/*
 * @(#)SerialVerify.java	1.3 97/12/05        
 *
 * Copyright (c) 1997 Sun Microsystems, Inc. All Rights Reserved.
 *
 * This software is the confidential and proprietary information of Sun
 * Microsystems, Inc. ("Confidential Information").  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sun.
 *
 * SUN MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SUN SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 *
 *
 */
import java.lang.*;
import java.io.*;
import java.util.*;
import java.lang.reflect.*;
import java.util.zip.*;
import java.text.MessageFormat;

/*
 * Serialization Evolution Test - 
 *
 * Used to see which classes have made Illegal, or legal, or both
 * legal and illegal changes in their persistent definitions.
 * 
 *
 * @author Swati Dhuleshia
 */

public class SerialVerify {

    /* 
     * Used during verification to hold the information about the class being
     * verified.
     */
    OriginalClass currentClass;

    /* 
     * Holds all of the classes and their information 
     */
    Hashtable classes;
    
    /* 
     * Used to write out and read in the information to and from the
     * file
     */
    static final String 
        CLASS = "CLSS ",
	SERIALIZABLE = "szable",
	EXTERNALIZABLE = "ezable",
	SUPER = "spr ",
	// non-static, non-transient field
	NSNTFIELD = "nsntfld ",                
	NSNTFIELDTYPE = "nsntfldtype ",      
	SUID = "suid "; 
 
    // Error log - set to System.err
     PrintStream log;


    // Test for legal changes? 
    boolean testForLegalChanges;

    // Test for illegal changes?
    boolean testForIllegalChanges;

    // Verbose mode or not
    boolean verbose;

    // tells if the class/package that the user asked for was found
    boolean found;

    // tells if the class currently being examined has had a problem
    boolean currProblem;

    // Number of classes verified or setup
    int numClasses;

    // Number of classes scanned
    int numScannedClasses = -1;

    // the properties Resource Bundle
    private static ResourceBundle messageRB;

    // the persistent fields of the current class (used in Verify)
    Hashtable persistentFields;

    // tells if there were any errors while this program ran
    boolean error;

    /*
     * where the packages to be scanned are put in to be sorted.
     * the key is the package name and the object is the list of
     * serializable classes in the package
     */
    Hashtable scannedPackages = new Hashtable();

    // where the packages names in sorted list are stored
    StringList pkgList = new StringList();

    public static void main(String[] args) {
	SerialVerify t = new SerialVerify(); 
     	t.run(args, System.out, System.out);

	if (!t.error) {
	    t.log.println();
	    if (t.numScannedClasses != -1) {
		t.log.println(getText("totalclassesscanned", 
				      Integer.toString(t.numScannedClasses)));
	    }
	    t.log.println();
	    t.log.println(getText
			  ("totalclassesserializable", 
			   Integer.toString(t.numClasses)));
	}

	/*
	 * This exit is made necessary by the fact that Class.forName method 
	 * used in scanClass is spawning off new threads with certain classes 
	 * such as java.awt.SystemColor
	 */
	if (t.error) {
	    System.exit(1);
	} else { 
	System.exit(0);
	}
    }

    /** 
     * Parse the command line and call appropriate methods
     * (either setUp or Verify) and also check for verbose mode.
     */
    void run(String[] args, PrintStream log, PrintStream ref) {
	
	this.log = log;
	
	boolean setup = false;
	boolean verify = false;
	File f = null;
	PrintWriter out = null;
	String fileName = null;
	Vector pkgNames = new Vector();

	if ((args.length < 2)) {
	    usage();
	    return;	    
	}

	// check for verbose mode
	if (args[0].equals("-v")) 
	    verbose = true;

	// check for setup or verify and their corresponding arguments
	for (int i = 0; i < args.length; i++) {
	    if (args[i].equals("-setup")) {
		setup = true;
		if (i < args.length - 2) {
		    fileName = args[++i];
		    i++;
		    while (i < args.length) {
			pkgNames.addElement(args[i]);
			i++;
		    }
		} else {
		    usage();
		}
	    } else if (args[i].equals("-verify")) {
		verify = true;
		int x = i - 1;
		// the default mode
		if (args.length == 2) {
		    testForLegalChanges = true;
		    testForIllegalChanges = true;
		}
		else {
		    if (x >= 0) {
			if (args[x].equals("-v")){
			    testForLegalChanges = true;
			    testForIllegalChanges = true;
			} else if (args[x].equals("-i")) {
			    testForIllegalChanges = true;
			} else if (args[x].equals("-l")) {
			    testForLegalChanges = true;
			} else {
			    usage();
			}
		    } else {
			usage();
		    }
		}
		if (i < args.length - 1) {
		    fileName = args[++i];
		}
		else {
		    usage();
		}
	    } 
	}

	
	if ((setup) && (fileName != null)) {
	    try {
		out = new PrintWriter
		    (new FileOutputStream(fileName));
	    } catch (Exception e) {
		System.out.println(e);
		error = true;
		System.exit(1);
	    }
	    String path = System.getProperty("java.class.path");

	    if (verbose) {
		System.out.println(getText("blank1", path));
	    }

	}

	/*
	 * Call either setup or verify after checking if the filename is
	 * appropriate.
	 */
	if (fileName != null) {
	    f = new File(fileName);
	    if (f.exists()) {
		if (verify) {
		    verify(fileName);
		} else if (setup) {
		    numScannedClasses = 0;
		    if (pkgNames.size() > 0) {
			if (verbose)
			    System.out.println(getText("doingSetup"));  
			for (int i = 0; i < pkgNames.size(); i++) {
			    String pkgName = (String) pkgNames.elementAt(i);
			    if (pkgName != null) {
				setUp(pkgName, fileName);
			    }
			}
			writeClasses(out);
		    } else {
			usage();
		    }
		} else {
		    usage();
		}
	    } else {
		if (verify) {
		    System.err.println(getText("nosuchfile"));
		} else if (setup) {
		    numScannedClasses = 0;
		    if (pkgNames.size() > 0) {
			 if (verbose)
			     System.out.println(getText("doingSetup"));   
			for (int i = 0; i < pkgNames.size(); i++) {
			    String pkgName = (String) pkgNames.elementAt(i);
			    if (pkgName != null) {
				setUp(pkgName, fileName);
			    }
			}
			writeClasses(out);
		    } else {
			usage();
		    }
		} else {
		    usage();
		}
	    }
	} else {
	    usage();
	}
	if (out != null) {
	    out.flush();
	    out.close();
	}
    }
       
    /**
     * Print out usage
     */
    void usage() {
	System.err.println(getText("usage"));
	System.exit(1);
    }

    /** 
     * Overall, setUp is used to accumulate information about Serializable and
     * Externalizable classes of the given class/package name and write this 
     * information to a given file. Later, after the classes have evolved, 
     * this information can be used by verify to make sure that the changes 
     * made to the classes are compatible, as defined by the Serialization 
     * Specification.
     *
     * More specifically, this method looks at the given package or class
     * name and calls either respectively scanFile or scanClass. If the 
     * given argument is a package, then all of the classes in the package
     * are written out to the given file. If the given argument is just
     * a class, then only that class is written out to the given file.
     */
    private void setUp(String fullPkg, String outFileName) {   

	// check if it's a class and call scanClass if it is.
	try {
	    Class c = Class.forName(fullPkg);
	    try {
		Class serial = Class.forName("java.io.Serializable");
		Class extern = Class.forName("java.io.Externalizable");
		found = true;
		numScannedClasses++;
		// only classes that are either Serializable or Externalizable
		if (serial.isAssignableFrom(c)) {
		    if (!scannedPackages.containsKey(fullPkg)) {
			scannedPackages.put(fullPkg, new StringList());
			pkgList.insertString(fullPkg);
		    }
		    scanClass((StringList)scannedPackages.get(fullPkg), 
			      fullPkg);
		} else {
		    log.println(getText("notserializable", fullPkg));
		}
	    } catch(Exception e) {
		error = true;
		System.err.println(e);
		System.exit(1);
	    }
	    return;
	} catch (Exception e) {
	    // FALL THROUGH   
	}

	String path = System.getProperty("java.class.path");

 	// go through the class path variable and look for the given pkg. name
	while (path != null) {
	    File f;
	    String s;
	    int index;

	    index = path.indexOf(File.pathSeparatorChar);
	    if (index == -1) {
		s = path;
		path = null;
	    } else {
		s = path.substring(0, index);
		path = path.substring(index + 1);
	    }
	    
	    if (s.endsWith(File.separator)) {
		s = s.substring(0, s.length() - File.separator.length());
	    }

	    f = new File(s);
		if (f.isDirectory()) {
		    String files[] = f.list();
		    
		    for (int i = 0; i < files.length; i++) {
			scanFile(new File(f, files[i]), "", fullPkg);
		    }
		} else {
		    scanFile( f, "", fullPkg);
		}
	}
	if (!found) {
	    log.println(getText("notfound", fullPkg));
	    error = true;
	}
    }
    /**
     * Deals with the given file appropriately. For a zip file, looks for
     * classes of the given package name (fullPkg) in the zip file. For a
     * directory, checks if it needs to go further into a directory to
     * get to the given package name. If it does, then it recursively calls
     * scanFile. For a .class file, it calls scanClass on the class name,
     * if the .class file is in the package that we are looking for.
     */
    void scanFile(File entry, String pkg, String fullPkg) {

	// if it's a zipfile, look for classes in our package.
	if (entry.getName().endsWith(".zip")) {
	    try {
		ZipFile z = new ZipFile(entry);
		for (Enumeration e = z.entries(); e.hasMoreElements();) {
		    ZipEntry ze = (ZipEntry)e.nextElement();
		    String name = ze.getName();
		    if (name.endsWith(".class") && 
			(pkg + name.replace('/','.')).startsWith(fullPkg)) {
			found = true;
			if (!scannedPackages.containsKey(pkg)) {
			    scannedPackages.put(pkg, new StringList());
			    pkgList.insertString(pkg);
			}
			numScannedClasses++;
			StringList tmpList = (StringList) 
			    scannedPackages.get(pkg);
			scanClass(tmpList, name.substring
				  (0, name.length() -6).replace('/', '.'));
		    }
		}
	    } catch (Exception e) {
		// do nothing if there are no entries etc in this file
	    }
	    /*
	     * if it's a class and it's in our package, scan the Class.
	     */
	} else if (entry.getName().endsWith(".class")) {
	    if (pkg.startsWith(fullPkg)) {
		found = true;
		if (!scannedPackages.containsKey(pkg)) {
		    scannedPackages.put(pkg, new StringList());
		    pkgList.insertString(pkg);
		}
		numScannedClasses++;
		StringList tmpList = (StringList) scannedPackages.get(pkg);
		scanClass(tmpList, 
			  pkg + 
			  entry.getName().substring(0, 
						    entry.getName().length() - 
						    6));
	    }
	    /*
	     * if it's a directory and has potential to lead us to the 
	     * package we want, then recurse.
	     */
	} else if (entry.isDirectory()) {
	    if ((pkg.startsWith(fullPkg)) || (fullPkg.startsWith(pkg))) {
		String[] files = entry.list();
		for (int i = 0; i < files.length; i++) {
		    File f = new File(entry, files[i]);
		    scanFile(f, pkg + entry.getName() + ".", fullPkg);
		}
	    }
	} else {
	    // Ignore
	}
    }	
    
    /* 
     * For a given class name, see if the class is Serializable. if so,
     * add it to the list of classes to be written out.
     */
    private void scanClass(StringList list, String name) {
	
	// ignore the special cases of String and ObjectStreamClass
	if ((name.equals("java.lang.String")) || 
	    (name.equals("java.io.ObjectStreamClass"))) {
	    return;
	}

	try {
	    Class c = Class.forName(name);
	    
	    Class serial = Class.forName("java.io.Serializable");
	    Class extern = Class.forName("java.io.Externalizable");

	    // only classes that are either Serializable or Externalizable
	    if (serial.isAssignableFrom(c)) {
		list.insertString(name);
	    } 
	} catch (Exception e) {
	    error = true;
	    System.out.println(e);
	    System.exit(1);
	}
    }

    
    /*
     * Write out the classes to the file in alphabetical order
     */
    private void writeClasses(PrintWriter out) {
	
	for (int i = 0; i < pkgList.size(); i++) {
	    String currPackage = pkgList.getNextString();
	    if (scannedPackages.containsKey(currPackage)) {
		StringList currList = (StringList) 
		    scannedPackages.get(currPackage);
		for (int j = 0; j < currList.size(); j++) {
		    String name = currList.getNextString();
		    writeClass(out, name);
		}
	    }
	}
    }
    
    /**
     * Write out the info about the given class 
     */
    private void writeClass(PrintWriter out, String name) {
	
	try {
	    Class c = Class.forName(name);
	    
	    Class serial = Class.forName("java.io.Serializable");
	    Class extern = Class.forName("java.io.Externalizable");

	    // only classes that are either Serializable or Externalizable
	    if (serial.isAssignableFrom(c)) {
		boolean externalizable = false;
		numClasses++;
		if (verbose)
		    System.out.println(getText("scanserializableclass", name));
			
		// print out class name
		out.println(CLASS + name);

		// print out whether it is serializable or externalizable
		if (extern.isAssignableFrom(c)) {
		    out.println(EXTERNALIZABLE);
		    externalizable = true;
		}
		else 
		    out.println(SERIALIZABLE);
		
		/*
		 * only need information about serializable (but not
		 * externalizable classes
		 */
		if (!externalizable) {

		    // print out the SerialVersionUID number
		    ObjectStreamClass osc = ObjectStreamClass.lookup(c);
		    out.println(SUID + osc.getSerialVersionUID());
		   
		    /*
		     * get the fields through ObjectStreamClass if possible
		     * otherwise, use the reflect method and just get the 
		     * non-transient, non-static fields (this should actually
		     * never happen)
		     */
		    Field[] fields = c.getDeclaredFields();
		    if (fields != null) {
			for (int i = 0; i < fields.length; i++) {
				/* 
				 * only deal with fields that are non-static and 
				 * non-transient.
				 */
			    int mod = fields[i].getModifiers();
				
			    if ((!(Modifier.isStatic(mod))) && 
				(!(Modifier.isTransient(mod)))) {
				out.println(NSNTFIELD + fields[i].getName());
				out.println
				    (NSNTFIELDTYPE + fields[i].getType());
			    }
			}
		    }
		    
		    // write out the superclasses themselves
		    for (Class spr = c.getSuperclass(); spr != null; 
			 spr = spr.getSuperclass()) {
			out.println(SUPER + spr.getName());
		    }
		}
	    } else {
		System.out.println(getText("notserializable", name));
	    }
	} catch (Exception e) { 
	    error = true; 
	    System.err.println(e);
	} 
    }


    /**
     * First, reads in all the information about the original classes from the 
     * given file into a hashtable. Then, compares this information with the
     * information of the evolved classes and looks for legal changes in
     * the class such as persistent field addition, modifier changes that 
     * result in persistent field addition, and superclass addition or deletion
     */
    private void verify (String filename) {
	
 	// first read in information about how the classes were originally
	readOrigClassInfo(filename);

	if (verbose)
	    System.out.println(getText("nowverifying")); 

	// now we see if the current classes has made legal changes 
	for (Enumeration e = classes.keys(); e.hasMoreElements();) {
	    numClasses++;
	    Class serial = null; 
	    Class extern = null;
	    String name = (String)e.nextElement();
	    currentClass = (OriginalClass)classes.get(name);
	    try {
		serial = Class.forName("java.io.Serializable");
		extern = Class.forName("java.io.Externalizable");
	    } catch (Exception ex) {
		System.out.println(e);
		error = true;
		return; // a fatal error
	    }
	    
	    try {
		currProblem = false; 
		Class newClass = Class.forName(name);
	 
		if (testForIllegalChanges) {
		    /*
		     * check that if Serializable before still Serializable
		     * and if Externalizable before still Externalizable
		     */
		    if (currentClass.isItExternalizable()) 
			checkIfStillExternalizable(currentClass, newClass);
		    else 
			checkIfStillSerializable(currentClass, newClass);
		}
		
		/*
		 * do the tests only if still serializable and is not
		 * externalizable
		 */
		if ((serial.isAssignableFrom(newClass)) && 
		    (!extern.isAssignableFrom(newClass)) &&
		    (!currentClass.isItExternalizable())) {
		
		    persistentFields = getPersistentFields(newClass);
		    
		    /*
		     * check for added fields and modifier changes
		     * that result in added persistent fields (such
		     * as going from static to non-static
		     */	
		    if (testForLegalChanges) {
			checkForAddedFields(currentClass, newClass);
		    }
		    
		    if (testForIllegalChanges) {
			/*
			 * check for deleted fields and modifier changes
			 * and type changes that result in deleted persistent
			 * fields.
			 */
			checkForDeletedFields(currentClass, newClass);
			
			// check that the SerialVersionUID matches
			checkSuid(currentClass, newClass);
		    }
		 
		    if ((testForLegalChanges) || (testForIllegalChanges)) {
			/*
			 * check for hierarchy changes such as superclass 
			 * deletion, superclass addition, and superclass 
			 * becoming a subclass
			 */
			checkForHierarchyChanges(currentClass, newClass);
		    }
		}
	    } catch (ClassNotFoundException ex) { 
		log.println();
		log.println(getText("inclass", name));
		log.println(getText("classdeleted")); 
	    }
	}
    }

    /**
     * Read information about the original classes from a file
     */
    void readOrigClassInfo(String filename) {
	// going to put all of the classes from the file into this hash table
	classes = new Hashtable();
	
	if (verbose)
	    System.out.println(getText("readingoriginalclasses"));

	// check out how the classes were defined originally
	try {
	    FileInputStream input = new FileInputStream(filename);
	    BufferedReader in = new BufferedReader
		(new InputStreamReader(input));
	    String line;
	
	    while ((line = in.readLine()) != null) {
		if (line.startsWith(CLASS)) { 
		    String name = line.substring(CLASS.length());
		    setCurrentClass(name);
		    if (verbose)
			System.out.println(getText("readingclass", name));
		} else if (line.startsWith(SUID)) {
		    currentClass.setSuid(Long.parseLong(line.substring
							(SUID.length())));
		} else if (line.startsWith(EXTERNALIZABLE)) {
		    currentClass.setExternalizable(true);
		} else if (line.startsWith(SERIALIZABLE)) {
		    currentClass.setExternalizable(false);
		} else if (line.startsWith(NSNTFIELD)) { 
		    currentClass.addField(line.substring(NSNTFIELD.length()));
		} else if (line.startsWith(NSNTFIELDTYPE)) { 
		    currentClass.addFieldType(line.substring
					      (NSNTFIELDTYPE.length()));
		} else if (line.startsWith(SUPER)) {
		    currentClass.addSuperType(line.substring(SUPER.length()));
		} else { 
		    System.out.println
			(getText("definitionfileerror"));
		    System.exit(1);
		}
	    }
	} catch (IOException e) {
	    error = true;
	    System.out.println(e);}
    }

    
    /**
     * This checks to make sure that the following incompatible changes did not
     * take place:
     * the fields have not been deleted
     * the fields have not been changed to static or transient
     * the fields types are still the same
     */
    private void checkForDeletedFields(OriginalClass d, Class c) {
	
	Vector oldFields = d.getFields();
	Vector oldFieldTypes = d.getFieldTypes();
 
	// check for all the original non-static, non-transient fields
	
	for (int i = 0; i < oldFields.size(); i++) {
	    String s = (String)oldFields.elementAt(i);
	    String t = (String)oldFieldTypes.elementAt(i);
	    if (s == null)
		break;
		// check for deleted field error
	    if (persistentFields != null) {
		if (!persistentFields.containsKey(s)) {
		    aProblem(c, getText("persistentfielddeleted", s));
		} else {
		    String newtype = (String)persistentFields.get(s);
		    if (!(t.equals(newtype))) {
			aProblem(c, getText
				 ("fieldtypechanged", s, t, newtype));
		    }
		}  
	    }
	}
    }

    /** 
     * Get the persistent fields of the class by using the ObjectStreamClass
     * getFields method.
     */
    Hashtable getPersistentFields(Class newClass) {

	Hashtable ht = new Hashtable();
	ObjectStreamClass osc = ObjectStreamClass.lookup(newClass);
	 
	Field[] fields = newClass.getDeclaredFields();
	if (fields != null) {
	    for (int i = 0; i < fields.length; i++) {
		/* 
		 * only deal with fields that are non-static and 
		 * non-transient.
		 */
		int mod = fields[i].getModifiers();
			
		if ((!(Modifier.isStatic(mod))) && 
		    (!(Modifier.isTransient(mod)))) {
		    ht.put(fields[i].getName(), 
			   (fields[i].getType()).toString());
		}

	    }
	}
	return ht;
    }

    /**
     * Check for persistent fields that may have been added.
     */
    void checkForAddedFields(OriginalClass oldClass, Class newClass) {

 	// put all the old fields in a hashtable
	Vector oldFields = oldClass.getFields();

	if (persistentFields != null) {
	    for (Enumeration e = persistentFields.keys(); 
		 e.hasMoreElements();) {
		String fieldName = (String)e.nextElement();
		if (!(oldFields.contains(fieldName))) {
		    String message = (getText
				      ("persistentfieldadded", fieldName));
		    aProblem(newClass, message);
		}
	    }
	}
    }

    /**  
     * This checks to make sure that the SerialVersionUID's are still the same
     * Changing them is an incompatible change!
     */
    private void checkSuid(OriginalClass d, Class c) {
	
	long oldsuid = d.getSuid();
	ObjectStreamClass obc = ObjectStreamClass.lookup(c);	
	long newsuid = obc.getSerialVersionUID();
	
	if (oldsuid != newsuid) {
	    aProblem
		(c, getText("suidchanged", 
			    Long.toString(oldsuid), 
			    Long.toString(newsuid))); 
	}
    }

    private void checkIfStillSerializable(OriginalClass currentClass, 
					  Class newClass)  {
	Class serial = null;
	Class extern = null;
	
	try {
	    serial = Class.forName("java.io.Serializable");
	    extern = Class.forName("java.io.Externalizable");
	} catch (Exception e) {
	    System.out.println(e);
	    error = true;
	    return;
	}
	if (!(serial.isAssignableFrom(newClass))) 
	    aProblem(newClass, getText("serializableremoved"));
	if ((extern.isAssignableFrom(newClass)))
	    aProblem(newClass, getText("serialtoextern"));
	
    }
	  
    private void checkIfStillExternalizable(OriginalClass currentClass, 
					  Class newClass) {
	Class serial = null;
	Class extern = null;

	try {
	    serial = Class.forName("java.io.Serializable");
	    extern = Class.forName("java.io.Externalizable");
	} catch (Exception e) {
	    System.out.println(e);
	    error = true;
	    return;
	}

 	// if Externalizable before, still Externalizable?

	if (!(extern.isAssignableFrom(newClass))) {
	    if ((serial.isAssignableFrom(newClass))) {
		aProblem(newClass, getText("externtoserial"));
	    }
	    else {
		aProblem(newClass, getText("externalizableremoved")); 
	    }
	}
    }
    
    /**
     * Check for changes in hierarchy such as addition of new super type
     * or deletion of a supertype.
     */
    void checkForHierarchyChanges(OriginalClass oldClass, Class newClass) {
	
	// get the supertype vectors
	Vector oldSupers = oldClass.getSuperTypes();
	Vector newSupers = getNewClassSuperTypes(newClass);

	if (testForIllegalChanges) {
	    // check if a subclass has not become a superclass
	    checkForCircularityError(newClass, oldSupers);
	}
	
	if (testForLegalChanges) {
	    // check if superclass has been added
	    checkForAddedSupers(newClass, oldSupers, newSupers);
	    
	    // check if superclass has been deleted
	    checkForDeletedSupers(newClass, oldSupers, newSupers);
	}
    }

    /**
     * Get a class's supertypes in a vector form
     */
    Vector getNewClassSuperTypes(Class newClass) {

	Vector superTypes = new Vector();
	
	// write out the superclasses themselves
	for (Class spr = newClass.getSuperclass(); spr != null; 
	     spr = spr.getSuperclass()) {
	    superTypes.addElement(spr.getName());
	}
	return(superTypes);
    }

    /**
     * Check if any supers have been added to the given class
     */
    void checkForAddedSupers(Class newClass, Vector oldSprs, Vector newSprs) {

	for (Enumeration e = newSprs.elements(); e.hasMoreElements();) {
	    String newSpr = (String)e.nextElement();
	    if (!oldSprs.contains(newSpr)) {
		aProblem(newClass, getText("superclassadded", newSpr));
	    }
	}
    }

    /**
     * Check if any supers have been deleted from the given class
     */
    void checkForDeletedSupers(Class newClass, Vector oldSprs, 
			       Vector newSprs) {

	for (Enumeration e = oldSprs.elements(); e.hasMoreElements();) {
	    String oldSpr = (String)e.nextElement();
	    if (!newSprs.contains(oldSpr)) {
		aProblem(newClass, getText("superclassdeleted", oldSpr));
	    }
	}
    }
    
    /**
     * This checks for the incompatible change of a previous superclass of a
     * class now becoming a subclass
     */ 
    private void checkForCircularityError(Class c, Vector oldSuperTypes) {

	// for each of the supertypes, check that they are not now subtypes
	for (int i = 0; i < oldSuperTypes.size(); i++) {
	    String supertype = (String) oldSuperTypes.elementAt(i);
	    try {
		if (c.isAssignableFrom(Class.forName(supertype))) {
		    aProblem(c, getText
			     ("hierarchychanged", supertype, c.getName())); 
		}    
	    }
	    catch (ClassNotFoundException e) { 
		// ignore. will be caught elsewhere  
	    }
	} 
    }
    
    /** 
     * writes to error-log the class name and the error message
     */
    private void aProblem(Class c, String msg) {
	
	if (currProblem == false) {
	    log.println();
	    log.println(getText("inclass", c.getName()));
	    currProblem = true;
	}
	log.println(msg);
    }
 
    /**
     * Used by verify to set the currentClass variable to the class with
     * with the given name from the hashtable
     */
    private void setCurrentClass(String name) {
	// just in case the class is already there for some reason! 
	OriginalClass d = (OriginalClass) classes.get(name);
	if (d == null) {
	    d = new OriginalClass();
	    classes.put(name, d);
	}
	currentClass = d;     
    }

    
    /** 
     * Used to print messages to the end user. Need to do it this way
     * to support internationalization
     */
    private static String getText(String key, String fixed1, String fixed2, 
				 String fixed3) {
	if (messageRB == null) {
	    initResource();
	}
	try {
	    String message = messageRB.getString(key);
	    String[] args = new String[3];
	    args[0] = fixed1;
	    args[1] = fixed2;
	    args[2] = fixed3;
	    return MessageFormat.format(message, args);
	} catch (MissingResourceException e) {
	    if (fixed1 == null)  fixed1 = "null";
	    if (fixed2 == null)  fixed2 = "null";
	    if (fixed3 == null)  fixed3 = "null";
	    String args[] = { key, fixed1, fixed2, fixed3 };
	    String message = 
		"THE MESSAGE FILE IS BROKEN: key={0}, arguments={1}, {2}, {3}";
	    return MessageFormat.format(message, args);
	}
    }

    private static String getText(String key) {
	return(getText(key, null, null, null));
    } 
    private static String getText(String key, String fixed1) {
	return(getText(key, fixed1, null, null));
    } 

    private static String getText(String key, String fixed1, String fixed2) {
	return(getText(key, fixed1, fixed2, null));
    }

    /**
     * Get the local resource bundle. Used for internationalization.
     */
    private static void initResource() {
	try {
	    messageRB =
		ResourceBundle.getBundle
		("resources.SerialVerify");
	} catch (MissingResourceException e) {
	    throw new Error("Fatal: Resource is missing");
	}
    }
}
    
    
/**
 * This holds all of the data for a original serializable class
 */
class OriginalClass {

    // a boolean that tells if it is serializable or externalizable
    private boolean isExternalizable;
    
    // an array of the fields
    private Vector fields = new Vector();
	
    // an array of the corresponding field types above
    private Vector fieldtypes = new Vector();
    
    // an array to hold all the supertypes of the class - used to 
    // test that class has not moved in the hierarchy.
    private Vector supertypes = new Vector();
	
    // the serialVersionUID number
    private long suid;

    /**
     * add a field name to the field array
     */
    void addField(String f) {
	fields.addElement(f);
    }
  
    /**
     * add a field type to the field array
     */
    void addFieldType(String ft) {
	fieldtypes.addElement(ft);
    }
	
    /**
     * add a supertype to the super array
     */
    void addSuperType(String st) {
	supertypes.addElement(st);
    }

    /**
     * set the isItExternalizable field with the given boolean value
     */
    void setExternalizable(boolean isExternalizable) {
	this.isExternalizable = isExternalizable;
    }
    
    /**
     * set the suid field with the given long value
     */
    void setSuid(long suid) {
	this.suid = suid;
    }
	
    /**
     * get the fields in a vector
     */
    Vector getFields() {
	return(fields);
    }
    
    /**
     * get the fieldtypes vector 
     */
    Vector getFieldTypes() {
	    return fieldtypes;
    }
    
    /**
     * get the supertypes vector
     */
    Vector getSuperTypes() { 
	return supertypes;
    }
	
    /**
     * get the suid field 
     */
    long getSuid() {
	return suid;
    }
    
    /**
     * return the isItExternalizable field
     */
    boolean isItExternalizable() {
	return (isExternalizable);
    }
}

/**
 * Used to keep a sorted list of classes to be written to the file
 */
class StringList {

    ListElement head;
    ListElement curr;
    int size;
    
    /**
     * Insert the given string into the list in the proper place
     * (alphabetically ordered)
     */
    void insertString(String str) {
	if (head == null) {
	    head = new ListElement();
	    curr = head;
	    head.setString(str);
	    size++;
	} else {
	    // case it goes in the beginning
	    if (str.compareTo(head.getString()) < 0) {
		ListElement tmp = new ListElement();
		tmp.setString(str);
		tmp.setNext(head);
		head = tmp;
		curr = head;
		size++;
	    } else {
		ListElement first = head;
		ListElement second = head.getNext();
		while (first != null) {
		    if (second != null) {
			// add to the middle
			if (str.compareTo(second.getString()) < 0) {
			    ListElement tmp = new ListElement();
			    tmp.setString(str);
			    first.setNext(tmp);
			    tmp.setNext(second);
			    size++;
			    break;
			} 
			// keep on going through list
			else {
			    ListElement tmp = second;
			    second = second.getNext();
			    first = tmp;
			}
		    } else {
			// add to the end
		        ListElement tmp = new ListElement();
			tmp.setString(str);
			first.setNext(tmp);
			size++;
			break;
		    }
		}
	    }
	}
    }

    /**
     * return the number of elements in the list
     */
    int size() {
	return size;
    }

    /**
     * get the next string in the list.
     * returns null if there are no more strings.
     */
    String getNextString() {
	if (head != null) {
	    if (curr != null) {
		String str = curr.getString();
		curr = curr.getNext();
		return str;
	    }
	} 
	return null;
    }
}

/**
 * Contains the String data and the pointer to the next element
 * in the list.
 */
class ListElement {

    String str;
    ListElement next;

    void setString(String str) {
	this.str = str;
    }
    
    void setNext(ListElement elem) {
	next = elem;
    }

    String getString() {
	return str;
    }

    ListElement getNext() {
	return next;
    }
}
 
