/*
 * Example code: calling a user C function from server-side JavaScript
 *
 * Copyright (c) 1996
 * Netscape Communications Corporation
 * All rights reserved
 * 
 */

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include "lwccall.h"

/*
 * 
 * Notes:
 * 
 * LiveWire needs to know about shared libraries (.dll or .so) that you
 * want to use. You tell LiveWire about shared libraries by adding
 * something like this to your livewire.conf, for one of your .web
 * applications:
 * 
 *     library=/full/path/to/your/file.so
 * 
 * If you change your shared library, you need to restart your server
 * before it will be used. In other words, restarting the .web
 * application that uses a shared library will not reload (dlopen
 * on unix, LoadLibary on NT) the library. You have to restart your
 * server.
 * 
 * Your .web application needs to know about entry points in your 
 * shared library. You tell your .web application about entry points
 * with something like this:
 * 
 *     registerCFunction("echoCCallArguments",
 *                       "c:\\yourpath\\mystuff.dll",
 *                       "mystuff_EchoCCallArguments")
 * 
 * On unix it might look like this:
 * 
 *     registerCFunction("echoCCallArguments",
 *                       "/yourpath/mystuff.so",
 *                       "mystuff_EchoCCallArguments")
 * 
 * This registration needs to be done once for each .web application
 * that wants to use the entry point. Suggestion: a good time to
 * register the entry point is when the .web application is started.
 * To do this, put the registration in your 'start' page. It does
 * not need to be registered in each page that makes a call to your
 * function. It does have to be registered before you can call it.
 * 
 * The first argument to registerCFunction is the name you assign
 * to the function. Use the same string as the first argument in
 * the callC server-side JavaScript function. The second argument
 * is the full path to your shared library. This string must be the
 * same as what you put in your livewire.conf file. The third
 * argument is the name of your C function.
 * 
 * If you register and call a C function in a single html page, it
 * might look something like this:
 * 
 *   <server>
 *     var isRegistered = registerCFunction("echoCCallArguments",
 *         "c:\\yourpath\\mystuff.dll","mystuff_EchoCCallArguments")
 *     if (isRegistered == true) {
 *         var returnValue = callC("echoCCallArguments","first arg",
 *                                 42,true,"last arg")
 *         write(returnValue)
 *     }
 *   </server>
 * 
 * The html emitted by this code would look like this:
 *
 *     argc = 4<br>
 *     argv[0].tag: string; value = first arg<br>
 *     argv[1].tag: double; value = 42<br>
 *     argv[2].tag: boolean; value = true<br>
 *     argv[3].tag: string; value = last arg<br>
 *
 * Typical command lines:
 * 
 *     VC++4.0 for NT
 *         cl -LD lwccall.c /link /EXPORT:mystuff_EchoCCallArguments
 *
 * For more info, see notes in lwccall.h.
 *
 */

/***********************************************************************
 * Support routines
 */

/*
 * Name: freeString
 * Returns: void
 * Description: This function is a callback function. It is returned to 
 *     JavaScript along with a string result. After JavaScript has
 *     captured the string, it calls this function so you can do any
 *     cleanup you need to do.
 */
static void 
freeString(char* s)
{
    if (s)
        free(s);
}

/*
 * Name: addToResult
 * Returns: pointer to new string
 * Description: Create a new string that is the concatenation of
 *     'oldResult' and 'more'. Release the old string. It's okay for
 *     either argument to be NULL.
 * Assumption: oldResult can be released by free().
 */
static char* 
addToResult(char* oldResult, char* more)
{
    int oldResultLen = oldResult ? strlen(oldResult) : 0;
    int moreLen = more ? strlen(more) : 0;
    int newResultLen = oldResultLen + moreLen;
    char* newResult = (char*)malloc(newResultLen+1);

    if (oldResult) {
        strcpy(newResult,oldResult);
        if (more)
            strcat(newResult,more);
        free(oldResult);
    } 
    else {
        if (more)
            strcpy(newResult,more);
        else
            *newResult = '\0';
    }
    return newResult;
}

/***********************************************************************
 * Exported functions
 */

/*
 * Name: mystuff_EchoCCallArguments
 * Returns: void
 * Description: This is an exported function to be called by server-side
 *     JavaScript. It creates a string result that is html that reports
 *     both the type and the value of each of the arguments passed in.
 */
void
mystuff_EchoCCallArguments(int argc, 
                           struct LivewireCCallData argv[],
                           struct LivewireCCallData *result)
{
    char* resultString = 0;
    int i;
    char buf[164];

    sprintf(buf,"argc = %d<br>\n",argc);
    resultString = addToResult(resultString,buf);

    i = 0;
    while (i < argc) {
        switch(argv[i].tag) {
        case LIVEWIRE_CCALLDATA_UNKNOWN:
            sprintf(buf,"argv[%d].tag: unknown<br>\n",i);
            resultString = addToResult(resultString,buf);
            break;
        case LIVEWIRE_CCALLDATA_NULL:
            sprintf(buf,"argv[%d].tag: null<br>\n",i);
            resultString = addToResult(resultString,buf);
            break;
        case LIVEWIRE_CCALLDATA_INTEGER:
            sprintf(buf,"argv[%d].tag: integer; value = %d<br>\n",
                    i,argv[i].u.i);
            resultString = addToResult(resultString,buf);
            break;
        case LIVEWIRE_CCALLDATA_DOUBLE:
            sprintf(buf,"argv[%d].tag: double; value = %.20g<br>\n",
                    i,argv[i].u.d);
            resultString = addToResult(resultString,buf);
            break;
        case LIVEWIRE_CCALLDATA_STRING:
            sprintf(buf,"argv[%d].tag: string; value = ",i);
            resultString = addToResult(resultString,buf);
            resultString = addToResult(resultString,argv[i].u.s);
            resultString = addToResult(resultString,"<br>\n");
            break;
        case LIVEWIRE_CCALLDATA_BOOLEAN:
            sprintf(buf,"argv[%d].tag: boolean; value = %s<br>\n",
                    i,argv[i].u.i?"true":"false");
            resultString = addToResult(resultString,buf);
            break;
        default:
            sprintf(buf,"argv[%d].tag is corrupt<br>\n",i);
            resultString = addToResult(resultString,buf);
            break;
        }
        i++;
    }

    result->tag = LIVEWIRE_CCALLDATA_STRING;
    result->u.s = resultString;
    result->freeString = freeString;
}
