/*
**  FILE:          postit.c
**  AUTHOR:        Kent Landfield
**
**  ABSTRACT:      Post source and info submissions to c.s.m
**
** This software is Copyright (c) 1990, 1991 by Kent Landfield.
**
** Permission is hereby granted to copy, distribute or otherwise 
** use any part of this package as long as you do not try to make 
** money from it or pretend that you wrote it.  This copyright 
** notice must be maintained in any copy made.
**
** NOTES TO DO:
**      1. Need to assure that  this works on the command line as 
**         well as interactively.  Still need to deal with the 
**         Supersedes: and Patch-To: lines.
**      2. Comment the hell out of this and the reasons why things
**         were done the way they were.
**      3. Add the ability to fully process an article and place it
**         in a queue to be dealt with later so that a script and
**         cron could assure that the same number of postings went
**         out each day.
*/

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <ctype.h>
#include <errno.h>
#include <time.h>
#include <varargs.h>
#include "postit.h"

#ifdef TEST
  static char sccsid[] = "@(#)postit.c	1.11 5/31/92 -Kent+ imd.sources";
# define NEWSGROUP       "imd.sources"
# define INFO_NEWSGROUPS "imd.sources,imd.info"
# define NG_DIR          "/u1/csm/post/test"
# define MAXPOSTSIZE     100000
#else
  static char sccsid[] = "@(#)postit.c	1.11 5/31/92 -Kent+ comp.sources.misc";
# define NEWSGROUP       "comp.sources.misc"
# define INFO_NEWSGROUPS "comp.sources.misc,comp.sources.d"
# define NG_DIR          "/u1/csm"
# define MAXPOSTSIZE      95000   /* 61440 */
#endif /*TEST*/

#define FOLLOWUPTO "comp.sources.d"
#define APPROVED   "kent@sparky.imd.sterling.com" /* For Approved:   */
#define HOSTPOST   "sparky.IMD.Sterling.COM"      /* For References: */
#define GRP_PREFIX "csm"                          /* For References: */
#define DAYS_TILL_EXPIRE  45                      /* For Expires:    */

/*
**  Configuration parameters.
*/
#define EDITOR      "vi"                      /* Default editor     */
#define INEWS       "/usr/bin/inews"          /* Where inews lives  */
#define QUEUE_DIR   "/u1/csm"                 /* .issue and .index  */
#define SENDMAIL    "/usr/ucb/Mail"           /* Mailer program     */
#define SUBJECTHDR  "Subject: "
#define SHELL       "/bin/ksh"

/*
** validation/checksum defines
*/
#define HASHMD4    "/usr/local/bin/hashmd4" /* MD4 Signature program  */
#define CHECKSUMHDR "X-Md4-Signature: "     /* MD4 Checksum header    */

/*
** Type of posting defines
*/
#define NONE  -1
#define SOURCE 0
#define INFO   1
#define PATCH  2

/*
** General defines
*/
#define YES 1
#define NO  0
#define SEC_PER_DAY  86400

char *days[]   = {"Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat"};
char *months[] = {"Jan", "Feb", "Mar", "Apr", "May", "Jun", 
                  "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"};

char *NG_HOME;          /* Newsgroup Home directory      */
char *progname;         /* name of executable            */
char buf2[BUFSIZ];      /* read-ahead  buffer            */
char lowbuf[2*BUFSIZ];
char s[2*BUFSIZ];
char Tempfile[BUFSIZ];
char Archbuff[BUFSIZ];
char Buff[BUFSIZ];
char Issuepath[BUFSIZ];

int  Debug;
int  Display_only;
int  Interactive;
int  Reinit;
int  Verbose;
int  Repost = FALSE;
int  Last_post_type = SOURCE;
int  PostingType = SOURCE;
int  First_ref = 1;

FILE *errfp;            /* standard error file pointer   */
FILE *logfp;            /* standard output file pointer  */
FILE *Issuefp;
FILE *Submission;
FILE *Out;

struct posting {
	int type; 
	int volume;
	int issue;
	int info;
	int part;
	int lastpart;
	int patch_num;
	char subject[BUFSIZ];
	char archname[BUFSIZ];
	char submitter[BUFSIZ];
	char patchto[BUFSIZ];
	char supersedes[BUFSIZ];
	char author_name[BUFSIZ];
	char author_signon[BUFSIZ];
	char env[BUFSIZ];
	char references[BUFSIZ];
	char expires[BUFSIZ];
};
struct posting post;
struct header header;
struct stat stbuf;
	
char *post_to_newsgroup(); /* local function declaration    */
char *read_line();         /* local function declaration    */
FILE *efopen();            /* local function declaration    */
void patch_to_format();    /* local function declaration    */
void environment();        /* local function declaration    */

char *strchr();            /* libc.a function declaration   */
char *strrchr();           /* libc.a function declaration   */
char *strcpy();            /* libc.a function declaration   */
char *strcat();            /* libc.a function declaration   */
char *getenv();            /* libc.a function declaration   */
FILE *fopen();             /* libc.a function declaration   */
void exit();               /* libc.a function declaration   */
int  strlen();
int  atoi();
int  fprintf();
int  unlink();
int  fputs();
int  strncmp();
int  fclose();
int  printf();
int  system();
int  sscanf();
int  tolower();

#ifndef SYSV
int  perror();
#else
void  perror();
#endif /*SYSV*/

/*
** Local declarations
*/
char *str_lower();
int  GetChecksum();
void init_header();
void init_post();
void read_issue_info();
void usage();
void submission_header();
void get_article_type();
void update_sequence_numbers();
void store_line();
void dump_message();
void archivename_line();
void subject_line();
void submitted_by();
void get_parts();
void format_posting();
void edit_posting();
void post_file();
void data();
void print_val();

int main(argc, argv)
int argc;
char **argv;
{
   int getopt();
   int strlen();

   int c;
   extern char *optarg;
   extern int optind;
   extern int opterr;

   opterr = 0;
   logfp = stdout;
   errfp = stderr;
   progname = argv[0];

   Display_only = 0;
   Debug = 0;
   Verbose = 0;
   Reinit = FALSE;

   /*
   ** Where is the posting queue directory ?
   */

   if ((NG_HOME = getenv("POSTQUEUE")) == NULL)
        NG_HOME = NG_DIR;

   Interactive = isatty(0);

   init_post();
   read_issue_info();

   if (argc > 1) {
      while ((c = getopt(argc,argv,"a:dDe:f:i:I:m:p:s:t:Rr:v:VW")) != EOF) {
         switch (c) {
             /*
             ** General flags 
             */
             case 'r': /* Article repost */
                 Repost = TRUE;
                 post.issue = atoi(optarg);
                 break;
             case 'V':
                 Verbose++;
                 break;
             case 'd':
                 Display_only++;
                 break;
             case 'R':
                 (void) fprintf(errfp,"%s\n", "Reiniting");
                 Reinit = TRUE;
                 break;
             case 'D':
                 Debug++;
                 break;
             case 'W':
                 (void) fprintf(errfp,"%s\n", sccsid);
                 return(2);
             /*
             ** Header line information
             */
            case 'a':    /* archive-name: */
                 (void) strcpy(post.archname, optarg);
                 Interactive = FALSE;
                 break;
             case 'm':    /* Submitter's email address - Submitted-by: */
                 (void) strcpy(post.author_signon, optarg);
                 Interactive = FALSE;
                 break;
             case 'f':    /* Submitter's full name - Submitted-by: */
                 (void) strcpy(post.author_name, optarg);
                 Interactive = FALSE;
                 break;
             case 'i':    /* issue number - override .issue file   */
                 post.issue = atoi(optarg);
                 Interactive = FALSE;
                 break;
             case 'I':   /* INF posting number - override .issue file */
                 post.info = atoi(optarg);
                 Interactive = FALSE;
                 PostingType = INFO;
                 break;
             case 'p':   /* posting part number */
                 post.part = atoi(optarg);
                 break;
             case 's':   /* Subject: line topic information */
                 (void) strcpy(post.subject, optarg);
                 break;
             case 't':    /* Total number of parts in the submission */
                 post.lastpart = atoi(optarg);
                 break;
             case 'v':    /* volume number */
                 post.volume = atoi(optarg);
                 Interactive = FALSE;
                 break;
             case 'e':   /* Environment: information */
                 (void) strcpy(post.env, optarg);
                 break;
             default:
                 usage();
                 return(1);
         }
      }
   }

   /* NEED TO ASSURE THAT IF CALLED NON-INTERACTIVELY THAT ALL THE */
   /* REQUIRED INFORMATION HAS BEEN SUPPLIED.                      */

   for (; optind < argc; optind++) {  /* process submissions to post */
        submission_header(argv[optind]);

        /* 
        ** If all the info was specified on the command line 
        ** then bail out since they only get to specify one
        ** package on the command line at a time...
        */

        if (!Interactive)
            break;
        
        /* Increment the appropriate counter... */

        if (PostingType == SOURCE) 
            post.part++;

        if (PostingType == PATCH) {
            if (post.lastpart == 1) /* not a multi-part */
                post.patch_num++;
            else
                post.part++;
        }

        /*
        ** NEED TO CHECK TO SEE IF THE SUBMITTER OF A SUBSEQUENT 
        ** ARTICLE IS THE SAME AS THE  PREVIOUS ARTICLE. ELSE, 
        ** RESET SOME HEADER VARIABLES.
        */
   }
   return(0);                              /* terminate this process */
}

void usage()    
{
	(void) fprintf(errfp,"\nusage:\n\
  %s [ -dDnVW ]       [ -i issue-num ]  [ -a archive-name ]\n\
         [ -t num-parts ] [ -p part-num ]   [ -s subject ]\n\
         [ -r issue-num ] [ -m email-addr ] [ -f full-name ]\n\
         [ -R last-art ]  [ -e keywords ]   [ -I info-num ]\n\
         [ -v volume ]       file [ .. ]\n\
\n\
options:\n\
    -D            Debug. Input lines are printed.\n\
    -V            Verbose.\n\
    -W            Print out 'what' version information.\n\
    -d            Display submission header information only.\n\
                  No posting is done.\n\
    -R last-article in series posted\n\
                  The -R allows the moderator to reinitialize the\n\
                  internal variables so as to save time having to\n\
                  type them in again.\n\
    -a archive-name  Specify the number of the issue.\n\
    -i issue-num  Specify the number of the issue.\n\
    -I info-num   Specify the number of the INF posting.\n\
    -t num-parts  Specify the number of issues in a multipart posting.\n\
    -p part-num   Specify the current part number of the issue in a\n\
                  multipart posting.\n\
    -s subject    Specify the subject of the issue.\n\
    -v volume     Specify the volume of the issue.\n\
    -r issue-num  Repost the specified issue.\n\
    -m mail-addr  Submitter's email address.\n\
    -f full-name  Submitter's full name.\n\
    -e keywords   Environment keywords.\n\
\n",
	progname);
}

void init_post()
{
	post.type = SOURCE;
	post.volume = -1;
	post.issue = -1;
	post.info = -1;
	post.part = 1;
	post.lastpart = 1;
	post.patch_num = 1;
	post.subject[0] = '\0';
	post.archname[0] = '\0';
	post.submitter[0] = '\0';
	post.patchto[0] = '\0';
	post.supersedes[0] = '\0';
	post.author_signon[0] = '\0';
	post.author_name[0] = '\0';
	post.env[0] = '\0';
	post.references[0];
	post.expires[0];
}

int fill_header(submitfp,hdr)
FILE *submitfp;
struct header *hdr;
{
    char *eop;
    int lineno;
    int hcnt;

    init_header(hdr);
 
    lineno = 0;
    hcnt = 0;
 
    while (read_line(submitfp) != NULL) {
        lineno++;
        if (Debug && !Verbose)
            (void) fputs(s,errfp);
        /*
        ** The first blank line ends the header section if it is a
        ** mail submission. If I have to repack it, it may not have
        ** anything in it...
        **
        ** IDEA: In order to assure that this works with forwarded
        **       articles, search for a second header termination.
        */
        if ((*s == '\n') || (strchr(s,':') == NULL)) {
            if (Reinit == TRUE) 
               hcnt++;
            else
               hcnt = 2;

            if (hcnt == 2) {
                break;
            }
        }
 
        eop = s;
        while (*++eop)
            if (*eop == '\n')
                *eop = '\0';
 
        /*
        ** Determine the type of the line and then
        ** store the line in its appropriate structure
        ** element for later retrieval.
        */
        store_line(hdr);
    }
    return(lineno);
}

reinit()
{
    char *sp;
    char num[10];

    /*
    ** Need to determine the part/patch numbering 
    */
    if ((sp = strrchr(header.subject,',')) != NULL) {
         sp += 2;
         if (strncmp(sp,"Part",4) == 0) {
             (void) strcpy(num,sp+4);
             if ((sp = strrchr(num,'/')) != NULL) {
                 *sp = '\0';
                  post.part = atoi(num);
                  ++sp;
	          post.lastpart = atoi(sp);
             }
             else 
                  post.part = atoi(num);
         }
         /* STILL NEED TO DO PATCHES... */
    }
    /*
    ** Assure that the references line is filled in if
    ** it is needed... Reinits are used on multipart
    ** packages (mainly) so a minumual check is needed.
    */
    if (header.messageid[0] != '\0')
       (void) strcpy(post.references, header.messageid);

    if (header.references[0] != '\0')
       (void) strcpy(post.references, header.references);

    if (header.supersedes[0] != '\0')
       (void) strcpy(post.supersedes, header.supersedes);

    get_article_type();
    archivename_line();   /* get the Archive-name: line info */
    subject_line();       /* get the Subject: line info      */
    submitted_by();       /* get the Submitted-by: line info */
    get_parts();
    if (PostingType == PATCH)   /* Is a patch-to needed ? */
        patch_to_format(PATCH_TO,"Patch-To:",post.patchto,header.patch_to);
    patch_to_format(SUPERSEDES,"Supersedes:",post.supersedes,header.supersedes);
    environment();
    First_ref = 0;       /* Assure generate_references is not called */
    Reinit = FALSE;      /* Our work here is done... */
    buf2[0] = '\0';
    (void) fclose(Submission);
}

void submission_header(flname)
    char *flname;
{
    char *mktemp();

    int lineno;
 
    Submission = efopen(flname, "r");

    /*
    ** Read the submitted article and fill in
    ** the header structure for future use...
    */

    lineno = fill_header(Submission, &header);

    if (Interactive)  
        dump_message(flname);

    if (Display_only)
        return;

    if (Reinit == TRUE) {
        reinit();
        return;
    }
    /*
    **  Open temporary output files. 
    */
    Out = efopen(mktemp(strcpy(Tempfile, "/usr/tmp/modXXXXXX" )), "w");
    get_article_type();
    update_sequence_numbers();

    archivename_line();   /* get the Archive-name: line info */
    subject_line();       /* get the Subject: line info      */
    submitted_by();       /* get the Submitted-by: line info */
    get_parts();

    if (PostingType == PATCH)   /* Is a patch-to needed ? */
        patch_to_format(PATCH_TO,"Patch-To:",post.patchto,header.patch_to);
    patch_to_format(SUPERSEDES,"Supersedes:",post.supersedes,header.supersedes);
    environment();

    format_posting();

    if (lineno == 1) {
        /*
        ** This is not a mail message but a 
        ** shar file made by the moderator.
        */
        (void) fputs(s, Out);
    }
    (void) fputs(buf2, Out);   /* Print out the read ahead buffer */

    while (fgets(s,sizeof(s),Submission) != NULL) 
        (void) fputs(s, Out);

    buf2[0] = '\0';

    if (PostingType != INFO)
        (void)fprintf(Out, "exit 0 # Just in case...\n");

    (void) fclose(Out);
    (void) fclose(Submission);

    /* 
    ** At this point the article to be posted is in the
    ** Tempfile and is ready to post.  Place the user into
    ** the editor so that they can fix up and verify the
    ** posting prior to it being sent.
    */
    edit_posting(Tempfile);

    /*
    ** Add the X-Md4-Signature header line... 
    */
    (void) printf("Computing Issue's MD4 Signature...\n");
    (void) sprintf(Buff, "exec %s %s", HASHMD4, Tempfile);
    (void) system(Buff);

    /*
    ** Check to assure that the article is within acceptable
    ** size levels. Anything over 60K can cause problems to
    ** 286 and PDP11 systems and any other small  model system
    */ 

    if (stat(Tempfile,&stbuf) == 0) {
        (void) printf("Article size = %d\n",stbuf.st_size);
        if (stbuf.st_size > MAXPOSTSIZE) {
            (void) printf("EXCEEDS %d maximum posting size\n",MAXPOSTSIZE);
            (void) printf("Bypassing the posting of %s\n", flname);
            (void) unlink(Tempfile);
            return;
        }
    }

    post_file();
    (void) unlink(Tempfile);
    return;
}

void patch_to_format(which, linehdr, poststor, headerstor)
int which;
char *linehdr;
char poststor[];    
char headerstor[];    
{
    char *p;
    char issue_num[128];
    char vname[128];
    char packname[128];

    if (!Interactive)   /* ADD SUPPORT FOR Supersedes: AND Patch-To: */
        return;         /* FOR NOW, JUST PUNT THE LACK OF SUPPORT... */

    if (which == SUPERSEDES && poststor[0] == '\0') {
        if (headerstor[0] == '\0') {
            if (yes_or_no("n","Supersedes an existing package ? [n] ") == NO)
                return;
        }
    }
   
    /*
    ** Check if a similar line was used in the previous posting.
    */
    if (poststor[0] != '\0') {
        /* 
        ** See if the user wishes to use the 
        ** currently specified information.
        */
        if (yes_or_no("y","Use %s [%s] ? [y] ",linehdr,poststor) == YES)
                return;
    }

    /* 
    ** Let's see if the user supplied one in the posting that was
    ** sent to the moderator...
    */
    if (headerstor[0] != '\0') {
        if (yes_or_no("y","Use %s [%s] ? [y] ",linehdr,headerstor) == YES) {
            (void) strcpy(poststor, headerstor);
            return;
        }
    }

    /*
    ** Let's prompt for it..
    */

    for(;;) {
	poststor[0] = '\0';
        vname[0] = '\0';
        issue_num[0] = '\0';
        (void) strcpy(packname, post.archname);

        do {
            (void) printf("Enter Package name [%s] ? ", packname);
            (void) fgets(Buff, sizeof Buff, stdin);
            if (*Buff != '\n') {
                (void) strcpy(packname, Buff);
                if ((p = strchr(packname, '\n')) != NULL)
                    *p = '\0';
            }
        } while (yes_or_no("y","%s [%s] ? [y] ",linehdr,packname) == NO);

        do {
            (void) printf("Enter Volume:  ");
            (void) fgets(vname, sizeof vname, stdin);
            if ((p = strchr(vname, '\n')) != NULL)
                *p = '\0';

            /*
            ** Check to assure that we have digits... 
            */
            for(p = vname; *p; ++p)  {
               if (!isdigit(*p)) {
                    vname[0] = '\0';
                    (void) fprintf(errfp,"\007digits please...\n");
                    break;
               }
            }
        } while(vname[0] == '\0');

        do {
            (void) printf("Enter Issue range:   ");
            (void) fgets(issue_num, sizeof issue_num, stdin);
            if ((p = strchr(issue_num, '\n')) != NULL)
                    *p = '\0';
            /*
            ** Check to assure that we have digits, '-' or ','.
            */
            for(p = issue_num; *p; ++p)  {
               if (!isdigit(*p) && (*p != '-') && (*p != ',')) {
                    issue_num[0] = '\0';
                    (void) fprintf(errfp,"\007digits, '-' or ',' please...\n");
                    break;
               }
            }
        } while(issue_num[0] == '\0');
       

        (void) sprintf(poststor,"%s: Volume %s, Issue %s",
			packname, vname, issue_num);

        if (yes_or_no("y","%s %s ? [y] ",linehdr,poststor) == YES)
            return;
    }
}

void post_file()
{
    FILE *F;

    if (Repost != TRUE) {
       /* update our sequence number. */
       if ((Issuefp = fopen(Issuepath, "w")) == NULL)
           perror(Issuepath);
       else {
           (void)fprintf(Issuefp,"%d %d %d\n",post.volume,post.issue,post.info);
           (void)fclose(Issuefp);
       }  
    }
 
    (void) printf(".issue updated.\n");
 
    /* Post it. */
    (void) printf("Posting...\n");
    (void) sprintf(Buff, "exec %s -h <%s", INEWS, Tempfile);
    (void) system(Buff);
       
    /* Update the log. */
    (void)sprintf(Buff, "%s/.index", NG_HOME);

    F = efopen(Buff, "a");

    /* NEED TO HAVE format() CAPABILITIES ADDED INSTEAD OF THIS JUNK... */

    (void) fprintf(F, "%s%s\t", Archbuff, strlen(Archbuff) < 8 ? "\t" : "");

    if (PostingType == INFO)
        (void) fprintf(F, "v%2.2dINF%1.1d:  %s",
                       post.volume,post.info,post.subject);
    else {
        (void) fprintf(F, "v%2.2di%3.3d:  %s",
                       post.volume,post.issue,post.subject);
        if (PostingType == PATCH) {
            if (post.lastpart > 1) /* Is this a multipart patch posting */
                (void) fprintf(F, ", Patch%2.2d%c/%d", 
                               post.patch_num, post.part+96, post.lastpart);
            else
                (void) fprintf(F, ", Patch%2.2d", post.patch_num);
        }
        else
            (void) fprintf(F, ", Part%2.2d/%2.2d", post.part, post.lastpart);
    }

    (void) fprintf(F, "\n");

    /*
    ** get the checksum info line from the issue
    */
    if (GetChecksum(Tempfile, Buff))
        (void) fprintf(F, "\t%s%s\n", CHECKSUMHDR, Buff);
    else
        (void) fprintf(F, "\tNo checksum\n");
    
    (void)fclose(F);
}

/*
**  Call $EDITOR for last-minute cleanups.
*/
void edit_posting(tempfile)
char *tempfile;
{
    register char *p;

    if ((p = getenv("EDITOR")) == NULL)
        p = EDITOR;

    for ( ; ; ) {
        (void)sprintf(Buff, "exec %s %s", p, tempfile);
        (void)system(Buff);

        do {
            (void) printf("(P)roceed, (E)dit or (Q)uit (!)shell [P]?  ");
            (void) fgets(Buff, sizeof Buff, stdin);
            if (Buff[0] == '!')
                (void) system(SHELL);

        } while (Buff[0] == '!');
      
        if (Buff[0] == '\n' || Buff[0] == 'p' || Buff[0] == 'P') {
            /* NEED TO ADD AN ERROR CHECKING ROUTINE HERE   */ 
            /* IF THERE IS A PROBLEM, INFORMA THEM AND THEN */
            /* THROW THEM BACK INTO THE EDITOR...           */
            return;
        }
        if (Buff[0] == 'Q' || Buff[0] == 'q') {
           (void) fprintf(errfp,"Terminating. Leaving tempfile %s\n",Tempfile);
           exit (1); 
        }
    }
}

void format_posting()
{
    char *generate_references();
    char *expires();

    /*
    ** MAIN NEWS AND RFC822 HEADERS
    */

    /*
    ** From:
    **
    ** Assure that the From: line is set to the 
    ** person who is submitting the article to 
    ** the newsgroup.
    */
    (void) fprintf(Out, "From: %s\n", post.submitter);

    /*
    ** Subject:
    **
    ** Write out the subject line depending on the type of the posting..
    **
    ** Source Posting:
    **    Subject:  v16i061: lc - Catalog Files In Columns, Patch01
    **
    ** Reposted Source Posting:
    **    Subject:  REPOST: v16i061: lc - Catalog Files In Columns, Patch01
    **
    ** INFormational Posting:
    **    Subject:  v16INF1:  Introduction to comp.sources.misc
    */

    if (Repost == TRUE)
        (void) fprintf(Out,"Subject:  REPOST: v%2.2di%3.3d:  %s",
                       post.volume,post.issue,post.subject);
    else if (PostingType == SOURCE || PostingType == PATCH)
        (void) fprintf(Out,"Subject:  v%2.2di%3.3d:  %s",
                       post.volume,post.issue,post.subject);
    else if (PostingType == INFO) 
        (void) fprintf(Out,"Subject:  v%2.2dINF%1.1d:  %s\n",
                       post.volume,post.info,post.subject);

    /*
    ** Add the appropriate Part??/?? or Patch?? to the end of the
    ** Subject: line.
    **
    ** The general rules are as follows... 
    **    Patch Posting  -  Patch posting and Patch number attached.
    **    Source Posting -  Source posting and Part??/?? attached.
    **    INFO Posting   - NO part number or patch number attached.
    */             

    if (PostingType == PATCH) {
        /*
        ** We have a patch here. Need to check to see if it is a multipart
        ** patch or a individual patch. For multipart patches use a 
        ** Patch01a/5 format where the Patch01 is the software patch number,
        ** the 'a' is the part indicator of the multipart patch and the /5
        ** indicates the total number of parts in this multipart patch.
        **
        ** Subject: v99i997:  package - Neat piece of software, Patch01a/5
        ** Archive-name: package/patch01a
        ** 
        ** Subject: v99i998:  package - Neat piece of software, Patch01b/5
        ** Archive-name: package/patch01b
        ** 
        ** Alternatively, could do: "..., Patch01a/d", but using a digit
        ** rather than letter to indicate the number of "parts" is clearer.
        **
        ** Warning:
        ** There are only 26 letters in the alphabet. This breaks on the 
        ** 27th part of a multipart patch posting...
        */
        if (post.lastpart > 1)  /* Is it a multipart patch posting */
            (void) fprintf(Out, ", Patch%2.2d%c/%d\n",
                          post.patch_num, post.part+96, post.lastpart);
        else 
            (void) fprintf(Out, ", Patch%2.2d\n", post.patch_num);
    }
    else if (PostingType != INFO)
        (void) fprintf(Out, ", Part%2.2d/%2.2d\n", post.part, post.lastpart);

    /*
    ** Newsgroups:
    */
    
    (void) fprintf(Out, "Newsgroups: %s\n", post_to_newsgroup());

    /*
    ** Keywords:
    **
    ** Did the user supply keywords ?? If so, include them.
    */
    if (header.keywords[0] != '\0')
        (void) fprintf(Out, "Keywords: %s\n", header.keywords);

    /* 
    ** This section sets up the Message-ID: if the posting is an
    ** initial part of a multi-part posting.  If the posting is
    ** a secondary article within a multi-part posting, the References:
    ** line is used to aid threaded newsreaders so that the entire
    ** set is seen as a single posting.
    **
    ** Set up on the first article ...
    */

    if (post.lastpart > 1) {
        if (((post.part == 0) || (post.part == 1)) && First_ref)  {
            First_ref = 0;
            /*
            ** Put out the header on the initial part...
            */
            (void) fprintf(Out, "Message-ID: %s\n", generate_references());
        }
        else {
            /*
            ** Put out the header on the initial part...
            */
            (void) fprintf(Out, "References: %s\n", post.references);
        }
    }

    /*
    ** Followup-To:
    */

    (void) fprintf(Out, "Followup-To: %s\n", FOLLOWUPTO);

    /*
    ** Organization:
    **
    ** Did the user supply an organization line ?? If so, include it.
    */

    if (header.organization[0] != '\0')
        (void) fprintf(Out, "Organization: %s\n", header.organization);

    /*
    ** Followup-To:
    **
    ** Did the user supply a Followup-To: line ?? If so, include it.
    */

    if (header.followupto[0] != '\0')
        (void) fprintf(Out, "Followup-To: %s\n", header.followupto);

    /*
    ** Reply-To:
    **
    ** Did the user supply a Reply-To: line ?? If so, include it.
    */

    if (header.replyto[0] != '\0')
        (void) fprintf(Out, "Reply-To: %s\n", header.replyto);

    /*
    ** Summary:
    **
    ** Did the user supply a Summary: line ?? If so, include it.
    */

    if (header.summary[0] != '\0')
        (void) fprintf(Out, "Summary: %s\n", header.summary);

    /*
    ** Expires:
    **
    ** Is this an INF posting ?  
    */

    if (PostingType == INFO) {
        (void) fprintf(Out, "Expires: %s", expires(DAYS_TILL_EXPIRE));
    }

    /*
    ** Approved:
    */
    (void) fprintf(Out, "Approved: %s\n", APPROVED);
 
    /*
    ** Separate the RFC822 headers from the auxiliary headers
    */
    (void) fprintf(Out, "\n");

    /*
    ** AUXILIARY HEADERS
    */

    /*
    ** Submitted-by: 
    */

    (void) fprintf(Out, "Submitted-by: %s\n", post.submitter);

    /*
    ** Posting-number:
    */

    if (PostingType == INFO)
        (void) fprintf(Out, "Posting-number: Volume %d, Info %d\n", 
                       post.volume, post.info);
    else
        (void) fprintf(Out, "Posting-number: Volume %d, Issue %d\n", 
                       post.volume, post.issue);
    /*
    ** Archive-name:
    */

    if (PostingType == INFO)
        (void)strcpy(Archbuff, post.archname);

    else if (PostingType == PATCH) {  /* Is this a patch posting ? */
       if (post.lastpart > 1)  /* Is it a multipart patch posting */
           (void)sprintf(Archbuff, "%s/patch%2.2d%c", 
                          post.archname, post.patch_num, post.part+96);
       else 
           (void)sprintf(Archbuff,"%s/patch%2.2d",post.archname,post.patch_num);
    }
    else
        (void)sprintf(Archbuff, "%s/part%2.2d", post.archname, post.part);

    (void) fprintf(Out, "Archive-name: %s\n", Archbuff);

    /*
    ** Environment:
    ** 
    ** The operational/compiliation requirements needed to run the issue..
    **
    ** This may entail a keyword lookup for base keywords with automatic
    ** addition to the table file when the moderator has to enter a new 
    ** keyword never before used.  In this way the Keword table file will
    ** always contain every keyword ever used by the moderator.
    */

    if (post.env[0] != '\0')
        (void) fprintf(Out, "Environment: %s\n", post.env);
   
    /*
    ** Patch-To:
    */

    if (post.patchto[0] != '\0')
        (void) fprintf(Out, "Patch-To: %s\n", post.patchto);

    /*
    ** Supersedes:
    */
    if (post.supersedes[0] != '\0')
        (void) fprintf(Out, "Supersedes: %s\n", post.supersedes);

    if (PostingType == INFO) {
        if (header.last_modified[0] != '\0')
            (void) fprintf(Out, "Last-modified: %s\n", header.last_modified);
    }

    /*
    ** ADD ADDITIONAL AUXILIARY HEADER SUPPORT HERE
    */ 

    /*
    ** Separate the auxiliary headers from the body of the text.
    */
    (void) fprintf(Out, "\n");
}

void get_parts()
{
    char *p;

    /*
    ** Has the user specified the info on the command line...
    */
    if (!Interactive || PostingType == INFO)
        return;

    /*
    ** check for potential multiple parts
    */
    do {
        if (PostingType == PATCH)
            (void) printf("Patch Number [%d]:   ", post.patch_num);
        else
            (void) printf("Part Number [%d]:   ", post.part);

        (void)fgets(Buff, sizeof Buff, stdin);
        if (*Buff != '\n') {
            if ((p = strchr(Buff,'\n')) != NULL)
                 *p = '\0';
            /*
            ** Check to assure that we have digits... 
            */
            for(p = Buff; *p; ++p)  {
                if (!isdigit(*p) && *p != '-') {
                    Buff[0] = '\0';
                    (void) fprintf(errfp,"\007digits please...\n");
                    break;
                }
            }
            if (*Buff != '\0') {
                if (PostingType == PATCH)
                    post.patch_num = atoi(Buff);
                else
                    post.part = atoi(Buff);
            }
        }
    } while (*Buff == '\0');
    
    /*
    ** Check to see if this issue is to be a Patch posting.
    ** If so, get the part number of the potential set of
    ** patch parts...
    */
    if (PostingType == PATCH) {
        do {
           (void) printf("Patch%2.2d - Enter Patch Part number [%d]:   ", 
                         post.patch_num, post.part);
           (void)fgets(Buff, sizeof Buff, stdin);
           if (*Buff != '\n') {
               if ((p = strchr(Buff,'\n')) != NULL)
                   *p = '\0';
               /*
               ** Check to assure that we have digits... 
               */
               for(p = Buff; *p; ++p)  {
                   if (!isdigit(*p)) {
                       Buff[0] = '\0';
                       (void) fprintf(errfp,"\007digits please...\n");
                       break;
                    }
                }
                if (*Buff != '\0') 
                    post.part = atoi(Buff);

            }
        } while (*Buff == '\0');
    }
       
    /*
    ** Time to check how many parts there are in the complete submission. 
    */
    if (post.lastpart == 1) {
        do {
            (void) printf("Total number of parts [%d]:   ",post.lastpart);
            (void)fgets(Buff, sizeof Buff, stdin);
            if (*Buff != '\n') {
                if ((p = strchr(Buff,'\n')) != NULL)
                    *p = '\0';
                /*
                ** Check to assure that we have digits... 
                */
                for(p = Buff; *p; ++p)  {
                    if (!isdigit(*p)) {
                        Buff[0] = '\0';
                        (void) fprintf(errfp,"\007digits please...\n");
                        break;
                    }
                }
                if (*Buff != '\0') 
                    post.lastpart = atoi(Buff);
            } 
        } while (*Buff == '\0');
    }
}

void submitted_by()
{
    char *p;
    char addr[256];
    char fname[256];

    /*
    ** Check to assure that the Submitted-by: info is available if
    ** the posting is being done from the command line.
    */
    if (!Interactive && (post.author_name[0] != '\0' && post.author_signon[0] != '\0')) {
        (void) sprintf(post.submitter, "%s <%s>", post.author_name, post.author_signon);
        return; /* We're good..*/
    }

    /*
    ** Check to assure that there is a Submitted-by entry.
    */
    if (post.submitter[0] != '\0') {
        /* 
        ** check to assure that the user's address is in 
        ** '@' notation or '!' notation (as a last resort).
        ** There should be a .domain specified, if the '@' 
	** notation is used even if it it just the .UUCP domain.
        */
        if (((strchr(post.submitter,'@') != NULL) && 
            (strchr(post.submitter,'.') != NULL)) ||
            (strchr(post.submitter,'!') != NULL)) {
            if (yes_or_no("y","Use Submitter's address: [%s] ? [y] ",
                                                     post.submitter) == YES)
                return;
        }
    }

    /* 
    ** Let's see if the user supplied one in the posting that was
    ** sent to the moderator... or is this a multiple part continuation...
    */
    if (header.submitted_by[0] != '\0') {
        if (yes_or_no("y","Use Submitter's address: [%s] ? [y] ",
                                            header.submitted_by) == YES) {
            (void) strcpy(post.submitter, header.submitted_by);
            return;
        }
    }


    /* 
    ** Lets see if the user supplied enough information to construct one...
    */
    if (post.author_name[0] != '\0' && post.author_signon[0] != '\0') {
        (void) sprintf(post.submitter, "%s (%s)", post.author_signon, post.author_name);
        if (yes_or_no("y","Use Submitter's address: [%s] ? [y] ",
                                                 post.submitter) == YES)
            return;
    }

    /*
    ** Let's prompt for it..
    */

    (void) printf("From: %s\n", header.from);

    /* NEED TO ALLOW FOR CUT AND PASTE FROM THE WINDOWS HERE... */

    /* NEED TO HAVE THE PROGRAM CORRECT THE ADDRESS IF IT CAN BELOW */

    for(;;) {
        addr[0] = '\0';
        fname[0] = '\0';

        (void) printf("Enter Submitter's fullname:  ");
        (void) fgets(fname, sizeof fname, stdin);
        if ((p = strchr(fname, '\n')) != NULL)
            *p = '\0';

        for(;;) {
            (void) printf("Enter Submitter's address:   ");
            (void) fgets(addr, sizeof addr, stdin);
            if ((p = strchr(addr, '\n')) != NULL)
                *p = '\0';

            (void) sprintf(post.submitter,"%s (%s)", addr, fname);

            if (((strchr(post.submitter,'@') == NULL) || 
                 (strchr(post.submitter,'.') == NULL)) &&
                 (strchr(post.submitter,'!') == NULL)) {
                (void) printf("\007\007");
                (void) printf("Invalid address specified ([@,.],!).\n");
            }
            else
                break;
        }

        if (yes_or_no("y","Submitted-by: %s ? [y] ",post.submitter) == YES) {
            if (((strchr(post.submitter,'@') != NULL) && 
                 (strchr(post.submitter,'.') != NULL)) ||
                 (strchr(post.submitter,'!') != NULL)) 
                return;
            (void) printf("\007\007");
            (void) printf("Invalid address specified ([@,.],!).\n");
        }
    }
}

void archivename_line()
{
    char *p;

    /*
    ** Was the archive name passed on the command line ??
    */
    if (!Interactive && post.archname[0] != '\0')
        return;  /*Yep!*/


    if (PostingType != INFO) {
        /*
        ** Is this a multipart posting...
        */
        if (post.archname[0] != '\0') {
            if (yes_or_no("y","Use Archive-name: [%s] ? [y] ",post.archname) == YES) 
                return;
        }
    }

    /*
    ** Time to get the Archive-name: line to be used in the article.
    ** If the user supplied one, check to see if the moderator
    ** wants to use the supplied one. If not, allow the moderator
    ** to supply a new Subject: line to be used.
    **
    ** Lets see if the user supplied one in the posting that was
    ** sent to the moderator...
    */
    if (header.archive_name[0] != '\0') {
        /*
        ** Get rid of any part number..
        */
        if ((p = strchr(header.archive_name,'/')) != NULL)
            *p = '\0';
        if (yes_or_no("y","Use Archive-name: [%s] ? [y] ",header.archive_name) == YES) {
            (void) strcpy(post.archname, header.archive_name);
            return;
        }
    }

    for(;;) {
        post.archname[0] = '\0';

        do {
            (void) printf("Enter Archive-name:   ");
            (void) fgets(post.archname, sizeof post.archname, stdin);
            if ((p = strchr(post.archname, '\n')) != NULL)
                *p = '\0';
            /*
            ** Check to assure that the archive-name is 12 characters
            ** or less. This is so archive administrators can compress
            ** the archive members if archive-name archiving is done.
            */
            if (strlen(post.archname) > 12) {
               (void) printf("\007Archive-name: too long. Re-enter...\n");
               post.archname[0] = '\0';
            }
        } while (post.archname[0] == '\0');

        if (yes_or_no("y","Archive-name: %s ? [y] ",post.archname) == YES) 
            return;
    }
}

void subject_line()
{
    char *p;

    /*
    ** Did the moderator pass the subject on the command line ??
    */
    if (!Interactive && post.subject[0] != '\0')
        return; /*Yep*/

    if (PostingType != INFO) {
        /*
        ** Time to get the Subject line to be used in the article.
        ** First see if this is a series based article posting.
        ** As the moderator if they wish to use the last subject...
        */
        if (*post.subject) {
            (void) printf("Use Subject: %s [y] ",post.subject);
            (void) fgets(Buff, sizeof Buff, stdin);
            if (*Buff == 'y' || *Buff == '\n') 
                return;
        }
    }

    /*
    ** Nope, either first time or the moderator does not want
    ** to use the previous subject line.
    ** If the user supplied one, check to see if the moderator
    ** wants to use the supplied one. If not, allow the moderator
    ** to supply a new Subject: line to be used.
    */

    if (*header.subject) {
        if (Reinit) {
           (void) strcpy(Buff, header.subject);
           /* Strip part info */
           if ((p = strrchr(Buff,',')) != NULL)
                *p = '\0';
           /* Strip package info */
           if ((p = strchr(Buff,'-')) != NULL)
               p += 2;
           (void) strcpy(header.subject, p);
        }
        (void) printf("Original subject: '%s'\n", header.subject);
        (void) printf("Do you wish to use the supplied subject line ? [y] ");
        (void) fgets(Buff, sizeof Buff, stdin);
        if (*Buff == 'y' || *Buff == '\n') {
            if (PostingType == SOURCE || PostingType == PATCH)
                (void) sprintf(post.subject, "%s - %s", post.archname, header.subject);
            else
                (void) sprintf(post.subject, "%s", header.subject);
            return;
        }
    }
    else 
        (void) printf("Submitter did not supply a Subject: line..\n");

    do {
        post.subject[0] = '\0';
        do {
            (void) printf("Enter Subject:   ");
            (void) fgets(Buff, sizeof Buff, stdin);
            (void) sprintf(post.subject, "%s - %s", post.archname, Buff);
            if ((p = strchr(post.subject, '\n')) != NULL)
                *p = '\0';
        } while (post.subject[0] == '\0');
        (void) printf("Subject: %s ? [y] ", post.subject);
        (void) fgets(Buff, sizeof Buff, stdin);
    } while (*Buff != 'y' && *Buff != '\n');
}

void update_sequence_numbers()
{
    char *p;

    if (Interactive) {
        if (Repost == TRUE) {
            do {
                (void) printf("Issue number of article to repost:  ");
                (void) fgets(Buff, sizeof Buff, stdin);
                if ((p = strchr(Buff, '\n')) != NULL)
                    *p = '\0';
                post.issue = atoi(Buff);
                (void) printf("Issue: %d ? [y] ", post.issue);
                (void) fgets(Buff, sizeof Buff, stdin);
            } while (*Buff != 'y' && *Buff != '\n');
        } 
        else if (PostingType == INFO)
            post.info++;
        else
            post.issue++;
    }
		
    if (Interactive)  
        (void) fprintf(logfp,"Preparing %s volume %d, issue %d, Info %d.\n",
                       NEWSGROUP, post.volume, post.issue, post.info);
}

void read_issue_info()
{
    (void) sprintf(Issuepath, "%s/.issue", NG_HOME);
 
    Issuefp = efopen(Issuepath,"r");
    if (fgets(Buff, sizeof Buff, Issuefp) == NULL) {
	(void) fprintf(stderr,"Unable to read %s info...bye\n", Issuepath);
        exit(1);
    }
    if (sscanf(Buff, "%d %d %d", &post.volume, &post.issue, &post.info) != 3) {
	(void) fprintf(stderr,"%s format error...bye\n", Issuepath);
        exit(1);
    }
    (void)fclose(Issuefp);
}

void get_article_type()
{
    char default_type[10];
    char *cp;
    int last_reposted;

    if (!Interactive)
        return;

    last_reposted = Repost;
    Repost = FALSE;
    
    for(;;) {
        (void) printf("\nType of posting, (I)nfo, (R)epost, (P)atch or (S)ource ");
        switch(Last_post_type) {
            case INFO    : (void) strcpy(default_type,"i"); break;
            case SOURCE  : (void) strcpy(default_type,"s"); break;
            case PATCH   : (void) strcpy(default_type,"p"); break;
            default      : (void) strcpy(default_type,"s"); break;
        }
    
        /*
        ** Determine if the last was a Repost. If so add the 'r'
        ** To the end of the choice defaults.  
        */
        if (last_reposted == TRUE) 
            (void) strcat(default_type,"r"); 

        (void) printf("[%s] ", default_type);
        (void) fgets(Buff, sizeof Buff, stdin);

        /*
        ** Let's get the newline out of the way...
        */
        if (*Buff == '\n')
           (void) strcpy(Buff, default_type);

        /* get rid of newline potentially left by fgets */

        if ((cp = strchr(Buff,'\n')) != NULL)
             *cp = '\0';

         cp = str_lower(Buff);
        
         while (*cp) {
           switch(*cp++) {
               case 'i' : PostingType = INFO;   break;
               case 's' : PostingType = SOURCE; break;
               case 'p' : PostingType = PATCH;  break;
               case 'r' : Repost = TRUE;        break;
               default  : PostingType = NONE;   
                          (void) printf("\007Enter a valid selection...\n");
                        break;
           }
        }
        if (PostingType == NONE)
           continue;

        Last_post_type = PostingType;
        return;
    }
}

char *post_to_newsgroup()
{
    /*  
    ** PROBABLY NEED TO ADD THE ABILITY TO SPECIFY THE NEWSGROUPS TO POST
    ** TO ON THE COMMAND LINE. FOR NOW, IF WE ARE NON-INTERACTIVE, JUST
    ** RETURN THE NEWSGROUP DEFINE
    */

    if (PostingType != INFO || !Interactive)
       return(NEWSGROUP);

    /*
    ** for inf postings, I may want to post it to c.s.d and
    ** comp.archives...  I need to choose...
    */
    (void) printf("Post to %s (O)nly or %s (A)ll ? [O] ",
                  NEWSGROUP, INFO_NEWSGROUPS);

    (void) fgets(Buff, sizeof Buff, stdin);
    if (*Buff == 'o' || *Buff == '\n' || *Buff == 'O')
        return(NEWSGROUP);
    else
        return(INFO_NEWSGROUPS);
    
    /*
    ** THIS REALLY SHOULD CHECK TO ASSURE THAT I ENTER A VALID ANSWER.
    ** MAYBE IT SHOULD ALSO ALLOW ME TO SPECIFY A NEWSGROUP LIST TO
    ** POST TO...
    */
}

/*
**  Scan article for MD4 checksum header.
*/
int GetChecksum(Name, buff)
char  *Name;
char  *buff;
{
    FILE *sumfile;
    char line[256];
    char *p;
    char *strchr();
    int len;

    sumfile = efopen(Name, "r");
    
    len = sizeof(CHECKSUMHDR) -1;
 
    /* Read up to the headers. */

    while (fgets(line, sizeof line, sumfile) && line[0] != '\n') {
        if ((p = strchr(line, '\n')) != NULL)
            *p = '\0';
        if (strncmp(line, CHECKSUMHDR, len) == 0) {
            (void)strcpy(buff, &line[len]);
            (void)fclose(sumfile);
            return (TRUE);
        }
    }
    (void) fclose(sumfile);
    return(FALSE);
}

char *read_line(file)
FILE *file;
{
    char *rp;

    /*
    ** If first time through, fill s with read, else
    ** use line we previously read into buf2.
    */

    if (buf2[0] == '\0') {
       if (fgets(s,sizeof(s),file) == NULL) 
           return(NULL);
    }
    else 
       (void) strcpy(s,buf2);

    if (fgets(buf2,sizeof(buf2),file) == NULL) 
        return(NULL);

#ifdef DEBUG
(void) printf("S = %s",s);
(void) printf("buf2 = %s",buf2);
#endif

    /*
    ** If the next line starts with a whitespace, it is a continuation
    ** line and it must be concatinated to the previous line.
    */

    while(buf2[0] == ' ' || buf2[0] == '\t') {
       if ((rp = strchr(s,'\n')) != NULL)
            *rp = '\0';
       rp = buf2;
       while(*rp && (*rp == ' ' || *rp == '\t')) 
           ++rp;
       (void) strcat(s, rp);
       if (fgets(buf2,sizeof(buf2),file) == NULL) 
           return(NULL);
#ifdef DEBUG
(void) printf("Concatenate S = [%s]\n",s);
(void) printf("Reread buf2 = %s\n",buf2);
#endif
    }
    return (s);
}

FILE *efopen(file,mode)       
char *file, *mode;
{
   FILE *fp;

   if ((fp = fopen (file, mode)) == NULL) {
      (void) fprintf (errfp, "Can't open file %s\n", file);
      exit(1);
   }
   return (fp);
}

void print_val(fp,frmt,value)
FILE *fp;
char *frmt;
char *value;
{
    if (*value)
        (void) fprintf(fp,frmt,value);
}

void dump_message(fname)
char *fname;
{
    print_val(logfp,"Message:              [%s]\n",fname);
    print_val(logfp,"   From:              [%s]\n",header.from);
    print_val(logfp,"   To:                [%s]\n",header.to);
    print_val(logfp,"   Subject:           [%s]\n",header.subject);
    print_val(logfp,"   Submitted-by:      [%s]\n",header.submitted_by);
    print_val(logfp,"   Archive-name:      [%s]\n",header.archive_name);
    print_val(logfp,"   Posting-number:    [%s]\n",header.posting_number);
    print_val(logfp,"   Patch-To:          [%s]\n",header.patch_to);
    print_val(logfp,"   Sender:            [%s]\n",header.sender);
    print_val(logfp,"   Reply-To:          [%s]\n",header.replyto);
    print_val(logfp,"   CC:                [%s]\n",header.cc);
    print_val(logfp,"   BCC:               [%s]\n",header.bcc);
    print_val(logfp,"   Comments:          [%s]\n",header.comments);
    print_val(logfp,"   In-Reply-To:       [%s]\n",header.inreplyto);
    print_val(logfp,"   Date:              [%s]\n",header.date);
    print_val(logfp,"   Message-ID:        [%s]\n",header.messageid);
    print_val(logfp,"   Keywords:          [%s]\n",header.keywords);
    print_val(logfp,"   Summary:           [%s]\n",header.summary);
    print_val(logfp,"   Xref:              [%s]\n",header.xref);
    print_val(logfp,"   Supersedes:        [%s]\n",header.supersedes);
    print_val(logfp,"   Return-path:       [%s]\n",header.returnpath);
    print_val(logfp,"   Organization:      [%s]\n",header.organization);
    print_val(logfp,"   Environment:       [%s]\n",header.environment);
    if (Verbose) {
        print_val(logfp,"   Received: from     [%s]\n",header.receivedfrom);
        print_val(logfp,"   Received: by       [%s]\n",header.receivedby);
        print_val(logfp,"   Received: via      [%s]\n",header.receivedvia);
        print_val(logfp,"   Received: with     [%s]\n",header.receivedwith);
        print_val(logfp,"   Received: id       [%s]\n",header.receivedid);
        print_val(logfp,"   Received: for      [%s]\n",header.receivedfor);
        print_val(logfp,"   Resent-Reply-To:   [%s]\n",header.resentreplyto);
        print_val(logfp,"   Resent-From:       [%s]\n",header.resentfrom);
        print_val(logfp,"   Resent-Sender:     [%s]\n",header.resentsender);
        print_val(logfp,"   Resent-Date:       [%s]\n",header.resentdate);
        print_val(logfp,"   Resent-To:         [%s]\n",header.resentto);
        print_val(logfp,"   Resent-Message-ID: [%s]\n",header.resentmessageid);
        print_val(logfp,"   Resent-cc:         [%s]\n",header.resentcc);
        print_val(logfp,"   Resent-bcc:        [%s]\n",header.resentbcc);
        print_val(logfp,"   References:        [%s]\n",header.references);
        print_val(logfp,"   Encrypted:         [%s]\n",header.encrypted);
        print_val(logfp,"   Followup-To:       [%s]\n",header.followupto);
    }
}

char *str_lower(str)
char *str;
{
    char *xp;

    xp = str;

    while (*xp) {
        if (isupper(*xp)) 
            *xp = tolower(*xp);
        ++xp;
    }
    return (str);
}

/*
** its: Returns TRUE if the string s1 starts off the lowbuf line.
** (This routine is not case sensitive.)
*/

int its(s1)
register char *s1;
{
    int strlen();
    int strncmp();

    (void) str_lower(s1);

    if (strncmp(lowbuf,s1,strlen(s1)) == 0)
        return(TRUE);
    return(FALSE);
}

int line_type()
{
    if (its("Date: "))
        return DATE;
    if (its("From: "))
        return FROM;
    if (its("Subject: "))
        return SUBJECT;
    if (its("Sender: "))
        return SENDER;
    if (its("Reply-To: "))
        return REPLY_TO;
    if (its("To: "))
        return TO;
    if (its("Cc: "))
        return CC;
    if (its("Bcc: "))
        return BCC;
    if (its("Comments: "))
        return COMMENTS;
    if (its("In-Reply-To: "))
        return IN_REPLY_TO;
    if (its("Message-ID: "))
        return MESSAGEID;
    if (its("Return-Path: "))
        return RETURNPATH;
    if (its("received: from"))
        return RECEIVEDFROM;
    if (its("received: by"))
        return RECEIVEDBY;
    if (its("received: via"))
        return RECEIVEDVIA;
    if (its("received: with"))
        return RECEIVEDWITH;
    if (its("received: id"))
        return RECEIVEDID;
    if (its("received: for "))
        return RECEIVEDFOR;
    if (its("resent-reply-to: "))
        return RESENTREPLYTO;
    if (its("resent-from: "))
        return RESENTFROM;
    if (its("resent-sender: "))
        return RESENTSENDER;
    if (its("resent-date: "))
        return RESENTDATE;
    if (its("resent-to: "))
        return RESENTTO;
    if (its("resent-message-id: "))
        return RESENTMESSAGEID;
    if (its("resent-cc: "))
        return RESENTCC;
    if (its("Followup-To: "))
        return FOLLOWUP_TO;
    if (its("resent-bcc: "))
        return RESENTBCC;
    if (its("references: "))
        return REFERENCES;
    if (its("keywords: "))
        return KEYWORDS;
    if (its("encrypted: "))
        return ENCRYPTED;
    if (its("Archive-name: "))
        return ARCH_NAME;
    if (its("Submitted-by: "))
        return SUBMITTED_BY;
    if (its("Summary: "))
        return SUMMARY;
    if (its("Supersedes: "))
        return SUPERSEDES;
    if (its("Organization: "))
        return ORGANIZATION;
    if (its("Last-modified: "))
        return LAST_MODIFIED;
    if (its("Environment: "))
        return ENVIRONMENT;
    if (its("Patch-To: "))
        return PATCH_TO;
    if (its("Xref: "))
        return XREF;

    return OTHER;
}

void data(hpfield, size,fldname)
char   *hpfield;
int    size;
char *fldname;
{
    register char *ptr;
    register char *p;
    char *strncpy();

    for (ptr = strchr(s, ':'); isspace(*++ptr); )
        ;
    if (*ptr != '\0') {
        (void) strncpy(hpfield, ptr, size - 1);
        /*
         * Strip trailing newlines, blanks, and tabs from hpfield.
         */
        for (p = hpfield; *p; ++p)
             ;
        while (--p >= hpfield && (*p == '\n' || *p == ' ' || *p == '\t'))
             ;
        *++p = '\0';
    }
    if (Debug && Verbose)
       (void) fprintf(logfp,"%s: %s\n",fldname, hpfield);
}

void init_header(hdr)
   struct header *hdr;
{
   hdr->date[0] = '\0';
   hdr->from[0] = '\0';
   hdr->subject[0] = '\0';
   hdr->sender[0] = '\0';
   hdr->replyto[0] = '\0';
   hdr->to[0] = '\0';
   hdr->cc[0] = '\0';
   hdr->bcc[0] = '\0';
   hdr->comments[0] = '\0';
   hdr->inreplyto[0] = '\0';
   hdr->messageid[0] = '\0';
   hdr->returnpath[0] = '\0';
   hdr->receivedfrom[0] = '\0';
   hdr->receivedby[0] = '\0';
   hdr->receivedvia[0] = '\0';
   hdr->receivedwith[0] = '\0';
   hdr->receivedid[0] = '\0';
   hdr->receivedfor[0] = '\0';
   hdr->resentreplyto[0] = '\0';
   hdr->resentfrom[0] = '\0';
   hdr->resentsender[0] = '\0';
   hdr->resentdate[0] = '\0';
   hdr->resentto[0] = '\0';
   hdr->resentmessageid[0] = '\0';
   hdr->resentcc[0] = '\0';
   hdr->resentbcc[0] = '\0';
   hdr->followupto[0] = '\0';
   hdr->references[0] = '\0';
   hdr->encrypted[0] = '\0';
   hdr->keywords[0] = '\0';
   hdr->summary[0] = '\0';           /* Summary:              */
   hdr->supersedes[0] = '\0';        /* Supersedes:           */
   hdr->xref[0] = '\0';              /* Xref:                 */
   hdr->submitted_by[0] = '\0';      /* Submitted_by:         */
   hdr->archive_name[0] = '\0';      /* Archive-name:         */
   hdr->organization[0] = '\0';      /* Organization:         */
   hdr->environment[0] = '\0';       /* Environment:          */
   hdr->patch_to[0] = '\0';          /* Patch-To:             */
}

void store_line(hdr)
   struct header *hdr;
{
   char *sp;
   char *dp;
   char wrkbuf[128];

   /* 
   ** Need to lower case the input line for comparison in its().
   ** the case does not matter in the keywords.
   */
   (void) strcpy(lowbuf,s);
   (void) str_lower(lowbuf);

   switch(line_type()) {
       case DATE:
            data(hdr->date, sizeof(hdr->date),"DATE");
            break;
       case FROM:
            data(hdr->from, sizeof(hdr->from),"FROM");
            /*
            ** Save the author's name and sign on if specified
            ** Can be in any of the following formats:
            **      kent@sparky.uucp
            **      kent@sparky.imd.sterling.com (Kent Landfield)
            **      Kent Landfield <kent@sparky.imd.sterling.com>
            **      uunet!sparky!kent (Kent Landfield)
            */
            if ((sp = strchr(s,':')) != NULL) {
                (void) strcpy(post.author_signon,(sp+2));
                /*
                ** Has a name been attached to the signon ?
                */
                if ((dp = strchr(post.author_signon,'<')) != NULL) {
                    *(dp-1) = '\0';
                    (void) strcpy(post.author_name, post.author_signon);
                    (void) strcpy(post.author_signon, ++dp);
                    /*
                    ** Save the name, removing the <>.
                    */
                    if ((dp = strchr(post.author_signon,'>')) != NULL)
                        *dp = '\0';
                }
                else if ((dp = strchr(post.author_signon,'(')) != NULL) {
                    *(dp-1) = '\0';
                    /*
                    ** Save the name, removing the ().
                    */
                    (void) strcpy(post.author_name, ++dp);
                    if ((dp = strchr(post.author_name,')')) != NULL)
                        *dp = '\0';
                }
                /*
                ** Assure that the signon is in '@' notation instead of 
                ** '!' notation.
                */
                if ((dp = strrchr(post.author_signon,'!')) != NULL) {
                    sp = dp+1;    /* point to the name */
                    *dp = '\0';
                    /*
                    ** check and see if there are more then one hop... 
                    */
                    if ((dp = strrchr(post.author_signon,'!')) != NULL) 
                       ++dp;    /* point to the hostname */
                    else 
                       dp = post.author_signon;

                    (void) sprintf(wrkbuf,"%s@%s", sp, dp);
                    (void) strcpy(post.author_signon,wrkbuf);
                }
            }
            break;
       case SUBJECT:
            data(hdr->subject, sizeof(hdr->subject),"SUBJECT");
            break;
        case SENDER:
            data(hdr->sender, sizeof(hdr->sender),"SENDER");
            break;
       case REPLY_TO:
            data(hdr->replyto, sizeof(hdr->replyto),"REPLY-TO");
            break;
       case TO:
            data(hdr->to, sizeof(hdr->to),"TO");
            break;
       case CC:
            data(hdr->cc, sizeof(hdr->cc),"CC");
            break;
       case BCC:
            data(hdr->bcc, sizeof(hdr->bcc),"BCC");
            break;
       case COMMENTS:
            data(hdr->comments, sizeof(hdr->comments),"COMMENTS");
            break;
       case IN_REPLY_TO:
            data(hdr->inreplyto, sizeof(hdr->inreplyto),"IN-REPLY-TO");
            break;
       case MESSAGEID:
            data(hdr->messageid, sizeof(hdr->messageid),"MESSAGE-ID");
            break;
       case RETURNPATH:
            data(hdr->returnpath, sizeof(hdr->returnpath),"RETURN-PATH");
            break;
       case RECEIVEDFROM:
            data(hdr->receivedfrom, sizeof(hdr->receivedfrom),"RECEIVED");
            break;
       case RECEIVEDBY:
            data(hdr->receivedby, sizeof(hdr->receivedby),"RECEIVED");
            break;
       case RECEIVEDVIA:
            data(hdr->receivedvia, sizeof(hdr->receivedvia),"RECEIVED");
            break;
       case RECEIVEDWITH:
            data(hdr->receivedwith, sizeof(hdr->receivedwith),"RECEIVED");
            break;
       case RECEIVEDID:
            data(hdr->receivedid, sizeof(hdr->receivedid),"RECEIVED");
            break;
       case RECEIVEDFOR:
            data(hdr->receivedfor, sizeof(hdr->receivedfor),"RECEIVED");
            break;
       case RESENTREPLYTO:
            data(hdr->resentreplyto, sizeof(hdr->resentreplyto),"RESENT-REPLY-TO");
            break;
       case RESENTFROM:
            data(hdr->resentfrom, sizeof(hdr->resentfrom),"RESENT-FROM");
            break;
       case RESENTSENDER:
            data(hdr->resentsender, sizeof(hdr->resentsender),"RESENT-SENDER");
            break;
       case RESENTDATE:
            data(hdr->resentdate, sizeof(hdr->resentdate),"RESENT-DATE");
            break;
       case RESENTTO:
            data(hdr->resentto, sizeof(hdr->resentto),"RESENT-TO");
            break;
       case RESENTMESSAGEID:
            data(hdr->resentmessageid, sizeof(hdr->resentmessageid),"RESENT-MESSAGE-ID");
            break;
       case RESENTCC:
            data(hdr->resentcc, sizeof(hdr->resentcc),"RESENT-CC");
            break;
       case RESENTBCC:
            data(hdr->resentbcc, sizeof(hdr->resentbcc),"RESENT-BCC");
            break;
       case REFERENCES:
            data(hdr->references, sizeof(hdr->references),"REFERENCES");
            break;
       case KEYWORDS:
            data(hdr->keywords, sizeof(hdr->keywords),"KEYWORDS");
            break;
       case ENCRYPTED:
            data(hdr->encrypted, sizeof(hdr->encrypted),"ENCRYPTED");
            break;
       case SUPERSEDES:
            data(hdr->supersedes, sizeof(hdr->supersedes),"SUPERSEDES");
            break;
       case XREF:
            data(hdr->xref, sizeof(hdr->xref),"XREF");
            break;
       case SUMMARY:
            data(hdr->summary, sizeof(hdr->summary),"SUMMARY");
            break;
       case ENVIRONMENT:
            data(hdr->environment,sizeof(hdr->environment),"ENVIRONMENT");
            break;
       case ORGANIZATION:
            data(hdr->organization,sizeof(hdr->organization),"ORGANIZATION");
            break;
       case PATCH_TO:
            data(hdr->patch_to,sizeof(hdr->patch_to),"PATCH_TO");
            break;
       case ARCH_NAME:
            data(hdr->archive_name,sizeof(hdr->archive_name),"ARCH_NAME");
            break;
       case POSTING_NUMBER:
            data(hdr->posting_number,sizeof(hdr->posting_number),"POSTING_NUMBER");
       case LAST_MODIFIED:
            data(hdr->last_modified,sizeof(hdr->last_modified),"LAST_MODIFIED");
            break;
       case SUBMITTED_BY:
            data(hdr->submitted_by,sizeof(hdr->submitted_by),"SUBMITTED_BY");
            /*
            ** Save the author's name and sign on if specified
            ** Can be in any of the following formats:
            **      kent@sparky.uucp
            **      kent@sparky.imd.sterling.com (Kent Landfield)
            **      Kent Landfield <kent@sparky.imd.sterling.com>
            */
            if ((sp = strchr(s,':')) != NULL) {
                (void) strcpy(post.author_signon,(sp+2));
                /*
                ** Has a name been attached to the signon ?
                */
                if ((dp = strchr(post.author_signon,'<')) != NULL) {
                    *(dp-1) = '\0';
                    (void) strcpy(post.author_name, post.author_signon);
                    (void) strcpy(post.author_signon, ++dp);
                    /*
                    ** Save the name, removing the <>.
                    */
                    if ((dp = strchr(post.author_signon,'>')) != NULL)
                        *dp = '\0';
                }
                else if ((dp = strchr(post.author_signon,'(')) != NULL) {
                    *(dp-1) = '\0';
                    /*
                    ** Save the name, removing the ().
                    */
                    (void) strcpy(post.author_name, ++dp);
                    if ((dp = strchr(post.author_name,')')) != NULL)
                        *dp = '\0';
                }
            }
            break;
    }
    return;
}

char *generate_references()
{
    long time();
    struct tm *localtime();
   
    char *cp;
    char dtstr[80];
    long clk;
    struct tm *n;
  
    clk = time((long *)0);
    n = localtime(&clk);
    (void) sprintf(dtstr,"%.2d%.2d%.2d", n->tm_hour, n->tm_min, n->tm_sec);
    (void) sprintf(post.references,"<%s-v%2.2di%3.3d=%-s.%s@%s>",
                   GRP_PREFIX, post.volume, post.issue, 
                   post.archname, dtstr, HOSTPOST);

    /*
    ** Assure that there is no white space in the header
    */
    while ((cp = strchr(post.references, ' ')) != NULL)
          *cp = '-';

    return(post.references);
}

/*
** Stream is a FILE * that changes throughout the program if necessary.
** When a new output file (stream) is opened, stream is set to point to
** that so this routine can be used without passing the stream.
*/

/*VARARGS0*/
int yes_or_no(va_alist)    /* yes_or_no(format,arg1,arg2,....argn) */
va_dcl
{
    va_list args;
    char *format;
    char *dflt;
    char msg_buf[BUFSIZ];
    char bf[BUFSIZ];
    
    va_start(args);
    
    dflt = va_arg(args, char *);     /* get the default answer */
    format = va_arg(args, char *);   /* get format to use      */
    (void) vsprintf(msg_buf,format,args);
    va_end(args);

    for (;;) {
        (void) printf("%s", msg_buf);
        (void) fgets(bf, sizeof bf, stdin);
        if (*bf == '\n')
            *bf = *dflt;
        switch(*bf) {
            case 'y': 
            case 'Y': return(YES);
            case 'n':
            case 'N': return(NO);
            default : (void) printf(" Please enter y or n ...\n");
        }
    }
}

void environment()
{
    char *p;

    if (!Interactive || PostingType == INFO)
        return;        

    /*
    ** Check if a similar line was used in the previous posting.
    */
    if (post.env[0] != '\0') {
        /* 
        ** See if the user wishes to use the 
        ** currently specified information.
        */
        if (yes_or_no("y","Use Environment [%s] ? [y] ", post.env) == YES)
            return;
    }

    /* 
    ** Let's see if the user supplied one in the posting that was
    ** sent to the moderator...
    */
    if (header.environment[0] != '\0') {
        if (yes_or_no("y","Use Environment [%s] ? [y] ", header.environment) == YES) {
            (void) strcpy(post.env, header.environment);
            return;
        }
    }

    /*
    ** Let's prompt for it..
    */

    for(;;) {
	post.env[0] = '\0';

        (void) printf("Enter Environment Keywords:  ");
        (void) fgets(post.env, sizeof post.env, stdin);
        if (*post.env == '\n') {
            if (yes_or_no("n","Skip Environment: ? [n] ") == YES) {
                post.env[0] = '\0';
                return;
            }
        }
        if ((p = strchr(post.env, '\n')) != NULL)
                *p = '\0';

        if (yes_or_no("y","Environment: %s ? [y] ", post.env) == YES)
            return;
    }
}

char *expires(interval)
int interval;
{
    long    clk, time();
    struct tm *now, *gmtime();
    static char datestr[40];

    /*  
    ** This is to be used to keep the informational postings 
    ** around longer than the normal default expiration.
    ** This way the INF postings are around when needed and
    ** not just when posted.
    */
    clk = time((long *)0);
    /*
    ** Add in the default day interval
    */
    if (interval > 0)
        clk += (interval * SEC_PER_DAY);

    now = gmtime(&clk); 

    (void) sprintf(datestr,"%s, %d %s %d %02d:%02d:%02d GMT\n",
           days[now->tm_wday], now->tm_mday, months[now->tm_mon],
           now->tm_year+1900,now->tm_hour,now->tm_min,now->tm_sec);
    return(datestr);
}
