(* Copyright 1997-1998 John D. Polstra.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgment:
 *      This product includes software developed by John D. Polstra.
 * 4. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: MD5Wr.m3,v 1.4 1998/04/22 02:38:22 jdp Exp $ *)

MODULE MD5Wr;

IMPORT MD5, Thread, Wr, WrClass;

REVEAL
  T = Public BRANDED OBJECT
    wr: Wr.T;
    closeChild: BOOLEAN;
    md5: MD5.T;
    signature: TEXT := NIL;
  OVERRIDES
    init := Init;
    getSignature := GetSignature;
    seek := Seek;
    flush := Flush;
    close := Close;
  END;

PROCEDURE Close(self: T)
  RAISES {Thread.Alerted, Wr.Failure} =
  BEGIN
    IF self.closeChild THEN
      Wr.Close(self.wr);
    END;
  END Close;

PROCEDURE Flush(self: T)
  RAISES {Thread.Alerted, Wr.Failure} =
  BEGIN
    Seek(self, self.cur);
    Wr.Flush(self.wr);
  END Flush;

PROCEDURE GetSignature(self: T): TEXT =
  BEGIN
    <* ASSERT self.closed *>
    IF self.signature = NIL THEN
      self.signature := self.md5.finish();
    END;
    RETURN self.signature;
  END GetSignature;

PROCEDURE Init(self: T;
               wr: Wr.T;
               md5: MD5.T := NIL;
	       closeChild: BOOLEAN := TRUE): T =
  BEGIN
    WrClass.Lock(self);
    TRY
      self.wr := wr;
      self.closeChild := closeChild;
      self.md5 := md5;

      IF self.md5 = NIL THEN self.md5 := MD5.New() END;
      self.buff := NEW(REF ARRAY OF CHAR, 8192);
      self.st := 0;  (* Always *)
      self.hi := NUMBER(self.buff^);
      self.seekable := FALSE;
      self.buffered := wr.buffered;
      self.closed := FALSE;
    FINALLY
      WrClass.Unlock(self);
    END;
    RETURN self;
  END Init;

PROCEDURE Seek(self: T; n: CARDINAL)
  RAISES {Thread.Alerted, Wr.Failure} =
  BEGIN
    <* ASSERT n = self.cur *>

    WITH data = SUBARRAY(self.buff^, 0, self.cur - self.lo) DO
      Wr.PutString(self.wr, data);
      self.md5.update(data);
    END;

    self.lo := self.cur;
    self.hi := self.lo + NUMBER(self.buff^);
  END Seek;

BEGIN
END MD5Wr.
