import java.awt.*;
import java.awt.event.*;
import java.awt.datatransfer.*;
import java.io.*;
import java.util.Hashtable;
import java.util.Enumeration;
import java.util.StringTokenizer;
import java.util.Vector;

/**
 This is the main list frame.  It is divided into three parts.  On the
 left is a list of keys, one for each record in the file being operated
 on.  At the bottom is a status and control panel.  Various status messages
 and some control radio buttons are shown here.  On the right is the
 entry panel.  The entry panel will display the most recently selected
 record.
 */
public class ListFrame extends Frame implements ItemListener, ActionListener, FocusListener, ClipboardOwner {
    static boolean m_Debug; // flag for debugging
    static int m_FrameBase = 0; // storage for frame base number
    int m_FrameNum; // number for this frame
    EditPanel m_Edit; // edit panel
    List m_Keys; // list of keys
    Hashtable m_Index; // hash to store index
    String m_FileName; // name of file being worked on
//    String m_LastRecord; // key pointing to the last record
    String m_Fields; // string to store field names
    String m_Types; // string to store list of types
    File m_File; // pointer to file, from which random access file is made
    RandomAccessFile m_RAFile; // pointer to file
    JavaBib m_Master; // reference to container for all lists
    BibTeXRecord m_ActiveRecord; // pointer to active record
    Menu m_FilesMenu; // menu for files that are open
    Menu m_CiteMenu; // menu which contains citation options
    Clipboard m_Clipboard; // clipboard for this application
    TextField m_CiteText; // field with citation text
    TextField m_Key; // key that is currently active
    SearchDialog m_Search; // memory of searches
    FindDialog m_Find; // memory of finds
    Checkbox m_Multiple; // check to switch to multiple

    static final int FILE_POS = 0;
    static final int LAST_POS = 1;
    static final int NEXT_POS = 2;

    ListFrame(JavaBib master) {
        super("new BibTeX file");
        // deal with frame number
        m_FrameNum = m_FrameBase;
        m_FrameBase++;
        // store reference to master
        m_Master = master;
        // set debug to true, by default
        m_Debug = true;
        // null active record
        m_ActiveRecord = null;
        //        m_LastRecord = null;
        setBackground(Color.lightGray);
        // set up the main components
        m_Edit = new EditPanel();
        m_Edit.addKeyFocusListener(this);
        m_Edit.addCrossRefFocusListener(this);
        m_Keys = new List();
        m_Keys.addItemListener(this);
        m_Keys.setBackground(Color.lightGray);
        // build the frame
        add(m_Keys, "West");
        add(m_Edit, "Center");
        add(makeControlPanel(), "South");
        setMenuBar(makeMenuBar());
        // to test, and set up cite options
        setCiteMenu("cite citet citep citeauthor");
        // set the size
        pack();
        // set the file name to null to start
        m_FileName = null;
        m_Fields = " ";
        m_Types = " ";
        // declare a new hashtable for storing pointers
        m_Index = new Hashtable();
        // declare a new clipboard, if needed
        if(m_Clipboard == null) m_Clipboard = getToolkit().getSystemClipboard();
    }

    /**
     Read entries from a stream and append them to the current file.  If keys
     need to be changed, display the record and ask the user to enter a new
     key.  The new key is entered in a dialog, but the record is displayed in
     the edit panel.
     The operation takes two passes, so the stream must be rewindable.  On the
     first pass, each record is read and the key is checked.  If the key
     has a match, ask for a change.  Any changes are stored in a hash.  On the
     second pass, records are read and added.  If they contain a crossref
     field for which the key has changed, this is updated.  Also, keys are
     updated as needed.  The records are then writen to the file.
     */
    public void addRecords(Reader in) {
        // open a file to dump the stuff coming in too, so that we don't
        // have to worry about trying to set a mark
        File temp = new File(m_File.getAbsolutePath() + ".temp");
        FileWriter fw = null;
        try {fw = new FileWriter(temp);}
        catch(IOException e) {
            debug(e.toString());
            return;}
        // storage for keys
        Hashtable keys = new Hashtable();
        // loop through, reading all records
        boolean flag = true;
        try {
            while(flag) {
                // read a record
                BibTeXRecord rec = new BibTeXRecord();
                rec.read(in);
                // set flag for next loop
                flag = !rec.isEmpty();
                // get the record name
                String recKey = rec.getKey();
                // check for duplicate key
                if(m_Index.containsKey(recKey)
                   || keys.containsKey(recKey)) {
                    // key is either in index or in keys read so far,
                    // need to change
                    m_Edit.setContents(rec);
                    m_Edit.validate();
                    // loop until new key is valid or have hit cancel
                    // note, on cancel, record is not included in write
                    boolean test = true;
                    while(test) {
                        // creating input dialogs is a pain
                        String[] q = new String[1];
                        q[0] = "Citation key:";
                        InputDialog id = new InputDialog(this, "Citation key exists in current index, enter a replacement.",
                                                         q, "Try Replacement", "Skip Record");
                        id.setVisible(true);
                        // on return, get answer to question
                        if(id.isNewData()) {
                            q = id.getNewData();
                            recKey = q[0];
                            test = m_Index.containsKey(recKey)
                                || keys.containsKey(recKey);}
                        else {
                            test = false;
                            recKey = null;}}}
                // record key is now valid or null
                if(recKey != null) {
                    keys.put(rec.getKey(), recKey);}
                // write the record to the file
                rec.write(fw);}}
        catch(IOException e) {
            debug(e.toString());}
        // close the file writer, as we have to turn it into a reader
        try {fw.close();}
        catch(IOException e) {debug(e.toString());}
        // ready for second pass
        // start by opening a reader to the file
        FileReader fr = null;
        try {fr = new FileReader(temp);}
        catch(FileNotFoundException e) {debug(e.toString());}
        // position the file pointer to be at the end of the file
        try {
            long loc = m_RAFile.length();
            m_RAFile.seek(loc);}
        catch(IOException e) {
            debug(e.toString());}
        // read each record, checking crossreference and presence
        // of key
        flag = true;
        try {
            while(flag) {
                // read a record
                BibTeXRecord rec = new BibTeXRecord();
                rec.read(fr);
                // set flag
                flag = !rec.isEmpty();
                // check for presence of key
                if(keys.containsKey(rec.getKey())) {
                    // in list to be included
                    // check for crossreference
                    if(rec.containsKey("CROSSREF")) {
                        // contains a crossref, check if it is in list
                        String xref = ((BibTeXField)rec.get("CROSSREF")).getBareText();
                        // test for key
                        if(!keys.containsKey(xref)) {
                            continue;}
                        // xref is here, need to fix the entry
                        BibTeXField bf = new BibTeXField("CROSSREF",
                                                         (String)keys.get(xref));
                        rec.put("CROSSREF", bf);}
                    // valid on all counts, write it.  We are going to
                    // write it directly, and build the index
                    // again after we are done
                    rec.setKey((String)keys.get(rec.getKey()));
                    // write the record
                    rec.write(m_RAFile);}}}
        catch(IOException e) {
            debug(e.toString());}
        // close things and delete the temporary file
        try {
            fr.close();
            temp.delete();}
        catch(IOException e) {
            debug(e.toString());}
        // all done, now rebuild the index
        m_Index = makeIndex(m_RAFile);
        // update the display
        setKeyList(m_Index);
    }

    /**
     Create a string with the selected entries.  This is a four pass operation.
     1) Find fields with crossreferences
     2) Add records with crossreference to string
     3) Add remaining records, including any required but not selected
     */
    public String getSelectedRecords() {
        // get selected records
        String[] items = m_Keys.getSelectedItems();
        if(items.length == 0) return null;
        // put selection into a vector
        Vector keyVec = new Vector();
        for(int i=0; i<items.length; i++) {
            keyVec.addElement(items[i]);}
        // create the string buffer for storing the stuff
        StringBuffer sb = new StringBuffer("");
        // read all the records
        Enumeration enum = keyVec.elements();
        while(enum.hasMoreElements()) {
            String key = (String)enum.nextElement();
            // read this record
            if(m_Index.containsKey(key)) {
                // get the file position
                long pos = (Long.valueOf(((String[])m_Index.get(key))[FILE_POS])).longValue();
                // get the record
                try {
                    m_RAFile.seek(pos);
                    BibTeXRecord rec = new BibTeXRecord();
                    rec.read(m_RAFile);
                    // check for a crossreference field
                    if(rec.containsKey("CROSSREF")) {
                        // yes, add record to string
                        sb.append(rec.toString());
                        // get the crossreference
                        String xref = ((BibTeXField)rec.get("CROSSREF")).getBareText();
                        // add it to vector, if necessary
                        if(!keyVec.contains(xref)) keyVec.addElement(xref);
                        // remove this entry from the vector
                        keyVec.remove(key);}}
                catch(IOException e) {
                    debug(e.toString());}}}
        // first pass done, now repeat to get the rest
        enum = keyVec.elements();
        while(enum.hasMoreElements()) {
            String key = (String)enum.nextElement();
            // read this record
            if(m_Index.containsKey(key)) {
                // get the file position
                long pos = (Long.valueOf(((String[])m_Index.get(key))[FILE_POS])).longValue();
                // get the record
                try {
                    m_RAFile.seek(pos);
                    BibTeXRecord rec = new BibTeXRecord();
                    rec.read(m_RAFile);
                    // now just add it
                    sb.append(rec.toString());}
                catch(IOException e) {
                    debug(e.toString());}}}
        // got them all, return the string
        return sb.toString();
    }

    /**
     Get the file name to which this list is attached
     */
    public String getFileName() {
        if(m_File != null) {
            return m_File.getAbsolutePath();}
        return null;
    }

    /**
     Update the window menu
     */
    public void setWindowMenu(Menu men) {
        getMenuBar().remove(m_FilesMenu);
        m_FilesMenu = men;
        getMenuBar().add(m_FilesMenu);
    }

    /**
     set the citation window
     */
    public void setCiteMenu(String options) {
        // remove the existing citation menu
        int count = getMenuBar().getMenuCount();
        Menu search = getMenuBar().getMenu(count - 1);
        getMenuBar().remove(m_FilesMenu);
        getMenuBar().remove(search);
        getMenuBar().remove(m_CiteMenu);
        // create the new cite menu
        m_CiteMenu = new Menu("Cite");
        // assume that options are split by spaces
        StringTokenizer st = new StringTokenizer(options);
        while(st.hasMoreTokens()) {
            MenuItem mi = new MenuItem("\\" + st.nextToken());
            mi.addActionListener(this);
            m_CiteMenu.add(mi);}
        // put the menu bar back together
        getMenuBar().add(m_CiteMenu);
        getMenuBar().add(search);
        getMenuBar().add(m_FilesMenu);
        // all done
    }

    /**
     Return the identifier number for this frame
     */
    public int getFrameNumber() {
        return m_FrameNum;}

    /**
     Create the menu bar
     */
    private MenuBar makeMenuBar() {
        MenuBar mb = new MenuBar();
        // file menu
        Menu m = new Menu("File");
        MenuItem mi = new MenuItem("Open");
        mi.addActionListener(m_Master);
        mi.addActionListener(this);
        m.add(mi);
        mi = new MenuItem("Close");
        mi.addActionListener(this);
        mi.addActionListener(m_Master);
        m.add(mi);
        mi = new MenuItem("-");
        m.add(mi);
        mi = new MenuItem("Compact");
        mi.addActionListener(this);
        m.add(mi);
        mi = new MenuItem("-");
        m.add(mi);
        mi = new MenuItem("Import");
        mi.addActionListener(this);
        m.add(mi);
        mi = new MenuItem("Export");
        mi.addActionListener(this);
        m.add(mi);
        mi = new MenuItem("-");
        m.add(mi);
        // continue file menu
        mi = new MenuItem("Exit");
        mi.addActionListener(this);
        mi.addActionListener(m_Master);
        m.add(mi);
        mb.add(m);
        // edit menu
        m = new Menu("Edit");
        mi = new MenuItem("New");
        mi.addActionListener(this);
        m.add(mi);
        mi = new MenuItem("Clear");
        mi.addActionListener(this);
        m.add(mi);
        mi = new MenuItem("Reset");
        mi.addActionListener(this);
        m.add(mi);
        mi = new MenuItem("Record");
        mi.addActionListener(this);
        m.add(mi);
        mi = new MenuItem("Cancel");
        mi.addActionListener(this);
        m.add(mi);
        mi = new MenuItem("-");
        m.add(mi);
        // records sub-menu
        Menu m2 = new Menu("Records");
        mi = new MenuItem("Cut");
        mi.addActionListener(this);
        mi.addActionListener(m_Master);
        m2.add(mi);
        mi = new MenuItem("Copy");
        mi.addActionListener(this);
        mi.addActionListener(m_Master);
        m2.add(mi);
        mi = new MenuItem("Paste");
        mi.addActionListener(this);
        mi.addActionListener(m_Master);
        m2.add(mi);
        mi = new MenuItem("Delete");
        mi.addActionListener(this);
        mi.addActionListener(m_Master);
        m2.add(mi);
        mi = new MenuItem("-");
        m2.add(mi);
        mi = new MenuItem("Select All");
        mi.addActionListener(this);
        m2.add(mi);
        mi = new MenuItem("Deselect All");
        mi.addActionListener(this);
        m2.add(mi);
        // continue edit menu
        mi = new MenuItem("-");
        m.add(mi);
        mi = new MenuItem("Options");
        mi.addActionListener(this);
        mi.addActionListener(m_Master);
        m.add(mi);
        mb.add(m);
        mb.add(m2);
        // cite menu
        m_CiteMenu = new Menu("Cite");
        mb.add(m_CiteMenu);
        // search menu
        m = new Menu("Search");
        mi = new MenuItem("Find");
        mi.addActionListener(this);
        m.add(mi);
        mi = new MenuItem("Search");
        mi.addActionListener(this);
        m.add(mi);
        mi = new MenuItem("Exit");
        mi.addActionListener(this);
        m.add(mi);
        mb.add(m);
        // window menu
        if(m_FilesMenu == null) m_FilesMenu = new Menu("Window");
        mb.add(m_FilesMenu);
        // add a help menu
        m = new Menu("Help");
        mi = new MenuItem("About");
        mi.addActionListener(this);
        m.add(mi);
        mb.setHelpMenu(m);
        return mb;
    }

    /**
     Construct the control panel
     */
    private Panel makeControlPanel() {
        Panel pan = new Panel(new BorderLayout());
        Panel leftpan = new Panel(new GridLayout(0,1));
        Panel rightpan = new Panel(new BorderLayout());
        Panel toppan = new Panel(new GridLayout());
        Panel botpan = new Panel(new BorderLayout());
        // citation pane
        m_CiteText = new TextField();
        m_CiteText.setEditable(false);
        m_CiteText.setText("\\cite{}");
        m_CiteText.setBackground(Color.lightGray);
        botpan.add(m_CiteText, "Center");
        // control pane
        Button but = new Button("New");
        but.addActionListener(this);
        toppan.add(but);
        but = new Button("Clear");
        but.addActionListener(this);
        toppan.add(but);
        but = new Button("Reset");
        but.addActionListener(this);
        toppan.add(but);
        but = new Button("Record");
        but.addActionListener(this);
        toppan.add(but);
        but = new Button("Cancel");
        but.addActionListener(this);
        toppan.add(but);
        toppan.add(new Label(" "));
        but = new Button("|<");
        but.addActionListener(this);
        toppan.add(but);
        but = new Button("<<");
        but.addActionListener(this);
        toppan.add(but);
        m_Key = new TextField();
        m_Key.setEditable(false);
        m_Key.setBackground(Color.lightGray);
        toppan.add(m_Key);
        but = new Button(">>");
        but.addActionListener(this);
        toppan.add(but);
        but = new Button(">|");
        but.addActionListener(this);
        toppan.add(but);
        // selection style
        CheckboxGroup cbg = new CheckboxGroup();
        Checkbox cb = new Checkbox("single", cbg, true);
        cb.addItemListener(this);
        leftpan.add(cb);
        m_Multiple = new Checkbox("multiple", cbg, false);
        m_Multiple.addItemListener(this);
        leftpan.add(m_Multiple);
        // put it together
        rightpan.add(toppan, "North");
        rightpan.add(botpan, "South");
        pan.add(leftpan, "West");
        pan.add(rightpan, "Center");
        return pan;
    }

    /**
     Compact a database.  Basic task is to remove all extraneous material.
     This extraneous material is mostly comment lines left behind when a
     record was deleted or moved.
     Proceedure:
     1) Close file.
     2) Rename file.
     3) Open new file with original name.
     4) Read record from old file and write to new file, replacing
     preamble with two blank lines
     5) Repeat (4) for all records
     6) Rebuild index
     */
    public void compact() {
        // clear the keys
        m_Keys.removeAll();
        // close file
        try{m_RAFile.close();}
        catch(IOException e) {debug(e.toString());}
        String path = m_File.getAbsolutePath();
        // rename file
        File temp = new File(path + ".temp");
//        debug("rename: " + m_File.renameTo(temp));
        // open the new file
//        File compact = new File(path);
        // open the streams
        RandomAccessFile fr = null;
        try{
            try{
                fr = new RandomAccessFile(m_File, "r");}
            catch(FileNotFoundException e) {debug(e.toString());}}
        catch(IOException e) {debug(e.toString());}
        RandomAccessFile fw = null;
        try{
            fw = new RandomAccessFile(temp, "rw");}
        catch(IOException e) {debug(e.toString());}
        // read, set, and write
        String EOLN = System.getProperty("line.separator");
        BibTeXRecord rec = new BibTeXRecord();
        try {
            fr.seek(0);
            while(fr.getFilePointer() < fr.length()) {
                rec.read(fr);
                rec.setPreamble(EOLN + EOLN);
//                debug(rec.toString());
                // to track progress, put name on list
                addItemByOrder(m_Keys, rec.getKey());
                rec.write(fw);}}
        catch(IOException e) {debug(e.toString());}
        // close the files
        try{
            fr.close();
            fw.close();}
        catch(IOException e) {debug(e.toString());}
        m_Keys.removeAll();
        // delete old file
        m_File.delete();
        temp.renameTo(m_File);
        try{m_RAFile = new RandomAccessFile(m_File, "rw");}
        catch(IOException e) {debug(e.toString());}
        // rebuild the index from the file
        openFile(path);
    }

    /**
     Load a BibTex file into a hash, with the hash key being the BibTeX
     key and all the information stored as a string as the hash object.
     Note that on writing, the records must be written so that the ones
     with a crossreference are located in the file BEFORE those that
     contain the missing data.  Also, fields such as 'booktitle' should be
     added to the referenced record as needed.
     */
    private Hashtable makeIndex(RandomAccessFile fr) {
        // create hash table to store records in
        Hashtable hash = new Hashtable();
        BibTeXRecord rec = new BibTeXRecord();
        // storage to make links
        String lastKey = null;
        try {
            try {
                // move to the beginning of the file
                fr.seek(0);
                // keep reading so long as the file pointer
                // is less than the length of the file
                while(fr.getFilePointer() < fr.length()) {
                    rec.read(fr); // read a record
                    String key = rec.getKey();
                    String[] data = new String[3];
                    data[FILE_POS] = String.valueOf(rec.getRecordStart()); // save position
                    data[NEXT_POS] = null;
                    if(lastKey != null) {
                        // not first, so update last
                        ((String[])hash.get(lastKey))[NEXT_POS] = key;
                        data[LAST_POS] = lastKey;}
                    else {
                        // first, so set last to -1
                        data[LAST_POS] = null;}
                    // store the name of this one, for when we cut out
                    lastKey = key;
                    // add data to the hash
                    hash.put(rec.getKey(), data);
                    // go through the field names and add them to the
                    // list if necessary
                    updateFieldList(rec);
                    updateTypeList(rec);
                }}
            catch(EOFException e) {
                debug(e.toString());}}
        catch(IOException e) {
            debug(e.toString());}
//        m_LastRecord = lastKey;
//        debug(m_Fields);
        return hash;
    }

    /**
     Check to see if any field labels from a particular record need to be
     added to the list of field labels.
     */
    public void updateFieldList(BibTeXRecord rec) {
        // enumerate over everything in the list
        Enumeration enum = rec.keys();
        while(enum.hasMoreElements()) {
            String field = (String)enum.nextElement();
            if(m_Fields.indexOf(" " + field + " ") < 0) {
                // not in field list, so add it
                m_Fields = m_Fields + field + " ";}}
    }

    /**
     Check to see if any field labels from a particular record need to be
     added to the list of field labels.
     */
    public void updateTypeList(BibTeXRecord rec) {
        // enumerate over everything in the list
        String type = rec.getType();
        if(m_Types.indexOf(" " + type + " ") < 0) {
            // not in field list, so add it
            m_Types = m_Types + type + " ";}
    }

    /**
     Write the index to a file.  The index has the following components:
     1) Name of BibTeX file.
     2) Length of BibTeX file
     3) Date of last change to BibTeX file
     4) List of fields in BibTeX file
     5) Index, as key, position, next, last
     */
    public void writeIndex() {
        // before trying to write the index, make sure that it is not empty
        if(m_Index.size() == 0) {
            // empty index, delete file and quit
            m_File.delete();
            return;}
        // close the random access file to get the date
        // construct the name of the file
        String EOLN = System.getProperty("line.separator");
        String iName = m_File.getAbsolutePath().substring(0,m_File.getAbsolutePath().length() - 4) + ".bdx";
        File iFile = new File(iName);
        try {
            FileWriter fr = new FileWriter(iFile);
            fr.write(m_File.getName() + EOLN); // file name
            fr.write(String.valueOf(m_File.length()) + EOLN); // length of file
            fr.write(String.valueOf(m_File.lastModified()) + EOLN); // time last modified
            fr.write(m_Fields + EOLN); // list of fields
            fr.write(m_Types + EOLN); // list of types
            // for ease of debugging, write the index in order
            boolean flag = true;
            String key = initialEntry(m_Index);
            while(flag) {
                // write the record
                String[] data = (String[])m_Index.get(key);
                fr.write(key + " " + data[FILE_POS] + " "
                         + data[LAST_POS] + " " + data[NEXT_POS] + EOLN);
                // set the value of the flag
                key = data[NEXT_POS];
                if(key == null) flag = false;}

            // now send out the index itself
//            Enumeration enum = m_Index.keys();
//            while(enum.hasMoreElements()) {
//                String key = (String)enum.nextElement();
//                String[] data = (String[])m_Index.get(key);
//                fr.write(key + " " + data[FILE_POS] + " "
//                         + data[LAST_POS] + " " + data[NEXT_POS] + EOLN);}
            // close it up
            fr.close();}
        catch(IOException e) {
            OKDialog.createOKDialog("Attempting to write index:\n" + e.toString());}
    }

    /**
     Find the end of the index.
     */
    public String finalEntry(Hashtable index) {
        String key = null;
        // get an element in the hash
        Enumeration enum = index.keys();
        if(enum.hasMoreElements()) {
            key = (String)enum.nextElement();
            // move forward from here until null it hit
            String[] data = (String[])index.get(key);
            while(data[NEXT_POS] != null) {
                key = data[NEXT_POS];
                data = (String[])index.get(key);}}
        // return it
        return key;
    }

    /**
     Find the beginning of the index.
     */
    public String initialEntry(Hashtable index) {
        String key = null;
        // get an element in the hash
        Enumeration enum = index.keys();
        if(enum.hasMoreElements()) {
            key = (String)enum.nextElement();
            // move backward from here until null it hit
            String[] data = (String[])index.get(key);
            while(data[LAST_POS] != null) {
                key = data[LAST_POS];
                data = (String[])index.get(key);}}
        // return it
        return key;
    }

    /**
     Housekeeping to conduct when closing the file
     */
    public void closeFile() {
        // close the random access file
        try{m_RAFile.close();}
        catch(IOException e) {debug(e.toString());}
        // write the index
        writeIndex();
    }

    /**
     Try to read an index file.  Once the index has been read, form a link
     to the file and check the length and last modified stamps.  If these
     do not match, recreate the index for this file.
     */
    public void readIndex(String fName) {
        File fl = new File(fName);
        FileReader fr = null;
        BufferedReader in = null;
        try {
            fr = new FileReader(fl);
            in = new BufferedReader(fr);
            try {
                String fileName = in.readLine();
                long length = (Long.valueOf(in.readLine())).longValue();
                long lastMod = (Long.valueOf(in.readLine())).longValue();
                m_Fields = in.readLine();
                m_Types = in.readLine();
                // now read the index into the index hash
                m_Index.clear();
                String line = in.readLine();
                while(line != null) {
                    StringTokenizer st = new StringTokenizer(line);
                    String key = st.nextToken();
                    String[] data = new String[3];
                    data[FILE_POS] = st.nextToken();
                    data[LAST_POS] = st.nextToken();
                    data[NEXT_POS] = st.nextToken();
                    if(data[LAST_POS].equals("null")) data[LAST_POS] = null;
                    if(data[NEXT_POS].equals("null")) data[NEXT_POS] = null;
                    m_Index.put(key, data);
                    line = in.readLine();}
                // have read the index, now check the file
                fileName = fl.getAbsolutePath().substring(0, fl.getAbsolutePath().length() - fileName.length()) + fileName;
                m_File = new File(fileName);
                m_RAFile = new RandomAccessFile(m_File, "rw");
                if((m_File.length() != length) || (m_File.lastModified() != lastMod)) {
                    // there have been changes, remake the index
                    m_Index = makeIndex(m_RAFile);}
            }
            catch(IOException e) {debug(e.toString());}
        }
        catch(FileNotFoundException e) {debug(e.toString());}
        // close the files
        try{
            if(in != null) in.close();
            if(fr != null) fr.close();}
        catch(IOException e) {debug(e.toString());}
    }

    /**
     Open a file.
     1) Read the index from the file
     2) Write the index entries to the list panel
     3) Select the first entry in the list
     4) Display the information for the first entry
     in the edit panel
     */
    public void openFile(String fileName) {
        // open a file dialog if the fileName is null, and keep opening
        // it until the user enters a file name
        if(!fileName.endsWith(".bib")) {
            // add a .bib ending if it is not there or it is not an index
            fileName = fileName + ".bib";}
        // get the name of the index file
        String iName = fileName.substring(0,fileName.length()-4) + ".bdx";
        m_File = new File(fileName);
        File iFile = new File(iName); // create a link to the index file
        setTitle(fileName);
        // check to see if the index exists
        if(iFile.exists()) {
//            debug("index exists, reading");
            // yes, read the index
            readIndex(iName);}
        else {
            // index does not exist, check the bibtex file
            if(m_File.exists()) {
                // bibtex file, need to create index and read file
                try {
                    try {
                        // open random access file and read index
                        m_RAFile = new RandomAccessFile(m_File, "rw");
                        m_Index = makeIndex(m_RAFile);}
                    catch(FileNotFoundException e) {
                        OKDialog.createOKDialog("Attepting to open file:\n" + e.toString());}}
                catch(IOException e) {
                    OKDialog.createOKDialog("Attempting to open file:\n" + e.toString());}}
            else {
                try {
                    try {
                        // open random access file and read index
                        m_RAFile = new RandomAccessFile(m_File, "rw");}
                    catch(FileNotFoundException e) {
                        OKDialog.createOKDialog("Attempting to create new file:\n" + e.toString());}}
                catch(IOException e) {
                    OKDialog.createOKDialog("Attempting to create new file:\n" + e.toString());}}}
        // set the list of keys
        setKeyList(m_Index);
    }

    /**
     Add an item to a list component alphabetically
     */
    private void addItemByOrder(List li, String item) {
        // add the choices alphabetically
        if(li.getItemCount() > 0) {
            int pos = 0;
            while(pos < li.getItemCount()) {
                if(li.getItem(pos).compareTo(item) > 0) {
                    li.addItem(item, pos);
                    break;}
                pos++;}
            if(pos == li.getItemCount()) {
                li.addItem(item);}}
        else {
            li.addItem(item);}
    }

    /**
     Process item events.  At this level, the come from the list
     */
    public void itemStateChanged(ItemEvent evt) {
        Object src = evt.getSource();
        // check the source to see what to do
//        debug(evt.toString());
        if(evt.getSource().equals(m_Keys)) {
            // state changed on list of keys
            String item = null;
            if(evt.getItem() != null) {
                debug("Item selected = " + evt.getItem());
                // get item from one clicked on
                item = m_Keys.getItem(((Integer)evt.getItem()).intValue());}
            else {
                debug("No item selected");
                // no item for event, use last selected one
                String[] items = m_Keys.getSelectedItems();
                if(items.length > 0) {
                    item = items[items.length - 1];}}
            // process the request
            processEntryUpdate(item);
        }
        // check for the checkbox entry
        if(src instanceof Checkbox) {
            // set mode of list appropriately
            if(((Checkbox)src).getLabel().equals("single")) {
                m_Keys.setMultipleMode(!((Checkbox)src).getState());}
            if(((Checkbox)src).getLabel().equals("multiple")) {
                m_Keys.setMultipleMode(((Checkbox)src).getState());}
            // clear the selections
            // leave last item still selected
            debug("Clearing selection, itemStateChanged()");
            int[] keys = m_Keys.getSelectedIndexes();
            for(int i=0; i<(keys.length - 1); i++) {
                m_Keys.deselect(i);}
            m_Keys.validate();
            // throw an item event
            ItemEvent evt2 = new ItemEvent(m_Keys, ItemEvent.ITEM_STATE_CHANGED, null, ItemEvent.DESELECTED);
            itemStateChanged(evt2);
        }
    }

    /**
     Check and entry.  If valid or rejected, read the new entry passed or
     set to blank if the entry passed is null.  This is only done if the
     checks are passed.
     */
    public boolean processEntryUpdate(String nextItem) {
        // return value is true if record is writen and able to proceed
        boolean ret = false;
        // check the entry for validity
        String msg = m_Edit.checkEntry();
        // get the record from the panel
        BibTeXRecord rec = m_Edit.getBibTeXRecord();
        if(m_ActiveRecord == null) m_ActiveRecord = rec; // for startup
        if(!rec.equals(m_ActiveRecord)) {
            // current record is not equal to the data in the panel
            if(!m_ActiveRecord.isEmpty()) {
                // initial record was not empty, need to erase it
                if(!AskDialog.createAskDialog("Active record is modified, save changes?", "Save", "Trash", this)) {
                    // read the record and return
                    if(nextItem != null) {
                        // read the record
                        readRecordToEdit(nextItem);
                        // update the citation display
                        updateCiteDisplay();}
                    else {
                        // set up new entry display
                        m_ActiveRecord = new BibTeXRecord();
                        // set the contents of the edit panel
                        m_Edit.setContents(m_ActiveRecord);
                        m_Edit.validate();
                    }
                    return true;}
                // new record is OK, so delete this one
                deleteRecord(m_ActiveRecord.getKey());}
            // going to save record, so check it out
            if(msg != null) {
                // display message and return
                OKDialog.createOKDialog(msg, this);
                // restore the contents of the panel
                m_Edit.setContents(rec);
                m_Edit.validate();
                return false;}
            // check for missing cross references or cross references
            // that are nested
            if(rec.containsKey("CROSSREF")) {
                String xref = ((BibTeXField)rec.get("CROSSREF")).getBareText();
                if(!m_Index.containsKey(xref)) {
                    // crossref key not found in index, error!
                    OKDialog.createOKDialog("The cross reference " + xref + " cannot be found.", this);
                    // restore contents and return
                    m_Edit.setContents(rec);
                    m_Edit.validate();
                    return false;}
                else {
                    // check here to see if crossreference is nested
                    // found it, now load the information
                    long pos = (Long.valueOf(((String[])m_Index.get(xref))[FILE_POS])).longValue();
                    try {
                        // set the pointer
                        m_RAFile.seek(pos);
                        // read the record
                        BibTeXRecord xrec = new BibTeXRecord();
                        xrec.read(m_RAFile);
                        // make sure this is not
                        // a nested crossref
                        if(xrec.containsKey("CROSSREF")) {
                            // nested crossreference, message to user.
                            OKDialog.createOKDialog("Record " + xrec.getKey() + " contains a cross reference.  Cross references may not be nested.", this);
                            return false;}}
                    catch(IOException e) {
                        OKDialog.createOKDialog("Checking for nested cross reference:\n" + e.toString());}}}
            // check the key
            if((m_Index.containsKey(rec.getKey()))
               &&(!rec.getKey().equals(m_ActiveRecord.getKey()))) {
                // duplicate key
                OKDialog.createOKDialog("The key " + rec.getKey() + " is already contained in this database.", this);
                // restore the contents of the panel
                m_Edit.setContents(rec);
                m_Edit.validate();
                return false;}
            else {
                // key not duplicate, write things
                writeRecord(rec);}
        }
        if(nextItem != null) {
            // read the record
            readRecordToEdit(nextItem);
            // select this record
            selectInList(m_Keys, nextItem);}
        else {
            // set up new entry display
            m_ActiveRecord = new BibTeXRecord();
            // set the contents of the edit panel
            m_Edit.setContents(m_ActiveRecord);
            m_Edit.validate();
        }
        // update the citation display
        updateCiteDisplay();
        return true;
    }

    /**
     Update the citation entry displayed at the bottom of the panel
     */
    public void updateCiteDisplay() {
        // get the selected records
        String[] keys = m_Keys.getSelectedItems();
        // get the current citation style
        String cite = m_CiteText.getText();
        cite = cite.substring(0, cite.indexOf("{"));
        // build the new citation
        StringBuffer sb = new StringBuffer(cite + "{");
        if(keys.length > 0) {
            sb.append(keys[0]);
            for(int i=1; i<keys.length; i++) {
                sb.append("," + keys[i]);}}
        sb.append("}");
        // set the text
        m_CiteText.setText(sb.toString());
    }

    /**
     Read a record from the file and load it into the edit panel
     */
    public void readRecordToEdit(String item) {
        if(item == null) {
            // no items are selected.  Set the display to blank
            m_ActiveRecord = new BibTeXRecord();
            m_Edit.setContents(m_ActiveRecord);
            m_Edit.validate();
            m_Key.setText("");
            return;}
        // set file pointer for loading record
        if(m_RAFile != null) {
            // get the position pointer
            if(m_Index.containsKey(item)) {
                long pos = (Long.valueOf(((String[])m_Index.get(item))[FILE_POS])).longValue();
                try {
                    // set the pointer
                    m_RAFile.seek(pos);
                    // read the record
                    m_ActiveRecord = new BibTeXRecord();
                    m_ActiveRecord.read(m_RAFile);
                    // put the data in the panel
                    m_Edit.setContents(m_ActiveRecord);
                    // read back into active record for comparison
                    m_ActiveRecord = m_Edit.getBibTeXRecord();
//                    m_Edit.setContents(m_ActiveRecord);
                    // check to see if there is a crossreference
                    if(m_ActiveRecord.containsKey("CROSSREF")) {
                        // get the crossreference key
                        String key = ((BibTeXField)m_ActiveRecord.get("CROSSREF")).getBareText();
                        if(m_Index.containsKey(key)) {
                            // load the record and set it
                            pos = (Long.valueOf(((String[])m_Index.get(key))[FILE_POS])).longValue();
                            m_RAFile.seek(pos);
                            // read the record
                            BibTeXRecord xref = new BibTeXRecord();
                            xref.read(m_RAFile);
                            // apply the data
                            m_Edit.setCrossReference(xref);}}
                    // validate the panel
                    m_Edit.validate();
                }
                catch(IOException e) {
                    OKDialog.createOKDialog("Attempting to read record:\n" + e.toString());}}
            // set the name in the controls at bottom
            m_Key.setText(m_ActiveRecord.getKey());
        }
    }

    /**
     Process a pressed key
     */
    public void keyPressed(KeyEvent evt) {
        debug(evt.toString());}

    /**
     Process a released key
     */
    public void keyReleased(KeyEvent evt) {
        debug(evt.toString());}

    /**
     Process a key typed
     */
    public void keyTyped(KeyEvent evt) {
        debug(evt.toString());}

    /**
     Process the action event
     */
    public void actionPerformed(ActionEvent evt) {
        // get the selection from the list
        String[] l_keys = m_Keys.getSelectedItems();
        String l_key = null;
        if(l_keys.length > 0) l_key = l_keys[0];
        // get source and evaluate
        Object src = evt.getSource();
        String label = "";
        if(src instanceof Button) {
            // get the button command
            if(evt.getActionCommand().equals("New")) {
                label = "Edit->New";}
            if(evt.getActionCommand().equals("Clear")) {
                label = "Edit->Clear";}
            if(evt.getActionCommand().equals("Reset")) {
                label = "Edit->Reset";}
            if(evt.getActionCommand().equals("Record")) {
                label = "Edit->Record";}
            if(evt.getActionCommand().equals("Cancel")) {
                label = "Edit->Cancel";}
            // commands that are not repeats of menu commands
            if(evt.getActionCommand().equals("<<")) {
                // now check to see if record has changed and user wants to keep changes
                if(!processEntryUpdate(l_key)) return;
                // first, check to see if list must be redone
                if(m_Keys.getItemCount() != m_Index.size()) setKeyList(m_Index);
                // move back one
                String key = m_Key.getText();
                if(key != "") {
                    String last = ((String[])m_Index.get(key))[LAST_POS];
                    if(last != null) {
                        // move to the last one, if user chooses
                        selectInList(m_Keys, last);
                        processEntryUpdate(last);}}}
            // move ahead one
            if(evt.getActionCommand().equals(">>")) {
                // now check to see if record has changed and user wants to keep changes
                if(!processEntryUpdate(l_key)) return;
                // first, check to see if list must be redone
                if(m_Keys.getItemCount() != m_Index.size()) setKeyList(m_Index);
                // move back one
                String key = m_Key.getText();
                if(key != "") {
                    String next = ((String[])m_Index.get(key))[NEXT_POS];
                    if(next != null) {
                        // move to the last one, if user chooses
                        selectInList(m_Keys, next);
                        processEntryUpdate(next);}}}
            // move to first record
            if(evt.getActionCommand().equals("|<")) {
                // now check to see if record has changed and user wants to keep changes
                if(!processEntryUpdate(l_key)) return;
                // first, check to see if list must be redone
                if(m_Keys.getItemCount() != m_Index.size()) setKeyList(m_Index);
                // move to initial record in index
                String key = initialEntry(m_Index);
                if(key != null) {
                    selectInList(m_Keys, key);
                    processEntryUpdate(key);}}
            // move to final record
            if(evt.getActionCommand().equals(">|")) {
                // now check to see if record has changed and user wants to keep changes
                if(!processEntryUpdate(l_key)) return;
                // first, check to see if list must be redone
                if(m_Keys.getItemCount() != m_Index.size()) setKeyList(m_Index);
                // move to final record in index
                String key = finalEntry(m_Index);
                if(key != null) {
                    selectInList(m_Keys, key);
                    processEntryUpdate(key);}}
        }
        if(src instanceof MenuItem) {
            // get a string representing parent and menu item
            label = ((Menu)((MenuItem)src).getParent()).getLabel()
                + "->" + ((MenuItem)src).getLabel();}
//        debug(label);
        // check to see about adding a new record
        if(label.equals("Edit->New")) {
            newRecord();
        }
        // check to see about clearing the record
        // to clear, get rid of fields, but save key and
        // type
        if(label.equals("Edit->Clear")) {
            // get the record from the edit panel
            BibTeXRecord rec = m_Edit.getBibTeXRecord();
            // clear all the fields
            rec.clear();
            // set the data in the panel from the record
            m_Edit.setContents(rec);
            m_Edit.validate();}
        if(label.equals("Edit->Reset")) {
            // set the data in the panel from the active record
            m_Edit.setContents(m_ActiveRecord);
            m_Edit.validate();}
        if(label.equals("Edit->Record")) {
            // set the data in the panel from the active record
            processEntryUpdate(m_Edit.getKeyTextField().getText());}
        if(label.equals("Edit->Cancel")) {
            // get the current selected item and set up panel
            String[] items = m_Keys.getSelectedItems();
            if(items.length > 0) {
                readRecordToEdit(items[0]);}
            else {
                // no record, set blank
                m_ActiveRecord = new BibTeXRecord();
                m_Edit.setContents(m_ActiveRecord);
                m_Edit.validate();}}
        // select all records
        if(label.equals("Records->Select All")) {
            // first switch to multiple mode
            // throw an item event
            m_Multiple.setState(true);
            ItemEvent evt2 = new ItemEvent(m_Multiple, ItemEvent.ITEM_STATE_CHANGED, null, ItemEvent.SELECTED);
            itemStateChanged(evt2);
            // go through each item, selecting it
            for(int i=0; i<m_Keys.getItemCount(); i++) {
                m_Keys.select(i);}
            // set the last one in the display
            if(m_Keys.getItemCount() > 0) {
                readRecordToEdit(m_Keys.getItem(m_Keys.getItemCount() - 1));}
            updateCiteDisplay();}
        // deselect all records
        if(label.equals("Records->Deselect All")) {
            // go through each item, deselecting it
            debug("Processing Deselect All Command.");
            for(int i=0; i<m_Keys.getItemCount(); i++) {
                m_Keys.deselect(i);}
            // no record, set blank
            m_ActiveRecord = new BibTeXRecord();
            m_Edit.setContents(m_ActiveRecord);
            m_Edit.validate();
            updateCiteDisplay();}
        // erase the information for one record
        if(label.equals("Records->Delete")) {
            // warn user that records will be deleted permanently
            if(!AskDialog.createAskDialog("Selected records will be permanently deleted.",
                                          "Continue", "Cancel", this)) return;
            // delete the records
            deleteSelected();}
        // check for request to post citation
        if(label.startsWith("Cite->")) {
            // rule out options request
            if(!label.equals("Cite->Options")) {
                // carve off citation part
                String style = label.substring(6);
                // get the text from the citation entry display
                String cite = m_CiteText.getText();
                cite = cite.substring(cite.indexOf("{"));
                // put in the text with the new command
                m_CiteText.setText(style + cite);
                // try pushing it onto the clipboard
                StringSelection ss = new StringSelection(style + cite);
                m_Clipboard.setContents(ss, this);}}
        // import a file
        if(label.equals("File->Import")) {
            // now check to see if record has changed and user wants to keep changes
            if(!processEntryUpdate(l_key)) return;
            // go to the import method
            importFile();
        }
        // export a file
        if(label.equals("File->Export")) {
            // go to the import method
            exportFile();
        }
        // compact the file
        if(label.equals("File->Compact")) {
            // compact the file
            compact();}
        // find
        if(label.equals("Search->Find")) {
            // now check to see if record has changed and user wants to keep changes
            if(!processEntryUpdate(l_key)) return;
            // create the find dialog
            if(m_Find == null) m_Find = new FindDialog(this);
            // set up the find dialog conditions
            m_Find.setList(m_Keys.getItems()); // list of items
            m_Find.setIndex(m_Index); // index
            m_Find.setFile(m_RAFile); // file for access
            m_Find.setItem(m_Key.getText()); // currently active record
            m_Find.setParent(this); // set the parent
            m_Find.setVisible(true);
            // back from find, set selection based on return
            // check for null
            if(m_Find.getItem() != null) {
                selectInList(m_Keys, m_Find.getItem());}}
        // search
        if(label.equals("Search->Search")) {
            // now check to see if record has changed and user wants to keep changes
            if(!processEntryUpdate(l_key)) return;
            // create the search dialog
            if(m_Search == null) m_Search = new SearchDialog(this, m_Types, m_Fields);
            // set the parameters
            m_Search.setIndex(m_Index);
            m_Search.setFile(m_RAFile);
            // show it
            m_Search.setVisible(true);
            // get results on return, and update list of keys
            String[] result = m_Search.getKeys();
            if(result == null) setKeyList(m_Index);
            else setKeyList(result);}
        // exit a search
        if(label.equals("Search->Exit")) {
            // now check to see if record has changed and user wants to keep changes
            if(!processEntryUpdate(l_key)) return;
            // set the key list based on initial index
            setKeyList(m_Index);}
        // display the egotistical information
        if(label.equals("Help->About")) {
            // show the OKDialog
            OKDialog.createOKDialog("JavaBib III\n"
                                    +"BibTeX Database Manager\nby\n"
                                    +"John Janmaat\n"
                                    +"Department of Economics\n"
                                    +"Acadia University\n"
                                    +"jjanmaat@acadiau.ca", this);}
    }

    /**
     Import a file.  Open a file dialog to ask for the name.  Set up a stream
     from this file, and pass control to the addRecords method.
     */
    public void importFile() {
        // first, a file dialog
        FileDialog fd = new FileDialog(this, "Import Records", FileDialog.LOAD);
        fd.setVisible(true);
        // get the details
        if(fd.getFile() != null) {
            // open a reader on the file name
            try {
                String name = fd.getDirectory() + fd.getFile();
                debug(name);
                FileReader fr = new FileReader(name);
                addRecords(fr);
                fr.close();}
            catch(IOException e) {
                OKDialog.createOKDialog("Attempting to import file:\n" + e.toString());}
        }
    }

    /**
     Export a file.  Open a file dialog to ask for the file name, and
     then set up a stream to this file.
     */
    public void exportFile() {
        // first, a file dialog
        FileDialog fd = new FileDialog(this, "Export Records", FileDialog.SAVE);
        fd.setVisible(true);
        String name = fd.getDirectory() + fd.getFile();
        // get the details
        if(fd.getFile() != null) {
            // get the selected records as a string
            String output = getSelectedRecords();
            // open a writer on the file name
            try {
                debug(name);
                FileWriter fw = new FileWriter(name);
                // write the selected records
                fw.write(output);
                fw.close();}
            catch(IOException e) {
                OKDialog.createOKDialog("Attempting to export file:\n" + e.toString());}
        }
    }

    /**
     Delete the selected items.
     */
    public void deleteSelected() {
        // get the selected items
        String[] items = m_Keys.getSelectedItems();
        // delete the records
        for(int i=0; i<items.length; i++) {
            if(m_Index.containsKey(items[i])) {
                deleteRecord(items[i]);}}
        // all done
    }

    /**
     Set the list of keys, given a reference to a hashtable that is the
     index.
     */
    public void setKeyList(Hashtable index) {
        // clear the list
        m_Keys.removeAll();
        // enumerate through the index, adding entries
        // to the list in sorted order
        Enumeration enum = index.keys();
        while(enum.hasMoreElements()) {
            addItemByOrder(m_Keys, (String)enum.nextElement());}
        // select the first item in the list
        // if listening is activated, this should lead to
        // the display being updated
        if(m_Keys.getItemCount() > 0) {
            m_Keys.select(0);
            // create an item event
            ItemEvent evt = new ItemEvent(m_Keys, ItemEvent.ITEM_STATE_CHANGED, null, ItemEvent.SELECTED);
            // call the item event processor
            itemStateChanged(evt);}
        else {
            // there are no elements in the list of keys
            m_ActiveRecord = new BibTeXRecord();
            m_Edit.setContents(m_ActiveRecord);
            m_Edit.validate();}
        // validate the display
        validate();
    }

    /**
     Set the list of keys, given a string array.
     */
    public void setKeyList(String[] list) {
        // clear the list
        m_Keys.removeAll();
        // add each element from list in turn
        for(int i=0; i<list.length; i++) {
            addItemByOrder(m_Keys, list[i]);}
        // select the first item
        if(m_Keys.getItemCount() > 0) {
            m_Keys.select(0);
            // create an item event
            ItemEvent evt = new ItemEvent(m_Keys, ItemEvent.ITEM_STATE_CHANGED, null, ItemEvent.SELECTED);
            // call the item event processor
            itemStateChanged(evt);}
        else {
            // there are no elements in the list of keys
            m_ActiveRecord = new BibTeXRecord();
            m_Edit.setContents(m_ActiveRecord);
            m_Edit.validate();}
        // validate the display
        validate();
    }

    /**
     Select an item in a list using the string
     */
    public void selectInList(java.awt.List list, String item) {
        for(int i=0; i<list.getItemCount(); i++) {
            if(list.getItem(i).equals(item)) {
                // check to make sure it is not already selected
                if(!list.isIndexSelected(i)) {
                    list.select(i);}
                // all done, bail out
                return;}}
    }

    /**
     Write a record to the file, erasing one record if needed
     */
    public void newRecord() {
        // call entry update process
        processEntryUpdate(null);
    }

    /**
     Write a record to the file.  The record is written at the end of the
     file, and pointers in the index are updated to reflect the changes
     */
    public void writeRecord(BibTeXRecord rec) {
        try {
//            debug("REC = " + rec.toString());
            // check to see if cross reference is set
            BibTeXRecord xref = null;
            if(rec.containsKey("CROSSREF")) {
                String lab = ((BibTeXField)rec.get("CROSSREF")).getBareText();
                // read the crossref record from the file
                long pos = (Long.valueOf(((String[])m_Index.get(lab))[FILE_POS])).longValue();
                // move the pointer to the right position
                m_RAFile.seek(pos);
                // read the record
                xref = new BibTeXRecord();
                xref.read(m_RAFile);}
            // move pointer to end of file
            long end = m_RAFile.length();
            m_RAFile.seek(end);
            // write the record
            rec.write(m_RAFile);
            // move the pointer back to the position before this write
            m_RAFile.seek(end);
            // read the record to get the file positions
            m_ActiveRecord = new BibTeXRecord();
            m_ActiveRecord.read(m_RAFile);
            // set the pointers
            String finRec = finalEntry(m_Index);
            if(finRec != null) {
                // last record exists, update it
                String[] data = (String[])m_Index.get(finRec);
                data[NEXT_POS] = (String)m_ActiveRecord.getKey();}
            // add entry for current record
            String[] data = new String[3];
            data[NEXT_POS] = null;
            data[LAST_POS] = finRec;
            // update the pointer to the last record
            finRec = (String)m_ActiveRecord.getKey();
            data[FILE_POS] = String.valueOf(m_ActiveRecord.getRecordStart());
            m_Index.put(finRec, data);
            // add the entry to the list
            addItemByOrder(m_Keys, finRec);
            // add the fields and types as needed
            updateFieldList(m_ActiveRecord);
            updateTypeList(m_ActiveRecord);
            // move the cross reference back to the end
            if(xref != null) {
                // delete the current xref
                deleteRecord(xref.getKey());
                // add a reference to booktitle
                if(!xref.containsKey("BOOKTITLE")) {
                    if(xref.containsKey("TITLE")) {
                        // make a new entry
                        BibTeXField bf
                            = new BibTeXField("BOOKTITLE",
                                              ((BibTeXField)xref.get("TITLE"))
                                              .getText());
                        // add the entry
                        xref.put("BOOKTITLE", bf);}}
//                debug("XREF = " + xref.toString());
                // and now write it
                writeRecord(xref);}
        }
        catch(IOException e) {
            // display error message in dialog
            OKDialog.createOKDialog("Attempting to write record:\n" + e.toString());}
    }

    /**
     Method to delete an entry.  The process:
     1) Wipe the record
     2) Update the pointers around the record
     */
    private void deleteRecord(String item) {
        // move the pointer to the position of this record
        if(m_RAFile != null) {
            // get the position pointer
            if(m_Index.containsKey(item)) {
                long pos = (Long.valueOf(((String[])m_Index.get(item))[FILE_POS])).longValue();
                try {
                    // set the pointer
                    m_RAFile.seek(pos);
                    // read the record
                    m_ActiveRecord = new BibTeXRecord();
                    m_ActiveRecord.read(m_RAFile);
                    // erase the record from the file
                    m_ActiveRecord.eraseRecord(m_RAFile);
                    // create a blank active record
                    m_ActiveRecord = new BibTeXRecord();
                    // put the data in the panel
                    m_Edit.setContents(m_ActiveRecord);
                    m_Edit.validate();
                    // remove the entry from the list
                    m_Keys.remove(item);
                    // fix the pointers
                    String[] data = ((String[])m_Index.get(item));
                    // fix last one
                    if(data[LAST_POS] != null) {
                        // there is a record before this one, get a reference
                        String[] data2 = (String[])m_Index.get(data[LAST_POS]);
                        // fix the values
                        data2[NEXT_POS] = data[NEXT_POS];}
                    if(data[NEXT_POS] != null) {
                        // there is a record after this one, get a reference
                        String[] data2 = (String[])m_Index.get(data[NEXT_POS]);
                        // fix the values
                        data2[LAST_POS] = data[LAST_POS];
                        // next record begins at beginning of current record
                        data2[FILE_POS] = data[FILE_POS];}
                    // remove the entry from the index
                    m_Index.remove(item);
                    // validate
                    m_Keys.validate();
                    m_Edit.validate();
                }
                catch(IOException e) {
                    // display the error message in a dialog
                    OKDialog.createOKDialog("Attempting to delete record:\n" + e.toString());}}
        }
    }

    /**
     Deal with a loss of clipboard ownership
     */
    public void lostOwnership(Clipboard clipboard, Transferable contents) {
        debug("lost clipboard ownership");
    }

    /**
     Print out the list of connections
     */
    public void printIndex() {
        String item = finalEntry(m_Index);
        while(item != null) {
            String[] data = (String[])m_Index.get(item);
            item = data[LAST_POS];
            debug(data[0] + ", " + data[1] + ", " + data[2]);}
    }

    /**
     Deal with focus gained.  For the key and crossreference fields,
     we don't care about this one.
     */
    public void focusGained(FocusEvent evt) {}

    /**
     Deal with focus lost.  When focus is lost from the key field, test
     that the field is contained in the index.  For the crossreference
     field, conduct the same test, and if true, move the data to the
     edit panel.
     */
    public void focusLost(FocusEvent evt) {
//        debug(evt.toString());
        if(evt.getSource().equals(m_Edit.getKeyTextField())) {
            // lost focus on key field.  Check to make sure we don't have
            // a duplicate key
            String key = m_Edit.getKeyTextField().getText();
//            if(m_Index.containsKey(key)) {
                // problem, key already exists
//                OKDialog.createOKDialog("Key already exists.  Record invalid without unique key.", this);}
        }
        if(evt.getSource().equals(m_Edit.getCrossRefTextField())) {
            // lost focus on crossreference field.  If this field contains
            // text, make sure that the key exists
            String key = m_Edit.getCrossRefTextField().getText();
            if(key.equals("")) {
                // field blank, remove the cross reference
                m_Edit.removeCrossReference();
                m_Edit.buildRecordPane();
                return;}
            if(!m_Index.containsKey(key)) {
                // key not found, message
                OKDialog.createOKDialog("Cross reference key not found.  Enter another or leave field blank.", this);}
            else {
                // found it, now load the information
                long pos = (Long.valueOf(((String[])m_Index.get(key))[FILE_POS])).longValue();
                try {
                    // set the pointer
                    m_RAFile.seek(pos);
                    // read the record
                    BibTeXRecord rec = new BibTeXRecord();
                    rec.read(m_RAFile);
                    // put the data in the panel.  First remove any old one
                    // before adding current one
                    m_Edit.removeCrossReference();
                    // before setting the information, make sure this is not
                    // a nested crossref
                    if(rec.containsKey("CROSSREF")) {
                        // nested crossreference, message to user.
                        OKDialog.createOKDialog("Record " + rec.getKey() + " contains a cross reference.  Cross references may not be nested.", this);}
                    else {
                        // not nested, set information
                        m_Edit.setCrossReference(rec);}
                    // validate the panel
                    m_Edit.validate();
                }
                catch(IOException e) {
                    OKDialog.createOKDialog("Attempting to load cross reference:\n" + e.toString());}}}
    }

    /**
     Method to output a string when a debug flag is set
     */
    public static void debug(String message)
    {
        if(m_Debug) System.out.println("ListFrame:"+message);}

    /**
     Method to set the debug flag
     */
    public static void setDebug(boolean flag) {m_Debug = flag;}
}