/*
 * compress.c
 *
 * data compression routines.
 *
 * idea taken from Robert Sedgewick's 'Algorithms in C'
 */
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <malloc.h>
#include "dos.h"
#include "compress.h"

static unsigned long *count = NULL;
struct Node *HufTree = NULL;
int end;

void ResetCount( void ) {

    if( HufTree ) {
	free( HufTree );
	HufTree = NULL;
    }

    if( count ) {
	free( count );
	count = NULL;
    }
}

void AddCount( unsigned char _far *buf, unsigned long size ) {

    if( count == NULL ) {
	count = malloc( COUNTTABSIZE );
	if( count == NULL )
	    return;
	memset( count, 0, COUNTTABSIZE );
    }

    while( size ) {
	++count[*buf];
	++buf;
	--size;
    }
}

unsigned BuildTree( void ) {

    int i, k, N;
    struct Node *min1, *min2;

    if( count == NULL )
	return CE_NOTAB;

    /* Look how many nodes we need */
    for( i = 0, end = 0; i < MAX_BYTE; ++i ) {
	if( count[i] ) ++end;
    }

    if( HufTree ) free( HufTree );
    HufTree = malloc( sizeof(struct Node) * end * 2 );
    if( HufTree == NULL )
	return CE_NOMEM;

    /* Initialize array */
    for( i = k = 0; i < end; ++k ) {
	if( count[k] ) {
	    HufTree[i].Parent = HufTree[i].LeftChild = HufTree[i].RightChild = NULL;
	    HufTree[i].Count = count[k];
	    HufTree[i].Value = (unsigned char)k;
	    ++i;
	}
    }

    for( ;; ++end ) {
	/* search for mins */
	for( i = 0, min1 = min2 = NULL; i < end; ++i ) {
	    if( HufTree[i].Parent == NULL ) {
		if( min1 == NULL || (HufTree[i].Count < min1->Count && min2 != NULL) )
		    min1 = &HufTree[i];
		else if( min2 == NULL || HufTree[i].Count < min2->Count )
		    min2 = &HufTree[i];
	    }
	}
	if( min2 == NULL )
	    break;

	/* make new root node */
	HufTree[end].Parent = NULL;
	HufTree[end].LeftChild = min1;
	HufTree[end].RightChild = min2;
	HufTree[end].Count = min1->Count + min2->Count;
	HufTree[end].Value = 0;
	/* mark children */
	min1->Parent = min2->Parent = &HufTree[end];
    }
    return CE_OK;
}


unsigned long *GetTree( void ) {

    return count;
}

void PutTree( unsigned long *tree ) {

    if( count ) free( count );
    count = tree;
}


unsigned long Encode( unsigned char _far *dst, unsigned long dstlen,
	     unsigned char _far *src, unsigned long srclen ) {

    int bitpos = 0;
    struct Node *n;
    unsigned int bits = 0;
    int i;
    unsigned long numbits = 0L;

    for( *dst = 0; srclen; --srclen, ++src ) {
	/* search for node */
	for( n = HufTree, i = 0; i < end; ++i, ++n ) {
	    if( n->Parent && (n->Value == *src) )
		break;
	}
	if( i == end )
	    return 0L;

	for( ; n->Parent; n = n->Parent ) {
	    if( n == n->Parent->RightChild )
		n->Parent->Move = 1;
	    else
		n->Parent->Move = 0;
	}

	/* flush bits */

	for( ; n->RightChild; ) {
	    bits = n->Move;
	    if( bits == 1 )
		n = n->RightChild;
	    else
		n = n->LeftChild;

	    *dst |= (bits & 1) << bitpos;
	    ++numbits;
	    ++bitpos;
	    if( bitpos > 7 ) {
		*++dst = 0;
		if( !dstlen-- )
		    return numbits;
		bitpos = 0;
	    }
	}
    }
    return numbits;
}


unsigned long Decode( unsigned char _far *dst, unsigned long dstlen,
	     unsigned char _far *src, unsigned long srclen ) {

    struct Node *Root, *n;
    unsigned char stream;
    int bitpos = 0;
    unsigned long len = 0L;

    Root = &HufTree[end-1];

    /* take the first byte */
    stream = *src++;

again:
    for( ; dstlen; ++len, --dstlen ) {
	/* loop down the tree (at the bottom of the tree there
	   are no more children) */
	for( n = Root; n->RightChild; ) {
	    if( stream & 1 )
		n = n->RightChild;
	    else
		n = n->LeftChild;
	    stream >>= 1;
	    ++bitpos;
	    --srclen;
	    if( srclen == 0L )
		return len;
	    if( bitpos > 7 ) {
		stream = *src++;
		bitpos = 0;
	    }
	}
	*dst++ = n->Value;
    }
    return len;
}

/* end of file compress.c */
