/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.io.util;

import com.google.common.base.Preconditions;
import com.google.common.base.Throwables;
import com.google.common.util.concurrent.RateLimiter;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.Closeable;
import java.io.DataInput;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.lang.invoke.MethodHandle;
import java.lang.invoke.MethodHandles;
import java.lang.reflect.Method;
import java.nio.ByteBuffer;
import java.nio.channels.Channels;
import java.nio.channels.ClosedChannelException;
import java.nio.channels.FileChannel;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.AtomicMoveNotSupportedException;
import java.nio.file.CopyOption;
import java.nio.file.DirectoryNotEmptyException;
import java.nio.file.FileStore;
import java.nio.file.FileSystems;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardCopyOption;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.FileAttributeView;
import java.nio.file.attribute.FileStoreAttributeView;
import java.text.DecimalFormat;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.cassandra.concurrent.ScheduledExecutors;
import org.apache.cassandra.config.CassandraRelevantProperties;
import org.apache.cassandra.io.FSError;
import org.apache.cassandra.io.FSErrorHandler;
import org.apache.cassandra.io.FSReadError;
import org.apache.cassandra.io.FSWriteError;
import org.apache.cassandra.io.sstable.CorruptSSTableException;
import org.apache.cassandra.service.StorageService;
import org.apache.cassandra.utils.JVMStabilityInspector;
import org.apache.cassandra.utils.NoSpamLogger;
import org.apache.cassandra.utils.SyncUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class FileUtils {
    public static final Charset CHARSET = StandardCharsets.UTF_8;
    private static final Logger logger = LoggerFactory.getLogger(FileUtils.class);
    private static final NoSpamLogger nospam1m = NoSpamLogger.getLogger(logger, 1L, TimeUnit.MINUTES);
    public static final long ONE_KB = 1024L;
    public static final long ONE_MB = 0x100000L;
    public static final long ONE_GB = 0x40000000L;
    public static final long ONE_TB = 0x10000000000L;
    private static final DecimalFormat df = new DecimalFormat("#.##");
    private static final AtomicReference<Optional<FSErrorHandler>> fsErrorHandler = new AtomicReference(Optional.empty());
    private static Class clsDirectBuffer;
    private static MethodHandle mhDirectBufferCleaner;
    private static MethodHandle mhCleanerClean;
    private static final File tempDir;
    private static final AtomicLong tempFileNum;

    public static File getTempDir() {
        return tempDir;
    }

    public static File createTempFile(String prefix, String suffix, File directory) {
        try {
            long num;
            String fileName;
            File candidate;
            do {
                num = tempFileNum.getAndIncrement();
            } while (!(candidate = new File(directory, fileName = prefix + Long.toString(num) + suffix)).createNewFile());
            return candidate;
        }
        catch (IOException e) {
            throw new FSWriteError((Throwable)e, directory);
        }
    }

    public static File createTempFile(String prefix, String suffix) {
        return FileUtils.createTempFile(prefix, suffix, tempDir);
    }

    public static File createDeletableTempFile(String prefix, String suffix) {
        File f = FileUtils.createTempFile(prefix, suffix, FileUtils.getTempDir());
        f.deleteOnExit();
        return f;
    }

    public static void createHardLink(String from, String to) {
        FileUtils.createHardLink(new File(from), new File(to));
    }

    public static void createHardLink(File from, File to) {
        if (to.exists()) {
            throw new RuntimeException("Tried to create duplicate hard link to " + to);
        }
        if (!from.exists()) {
            throw new RuntimeException("Tried to hard link to file that does not exist " + from);
        }
        try {
            Files.createLink(to.toPath(), from.toPath());
        }
        catch (IOException e) {
            throw new FSWriteError((Throwable)e, to);
        }
    }

    public static void createHardLinkWithConfirm(File from, File to) {
        try {
            FileUtils.createHardLink(from, to);
        }
        catch (FSWriteError ex) {
            throw ex;
        }
        catch (Throwable t) {
            throw new RuntimeException(String.format("Unable to hardlink from %s to %s", from, to), t);
        }
    }

    public static void createHardLinkWithConfirm(String from, String to) {
        FileUtils.createHardLinkWithConfirm(new File(from), new File(to));
    }

    public static void createHardLinkWithoutConfirm(String from, String to) {
        block2: {
            try {
                FileUtils.createHardLink(new File(from), new File(to));
            }
            catch (FSWriteError fse) {
                if (!logger.isTraceEnabled()) break block2;
                logger.trace("Could not hardlink file " + from + " to " + to, (Throwable)fse);
            }
        }
    }

    public static Throwable deleteWithConfirm(String filePath, Throwable accumulate) {
        return FileUtils.deleteWithConfirm(new File(filePath), accumulate, null);
    }

    public static Throwable deleteWithConfirm(File file, Throwable accumulate) {
        return FileUtils.deleteWithConfirm(file, accumulate, null);
    }

    public static Throwable deleteWithConfirm(File file, Throwable accumulate, RateLimiter rateLimiter) {
        try {
            double throttled;
            if (rateLimiter != null && (throttled = rateLimiter.acquire()) > 0.0) {
                nospam1m.warn("Throttling file deletion: waited {} seconds to delete {}", throttled, file);
            }
            Files.delete(file.toPath());
        }
        catch (Throwable t) {
            try {
                throw new FSWriteError(t, file);
            }
            catch (Throwable t2) {
                accumulate = org.apache.cassandra.utils.Throwables.merge(accumulate, t2);
            }
        }
        return accumulate;
    }

    public static void deleteWithConfirm(String file) {
        FileUtils.deleteWithConfirm(new File(file));
    }

    public static void deleteWithConfirm(File file) {
        org.apache.cassandra.utils.Throwables.maybeFail(FileUtils.deleteWithConfirm(file, null, null));
    }

    public static void deleteWithConfirmWithThrottle(File file, RateLimiter rateLimiter) {
        org.apache.cassandra.utils.Throwables.maybeFail(FileUtils.deleteWithConfirm(file, null, rateLimiter));
    }

    public static void copyWithOutConfirm(String from, String to) {
        block2: {
            try {
                Files.copy(Paths.get(from, new String[0]), Paths.get(to, new String[0]), new CopyOption[0]);
            }
            catch (IOException e) {
                if (!logger.isTraceEnabled()) break block2;
                logger.trace("Could not copy file" + from + " to " + to, (Throwable)e);
            }
        }
    }

    public static void copyWithConfirm(String from, String to) {
        FileUtils.copyWithConfirm(new File(from), new File(to));
    }

    public static void copyWithConfirm(File from, File to) {
        assert (from.exists());
        if (logger.isTraceEnabled()) {
            logger.trace("Copying {} to {}", (Object)from.getPath(), (Object)to.getPath());
        }
        try {
            Files.copy(from.toPath(), to.toPath(), new CopyOption[0]);
        }
        catch (IOException e) {
            throw new FSWriteError((Throwable)e, "Could not copy file" + from + " to " + to);
        }
    }

    public static void renameWithOutConfirm(String from, String to) {
        block2: {
            try {
                FileUtils.atomicMoveWithFallback(new File(from).toPath(), new File(to).toPath());
            }
            catch (IOException e) {
                if (!logger.isTraceEnabled()) break block2;
                logger.trace("Could not move file " + from + " to " + to, (Throwable)e);
            }
        }
    }

    public static void renameWithConfirm(String from, String to) {
        FileUtils.renameWithConfirm(new File(from), new File(to));
    }

    public static void renameWithConfirm(File from, File to) {
        assert (from.exists());
        if (logger.isTraceEnabled()) {
            logger.trace("Renaming {} to {}", (Object)from.getPath(), (Object)to.getPath());
        }
        try {
            FileUtils.atomicMoveWithFallback(from.toPath(), to.toPath());
        }
        catch (IOException e) {
            throw new RuntimeException(String.format("Failed to rename %s to %s", from.getPath(), to.getPath()), e);
        }
    }

    private static void atomicMoveWithFallback(Path from, Path to) throws IOException {
        try {
            Files.move(from, to, StandardCopyOption.REPLACE_EXISTING, StandardCopyOption.ATOMIC_MOVE);
        }
        catch (AtomicMoveNotSupportedException e) {
            logger.trace("Could not do an atomic move", (Throwable)e);
            Files.move(from, to, StandardCopyOption.REPLACE_EXISTING);
        }
    }

    public static void truncate(String path, long size) {
        try (FileChannel channel = FileChannel.open(Paths.get(path, new String[0]), StandardOpenOption.READ, StandardOpenOption.WRITE);){
            channel.truncate(size);
        }
        catch (FileNotFoundException | NoSuchFileException nfe) {
            throw new RuntimeException(nfe);
        }
        catch (IOException e) {
            throw new FSWriteError((Throwable)e, path);
        }
    }

    public static void closeQuietly(Closeable c) {
        try {
            if (c != null) {
                c.close();
            }
        }
        catch (Exception e) {
            logger.warn("Failed closing {}", (Object)c, (Object)e);
        }
    }

    public static void closeQuietly(AutoCloseable c) {
        try {
            if (c != null) {
                c.close();
            }
        }
        catch (Exception e) {
            logger.warn("Failed closing {}", (Object)c, (Object)e);
        }
    }

    public static void close(Closeable ... cs) throws IOException {
        FileUtils.close(Arrays.asList(cs));
    }

    public static void close(Iterable<? extends Closeable> cs) throws IOException {
        Throwable e = null;
        for (Closeable closeable : cs) {
            try {
                if (closeable == null) continue;
                closeable.close();
            }
            catch (Throwable ex) {
                if (e == null) {
                    e = ex;
                } else {
                    e.addSuppressed(ex);
                }
                logger.warn("Failed closing stream {}", (Object)closeable, (Object)ex);
            }
        }
        org.apache.cassandra.utils.Throwables.maybeFail(e, IOException.class);
    }

    public static void closeQuietly(Iterable<? extends AutoCloseable> cs) {
        for (AutoCloseable autoCloseable : cs) {
            try {
                if (autoCloseable == null) continue;
                autoCloseable.close();
            }
            catch (Exception ex) {
                logger.warn("Failed closing {}", (Object)autoCloseable, (Object)ex);
            }
        }
    }

    public static String getCanonicalPath(String filename) {
        try {
            return new File(filename).getCanonicalPath();
        }
        catch (IOException e) {
            throw new FSReadError((Throwable)e, filename);
        }
    }

    public static String getCanonicalPath(File file) {
        try {
            return file.getCanonicalPath();
        }
        catch (IOException e) {
            throw new FSReadError((Throwable)e, file);
        }
    }

    public static boolean isContained(File folder, File file) {
        Path folderPath = Paths.get(FileUtils.getCanonicalPath(folder), new String[0]);
        Path filePath = Paths.get(FileUtils.getCanonicalPath(file), new String[0]);
        return filePath.startsWith(folderPath);
    }

    public static String getRelativePath(String basePath, String path) {
        try {
            return Paths.get(basePath, new String[0]).relativize(Paths.get(path, new String[0])).toString();
        }
        catch (Exception ex) {
            String absDataPath = FileUtils.getCanonicalPath(basePath);
            return Paths.get(absDataPath, new String[0]).relativize(Paths.get(path, new String[0])).toString();
        }
    }

    public static void clean(ByteBuffer buffer) {
        if (buffer == null || !buffer.isDirect()) {
            return;
        }
        try {
            Object cleaner = mhDirectBufferCleaner.bindTo(buffer).invoke();
            if (cleaner != null) {
                mhCleanerClean.bindTo(cleaner).invoke();
            }
        }
        catch (RuntimeException e) {
            throw e;
        }
        catch (Throwable e) {
            throw new RuntimeException(e);
        }
    }

    public static void createDirectory(String directory) {
        FileUtils.createDirectory(new File(directory));
    }

    public static void createDirectory(File directory) {
        if (!directory.exists() && !directory.mkdirs()) {
            throw new FSWriteError((Throwable)new IOException("Failed to mkdirs " + directory), directory);
        }
    }

    public static boolean delete(String file) {
        if (!StorageService.instance.isDaemonSetupCompleted()) {
            logger.info("Deleting file during startup: {}", (Object)file);
        }
        File f = new File(file);
        return f.delete();
    }

    public static void delete(File ... files) {
        for (File file : files) {
            if (!StorageService.instance.isDaemonSetupCompleted()) {
                logger.info("Deleting file during startup: {}", (Object)file);
            }
            file.delete();
        }
    }

    public static void deleteAsync(final String file) {
        Runnable runnable = new Runnable(){

            @Override
            public void run() {
                FileUtils.deleteWithConfirm(new File(file));
            }
        };
        ScheduledExecutors.nonPeriodicTasks.execute(runnable);
    }

    public static long parseFileSize(String value) {
        if (!value.matches("\\d+(\\.\\d+)? (GiB|KiB|MiB|TiB|bytes)")) {
            throw new IllegalArgumentException(String.format("value %s is not a valid human-readable file size", value));
        }
        if (value.endsWith(" TiB")) {
            long result = Math.round(Double.valueOf(value.replace(" TiB", "")) * 1.099511627776E12);
            return result;
        }
        if (value.endsWith(" GiB")) {
            long result = Math.round(Double.valueOf(value.replace(" GiB", "")) * 1.073741824E9);
            return result;
        }
        if (value.endsWith(" KiB")) {
            long result = Math.round(Double.valueOf(value.replace(" KiB", "")) * 1024.0);
            return result;
        }
        if (value.endsWith(" MiB")) {
            long result = Math.round(Double.valueOf(value.replace(" MiB", "")) * 1048576.0);
            return result;
        }
        if (value.endsWith(" bytes")) {
            long result = Math.round(Double.valueOf(value.replace(" bytes", "")));
            return result;
        }
        throw new IllegalStateException(String.format("FileUtils.parseFileSize() reached an illegal state parsing %s", value));
    }

    public static String stringifyFileSize(double value) {
        if (value >= 1.099511627776E12) {
            double d = value / 1.099511627776E12;
            String val = df.format(d);
            return val + " TiB";
        }
        if (value >= 1.073741824E9) {
            double d = value / 1.073741824E9;
            String val = df.format(d);
            return val + " GiB";
        }
        if (value >= 1048576.0) {
            double d = value / 1048576.0;
            String val = df.format(d);
            return val + " MiB";
        }
        if (value >= 1024.0) {
            double d = value / 1024.0;
            String val = df.format(d);
            return val + " KiB";
        }
        String val = df.format(value);
        return val + " bytes";
    }

    public static void deleteRecursiveWithThrottle(File dir, RateLimiter rateLimiter) {
        if (dir.isDirectory()) {
            String[] children;
            for (String child : children = dir.list()) {
                FileUtils.deleteRecursiveWithThrottle(new File(dir, child), rateLimiter);
            }
        }
        FileUtils.deleteWithConfirmWithThrottle(dir, rateLimiter);
    }

    public static void deleteRecursive(File dir) {
        if (CassandraRelevantProperties.USE_NIX_RECURSIVE_DELETE.getBoolean() && dir.toPath().getFileSystem() == FileSystems.getDefault()) {
            FileUtils.deleteRecursiveUsingNixCommand(dir.toPath(), false);
            return;
        }
        FileUtils.deleteChildrenRecursive(dir);
        FileUtils.deleteWithConfirm(dir);
    }

    public static void deleteChildrenRecursive(File dir) {
        block5: {
            if (!dir.isDirectory()) break block5;
            String[] children = dir.list();
            if (children.length == 0) {
                return;
            }
            if (CassandraRelevantProperties.USE_NIX_RECURSIVE_DELETE.getBoolean() && dir.toPath().getFileSystem() == FileSystems.getDefault()) {
                for (String child : children) {
                    FileUtils.deleteRecursiveUsingNixCommand(dir.toPath().resolve(child), false);
                }
            } else {
                for (String child : children) {
                    FileUtils.deleteRecursive(new File(dir, child));
                }
            }
        }
    }

    private static void deleteRecursiveUsingNixCommand(Path path, boolean quietly) {
        Object[] cmd = new String[]{"rm", quietly ? "-drf" : "-dr", path.toAbsolutePath().toString()};
        try {
            String err;
            String out;
            Process p = Runtime.getRuntime().exec((String[])cmd);
            int result = p.waitFor();
            try (BufferedReader outReader = new BufferedReader(new InputStreamReader(p.getInputStream()));
                 BufferedReader errReader = new BufferedReader(new InputStreamReader(p.getErrorStream()));){
                out = outReader.lines().collect(Collectors.joining("\n"));
                err = errReader.lines().collect(Collectors.joining("\n"));
            }
            if (result != 0 && Files.exists(path, new LinkOption[0])) {
                logger.error("{} returned:\nstdout:\n{}\n\nstderr:\n{}", new Object[]{Arrays.toString(cmd), out, err});
                throw new IOException(String.format("%s returned non-zero exit code: %d%nstdout:%n%s%n%nstderr:%n%s", Arrays.toString(cmd), result, out, err));
            }
        }
        catch (IOException e) {
            throw new FSWriteError((Throwable)e, path.toString());
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new FSWriteError((Throwable)e, path.toString());
        }
    }

    public static void deleteRecursiveOnExit(File dir) {
        if (dir.isDirectory()) {
            String[] children;
            for (String child : children = dir.list()) {
                FileUtils.deleteRecursiveOnExit(new File(dir, child));
            }
        }
        logger.trace("Scheduling deferred deletion of file: {}", (Object)dir);
        dir.deleteOnExit();
    }

    public static void handleCorruptSSTable(CorruptSSTableException e) {
        fsErrorHandler.get().ifPresent(handler -> handler.handleCorruptSSTable(e));
    }

    public static void handleFSError(FSError e) {
        fsErrorHandler.get().ifPresent(handler -> handler.handleFSError(e));
    }

    public static void handleFSErrorAndPropagate(FSError e) {
        JVMStabilityInspector.inspectThrowable(e);
        throw Throwables.propagate(e);
    }

    public static long folderSize(File folder) {
        final long[] sizeArr = new long[]{0L};
        try {
            Files.walkFileTree(folder.toPath(), (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

                @Override
                public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) {
                    sizeArr[0] = sizeArr[0] + attrs.size();
                    return FileVisitResult.CONTINUE;
                }
            });
        }
        catch (IOException e) {
            logger.error("Error while getting {} folder size. {}", (Object)folder, (Object)e.getMessage());
        }
        return sizeArr[0];
    }

    public static void copyTo(DataInput in, OutputStream out, int length) throws IOException {
        byte[] buffer = new byte[65536];
        int copiedBytes = 0;
        while (copiedBytes + buffer.length < length) {
            in.readFully(buffer);
            out.write(buffer);
            copiedBytes += buffer.length;
        }
        if (copiedBytes < length) {
            int left = length - copiedBytes;
            in.readFully(buffer, 0, left);
            out.write(buffer, 0, left);
        }
    }

    public static boolean isSubDirectory(File parent, File child) throws IOException {
        parent = parent.getCanonicalFile();
        for (File toCheck = child = child.getCanonicalFile(); toCheck != null; toCheck = toCheck.getParentFile()) {
            if (!parent.equals(toCheck)) continue;
            return true;
        }
        return false;
    }

    public static void append(File file, String ... lines) {
        if (file.exists()) {
            FileUtils.write(file, Arrays.asList(lines), StandardOpenOption.APPEND);
        } else {
            FileUtils.write(file, Arrays.asList(lines), StandardOpenOption.CREATE);
        }
    }

    public static void appendAndSync(File file, String ... lines) {
        if (file.exists()) {
            FileUtils.write(file, Arrays.asList(lines), StandardOpenOption.APPEND, StandardOpenOption.SYNC);
        } else {
            FileUtils.write(file, Arrays.asList(lines), StandardOpenOption.CREATE, StandardOpenOption.SYNC);
        }
    }

    public static void replace(File file, String ... lines) {
        FileUtils.write(file, Arrays.asList(lines), StandardOpenOption.TRUNCATE_EXISTING);
    }

    public static void write(File file, List<String> lines, StandardOpenOption ... options) {
        HashSet<StandardOpenOption> optionsSet = new HashSet<StandardOpenOption>(Arrays.asList(options));
        if (optionsSet.isEmpty()) {
            optionsSet.add(StandardOpenOption.CREATE);
            optionsSet.add(StandardOpenOption.TRUNCATE_EXISTING);
        }
        boolean sync = optionsSet.remove(StandardOpenOption.SYNC);
        boolean dsync = optionsSet.remove(StandardOpenOption.DSYNC);
        optionsSet.add(StandardOpenOption.WRITE);
        Path filePath = file.toPath();
        try (FileChannel fc = filePath.getFileSystem().provider().newFileChannel(filePath, optionsSet, new FileAttribute[0]);
             BufferedWriter writer = new BufferedWriter(new OutputStreamWriter(Channels.newOutputStream(fc), CHARSET.newEncoder()));){
            for (CharSequence charSequence : lines) {
                writer.append(charSequence);
                writer.newLine();
            }
            if (sync) {
                SyncUtil.force(fc, true);
            } else if (dsync) {
                SyncUtil.force(fc, false);
            }
        }
        catch (ClosedChannelException cce) {
            throw new RuntimeException(cce);
        }
        catch (IOException ex) {
            throw new FSWriteError((Throwable)ex, file);
        }
    }

    public static List<String> readLines(File file) {
        try {
            return Files.readAllLines(file.toPath(), CHARSET);
        }
        catch (IOException ex) {
            if (ex instanceof NoSuchFileException) {
                return Collections.emptyList();
            }
            throw new RuntimeException(ex);
        }
    }

    public static void setFSErrorHandler(FSErrorHandler handler) {
        fsErrorHandler.getAndSet(Optional.ofNullable(handler));
    }

    public static long getTotalSpace(File file) {
        return FileUtils.handleLargeFileSystem(file.getTotalSpace());
    }

    public static long getFreeSpace(File file) {
        return FileUtils.handleLargeFileSystem(file.getFreeSpace());
    }

    public static long getUsableSpace(File file) {
        return FileUtils.handleLargeFileSystem(file.getUsableSpace());
    }

    public static FileStore getFileStore(Path path) throws IOException {
        return new SafeFileStore(Files.getFileStore(path));
    }

    private static long handleLargeFileSystem(long size) {
        return size < 0L ? Long.MAX_VALUE : size;
    }

    private FileUtils() {
    }

    public static void moveRecursively(Path source, Path target) throws IOException {
        logger.info("Moving {} to {}", (Object)source, (Object)target);
        if (Files.isDirectory(source, new LinkOption[0])) {
            Files.createDirectories(target, new FileAttribute[0]);
            for (File f : source.toFile().listFiles()) {
                String fileName = f.getName();
                FileUtils.moveRecursively(source.resolve(fileName), target.resolve(fileName));
            }
            FileUtils.deleteDirectoryIfEmpty(source);
        } else if (Files.exists(target, new LinkOption[0])) {
            logger.warn("Cannot move the file {} to {} as the target file already exists.", (Object)source, (Object)target);
        } else {
            Files.copy(source, target, StandardCopyOption.COPY_ATTRIBUTES);
            Files.delete(source);
        }
    }

    public static void deleteDirectoryIfEmpty(Path path) throws IOException {
        Preconditions.checkArgument(Files.isDirectory(path, new LinkOption[0]), String.format("%s is not a directory", path));
        try {
            logger.info("Deleting directory {}", (Object)path);
            Files.delete(path);
        }
        catch (DirectoryNotEmptyException e) {
            try (Stream<Path> paths = Files.list(path);){
                String content = paths.map(p -> p.getFileName().toString()).collect(Collectors.joining(", "));
                logger.warn("Cannot delete the directory {} as it is not empty. (Content: {})", (Object)path, (Object)content);
            }
        }
    }

    static {
        try {
            clsDirectBuffer = Class.forName("sun.nio.ch.DirectBuffer");
            Method mDirectBufferCleaner = clsDirectBuffer.getMethod("cleaner", new Class[0]);
            mhDirectBufferCleaner = MethodHandles.lookup().unreflect(mDirectBufferCleaner);
            Method mCleanerClean = mDirectBufferCleaner.getReturnType().getMethod("clean", new Class[0]);
            mhCleanerClean = MethodHandles.lookup().unreflect(mCleanerClean);
            ByteBuffer buf = ByteBuffer.allocateDirect(1);
            FileUtils.clean(buf);
        }
        catch (IllegalAccessException e) {
            logger.error("FATAL: Cassandra is unable to access required classes. This usually means it has been run without the aid of the standard startup scripts or the scripts have been edited. If this was intentional, and you are attempting to use Java 11+ you may need to add the --add-exports and --add-opens jvm options from either jvm11-server.options or jvm11-client.options", (Throwable)e);
            throw new RuntimeException(e);
        }
        catch (Throwable t) {
            logger.error("FATAL: Cannot initialize optimized memory deallocator.", t);
            JVMStabilityInspector.inspectThrowable(t);
            throw new RuntimeException(t);
        }
        tempDir = new File(CassandraRelevantProperties.JAVA_IO_TMPDIR.getString());
        tempFileNum = new AtomicLong();
    }

    private static final class SafeFileStore
    extends FileStore {
        private final FileStore fileStore;

        public SafeFileStore(FileStore fileStore) {
            this.fileStore = fileStore;
        }

        @Override
        public String name() {
            return this.fileStore.name();
        }

        @Override
        public String type() {
            return this.fileStore.type();
        }

        @Override
        public boolean isReadOnly() {
            return this.fileStore.isReadOnly();
        }

        @Override
        public long getTotalSpace() throws IOException {
            return FileUtils.handleLargeFileSystem(this.fileStore.getTotalSpace());
        }

        @Override
        public long getUsableSpace() throws IOException {
            return FileUtils.handleLargeFileSystem(this.fileStore.getUsableSpace());
        }

        @Override
        public long getUnallocatedSpace() throws IOException {
            return FileUtils.handleLargeFileSystem(this.fileStore.getUnallocatedSpace());
        }

        @Override
        public boolean supportsFileAttributeView(Class<? extends FileAttributeView> type) {
            return this.fileStore.supportsFileAttributeView(type);
        }

        @Override
        public boolean supportsFileAttributeView(String name) {
            return this.fileStore.supportsFileAttributeView(name);
        }

        @Override
        public <V extends FileStoreAttributeView> V getFileStoreAttributeView(Class<V> type) {
            return this.fileStore.getFileStoreAttributeView(type);
        }

        @Override
        public Object getAttribute(String attribute) throws IOException {
            return this.fileStore.getAttribute(attribute);
        }
    }
}

