/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.inlong.sdk.transform.process.function.string;

import org.apache.inlong.sdk.transform.decode.SourceData;
import org.apache.inlong.sdk.transform.process.Context;
import org.apache.inlong.sdk.transform.process.function.FunctionConstant;
import org.apache.inlong.sdk.transform.process.function.TransformFunction;
import org.apache.inlong.sdk.transform.process.operator.OperatorTools;
import org.apache.inlong.sdk.transform.process.parser.ValueParser;

import net.sf.jsqlparser.expression.Function;

/**
 * IsDecimalFunction  ->  is_decimal(string)
 * description:
 * - Return true if string can be parsed to a valid numeric.
 * - Return false otherwise (Including cases where string is null and '').
 */
@TransformFunction(type = FunctionConstant.STRING_TYPE, names = {
        "is_decimal"}, parameter = "(String str)", descriptions = {
                "- Return \"\" if 'str' is NULL;", "- Return true if 'str' can be parsed to a valid numeric;",
                "- Return false otherwise (Including cases where string is null and '')."
        }, examples = {
                "is_decimal('3he') = false",
                "is_decimal('3.5') = true"
        })
public class IsDecimalFunction implements ValueParser {

    private final ValueParser stringParser;

    public IsDecimalFunction(Function expr) {
        stringParser = OperatorTools.buildParser(expr.getParameters().getExpressions().get(0));
    }

    @Override
    public Object parse(SourceData sourceData, int rowIndex, Context context) {
        Object stringObject = stringParser.parse(sourceData, rowIndex, context);
        if (stringObject == null) {
            return false;
        }
        String string = OperatorTools.parseString(stringObject).toLowerCase();
        if (string.isEmpty()) {
            return false;
        }
        return string.matches("(\\s)*([+-])?(([0-9]*\\.)?([0-9]+)|([0-9]+)(\\.[0-9]*)?)([eE][\\+-]?[0-9]+)?(\\s)*");
    }
}
