/*
 * TclFilter.java
 *
 * Brazil project web application toolkit,
 * export version: 2.1 
 * Copyright (c) 1999-2004 Sun Microsystems, Inc.
 *
 * Sun Public License Notice
 *
 * The contents of this file are subject to the Sun Public License Version 
 * 1.0 (the "License"). You may not use this file except in compliance with 
 * the License. A copy of the License is included as the file "license.terms",
 * and also available at http://www.sun.com/
 * 
 * The Original Code is from:
 *    Brazil project web application toolkit release 2.1.
 * The Initial Developer of the Original Code is: suhler.
 * Portions created by suhler are Copyright (C) Sun Microsystems, Inc.
 * All Rights Reserved.
 * 
 * Contributor(s): cstevens, suhler.
 *
 * Version:  2.2
 * Created by suhler on 99/08/06
 * Last modified by suhler on 04/11/30 15:19:44
 */

package sunlabs.brazil.tcl;

import sunlabs.brazil.filter.Filter;
import sunlabs.brazil.server.FileHandler;
import sunlabs.brazil.server.Request;
import sunlabs.brazil.server.Server;
import sunlabs.brazil.session.SessionManager;
import sunlabs.brazil.util.http.MimeHeaders;

import tcl.lang.Interp;
import tcl.lang.ReflectObject;
import tcl.lang.TCL;
import tcl.lang.TclException;
import tcl.lang.TclObject;
import tcl.lang.TclUtil;

import java.io.File;

/**
 * Wrapper for writing FilterHandler filters in TCL.
 * Runs a tcl startup script when the handler is initialized.
 * Any time a request is made, the tcl <b>filter</b> proc
 * (which should be defined in the init script) is called,
 * provided with the
 * {@link sunlabs.brazil.server.Request} object as an argument.
 * <p>
 * This provides a bare-bones tcl interface.  The startup script
 * should provide a friendlier interface.
 * <p>
 * This handler requires the <code>tcl.jar</code>
 * jar file, a version of jacl included in the release.
 * <p>
 * The following server properties are used:
 * <dl class=props>
 * <dt>script	<dd> The name of the TCL file sourced on startup.
 *		The {@link #init} parameters are made available as the global
 *		variables <code>prefix</code> and <code>server</code>.
 * </dl>
 *
 * @author		Stephen Uhler
 * @version		2.2
 */

public class TclFilter implements Filter {
    Server server;
    String propsPrefix;
    String scriptName;

    static final String SCRIPT = "script";

    /**
     * extract the filter properties.
     */

    public boolean
    init(Server server, String prefix) {
    	this.server = server;
    	propsPrefix = prefix;
	// System.out.println("initing (" + prefix + ")");
	scriptName = server.props.getProperty(prefix + SCRIPT,prefix + "tcl");
	File scriptFile = new File(scriptName);
	if (!scriptFile.isAbsolute()) {
	    scriptFile = new File(server.props.getProperty(FileHandler.ROOT,"."), scriptName);
	}
	scriptName = scriptFile.getAbsolutePath();
	server.log(Server.LOG_DIAGNOSTIC, prefix, "Using: " + scriptName);
	return true;
    }

    /**
     * We don't need to look at the request.
     */

    public boolean
    respond(Request request) {
	return false;
    }

    /**
     * For now, only filter text/html.  This restriction will be
     * lifted once I figure out how
     */

    public boolean
    shouldFilter(Request request, MimeHeaders headers) {
	String type = headers.get("content-type");
	return (type != null && type.startsWith("text/html"));
    }

    /*
     * Find (or create) the interpreter.
     * Call the tcl callback script.
     * The <code>request</code> object reference is appended to the
     * <code>callback</code> parameter.
     * @return	true, if the callback script returns "true" or "1".
     */

    public byte[]
    filter(Request request, MimeHeaders headers, byte[] content) {

	/*
	 * Find the proper interp, creating it if needed.  If newly created
	 * (or if no SessionID variable was found) - initialize the interp
	 */

	String sessionId = request.props.getProperty("SessionID","common");
	request.log(Server.LOG_DIAGNOSTIC, "  Using session: " + sessionId);
	Interp interp = (Interp) SessionManager.getSession(sessionId,
		propsPrefix + "TCL", Interp.class);

    	int code = 0;
    	synchronized (interp) {
	    setupInterp(interp, sessionId);
	    try {
		// System.out.println("running eval");
		TclUtil.setVar(interp, "content", new String(content),
			TCL.GLOBAL_ONLY);
		interp.eval("filter" + " " +
		    ReflectObject.newInstance(interp, Request.class, request) +
		    " " + ReflectObject.newInstance(interp, MimeHeaders.class,
			headers)
		    );
		TclObject res = interp.getVar("content", TCL.GLOBAL_ONLY);
		content =  res.toString().getBytes();
	    } catch (TclException e) {
		code = e.getCompletionCode();
		String trace = e.toString();
		// System.out.println("Tcl Oops: "  + code + " " + e);
		if (code == 1) {
		    try {
			trace = interp.getVar("errorInfo", TCL.GLOBAL_ONLY).toString();
		    } catch (Exception e1) {}
		}
		request.log(Server.LOG_WARNING, propsPrefix + trace);
	    }
	}  // end sync block
	return content;
    }

    /**
     * Setup a tcl interpreter for this session
     */

    private void
    setupInterp(Interp interp, String id) {
	try {
	    interp.getVar("SessionID", TCL.GLOBAL_ONLY);
	    return;
	} catch (TclException e) {
	    // System.out.println("Creating new TCL interp");
	}
	try {
	    TclUtil.setVar(interp, "tcl_interactive", "0", TCL.GLOBAL_ONLY);
	    TclUtil.setVar(interp, "argv0", scriptName, TCL.GLOBAL_ONLY);
	    TclUtil.setVar(interp, "prefix", propsPrefix, TCL.GLOBAL_ONLY);
	    TclUtil.setVar(interp, "SessionID", id, TCL.GLOBAL_ONLY);
	    TclUtil.setVar(interp, "server",
		    ReflectObject.newInstance(interp, Server.class, server), 
		    TCL.GLOBAL_ONLY);
	    // System.out.println("About to run: " + scriptName);
	    interp.evalFile(scriptName);
	    // System.out.println("DONE");
	} catch (TclException e) {
	    int code = e.getCompletionCode();
	    String trace = e.toString();
	    if (code == 1) {
		try {
		    trace = interp.getVar("errorInfo",
			    TCL.GLOBAL_ONLY).toString();
		} catch (Exception e1) {}
	    }
        server.log(Server.LOG_WARNING, null, trace);
	}
    return;
    }
}
