"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const PathFilter_1 = require("./PathFilter");
const pathutils = require("./PathUtils");
/**
 * The entry in VSCodeResourceWatchService for an individual directory to
 * (recursively) watch. There should be a 1-1 relationship between WatchedPath
 * objects and projects that are monitored on behalf of the server.
 *
 * The receiveFileChanges(...) method is called by the parent, from which we
 * process and filter the change, then pass it to filewatcher core through the
 * IPlatformWatchListener interface.
 */
class VSCWatchedPath {
    constructor(pathRoot, ptw, parent) {
        this._pathInNormalizedForm = pathutils.normalizePath(pathRoot);
        this._pathFilter = new PathFilter_1.PathFilter(ptw);
        this._parent = parent;
        this._pathRoot = pathRoot;
        this._parent.parent.sendWatchResponseAsync(true, ptw);
    }
    receiveFileChanges(entries) {
        const newEvents = [];
        for (const wee of entries) {
            const relativePath = pathutils.convertAbsolutePathWithUnixSeparatorsToProjectRelativePath(wee.absolutePathWithUnixSeparators, this._pathInNormalizedForm);
            if (!relativePath || (!this._pathFilter.isFilteredOutByFilename(relativePath)
                && !this._pathFilter.isFilteredOutByPath(relativePath))) {
                newEvents.push(wee);
            }
        }
        if (newEvents.length > 0) {
            for (const event of newEvents) {
                this._parent.internal_handleEvent(event);
            }
        }
    }
    dispose() {
        /* Nothing to dispose */
    }
    get pathRoot() {
        return this._pathRoot;
    }
}
exports.VSCWatchedPath = VSCWatchedPath;
