/*******************************************************************************
 * Copyright (c) 2010 Engineering Group.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Marco Cortella (Engineering Group) - initial API and implementation and/or initial documentation
 *******************************************************************************/
package org.eclipse.ebam.emf;

import java.util.List;

import javax.persistence.EntityManager;
import javax.persistence.FlushModeType;
import javax.persistence.Query;

import org.eclipse.ebam.model.entities.SbiAlarm;
import org.eclipse.ebam.model.entities.Service;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/** Handle transactions of class Service
 * 
 * @author gavardi
 *
 */


public class ServiceTransactions extends EMF {

	/**
	 * Logger.
	 */
	private static Logger logger = 
		LoggerFactory.getLogger(ServiceTransactions.class);


	/** Get all services.
	 * 
	 * @return list of Services
	 * @throws Exception
	 */

	public final List getServiceList() throws Exception {
		List services = null;
		EntityManager em = getEMF().createEntityManager();
		try {
			// remove from cache because could have been changed outside of program
			getEMF().getCache().evict(Service.class);	
			Query query = em.createQuery("select s from "
					+ "Service s");
			//query.setParameter("code", code);
			services = query.getResultList();
		} catch (Exception e) {
			logger.error(e.getMessage(), e);
		} finally {			
			em.close();
		}
		return services;
	}

	/** get Service by id
	 * 
	 * @param id
	 * @return one Service
	 * @throws Exception
	 */

	public final Service getServiceById(Integer id) 
	throws Exception {
		Service service = null;
		EntityManager em = getEMF().createEntityManager();
		try {
			// remove from cache because could have been changed outside of program
			getEMF().getCache().evict(Service.class);	
			Query query = em.createQuery("select s from "
					+ "Service s where s.idService = :id");
			query.setParameter("id", id);
			service = (Service) query.getSingleResult();        

		} catch (Exception e) {
			logger.error(e.getMessage(), e);
		} finally {			
			em.close();
		}
		return service;
	}

	/** get Service by name
	 * 
	 * @param id
	 * @return one Service
	 * @throws Exception
	 */

	public final Service getServiceByName(String name) 
	throws Exception {
		Service service = null;
		EntityManager em = getEMF().createEntityManager();
		try {
			// remove from cache because could have been changed outside of program
			getEMF().getCache().evict(Service.class);	
			Query query = em.createQuery("select s from "
					+ "Service s where s.name = :name");
			query.setParameter("name", name);
			service = (Service) query.getSingleResult();        

		} catch (Exception e) {
			logger.error(e.getMessage(), e);
		} finally {			
			em.close();
		}
		return service;
	}

	
	/**Method that persists <code>Service</code> on database.
	 * @param j <code>Service</code> to save
	 * @return operation result
	 * @throws Exception error on saving 
	 */
	public final boolean createNewService(final Service service) throws Exception {
		boolean isSaved = false;
		EntityManager em = getEMF().createEntityManager();
		try {
			em.getTransaction().begin();

			// Set the flush-mode so the manager query does not force a 
			// write before the read.
			em.setFlushMode(FlushModeType.COMMIT);

			em.persist(service);

			em.getTransaction().commit();

			isSaved = true;

		} catch (Exception e) {
			if (em.getTransaction() != null && em.getTransaction().isActive()) {
				em.getTransaction().rollback();
			}			
			logger.error(e.getMessage(), e);
			throw e;
		} finally {			
			em.close();			
		}
		return isSaved;
	}


	/** merge an existng service
	 * 
	 * @param Service
	 * @return
	 * @throws Exception
	 */

	public final boolean mergeService(final Service service) throws Exception {
		boolean isSaved = false;
		EntityManager em = getEMF().createEntityManager();
		try {
			em.getTransaction().begin();

			// Set the flush-mode so the manager query does not force a 
			// write before the read.
			em.setFlushMode(FlushModeType.COMMIT);

			//take all contacts

			em.merge(service);

			em.getTransaction().commit();

			isSaved = true;

		} catch (Exception e) {
			if (em.getTransaction() != null && em.getTransaction().isActive()) {
				em.getTransaction().rollback();
			}			
			logger.error(e.getMessage(), e);
			throw e;
		} finally {			
			em.close();			
		}
		return isSaved;
	}

	/** Delete a service
	 * 
	 * @param service
	 * @return
	 * @throws Exception
	 */
	public final boolean deleteService(final Service service) throws Exception {
		boolean isSaved = false;
		EntityManager em = getEMF().createEntityManager();
		try {
			em.getTransaction().begin();

			em.setFlushMode(FlushModeType.COMMIT);

			Service toRemove = em.find(Service.class, service.getIdService());
			//toRemove.setMessages(null);
			em.merge(toRemove);
			em.remove(toRemove);

			em.getTransaction().commit();

			isSaved = true;

		} catch (Exception e) {
			if (em.getTransaction() != null && em.getTransaction().isActive()) {
				em.getTransaction().rollback();
			}			
			logger.error(e.getMessage(), e);
			throw e;
		} finally {			
			em.close();			
		}
		return isSaved;
	}


}
