package org.mandas.docker.client.messages.swarm;

import com.fasterxml.jackson.annotation.JsonProperty;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import org.mandas.docker.Nullable;

/**
 * Immutable implementation of {@link Reservations}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableReservations.builder()}.
 */
@SuppressWarnings({"all"})
final class ImmutableReservations implements Reservations {
  private final @Nullable Long nanoCpus;
  private final @Nullable Long memoryBytes;
  private final @Nullable List<ResourceSpec> resources;

  private ImmutableReservations(
      @Nullable Long nanoCpus,
      @Nullable Long memoryBytes,
      @Nullable List<ResourceSpec> resources) {
    this.nanoCpus = nanoCpus;
    this.memoryBytes = memoryBytes;
    this.resources = resources;
  }

  /**
   * @return The value of the {@code nanoCpus} attribute
   */
  @JsonProperty("NanoCPUs")
  @Override
  public @Nullable Long nanoCpus() {
    return nanoCpus;
  }

  /**
   * @return The value of the {@code memoryBytes} attribute
   */
  @JsonProperty("MemoryBytes")
  @Override
  public @Nullable Long memoryBytes() {
    return memoryBytes;
  }

  /**
   * @return The value of the {@code resources} attribute
   */
  @JsonProperty("GenericResources")
  @Override
  public @Nullable List<ResourceSpec> resources() {
    return resources;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Reservations#nanoCpus() nanoCpus} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for nanoCpus (can be {@code null})
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableReservations withNanoCpus(@Nullable Long value) {
    if (Objects.equals(this.nanoCpus, value)) return this;
    return new ImmutableReservations(value, this.memoryBytes, this.resources);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Reservations#memoryBytes() memoryBytes} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for memoryBytes (can be {@code null})
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableReservations withMemoryBytes(@Nullable Long value) {
    if (Objects.equals(this.memoryBytes, value)) return this;
    return new ImmutableReservations(this.nanoCpus, value, this.resources);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link Reservations#resources() resources}.
   * @param elements The elements to set
   * @return A modified copy of {@code this} object
   */
  public final ImmutableReservations withResources(@Nullable ResourceSpec... elements) {
    if (elements == null) {
      return new ImmutableReservations(this.nanoCpus, this.memoryBytes, null);
    }
    @Nullable List<ResourceSpec> newValue = Arrays.asList(elements) == null ? null : createUnmodifiableList(false, createSafeList(Arrays.asList(elements), true, false));
    return new ImmutableReservations(this.nanoCpus, this.memoryBytes, newValue);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link Reservations#resources() resources}.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param elements An iterable of resources elements to set
   * @return A modified copy of {@code this} object
   */
  public final ImmutableReservations withResources(@Nullable Iterable<? extends ResourceSpec> elements) {
    if (this.resources == elements) return this;
    @Nullable List<ResourceSpec> newValue = elements == null ? null : createUnmodifiableList(false, createSafeList(elements, true, false));
    return new ImmutableReservations(this.nanoCpus, this.memoryBytes, newValue);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableReservations} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableReservations
        && equalTo(0, (ImmutableReservations) another);
  }

  private boolean equalTo(int synthetic, ImmutableReservations another) {
    return Objects.equals(nanoCpus, another.nanoCpus)
        && Objects.equals(memoryBytes, another.memoryBytes)
        && Objects.equals(resources, another.resources);
  }

  /**
   * Computes a hash code from attributes: {@code nanoCpus}, {@code memoryBytes}, {@code resources}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + Objects.hashCode(nanoCpus);
    h += (h << 5) + Objects.hashCode(memoryBytes);
    h += (h << 5) + Objects.hashCode(resources);
    return h;
  }

  /**
   * Prints the immutable value {@code Reservations} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "Reservations{"
        + "nanoCpus=" + nanoCpus
        + ", memoryBytes=" + memoryBytes
        + ", resources=" + resources
        + "}";
  }

  /**
   * Creates an immutable copy of a {@link Reservations} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable Reservations instance
   */
  public static ImmutableReservations copyOf(Reservations instance) {
    if (instance instanceof ImmutableReservations) {
      return (ImmutableReservations) instance;
    }
    return ImmutableReservations.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableReservations ImmutableReservations}.
   * <pre>
   * ImmutableReservations.builder()
   *    .nanoCpus(Long | null) // nullable {@link Reservations#nanoCpus() nanoCpus}
   *    .memoryBytes(Long | null) // nullable {@link Reservations#memoryBytes() memoryBytes}
   *    .resources(List&amp;lt;org.mandas.docker.client.messages.swarm.ResourceSpec&amp;gt; | null) // nullable {@link Reservations#resources() resources}
   *    .build();
   * </pre>
   * @return A new ImmutableReservations builder
   */
  public static ImmutableReservations.Builder builder() {
    return new ImmutableReservations.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableReservations ImmutableReservations}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  static final class Builder implements Reservations.Builder {
    private Long nanoCpus;
    private Long memoryBytes;
    private List<ResourceSpec> resources = null;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code Reservations} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * Collection elements and entries will be added, not replaced.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(Reservations instance) {
      Objects.requireNonNull(instance, "instance");
      @Nullable Long nanoCpusValue = instance.nanoCpus();
      if (nanoCpusValue != null) {
        nanoCpus(nanoCpusValue);
      }
      @Nullable Long memoryBytesValue = instance.memoryBytes();
      if (memoryBytesValue != null) {
        memoryBytes(memoryBytesValue);
      }
      @Nullable List<ResourceSpec> resourcesValue = instance.resources();
      if (resourcesValue != null) {
        addAllResources(resourcesValue);
      }
      return this;
    }

    /**
     * Initializes the value for the {@link Reservations#nanoCpus() nanoCpus} attribute.
     * @param nanoCpus The value for nanoCpus (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("NanoCPUs")
    public final Builder nanoCpus(@Nullable Long nanoCpus) {
      this.nanoCpus = nanoCpus;
      return this;
    }

    /**
     * Initializes the value for the {@link Reservations#memoryBytes() memoryBytes} attribute.
     * @param memoryBytes The value for memoryBytes (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("MemoryBytes")
    public final Builder memoryBytes(@Nullable Long memoryBytes) {
      this.memoryBytes = memoryBytes;
      return this;
    }

    /**
     * Adds one element to {@link Reservations#resources() resources} list.
     * @param element A resources element
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder resource(ResourceSpec element) {
      if (this.resources == null) {
        this.resources = new ArrayList<ResourceSpec>();
      }
      this.resources.add(Objects.requireNonNull(element, "resources element"));
      return this;
    }

    /**
     * Adds elements to {@link Reservations#resources() resources} list.
     * @param elements An array of resources elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder resources(ResourceSpec... elements) {
      if (this.resources == null) {
        this.resources = new ArrayList<ResourceSpec>();
      }
      for (ResourceSpec element : elements) {
        this.resources.add(Objects.requireNonNull(element, "resources element"));
      }
      return this;
    }


    /**
     * Sets or replaces all elements for {@link Reservations#resources() resources} list.
     * @param elements An iterable of resources elements
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("GenericResources")
    public final Builder resources(@Nullable Iterable<? extends ResourceSpec> elements) {
      if (elements == null) {
        this.resources = null;
        return this;
      }
      this.resources = new ArrayList<ResourceSpec>();
      return addAllResources(elements);
    }

    /**
     * Adds elements to {@link Reservations#resources() resources} list.
     * @param elements An iterable of resources elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addAllResources(Iterable<? extends ResourceSpec> elements) {
      Objects.requireNonNull(elements, "resources element");
      if (this.resources == null) {
        this.resources = new ArrayList<ResourceSpec>();
      }
      for (ResourceSpec element : elements) {
        this.resources.add(Objects.requireNonNull(element, "resources element"));
      }
      return this;
    }

    /**
     * Builds a new {@link ImmutableReservations ImmutableReservations}.
     * @return An immutable instance of Reservations
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableReservations build() {
      return new ImmutableReservations(nanoCpus, memoryBytes, resources == null ? null : createUnmodifiableList(true, resources));
    }
  }

  private static <T> List<T> createSafeList(Iterable<? extends T> iterable, boolean checkNulls, boolean skipNulls) {
    ArrayList<T> list;
    if (iterable instanceof Collection<?>) {
      int size = ((Collection<?>) iterable).size();
      if (size == 0) return Collections.emptyList();
      list = new ArrayList<>(size);
    } else {
      list = new ArrayList<>();
    }
    for (T element : iterable) {
      if (skipNulls && element == null) continue;
      if (checkNulls) Objects.requireNonNull(element, "element");
      list.add(element);
    }
    return list;
  }

  private static <T> List<T> createUnmodifiableList(boolean clone, List<T> list) {
    switch(list.size()) {
    case 0: return Collections.emptyList();
    case 1: return Collections.singletonList(list.get(0));
    default:
      if (clone) {
        return Collections.unmodifiableList(new ArrayList<>(list));
      } else {
        if (list instanceof ArrayList<?>) {
          ((ArrayList<?>) list).trimToSize();
        }
        return Collections.unmodifiableList(list);
      }
    }
  }
}
