package org.mandas.docker.client.messages.swarm;

import com.fasterxml.jackson.annotation.JsonProperty;
import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import org.mandas.docker.Nullable;

/**
 * Immutable implementation of {@link ConfigSpec}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableConfigSpec.builder()}.
 */
@SuppressWarnings({"all"})
final class ImmutableConfigSpec implements ConfigSpec {
  private final String name;
  private final @Nullable Map<String, String> labels;
  private final @Nullable String data;

  private ImmutableConfigSpec(
      String name,
      @Nullable Map<String, String> labels,
      @Nullable String data) {
    this.name = name;
    this.labels = labels;
    this.data = data;
  }

  /**
   * @return The value of the {@code name} attribute
   */
  @JsonProperty("Name")
  @Override
  public String name() {
    return name;
  }

  /**
   * @return The value of the {@code labels} attribute
   */
  @JsonProperty("Labels")
  @Override
  public @Nullable Map<String, String> labels() {
    return labels;
  }

  /**
   * @return The value of the {@code data} attribute
   */
  @JsonProperty("Data")
  @Override
  public @Nullable String data() {
    return data;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ConfigSpec#name() name} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for name
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableConfigSpec withName(String value) {
    String newValue = Objects.requireNonNull(value, "name");
    if (this.name.equals(newValue)) return this;
    return new ImmutableConfigSpec(newValue, this.labels, this.data);
  }

  /**
   * Copy the current immutable object by replacing the {@link ConfigSpec#labels() labels} map with the specified map.
   * Nulls are not permitted as keys or values.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param entries The entries to be added to the labels map
   * @return A modified copy of {@code this} object
   */
  public final ImmutableConfigSpec withLabels(@Nullable Map<String, ? extends String> entries) {
    if (this.labels == entries) return this;
    @Nullable Map<String, String> newValue = entries == null ? null : createUnmodifiableMap(true, false, entries);
    return new ImmutableConfigSpec(this.name, newValue, this.data);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ConfigSpec#data() data} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for data (can be {@code null})
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableConfigSpec withData(@Nullable String value) {
    if (Objects.equals(this.data, value)) return this;
    return new ImmutableConfigSpec(this.name, this.labels, value);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableConfigSpec} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableConfigSpec
        && equalTo(0, (ImmutableConfigSpec) another);
  }

  private boolean equalTo(int synthetic, ImmutableConfigSpec another) {
    return name.equals(another.name)
        && Objects.equals(labels, another.labels)
        && Objects.equals(data, another.data);
  }

  /**
   * Computes a hash code from attributes: {@code name}, {@code labels}, {@code data}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + name.hashCode();
    h += (h << 5) + Objects.hashCode(labels);
    h += (h << 5) + Objects.hashCode(data);
    return h;
  }

  /**
   * Prints the immutable value {@code ConfigSpec} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "ConfigSpec{"
        + "name=" + name
        + ", labels=" + labels
        + ", data=" + data
        + "}";
  }

  /**
   * Creates an immutable copy of a {@link ConfigSpec} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable ConfigSpec instance
   */
  public static ImmutableConfigSpec copyOf(ConfigSpec instance) {
    if (instance instanceof ImmutableConfigSpec) {
      return (ImmutableConfigSpec) instance;
    }
    return ImmutableConfigSpec.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableConfigSpec ImmutableConfigSpec}.
   * <pre>
   * ImmutableConfigSpec.builder()
   *    .name(String) // required {@link ConfigSpec#name() name}
   *    .labels(Map&amp;lt;String, String&amp;gt; | null) // nullable {@link ConfigSpec#labels() labels}
   *    .data(String | null) // nullable {@link ConfigSpec#data() data}
   *    .build();
   * </pre>
   * @return A new ImmutableConfigSpec builder
   */
  public static ImmutableConfigSpec.Builder builder() {
    return new ImmutableConfigSpec.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableConfigSpec ImmutableConfigSpec}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  static final class Builder implements ConfigSpec.Builder {
    private static final long INIT_BIT_NAME = 0x1L;
    private long initBits = 0x1L;

    private String name;
    private Map<String, String> labels = null;
    private String data;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code ConfigSpec} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * Collection elements and entries will be added, not replaced.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(ConfigSpec instance) {
      Objects.requireNonNull(instance, "instance");
      this.name(instance.name());
      @Nullable Map<String, String> labelsValue = instance.labels();
      if (labelsValue != null) {
        putAllLabels(labelsValue);
      }
      @Nullable String dataValue = instance.data();
      if (dataValue != null) {
        data(dataValue);
      }
      return this;
    }

    /**
     * Initializes the value for the {@link ConfigSpec#name() name} attribute.
     * @param name The value for name 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Name")
    public final Builder name(String name) {
      this.name = Objects.requireNonNull(name, "name");
      initBits &= ~INIT_BIT_NAME;
      return this;
    }

    /**
     * Put one entry to the {@link ConfigSpec#labels() labels} map.
     * @param key The key in the labels map
     * @param value The associated value in the labels map
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addLabel(String key, String value) {
      if (this.labels == null) {
        this.labels = new LinkedHashMap<String, String>();
      }
      this.labels.put(
          Objects.requireNonNull(key, "labels key"),
          Objects.requireNonNull(value, value == null ? "labels value for key: " + key : null));
      return this;
    }

    /**
     * Put one entry to the {@link ConfigSpec#labels() labels} map. Nulls are not permitted
     * @param entry The key and value entry
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addLabel(Map.Entry<String, ? extends String> entry) {
      if (this.labels == null) {
        this.labels = new LinkedHashMap<String, String>();
      }
      String k = entry.getKey();
      String v = entry.getValue();
      this.labels.put(
          Objects.requireNonNull(k, "labels key"),
          Objects.requireNonNull(v, v == null ? "labels value for key: " + k : null));
      return this;
    }

    /**
     * Sets or replaces all mappings from the specified map as entries for the {@link ConfigSpec#labels() labels} map. Nulls are not permitted as keys or values, but parameter itself can be null
     * @param entries The entries that will be added to the labels map
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Labels")
    public final Builder labels(@Nullable Map<String, ? extends String> entries) {
      if (entries == null) {
        this.labels = null;
        return this;
      }
      this.labels = new LinkedHashMap<String, String>();
      return putAllLabels(entries);
    }

    /**
     * Put all mappings from the specified map as entries to {@link ConfigSpec#labels() labels} map. Nulls are not permitted
     * @param entries The entries that will be added to the labels map
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder putAllLabels(Map<String, ? extends String> entries) {
      if (this.labels == null) {
        this.labels = new LinkedHashMap<String, String>();
      }
      for (Map.Entry<String, ? extends String> e : entries.entrySet()) {
        String k = e.getKey();
        String v = e.getValue();
        this.labels.put(
            Objects.requireNonNull(k, "labels key"),
            Objects.requireNonNull(v, v == null ? "labels value for key: " + k : null));
      }
      return this;
    }

    /**
     * Initializes the value for the {@link ConfigSpec#data() data} attribute.
     * @param data The value for data (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Data")
    public final Builder data(@Nullable String data) {
      this.data = data;
      return this;
    }

    /**
     * Builds a new {@link ImmutableConfigSpec ImmutableConfigSpec}.
     * @return An immutable instance of ConfigSpec
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableConfigSpec build() {
      if (initBits != 0) {
        throw new IllegalStateException(formatRequiredAttributesMessage());
      }
      return new ImmutableConfigSpec(name, labels == null ? null : createUnmodifiableMap(false, false, labels), data);
    }

    private String formatRequiredAttributesMessage() {
      List<String> attributes = new ArrayList<>();
      if ((initBits & INIT_BIT_NAME) != 0) attributes.add("name");
      return "Cannot build ConfigSpec, some of required attributes are not set " + attributes;
    }
  }

  private static <K, V> Map<K, V> createUnmodifiableMap(boolean checkNulls, boolean skipNulls, Map<? extends K, ? extends V> map) {
    switch (map.size()) {
    case 0: return Collections.emptyMap();
    case 1: {
      Map.Entry<? extends K, ? extends V> e = map.entrySet().iterator().next();
      K k = e.getKey();
      V v = e.getValue();
      if (checkNulls) {
        Objects.requireNonNull(k, "key");
        Objects.requireNonNull(v, v == null ? "value for key: " + k : null);
      }
      if (skipNulls && (k == null || v == null)) {
        return Collections.emptyMap();
      }
      return Collections.singletonMap(k, v);
    }
    default: {
      Map<K, V> linkedMap = new LinkedHashMap<>(map.size() * 4 / 3 + 1);
      if (skipNulls || checkNulls) {
        for (Map.Entry<? extends K, ? extends V> e : map.entrySet()) {
          K k = e.getKey();
          V v = e.getValue();
          if (skipNulls) {
            if (k == null || v == null) continue;
          } else if (checkNulls) {
            Objects.requireNonNull(k, "key");
            Objects.requireNonNull(v, v == null ? "value for key: " + k : null);
          }
          linkedMap.put(k, v);
        }
      } else {
        linkedMap.putAll(map);
      }
      return Collections.unmodifiableMap(linkedMap);
    }
    }
  }
}
