/*******************************************************************************
 * @license
 * Copyright (c) 2012 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 * 
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/
/*eslint-env browser, amd, node*/
(function(root, factory) { // UMD
    if (typeof define === "function" && define.amd) { //$NON-NLS-0$
        define('orion/Deferred',factory);
    } else if (typeof exports === "object") { //$NON-NLS-0$
        module.exports = factory();
    } else {
        root.orion = root.orion || {};
        root.orion.Deferred = factory();
    }
}(this, function() {
    var queue = [],
        running = false;

    function run() {
        var fn;
        while ((fn = queue.shift())) {
            fn();
        }
        running = false;
    }

	var runAsync = (function() {
		if (typeof process !== "undefined" && typeof process.nextTick === "function") {
			var nextTick = process.nextTick;
    		return function() {
    			nextTick(run);
    		};
		} else if (typeof MutationObserver === "function") {
			var div = document.createElement("div");
			var observer = new MutationObserver(run);
			observer.observe(div, {
            	attributes: true
        	});
        	return function() {
        		div.setAttribute("class", "_tick");
        	};
		}
		return function() {
			setTimeout(run, 0);
		};
	})();

    function enqueue(fn) {
        queue.push(fn);
        if (!running) {
            running = true;
            runAsync();
        }
    }

    function noReturn(fn) {
        return function(result) {
            fn(result);
        };
    }
    
    function settleDeferred(fn, result, deferred) {
    	try {
    		var listenerResult = fn(result);
    		var listenerThen = listenerResult && (typeof listenerResult === "object" || typeof listenerResult === "function") && listenerResult.then;
    		if (typeof listenerThen === "function") {
    			if (listenerResult === deferred.promise) {
    				deferred.reject(new TypeError());
    			} else {
    				var listenerResultCancel = listenerResult.cancel;
    				if (typeof listenerResultCancel === "function") {
    					deferred._parentCancel = listenerResultCancel.bind(listenerResult);
    				} else {
    					delete deferred._parentCancel;
    				}
    				listenerThen.call(listenerResult, noReturn(deferred.resolve), noReturn(deferred.reject), noReturn(deferred.progress));
    			}
    		} else {
    			deferred.resolve(listenerResult);
    		}
    	} catch (e) {
    		deferred.reject(e);
    	}
    }


    /**
     * @name orion.Promise
     * @class Interface representing an eventual value.
     * @description Promise is an interface that represents an eventual value returned from the single completion of an operation.
     *
     * <p>For a concrete class that implements Promise and provides additional API, see {@link orion.Deferred}.</p>
     * @see orion.Deferred
     * @see orion.Deferred#promise
     */
    /**
     * @name then
     * @function
     * @memberOf orion.Promise.prototype
     * @description Adds handlers to be called on fulfillment or progress of this promise.
     * @param {Function} [onResolve] Called when this promise is resolved.
     * @param {Function} [onReject] Called when this promise is rejected.
     * @param {Function} [onProgress] May be called to report progress events on this promise.
     * @returns {orion.Promise} A new promise that is fulfilled when the given <code>onResolve</code> or <code>onReject</code>
     * callback is finished. The callback's return value gives the fulfillment value of the returned promise.
     */
    /**
     * Cancels this promise.
     * @name cancel
     * @function
     * @memberOf orion.Promise.prototype
     * @param {Object} reason The reason for canceling this promise.
     * @param {Boolean} [strict]
     */

    /**
     * @name orion.Deferred
     * @borrows orion.Promise#then as #then
     * @borrows orion.Promise#cancel as #cancel
     * @class Provides abstraction over asynchronous operations.
     * @description Deferred provides abstraction over asynchronous operations.
     *
     * <p>Because Deferred implements the {@link orion.Promise} interface, a Deferred may be used anywhere a Promise is called for.
     * However, in most such cases it is recommended to use the Deferred's {@link #promise} field instead, which exposes a 
     * simplified, minimally <a href="https://github.com/promises-aplus/promises-spec">Promises/A+</a>-compliant interface to callers.</p>
     */
    function Deferred() {
        var result, state, listeners = [],
            _this = this;

        function notify() {
            var listener;
            while ((listener = listeners.shift())) {
                var deferred = listener.deferred;
                var methodName = state === "fulfilled" ? "resolve" : "reject"; //$NON-NLS-0$ //$NON-NLS-1$ //$NON-NLS-2$
                var fn = listener[methodName];
                if (typeof fn === "function") { //$NON-NLS-0$
                	settleDeferred(fn, result, deferred);
                } else {
                    deferred[methodName](result);
                }
            }
        }

        function _reject(error) {
            delete _this._parentCancel;
            state = "rejected";
            result = error;
            if (listeners.length) {
                enqueue(notify);
            }
        }

        function _resolve(value) {
            function once(fn) {
                return function(result) {
                    if (!state || state === "assumed") {
                          fn(result);
                    }
                };
            }
            delete _this._parentCancel;
            try {
                var valueThen = value && (typeof value === "object" || typeof value === "function") && value.then;
                if (typeof valueThen === "function") {
                    if (value === _this) {
                        _reject(new TypeError());
                    } else {
                        state = "assumed";
                        var valueCancel = value && value.cancel;
                        if (typeof valueCancel !== "function") {
                            var deferred = new Deferred();
                            value = deferred.promise;
                            try {
                                valueThen(deferred.resolve, deferred.reject, deferred.progress);
                            } catch (thenError) {
                                deferred.reject(thenError);
                            }
                            valueCancel = value.cancel;
                            valueThen = value.then;
                        }
                        result = value;
                        valueThen.call(value, once(_resolve), once(_reject));
                        _this._parentCancel = valueCancel.bind(value);
                    }
                } else {
                    state = "fulfilled";
                    result = value;
                    if (listeners.length) {
                        enqueue(notify);
                    }
                }
            } catch (error) {
                once(_reject)(error);
            }
        }

        function cancel() {
            var parentCancel = _this._parentCancel;
            if (parentCancel) {
                delete _this._parentCancel;
                parentCancel();
            } else if (!state) {
                var cancelError = new Error("Cancel");
                cancelError.name = "Cancel";
                _reject(cancelError);
            }
        }


        /**
         * Resolves this Deferred.
         * @name resolve
         * @function
         * @memberOf orion.Deferred.prototype
         * @param {Object} value
         * @returns {orion.Promise}
         */
        this.resolve = function(value) {
            if (!state) {
                _resolve(value);
            }
            return _this;
        };

        /**
         * Rejects this Deferred.
         * @name reject
         * @function
         * @memberOf orion.Deferred.prototype
         * @param {Object} error
         * @param {Boolean} [strict]
         * @returns {orion.Promise}
         */
        this.reject = function(error) {
            if (!state) {
                _reject(error);
            }
            return _this;
        };

        /**
         * Notifies listeners of progress on this Deferred.
         * @name progress
         * @function
         * @memberOf orion.Deferred.prototype
         * @param {Object} update The progress update.
         * @returns {orion.Promise}
         */
        this.progress = function(update) {
            if (!state) {
                listeners.forEach(function(listener) {
                    if (listener.progress) {
                        try {
                            listener.progress(update);
                        } catch (ignore) {
                            // ignore
                        }
                    }
                });
            }
            return _this.promise;
        };

        this.cancel = function() {
            if (_this._parentCancel) {
                setTimeout(cancel, 0);
            } else {
                cancel();
            }
            return _this;
        };

        // Note: "then" ALWAYS returns before having onResolve or onReject called as per http://promises-aplus.github.com/promises-spec/
        this.then = function(onFulfill, onReject, onProgress) {
        	var deferred = new Deferred();
            deferred._parentCancel = _this.promise.cancel;
            listeners.push({
                resolve: onFulfill,
                reject: onReject,
                progress: onProgress,
                deferred: deferred
            });
            if (state === "fulfilled" || state === "rejected") {
                enqueue(notify);
            }
            return deferred.promise;
        };

        /**
         * The promise exposed by this Deferred.
         * @name promise
         * @field
         * @memberOf orion.Deferred.prototype
         * @type orion.Promise
         */
        this.promise = {
            then: _this.then,
            cancel: _this.cancel
        };
    }

    /**
     * Returns a promise that represents the outcome of all the input promises.
     * <p>When <code>all</code> is called with a single parameter, the returned promise has <dfn>eager</dfn> semantics,
     * meaning that if any input promise rejects, the returned promise immediately rejects, without waiting for the rest of the
     * input promises to fulfill.</p>
     *
     * To obtain <dfn>lazy</dfn> semantics (meaning the returned promise waits for every input promise to fulfill), pass the
     * optional parameter <code>optOnError</code>.
     * @name all
     * @function
     * @memberOf orion.Deferred
     * @static
     * @param {orion.Promise[]} promises The input promises.
     * @param {Function} [optOnError] Handles a rejected input promise. <code>optOnError</code> is invoked for every rejected
     * input promise, and is passed the reason the input promise was rejected. <p><code>optOnError</code> can return a value, which
     * allows it to act as a transformer: the return value serves as the final fulfillment value of the rejected promise in the 
     * results array generated by <code>all</code>.
     * @returns {orion.Promise} A new promise. The returned promise is generally fulfilled to an <code>Array</code> whose elements
     * give the fulfillment values of the input promises. <p>However, if an input promise rejects and eager semantics is used, the 
     * returned promise will instead be fulfilled to a single error value.</p>
     */
    Deferred.all = function(promises, optOnError) {
        var count = promises.length,
            result = [],
            rejected = false,
            deferred = new Deferred();

        deferred.then(undefined, function() {
            rejected = true;
            promises.forEach(function(promise) {
                if (promise.cancel) {
                    promise.cancel();
                }
            });
        });

        function onResolve(i, value) {
            if (!rejected) {
                result[i] = value;
                if (--count === 0) {
                    deferred.resolve(result);
                }
            }
        }

        function onReject(i, error) {
            if (!rejected) {
                if (optOnError) {
                    try {
                        onResolve(i, optOnError(error));
                        return;
                    } catch (e) {
                        error = e;
                    }
                }
                deferred.reject(error);
            }
        }

        if (count === 0) {
            deferred.resolve(result);
        } else {
            promises.forEach(function(promise, i) {
                promise.then(onResolve.bind(undefined, i), onReject.bind(undefined, i));
            });
        }
        return deferred.promise;
    };

    /**
     * Applies callbacks to a promise or to a regular object.
     * @name when
     * @function
     * @memberOf orion.Deferred
     * @static
     * @param {Object|orion.Promise} value Either a {@link orion.Promise}, or a normal value.
     * @param {Function} onResolve Called when the <code>value</code> promise is resolved. If <code>value</code> is not a promise,
     * this function is called immediately.
     * @param {Function} onReject Called when the <code>value</code> promise is rejected. If <code>value</code> is not a promise, 
     * this function is never called.
     * @param {Function} onProgress Called when the <code>value</code> promise provides a progress update. If <code>value</code> is
     * not a promise, this function is never called.
     * @returns {orion.Promise} A new promise.
     */
    Deferred.when = function(value, onResolve, onReject, onProgress) {
        var promise, deferred;
        if (value && typeof value.then === "function") { //$NON-NLS-0$
            promise = value;
        } else {
            deferred = new Deferred();
            deferred.resolve(value);
            promise = deferred.promise;
        }
        return promise.then(onResolve, onReject, onProgress);
    };

    return Deferred;
}));
/*******************************************************************************
 * @license
 * Copyright (c) 2012 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
define('orion/EventTarget',[],function() {
	/**
	 * Creates an Event Target
	 *
	 * @name orion.EventTarget
	 * @class Base for creating an Orion event target
	 */
	function EventTarget() {
		this._namedListeners = {};
	}

	EventTarget.prototype = /** @lends orion.EventTarget.prototype */
	{
		/**
		 * Dispatches a named event along with an arbitrary set of arguments. Any arguments after <code>eventName</code>
		 * will be passed to the event listener(s).
		 * @param {Object} event The event to dispatch. The event object MUST have a type field
		 * @returns {boolean} false if the event has been canceled and any associated default action should not be performed
		 * listeners (if any) have resolved.
		 */
		dispatchEvent: function(event) {
			if (!event.type) {
				throw new Error("unspecified type");
			}
			var listeners = this._namedListeners[event.type];
			if (listeners) {
				listeners.forEach(function(listener) {
					try {
						if (typeof listener === "function") {
							listener(event);
						} else {
							listener.handleEvent(event);
						}
					} catch (e) {
						if (typeof console !== 'undefined') {
							console.log(e); // for now, probably should dispatch an ("error", e)
						}
					}			
				});
			}
			return !event.defaultPrevented;
		},

		/**
		 * Adds an event listener for a named event
		 * @param {String} eventName The event name
		 * @param {Function} listener The function called when an event occurs
		 */
		addEventListener: function(eventName, listener) {
			if (typeof listener === "function" || listener.handleEvent) {
				this._namedListeners[eventName] = this._namedListeners[eventName] || [];
				this._namedListeners[eventName].push(listener);
			}
		},

		/**
		 * Removes an event listener for a named event
		 * @param {String} eventName The event name
		 * @param {Function} listener The function called when an event occurs
		 */
		removeEventListener: function(eventName, listener) {
			var listeners = this._namedListeners[eventName];
			if (listeners) {
				for (var i = 0; i < listeners.length; i++) {
					if (listeners[i] === listener) {
						if (listeners.length === 1) {
							delete this._namedListeners[eventName];
						} else {
							listeners.splice(i, 1);
						}
						break;
					}
				}
			}
		}
	};
	EventTarget.prototype.constructor = EventTarget;
	
	EventTarget.attach = function(obj) {
		var eventTarget = new EventTarget();
		obj.dispatchEvent = eventTarget.dispatchEvent.bind(eventTarget);
		obj.addEventListener = eventTarget.addEventListener.bind(eventTarget);
		obj.removeEventListener = eventTarget.removeEventListener.bind(eventTarget);
	};
	
	return EventTarget;
});
/*******************************************************************************
 * @license
 * Copyright (c) 2011, 2015 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd, node*/
/* eslint-disable missing-nls */
(function(root, factory) { // UMD
    if (typeof define === "function" && define.amd) {
        define('orion/plugin',["orion/Deferred", "orion/EventTarget"], factory);
    } else if (typeof exports === "object") {
        module.exports = factory(require("orion/Deferred"), require("orion/EventTarget"));
    } else {
        root.orion = root.orion || {};
        root.orion.PluginProvider = factory(root.orion.Deferred, root.orion.EventTarget);
    }
}(this, function(Deferred, EventTarget) {

    function _equal(obj1, obj2) {
        var keys1 = Object.keys(obj1);
        var keys2 = Object.keys(obj2);
        if (keys1.length !== keys2.length) {
            return false;
        }
        keys1.sort();
        keys2.sort();
        for (var i = 0, len = keys1.length; i < len; i++) {
            var key = keys1[i];
            if (key !== keys2[i]) {
                return false;
            }
            var value1 = obj1[key],
                value2 = obj2[key];
            if (value1 === value2) {
                continue;
            }
            if (JSON.stringify(value1) !== JSON.stringify(value2)) {
                return false;
            }
        }
        return true;
    }

    function ObjectReference(objectId, methods) {
        this.__objectId = objectId;
        this.__methods = methods;
    }
    
    function PluginProvider(headers, serviceRegistry) {
        var _headers = headers;
        var _connected = false;

        var _currentMessageId = 0;
        var _currentObjectId = 0;
        var _currentServiceId = 0;

        var _requestReferences = {};
        var _responseReferences = {};
        var _objectReferences = {};
        var _serviceReferences = {};
        
        var _services;
        var _remoteServices = {};
        var _registry = serviceRegistry;
        var _connectCallback;
        
        var _ports = [];
        var _shared = false;
        
        var _target = null;
        if (typeof(window) === "undefined") {
            if (self.postMessage) {
                _target = self;
            } else {
                _shared = true;
            }
        } else if (window !== window.parent) {
            _target = window.parent;
        } else if (window.opener !== null) {
            _target = window.opener;
        }        

        function _publish(message, target) {
            target = target || _target;
            if (target) {
                if (typeof(ArrayBuffer) === "undefined") {
                    message = JSON.stringify(message);
                }
                if (target === self || _shared) {
                    target.postMessage(message);
                } else {
                    target.postMessage(message, "*");
                }
            }
        }
        var _notify = _publish;
        var _errHandler = function(evt){
        	_publish({method: "error", error: _serializeError(evt.error)});
        };
        addEventListener("error", _errHandler);
        
        var lastHeartbeat;
        var startTime = new Date().getTime();
        function log(state) {
            if (typeof(localStorage) !== "undefined" && localStorage.pluginLogging) {
            	console.log(state + "(" + (new Date().getTime() - startTime) + "ms)=" + self.location);
        	}
        }
        function heartbeat() {
            var time = new Date().getTime();
            // This timeout depends on the handshake timeout of the plugin registry. Update both accordingly.
            if (lastHeartbeat  && time - lastHeartbeat < 4000) return;
            lastHeartbeat = time;
            _publish({
                method: "loading"
            });
            log("heartbeat");
        }
        heartbeat();

        if (_shared) {
            self.addEventListener("connect", function(evt) {
                var port = evt.ports[0];
                _ports.push(port);
                if (_connected) {
                    var message = {
                        method: "plugin",
                        params: [_getPluginData()]
                    };
                    _publish(message, port);
                } else {
                    heartbeat();
                }
                port.addEventListener("message",  function(evt) {
                	_handleMessage(evt, port);
                });
                port.start();
            });
        }

        function _getPluginData() {
            var services = [];
            // we filter out the service implementation from the data
            Object.keys(_serviceReferences).forEach(function(serviceId) {
                var serviceReference = _serviceReferences[serviceId];
                services.push({
                    serviceId: serviceId,
                    names: serviceReference.names,
                    methods: serviceReference.methods,
                    properties: serviceReference.properties
                });
            });
            return {
            	updateRegistry: !!_registry,
                headers: _headers || {},
                services: services
            };
        }

        function _jsonXMLHttpRequestReplacer(name, value) {
            if (value && value instanceof XMLHttpRequest) {
                var status, statusText;
                try {
                    status = value.status;
                    statusText = value.statusText;
                } catch (e) {
                    // https://bugs.webkit.org/show_bug.cgi?id=45994
                    status = 0;
                    statusText = ""; //$NON-NLS-0
                }
                return {
                    status: status || 0,
                    statusText: statusText
                };
            }
            return value;
        }

        function _serializeError(error) {
            var result = error ? JSON.parse(JSON.stringify(error, _jsonXMLHttpRequestReplacer)) : error; // sanitizing Error object
            if (error instanceof Error) {
                result.__isError = true;
                result.message = result.message || error.message;
                result.name = result.name || error.name;
            }
            return result;
        }

        function _request(message, target) {
            target = target || _target;
            if (!target) {
                return new Deferred().reject(new Error("plugin not connected"));
            }

            message.id = String(_currentMessageId++);
            var d = new Deferred();
            _responseReferences[message.id] = d;
            d.then(null, function(error) {
                if (_connected && error instanceof Error && error.name === "Cancel") {
                    _notify({
                        requestId: message.id,
                        method: "cancel",
                        params: error.message ? [error.message] : []
                    }, target);
                }
            });

            var toStr = Object.prototype.toString;
            message.params.forEach(function(param, i) {
                if (toStr.call(param) === "[object Object]" && !(param instanceof ObjectReference)) {
                    var candidate, methods;
                    for (candidate in param) {
                        if (toStr.call(param[candidate]) === "[object Function]") {
                            methods = methods || [];
                            methods.push(candidate);
                        }
                    }
                    if (methods) {
                        var objectId = _currentObjectId++;
                        _objectReferences[objectId] = param;
                        var removeReference = function() {
                            delete _objectReferences[objectId];
                        };
                        d.then(removeReference, removeReference);
                        message.params[i] = new ObjectReference(objectId, methods);
                    }
                }
            });
            _notify(message, target);
            return d.promise;
        }

        function _throwError(messageId, error, target) {
            if (messageId || messageId === 0) {
                _notify({
                    id: messageId,
                    result: null,
                    error: error
                }, target);
            } else {
                console.log(error);
            }
        }

        function _callMethod(messageId, implementation, method, params, target) {
            params.forEach(function(param, i) {
                if (param && typeof param.__objectId !== "undefined") {
                    var obj = {};
                    param.__methods.forEach(function(method) {
                        obj[method] = function() {
                            return _request({
                                objectId: param.__objectId,
                                method: method,
                                params: Array.prototype.slice.call(arguments)
                            }, target);
                        };
                    });
                    params[i] = obj;
                }
            });
            var response = typeof messageId === "undefined" ? null : {
                id: messageId,
                result: null,
                error: null
            };
            try {
                var promiseOrResult = method.apply(implementation, params);
                if (!response) {
                    return;
                }

                if (promiseOrResult && typeof promiseOrResult.then === "function") {
                    _requestReferences[messageId] = promiseOrResult;
                    promiseOrResult.then(function(result) {
                        delete _requestReferences[messageId];
                        response.result = result;
                        _notify(response, target);
                    }, function(error) {
                        if (_requestReferences[messageId]) {
                            delete _requestReferences[messageId];
                            response.error = _serializeError(error);
                            _notify(response, target);
                        }
                    }, function() {
                        _notify({
                            responseId: messageId,
                            method: "progress",
                            params: Array.prototype.slice.call(arguments)
                        }, target);
                    });
                } else {
                    response.result = promiseOrResult;
                    _notify(response, target);
                }
            } catch (error) {
                if (response) {
                    response.error = _serializeError(error);
                    _notify(response, target);
                }
            }
        }

        function _handleMessage(evnt, target) {
            if (!_shared && evnt.source !== _target && typeof window !== "undefined") {
                return;
            }
            var data = evnt.data;
            var message = (typeof data !== "string" ? data : JSON.parse(data));
            try {
                if (message.method) { // request
                    var method = message.method,
                        params = message.params || [];
                    if ("serviceId" in message) {
                        var service = _serviceReferences[message.serviceId];
                        if (!service) {
                            _throwError(message.id, "service not found", target);
                        } else {
	                        service = service.implementation;
	                        if (method in service) {
	                            _callMethod(message.id, service, service[method], params, target);
	                        } else {
	                            _throwError(message.id, "method not found", target);
	                        }
                    	}
                    } else if ("objectId" in message) {
                        var object = _objectReferences[message.objectId];
                        if (!object) {
                            _throwError(message.id, "object not found", target);
                        } else if (method in object) {
                            _callMethod(message.id, object, object[method], params, target);
                        } else {
                            _throwError(message.id, "method not found", target);
                        }
                    } else if ("requestId" in message) {
                        var request = _requestReferences[message.requestId];
                        if (request && method === "cancel" && request.cancel) {
                            request.cancel.apply(request, params);
                        }
                    } else if ("responseId" in message) {
                        var response = _responseReferences[message.responseId];
                        if (response && method === "progress" && response.progress) {
                            response.progress.apply(response, params);
                        }
                    } else {
                        if ("plugin" === message.method) { //$NON-NLS-0$
                            var manifest = message.params[0];
                            _update({
                                services: manifest.services
                            });
                        } else {
                            throw new Error("Bad method: " + message.method);
                        }
                    }
                } else if (message.id) {
                    var deferred = _responseReferences[String(message.id)];
                    if (deferred) {
	                    delete _responseReferences[String(message.id)];
	                    if (message.error) {
	                        deferred.reject(message.error);
	                    } else {
	                        deferred.resolve(message.result);
	                    }
                    }
                }
            } catch (e) {
                console.log("Plugin._messageHandler " + e);
            }
        }        
        
        function _createServiceProxy(service) {
            var serviceProxy = {};
            if (service.methods) {
                service.methods.forEach(function(method) {
                    serviceProxy[method] = function() {
                        var message = {
                            serviceId: service.serviceId,
                            method: method,
                            params: Array.prototype.slice.call(arguments)
                        };
                        return _request(message);
                    };
                });

                if (serviceProxy.addEventListener && serviceProxy.removeEventListener && EventTarget) {
                    var eventTarget = new EventTarget();
                    var objectId = _currentObjectId++;
                    _objectReferences[objectId] = {
                        handleEvent: eventTarget.dispatchEvent.bind(eventTarget)
                    };
                    var listenerReference = new ObjectReference(objectId, ["handleEvent"]);

                    var _addEventListener = serviceProxy.addEventListener;
                    serviceProxy.addEventListener = function(type, listener) {
                        if (!eventTarget._namedListeners[type]) {
                            _addEventListener(type, listenerReference);
                        }
                        eventTarget.addEventListener(type, listener);
                    };
                    var _removeEventListener = serviceProxy.removeEventListener;
                    serviceProxy.removeEventListener = function(type, listener) {
                        eventTarget.removeEventListener(type, listener);
                        if (!eventTarget._namedListeners[type]) {
                            _removeEventListener(type, listenerReference);
                        }
                    };
                }
            }
            return serviceProxy;
        }

        function _createServiceProperties(service) {
            var properties = JSON.parse(JSON.stringify(service.properties));
            var objectClass = service.names || service.type || [];
            if (!Array.isArray(objectClass)) {
                objectClass = [objectClass];
            }
            properties.objectClass = objectClass;
            return properties;
        }

        function _registerService(service) {
        	if (!_registry) return;
            var serviceProxy = _createServiceProxy(service);
            var properties = _createServiceProperties(service);
            var registration = _registry.registerService(service.names || service.type, serviceProxy, properties);
            _remoteServices[service.serviceId] = {
                registration: registration,
                proxy: serviceProxy
            };
        }

        function _update(input) {
            var oldServices = _services || [];
            _services = input.services || [];

            if (!_equal(_services, oldServices)) {
	            var serviceIds = [];
				_services.forEach(function(service) {
					var serviceId = service.serviceId;
	                serviceIds.push(serviceId);
	                var remoteService = _remoteServices[serviceId];
	                if (remoteService) {
	                    if (_equal(service.methods, Object.keys(remoteService.proxy))) {
	                        var properties = _createServiceProperties(service);
	                        var reference = remoteService.registration.getReference();
	                        var currentProperties = {};
	                        reference.getPropertyKeys().forEach(function(_name) {
	                            currentProperties[_name] = reference.getProperty(_name);
	                        });
	                        if (!_equal(properties, currentProperties)) {
	                            remoteService.registration.setProperties(properties);
	                        }
	                        return;
	                    }
	                    remoteService.registration.unregister();
	                    delete _remoteServices[serviceId];
	                }
	                _registerService(service);
	            });
	            Object.keys(_remoteServices).forEach(function(serviceId) {
	                if (serviceIds.indexOf(serviceId) === -1) {
	                    _remoteServices[serviceId].registration.unregister();
	                    delete _remoteServices[serviceId];
	                }
	            });
           }
           
           if (_connectCallback) {
               _connectCallback();
               _connectCallback = null;
           }
        }

        this.updateHeaders = function(headers) {
            if (_connected) {
                throw new Error("Cannot update headers. Plugin Provider is connected");
            }
            _headers = headers;
        };

        this.registerService = function(names, implementation, properties) {
            if (_connected) {
                throw new Error("Cannot register service. Plugin Provider is connected");
            }

            if (typeof names === "string") {
                names = [names];
            } else if (!Array.isArray(names)) {
                names = [];
            }

            var method = null;
            var methods = [];
            for (method in implementation) {
                if (typeof implementation[method] === 'function') {
                    methods.push(method);
                }
            }
            _serviceReferences[_currentServiceId++] = {
                names: names,
                methods: methods,
                implementation: implementation,
                properties: properties || {},
                listeners: {}
            };
            heartbeat();
        };
        this.registerServiceProvider = this.registerService;

        this.connect = function(callback, errback) {
            if (_connected) {
                if (callback) {
                    callback();
                }
                return;
            }
            removeEventListener("error", _errHandler);
            var message = {
                method: "plugin",
                params: [_getPluginData()]
            };
            if (!_shared) {
                if (!_target) {
                    if (errback) {
                        errback("No valid plugin target");
                    }
                    return;
                }           
                addEventListener("message", _handleMessage, false);
                _publish(message);
            }
            if (typeof(window) !== "undefined") {
            	var head = document.getElementsByTagName("head")[0] || document.documentElement;
            	var title = head.getElementsByTagName("title")[0];
            	if (!title) {
	            	title = document.createElement("title");
	            	title.textContent = _headers ? _headers.name : '';
	            	head.appendChild(title);
	        	}
        	}

            _ports.forEach(function(port) {
                _publish(message, port);
            });
            _connected = true;
            if (_registry) {
            	_connectCallback = callback;
            } else {
	            if (callback) {
	                callback();
	            }
            }
        };

        this.disconnect = function() {
            if (_connected) {
                removeEventListener("message", _handleMessage);
                _ports.forEach(function(port) {
                    port.close();
                });
                _ports = null;
                _target = null;
                _connected = false;
            }
            // Note: re-connecting is not currently supported
        };            
    }
    
    return PluginProvider;
}));

/*******************************************************************************
 * @license
 * Copyright (c) 2011, 2013 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html).
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
define('orion/serviceregistry',["orion/Deferred", "orion/EventTarget"], function(Deferred, EventTarget) {

	/**
	 * @name orion.serviceregistry.ServiceReference
	 * @description Creates a new service reference.
	 * @class A reference to a service in the Orion service registry
	 * @param {String} serviceId The symbolic id of this service instance
	 * @param {String} name The service name
	 * @param {Object} properties A JSON object containing the service's declarative properties
	 */
	function ServiceReference(serviceId, objectClass, properties) {
		this._properties = properties || {};
		this._properties["service.id"] = serviceId;
		this._properties.objectClass = objectClass;
		this._properties["service.names"] = objectClass;
	}

	ServiceReference.prototype = /** @lends orion.serviceregistry.ServiceReference.prototype */
	{
		/**
		 * @name getPropertyKeys
		 * @description Returns the names of the declarative properties of this service.
		 * @function
		 * @public
		 * @memberof orion.serviceregistry.ServiceReference.prototype
		 * @returns the names of the declarative properties of this service
		 */
		getPropertyKeys: function() {
			var result = [];
			var name;
			for (name in this._properties) {
				if (this._properties.hasOwnProperty(name)) {
					result.push(name);
				}
			}
			return result;
		},
		/**
		 * @name getProperty
		 * @description Returns the declarative service property with the given name, or <code>undefined</code>
		 * if this service does not have such a property.
		 * @function
		 * @public
		 * @memberof orion.serviceregistry.ServiceReference.prototype
		 * @param {String} propertyName The name of the service property to return
		 * @returns The {String} property with the given name or <code>undefined</code>
		 */
		getProperty: function(propertyName) {
			return this._properties[propertyName];
		}
	};
	ServiceReference.prototype.constructor = ServiceReference;

	/**
	 * @name orion.serviceregistry.ServiceRegistration
	 * @description Creates a new service registration. This constructor is private and should only be called by the service registry.
	 * @class A reference to a registered service in the Orion service registry
	 * @param {String} serviceId The symbolic id of this service
	 * @param {String} serviceReference A reference to the service
	 * @param {Object} internalRegistry A JSON object containing the service's declarative properties
	 */
	function ServiceRegistration(serviceId, serviceReference, internalRegistry) {
		this._serviceId = serviceId;
		this._serviceReference = serviceReference;
		this._internalRegistry = internalRegistry;
	}

	ServiceRegistration.prototype = /** @lends orion.serviceregistry.ServiceRegistration.prototype */
	{
		/**
		 * @name unregister
		 * @description Unregister this service registration. Clients registered for <code>unregistering</code> service events
		 * will be notified of this change.
		 * @function
		 * @private
		 * @memberof orion.serviceregistry.ServiceRegistration.prototype
		 */
		unregister: function() {
			this._internalRegistry.unregisterService(this._serviceId);
		},

		/**
		 * @name getReference
		 * @description Returns the {@link orion.serviceregistry.ServiceReference} in this registration
		 * @function
		 * @private
		 * @memberof orion.serviceregistry.ServiceRegistration.prototype
		 * @param properties
		 * @returns the {@link orion.serviceregistry.ServiceReference} in this registration
		 * @throws An error is the service has been unregistered
		 */
		getReference: function() {
			if (!this._internalRegistry.isRegistered(this._serviceId)) {
				throw new Error("Service has already been unregistered: "+this._serviceId);
			}
			return this._serviceReference;
		},
		/**
		 * @name setProperties
		 * @description Sets the properties of this registration to the new given properties. Clients registered for <code>modified</code> service events
		 * will be notified of the change.
		 * @function
		 * @private
		 * @memberof orion.serviceregistry.ServiceRegistration.prototype
		 * @param {Object} properties
		 */
		setProperties: function(properties) {
			var oldProperties = this._serviceReference._properties;
			this._serviceReference._properties = properties || {};
			properties["service.id"] = this._serviceId;
			properties.objectClass = oldProperties.objectClass;
			properties["service.names"] = oldProperties["service.names"];
			this._internalRegistry.modifyService(this._serviceId);
		}
	};
	ServiceRegistration.prototype.constructor = ServiceRegistration;

	/**
	 * @name orion.serviceregistry.DeferredService
	 * @description Creates a new service promise to resolve the service at a later time.
	 * @class A service that is resolved later
	 * @private
	 * @param {orion.serviceregistry.ServiceReference} implementation The implementation of the service
	 * @param {Function} isRegistered A function to call to know if the service is already registered
	 */
	function DeferredService(implementation, isRegistered) {

		function _createServiceCall(methodName) {
			return function() {
					var d;
					try {
						if (!isRegistered()) {
							throw new Error("Service was unregistered");
						}
						var result = implementation[methodName].apply(implementation, Array.prototype.slice.call(arguments));
						if (result && typeof result.then === "function") {
							return result;
						} else {
							d = new Deferred();
							d.resolve(result);
						}
					} catch (e) {
							d = new Deferred();
							d.reject(e);
					}
					return d.promise;
			};
		}

		var method;
		for (method in implementation) {
			if (typeof implementation[method] === 'function') {
				this[method] = _createServiceCall(method);
			}
		}
	}

	/**
	 * @name orion.serviceregistry.ServiceEvent
	 * @description An event that is fired from the service registry. Clients must register to listen to events using 
	 * the {@link orion.serviceregistry.ServiceRegistry#addEventListener} function.
	 * <br> 
	 * There are three types of events that this registry will send:
	 * <ul>
	 * <li>modified - the service has been modified</li> 
	 * <li>registered - the service has been registered</li> 
	 * <li>unregistering - the service is unregistering</li>
	 * </ul> 
	 * @class A service event
	 * @param {String} type The type of the event, one of <code>modified</code>, <code>registered</code> or <code>unregistered</code>
	 * @param {orion.serviceregistry.ServiceReference} serviceReference the service reference the event is for
	 */
	function ServiceEvent(type, serviceReference) {
		this.type = type;
		this.serviceReference = serviceReference;
	}

	/**
	 * @name orion.serviceregistry.ServiceRegistry
	 * @description Creates a new service registry
	 * @class The Orion service registry
	 */
	function ServiceRegistry() {
		this._entries = [];
		this._namedReferences = {};
		this._serviceEventTarget = new EventTarget();
		var _this = this;
		this._internalRegistry = {
			/**
			 * @name isRegistered
			 * @description Returns if the service with the given identifier is registered or not.
			 * @function
			 * @private
			 * @memberof orion.serviceregistry.ServiceRegistry
			 * @param {String} serviceId the identifier of the service
			 * @returns <code>true</code> if the service with the given identifier is registered, <code>false</code> otherwise
			 */
			isRegistered: function(serviceId) {
				return _this._entries[serviceId] ? true : false;
			},
			
			/**
			 * @name unregisterService
			 * @description Unregisters a service with the given identifier. This function will notify
			 * clients registered for <code>unregistering</code> service events.
			 * @function
			 * @private
			 * @memberof orion.serviceregistry.ServiceRegistry
			 * @param {String} serviceId the identifier of the service
			 * @throws An error if the service has already been unregistered
			 * @see orion.serviceregistry.ServiceEvent
			 */
			unregisterService: function(serviceId) {
				var entry = _this._entries[serviceId];
				if (!entry) {
					throw new Error("Service has already been unregistered: "+serviceId);
				}
				var reference = entry.reference;
				_this._serviceEventTarget.dispatchEvent(new ServiceEvent("unregistering", reference));
				_this._entries[serviceId] = null;
				var objectClass = reference.getProperty("objectClass");
				objectClass.forEach(function(type) {
					var namedReferences = _this._namedReferences[type];
					for (var i = 0; i < namedReferences.length; i++) {
						if (namedReferences[i] === reference) {
							if (namedReferences.length === 1) {
								delete _this._namedReferences[type];
							} else {
								namedReferences.splice(i, 1);
							}
							break;
						}
					}
				});
			},
			/**
			 * @name modifyService
			 * @description Notifies that the service with the given identifier has been modified. This function will notify clients
			 * registered for <code>modified</code> service events.
			 * @function
			 * @private
			 * @memberof orion.serviceregistry.ServiceRegistry
			 * @param {String} serviceId the identifier of the service
			 * @throws An error if the service for the given identifier does not exist
			 * @see orion.serviceregistry.ServiceEvent
			 */
			modifyService: function(serviceId) {
				var entry = _this._entries[serviceId];
				if (!entry) {
					throw new Error("Service not found while trying to send modified event: "+serviceId);
				}
				var reference = entry.reference;
				_this._serviceEventTarget.dispatchEvent(new ServiceEvent("modified", reference));
			}
		};
	}

	ServiceRegistry.prototype = /** @lends orion.serviceregistry.ServiceRegistry.prototype */
	{
		/**
		 * @name getService
		 * @description Returns the service with the given name or reference.
		 * @function
		 * @public
		 * @memberof orion.serviceregistry.ServiceRegistry.prototype
		 * @param {String|orion.serviceregistry.ServiceReference} nameOrServiceReference The service name or a service reference
		 * @returns {orion.serviceregistry.ServiceReference|null} The service implementation, or <code>null</code> if no such service was found.
		 */
		getService: function(typeOrServiceReference) {
			var service;
			if (typeof typeOrServiceReference === "string") {
				var references = this._namedReferences[typeOrServiceReference];
				if (references) {
					references.some(function(reference) {
						service = this._entries[reference.getProperty("service.id")].service;
						return !!service;
					}, this);
				}
			} else {
				var entry = this._entries[typeOrServiceReference.getProperty("service.id")];
				if (entry) {
					service = entry.service;
				}
			}
			return service || null;
		},

		/**
		 * @name getServiceReferences
		 * @description Returns all references to the service with the given name.
		 * @function
		 * @public
		 * @memberof orion.serviceregistry.ServiceRegistry.prototype
		 * @param {String} name The name of the service to return
		 * @returns {orion.serviceregistry.ServiceReference[]} An array of service references
		 */
		getServiceReferences: function(name) {
			if (name) {
				return this._namedReferences[name] ? this._namedReferences[name] : [];
			}
			var result = [];
			this._entries.forEach(function(entry) {
				if (entry) {
					result.push(entry.reference);
				}
			});
			return result;
		},
		
		/**
		 * @name registerService
		 * @description Registers a service with this registry. This function will notify clients registered
		 * for <code>registered</code> service events.
		 * @function
		 * @public
		 * @memberof orion.serviceregistry.ServiceRegistry.prototype
		 * @param {String|String[]} names the name or names of the service being registered
		 * @param {Object} service The service implementation
		 * @param {Object} properties A JSON collection of declarative service properties
		 * @returns {orion.serviceregistry.ServiceRegistration} A service registration object for the service.
		 * @see orion.serviceregistry.ServiceEvent
		 */
		registerService: function(names, service, properties) {
			if (typeof service === "undefined" ||  service === null) {
				throw new Error("invalid service");
			}

			if (typeof names === "string") {
				names = [names];
			} else if (!Array.isArray(names)) {
				names = [];
			}

			var serviceId = this._entries.length;
			var reference = new ServiceReference(serviceId, names, properties);
			var namedReferences = this._namedReferences;
			names.forEach(function(name) {
				namedReferences[name] = namedReferences[name] || [];
				namedReferences[name].push(reference);
			});
			var deferredService = new DeferredService(service, this._internalRegistry.isRegistered.bind(null, serviceId));
			this._entries.push({
				reference: reference,
				service: deferredService
			});
			var internalRegistry = this._internalRegistry;
			this._serviceEventTarget.dispatchEvent(new ServiceEvent("registered", reference));
			return new ServiceRegistration(serviceId, reference, internalRegistry);
		},

		/**
		 * @name addEventListener
		 * @description Adds a listener for events on this registry.
		 * <br> 
		 * The events that this registry notifies about:
		 * <ul>
		 * <li>modified - the service has been modified</li> 
		 * <li>registered - the service has been registered</li> 
		 * <li>unregistering - the service is unregistering</li> 
		 * </ul> 
		 * @function
		 * @public
		 * @memberof orion.serviceregistry.ServiceRegistry.prototype
		 * @param {String} eventName The name of the event to be notified about.
		 * @param {Function} listener The listener to add
		 * @see orion.serviceregistry.ServiceEvent
		 */
		addEventListener: function(eventName, listener) {
			this._serviceEventTarget.addEventListener(eventName, listener);
		},

		/**
		 * @name removeEventListener
		 * @description Removes a listener for service events in this registry.
		 * @function
		 * @public
		 * @memberof orion.serviceregistry.ServiceRegistry.prototype
		 * @param {String} eventName The name of the event to stop listening for
		 * @param {Function} listener The listener to remove
		 * @see orion.serviceregistry.ServiceEvent
		 */
		removeEventListener: function(eventName, listener) {
			this._serviceEventTarget.removeEventListener(eventName, listener);
		}
	};
	ServiceRegistry.prototype.constructor = ServiceRegistry;

	//return the module exports
	return {
		ServiceRegistry: ServiceRegistry
	};
});
/*******************************************************************************
 * @license
 * Copyright (c) 2013 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 * 
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/
/*eslint-env browser, amd*/
define('orion/objects',[], function() {
	function mixin(target/*, source..*/) {
		var hasOwnProperty = Object.prototype.hasOwnProperty;
		for (var j = 1, len = arguments.length; j < len; j++) {
			var source = arguments[j];
			for (var key in source) {
				if (hasOwnProperty.call(source, key)) {
					target[key] = source[key];
				}
			}
		}
		return target;
	}

	/**
	 * @name orion.objects
	 * @class Object-oriented helpers.
	 */
	return {
		/**
		 * Creates a shallow clone of the given <code>object</code>.
		 * @name orion.objects.clone
		 * @function
		 * @static
		 * @param {Object|Array} object The object to clone. Must be a "normal" Object or Array. Other built-ins,
		 * host objects, primitives, etc, will not work.
		 * @returns {Object|Array} A clone of <code>object</code>.
		 */
		clone: function(object) {
			if (Array.isArray(object)) {
				return Array.prototype.slice.call(object);
			}
			var clone = Object.create(Object.getPrototypeOf(object));
			mixin(clone, object);
			return clone;
		},
		/**
		 * Mixes all <code>source</code>'s own enumerable properties into <code>target</code>. Multiple source objects
		 * can be passed as varargs.
		 * @name orion.objects.mixin
		 * @function
		 * @static
		 * @param {Object} target
		 * @param {Object} source
		 */
		mixin: mixin,
		/**
		 * Wraps an object into an Array if necessary.
		 * @name orion.objects.toArray
		 * @function
		 * @static
		 * @param {Object} obj An object.
		 * @returns {Array} Returns <code>obj</code> unchanged, if <code>obj</code> is an Array. Otherwise returns a 1-element Array
		 * whose sole element is <code>obj</code>.
		 */
		toArray: function(o) {
			return Array.isArray(o) ? o : [o];
		}
	};
});
/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env amd*/
define('javascript/lru',[
], function() {

    function node(key, value) {
        var n = Object.create(null);
        n._p = null;
        n._n = null;
        n._v = {key: key, value:value};
        return n;
    }

	/**
	 * @description Creates a new LRU cache with the given maximum size. If no size is given 
	 * an unbounded cache is created.
	 * 
	 * @constructor 
	 * @param {Number} size The maximum size of the LRU or -1 for an unbounded cache
	 * @returns {javascript.LRU} A new LRU instance
	 * @since 8.0
	 */
	function LRU(size) {
	    if(typeof size === 'undefined') {
	       this._max = -1;
	    } else {
	       this._max = size; 
	    }
	    this._start = this._end = null;
	    this._size = 0;
	    this._cache = Object.create(null);
	}
	
	/**
	 * @description Clears the entire cache
	 * @function
	 */
	LRU.prototype.clear = function clear() {
	    this._cache = Object.create(null);
	    this._start = null;
	    this._end = null;
	    this._size = 0;
	};
	/**
	 * @description Returns the current size of the map
	 * @function
	 * @returns {Number} The size of the map
	 */
	LRU.prototype.size = function size() {
	  return this._size;  
	};
	/**
	 * @description If the map contains the given key
	 * @function
	 * @param {String} key The key to check
	 * @returns {Boolean} If the map contains the key or not
	 */
	LRU.prototype.containsKey = function containsKey(key) {
	    return typeof this._cache[key] !== 'undefined';
	};
	/**
	 * @description Adds the given key / value pair to the map. If the addition is
	 * greater than the given maximum map size, the last entry will be removed 
	 * and the new entry added to the head of the map.
	 * 
	 * Putting a value that already exists in the map will move it to the head
	 * of the LRU discarding the existing value.
	 * 
	 * @function
	 * @param {String} key The key to map the given value to
	 * @param {*} value The value to map to the given key
	 */
	LRU.prototype.put = function put(key, value) {
	    if(this._max !== -1 && this._size+1 > this._max) {
	        //shuffle one off the end
	       this.remove(this._end._v.key);
	    }
	    this.remove(key);  //torch the existing value
	    var entry = node(key, value);
	    if(!this._start) {
	        this._start = this._end = entry;
	    } else {
	        entry = node(key, value);
	        entry._n = this._start;
	        this._start._p = entry;
	        this._start = entry;
	    }
	    this._cache[key] = entry;
	    this._size++;
	};
	/**
	 * @description Gets the value from the map with the given key. Returns
	 * null if no mapping exists.
	 * @function
	 * @param {String} key The key to look up
	 * @returns {*} The value mapped to the given key
	 */
	LRU.prototype.get = function get(key) {
	    if(this._size > 0) {
	        var entry = this._cache[key];
	        if(entry && entry._v) {
	          return entry._v.value;
	        }
	    }
	    return null;
	};
 		/**
		  * @description Removes the key and mapped value from the map and returnns
		  * the removed value or null if nothign was removed.
		  * @function
		  * @param {String} key The key to remove
		  * @returns {*} The removed value or null
		  */
		 LRU.prototype.remove = function remove(key) {
 		    if(this._size === 0) {
 		        return null;
 		    }
 		    var entry = this._cache[key];
 		    if(entry) {
 		        var p = entry._p;
 		        if(this._end === entry) {
 		        	this._end = p;
 		        }
 		        var n = entry._n;
 		        if(this._start === entry) {
 		        	this._start = entry._n;
 		        }
 		        if(p) {
 		            p._n = n;
 		        }
 		        if(n) {
 		            n._p = p;
 		        }
 		        delete this._cache[key];
 		        this._size--;
 		        return entry._v.value;
 		    }
 		    return null;
 		};
 		/**
		  * @description Returns the array of keys found in the map in the order they were inserted,
		  * so for this LRU map the first key would be the oldest mapped value
		  * @function
		  * @returns {String[]} The keys in the map in insertion order
		  */
		 LRU.prototype.keys = function keys() {
		    var keys = [];
		    if(this._end) {
		       var n = this._end;
		       while(n) {
		           keys.push(n._v.key);
		           n = n._p;
		       }
		    }
		    return keys;
 		};
	
	return LRU;
});

/*******************************************************************************
 * @license
 * Copyright (c) 2013, 2015 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html).
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env amd*/
/**
 * @see http://wiki.eclipse.org/Orion/Dependency_resolution
 */
define('javascript/scriptResolver',[
'orion/objects',
'orion/Deferred',
'javascript/lru'
], function(Objects, Deferred, LRU) {

    /**
     * @name ScriptResolver
     * @description Creates a new script resolver for finding workspace file based
     * on a given logical path and search options
     * @param {orion.ServiceRegistry} serviceRegistry The service registry object
     * @constructor
     * @since 8.0
     */
    function ScriptResolver(serviceRegistry) {
        this.serviceRegistry = serviceRegistry;
        this.cache = new LRU(10);
    }

    Objects.mixin(ScriptResolver.prototype, {
       /**
        * Returns an array of workspace file that match the given logical name and options
        * @param {String} logicalName The name of the file to look up, for example, 'orion/objects'
        * @param {Object} options The map of search options.
        *
        * >Supported options include:
        * >  * ext - the file extension type to look for, for example 'js'
        * >  * icon - the URL or relative path to the icon to describe found files
        * >  * type - the name to use for the content type of any found files
        * 
        * @returns {File | null} Array of found files or ```null```
        */
       getWorkspaceFile : function getWorkspaceFile(logicalName, options) {
          if(logicalName) {
              return this._getFile(logicalName, options);
          }
          return new Deferred().resolve(null);
       },
	   getFileClient: function() {
	   		if(!this.fileclient) {
	   			this.fileclient = this.serviceRegistry.getService("orion.core.file.client"); //$NON-NLS-1$
	   		}
	   		return this.fileclient;
	   },
       setSearchLocation: function(searchLocation) {
       		this.searchLocation = searchLocation;
       },
	   getSearchLocation: function() {
	   		if(typeof this.searchLocation === 'string' && this.searchLocation.length > 0) {
	   			return new Deferred().resolve(this.searchLocation);
	   		}
	   		return this.getFileClient().fileServiceRootURL();
	   },
       _getFile : function _getFile(name, options) {
           var files = this.cache.get(name);
           if(files) {
               return new Deferred().resolve(files);
           }
           var that = this;
           var opts = options ? options : Object.create(null);
           var ext = opts.ext ? opts.ext : 'js'; //$NON-NLS-1$
           var icon = opts.icon ? opts.icon : '../javascript/images/javascript.png'; //$NON-NLS-1$
           var type = opts.type ? opts.type : 'JavaScript'; //$NON-NLS-1$
           var dotext = '.'+ext;
           var pref = this._removePrefix(name);
           var filename = pref.length > 1 ? pref[1] : pref[0];
           var idx = filename.lastIndexOf('/');
           var searchname = filename.slice(idx+1);

           // Search for it
           return this.getFileClient().search(
                {
                	'resource': that.searchLocation || "",
                    'keyword': searchname,
                    'sort': 'Name asc', //$NON-NLS-1$
                    'nameSearch': true,
                    'fileType': ext,
                    'start': 0,
                    'rows': 30
                }
           ).then(function(res) {
               var r = res.response;
               var len = r.docs.length;
               if(r.numFound > 0) {
                   files = [];
                   var testname = filename.replace(/(?:\.?\.\/)*/, '');
                   testname = testname.replace(new RegExp("\\"+dotext+"$"), ''); //$NON-NLS-1$
                   testname = testname.replace(/\//g, "\\/"); //$NON-NLS-1$
                   for(var i = 0; i < len; i++) {
                       var file = r.docs[i];
                       //TODO haxxor - only keep ones that end in the logical name or the mapped logical name
                       var regex = ".*(?:"+testname+")$"; //$NON-NLS-1$ //$NON-NLS-2$
                       if(new RegExp(regex).test(file.Location.slice(0, file.Location.length-dotext.length))) {
                           files.push(that._newFileObj(file.Name, file.Location, that._trimName(file.Path), icon, type));
                       }
                   }
                   if(files.length > 0) {
                       that.cache.put(filename, files);
                       return files;
                   }
               }
               return null;
           });
       },

       /**
        * @description Removes the prefix of a name a la requirejs
        * @param {String} name The name to remove the prefix from
        * @returns {Array.<String>} The array of prefix followed by the trimmed name, or an array with a single entry (if no prefix was removed).
        * @since 10.0
        */
       _removePrefix: function _removePrefix(name) {
       		var idx = name.indexOf('!');
       		if(idx > -1) {
       			return name.split('!');
       		}
  			return [name];
       },

       /**
        * @description Resolves the files that match the given location
        * @function
        * @param {String} path The path to resolve against
        * @param {Array} files The array of files
        * @param {Object} metadata The file metadata from the workspace
        * @returns {Array} The filtered list of files for the relative path or an empty array, never null
        * @since 8.0
        */
       resolveRelativeFiles: function resolveRelativeFiles(path, files, metadata) {
		    if(files && files.length > 0 && metadata) {
		        var filepath = metadata.location;
		        var _files = [];
		        var pref = this._removePrefix(path);
		        var _p = pref.length > 1 ? pref[1] : pref[0];
		        filepath = filepath.slice(0, filepath.lastIndexOf('/'));
		        var relative = false;
		        if(_p.charAt(0) !== '.') {
	                filepath = this._appendPath(filepath, _p);
	            } else {
	            	relative = true;
	                //resolve the realtive path
	                var rel = /^\.\.\//.exec(_p);
	                if(rel) {
    	                while(rel != null) {
    	                    filepath = filepath.slice(0, filepath.lastIndexOf('/'));
    	                    _p = _p.slice(3);
    	                    rel = /^\.\.\//.exec(_p);
    	                }
    	                filepath = this._appendPath(filepath, _p);
	                } else {
	                    while(/^\.\//.test(_p)) {
	                       _p = _p.slice(2);
	                    }
	                    filepath = this._appendPath(filepath, _p);
	                }
	            }
		        for(var i = 0; i < files.length; i++) {
		            var file = files[i];
		            var loc = file.location ? file.location : file.Location;
                    if(loc === filepath) {
                        _files.push(file);
                    } else if(this._samePaths(file, filepath, metadata))	 {
                    	_files.push(file);
                    } else if(!relative) {
                    	var idx = loc.lastIndexOf('.');
		       			var p1 = loc;
		       			if(idx > -1) {
			      			p1 = loc.slice(0, idx);
			      		}
			      		var _test = _p.replace(/[/?|{}()*.#$^]/g, '\\$&'); //$NON-NLS-1$
			      		var reg = new RegExp(_test+"$");
			      		if(reg.test(p1)) {
			      			_files.push(file);
			      		}
                    }
		        }
		        return _files;
		    }
		    return [];
		},

       /**
        * Returns if the two paths are the same
        * @param {String} file The first path
        * @param {String} path2 The second path
        * @returns {Boolean} If the paths are the same
        */
       _samePaths: function _samePaths(file, path2, meta) {
       		if(file == null) {
       			return path2 == null;
       		}
       		if(typeof(file) === 'undefined') {
       			return typeof(path2) === 'undefined';
       		}
       		if(path2 == null) {
       			return file == null;
       		}
       		if(typeof(path2) === 'undefined') {
       			return typeof(file) === 'undefined';
       		}
   			//get rid of extensions and compare the names
   			var loc = file.location ? file.location : file.Location;
   			if(!loc) {
   				return false;
   			}
   			var idx = loc.lastIndexOf('.');
   			var p1 = loc;
   			if(idx > -1) {
      			p1 = loc.slice(0, idx);
      		}
      		if(path2 === p1) {
      			return true; //could be that only the extension was missing from the other path
      		}
      		idx = path2.lastIndexOf('.');
   			var p2 = path2;
   			if(idx > -1) {
      			p2 = path2.slice(0, idx);
      		}
      		if(p1 === p2) {
      			return true;
      		} else if(p1 === decodeURIComponent(p2)) {
      			return true;
      		}
      		return false;
       },

       /**
        * @description Adds the additional path to the given path
        * @function
        * @private
        * @param {String} path The original path
        * @param {String} addition The additonal path to append
        * @returns {String | null} Returns the new path as a string or null if either of the parameters are not strings
        * @since 8.0
        */
       _appendPath: function _appendPath(path, addition) {
            if(typeof(path) === 'string' && typeof(addition) === 'string') {
                var newpath = path;
                if(newpath.charAt(newpath.length-1) !== '/') {
	               newpath += '/';
                }
                if(addition.charAt(0) === '/') {
                    newpath += addition.slice(1);
                } else {
                    newpath += addition;
                }
                return newpath;
            }
            return null;
       },

       _trimName: function _trimeName(name) {
           //TODO haxxor - we don't need to see the root client path
           return name.replace(/^(?:org\.eclipse\.orion\.client)?(?:\/)?bundles\//, '');
       },

       _newFileObj: function _newFileObj(name, location, path, icon, type) {
           var meta = Object.create(null);
           meta.name = name;
           meta.location = location;
           meta.path = path;
           meta.contentType = Object.create(null);
           if(icon) {
                meta.contentType.icon = icon;
           }
           if(type) {
                meta.contentType.name = type;
           }
           return meta;
       }
    });

    return {
        ScriptResolver: ScriptResolver
    };
});

/* eslint-env amd */
/* eslint-disable missing-nls */
define('htmlparser2/decodeCodepoint',[
], function() {
	return {"0":65533,"128":8364,"130":8218,"131":402,"132":8222,"133":8230,"134":8224,"135":8225,"136":710,"137":8240,"138":352,"139":8249,"140":338,"142":381,"145":8216,"146":8217,"147":8220,"148":8221,"149":8226,"150":8211,"151":8212,"152":732,"153":8482,"154":353,"155":8250,"156":339,"158":382,"159":376};
});
/* eslint-env amd */
/* eslint-disable missing-nls */
define('htmlparser2/entities',[
], function() {
	return {  
   "Aacute":"\u00C1",
   "aacute":"\u00E1",
   "Abreve":"\u0102",
   "abreve":"\u0103",
   "ac":"\u223E",
   "acd":"\u223F",
   "acE":"\u223E\u0333",
   "Acirc":"\u00C2",
   "acirc":"\u00E2",
   "acute":"\u00B4",
   "Acy":"\u0410",
   "acy":"\u0430",
   "AElig":"\u00C6",
   "aelig":"\u00E6",
   "af":"\u2061",
   "Afr":"\uD835\uDD04",
   "afr":"\uD835\uDD1E",
   "Agrave":"\u00C0",
   "agrave":"\u00E0",
   "alefsym":"\u2135",
   "aleph":"\u2135",
   "Alpha":"\u0391",
   "alpha":"\u03B1",
   "Amacr":"\u0100",
   "amacr":"\u0101",
   "amalg":"\u2A3F",
   "amp":"&",
   "AMP":"&",
   "andand":"\u2A55",
   "And":"\u2A53",
   "and":"\u2227",
   "andd":"\u2A5C",
   "andslope":"\u2A58",
   "andv":"\u2A5A",
   "ang":"\u2220",
   "ange":"\u29A4",
   "angle":"\u2220",
   "angmsdaa":"\u29A8",
   "angmsdab":"\u29A9",
   "angmsdac":"\u29AA",
   "angmsdad":"\u29AB",
   "angmsdae":"\u29AC",
   "angmsdaf":"\u29AD",
   "angmsdag":"\u29AE",
   "angmsdah":"\u29AF",
   "angmsd":"\u2221",
   "angrt":"\u221F",
   "angrtvb":"\u22BE",
   "angrtvbd":"\u299D",
   "angsph":"\u2222",
   "angst":"\u00C5",
   "angzarr":"\u237C",
   "Aogon":"\u0104",
   "aogon":"\u0105",
   "Aopf":"\uD835\uDD38",
   "aopf":"\uD835\uDD52",
   "apacir":"\u2A6F",
   "ap":"\u2248",
   "apE":"\u2A70",
   "ape":"\u224A",
   "apid":"\u224B",
   "apos":"'",
   "ApplyFunction":"\u2061",
   "approx":"\u2248",
   "approxeq":"\u224A",
   "Aring":"\u00C5",
   "aring":"\u00E5",
   "Ascr":"\uD835\uDC9C",
   "ascr":"\uD835\uDCB6",
   "Assign":"\u2254",
   "ast":"*",
   "asymp":"\u2248",
   "asympeq":"\u224D",
   "Atilde":"\u00C3",
   "atilde":"\u00E3",
   "Auml":"\u00C4",
   "auml":"\u00E4",
   "awconint":"\u2233",
   "awint":"\u2A11",
   "backcong":"\u224C",
   "backepsilon":"\u03F6",
   "backprime":"\u2035",
   "backsim":"\u223D",
   "backsimeq":"\u22CD",
   "Backslash":"\u2216",
   "Barv":"\u2AE7",
   "barvee":"\u22BD",
   "barwed":"\u2305",
   "Barwed":"\u2306",
   "barwedge":"\u2305",
   "bbrk":"\u23B5",
   "bbrktbrk":"\u23B6",
   "bcong":"\u224C",
   "Bcy":"\u0411",
   "bcy":"\u0431",
   "bdquo":"\u201E",
   "becaus":"\u2235",
   "because":"\u2235",
   "Because":"\u2235",
   "bemptyv":"\u29B0",
   "bepsi":"\u03F6",
   "bernou":"\u212C",
   "Bernoullis":"\u212C",
   "Beta":"\u0392",
   "beta":"\u03B2",
   "beth":"\u2136",
   "between":"\u226C",
   "Bfr":"\uD835\uDD05",
   "bfr":"\uD835\uDD1F",
   "bigcap":"\u22C2",
   "bigcirc":"\u25EF",
   "bigcup":"\u22C3",
   "bigodot":"\u2A00",
   "bigoplus":"\u2A01",
   "bigotimes":"\u2A02",
   "bigsqcup":"\u2A06",
   "bigstar":"\u2605",
   "bigtriangledown":"\u25BD",
   "bigtriangleup":"\u25B3",
   "biguplus":"\u2A04",
   "bigvee":"\u22C1",
   "bigwedge":"\u22C0",
   "bkarow":"\u290D",
   "blacklozenge":"\u29EB",
   "blacksquare":"\u25AA",
   "blacktriangle":"\u25B4",
   "blacktriangledown":"\u25BE",
   "blacktriangleleft":"\u25C2",
   "blacktriangleright":"\u25B8",
   "blank":"\u2423",
   "blk12":"\u2592",
   "blk14":"\u2591",
   "blk34":"\u2593",
   "block":"\u2588",
   "bne":"=\u20E5",
   "bnequiv":"\u2261\u20E5",
   "bNot":"\u2AED",
   "bnot":"\u2310",
   "Bopf":"\uD835\uDD39",
   "bopf":"\uD835\uDD53",
   "bot":"\u22A5",
   "bottom":"\u22A5",
   "bowtie":"\u22C8",
   "boxbox":"\u29C9",
   "boxdl":"\u2510",
   "boxdL":"\u2555",
   "boxDl":"\u2556",
   "boxDL":"\u2557",
   "boxdr":"\u250C",
   "boxdR":"\u2552",
   "boxDr":"\u2553",
   "boxDR":"\u2554",
   "boxh":"\u2500",
   "boxH":"\u2550",
   "boxhd":"\u252C",
   "boxHd":"\u2564",
   "boxhD":"\u2565",
   "boxHD":"\u2566",
   "boxhu":"\u2534",
   "boxHu":"\u2567",
   "boxhU":"\u2568",
   "boxHU":"\u2569",
   "boxminus":"\u229F",
   "boxplus":"\u229E",
   "boxtimes":"\u22A0",
   "boxul":"\u2518",
   "boxuL":"\u255B",
   "boxUl":"\u255C",
   "boxUL":"\u255D",
   "boxur":"\u2514",
   "boxuR":"\u2558",
   "boxUr":"\u2559",
   "boxUR":"\u255A",
   "boxv":"\u2502",
   "boxV":"\u2551",
   "boxvh":"\u253C",
   "boxvH":"\u256A",
   "boxVh":"\u256B",
   "boxVH":"\u256C",
   "boxvl":"\u2524",
   "boxvL":"\u2561",
   "boxVl":"\u2562",
   "boxVL":"\u2563",
   "boxvr":"\u251C",
   "boxvR":"\u255E",
   "boxVr":"\u255F",
   "boxVR":"\u2560",
   "bprime":"\u2035",
   "breve":"\u02D8",
   "Breve":"\u02D8",
   "brvbar":"\u00A6",
   "bscr":"\uD835\uDCB7",
   "Bscr":"\u212C",
   "bsemi":"\u204F",
   "bsim":"\u223D",
   "bsime":"\u22CD",
   "bsolb":"\u29C5",
   "bsol":"\\",
   "bsolhsub":"\u27C8",
   "bull":"\u2022",
   "bullet":"\u2022",
   "bump":"\u224E",
   "bumpE":"\u2AAE",
   "bumpe":"\u224F",
   "Bumpeq":"\u224E",
   "bumpeq":"\u224F",
   "Cacute":"\u0106",
   "cacute":"\u0107",
   "capand":"\u2A44",
   "capbrcup":"\u2A49",
   "capcap":"\u2A4B",
   "cap":"\u2229",
   "Cap":"\u22D2",
   "capcup":"\u2A47",
   "capdot":"\u2A40",
   "CapitalDifferentialD":"\u2145",
   "caps":"\u2229\uFE00",
   "caret":"\u2041",
   "caron":"\u02C7",
   "Cayleys":"\u212D",
   "ccaps":"\u2A4D",
   "Ccaron":"\u010C",
   "ccaron":"\u010D",
   "Ccedil":"\u00C7",
   "ccedil":"\u00E7",
   "Ccirc":"\u0108",
   "ccirc":"\u0109",
   "Cconint":"\u2230",
   "ccups":"\u2A4C",
   "ccupssm":"\u2A50",
   "Cdot":"\u010A",
   "cdot":"\u010B",
   "cedil":"\u00B8",
   "Cedilla":"\u00B8",
   "cemptyv":"\u29B2",
   "cent":"\u00A2",
   "centerdot":"\u00B7",
   "CenterDot":"\u00B7",
   "cfr":"\uD835\uDD20",
   "Cfr":"\u212D",
   "CHcy":"\u0427",
   "chcy":"\u0447",
   "check":"\u2713",
   "checkmark":"\u2713",
   "Chi":"\u03A7",
   "chi":"\u03C7",
   "circ":"\u02C6",
   "circeq":"\u2257",
   "circlearrowleft":"\u21BA",
   "circlearrowright":"\u21BB",
   "circledast":"\u229B",
   "circledcirc":"\u229A",
   "circleddash":"\u229D",
   "CircleDot":"\u2299",
   "circledR":"\u00AE",
   "circledS":"\u24C8",
   "CircleMinus":"\u2296",
   "CirclePlus":"\u2295",
   "CircleTimes":"\u2297",
   "cir":"\u25CB",
   "cirE":"\u29C3",
   "cire":"\u2257",
   "cirfnint":"\u2A10",
   "cirmid":"\u2AEF",
   "cirscir":"\u29C2",
   "ClockwiseContourIntegral":"\u2232",
   "CloseCurlyDoubleQuote":"\u201D",
   "CloseCurlyQuote":"\u2019",
   "clubs":"\u2663",
   "clubsuit":"\u2663",
   "colon":":",
   "Colon":"\u2237",
   "Colone":"\u2A74",
   "colone":"\u2254",
   "coloneq":"\u2254",
   "comma":",",
   "commat":"@",
   "comp":"\u2201",
   "compfn":"\u2218",
   "complement":"\u2201",
   "complexes":"\u2102",
   "cong":"\u2245",
   "congdot":"\u2A6D",
   "Congruent":"\u2261",
   "conint":"\u222E",
   "Conint":"\u222F",
   "ContourIntegral":"\u222E",
   "copf":"\uD835\uDD54",
   "Copf":"\u2102",
   "coprod":"\u2210",
   "Coproduct":"\u2210",
   "copy":"\u00A9",
   "COPY":"\u00A9",
   "copysr":"\u2117",
   "CounterClockwiseContourIntegral":"\u2233",
   "crarr":"\u21B5",
   "cross":"\u2717",
   "Cross":"\u2A2F",
   "Cscr":"\uD835\uDC9E",
   "cscr":"\uD835\uDCB8",
   "csub":"\u2ACF",
   "csube":"\u2AD1",
   "csup":"\u2AD0",
   "csupe":"\u2AD2",
   "ctdot":"\u22EF",
   "cudarrl":"\u2938",
   "cudarrr":"\u2935",
   "cuepr":"\u22DE",
   "cuesc":"\u22DF",
   "cularr":"\u21B6",
   "cularrp":"\u293D",
   "cupbrcap":"\u2A48",
   "cupcap":"\u2A46",
   "CupCap":"\u224D",
   "cup":"\u222A",
   "Cup":"\u22D3",
   "cupcup":"\u2A4A",
   "cupdot":"\u228D",
   "cupor":"\u2A45",
   "cups":"\u222A\uFE00",
   "curarr":"\u21B7",
   "curarrm":"\u293C",
   "curlyeqprec":"\u22DE",
   "curlyeqsucc":"\u22DF",
   "curlyvee":"\u22CE",
   "curlywedge":"\u22CF",
   "curren":"\u00A4",
   "curvearrowleft":"\u21B6",
   "curvearrowright":"\u21B7",
   "cuvee":"\u22CE",
   "cuwed":"\u22CF",
   "cwconint":"\u2232",
   "cwint":"\u2231",
   "cylcty":"\u232D",
   "dagger":"\u2020",
   "Dagger":"\u2021",
   "daleth":"\u2138",
   "darr":"\u2193",
   "Darr":"\u21A1",
   "dArr":"\u21D3",
   "dash":"\u2010",
   "Dashv":"\u2AE4",
   "dashv":"\u22A3",
   "dbkarow":"\u290F",
   "dblac":"\u02DD",
   "Dcaron":"\u010E",
   "dcaron":"\u010F",
   "Dcy":"\u0414",
   "dcy":"\u0434",
   "ddagger":"\u2021",
   "ddarr":"\u21CA",
   "DD":"\u2145",
   "dd":"\u2146",
   "DDotrahd":"\u2911",
   "ddotseq":"\u2A77",
   "deg":"\u00B0",
   "Del":"\u2207",
   "Delta":"\u0394",
   "delta":"\u03B4",
   "demptyv":"\u29B1",
   "dfisht":"\u297F",
   "Dfr":"\uD835\uDD07",
   "dfr":"\uD835\uDD21",
   "dHar":"\u2965",
   "dharl":"\u21C3",
   "dharr":"\u21C2",
   "DiacriticalAcute":"\u00B4",
   "DiacriticalDot":"\u02D9",
   "DiacriticalDoubleAcute":"\u02DD",
   "DiacriticalGrave":"`",
   "DiacriticalTilde":"\u02DC",
   "diam":"\u22C4",
   "diamond":"\u22C4",
   "Diamond":"\u22C4",
   "diamondsuit":"\u2666",
   "diams":"\u2666",
   "die":"\u00A8",
   "DifferentialD":"\u2146",
   "digamma":"\u03DD",
   "disin":"\u22F2",
   "div":"\u00F7",
   "divide":"\u00F7",
   "divideontimes":"\u22C7",
   "divonx":"\u22C7",
   "DJcy":"\u0402",
   "djcy":"\u0452",
   "dlcorn":"\u231E",
   "dlcrop":"\u230D",
   "dollar":"$",
   "Dopf":"\uD835\uDD3B",
   "dopf":"\uD835\uDD55",
   "Dot":"\u00A8",
   "dot":"\u02D9",
   "DotDot":"\u20DC",
   "doteq":"\u2250",
   "doteqdot":"\u2251",
   "DotEqual":"\u2250",
   "dotminus":"\u2238",
   "dotplus":"\u2214",
   "dotsquare":"\u22A1",
   "doublebarwedge":"\u2306",
   "DoubleContourIntegral":"\u222F",
   "DoubleDot":"\u00A8",
   "DoubleDownArrow":"\u21D3",
   "DoubleLeftArrow":"\u21D0",
   "DoubleLeftRightArrow":"\u21D4",
   "DoubleLeftTee":"\u2AE4",
   "DoubleLongLeftArrow":"\u27F8",
   "DoubleLongLeftRightArrow":"\u27FA",
   "DoubleLongRightArrow":"\u27F9",
   "DoubleRightArrow":"\u21D2",
   "DoubleRightTee":"\u22A8",
   "DoubleUpArrow":"\u21D1",
   "DoubleUpDownArrow":"\u21D5",
   "DoubleVerticalBar":"\u2225",
   "DownArrowBar":"\u2913",
   "downarrow":"\u2193",
   "DownArrow":"\u2193",
   "Downarrow":"\u21D3",
   "DownArrowUpArrow":"\u21F5",
   "DownBreve":"\u0311",
   "downdownarrows":"\u21CA",
   "downharpoonleft":"\u21C3",
   "downharpoonright":"\u21C2",
   "DownLeftRightVector":"\u2950",
   "DownLeftTeeVector":"\u295E",
   "DownLeftVectorBar":"\u2956",
   "DownLeftVector":"\u21BD",
   "DownRightTeeVector":"\u295F",
   "DownRightVectorBar":"\u2957",
   "DownRightVector":"\u21C1",
   "DownTeeArrow":"\u21A7",
   "DownTee":"\u22A4",
   "drbkarow":"\u2910",
   "drcorn":"\u231F",
   "drcrop":"\u230C",
   "Dscr":"\uD835\uDC9F",
   "dscr":"\uD835\uDCB9",
   "DScy":"\u0405",
   "dscy":"\u0455",
   "dsol":"\u29F6",
   "Dstrok":"\u0110",
   "dstrok":"\u0111",
   "dtdot":"\u22F1",
   "dtri":"\u25BF",
   "dtrif":"\u25BE",
   "duarr":"\u21F5",
   "duhar":"\u296F",
   "dwangle":"\u29A6",
   "DZcy":"\u040F",
   "dzcy":"\u045F",
   "dzigrarr":"\u27FF",
   "Eacute":"\u00C9",
   "eacute":"\u00E9",
   "easter":"\u2A6E",
   "Ecaron":"\u011A",
   "ecaron":"\u011B",
   "Ecirc":"\u00CA",
   "ecirc":"\u00EA",
   "ecir":"\u2256",
   "ecolon":"\u2255",
   "Ecy":"\u042D",
   "ecy":"\u044D",
   "eDDot":"\u2A77",
   "Edot":"\u0116",
   "edot":"\u0117",
   "eDot":"\u2251",
   "ee":"\u2147",
   "efDot":"\u2252",
   "Efr":"\uD835\uDD08",
   "efr":"\uD835\uDD22",
   "eg":"\u2A9A",
   "Egrave":"\u00C8",
   "egrave":"\u00E8",
   "egs":"\u2A96",
   "egsdot":"\u2A98",
   "el":"\u2A99",
   "Element":"\u2208",
   "elinters":"\u23E7",
   "ell":"\u2113",
   "els":"\u2A95",
   "elsdot":"\u2A97",
   "Emacr":"\u0112",
   "emacr":"\u0113",
   "empty":"\u2205",
   "emptyset":"\u2205",
   "EmptySmallSquare":"\u25FB",
   "emptyv":"\u2205",
   "EmptyVerySmallSquare":"\u25AB",
   "emsp13":"\u2004",
   "emsp14":"\u2005",
   "emsp":"\u2003",
   "ENG":"\u014A",
   "eng":"\u014B",
   "ensp":"\u2002",
   "Eogon":"\u0118",
   "eogon":"\u0119",
   "Eopf":"\uD835\uDD3C",
   "eopf":"\uD835\uDD56",
   "epar":"\u22D5",
   "eparsl":"\u29E3",
   "eplus":"\u2A71",
   "epsi":"\u03B5",
   "Epsilon":"\u0395",
   "epsilon":"\u03B5",
   "epsiv":"\u03F5",
   "eqcirc":"\u2256",
   "eqcolon":"\u2255",
   "eqsim":"\u2242",
   "eqslantgtr":"\u2A96",
   "eqslantless":"\u2A95",
   "Equal":"\u2A75",
   "equals":"=",
   "EqualTilde":"\u2242",
   "equest":"\u225F",
   "Equilibrium":"\u21CC",
   "equiv":"\u2261",
   "equivDD":"\u2A78",
   "eqvparsl":"\u29E5",
   "erarr":"\u2971",
   "erDot":"\u2253",
   "escr":"\u212F",
   "Escr":"\u2130",
   "esdot":"\u2250",
   "Esim":"\u2A73",
   "esim":"\u2242",
   "Eta":"\u0397",
   "eta":"\u03B7",
   "ETH":"\u00D0",
   "eth":"\u00F0",
   "Euml":"\u00CB",
   "euml":"\u00EB",
   "euro":"\u20AC",
   "excl":"!",
   "exist":"\u2203",
   "Exists":"\u2203",
   "expectation":"\u2130",
   "exponentiale":"\u2147",
   "ExponentialE":"\u2147",
   "fallingdotseq":"\u2252",
   "Fcy":"\u0424",
   "fcy":"\u0444",
   "female":"\u2640",
   "ffilig":"\uFB03",
   "fflig":"\uFB00",
   "ffllig":"\uFB04",
   "Ffr":"\uD835\uDD09",
   "ffr":"\uD835\uDD23",
   "filig":"\uFB01",
   "FilledSmallSquare":"\u25FC",
   "FilledVerySmallSquare":"\u25AA",
   "fjlig":"fj",
   "flat":"\u266D",
   "fllig":"\uFB02",
   "fltns":"\u25B1",
   "fnof":"\u0192",
   "Fopf":"\uD835\uDD3D",
   "fopf":"\uD835\uDD57",
   "forall":"\u2200",
   "ForAll":"\u2200",
   "fork":"\u22D4",
   "forkv":"\u2AD9",
   "Fouriertrf":"\u2131",
   "fpartint":"\u2A0D",
   "frac12":"\u00BD",
   "frac13":"\u2153",
   "frac14":"\u00BC",
   "frac15":"\u2155",
   "frac16":"\u2159",
   "frac18":"\u215B",
   "frac23":"\u2154",
   "frac25":"\u2156",
   "frac34":"\u00BE",
   "frac35":"\u2157",
   "frac38":"\u215C",
   "frac45":"\u2158",
   "frac56":"\u215A",
   "frac58":"\u215D",
   "frac78":"\u215E",
   "frasl":"\u2044",
   "frown":"\u2322",
   "fscr":"\uD835\uDCBB",
   "Fscr":"\u2131",
   "gacute":"\u01F5",
   "Gamma":"\u0393",
   "gamma":"\u03B3",
   "Gammad":"\u03DC",
   "gammad":"\u03DD",
   "gap":"\u2A86",
   "Gbreve":"\u011E",
   "gbreve":"\u011F",
   "Gcedil":"\u0122",
   "Gcirc":"\u011C",
   "gcirc":"\u011D",
   "Gcy":"\u0413",
   "gcy":"\u0433",
   "Gdot":"\u0120",
   "gdot":"\u0121",
   "ge":"\u2265",
   "gE":"\u2267",
   "gEl":"\u2A8C",
   "gel":"\u22DB",
   "geq":"\u2265",
   "geqq":"\u2267",
   "geqslant":"\u2A7E",
   "gescc":"\u2AA9",
   "ges":"\u2A7E",
   "gesdot":"\u2A80",
   "gesdoto":"\u2A82",
   "gesdotol":"\u2A84",
   "gesl":"\u22DB\uFE00",
   "gesles":"\u2A94",
   "Gfr":"\uD835\uDD0A",
   "gfr":"\uD835\uDD24",
   "gg":"\u226B",
   "Gg":"\u22D9",
   "ggg":"\u22D9",
   "gimel":"\u2137",
   "GJcy":"\u0403",
   "gjcy":"\u0453",
   "gla":"\u2AA5",
   "gl":"\u2277",
   "glE":"\u2A92",
   "glj":"\u2AA4",
   "gnap":"\u2A8A",
   "gnapprox":"\u2A8A",
   "gne":"\u2A88",
   "gnE":"\u2269",
   "gneq":"\u2A88",
   "gneqq":"\u2269",
   "gnsim":"\u22E7",
   "Gopf":"\uD835\uDD3E",
   "gopf":"\uD835\uDD58",
   "grave":"`",
   "GreaterEqual":"\u2265",
   "GreaterEqualLess":"\u22DB",
   "GreaterFullEqual":"\u2267",
   "GreaterGreater":"\u2AA2",
   "GreaterLess":"\u2277",
   "GreaterSlantEqual":"\u2A7E",
   "GreaterTilde":"\u2273",
   "Gscr":"\uD835\uDCA2",
   "gscr":"\u210A",
   "gsim":"\u2273",
   "gsime":"\u2A8E",
   "gsiml":"\u2A90",
   "gtcc":"\u2AA7",
   "gtcir":"\u2A7A",
   "gt":">",
   "GT":">",
   "Gt":"\u226B",
   "gtdot":"\u22D7",
   "gtlPar":"\u2995",
   "gtquest":"\u2A7C",
   "gtrapprox":"\u2A86",
   "gtrarr":"\u2978",
   "gtrdot":"\u22D7",
   "gtreqless":"\u22DB",
   "gtreqqless":"\u2A8C",
   "gtrless":"\u2277",
   "gtrsim":"\u2273",
   "gvertneqq":"\u2269\uFE00",
   "gvnE":"\u2269\uFE00",
   "Hacek":"\u02C7",
   "hairsp":"\u200A",
   "half":"\u00BD",
   "hamilt":"\u210B",
   "HARDcy":"\u042A",
   "hardcy":"\u044A",
   "harrcir":"\u2948",
   "harr":"\u2194",
   "hArr":"\u21D4",
   "harrw":"\u21AD",
   "Hat":"^",
   "hbar":"\u210F",
   "Hcirc":"\u0124",
   "hcirc":"\u0125",
   "hearts":"\u2665",
   "heartsuit":"\u2665",
   "hellip":"\u2026",
   "hercon":"\u22B9",
   "hfr":"\uD835\uDD25",
   "Hfr":"\u210C",
   "HilbertSpace":"\u210B",
   "hksearow":"\u2925",
   "hkswarow":"\u2926",
   "hoarr":"\u21FF",
   "homtht":"\u223B",
   "hookleftarrow":"\u21A9",
   "hookrightarrow":"\u21AA",
   "hopf":"\uD835\uDD59",
   "Hopf":"\u210D",
   "horbar":"\u2015",
   "HorizontalLine":"\u2500",
   "hscr":"\uD835\uDCBD",
   "Hscr":"\u210B",
   "hslash":"\u210F",
   "Hstrok":"\u0126",
   "hstrok":"\u0127",
   "HumpDownHump":"\u224E",
   "HumpEqual":"\u224F",
   "hybull":"\u2043",
   "hyphen":"\u2010",
   "Iacute":"\u00CD",
   "iacute":"\u00ED",
   "ic":"\u2063",
   "Icirc":"\u00CE",
   "icirc":"\u00EE",
   "Icy":"\u0418",
   "icy":"\u0438",
   "Idot":"\u0130",
   "IEcy":"\u0415",
   "iecy":"\u0435",
   "iexcl":"\u00A1",
   "iff":"\u21D4",
   "ifr":"\uD835\uDD26",
   "Ifr":"\u2111",
   "Igrave":"\u00CC",
   "igrave":"\u00EC",
   "ii":"\u2148",
   "iiiint":"\u2A0C",
   "iiint":"\u222D",
   "iinfin":"\u29DC",
   "iiota":"\u2129",
   "IJlig":"\u0132",
   "ijlig":"\u0133",
   "Imacr":"\u012A",
   "imacr":"\u012B",
   "image":"\u2111",
   "ImaginaryI":"\u2148",
   "imagline":"\u2110",
   "imagpart":"\u2111",
   "imath":"\u0131",
   "Im":"\u2111",
   "imof":"\u22B7",
   "imped":"\u01B5",
   "Implies":"\u21D2",
   "incare":"\u2105",
   "in":"\u2208",
   "infin":"\u221E",
   "infintie":"\u29DD",
   "inodot":"\u0131",
   "intcal":"\u22BA",
   "int":"\u222B",
   "Int":"\u222C",
   "integers":"\u2124",
   "Integral":"\u222B",
   "intercal":"\u22BA",
   "Intersection":"\u22C2",
   "intlarhk":"\u2A17",
   "intprod":"\u2A3C",
   "InvisibleComma":"\u2063",
   "InvisibleTimes":"\u2062",
   "IOcy":"\u0401",
   "iocy":"\u0451",
   "Iogon":"\u012E",
   "iogon":"\u012F",
   "Iopf":"\uD835\uDD40",
   "iopf":"\uD835\uDD5A",
   "Iota":"\u0399",
   "iota":"\u03B9",
   "iprod":"\u2A3C",
   "iquest":"\u00BF",
   "iscr":"\uD835\uDCBE",
   "Iscr":"\u2110",
   "isin":"\u2208",
   "isindot":"\u22F5",
   "isinE":"\u22F9",
   "isins":"\u22F4",
   "isinsv":"\u22F3",
   "isinv":"\u2208",
   "it":"\u2062",
   "Itilde":"\u0128",
   "itilde":"\u0129",
   "Iukcy":"\u0406",
   "iukcy":"\u0456",
   "Iuml":"\u00CF",
   "iuml":"\u00EF",
   "Jcirc":"\u0134",
   "jcirc":"\u0135",
   "Jcy":"\u0419",
   "jcy":"\u0439",
   "Jfr":"\uD835\uDD0D",
   "jfr":"\uD835\uDD27",
   "jmath":"\u0237",
   "Jopf":"\uD835\uDD41",
   "jopf":"\uD835\uDD5B",
   "Jscr":"\uD835\uDCA5",
   "jscr":"\uD835\uDCBF",
   "Jsercy":"\u0408",
   "jsercy":"\u0458",
   "Jukcy":"\u0404",
   "jukcy":"\u0454",
   "Kappa":"\u039A",
   "kappa":"\u03BA",
   "kappav":"\u03F0",
   "Kcedil":"\u0136",
   "kcedil":"\u0137",
   "Kcy":"\u041A",
   "kcy":"\u043A",
   "Kfr":"\uD835\uDD0E",
   "kfr":"\uD835\uDD28",
   "kgreen":"\u0138",
   "KHcy":"\u0425",
   "khcy":"\u0445",
   "KJcy":"\u040C",
   "kjcy":"\u045C",
   "Kopf":"\uD835\uDD42",
   "kopf":"\uD835\uDD5C",
   "Kscr":"\uD835\uDCA6",
   "kscr":"\uD835\uDCC0",
   "lAarr":"\u21DA",
   "Lacute":"\u0139",
   "lacute":"\u013A",
   "laemptyv":"\u29B4",
   "lagran":"\u2112",
   "Lambda":"\u039B",
   "lambda":"\u03BB",
   "lang":"\u27E8",
   "Lang":"\u27EA",
   "langd":"\u2991",
   "langle":"\u27E8",
   "lap":"\u2A85",
   "Laplacetrf":"\u2112",
   "laquo":"\u00AB",
   "larrb":"\u21E4",
   "larrbfs":"\u291F",
   "larr":"\u2190",
   "Larr":"\u219E",
   "lArr":"\u21D0",
   "larrfs":"\u291D",
   "larrhk":"\u21A9",
   "larrlp":"\u21AB",
   "larrpl":"\u2939",
   "larrsim":"\u2973",
   "larrtl":"\u21A2",
   "latail":"\u2919",
   "lAtail":"\u291B",
   "lat":"\u2AAB",
   "late":"\u2AAD",
   "lates":"\u2AAD\uFE00",
   "lbarr":"\u290C",
   "lBarr":"\u290E",
   "lbbrk":"\u2772",
   "lbrace":"{",
   "lbrack":"[",
   "lbrke":"\u298B",
   "lbrksld":"\u298F",
   "lbrkslu":"\u298D",
   "Lcaron":"\u013D",
   "lcaron":"\u013E",
   "Lcedil":"\u013B",
   "lcedil":"\u013C",
   "lceil":"\u2308",
   "lcub":"{",
   "Lcy":"\u041B",
   "lcy":"\u043B",
   "ldca":"\u2936",
   "ldquo":"\u201C",
   "ldquor":"\u201E",
   "ldrdhar":"\u2967",
   "ldrushar":"\u294B",
   "ldsh":"\u21B2",
   "le":"\u2264",
   "lE":"\u2266",
   "LeftAngleBracket":"\u27E8",
   "LeftArrowBar":"\u21E4",
   "leftarrow":"\u2190",
   "LeftArrow":"\u2190",
   "Leftarrow":"\u21D0",
   "LeftArrowRightArrow":"\u21C6",
   "leftarrowtail":"\u21A2",
   "LeftCeiling":"\u2308",
   "LeftDoubleBracket":"\u27E6",
   "LeftDownTeeVector":"\u2961",
   "LeftDownVectorBar":"\u2959",
   "LeftDownVector":"\u21C3",
   "LeftFloor":"\u230A",
   "leftharpoondown":"\u21BD",
   "leftharpoonup":"\u21BC",
   "leftleftarrows":"\u21C7",
   "leftrightarrow":"\u2194",
   "LeftRightArrow":"\u2194",
   "Leftrightarrow":"\u21D4",
   "leftrightarrows":"\u21C6",
   "leftrightharpoons":"\u21CB",
   "leftrightsquigarrow":"\u21AD",
   "LeftRightVector":"\u294E",
   "LeftTeeArrow":"\u21A4",
   "LeftTee":"\u22A3",
   "LeftTeeVector":"\u295A",
   "leftthreetimes":"\u22CB",
   "LeftTriangleBar":"\u29CF",
   "LeftTriangle":"\u22B2",
   "LeftTriangleEqual":"\u22B4",
   "LeftUpDownVector":"\u2951",
   "LeftUpTeeVector":"\u2960",
   "LeftUpVectorBar":"\u2958",
   "LeftUpVector":"\u21BF",
   "LeftVectorBar":"\u2952",
   "LeftVector":"\u21BC",
   "lEg":"\u2A8B",
   "leg":"\u22DA",
   "leq":"\u2264",
   "leqq":"\u2266",
   "leqslant":"\u2A7D",
   "lescc":"\u2AA8",
   "les":"\u2A7D",
   "lesdot":"\u2A7F",
   "lesdoto":"\u2A81",
   "lesdotor":"\u2A83",
   "lesg":"\u22DA\uFE00",
   "lesges":"\u2A93",
   "lessapprox":"\u2A85",
   "lessdot":"\u22D6",
   "lesseqgtr":"\u22DA",
   "lesseqqgtr":"\u2A8B",
   "LessEqualGreater":"\u22DA",
   "LessFullEqual":"\u2266",
   "LessGreater":"\u2276",
   "lessgtr":"\u2276",
   "LessLess":"\u2AA1",
   "lesssim":"\u2272",
   "LessSlantEqual":"\u2A7D",
   "LessTilde":"\u2272",
   "lfisht":"\u297C",
   "lfloor":"\u230A",
   "Lfr":"\uD835\uDD0F",
   "lfr":"\uD835\uDD29",
   "lg":"\u2276",
   "lgE":"\u2A91",
   "lHar":"\u2962",
   "lhard":"\u21BD",
   "lharu":"\u21BC",
   "lharul":"\u296A",
   "lhblk":"\u2584",
   "LJcy":"\u0409",
   "ljcy":"\u0459",
   "llarr":"\u21C7",
   "ll":"\u226A",
   "Ll":"\u22D8",
   "llcorner":"\u231E",
   "Lleftarrow":"\u21DA",
   "llhard":"\u296B",
   "lltri":"\u25FA",
   "Lmidot":"\u013F",
   "lmidot":"\u0140",
   "lmoustache":"\u23B0",
   "lmoust":"\u23B0",
   "lnap":"\u2A89",
   "lnapprox":"\u2A89",
   "lne":"\u2A87",
   "lnE":"\u2268",
   "lneq":"\u2A87",
   "lneqq":"\u2268",
   "lnsim":"\u22E6",
   "loang":"\u27EC",
   "loarr":"\u21FD",
   "lobrk":"\u27E6",
   "longleftarrow":"\u27F5",
   "LongLeftArrow":"\u27F5",
   "Longleftarrow":"\u27F8",
   "longleftrightarrow":"\u27F7",
   "LongLeftRightArrow":"\u27F7",
   "Longleftrightarrow":"\u27FA",
   "longmapsto":"\u27FC",
   "longrightarrow":"\u27F6",
   "LongRightArrow":"\u27F6",
   "Longrightarrow":"\u27F9",
   "looparrowleft":"\u21AB",
   "looparrowright":"\u21AC",
   "lopar":"\u2985",
   "Lopf":"\uD835\uDD43",
   "lopf":"\uD835\uDD5D",
   "loplus":"\u2A2D",
   "lotimes":"\u2A34",
   "lowast":"\u2217",
   "lowbar":"_",
   "LowerLeftArrow":"\u2199",
   "LowerRightArrow":"\u2198",
   "loz":"\u25CA",
   "lozenge":"\u25CA",
   "lozf":"\u29EB",
   "lpar":"(",
   "lparlt":"\u2993",
   "lrarr":"\u21C6",
   "lrcorner":"\u231F",
   "lrhar":"\u21CB",
   "lrhard":"\u296D",
   "lrm":"\u200E",
   "lrtri":"\u22BF",
   "lsaquo":"\u2039",
   "lscr":"\uD835\uDCC1",
   "Lscr":"\u2112",
   "lsh":"\u21B0",
   "Lsh":"\u21B0",
   "lsim":"\u2272",
   "lsime":"\u2A8D",
   "lsimg":"\u2A8F",
   "lsqb":"[",
   "lsquo":"\u2018",
   "lsquor":"\u201A",
   "Lstrok":"\u0141",
   "lstrok":"\u0142",
   "ltcc":"\u2AA6",
   "ltcir":"\u2A79",
   "lt":"<",
   "LT":"<",
   "Lt":"\u226A",
   "ltdot":"\u22D6",
   "lthree":"\u22CB",
   "ltimes":"\u22C9",
   "ltlarr":"\u2976",
   "ltquest":"\u2A7B",
   "ltri":"\u25C3",
   "ltrie":"\u22B4",
   "ltrif":"\u25C2",
   "ltrPar":"\u2996",
   "lurdshar":"\u294A",
   "luruhar":"\u2966",
   "lvertneqq":"\u2268\uFE00",
   "lvnE":"\u2268\uFE00",
   "macr":"\u00AF",
   "male":"\u2642",
   "malt":"\u2720",
   "maltese":"\u2720",
   "Map":"\u2905",
   "map":"\u21A6",
   "mapsto":"\u21A6",
   "mapstodown":"\u21A7",
   "mapstoleft":"\u21A4",
   "mapstoup":"\u21A5",
   "marker":"\u25AE",
   "mcomma":"\u2A29",
   "Mcy":"\u041C",
   "mcy":"\u043C",
   "mdash":"\u2014",
   "mDDot":"\u223A",
   "measuredangle":"\u2221",
   "MediumSpace":"\u205F",
   "Mellintrf":"\u2133",
   "Mfr":"\uD835\uDD10",
   "mfr":"\uD835\uDD2A",
   "mho":"\u2127",
   "micro":"\u00B5",
   "midast":"*",
   "midcir":"\u2AF0",
   "mid":"\u2223",
   "middot":"\u00B7",
   "minusb":"\u229F",
   "minus":"\u2212",
   "minusd":"\u2238",
   "minusdu":"\u2A2A",
   "MinusPlus":"\u2213",
   "mlcp":"\u2ADB",
   "mldr":"\u2026",
   "mnplus":"\u2213",
   "models":"\u22A7",
   "Mopf":"\uD835\uDD44",
   "mopf":"\uD835\uDD5E",
   "mp":"\u2213",
   "mscr":"\uD835\uDCC2",
   "Mscr":"\u2133",
   "mstpos":"\u223E",
   "Mu":"\u039C",
   "mu":"\u03BC",
   "multimap":"\u22B8",
   "mumap":"\u22B8",
   "nabla":"\u2207",
   "Nacute":"\u0143",
   "nacute":"\u0144",
   "nang":"\u2220\u20D2",
   "nap":"\u2249",
   "napE":"\u2A70\u0338",
   "napid":"\u224B\u0338",
   "napos":"\u0149",
   "napprox":"\u2249",
   "natural":"\u266E",
   "naturals":"\u2115",
   "natur":"\u266E",
   "nbsp":"\u00A0",
   "nbump":"\u224E\u0338",
   "nbumpe":"\u224F\u0338",
   "ncap":"\u2A43",
   "Ncaron":"\u0147",
   "ncaron":"\u0148",
   "Ncedil":"\u0145",
   "ncedil":"\u0146",
   "ncong":"\u2247",
   "ncongdot":"\u2A6D\u0338",
   "ncup":"\u2A42",
   "Ncy":"\u041D",
   "ncy":"\u043D",
   "ndash":"\u2013",
   "nearhk":"\u2924",
   "nearr":"\u2197",
   "neArr":"\u21D7",
   "nearrow":"\u2197",
   "ne":"\u2260",
   "nedot":"\u2250\u0338",
   "NegativeMediumSpace":"\u200B",
   "NegativeThickSpace":"\u200B",
   "NegativeThinSpace":"\u200B",
   "NegativeVeryThinSpace":"\u200B",
   "nequiv":"\u2262",
   "nesear":"\u2928",
   "nesim":"\u2242\u0338",
   "NestedGreaterGreater":"\u226B",
   "NestedLessLess":"\u226A",
   "NewLine":"\n",
   "nexist":"\u2204",
   "nexists":"\u2204",
   "Nfr":"\uD835\uDD11",
   "nfr":"\uD835\uDD2B",
   "ngE":"\u2267\u0338",
   "nge":"\u2271",
   "ngeq":"\u2271",
   "ngeqq":"\u2267\u0338",
   "ngeqslant":"\u2A7E\u0338",
   "nges":"\u2A7E\u0338",
   "nGg":"\u22D9\u0338",
   "ngsim":"\u2275",
   "nGt":"\u226B\u20D2",
   "ngt":"\u226F",
   "ngtr":"\u226F",
   "nGtv":"\u226B\u0338",
   "nharr":"\u21AE",
   "nhArr":"\u21CE",
   "nhpar":"\u2AF2",
   "ni":"\u220B",
   "nis":"\u22FC",
   "nisd":"\u22FA",
   "niv":"\u220B",
   "NJcy":"\u040A",
   "njcy":"\u045A",
   "nlarr":"\u219A",
   "nlArr":"\u21CD",
   "nldr":"\u2025",
   "nlE":"\u2266\u0338",
   "nle":"\u2270",
   "nleftarrow":"\u219A",
   "nLeftarrow":"\u21CD",
   "nleftrightarrow":"\u21AE",
   "nLeftrightarrow":"\u21CE",
   "nleq":"\u2270",
   "nleqq":"\u2266\u0338",
   "nleqslant":"\u2A7D\u0338",
   "nles":"\u2A7D\u0338",
   "nless":"\u226E",
   "nLl":"\u22D8\u0338",
   "nlsim":"\u2274",
   "nLt":"\u226A\u20D2",
   "nlt":"\u226E",
   "nltri":"\u22EA",
   "nltrie":"\u22EC",
   "nLtv":"\u226A\u0338",
   "nmid":"\u2224",
   "NoBreak":"\u2060",
   "NonBreakingSpace":"\u00A0",
   "nopf":"\uD835\uDD5F",
   "Nopf":"\u2115",
   "Not":"\u2AEC",
   "not":"\u00AC",
   "NotCongruent":"\u2262",
   "NotCupCap":"\u226D",
   "NotDoubleVerticalBar":"\u2226",
   "NotElement":"\u2209",
   "NotEqual":"\u2260",
   "NotEqualTilde":"\u2242\u0338",
   "NotExists":"\u2204",
   "NotGreater":"\u226F",
   "NotGreaterEqual":"\u2271",
   "NotGreaterFullEqual":"\u2267\u0338",
   "NotGreaterGreater":"\u226B\u0338",
   "NotGreaterLess":"\u2279",
   "NotGreaterSlantEqual":"\u2A7E\u0338",
   "NotGreaterTilde":"\u2275",
   "NotHumpDownHump":"\u224E\u0338",
   "NotHumpEqual":"\u224F\u0338",
   "notin":"\u2209",
   "notindot":"\u22F5\u0338",
   "notinE":"\u22F9\u0338",
   "notinva":"\u2209",
   "notinvb":"\u22F7",
   "notinvc":"\u22F6",
   "NotLeftTriangleBar":"\u29CF\u0338",
   "NotLeftTriangle":"\u22EA",
   "NotLeftTriangleEqual":"\u22EC",
   "NotLess":"\u226E",
   "NotLessEqual":"\u2270",
   "NotLessGreater":"\u2278",
   "NotLessLess":"\u226A\u0338",
   "NotLessSlantEqual":"\u2A7D\u0338",
   "NotLessTilde":"\u2274",
   "NotNestedGreaterGreater":"\u2AA2\u0338",
   "NotNestedLessLess":"\u2AA1\u0338",
   "notni":"\u220C",
   "notniva":"\u220C",
   "notnivb":"\u22FE",
   "notnivc":"\u22FD",
   "NotPrecedes":"\u2280",
   "NotPrecedesEqual":"\u2AAF\u0338",
   "NotPrecedesSlantEqual":"\u22E0",
   "NotReverseElement":"\u220C",
   "NotRightTriangleBar":"\u29D0\u0338",
   "NotRightTriangle":"\u22EB",
   "NotRightTriangleEqual":"\u22ED",
   "NotSquareSubset":"\u228F\u0338",
   "NotSquareSubsetEqual":"\u22E2",
   "NotSquareSuperset":"\u2290\u0338",
   "NotSquareSupersetEqual":"\u22E3",
   "NotSubset":"\u2282\u20D2",
   "NotSubsetEqual":"\u2288",
   "NotSucceeds":"\u2281",
   "NotSucceedsEqual":"\u2AB0\u0338",
   "NotSucceedsSlantEqual":"\u22E1",
   "NotSucceedsTilde":"\u227F\u0338",
   "NotSuperset":"\u2283\u20D2",
   "NotSupersetEqual":"\u2289",
   "NotTilde":"\u2241",
   "NotTildeEqual":"\u2244",
   "NotTildeFullEqual":"\u2247",
   "NotTildeTilde":"\u2249",
   "NotVerticalBar":"\u2224",
   "nparallel":"\u2226",
   "npar":"\u2226",
   "nparsl":"\u2AFD\u20E5",
   "npart":"\u2202\u0338",
   "npolint":"\u2A14",
   "npr":"\u2280",
   "nprcue":"\u22E0",
   "nprec":"\u2280",
   "npreceq":"\u2AAF\u0338",
   "npre":"\u2AAF\u0338",
   "nrarrc":"\u2933\u0338",
   "nrarr":"\u219B",
   "nrArr":"\u21CF",
   "nrarrw":"\u219D\u0338",
   "nrightarrow":"\u219B",
   "nRightarrow":"\u21CF",
   "nrtri":"\u22EB",
   "nrtrie":"\u22ED",
   "nsc":"\u2281",
   "nsccue":"\u22E1",
   "nsce":"\u2AB0\u0338",
   "Nscr":"\uD835\uDCA9",
   "nscr":"\uD835\uDCC3",
   "nshortmid":"\u2224",
   "nshortparallel":"\u2226",
   "nsim":"\u2241",
   "nsime":"\u2244",
   "nsimeq":"\u2244",
   "nsmid":"\u2224",
   "nspar":"\u2226",
   "nsqsube":"\u22E2",
   "nsqsupe":"\u22E3",
   "nsub":"\u2284",
   "nsubE":"\u2AC5\u0338",
   "nsube":"\u2288",
   "nsubset":"\u2282\u20D2",
   "nsubseteq":"\u2288",
   "nsubseteqq":"\u2AC5\u0338",
   "nsucc":"\u2281",
   "nsucceq":"\u2AB0\u0338",
   "nsup":"\u2285",
   "nsupE":"\u2AC6\u0338",
   "nsupe":"\u2289",
   "nsupset":"\u2283\u20D2",
   "nsupseteq":"\u2289",
   "nsupseteqq":"\u2AC6\u0338",
   "ntgl":"\u2279",
   "Ntilde":"\u00D1",
   "ntilde":"\u00F1",
   "ntlg":"\u2278",
   "ntriangleleft":"\u22EA",
   "ntrianglelefteq":"\u22EC",
   "ntriangleright":"\u22EB",
   "ntrianglerighteq":"\u22ED",
   "Nu":"\u039D",
   "nu":"\u03BD",
   "num":"#",
   "numero":"\u2116",
   "numsp":"\u2007",
   "nvap":"\u224D\u20D2",
   "nvdash":"\u22AC",
   "nvDash":"\u22AD",
   "nVdash":"\u22AE",
   "nVDash":"\u22AF",
   "nvge":"\u2265\u20D2",
   "nvgt":">\u20D2",
   "nvHarr":"\u2904",
   "nvinfin":"\u29DE",
   "nvlArr":"\u2902",
   "nvle":"\u2264\u20D2",
   "nvlt":"<\u20D2",
   "nvltrie":"\u22B4\u20D2",
   "nvrArr":"\u2903",
   "nvrtrie":"\u22B5\u20D2",
   "nvsim":"\u223C\u20D2",
   "nwarhk":"\u2923",
   "nwarr":"\u2196",
   "nwArr":"\u21D6",
   "nwarrow":"\u2196",
   "nwnear":"\u2927",
   "Oacute":"\u00D3",
   "oacute":"\u00F3",
   "oast":"\u229B",
   "Ocirc":"\u00D4",
   "ocirc":"\u00F4",
   "ocir":"\u229A",
   "Ocy":"\u041E",
   "ocy":"\u043E",
   "odash":"\u229D",
   "Odblac":"\u0150",
   "odblac":"\u0151",
   "odiv":"\u2A38",
   "odot":"\u2299",
   "odsold":"\u29BC",
   "OElig":"\u0152",
   "oelig":"\u0153",
   "ofcir":"\u29BF",
   "Ofr":"\uD835\uDD12",
   "ofr":"\uD835\uDD2C",
   "ogon":"\u02DB",
   "Ograve":"\u00D2",
   "ograve":"\u00F2",
   "ogt":"\u29C1",
   "ohbar":"\u29B5",
   "ohm":"\u03A9",
   "oint":"\u222E",
   "olarr":"\u21BA",
   "olcir":"\u29BE",
   "olcross":"\u29BB",
   "oline":"\u203E",
   "olt":"\u29C0",
   "Omacr":"\u014C",
   "omacr":"\u014D",
   "Omega":"\u03A9",
   "omega":"\u03C9",
   "Omicron":"\u039F",
   "omicron":"\u03BF",
   "omid":"\u29B6",
   "ominus":"\u2296",
   "Oopf":"\uD835\uDD46",
   "oopf":"\uD835\uDD60",
   "opar":"\u29B7",
   "OpenCurlyDoubleQuote":"\u201C",
   "OpenCurlyQuote":"\u2018",
   "operp":"\u29B9",
   "oplus":"\u2295",
   "orarr":"\u21BB",
   "Or":"\u2A54",
   "or":"\u2228",
   "ord":"\u2A5D",
   "order":"\u2134",
   "orderof":"\u2134",
   "ordf":"\u00AA",
   "ordm":"\u00BA",
   "origof":"\u22B6",
   "oror":"\u2A56",
   "orslope":"\u2A57",
   "orv":"\u2A5B",
   "oS":"\u24C8",
   "Oscr":"\uD835\uDCAA",
   "oscr":"\u2134",
   "Oslash":"\u00D8",
   "oslash":"\u00F8",
   "osol":"\u2298",
   "Otilde":"\u00D5",
   "otilde":"\u00F5",
   "otimesas":"\u2A36",
   "Otimes":"\u2A37",
   "otimes":"\u2297",
   "Ouml":"\u00D6",
   "ouml":"\u00F6",
   "ovbar":"\u233D",
   "OverBar":"\u203E",
   "OverBrace":"\u23DE",
   "OverBracket":"\u23B4",
   "OverParenthesis":"\u23DC",
   "para":"\u00B6",
   "parallel":"\u2225",
   "par":"\u2225",
   "parsim":"\u2AF3",
   "parsl":"\u2AFD",
   "part":"\u2202",
   "PartialD":"\u2202",
   "Pcy":"\u041F",
   "pcy":"\u043F",
   "percnt":"%",
   "period":".",
   "permil":"\u2030",
   "perp":"\u22A5",
   "pertenk":"\u2031",
   "Pfr":"\uD835\uDD13",
   "pfr":"\uD835\uDD2D",
   "Phi":"\u03A6",
   "phi":"\u03C6",
   "phiv":"\u03D5",
   "phmmat":"\u2133",
   "phone":"\u260E",
   "Pi":"\u03A0",
   "pi":"\u03C0",
   "pitchfork":"\u22D4",
   "piv":"\u03D6",
   "planck":"\u210F",
   "planckh":"\u210E",
   "plankv":"\u210F",
   "plusacir":"\u2A23",
   "plusb":"\u229E",
   "pluscir":"\u2A22",
   "plus":"+",
   "plusdo":"\u2214",
   "plusdu":"\u2A25",
   "pluse":"\u2A72",
   "PlusMinus":"\u00B1",
   "plusmn":"\u00B1",
   "plussim":"\u2A26",
   "plustwo":"\u2A27",
   "pm":"\u00B1",
   "Poincareplane":"\u210C",
   "pointint":"\u2A15",
   "popf":"\uD835\uDD61",
   "Popf":"\u2119",
   "pound":"\u00A3",
   "prap":"\u2AB7",
   "Pr":"\u2ABB",
   "pr":"\u227A",
   "prcue":"\u227C",
   "precapprox":"\u2AB7",
   "prec":"\u227A",
   "preccurlyeq":"\u227C",
   "Precedes":"\u227A",
   "PrecedesEqual":"\u2AAF",
   "PrecedesSlantEqual":"\u227C",
   "PrecedesTilde":"\u227E",
   "preceq":"\u2AAF",
   "precnapprox":"\u2AB9",
   "precneqq":"\u2AB5",
   "precnsim":"\u22E8",
   "pre":"\u2AAF",
   "prE":"\u2AB3",
   "precsim":"\u227E",
   "prime":"\u2032",
   "Prime":"\u2033",
   "primes":"\u2119",
   "prnap":"\u2AB9",
   "prnE":"\u2AB5",
   "prnsim":"\u22E8",
   "prod":"\u220F",
   "Product":"\u220F",
   "profalar":"\u232E",
   "profline":"\u2312",
   "profsurf":"\u2313",
   "prop":"\u221D",
   "Proportional":"\u221D",
   "Proportion":"\u2237",
   "propto":"\u221D",
   "prsim":"\u227E",
   "prurel":"\u22B0",
   "Pscr":"\uD835\uDCAB",
   "pscr":"\uD835\uDCC5",
   "Psi":"\u03A8",
   "psi":"\u03C8",
   "puncsp":"\u2008",
   "Qfr":"\uD835\uDD14",
   "qfr":"\uD835\uDD2E",
   "qint":"\u2A0C",
   "qopf":"\uD835\uDD62",
   "Qopf":"\u211A",
   "qprime":"\u2057",
   "Qscr":"\uD835\uDCAC",
   "qscr":"\uD835\uDCC6",
   "quaternions":"\u210D",
   "quatint":"\u2A16",
   "quest":"?",
   "questeq":"\u225F",
   "quot":"\"",
   "QUOT":"\"",
   "rAarr":"\u21DB",
   "race":"\u223D\u0331",
   "Racute":"\u0154",
   "racute":"\u0155",
   "radic":"\u221A",
   "raemptyv":"\u29B3",
   "rang":"\u27E9",
   "Rang":"\u27EB",
   "rangd":"\u2992",
   "range":"\u29A5",
   "rangle":"\u27E9",
   "raquo":"\u00BB",
   "rarrap":"\u2975",
   "rarrb":"\u21E5",
   "rarrbfs":"\u2920",
   "rarrc":"\u2933",
   "rarr":"\u2192",
   "Rarr":"\u21A0",
   "rArr":"\u21D2",
   "rarrfs":"\u291E",
   "rarrhk":"\u21AA",
   "rarrlp":"\u21AC",
   "rarrpl":"\u2945",
   "rarrsim":"\u2974",
   "Rarrtl":"\u2916",
   "rarrtl":"\u21A3",
   "rarrw":"\u219D",
   "ratail":"\u291A",
   "rAtail":"\u291C",
   "ratio":"\u2236",
   "rationals":"\u211A",
   "rbarr":"\u290D",
   "rBarr":"\u290F",
   "RBarr":"\u2910",
   "rbbrk":"\u2773",
   "rbrace":"}",
   "rbrack":"]",
   "rbrke":"\u298C",
   "rbrksld":"\u298E",
   "rbrkslu":"\u2990",
   "Rcaron":"\u0158",
   "rcaron":"\u0159",
   "Rcedil":"\u0156",
   "rcedil":"\u0157",
   "rceil":"\u2309",
   "rcub":"}",
   "Rcy":"\u0420",
   "rcy":"\u0440",
   "rdca":"\u2937",
   "rdldhar":"\u2969",
   "rdquo":"\u201D",
   "rdquor":"\u201D",
   "rdsh":"\u21B3",
   "real":"\u211C",
   "realine":"\u211B",
   "realpart":"\u211C",
   "reals":"\u211D",
   "Re":"\u211C",
   "rect":"\u25AD",
   "reg":"\u00AE",
   "REG":"\u00AE",
   "ReverseElement":"\u220B",
   "ReverseEquilibrium":"\u21CB",
   "ReverseUpEquilibrium":"\u296F",
   "rfisht":"\u297D",
   "rfloor":"\u230B",
   "rfr":"\uD835\uDD2F",
   "Rfr":"\u211C",
   "rHar":"\u2964",
   "rhard":"\u21C1",
   "rharu":"\u21C0",
   "rharul":"\u296C",
   "Rho":"\u03A1",
   "rho":"\u03C1",
   "rhov":"\u03F1",
   "RightAngleBracket":"\u27E9",
   "RightArrowBar":"\u21E5",
   "rightarrow":"\u2192",
   "RightArrow":"\u2192",
   "Rightarrow":"\u21D2",
   "RightArrowLeftArrow":"\u21C4",
   "rightarrowtail":"\u21A3",
   "RightCeiling":"\u2309",
   "RightDoubleBracket":"\u27E7",
   "RightDownTeeVector":"\u295D",
   "RightDownVectorBar":"\u2955",
   "RightDownVector":"\u21C2",
   "RightFloor":"\u230B",
   "rightharpoondown":"\u21C1",
   "rightharpoonup":"\u21C0",
   "rightleftarrows":"\u21C4",
   "rightleftharpoons":"\u21CC",
   "rightrightarrows":"\u21C9",
   "rightsquigarrow":"\u219D",
   "RightTeeArrow":"\u21A6",
   "RightTee":"\u22A2",
   "RightTeeVector":"\u295B",
   "rightthreetimes":"\u22CC",
   "RightTriangleBar":"\u29D0",
   "RightTriangle":"\u22B3",
   "RightTriangleEqual":"\u22B5",
   "RightUpDownVector":"\u294F",
   "RightUpTeeVector":"\u295C",
   "RightUpVectorBar":"\u2954",
   "RightUpVector":"\u21BE",
   "RightVectorBar":"\u2953",
   "RightVector":"\u21C0",
   "ring":"\u02DA",
   "risingdotseq":"\u2253",
   "rlarr":"\u21C4",
   "rlhar":"\u21CC",
   "rlm":"\u200F",
   "rmoustache":"\u23B1",
   "rmoust":"\u23B1",
   "rnmid":"\u2AEE",
   "roang":"\u27ED",
   "roarr":"\u21FE",
   "robrk":"\u27E7",
   "ropar":"\u2986",
   "ropf":"\uD835\uDD63",
   "Ropf":"\u211D",
   "roplus":"\u2A2E",
   "rotimes":"\u2A35",
   "RoundImplies":"\u2970",
   "rpar":")",
   "rpargt":"\u2994",
   "rppolint":"\u2A12",
   "rrarr":"\u21C9",
   "Rrightarrow":"\u21DB",
   "rsaquo":"\u203A",
   "rscr":"\uD835\uDCC7",
   "Rscr":"\u211B",
   "rsh":"\u21B1",
   "Rsh":"\u21B1",
   "rsqb":"]",
   "rsquo":"\u2019",
   "rsquor":"\u2019",
   "rthree":"\u22CC",
   "rtimes":"\u22CA",
   "rtri":"\u25B9",
   "rtrie":"\u22B5",
   "rtrif":"\u25B8",
   "rtriltri":"\u29CE",
   "RuleDelayed":"\u29F4",
   "ruluhar":"\u2968",
   "rx":"\u211E",
   "Sacute":"\u015A",
   "sacute":"\u015B",
   "sbquo":"\u201A",
   "scap":"\u2AB8",
   "Scaron":"\u0160",
   "scaron":"\u0161",
   "Sc":"\u2ABC",
   "sc":"\u227B",
   "sccue":"\u227D",
   "sce":"\u2AB0",
   "scE":"\u2AB4",
   "Scedil":"\u015E",
   "scedil":"\u015F",
   "Scirc":"\u015C",
   "scirc":"\u015D",
   "scnap":"\u2ABA",
   "scnE":"\u2AB6",
   "scnsim":"\u22E9",
   "scpolint":"\u2A13",
   "scsim":"\u227F",
   "Scy":"\u0421",
   "scy":"\u0441",
   "sdotb":"\u22A1",
   "sdot":"\u22C5",
   "sdote":"\u2A66",
   "searhk":"\u2925",
   "searr":"\u2198",
   "seArr":"\u21D8",
   "searrow":"\u2198",
   "sect":"\u00A7",
   "semi":";",
   "seswar":"\u2929",
   "setminus":"\u2216",
   "setmn":"\u2216",
   "sext":"\u2736",
   "Sfr":"\uD835\uDD16",
   "sfr":"\uD835\uDD30",
   "sfrown":"\u2322",
   "sharp":"\u266F",
   "SHCHcy":"\u0429",
   "shchcy":"\u0449",
   "SHcy":"\u0428",
   "shcy":"\u0448",
   "ShortDownArrow":"\u2193",
   "ShortLeftArrow":"\u2190",
   "shortmid":"\u2223",
   "shortparallel":"\u2225",
   "ShortRightArrow":"\u2192",
   "ShortUpArrow":"\u2191",
   "shy":"\u00AD",
   "Sigma":"\u03A3",
   "sigma":"\u03C3",
   "sigmaf":"\u03C2",
   "sigmav":"\u03C2",
   "sim":"\u223C",
   "simdot":"\u2A6A",
   "sime":"\u2243",
   "simeq":"\u2243",
   "simg":"\u2A9E",
   "simgE":"\u2AA0",
   "siml":"\u2A9D",
   "simlE":"\u2A9F",
   "simne":"\u2246",
   "simplus":"\u2A24",
   "simrarr":"\u2972",
   "slarr":"\u2190",
   "SmallCircle":"\u2218",
   "smallsetminus":"\u2216",
   "smashp":"\u2A33",
   "smeparsl":"\u29E4",
   "smid":"\u2223",
   "smile":"\u2323",
   "smt":"\u2AAA",
   "smte":"\u2AAC",
   "smtes":"\u2AAC\uFE00",
   "SOFTcy":"\u042C",
   "softcy":"\u044C",
   "solbar":"\u233F",
   "solb":"\u29C4",
   "sol":"/",
   "Sopf":"\uD835\uDD4A",
   "sopf":"\uD835\uDD64",
   "spades":"\u2660",
   "spadesuit":"\u2660",
   "spar":"\u2225",
   "sqcap":"\u2293",
   "sqcaps":"\u2293\uFE00",
   "sqcup":"\u2294",
   "sqcups":"\u2294\uFE00",
   "Sqrt":"\u221A",
   "sqsub":"\u228F",
   "sqsube":"\u2291",
   "sqsubset":"\u228F",
   "sqsubseteq":"\u2291",
   "sqsup":"\u2290",
   "sqsupe":"\u2292",
   "sqsupset":"\u2290",
   "sqsupseteq":"\u2292",
   "square":"\u25A1",
   "Square":"\u25A1",
   "SquareIntersection":"\u2293",
   "SquareSubset":"\u228F",
   "SquareSubsetEqual":"\u2291",
   "SquareSuperset":"\u2290",
   "SquareSupersetEqual":"\u2292",
   "SquareUnion":"\u2294",
   "squarf":"\u25AA",
   "squ":"\u25A1",
   "squf":"\u25AA",
   "srarr":"\u2192",
   "Sscr":"\uD835\uDCAE",
   "sscr":"\uD835\uDCC8",
   "ssetmn":"\u2216",
   "ssmile":"\u2323",
   "sstarf":"\u22C6",
   "Star":"\u22C6",
   "star":"\u2606",
   "starf":"\u2605",
   "straightepsilon":"\u03F5",
   "straightphi":"\u03D5",
   "strns":"\u00AF",
   "sub":"\u2282",
   "Sub":"\u22D0",
   "subdot":"\u2ABD",
   "subE":"\u2AC5",
   "sube":"\u2286",
   "subedot":"\u2AC3",
   "submult":"\u2AC1",
   "subnE":"\u2ACB",
   "subne":"\u228A",
   "subplus":"\u2ABF",
   "subrarr":"\u2979",
   "subset":"\u2282",
   "Subset":"\u22D0",
   "subseteq":"\u2286",
   "subseteqq":"\u2AC5",
   "SubsetEqual":"\u2286",
   "subsetneq":"\u228A",
   "subsetneqq":"\u2ACB",
   "subsim":"\u2AC7",
   "subsub":"\u2AD5",
   "subsup":"\u2AD3",
   "succapprox":"\u2AB8",
   "succ":"\u227B",
   "succcurlyeq":"\u227D",
   "Succeeds":"\u227B",
   "SucceedsEqual":"\u2AB0",
   "SucceedsSlantEqual":"\u227D",
   "SucceedsTilde":"\u227F",
   "succeq":"\u2AB0",
   "succnapprox":"\u2ABA",
   "succneqq":"\u2AB6",
   "succnsim":"\u22E9",
   "succsim":"\u227F",
   "SuchThat":"\u220B",
   "sum":"\u2211",
   "Sum":"\u2211",
   "sung":"\u266A",
   "sup1":"\u00B9",
   "sup2":"\u00B2",
   "sup3":"\u00B3",
   "sup":"\u2283",
   "Sup":"\u22D1",
   "supdot":"\u2ABE",
   "supdsub":"\u2AD8",
   "supE":"\u2AC6",
   "supe":"\u2287",
   "supedot":"\u2AC4",
   "Superset":"\u2283",
   "SupersetEqual":"\u2287",
   "suphsol":"\u27C9",
   "suphsub":"\u2AD7",
   "suplarr":"\u297B",
   "supmult":"\u2AC2",
   "supnE":"\u2ACC",
   "supne":"\u228B",
   "supplus":"\u2AC0",
   "supset":"\u2283",
   "Supset":"\u22D1",
   "supseteq":"\u2287",
   "supseteqq":"\u2AC6",
   "supsetneq":"\u228B",
   "supsetneqq":"\u2ACC",
   "supsim":"\u2AC8",
   "supsub":"\u2AD4",
   "supsup":"\u2AD6",
   "swarhk":"\u2926",
   "swarr":"\u2199",
   "swArr":"\u21D9",
   "swarrow":"\u2199",
   "swnwar":"\u292A",
   "szlig":"\u00DF",
   "Tab":"\t",
   "target":"\u2316",
   "Tau":"\u03A4",
   "tau":"\u03C4",
   "tbrk":"\u23B4",
   "Tcaron":"\u0164",
   "tcaron":"\u0165",
   "Tcedil":"\u0162",
   "tcedil":"\u0163",
   "Tcy":"\u0422",
   "tcy":"\u0442",
   "tdot":"\u20DB",
   "telrec":"\u2315",
   "Tfr":"\uD835\uDD17",
   "tfr":"\uD835\uDD31",
   "there4":"\u2234",
   "therefore":"\u2234",
   "Therefore":"\u2234",
   "Theta":"\u0398",
   "theta":"\u03B8",
   "thetasym":"\u03D1",
   "thetav":"\u03D1",
   "thickapprox":"\u2248",
   "thicksim":"\u223C",
   "ThickSpace":"\u205F\u200A",
   "ThinSpace":"\u2009",
   "thinsp":"\u2009",
   "thkap":"\u2248",
   "thksim":"\u223C",
   "THORN":"\u00DE",
   "thorn":"\u00FE",
   "tilde":"\u02DC",
   "Tilde":"\u223C",
   "TildeEqual":"\u2243",
   "TildeFullEqual":"\u2245",
   "TildeTilde":"\u2248",
   "timesbar":"\u2A31",
   "timesb":"\u22A0",
   "times":"\u00D7",
   "timesd":"\u2A30",
   "tint":"\u222D",
   "toea":"\u2928",
   "topbot":"\u2336",
   "topcir":"\u2AF1",
   "top":"\u22A4",
   "Topf":"\uD835\uDD4B",
   "topf":"\uD835\uDD65",
   "topfork":"\u2ADA",
   "tosa":"\u2929",
   "tprime":"\u2034",
   "trade":"\u2122",
   "TRADE":"\u2122",
   "triangle":"\u25B5",
   "triangledown":"\u25BF",
   "triangleleft":"\u25C3",
   "trianglelefteq":"\u22B4",
   "triangleq":"\u225C",
   "triangleright":"\u25B9",
   "trianglerighteq":"\u22B5",
   "tridot":"\u25EC",
   "trie":"\u225C",
   "triminus":"\u2A3A",
   "TripleDot":"\u20DB",
   "triplus":"\u2A39",
   "trisb":"\u29CD",
   "tritime":"\u2A3B",
   "trpezium":"\u23E2",
   "Tscr":"\uD835\uDCAF",
   "tscr":"\uD835\uDCC9",
   "TScy":"\u0426",
   "tscy":"\u0446",
   "TSHcy":"\u040B",
   "tshcy":"\u045B",
   "Tstrok":"\u0166",
   "tstrok":"\u0167",
   "twixt":"\u226C",
   "twoheadleftarrow":"\u219E",
   "twoheadrightarrow":"\u21A0",
   "Uacute":"\u00DA",
   "uacute":"\u00FA",
   "uarr":"\u2191",
   "Uarr":"\u219F",
   "uArr":"\u21D1",
   "Uarrocir":"\u2949",
   "Ubrcy":"\u040E",
   "ubrcy":"\u045E",
   "Ubreve":"\u016C",
   "ubreve":"\u016D",
   "Ucirc":"\u00DB",
   "ucirc":"\u00FB",
   "Ucy":"\u0423",
   "ucy":"\u0443",
   "udarr":"\u21C5",
   "Udblac":"\u0170",
   "udblac":"\u0171",
   "udhar":"\u296E",
   "ufisht":"\u297E",
   "Ufr":"\uD835\uDD18",
   "ufr":"\uD835\uDD32",
   "Ugrave":"\u00D9",
   "ugrave":"\u00F9",
   "uHar":"\u2963",
   "uharl":"\u21BF",
   "uharr":"\u21BE",
   "uhblk":"\u2580",
   "ulcorn":"\u231C",
   "ulcorner":"\u231C",
   "ulcrop":"\u230F",
   "ultri":"\u25F8",
   "Umacr":"\u016A",
   "umacr":"\u016B",
   "uml":"\u00A8",
   "UnderBar":"_",
   "UnderBrace":"\u23DF",
   "UnderBracket":"\u23B5",
   "UnderParenthesis":"\u23DD",
   "Union":"\u22C3",
   "UnionPlus":"\u228E",
   "Uogon":"\u0172",
   "uogon":"\u0173",
   "Uopf":"\uD835\uDD4C",
   "uopf":"\uD835\uDD66",
   "UpArrowBar":"\u2912",
   "uparrow":"\u2191",
   "UpArrow":"\u2191",
   "Uparrow":"\u21D1",
   "UpArrowDownArrow":"\u21C5",
   "updownarrow":"\u2195",
   "UpDownArrow":"\u2195",
   "Updownarrow":"\u21D5",
   "UpEquilibrium":"\u296E",
   "upharpoonleft":"\u21BF",
   "upharpoonright":"\u21BE",
   "uplus":"\u228E",
   "UpperLeftArrow":"\u2196",
   "UpperRightArrow":"\u2197",
   "upsi":"\u03C5",
   "Upsi":"\u03D2",
   "upsih":"\u03D2",
   "Upsilon":"\u03A5",
   "upsilon":"\u03C5",
   "UpTeeArrow":"\u21A5",
   "UpTee":"\u22A5",
   "upuparrows":"\u21C8",
   "urcorn":"\u231D",
   "urcorner":"\u231D",
   "urcrop":"\u230E",
   "Uring":"\u016E",
   "uring":"\u016F",
   "urtri":"\u25F9",
   "Uscr":"\uD835\uDCB0",
   "uscr":"\uD835\uDCCA",
   "utdot":"\u22F0",
   "Utilde":"\u0168",
   "utilde":"\u0169",
   "utri":"\u25B5",
   "utrif":"\u25B4",
   "uuarr":"\u21C8",
   "Uuml":"\u00DC",
   "uuml":"\u00FC",
   "uwangle":"\u29A7",
   "vangrt":"\u299C",
   "varepsilon":"\u03F5",
   "varkappa":"\u03F0",
   "varnothing":"\u2205",
   "varphi":"\u03D5",
   "varpi":"\u03D6",
   "varpropto":"\u221D",
   "varr":"\u2195",
   "vArr":"\u21D5",
   "varrho":"\u03F1",
   "varsigma":"\u03C2",
   "varsubsetneq":"\u228A\uFE00",
   "varsubsetneqq":"\u2ACB\uFE00",
   "varsupsetneq":"\u228B\uFE00",
   "varsupsetneqq":"\u2ACC\uFE00",
   "vartheta":"\u03D1",
   "vartriangleleft":"\u22B2",
   "vartriangleright":"\u22B3",
   "vBar":"\u2AE8",
   "Vbar":"\u2AEB",
   "vBarv":"\u2AE9",
   "Vcy":"\u0412",
   "vcy":"\u0432",
   "vdash":"\u22A2",
   "vDash":"\u22A8",
   "Vdash":"\u22A9",
   "VDash":"\u22AB",
   "Vdashl":"\u2AE6",
   "veebar":"\u22BB",
   "vee":"\u2228",
   "Vee":"\u22C1",
   "veeeq":"\u225A",
   "vellip":"\u22EE",
   "verbar":"|",
   "Verbar":"\u2016",
   "vert":"|",
   "Vert":"\u2016",
   "VerticalBar":"\u2223",
   "VerticalLine":"|",
   "VerticalSeparator":"\u2758",
   "VerticalTilde":"\u2240",
   "VeryThinSpace":"\u200A",
   "Vfr":"\uD835\uDD19",
   "vfr":"\uD835\uDD33",
   "vltri":"\u22B2",
   "vnsub":"\u2282\u20D2",
   "vnsup":"\u2283\u20D2",
   "Vopf":"\uD835\uDD4D",
   "vopf":"\uD835\uDD67",
   "vprop":"\u221D",
   "vrtri":"\u22B3",
   "Vscr":"\uD835\uDCB1",
   "vscr":"\uD835\uDCCB",
   "vsubnE":"\u2ACB\uFE00",
   "vsubne":"\u228A\uFE00",
   "vsupnE":"\u2ACC\uFE00",
   "vsupne":"\u228B\uFE00",
   "Vvdash":"\u22AA",
   "vzigzag":"\u299A",
   "Wcirc":"\u0174",
   "wcirc":"\u0175",
   "wedbar":"\u2A5F",
   "wedge":"\u2227",
   "Wedge":"\u22C0",
   "wedgeq":"\u2259",
   "weierp":"\u2118",
   "Wfr":"\uD835\uDD1A",
   "wfr":"\uD835\uDD34",
   "Wopf":"\uD835\uDD4E",
   "wopf":"\uD835\uDD68",
   "wp":"\u2118",
   "wr":"\u2240",
   "wreath":"\u2240",
   "Wscr":"\uD835\uDCB2",
   "wscr":"\uD835\uDCCC",
   "xcap":"\u22C2",
   "xcirc":"\u25EF",
   "xcup":"\u22C3",
   "xdtri":"\u25BD",
   "Xfr":"\uD835\uDD1B",
   "xfr":"\uD835\uDD35",
   "xharr":"\u27F7",
   "xhArr":"\u27FA",
   "Xi":"\u039E",
   "xi":"\u03BE",
   "xlarr":"\u27F5",
   "xlArr":"\u27F8",
   "xmap":"\u27FC",
   "xnis":"\u22FB",
   "xodot":"\u2A00",
   "Xopf":"\uD835\uDD4F",
   "xopf":"\uD835\uDD69",
   "xoplus":"\u2A01",
   "xotime":"\u2A02",
   "xrarr":"\u27F6",
   "xrArr":"\u27F9",
   "Xscr":"\uD835\uDCB3",
   "xscr":"\uD835\uDCCD",
   "xsqcup":"\u2A06",
   "xuplus":"\u2A04",
   "xutri":"\u25B3",
   "xvee":"\u22C1",
   "xwedge":"\u22C0",
   "Yacute":"\u00DD",
   "yacute":"\u00FD",
   "YAcy":"\u042F",
   "yacy":"\u044F",
   "Ycirc":"\u0176",
   "ycirc":"\u0177",
   "Ycy":"\u042B",
   "ycy":"\u044B",
   "yen":"\u00A5",
   "Yfr":"\uD835\uDD1C",
   "yfr":"\uD835\uDD36",
   "YIcy":"\u0407",
   "yicy":"\u0457",
   "Yopf":"\uD835\uDD50",
   "yopf":"\uD835\uDD6A",
   "Yscr":"\uD835\uDCB4",
   "yscr":"\uD835\uDCCE",
   "YUcy":"\u042E",
   "yucy":"\u044E",
   "yuml":"\u00FF",
   "Yuml":"\u0178",
   "Zacute":"\u0179",
   "zacute":"\u017A",
   "Zcaron":"\u017D",
   "zcaron":"\u017E",
   "Zcy":"\u0417",
   "zcy":"\u0437",
   "Zdot":"\u017B",
   "zdot":"\u017C",
   "zeetrf":"\u2128",
   "ZeroWidthSpace":"\u200B",
   "Zeta":"\u0396",
   "zeta":"\u03B6",
   "zfr":"\uD835\uDD37",
   "Zfr":"\u2128",
   "ZHcy":"\u0416",
   "zhcy":"\u0436",
   "zigrarr":"\u21DD",
   "zopf":"\uD835\uDD6B",
   "Zopf":"\u2124",
   "Zscr":"\uD835\uDCB5",
   "zscr":"\uD835\uDCCF",
   "zwj":"\u200D",
   "zwnj":"\u200C"
};
});
/* eslint-env amd */
/* eslint-disable missing-nls */
define('htmlparser2/legacy',[
], function() {
	return {  
   "Aacute":"\u00C1",
   "aacute":"\u00E1",
   "Acirc":"\u00C2",
   "acirc":"\u00E2",
   "acute":"\u00B4",
   "AElig":"\u00C6",
   "aelig":"\u00E6",
   "Agrave":"\u00C0",
   "agrave":"\u00E0",
   "amp":"&",
   "AMP":"&",
   "Aring":"\u00C5",
   "aring":"\u00E5",
   "Atilde":"\u00C3",
   "atilde":"\u00E3",
   "Auml":"\u00C4",
   "auml":"\u00E4",
   "brvbar":"\u00A6",
   "Ccedil":"\u00C7",
   "ccedil":"\u00E7",
   "cedil":"\u00B8",
   "cent":"\u00A2",
   "copy":"\u00A9",
   "COPY":"\u00A9",
   "curren":"\u00A4",
   "deg":"\u00B0",
   "divide":"\u00F7",
   "Eacute":"\u00C9",
   "eacute":"\u00E9",
   "Ecirc":"\u00CA",
   "ecirc":"\u00EA",
   "Egrave":"\u00C8",
   "egrave":"\u00E8",
   "ETH":"\u00D0",
   "eth":"\u00F0",
   "Euml":"\u00CB",
   "euml":"\u00EB",
   "frac12":"\u00BD",
   "frac14":"\u00BC",
   "frac34":"\u00BE",
   "gt":">",
   "GT":">",
   "Iacute":"\u00CD",
   "iacute":"\u00ED",
   "Icirc":"\u00CE",
   "icirc":"\u00EE",
   "iexcl":"\u00A1",
   "Igrave":"\u00CC",
   "igrave":"\u00EC",
   "iquest":"\u00BF",
   "Iuml":"\u00CF",
   "iuml":"\u00EF",
   "laquo":"\u00AB",
   "lt":"<",
   "LT":"<",
   "macr":"\u00AF",
   "micro":"\u00B5",
   "middot":"\u00B7",
   "nbsp":"\u00A0",
   "not":"\u00AC",
   "Ntilde":"\u00D1",
   "ntilde":"\u00F1",
   "Oacute":"\u00D3",
   "oacute":"\u00F3",
   "Ocirc":"\u00D4",
   "ocirc":"\u00F4",
   "Ograve":"\u00D2",
   "ograve":"\u00F2",
   "ordf":"\u00AA",
   "ordm":"\u00BA",
   "Oslash":"\u00D8",
   "oslash":"\u00F8",
   "Otilde":"\u00D5",
   "otilde":"\u00F5",
   "Ouml":"\u00D6",
   "ouml":"\u00F6",
   "para":"\u00B6",
   "plusmn":"\u00B1",
   "pound":"\u00A3",
   "quot":"\"",
   "QUOT":"\"",
   "raquo":"\u00BB",
   "reg":"\u00AE",
   "REG":"\u00AE",
   "sect":"\u00A7",
   "shy":"\u00AD",
   "sup1":"\u00B9",
   "sup2":"\u00B2",
   "sup3":"\u00B3",
   "szlig":"\u00DF",
   "THORN":"\u00DE",
   "thorn":"\u00FE",
   "times":"\u00D7",
   "Uacute":"\u00DA",
   "uacute":"\u00FA",
   "Ucirc":"\u00DB",
   "ucirc":"\u00FB",
   "Ugrave":"\u00D9",
   "ugrave":"\u00F9",
   "uml":"\u00A8",
   "Uuml":"\u00DC",
   "uuml":"\u00FC",
   "Yacute":"\u00DD",
   "yacute":"\u00FD",
   "yen":"\u00A5",
   "yuml":"\u00FF"
};
});
/* eslint-env amd */
/* eslint-disable missing-nls */
define('htmlparser2/xml',[
], function() {
	return {"amp":"&","apos":"'","gt":">","lt":"<","quot":"\""};
});
/* eslint-env amd */
/* eslint-disable missing-nls */
define('htmlparser2/tokenizer',[
'htmlparser2/decodeCodepoint',
'htmlparser2/entities',
'htmlparser2/legacy',
'htmlparser2/xml'
], function(decodeCodePoint, entityMap, legacyMap, xmlMap) {

	    var i = 0,
	
	    TEXT                      = i++,
	    BEFORE_TAG_NAME           = i++, //after <
	    IN_TAG_NAME               = i++,
	    IN_SELF_CLOSING_TAG       = i++,
	    BEFORE_CLOSING_TAG_NAME   = i++,
	    IN_CLOSING_TAG_NAME       = i++,
	    AFTER_CLOSING_TAG_NAME    = i++,
	
	    //attributes
	    BEFORE_ATTRIBUTE_NAME     = i++,
	    IN_ATTRIBUTE_NAME         = i++,
	    AFTER_ATTRIBUTE_NAME      = i++,
	    BEFORE_ATTRIBUTE_VALUE    = i++,
	    IN_ATTRIBUTE_VALUE_DQ     = i++, // "
	    IN_ATTRIBUTE_VALUE_SQ     = i++, // '
	    IN_ATTRIBUTE_VALUE_NQ     = i++,
	
	    //declarations
	    BEFORE_DECLARATION        = i++, // !
	    IN_DECLARATION            = i++,
	
	    //processing instructions
	    IN_PROCESSING_INSTRUCTION = i++, // ?
	
	    //comments
	    BEFORE_COMMENT            = i++,
	    IN_COMMENT                = i++,
	    AFTER_COMMENT_1           = i++,
	    AFTER_COMMENT_2           = i++,
	
	    //cdata
	    BEFORE_CDATA_1            = i++, // [
	    BEFORE_CDATA_2            = i++, // C
	    BEFORE_CDATA_3            = i++, // D
	    BEFORE_CDATA_4            = i++, // A
	    BEFORE_CDATA_5            = i++, // T
	    BEFORE_CDATA_6            = i++, // A
	    IN_CDATA                  = i++, // [
	    AFTER_CDATA_1             = i++, // ]
	    AFTER_CDATA_2             = i++, // ]
	
	    //special tags
	    BEFORE_SPECIAL            = i++, //S
	    BEFORE_SPECIAL_END        = i++,   //S
	
	    BEFORE_SCRIPT_1           = i++, //C
	    BEFORE_SCRIPT_2           = i++, //R
	    BEFORE_SCRIPT_3           = i++, //I
	    BEFORE_SCRIPT_4           = i++, //P
	    BEFORE_SCRIPT_5           = i++, //T
	    AFTER_SCRIPT_1            = i++, //C
	    AFTER_SCRIPT_2            = i++, //R
	    AFTER_SCRIPT_3            = i++, //I
	    AFTER_SCRIPT_4            = i++, //P
	    AFTER_SCRIPT_5            = i++, //T
	
	    BEFORE_STYLE_1            = i++, //T
	    BEFORE_STYLE_2            = i++, //Y
	    BEFORE_STYLE_3            = i++, //L
	    BEFORE_STYLE_4            = i++, //E
	    AFTER_STYLE_1             = i++, //T
	    AFTER_STYLE_2             = i++, //Y
	    AFTER_STYLE_3             = i++, //L
	    AFTER_STYLE_4             = i++, //E
	
	    BEFORE_ENTITY             = i++, //&
	    BEFORE_NUMERIC_ENTITY     = i++, //#
	    IN_NAMED_ENTITY           = i++,
	    IN_NUMERIC_ENTITY         = i++,
	    IN_HEX_ENTITY             = i++, //X
	
	    j = 0,
	
	    SPECIAL_NONE              = j++,
	    SPECIAL_SCRIPT            = j++,
	    SPECIAL_STYLE             = j++;
	
	function whitespace(c){
		return c === " " || c === "\n" || c === "\t" || c === "\f" || c === "\r";
	}
	
	function characterState(char, SUCCESS){
		return function(c){
			if(c === char) this._state = SUCCESS;
		};
	}
	
	function ifElseState(upper, SUCCESS, FAILURE){
		var lower = upper.toLowerCase();
	
		if(upper === lower){
			return function(c){
				if(c === lower){
					this._state = SUCCESS;
				} else {
					this._state = FAILURE;
					this._index--;
				}
			};
		} else {
			return function(c){
				if(c === lower || c === upper){
					this._state = SUCCESS;
				} else {
					this._state = FAILURE;
					this._index--;
				}
			};
		}
	}
	
	function consumeSpecialNameChar(upper, NEXT_STATE){
		var lower = upper.toLowerCase();
	
		return function(c){
			if(c === lower || c === upper){
				this._state = NEXT_STATE;
			} else {
				this._state = IN_TAG_NAME;
				this._index--; //consume the token again
			}
		};
	}
	
	function Tokenizer(options, cbs){
		this._state = TEXT;
		this._buffer = "";
		this._sectionStart = 0;
		this._index = 0;
		this._bufferOffset = 0; //chars removed from _buffer
		this._baseState = TEXT;
		this._special = SPECIAL_NONE;
		this._cbs = cbs;
		this._running = true;
		this._ended = false;
		this._xmlMode = !!(options && options.xmlMode);
		this._decodeEntities = !!(options && options.decodeEntities);
		this._quoteStart = 0;
	}
	
	Tokenizer.prototype._stateText = function(c){
		if(c === "<"){
			if(this._index > this._sectionStart){
				this._cbs.ontext(this._getSection());
			}
			this._state = BEFORE_TAG_NAME;
			this._sectionStart = this._index;
		} else if(this._decodeEntities && this._special === SPECIAL_NONE && c === "&"){
			if(this._index > this._sectionStart){
				this._cbs.ontext(this._getSection());
			}
			this._baseState = TEXT;
			this._state = BEFORE_ENTITY;
			this._sectionStart = this._index;
		}
	};
	
	Tokenizer.prototype._stateBeforeTagName = function(c){
		if(c === "/"){
			this._state = BEFORE_CLOSING_TAG_NAME;
		} else if(c === ">" || this._special !== SPECIAL_NONE || whitespace(c)) {
			this._state = TEXT;
		} else if(c === "!"){
			this._state = BEFORE_DECLARATION;
			this._sectionStart = this._index + 1;
		} else if(c === "?"){
			this._state = IN_PROCESSING_INSTRUCTION;
			this._sectionStart = this._index + 1;
		} else if(c === "<"){
			this._cbs.ontext(this._getSection());
			this._sectionStart = this._index;
		} else {
			this._state = (!this._xmlMode && (c === "s" || c === "S")) ?
							BEFORE_SPECIAL : IN_TAG_NAME;
			this._sectionStart = this._index;
		}
	};
	
	Tokenizer.prototype._stateInTagName = function(c){
		if(c === "/" || c === ">" || c === '<' || whitespace(c)){
			this._emitToken("onopentagname");
			this._state = BEFORE_ATTRIBUTE_NAME;
			this._index--;
		}
	};
	
	Tokenizer.prototype._stateBeforeCloseingTagName = function(c){
		if(whitespace(c));
		else if(c === ">"){
			this._state = TEXT;
		} else if(this._special !== SPECIAL_NONE){
			if(c === "s" || c === "S"){
				this._state = BEFORE_SPECIAL_END;
			} else {
				this._state = TEXT;
				this._index--;
			}
		} else {
			this._state = IN_CLOSING_TAG_NAME;
			this._sectionStart = this._index;
		}
	};
	
	Tokenizer.prototype._stateInCloseingTagName = function(c){
		if(c === ">" || whitespace(c)){
			this._emitToken("onclosetag");
			this._state = AFTER_CLOSING_TAG_NAME;
			this._index--;
		}
	};
	
	Tokenizer.prototype._stateAfterCloseingTagName = function(c){
		//skip everything until ">"
		if(c === ">"){
			this._state = TEXT;
			this._sectionStart = this._index + 1;
		}
	};
	
	Tokenizer.prototype._stateBeforeAttributeName = function(c){
		if(c === ">"){
			this._cbs.onopentagend();
			this._state = TEXT;
			this._sectionStart = this._index + 1;
		} else if(c === "<"){
			// TODO Orion 11.0 This case allows parsing of incomplete tags as tags not attrs
			this._cbs.onopentagend();
			this._state = BEFORE_TAG_NAME;
			this._sectionStart = this._index;
			this._index--;
		} else if(c === "/"){
			this._state = IN_SELF_CLOSING_TAG;
		} else if(!whitespace(c)){
			this._state = IN_ATTRIBUTE_NAME;
			this._sectionStart = this._index;
		}
	};
	
	Tokenizer.prototype._stateInSelfClosingTag = function(c){
		if(c === ">"){
			this._cbs.onselfclosingtag();
			this._state = TEXT;
			this._sectionStart = this._index + 1;
		} else if(!whitespace(c)){
			this._state = BEFORE_ATTRIBUTE_NAME;
			this._index--;
		}
	};
	
	Tokenizer.prototype._stateInAttributeName = function(c){
		if(c === "=" || c === "/" || c === ">" || whitespace(c)){
			this._cbs.onattribname(this._getSection());
			this._sectionStart = -1;
			this._state = AFTER_ATTRIBUTE_NAME;
			this._index--;
		}
	};
	
	Tokenizer.prototype._stateAfterAttributeName = function(c){
		if(c === "="){
			this._state = BEFORE_ATTRIBUTE_VALUE;
			this._quoteStart = this._index + 1;//position after the '='
		} else if(c === "/" || c === ">"){
			this._cbs.onattribend();
			this._state = BEFORE_ATTRIBUTE_NAME;
			this._index--;
		} else if(!whitespace(c)){
			this._cbs.onattribend();
			this._state = IN_ATTRIBUTE_NAME;
			this._sectionStart = this._index;
		}
	};
	
	Tokenizer.prototype._stateBeforeAttributeValue = function(c){
		if(c === "\""){
			this._state = IN_ATTRIBUTE_VALUE_DQ;
			this._sectionStart = this._index + 1;
		} else if(c === "'"){
			this._state = IN_ATTRIBUTE_VALUE_SQ;
			this._sectionStart = this._index + 1;
		} else if(!whitespace(c)){
			this._state = IN_ATTRIBUTE_VALUE_NQ;
			this._sectionStart = this._index;
			this._index--; //reconsume token
		}
	};
	
	Tokenizer.prototype._stateInAttributeValueDoubleQuotes = function(c){
		if(c === "\""){
			var valueStart = this._quoteStart;
			this._emitToken("onattribdata");
			this._cbs.onattribend(true, valueStart);
			this._state = BEFORE_ATTRIBUTE_NAME;
		} else if(this._decodeEntities && c === "&"){
			this._emitToken("onattribdata");
			this._baseState = this._state;
			this._state = BEFORE_ENTITY;
			this._sectionStart = this._index;
		} else if (c === '<') {
			valueStart = this._quoteStart;
			this._emitToken("onattribdata");
			this._cbs.onattribend(false, valueStart);
			this._state = BEFORE_ATTRIBUTE_NAME;
			this._index--;
		}
	};
	
	Tokenizer.prototype._stateInAttributeValueSingleQuotes = function(c){
		if(c === "'"){
			var valueStart = this._quoteStart;
			this._emitToken("onattribdata");
			this._cbs.onattribend(true, valueStart);
			this._state = BEFORE_ATTRIBUTE_NAME;
		} else if(this._decodeEntities && c === "&"){
			this._emitToken("onattribdata");
			this._baseState = this._state;
			this._state = BEFORE_ENTITY;
			this._sectionStart = this._index;
		} else if (c === '<') {
			valueStart = this._quoteStart;
			this._emitToken("onattribdata");
			this._cbs.onattribend(false, valueStart);
			this._state = BEFORE_ATTRIBUTE_NAME;
			this._index--;
		}
	};
	
	Tokenizer.prototype._stateInAttributeValueNoQuotes = function(c){
		if(whitespace(c) || c === ">"){
			var valueStart = this._quoteStart;
			this._emitToken("onattribdata");
			this._cbs.onattribend(false, valueStart);
			this._state = BEFORE_ATTRIBUTE_NAME;
			this._index--;
		} else if(this._decodeEntities && c === "&"){
			this._emitToken("onattribdata");
			this._baseState = this._state;
			this._state = BEFORE_ENTITY;
			this._sectionStart = this._index;
		}
	};
	
	Tokenizer.prototype._stateBeforeDeclaration = function(c){
		this._state = c === "[" ? BEFORE_CDATA_1 :
						c === "-" ? BEFORE_COMMENT :
							IN_DECLARATION;
	};
	
	Tokenizer.prototype._stateInDeclaration = function(c){
		if(c === ">"){
			this._cbs.ondeclaration(this._getSection());
			this._state = TEXT;
			this._sectionStart = this._index + 1;
		}
	};
	
	Tokenizer.prototype._stateInProcessingInstruction = function(c){
		if(c === ">"){
			this._cbs.onprocessinginstruction(this._getSection());
			this._state = TEXT;
			this._sectionStart = this._index + 1;
		}
	};
	
	Tokenizer.prototype._stateBeforeComment = function(c){
		if(c === "-"){
			this._state = IN_COMMENT;
			this._sectionStart = this._index + 1;
		} else {
			this._state = IN_DECLARATION;
		}
	};
	
	Tokenizer.prototype._stateInComment = function(c){
		if(c === "-") this._state = AFTER_COMMENT_1;
	};
	
	Tokenizer.prototype._stateAfterComment1 = function(c){
		if(c === "-"){
			this._state = AFTER_COMMENT_2;
		} else {
			this._state = IN_COMMENT;
		}
	};
	
	Tokenizer.prototype._stateAfterComment2 = function(c){
		if(c === ">"){
			//remove 2 trailing chars
			this._cbs.oncomment(this._buffer.substring(this._sectionStart, this._index - 2));
			this._state = TEXT;
			this._sectionStart = this._index + 1;
		} else if(c !== "-"){
			this._state = IN_COMMENT;
		}
		// else: stay in AFTER_COMMENT_2 (`--->`)
	};
	
	Tokenizer.prototype._stateBeforeCdata1 = ifElseState("C", BEFORE_CDATA_2, IN_DECLARATION);
	Tokenizer.prototype._stateBeforeCdata2 = ifElseState("D", BEFORE_CDATA_3, IN_DECLARATION);
	Tokenizer.prototype._stateBeforeCdata3 = ifElseState("A", BEFORE_CDATA_4, IN_DECLARATION);
	Tokenizer.prototype._stateBeforeCdata4 = ifElseState("T", BEFORE_CDATA_5, IN_DECLARATION);
	Tokenizer.prototype._stateBeforeCdata5 = ifElseState("A", BEFORE_CDATA_6, IN_DECLARATION);
	
	Tokenizer.prototype._stateBeforeCdata6 = function(c){
		if(c === "["){
			this._state = IN_CDATA;
			this._sectionStart = this._index + 1;
		} else {
			this._state = IN_DECLARATION;
			this._index--;
		}
	};
	
	Tokenizer.prototype._stateInCdata = function(c){
		if(c === "]") this._state = AFTER_CDATA_1;
	};
	
	Tokenizer.prototype._stateAfterCdata1 = characterState("]", AFTER_CDATA_2);
	
	Tokenizer.prototype._stateAfterCdata2 = function(c){
		if(c === ">"){
			//remove 2 trailing chars
			this._cbs.oncdata(this._buffer.substring(this._sectionStart, this._index - 2));
			this._state = TEXT;
			this._sectionStart = this._index + 1;
		} else if(c !== "]") {
			this._state = IN_CDATA;
		}
		//else: stay in AFTER_CDATA_2 (`]]]>`)
	};
	
	Tokenizer.prototype._stateBeforeSpecial = function(c){
		if(c === "c" || c === "C"){
			this._state = BEFORE_SCRIPT_1;
		} else if(c === "t" || c === "T"){
			this._state = BEFORE_STYLE_1;
		} else {
			this._state = IN_TAG_NAME;
			this._index--; //consume the token again
		}
	};
	
	Tokenizer.prototype._stateBeforeSpecialEnd = function(c){
		if(this._special === SPECIAL_SCRIPT && (c === "c" || c === "C")){
			this._state = AFTER_SCRIPT_1;
		} else if(this._special === SPECIAL_STYLE && (c === "t" || c === "T")){
			this._state = AFTER_STYLE_1;
		}
		else this._state = TEXT;
	};
	
	Tokenizer.prototype._stateBeforeScript1 = consumeSpecialNameChar("R", BEFORE_SCRIPT_2);
	Tokenizer.prototype._stateBeforeScript2 = consumeSpecialNameChar("I", BEFORE_SCRIPT_3);
	Tokenizer.prototype._stateBeforeScript3 = consumeSpecialNameChar("P", BEFORE_SCRIPT_4);
	Tokenizer.prototype._stateBeforeScript4 = consumeSpecialNameChar("T", BEFORE_SCRIPT_5);
	
	Tokenizer.prototype._stateBeforeScript5 = function(c){
		if(c === "/" || c === ">" || whitespace(c)){
			this._special = SPECIAL_SCRIPT;
		}
		this._state = IN_TAG_NAME;
		this._index--; //consume the token again
	};
	
	Tokenizer.prototype._stateAfterScript1 = ifElseState("R", AFTER_SCRIPT_2, TEXT);
	Tokenizer.prototype._stateAfterScript2 = ifElseState("I", AFTER_SCRIPT_3, TEXT);
	Tokenizer.prototype._stateAfterScript3 = ifElseState("P", AFTER_SCRIPT_4, TEXT);
	Tokenizer.prototype._stateAfterScript4 = ifElseState("T", AFTER_SCRIPT_5, TEXT);
	
	Tokenizer.prototype._stateAfterScript5 = function(c){
		if(c === ">" || whitespace(c)){
			this._special = SPECIAL_NONE;
			this._state = IN_CLOSING_TAG_NAME;
			this._sectionStart = this._index - 6;
			this._index--; //reconsume the token
		}
		else this._state = TEXT;
	};
	
	Tokenizer.prototype._stateBeforeStyle1 = consumeSpecialNameChar("Y", BEFORE_STYLE_2);
	Tokenizer.prototype._stateBeforeStyle2 = consumeSpecialNameChar("L", BEFORE_STYLE_3);
	Tokenizer.prototype._stateBeforeStyle3 = consumeSpecialNameChar("E", BEFORE_STYLE_4);
	
	Tokenizer.prototype._stateBeforeStyle4 = function(c){
		if(c === "/" || c === ">" || whitespace(c)){
			this._special = SPECIAL_STYLE;
		}
		this._state = IN_TAG_NAME;
		this._index--; //consume the token again
	};
	
	Tokenizer.prototype._stateAfterStyle1 = ifElseState("Y", AFTER_STYLE_2, TEXT);
	Tokenizer.prototype._stateAfterStyle2 = ifElseState("L", AFTER_STYLE_3, TEXT);
	Tokenizer.prototype._stateAfterStyle3 = ifElseState("E", AFTER_STYLE_4, TEXT);
	
	Tokenizer.prototype._stateAfterStyle4 = function(c){
		if(c === ">" || whitespace(c)){
			this._special = SPECIAL_NONE;
			this._state = IN_CLOSING_TAG_NAME;
			this._sectionStart = this._index - 5;
			this._index--; //reconsume the token
		}
		else this._state = TEXT;
	};
	
	Tokenizer.prototype._stateBeforeEntity = ifElseState("#", BEFORE_NUMERIC_ENTITY, IN_NAMED_ENTITY);
	Tokenizer.prototype._stateBeforeNumericEntity = ifElseState("X", IN_HEX_ENTITY, IN_NUMERIC_ENTITY);
	
	//for entities terminated with a semicolon
	Tokenizer.prototype._parseNamedEntityStrict = function(){
		//offset = 1
		if(this._sectionStart + 1 < this._index){
			var entity = this._buffer.substring(this._sectionStart + 1, this._index),
			    map = this._xmlMode ? xmlMap : entityMap;
	
			if(map.hasOwnProperty(entity)){
				this._emitPartial(map[entity]);
				this._sectionStart = this._index + 1;
			}
		}
	};
	
	
	//parses legacy entities (without trailing semicolon)
	Tokenizer.prototype._parseLegacyEntity = function(){
		var start = this._sectionStart + 1,
		    limit = this._index - start;
	
		if(limit > 6) limit = 6; //the max length of legacy entities is 6
	
		while(limit >= 2){ //the min length of legacy entities is 2
			var entity = this._buffer.substr(start, limit);
	
			if(legacyMap.hasOwnProperty(entity)){
				this._emitPartial(legacyMap[entity]);
				this._sectionStart += limit + 1;
				return;
			} else {
				limit--;
			}
		}
	};
	
	Tokenizer.prototype._stateInNamedEntity = function(c){
		if(c === ";"){
			this._parseNamedEntityStrict();
			if(this._sectionStart + 1 < this._index && !this._xmlMode){
				this._parseLegacyEntity();
			}
			this._state = this._baseState;
		} else if((c < "a" || c > "z") && (c < "A" || c > "Z") && (c < "0" || c > "9")){
			if(this._xmlMode);
			else if(this._sectionStart + 1 === this._index);
			else if(this._baseState !== TEXT){
				if(c !== "="){
					this._parseNamedEntityStrict();
				}
			} else {
				this._parseLegacyEntity();
			}
	
			this._state = this._baseState;
			this._index--;
		}
	};
	
	Tokenizer.prototype._decodeNumericEntity = function(offset, base){
		var sectionStart = this._sectionStart + offset;
	
		if(sectionStart !== this._index){
			//parse entity
			var entity = this._buffer.substring(sectionStart, this._index);
			var parsed = parseInt(entity, base);
	
			this._emitPartial(decodeCodePoint(parsed));
			this._sectionStart = this._index;
		} else {
			this._sectionStart--;
		}
	
		this._state = this._baseState;
	};
	
	Tokenizer.prototype._stateInNumericEntity = function(c){
		if(c === ";"){
			this._decodeNumericEntity(2, 10);
			this._sectionStart++;
		} else if(c < "0" || c > "9"){
			if(!this._xmlMode){
				this._decodeNumericEntity(2, 10);
			} else {
				this._state = this._baseState;
			}
			this._index--;
		}
	};
	
	Tokenizer.prototype._stateInHexEntity = function(c){
		if(c === ";"){
			this._decodeNumericEntity(3, 16);
			this._sectionStart++;
		} else if((c < "a" || c > "f") && (c < "A" || c > "F") && (c < "0" || c > "9")){
			if(!this._xmlMode){
				this._decodeNumericEntity(3, 16);
			} else {
				this._state = this._baseState;
			}
			this._index--;
		}
	};
	
	Tokenizer.prototype._cleanup = function (){
		if(this._sectionStart < 0){
			this._buffer = "";
			this._index = 0;
			this._bufferOffset += this._index;
		} else if(this._running){
			if(this._state === TEXT){
				if(this._sectionStart !== this._index){
					this._cbs.ontext(this._buffer.substr(this._sectionStart));
				}
				this._buffer = "";
				this._index = 0;
				this._bufferOffset += this._index;
			} else if(this._sectionStart === this._index){
				//the section just started
				this._buffer = "";
				this._index = 0;
				this._bufferOffset += this._index;
			} else {
				//remove everything unnecessary
				this._buffer = this._buffer.substr(this._sectionStart);
				this._index -= this._sectionStart;
				this._bufferOffset += this._sectionStart;
			}
	
			this._sectionStart = 0;
		}
	};
	
	//TODO make events conditional
	Tokenizer.prototype.write = function(chunk){
		if(this._ended) this._cbs.onerror(Error(".write() after done!"));
	
		this._buffer += chunk;
		this._parse();
	};
	
	Tokenizer.prototype._parse = function(){
		while(this._index < this._buffer.length && this._running){
			var c = this._buffer.charAt(this._index);
			if(this._state === TEXT) {
				this._stateText(c);
			} else if(this._state === BEFORE_TAG_NAME){
				this._stateBeforeTagName(c);
			} else if(this._state === IN_TAG_NAME) {
				this._stateInTagName(c);
			} else if(this._state === BEFORE_CLOSING_TAG_NAME){
				this._stateBeforeCloseingTagName(c);
			} else if(this._state === IN_CLOSING_TAG_NAME){
				this._stateInCloseingTagName(c);
			} else if(this._state === AFTER_CLOSING_TAG_NAME){
				this._stateAfterCloseingTagName(c);
			} else if(this._state === IN_SELF_CLOSING_TAG){
				this._stateInSelfClosingTag(c);
			}
	
			/*
			*	attributes
			*/
			else if(this._state === BEFORE_ATTRIBUTE_NAME){
				this._stateBeforeAttributeName(c);
			} else if(this._state === IN_ATTRIBUTE_NAME){
				this._stateInAttributeName(c);
			} else if(this._state === AFTER_ATTRIBUTE_NAME){
				this._stateAfterAttributeName(c);
			} else if(this._state === BEFORE_ATTRIBUTE_VALUE){
				this._stateBeforeAttributeValue(c);
			} else if(this._state === IN_ATTRIBUTE_VALUE_DQ){
				this._stateInAttributeValueDoubleQuotes(c);
			} else if(this._state === IN_ATTRIBUTE_VALUE_SQ){
				this._stateInAttributeValueSingleQuotes(c);
			} else if(this._state === IN_ATTRIBUTE_VALUE_NQ){
				this._stateInAttributeValueNoQuotes(c);
			}
	
			/*
			*	declarations
			*/
			else if(this._state === BEFORE_DECLARATION){
				this._stateBeforeDeclaration(c);
			} else if(this._state === IN_DECLARATION){
				this._stateInDeclaration(c);
			}
	
			/*
			*	processing instructions
			*/
			else if(this._state === IN_PROCESSING_INSTRUCTION){
				this._stateInProcessingInstruction(c);
			}
	
			/*
			*	comments
			*/
			else if(this._state === BEFORE_COMMENT){
				this._stateBeforeComment(c);
			} else if(this._state === IN_COMMENT){
				this._stateInComment(c);
			} else if(this._state === AFTER_COMMENT_1){
				this._stateAfterComment1(c);
			} else if(this._state === AFTER_COMMENT_2){
				this._stateAfterComment2(c);
			}
	
			/*
			*	cdata
			*/
			else if(this._state === BEFORE_CDATA_1){
				this._stateBeforeCdata1(c);
			} else if(this._state === BEFORE_CDATA_2){
				this._stateBeforeCdata2(c);
			} else if(this._state === BEFORE_CDATA_3){
				this._stateBeforeCdata3(c);
			} else if(this._state === BEFORE_CDATA_4){
				this._stateBeforeCdata4(c);
			} else if(this._state === BEFORE_CDATA_5){
				this._stateBeforeCdata5(c);
			} else if(this._state === BEFORE_CDATA_6){
				this._stateBeforeCdata6(c);
			} else if(this._state === IN_CDATA){
				this._stateInCdata(c);
			} else if(this._state === AFTER_CDATA_1){
				this._stateAfterCdata1(c);
			} else if(this._state === AFTER_CDATA_2){
				this._stateAfterCdata2(c);
			}
	
			/*
			* special tags
			*/
			else if(this._state === BEFORE_SPECIAL){
				this._stateBeforeSpecial(c);
			} else if(this._state === BEFORE_SPECIAL_END){
				this._stateBeforeSpecialEnd(c);
			}
	
			/*
			* script
			*/
			else if(this._state === BEFORE_SCRIPT_1){
				this._stateBeforeScript1(c);
			} else if(this._state === BEFORE_SCRIPT_2){
				this._stateBeforeScript2(c);
			} else if(this._state === BEFORE_SCRIPT_3){
				this._stateBeforeScript3(c);
			} else if(this._state === BEFORE_SCRIPT_4){
				this._stateBeforeScript4(c);
			} else if(this._state === BEFORE_SCRIPT_5){
				this._stateBeforeScript5(c);
			}
	
			else if(this._state === AFTER_SCRIPT_1){
				this._stateAfterScript1(c);
			} else if(this._state === AFTER_SCRIPT_2){
				this._stateAfterScript2(c);
			} else if(this._state === AFTER_SCRIPT_3){
				this._stateAfterScript3(c);
			} else if(this._state === AFTER_SCRIPT_4){
				this._stateAfterScript4(c);
			} else if(this._state === AFTER_SCRIPT_5){
				this._stateAfterScript5(c);
			}
	
			/*
			* style
			*/
			else if(this._state === BEFORE_STYLE_1){
				this._stateBeforeStyle1(c);
			} else if(this._state === BEFORE_STYLE_2){
				this._stateBeforeStyle2(c);
			} else if(this._state === BEFORE_STYLE_3){
				this._stateBeforeStyle3(c);
			} else if(this._state === BEFORE_STYLE_4){
				this._stateBeforeStyle4(c);
			}
	
			else if(this._state === AFTER_STYLE_1){
				this._stateAfterStyle1(c);
			} else if(this._state === AFTER_STYLE_2){
				this._stateAfterStyle2(c);
			} else if(this._state === AFTER_STYLE_3){
				this._stateAfterStyle3(c);
			} else if(this._state === AFTER_STYLE_4){
				this._stateAfterStyle4(c);
			}
	
			/*
			* entities
			*/
			else if(this._state === BEFORE_ENTITY){
				this._stateBeforeEntity(c);
			} else if(this._state === BEFORE_NUMERIC_ENTITY){
				this._stateBeforeNumericEntity(c);
			} else if(this._state === IN_NAMED_ENTITY){
				this._stateInNamedEntity(c);
			} else if(this._state === IN_NUMERIC_ENTITY){
				this._stateInNumericEntity(c);
			} else if(this._state === IN_HEX_ENTITY){
				this._stateInHexEntity(c);
			}
	
			else {
				this._cbs.onerror(Error("unknown _state"), this._state);
			}
	
			this._index++;
		}
		this._eof = this._index; // TODO Orion 11.0 Track the end of file for recovery
		this._cleanup();
	};
	
	Tokenizer.prototype.pause = function(){
		this._running = false;
	};
	Tokenizer.prototype.resume = function(){
		this._running = true;
	
		if(this._index < this._buffer.length){
			this._parse();
		}
		if(this._ended){
			this._finish();
		}
	};
	
	Tokenizer.prototype.end = function(chunk){
		if(this._ended) this._cbs.onerror(Error(".end() after done!"));
		if(chunk) this.write(chunk);
	
		this._ended = true;
	
		if(this._running) this._finish();
	};
	
	Tokenizer.prototype._finish = function(){
		//if there is remaining data, emit it in a reasonable way
		if(this._sectionStart < this._index){
			this._handleTrailingData();
		}
	
		this._cbs.onend();
	};
	
	Tokenizer.prototype._handleTrailingData = function(){
		var data = this._buffer.substr(this._sectionStart);
	
		if(this._state === IN_CDATA || this._state === AFTER_CDATA_1 || this._state === AFTER_CDATA_2){
			this._cbs.oncdata(data);
		} else if(this._state === IN_COMMENT || this._state === AFTER_COMMENT_1 || this._state === AFTER_COMMENT_2){
			this._cbs.oncomment(data);
		} else if(this._state === IN_NAMED_ENTITY && !this._xmlMode){
			this._parseLegacyEntity();
			if(this._sectionStart < this._index){
				this._state = this._baseState;
				this._handleTrailingData();
			}
		} else if(this._state === IN_NUMERIC_ENTITY && !this._xmlMode){
			this._decodeNumericEntity(2, 10);
			if(this._sectionStart < this._index){
				this._state = this._baseState;
				this._handleTrailingData();
			}
		} else if(this._state === IN_HEX_ENTITY && !this._xmlMode){
			this._decodeNumericEntity(3, 16);
			if(this._sectionStart < this._index){
				this._state = this._baseState;
				this._handleTrailingData();
			}
		} else if(
			this._state !== IN_TAG_NAME &&
			this._state !== BEFORE_ATTRIBUTE_NAME &&
			this._state !== BEFORE_ATTRIBUTE_VALUE &&
			this._state !== AFTER_ATTRIBUTE_NAME &&
			this._state !== IN_ATTRIBUTE_NAME &&
			this._state !== IN_ATTRIBUTE_VALUE_SQ &&
			this._state !== IN_ATTRIBUTE_VALUE_DQ &&
			this._state !== IN_ATTRIBUTE_VALUE_NQ &&
			this._state !== IN_CLOSING_TAG_NAME
		){
			this._cbs.ontext(data);
		}
		//else, ignore remaining data
		//TODO add a way to remove current tag
	};
	
	Tokenizer.prototype.reset = function(){
		Tokenizer.call(this, {xmlMode: this._xmlMode, decodeEntities: this._decodeEntities}, this._cbs);
	};
	
	Tokenizer.prototype.getAbsoluteIndex = function(){
		return this._bufferOffset + this._index;
	};
	
	Tokenizer.prototype._getSection = function(){
		return this._buffer.substring(this._sectionStart, this._index);
	};
	
	Tokenizer.prototype._emitToken = function(name){
		this._cbs[name](this._getSection());
		this._sectionStart = -1;
		this._quoteStart = -1;
	};
	
	Tokenizer.prototype._emitPartial = function(value){
		if(this._baseState !== TEXT){
			this._cbs.onattribdata(value); //TODO implement the new event
		} else {
			this._cbs.ontext(value);
		}
	};
	
	return Tokenizer;
});

/*******************************************************************************
 * @license
 * Copyright (c) 2013, 2015 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*
 * Shim for Node events API
 * http://nodejs.org/api/events.html
 */
/*eslint-env amd */
/*global console*/
define('eslint/lib/events',[
	"orion/EventTarget",
	"orion/objects"
], function(EventTarget, objects) {
	var DEFAULT_MAX_LISTENERS = 10;

	function EventEmitter() {
		this._eventTarget = new EventTarget();
	}

	function addListener(eventName, listener) {
		if (typeof listener !== "function") {
			throw new Error("addListener only takes instances of Function");
		}
		var max = typeof this._maxListeners !== "undefined" ? this._maxListeners : DEFAULT_MAX_LISTENERS;
		var count;
		if (max !== 0 && (count = EventEmitter.listenerCount(this, eventName) >= max)) {
			if (typeof console !== "undefined") {
				console.error("Possible EventEmitter memory leak: " + count + " listeners added.");
			}
		}
		this.emit("newListener", listener);
		this._eventTarget.addEventListener(eventName, listener);
		return this;
	}

	EventEmitter.prototype.constructor = EventEmitter;
	objects.mixin(EventEmitter.prototype, {
		_maxListeners: 10,
		addListener: addListener,
		on: addListener,
		once: function(eventName, listener) {
			var emitter = this;
			var oneTimeListener = /* @callback */ function(event) {
				try {
					listener.apply(this, Array.prototype.slice.call(arguments));
				} finally {
					emitter.removeListener(eventName, oneTimeListener);
				}
			};
			this.addListener(eventName, oneTimeListener);
			return this;
		},
		removeListener: function(eventName, listener) {
			if (typeof listener !== "function") {
				throw new Error("removeListener only takes instances of Function");
			}
			this._eventTarget.removeEventListener(eventName, listener);
			this.emit("removeListener", listener);
			return this;
		},
		removeAllListeners: function(eventName) {
			var namedListeners = this._eventTarget._namedListeners;
			var emitter = this;
			var removeAllListenersFor = function(eventName) {
				var listeners = namedListeners[eventName];
				if (!listeners) {
					return;
				}
				listeners.forEach(emitter.emit.bind(emitter, "removeListener"));
				delete namedListeners[eventName];
			};
			if (typeof eventName === "undefined") {
				Object.keys(namedListeners).forEach(removeAllListenersFor);
			} else {
				removeAllListenersFor(eventName);
			}
			return this;
		},
		setMaxListeners: function(n) {
			if (typeof n !== "number") {
				throw new Error("setMaxListeners only takes a number");
			}
			this._maxListeners = n;
		},
		listeners: function(eventName) {
			var listeners = this._eventTarget._namedListeners[eventName];
			return listeners ? listeners.slice() : [];
		},
		emit: function emit(eventName /*, arg1, arg2, ...*/) {
			var listeners = this._eventTarget._namedListeners[eventName];
			if (!listeners) {
				if (eventName === "error") {
					throw new Error("Uncaught, unspecified 'error' event.");
				}
				return false;
			}
			var args = Array.prototype.slice.call(arguments, 1);
			var emitter = this;
			listeners.forEach(function(listener) {
				// To match Node's behavior we intentionally allow an exception thrown by listener to blow up the stack.
				listener.apply(emitter, args);
			});
			return true;
		}
	});
	EventEmitter.listenerCount = function(emitter, eventName) {
		var listeners = emitter._eventTarget._namedListeners[eventName];
		return listeners ? listeners.length : 0;
	};

	return {
		EventEmitter: EventEmitter
	};
});
/* eslint-env amd */
/* eslint-disable missing-nls */
define('htmlparser2/parser',[
'htmlparser2/tokenizer',
'eslint/lib/events'
], function(Tokenizer, events) {
	
	/*
		Options:
	
		xmlMode: Disables the special behavior for script/style tags (false by default)
		lowerCaseAttributeNames: call .toLowerCase for each attribute name (true if xmlMode is `false`)
		lowerCaseTags: call .toLowerCase for each tag name (true if xmlMode is `false`)
	*/
	
	/*
		Callbacks:
	
		oncdataend,
		oncdatastart,
		onclosetag,
		oncomment,
		oncommentend,
		onerror,
		onopentag,
		onprocessinginstruction,
		onreset,
		ontext
	*/
	
	var formTags = {
		input: true,
		option: true,
		optgroup: true,
		select: true,
		button: true,
		datalist: true,
		textarea: true
	};
	
	var openImpliesClose = {
		tr      : { tr:true, th:true, td:true },
		th      : { th:true },
		td      : { thead:true, th:true, td:true },
		body    : { head:true, link:true, script:true },
		li      : { li:true },
		p       : { p:true },
		h1      : { p:true },
		h2      : { p:true },
		h3      : { p:true },
		h4      : { p:true },
		h5      : { p:true },
		h6      : { p:true },
		select  : formTags,
		input   : formTags,
		output  : formTags,
		button  : formTags,
		datalist: formTags,
		textarea: formTags,
		option  : { option:true },
		optgroup: { optgroup:true }
	};
	
	var voidElements = {
		__proto__: null,
		area: true,
		base: true,
		basefont: true,
		br: true,
		col: true,
		command: true,
		embed: true,
		frame: true,
		hr: true,
		img: true,
		input: true,
		isindex: true,
		keygen: true,
		link: true,
		meta: true,
		param: true,
		source: true,
		track: true,
		wbr: true,
	
		//common self closing svg elements
		path: true,
		circle: true,
		ellipse: true,
		line: true,
		rect: true,
		use: true,
		stop: true,
		polyline: true,
		polygon: true
	};
	
	var re_nameEnd = /\s|\//;
	
	function Parser(cbs, options){
		this._options = options || {};
		this._cbs = cbs || {};
	
		this._tagname = "";
		this._attribname = "";
		this._attribvalue = null; // TODO Orion 11.0 Make null to distinguish between no value and =""
		this._attribrange = [];
		this._attribs = null;
		this._stack = [];
	
		this.startIndex = 0;
		this.endIndex = null;
	
		this._lowerCaseTagNames = "lowerCaseTags" in this._options ?
										!!this._options.lowerCaseTags :
										!this._options.xmlMode;
		this._lowerCaseAttributeNames = "lowerCaseAttributeNames" in this._options ?
										!!this._options.lowerCaseAttributeNames :
										!this._options.xmlMode;
	
		this._tokenizer = new Tokenizer(this._options, this);
	
		if(this._cbs.onparserinit) this._cbs.onparserinit(this);
	}
	
	/**
	 * ORION
	 * shim for node.js util.inherits
	 */
	function inherits(ctor, sctor) {
		ctor.prototype = Object.create(sctor.prototype);
		ctor._super = sctor;
	}
	
	inherits(Parser, events.EventEmitter);
	
	Parser.prototype._updatePosition = function(initialOffset){
		if(this.endIndex === null){
			if(this._tokenizer._sectionStart <= initialOffset){
				this.startIndex = 0;
			} else {
				this.startIndex = this._tokenizer._sectionStart - initialOffset;
			}
		}
		else this.startIndex = this.endIndex + 1;
		this.endIndex = this._tokenizer.getAbsoluteIndex();
	};
	
	//Tokenizer event handlers
	Parser.prototype.ontext = function(data){
		this._updatePosition(1);
		this.endIndex--;
	
		if(this._cbs.ontext) this._cbs.ontext(data);
	};
	
	Parser.prototype.onopentagname = function(name){
		if(this._lowerCaseTagNames){
			name = name.toLowerCase();
		}
	
		this._tagname = name;
	
		if(!this._options.xmlMode && name in openImpliesClose) {
			for(
				var el;
				(el = this._stack[this._stack.length - 1]) in openImpliesClose[name];
				this.onclosetag(el)
			);
		}
	
		if(this._options.xmlMode || !(name in voidElements)){
			this._stack.push(name);
		}
	
		if(this._cbs.onopentagname) this._cbs.onopentagname(name, this._tokenizer._sectionStart-1);
		if(this._cbs.onopentag) this._attribs = {};
	};
	
	Parser.prototype.onopentagend = function(){
		this._updatePosition(1);
	
		if(this._attribs){
			if(this._cbs.onopentag) this._cbs.onopentag(this._tagname, this._attribs, [this.startIndex, this.endIndex+1]); //ORION
			this._attribs = null;
		}
	
		if(!this._options.xmlMode && this._cbs.onclosetag && this._tagname in voidElements){
			this._cbs.onclosetag(this._tagname);
		}
	
		this._tagname = "";
	};
	
	Parser.prototype.onclosetag = function(name){
		this._updatePosition(1);
	
		if(this._lowerCaseTagNames){
			name = name.toLowerCase();
		}
	
		if(this._stack.length && (!(name in voidElements) || this._options.xmlMode)){
			var pos = this._stack.lastIndexOf(name);
			if(pos !== -1){
				if(this._cbs.onclosetag){
					pos = this._stack.length - pos;
					while(pos--) this._cbs.onclosetag(this._stack.pop(), [this.startIndex, this.endIndex+1]); //ORION
				}
				else this._stack.length = pos;
			} else if(name === "p" && !this._options.xmlMode){
				this.onopentagname(name);
				this._closeCurrentTag();
			}
		} else if(!this._options.xmlMode && (name === "br" || name === "p")){
			this.onopentagname(name);
			this._closeCurrentTag();
		}
	};
	
	Parser.prototype.onselfclosingtag = function(){
		if(this._options.xmlMode || this._options.recognizeSelfClosing){
			this._closeCurrentTag();
		} else {
			this.onopentagend();
		}
	};
	
	Parser.prototype._closeCurrentTag = function(){
		var name = this._tagname;
	
		this.onopentagend();
	
		//self-closing tags will be on the top of the stack
		//(cheaper check than in onclosetag)
		if(this._stack[this._stack.length - 1] === name){
			if(this._cbs.onclosetag){
				this._cbs.onclosetag(name);
			}
			this._stack.pop();
		}
	};
	
	Parser.prototype.onattribname = function(name){
		this._attribrange[0] = this._tokenizer._sectionStart; // TODO Orion 11.0 Collect attribute ranges
		if (this._cbs.onattribname) this._cbs.onattribname(name, this._attribrange[0]); // TODO Orion 11.0 Collect open tags to recover
		if(this._lowerCaseAttributeNames){
			name = name.toLowerCase();
		}
		this._attribname = name;
	};
	
	Parser.prototype.onattribdata = function(value){
		if (this._attribvalue){
			this._attribvalue += value;
		} else {
			this._attribvalue = value; // TODO Orion 11.0 Distinguish between no value and =""
		}

	};
	
	Parser.prototype.onattribend = function(isQuoted, valueStart){
		// TODO Orion 11.0 Collect attribute ranges
		if (valueStart) {
			this._attribrange[1] = valueStart;
			this._attribrange[2] = this._tokenizer.getAbsoluteIndex();
			// If the attribute is in quotes include them in the range
			if (isQuoted){
				this._attribrange[2]++;
			}
		} else {
			this._attribrange[1] = this._tokenizer.getAbsoluteIndex();
			// If the attribute is in quotes include them in the range
			if (isQuoted){
				this._attribrange[1]++;
			}
		}
		if(this._cbs.onattribute) this._cbs.onattribute(this._attribname, this._attribvalue, this._attribrange);
		if(
			this._attribs &&
			!Object.prototype.hasOwnProperty.call(this._attribs, this._attribname)
		){
			this._attribs[this._attribname] = this._attribvalue;
		}
		this._attribname = "";
		this._attribvalue = null;
	};
	
	Parser.prototype._getInstructionName = function(value){
		var idx = value.search(re_nameEnd),
		    name = idx < 0 ? value : value.substr(0, idx);
	
		if(this._lowerCaseTagNames){
			name = name.toLowerCase();
		}
	
		return name;
	};
	
	Parser.prototype.ondeclaration = function(value){
		if(this._cbs.onprocessinginstruction){
			var name = this._getInstructionName(value);
			this._cbs.onprocessinginstruction("!" + name, "!" + value, [this.startIndex, this._tokenizer.getAbsoluteIndex()]); //ORION
		}
	};
	
	Parser.prototype.onprocessinginstruction = function(value){
		if(this._cbs.onprocessinginstruction){
			var name = this._getInstructionName(value);
			this._cbs.onprocessinginstruction("?" + name, "?" + value, [this.startIndex, this._tokenizer.getAbsoluteIndex()]); //ORION
		}
	};
	
	Parser.prototype.oncomment = function(value){
		this._updatePosition(4);
	
		if(this._cbs.oncomment) this._cbs.oncomment(value, [this.startIndex, this.endIndex]); //ORION
		if(this._cbs.oncommentend) this._cbs.oncommentend([this.startIndex, this.endIndex]); //ORION
	};
	
	Parser.prototype.oncdata = function(value){
		this._updatePosition(1);
	
		if(this._options.xmlMode || this._options.recognizeCDATA){
			if(this._cbs.oncdatastart) this._cbs.oncdatastart();
			if(this._cbs.ontext) this._cbs.ontext(value);
			if(this._cbs.oncdataend) this._cbs.oncdataend();
		} else {
			this.oncomment("[CDATA[" + value + "]]", [this.startIndex, this.endIndex]); //ORION
		}
	};
	
	Parser.prototype.onerror = function(err){
		if(this._cbs.onerror) this._cbs.onerror(err);
	};
	
	Parser.prototype.onend = function(){
		if(this._cbs.onclosetag){
			for(
				var i = this._stack.length;
				i > 0;
				this._cbs.onclosetag(this._stack[--i])
			);
		}
		if(this._cbs.onend) {
			this._cbs.onend([this.startIndex, this._tokenizer._eof]); //ORION
		}
	};
	
	
	//Resets the parser to a blank state, ready to parse a new HTML document
	Parser.prototype.reset = function(){
		if(this._cbs.onreset) this._cbs.onreset();
		this._tokenizer.reset();
	
		this._tagname = "";
		this._attribname = "";
		this._attribs = null;
		this._stack = [];
	
		if(this._cbs.onparserinit) this._cbs.onparserinit(this);
	};
	
	//Parses a complete HTML document and pushes it to the handler
	Parser.prototype.parseComplete = function(data){
		this.reset();
		this.end(data);
	};
	
	Parser.prototype.write = function(chunk){
		this._tokenizer.write(chunk);
	};
	
	Parser.prototype.end = function(chunk){
		this._tokenizer.end(chunk);
	};
	
	Parser.prototype.pause = function(){
		this._tokenizer.pause();
	};
	
	Parser.prototype.resume = function(){
		this._tokenizer.resume();
	};
	
	//alias for backwards compat
	Parser.prototype.parseChunk = Parser.prototype.write;
	Parser.prototype.done = Parser.prototype.end;
	
	return  Parser;
});

/*******************************************************************************
 * @license
 * Copyright (c) 2013, 2016 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html).
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env amd*/
/*globals Tautologistics */
define('webtools/htmlAstManager',[
	'orion/Deferred',
	'orion/objects',
	'javascript/lru',
	'htmlparser2/parser',
], function(Deferred, Objects, LRU, HtmlParser2) {

	var handler = {
		ast: null,
		tagstack: [],
		errors: [],
		attribstack: [],
		/** @callback */
	    onopentag: function(name, attribs, range){
	    	this.trailingTag = null;
	    	var node = Object.create(null);
	    	node.range = [0, 0];
	    	if(Array.isArray(range)) {
	    		node.range[0] = range[0];
	    		node.range[1] = range[1]; // Inline tags don't have separate onclosetag calls
	    		node.openrange = [range[0], range[1]];
	    	} 
	    	node.name = name;
	    	node.type = 'tag'; //$NON-NLS-1$
	    	
	    	node.attributes = Object.create(null);
	    	for (var i=0; i<this.attribstack.length; i++){
	    		var attrib = this.attribstack[i];
	    		if (attrib.name){
	    			node.attributes[attrib.name] = attrib;
	    		}
	    	}
 	    	node.children = [];
	    	var tag = this._getLastTag();
	    	if(tag) {
	    		tag.children.push(node);
	    	} else {
	    		this.ast.children.push(node);
	    	}
	    	this.attribstack = [];
	    	this.tagstack.push(node);
	    },
	    /** @callback */
	    onclosetag: function(tagname, range){
	    	var tag = this._getLastTag();
	    	if(tag && tag.name === tagname) {
	    		if (range){
	    			tag.range[1] = range[1];
	    			tag.endrange = [range[0], range[1]];
    			} else {
    				// No matching closing tag or void element
    				// TODO Need to add tests for this, can it have children as well as text?
    				if (tag.openrange && tag.text){
    					tag.range[1] = tag.openrange[1] + tag.text.value.length;
    					tag.endrange = [tag.range[1], tag.range[1]];
    				}	
    			}
	    		this.tagstack.pop();
	    	}
	    },
	    /** @callback */
	    onopentagname: function(name, rangeStart) {
	    	this.trailingTag = {name: name, start: rangeStart};
	    },
	    /** @callback */
	    onattribname: function(name, rangeStart) {
	    	this.trailingAttrib = {name: name, start: rangeStart};
	    },
	    /** @callback */
	    onattribute: function(name, value, range) {
	    	this.trailingAttrib = null;
	    	var node = Object.create(null);
	    	node.value = value;
	    	if (Array.isArray(range)) {
			if (range.length === 3) {
				node.range = [range[0], range[2]];
				if ((range[2] - range[1]) <= 1) {
					// no actual value was specified
					node.value = null;
				} else {
					node.valueRange = [range[1], range[2]];
				}
			} else {
				node.range = [range[0], range[1]];
			}
		} else {
			node.range = [0, 0];
		}
	    	node.name = name;
	    	node.type = 'attr'; //$NON-NLS-1$
	    	this.attribstack.push(node);
	    },
	    /** @callback *//** @callback */
	    onprocessinginstruction: function(name, data, range) {
	    	var node = Object.create(null);
	    	node.range = Array.isArray(range) ? [range[0], range[1]] : [0, 0];
	    	node.name = name;
	    	node.type = 'instr'; //$NON-NLS-1$
	    	node.value = data;
	    	var tag = this._getLastTag();
	    	if(tag) {
	    		tag.children.push(node);
	    	} else {
	    		this.ast.children.push(node);
	    	}
	    },
	    /** @callback */
	    oncomment: function(data, range) {
	    	var node = Object.create(null);
	    	node.range = Array.isArray(range) ? [range[0], range[1]] : [0, 0];
	    	node.type = 'comment'; //$NON-NLS-1$
	    	node.data = data;
	    	
	    	var tag = this._getLastTag();
	    	if(tag) {
	    		tag.children.push(node);
	    	} else {
	    		this.ast.children.push(node);
	    	}
	    },
	    /** @callback */
	    oncdatastart: function() {
	    	var node = Object.create(null);
	    	node.range = [0, 0];
	    	node.type = 'cdata'; //$NON-NLS-1$
	    },
	    /** @callback */
	    ontext: function(text) {
	    	var node = Object.create(null);
	    	node.range = [0, 0];
	    	node.type = 'text'; //$NON-NLS-1$
	    	node.value = text;
	    	var tag = this._getLastTag();
	    	if(tag) {
	    		tag.text = node;
	    	}
	    },
	    /** @callback */
	    onerror: function(error) {
	    	var err = Object.create(null);
	    	err.error = error;
	    	err.range = [0, 0];
	    	this.errors.push(err);
	    },
	    /** @callback */
	    onend: function(range) {
	    	// The ordering is important here as trailing attributes need to be added to the tag
	    	if (this.trailingAttrib){
	    		// TODO Recover trailing value
	    		this.onattribute(this.trailingAttrib.name, null, [this.trailingAttrib.start,range[1]]);
	    	}
	    	if (this.trailingTag){
	    		this.onopentag(this.trailingTag.name, null, [this.trailingTag.start,range[1]]);
	    	}
	    	
	    	if(Array.isArray(range)) {
	    		this.ast.range[0] = this.ast.children.length > 0 ? this.ast.children[0].range[0] : 0;
	    		this.ast.range[1] = range[1];
	    	}
	    },
	    /** @callback */
	    onreset: function() {
			this.ast = Object.create(null);
			this.ast.range = [0,0];
			this.ast.children = [];
			this.tagstack = [];
			this.comments = [];
			this.errors = [];
			this.attribstack = [];
	    },
	    /** @callback */
	    _getLastTag: function() {
	    	if(this.tagstack && this.tagstack.length > 0) {
	    		return this.tagstack[this.tagstack.length-1];
	    	} 
	    	return null;
	    }
	};
	
	var registry;

	/**
	 * Provides a shared AST.
	 * @class Provides a shared parsed AST.
	 * @param {Object} serviceRegistry The platform service registry
	 * @since 8.0
	 */
	function HtmlAstManager(serviceRegistry) {
		this.cache = new LRU(10);
		registry = serviceRegistry;
	}
	
	/**
	 * @description Delegate to log timings to the metrics service
	 * @param {Number} end The end time
	 * @since 12.0
	 */
	function logTiming(end) {
		if(registry) {
			var metrics = registry.getService("orion.core.metrics.client"); //$NON-NLS-1$
			if(metrics) {
				metrics.logTiming('language tools', 'parse', end, 'text/html'); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
			}
		}
	}

	Objects.mixin(HtmlAstManager.prototype, /** @lends webtools.HtmlAstManager.prototype */ {
		/**
		 * @param {orion.editor.EditorContext} editorContext
		 * @returns {orion.Promise} A promise resolving to the AST.
		 */
		getAST: function(editorContext) {
			var _self = this;
			return editorContext.getFileMetadata().then(function(metadata) {
				metadata = metadata || {};
				var loc = _self._getKey(metadata);
				var ast = _self.cache.get(loc);
				if (ast) {
					return new Deferred().resolve(ast);
				}
				return editorContext.getText().then(function(text) {
					ast = _self.parse(text);
					_self.cache.put(loc, ast);
					if(metadata.location) {
					    //only set this if the original metadata has a real location
					    ast.fileLocation = metadata.location;
					}
					return ast;
				});
			});
		},
		/**
		 * Returns the key to use when caching
		 * @param {Object|String} metadata The file infos
		 */
		_getKey: function _getKey(metadata) {
			if(typeof metadata === 'string') {
				return metadata;
			}
			if(!metadata.location) {
				return 'unknown'; //$NON-NLS-1$
			}
			return metadata.location;
		},

		/**
		 * @private
		 * @param {String} text The code to parse.
		 * @returns {Object} The AST.
		 */
		parse: function(text) {
			var parser = new HtmlParser2(handler, {decodeEntities: true, recognizeSelfClosing: true});
			var start = Date.now();
			parser.reset();
			parser.write(text);
			parser.done();
			var end = Date.now()-start;
			if(handler.ast) {
				handler.ast.source = text;
			}
			logTiming(end);
			return handler.ast;
		},

		/**
		 * Callback from the orion.edit.model service
		 * @param {Object} event An <tt>orion.edit.model</tt> event.
		 * @see https://wiki.eclipse.org/Orion/Documentation/Developer_Guide/Plugging_into_the_editor#orion.edit.model
		 */
		onModelChanging: function(event) {
		    if(this.inputChanged) {
		        //TODO haxxor, eat the first model changing event which immediately follows
		        //input changed
		        this.inputChanged = null;
		    } else {
		        this.cache.remove(this._getKey(event.file));
		    }
		},
		/**
		 * Callback from the orion.edit.model service
		 * @param {Object} event An <tt>orion.edit.model</tt> event.
		 * @see https://wiki.eclipse.org/Orion/Documentation/Developer_Guide/Plugging_into_the_editor#orion.edit.model
		 */
		onInputChanged: function(event) {
		    this.inputChanged = event;
		},
		/**
		 * Callback from the FileClient
		 * @param {Object} event a <tt>fileChanged</tt> event
		 */
		onFileChanged: function(event) {
			if(event && event.type === 'Changed' && Array.isArray(event.modified)) {
				event.modified.forEach(function(file) {
					if(typeof file === 'string') {
						this.cache.remove(this._getKey(file));
					}
				}.bind(this));
			}
		}
	});
	return {
		HtmlAstManager : HtmlAstManager
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2011, 2012 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
define('orion/URITemplate',[],function(){
	
	var OPERATOR = {
		NUL: {first:"", sep:",", named: false, ifemp: "", allow: "U"}, //$NON-NLS-2$ //$NON-NLS-1$ //$NON-NLS-0$
		"+": {first:"", sep:",", named: false, ifemp: "", allow: "U+R"}, //$NON-NLS-3$ //$NON-NLS-2$ //$NON-NLS-1$ //$NON-NLS-0$
		".": {first:".", sep:",", named: false, ifemp: "", allow: "U"}, //$NON-NLS-3$ //$NON-NLS-2$ //$NON-NLS-1$ //$NON-NLS-0$
		"/": {first:"/", sep:"/", named: false, ifemp: "", allow: "U"}, //$NON-NLS-3$ //$NON-NLS-2$ //$NON-NLS-1$ //$NON-NLS-0$
		";": {first:";", sep:";", named: true, ifemp: "", allow: "U"}, //$NON-NLS-3$ //$NON-NLS-2$ //$NON-NLS-1$ //$NON-NLS-0$
		"?": {first:"?", sep:"&", named: true, ifemp: "=", allow: "U"}, //$NON-NLS-4$ //$NON-NLS-3$ //$NON-NLS-2$ //$NON-NLS-1$ //$NON-NLS-0$
		"&": {first:"&", sep:"&", named: true, ifemp: "=", allow: "U"}, //$NON-NLS-4$ //$NON-NLS-3$ //$NON-NLS-2$ //$NON-NLS-1$ //$NON-NLS-0$
		"#": {first:"#", sep:",", named: false, ifemp: "", allow: "U+R"}, //$NON-NLS-3$ //$NON-NLS-2$ //$NON-NLS-1$ //$NON-NLS-0$
		",": {first:"", sep:",", named: false, ifemp: "", allow: "U+R-,"} //$NON-NLS-3$ //$NON-NLS-2$ //$NON-NLS-1$ //$NON-NLS-0$
	};

	var VARSPEC_REGEXP = /^((?:(?:[a-zA-Z0-9_])|(?:%[0-9A-F][0-9A-F]))(?:(?:[a-zA-Z0-9_.])|(?:%[0-9A-F][0-9A-F]))*)(?:(\*)|:([0-9]+))?$/;
	var PCT_ENCODED_G = /%25[0-9A-F][0-9A-F]/g;

	function Literal(text) {
		this._text = text;
	}

	Literal.prototype = {
		expand: function(vars) {
			return encodeURI(this._text);
		}
	};
	
	function decodePercent(str) {
		return str.replace("%25", "%");
	}
	
	function encodeString(value, encoding) {
		if (encoding === "U") { //$NON-NLS-0$
			return encodeURIComponent(value).replace(/[!'()*]/g, function(str) {
				return '%' + str.charCodeAt(0).toString(16).toUpperCase(); //$NON-NLS-0$
			});
		}
		if (encoding === "U+R") { //$NON-NLS-0$
			return encodeURI(value).replace(/%5B/g, '[').replace(/%5D/g, ']').replace(PCT_ENCODED_G, decodePercent); //$NON-NLS-1$ //$NON-NLS-0$
		}
		if (encoding === "U+R-,") { //$NON-NLS-0$
			return encodeURI(value).replace(/%5B/g, '[').replace(/%5D/g, ']').replace(/,/g, '%2C'); //$NON-NLS-2$ //$NON-NLS-1$ //$NON-NLS-0$
		}
		throw new Error("Unknown allowed character set: " + encoding);
	}
	
	function encodeArray(value, encoding, separator) {
		var result = [];
		for (var i=0; i < value.length; i++) {
			if (typeof(value[i]) !== "undefined") { //$NON-NLS-0$
				result.push(encodeString(value[i], encoding));
			}
		}
		return result.join(separator);
	}
	
	function encodeObject(value, encoding, nameValueSeparator, pairSeparator ) {
		var keys = Object.keys(value);
		var result = [];
		for (var i=0; i < keys.length; i++) {
			if (typeof(value[keys[i]]) !== "undefined") { //$NON-NLS-0$
				result.push(encodeString(keys[i], encoding) + nameValueSeparator + encodeString(value[keys[i]], encoding));
			}
		}
		return result.join(pairSeparator);
	}
	
	function parseVarSpecs(text) {
		var result = [];
		var rawSpecs = text.split(","); //$NON-NLS-0$
		for (var i=0; i < rawSpecs.length; i++) {
			var match = rawSpecs[i].match(VARSPEC_REGEXP);
			if (match === null) {
				throw new Error("Bad VarSpec: " + text); //$NON-NLS-0$
			}
			result.push({
				name: match[1], 
				explode: !!match[2], 
				prefix: match[3] ? parseInt(match[3], 10) : -1
			}); 
		}
		return result;
	}
	
	function Expression(text) {
		if (text.length === 0) {
			throw new Error("Invalid Expression: 0 length expression"); //$NON-NLS-0$
		}
		
		this._operator = OPERATOR[text[0]];
		if (this._operator) {
			text = text.substring(1);
		} else {
			this._operator = OPERATOR.NUL;
		}
		
		this._varSpecList = parseVarSpecs(text);
	}
	
	Expression.prototype = {
		expand: function(params) {
			var result = [];
			for (var i=0; i < this._varSpecList.length; i++) {
				var varSpec = this._varSpecList[i];
				var name = varSpec.name;
				var value = params[name];
				var valueType = typeof(value);
				if (valueType !== "undefined" && value !== null) { //$NON-NLS-0$
					var resultText = result.length === 0 ? this._operator.first: this._operator.sep;			
					if (valueType === "string") { //$NON-NLS-0$
						if (this._operator.named) {
							resultText += encodeString(name, "U+R"); //$NON-NLS-0$
							resultText += (value.length === 0) ? this._operator.ifemp : "="; //$NON-NLS-0$
						}
						if (varSpec.prefix !== -1 && varSpec.prefix < value.length) {
							value = value.substring(0, varSpec.prefix);
						}
						
						resultText += encodeString(value, this._operator.allow);
					} else if (Array.isArray(value)) {
						if (value.length === 0) {
							continue; // treated as undefined and skipped
						}
						if (!varSpec.explode) {
							var encodedArray = encodeArray(value, this._operator.allow, ","); //$NON-NLS-0$
							if (this._operator.named) {
								resultText += encodeString(name, "U+R"); //$NON-NLS-0$
								resultText += (encodedArray.length === 0) ? this._operator.ifemp : "="; //$NON-NLS-0$
							}
							resultText += encodedArray;
						} else {
							resultText += encodeArray(value, this._operator.allow, this._operator.sep);
						}				
					} else if (valueType === "object") { //$NON-NLS-0$
						if (Object.keys(value).length === 0) {
							continue; // treated as undefined and skipped
						}
						if (!varSpec.explode) {
							var encodedObject = encodeObject(value, this._operator.allow, ",", ","); //$NON-NLS-1$ //$NON-NLS-0$
							if (this._operator.named) {
								resultText += encodeString(name, "U+R"); //$NON-NLS-0$
								resultText += (encodedObject.length === 0) ? this._operator.ifemp : "="; //$NON-NLS-0$
							}
							resultText += encodedObject; //$NON-NLS-0$
						} else {
							resultText += encodeObject(value, this._operator.allow, "=", this._operator.sep); //$NON-NLS-0$
						}
					} else {
						throw new Error("bad param type: " + name + " : " + valueType); //$NON-NLS-1$ //$NON-NLS-0$
					}
					result.push(resultText);
				}
			}
			return result.join("");
		}
	};

	function parseTemplate(text) {
		var result = [];
		var current = 0;	
		var curlyStartIndex = text.indexOf("{", current); //$NON-NLS-0$
		while (curlyStartIndex !== -1) {
			result.push(new Literal(text.substring(current, curlyStartIndex)));
			var curlyEndIndex = text.indexOf("}", curlyStartIndex + 1); //$NON-NLS-0$
			if (curlyEndIndex === -1) {
				throw new Error("Invalid template: " + text); //$NON-NLS-0$
			}
			result.push(new Expression(text.substring(curlyStartIndex + 1, curlyEndIndex)));
			current = curlyEndIndex + 1;
			curlyStartIndex = text.indexOf("{", current);			 //$NON-NLS-0$
		}
		result.push(new Literal(text.substring(current)));
		return result;
	}

	/**
	 * @name orion.URITemplate
	 * @class A URITemplate describes a range of Uniform Resource Identifiers through variable expansion, and allows for particular URIs to 
	 * be generated by expanding variables to actual values.</p>
	 * <p>Because the syntax and encoding rules of URIs can be complex, URITemplates are recommended over manual construction of URIs through 
	 * string concatenation or other means.</p>
	 * <p>A URITemplate is created by invoking the constructor, passing a <em>template string</em>:</p>
	 * <p><code>new URITemplate(template)</code></p>
	 * <p>The <dfn>template string</dfn> is an expression following a well-defined syntax (see <a href="http://tools.ietf.org/html/rfc6570#section-1.2">here</a>
	 * for an introduction). Most notably, the template may include variables.</p>
	 * <p>Once created, a URITemplate's {@link #expand} method can be invoked to generate a URI. Arguments to {@link #expand} give the values to be 
	 * substituted for the template variables.</p>
	 * @description Creates a new URITemplate.
	 * @param {String} template The template string. Refer to <a href="http://tools.ietf.org/html/rfc6570#section-2">RFC 6570</a> for details
	 * of the template syntax.
	 */
	function URITemplate(template) {
		this._templateComponents = parseTemplate(template);
	}
	
	URITemplate.prototype = /** @lends orion.URITemplate.prototype */ {
		/**
		 * Expands this URITemplate to a URI.
		 * @param {Object} params The parameters to use for expansion. This object is a map of keys (variable names) to values (the variable's
		 * value in the <a href="http://tools.ietf.org/html/rfc6570#section-3.2.1">expansion algorithm</a>).
		 * @returns {String} The resulting URI.
		 */
		expand: function(params) {
			var result = [];
			for (var i = 0; i < this._templateComponents.length; i++) {
				result.push(this._templateComponents[i].expand(params));
			}
			return result.join("");
		}
	};

	return URITemplate;
});
/*******************************************************************************
 * @license
 * Copyright (c) 2014, 2015 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env amd*/
define('htmlparser2/visitor',[
], function() {

	var Visitor = {
        
        BREAK: 1,
        SKIP: 2,
        
        /**
         * Visit the given AST top-down node -> attributes -> value -> end visit
         * @param {Object} dom	The AST to visit
         * @param {Function} callback The visitor callback 
         */
        visit: function visit(dom, callback) {
            if(Array.isArray(dom) && callback && typeof(callback.visitNode) === 'function') {
                for(var i = 0; i < dom.length; i++) {
                    var ret = visitNode(callback, dom[i], null);
                    endVisitNode(callback, dom[i]);
                    if(ret === this.BREAK) {
                        return;
                    }
                }    
            }
        }
    };

	/**
	 * Visits a given node with a given callback function. if the function does not implement #visitNode this function does nothing
	 * @param {Function} callback The visitor callback. Can optionally implement #visitNode
	 * @param {Object} node The current node being visited
	 * @param {Object} last the last node that was visited
	 * @returns {Number} Returns #Break or #SKIP or nothing   
	 */
    function visitNode(callback, node, last) {
    	if(typeof(callback.visitNode) === 'function') {
	        node.parent = last;
	        var ret = callback.visitNode(node);
	        if(ret === Visitor.BREAK || ret === Visitor.SKIP) {
	            return ret;
	        } 
	        if(node.attributes) {
	            var attrs = node.attributes;
	            var keys = Object.keys(attrs);
	            var attr;
	            for(var i = 0; i < keys.length; i++) {
	                attr = attrs[keys[i]];
	                attr.parent = node;
	                attr.kind = keys[i];
	                ret = callback.visitNode(attr);
	                if(typeof(callback.endVisitNode) === 'function') {
			    		callback.endVisitNode(attr);
			    	}
	                if(ret === Visitor.SKIP) {
	                    break;
	                } else if(ret === Visitor.BREAK) {
	                    return ret;
	                }
	            }
	        }
	        var kids = node.children;
	        if(kids) {
	            for(i = 0; i < kids.length; i++) {
	                ret = visitNode(callback, kids[i], node);
	                if(typeof(callback.endVisitNode) === 'function') {
			    		callback.endVisitNode(kids[i]);
			    	}
	                if(ret === Visitor.BREAK) {
	                    return ret;
	                } else if(ret === Visitor.SKIP) {
	                    continue;
	                }
	            }
	        }
        }
    }
    
    /**
     * Ends the visit on the given node with the given callback. Allows for post-processing when we are going to leave a node. If the callback does not implement
     * #endVisitNode this function does nothing
     * @param {Function} callback The visitor callback. Can optionally implement #endVisitNode
     * @param {Object} node The node we are ending the visit for
     * @since 10.0
     */
    function endVisitNode(callback, node) {
    	if(typeof(callback.endVisitNode) === 'function') {
    		callback.endVisitNode(node);
    	}
    }

    return Visitor;    
});
/*******************************************************************************
/*******************************************************************************
 * @license
 * Copyright (c) 2013, 2015 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env amd*/
define('webtools/util',[
'htmlparser2/visitor'
], function(Visitor) {

	var Util = {
		
		punc: '\n\t\r (){}[]:;,.+=-*^&@!%~`\'\"\/\\',  //$NON-NLS-0$
		
		/**
		 * Returns the ast node at the given offset or the parent node enclosing it
		 * @param {Object} ast The AST to inspect
		 * @param {Number} offset The offset into the source 
		 * @returns {Object} The AST node at the given offset or null 
		 * @since 10.0
		 */
		findNodeAtOffset: function(ast, offset) {
			var found = null;
			var dom = ast;
			if (!Array.isArray(ast) && ast.children){
				dom = ast.children;
			}
			 Visitor.visit(dom, {
	            visitNode: function(node) {
					if(node.range[0] <= offset) {
						found = node;
					} else {
					    return Visitor.BREAK;
					}      
	            },
	            endVisitNode: function(node) {
	            	if(found && offset >= found.range[1] && offset > node.range[0]) {
	            		found = node;
	            	}
	            }
	        });
	        return found;
		},
		
		/**
		 * @description Finds the word from the start position
		 * @function
		 * @public
		 * @memberof webtools.Util
		 * @param {String} text The text of the source to find the word in
		 * @param {Number} start The current start position of the carat
		 * @returns {String} Returns the computed word from the given string and offset or <code>null</code>
		 */
		findWord: function(text, start) {
			if(text && start) {
				var ispunc = this.punc.indexOf(text.charAt(start)) > -1;
				var pos = ispunc ? start-1 : start;
				while(pos >= 0) {
					if(this.punc.indexOf(text.charAt(pos)) > -1) {
						break;
					}
					pos--;
				}
				var s = pos;
				pos = start;
				while(pos <= text.length) {
					if(this.punc.indexOf(text.charAt(pos)) > -1) {
						break;
					}
					pos++;
				}
				if((s === start || (ispunc && (s === start-1))) && pos === start) {
					return null;
				} else if(s === start) {
					return text.substring(s, pos);
				}
				return text.substring(s+1, pos);
			}
			return null;
		},
		
		/**
		 * @description Finds the token in the given token stream for the given start offset
		 * @function
		 * @public
		 * @memberof webtools.Util
		 * @param {Number} offset The offset intot the source
		 * @param {Array|Object} tokens The array of tokens to search
		 * @returns {Object} The token that starts at the given start offset
		 */
		findToken: function(offset, tokens) {
			if(offset !== null && offset > -1 && tokens && tokens.length > 0) {
				var min = 0,
					max = tokens.length-1,
					token, 
					idx = 0;
					token = tokens[0];
				if(offset >= token.range[0] && offset < token.range[1]) {
					token.index = 0;
					return token;
				}
				token = tokens[max];
				if(offset >= token.range[0]) {
					token.index = max;
					return token;
				}
				token = null;
				while(min <= max) {
					idx = Math.floor((min + max) / 2);
					token = tokens[idx];
					if(offset < token.range[0]) {
						max = idx-1;
					}
					else if(offset > token.range[1]) {
						min = idx+1;
					}
					else if(offset === token.range[1]) {
						var next = tokens[idx+1];
						if(next.range[0] === token.range[1]) {
							min = idx+1;
						}
						else {
							token.index = idx;
							return token;
						}
					}
					else if(offset >= token.range[0] && offset < token.range[1]) {
						token.index = idx;
						return token;
					}
					if(min === max) {
						token = tokens[min];
						if(offset >= token.range[0] && offset <= token.range[1]) {
							token.index = min;
							return token;
						}
						return null;
					}
				}
			}
			return null;
		},
		
		/**
		 * @description Finds the style blocks from an HTML file and returns the code and offset for found blocks
		 * @function
		 * @public
		 * @param {String} buffer The file contents
		 * @param {Number} offset The offset into the buffer to find the enclosing block for
		 * @returns {Object} An object of script block items {text, offset}
		 * @since 8.0
		 */
		findStyleBlocks: function(buffer, offset) {
			var blocks = [];
			var val = null, regex = /<\s*style(?:type\s*=\s*"([^"]*)"|[^>]|\n)*>((?:.|\r?\n)*?)<\s*\/style(?:[^>]|\n)*>/ig;
			var comments = this.findHtmlCommentBlocks(buffer, offset);
			loop: while((val = regex.exec(buffer)) !== null) {
				var text = val[2];
				if(text.length < 1) {
					continue;
				}
				var index = val.index+val[0].indexOf('>')+1;  //$NON-NLS-0$
				if(typeof offset !== 'number' || (index <= offset && index+text.length >= offset)) {
					for(var i = 0; i < comments.length; i++) {
						if(comments[i].start <= index && comments[i].end >= index) {
							continue loop;
						}
					}
					blocks.push({
						text: text,
						offset: index
					});
				}
			}
			return blocks;
		},
		
		/**
		 * @description Finds all of the block comments in an HTML file
		 * @function
		 * @public
		 * @param {String} buffer The file contents
		 * @param {Number} offset The optional offset to compute the block(s) for
		 * @return {Array} The array of block objects {text, start, end}
		 * @since 8.0
		 */
		findHtmlCommentBlocks: function(buffer, offset) {
			var blocks = [];
			var val = null, regex = /<!--((?:.|\r?\n)*?)-->/ig;
			while((val = regex.exec(buffer)) !== null) {
				var text = val[1];
				if(text.length < 1) {
					continue;
				}
				if(typeof offset !== 'number' || (val.index <= offset && val.index+text.length >= val.index)) {
					blocks.push({
						text: text,
						start: val.index,
						end: val.index+text.length
					});
				}
			}
			return blocks;
		}
	};

	return Util;
});

 /*******************************************************************************
 * @license
 * Copyright (c) 2015 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env amd, node*/
define('webtools/htmlHover',[
'orion/objects',
'orion/URITemplate',
'webtools/util'
], function(Objects, URITemplate, util) {
	
	/**
	 * @description creates a new instance of the hover support
	 * @constructor
	 * @public
	 * @param {Object} htmlAstManager The shared AST manager for HTML DOMs
	 * @param {Object} resolver The shared script resolver
	 * @since 8.0
	 */
	function HTMLHover(htmlAstManager, resolver) {
	    this.htmlAstManager = htmlAstManager;
	    this.resolver = resolver;
	}
	
	Objects.mixin(HTMLHover.prototype, /** @lends webtools.HTMLHover.prototype*/ {
		
		_hoverableTags: ['div', 'h1', 'h2', 'h3', 'h4', 'h5', 'body', 'p', 'ol', 'ul', 'li', 'table', 'td', 'tr', 'textarea', 'select', 'form', 'pre'],
		
		/**
		 * @description Callback from the editor to compute the hover
		 * @function
		 * @public 
		 * @memberof webtools.HTMLHover.prototype
		 * @param {Object} editorContext The current editor context
		 * @param {Object} ctxt The current selection context
		 * @callback
		 */
		computeHoverInfo: function computeHover(editorContext, ctxt) {
			if(ctxt.proposal && ctxt.proposal.kind === 'html') {
				return ctxt.proposal.hover ? ctxt.proposal.hover : (ctxt.proposal.name ? ctxt.proposal.name : ctxt.proposal.description);
			}
			var that = this;
			return that.htmlAstManager.getAST(editorContext).then(function(ast) {
			    if(ast) {
			        var node = util.findNodeAtOffset(ast, ctxt.offset);
			        if(node) {
			            switch(node.type) {
			                case 'tag': {
			                	if (that._hoverableTags.indexOf(node.name) >= 0){
			                		//return that._getTagContentsHover(editorContext, node.range);
			                	}
			                    break;
			                }
			                case 'attr': {
			                    var path = node.value;
			                    switch(node.kind) {
			                        case 'href': {
			                            if(/\.(?:png|jpg|jpeg|bmp|gif)$/.test(path)) {
                            	            return that._getImageHover(editorContext, path);
                            	        } else if(/^data:image.*;base64/i.test(path)) {
                            	            return that._getImageHover(editorContext, path, true);
                            	        }
			                            return that._getFileHover(editorContext, path);
			                        }
			                        case 'src': {
			                            if(/\.(?:png|jpg|jpeg|bmp|gif)$/.test(path)) {
                            	            return that._getImageHover(editorContext, path);
                            	        } else if(/^data:image.*;base64/i.test(path)) {
                            	            return that._getImageHover(editorContext, path, true);
                            	        } else if(/\.js$/i.test(path)) {
                            	            return that._getFileHover(editorContext, path);
                            	        }
                            	        break;
			                        }
			                        case 'style': {
			                            //TODO support embedded style sheets
			                            break;
			                        }
			                    }
			                    break;
			                }
			            }
			        }
			    }
			    return null; 
			});
		},

		_getFileHover: function _getFileHover(editorContext, path) {
		    if(path) {
		        if(/^http/i.test(path)) {
    	            return this._formatFilesHover(path);
    	        }
		        var that = this;
		        var opts;
		        if(/\.css$/i.test(path)) {
		            opts = {ext:'css', type:'CSS', icon:'../webtools/images/css.png'}; //$NON-NLS-3$ //$NON-NLS-1$ //$NON-NLS-2$
		        } else if(/\.htm.*/i.test(path)) {
		            opts = {ext:'html', type:'HTML', icon:'../webtools/images/html.png'}; //$NON-NLS-3$ //$NON-NLS-1$ //$NON-NLS-2$
		        } else if(!/\.js$/i.test(path)) {
		            return null;
		        }
		        return editorContext.getFileMetadata().then(function(meta) {
		        	if(Array.isArray(meta.parents) && meta.parents.length > 0) {
						that.resolver.setSearchLocation(meta.parents[meta.parents.length - 1].Location);
					} else {
						that.resolver.setSearchLocation(null);	
					}
    		        return that.resolver.getWorkspaceFile(path, opts).then(function(files) {
    		        	var rels  = that.resolver.resolveRelativeFiles(path, files, meta);
        		    	if(rels && rels.length > 0) {
        		        	return that._formatFilesHover(path, rels);
        		    	}
    		        });    
		        });
		    }
		    return null;
		},
		
		/**
    	 * @description Formats the list of files as links for the hover
    	 * @function
    	 * @private
    	 * @param {String} path The path we are navigating to
    	 * @param {Array.<javascript.ScriptResolver.File>} files The array of files to linkify
    	 * @returns {String} The mardown to show in the hover
    	 */
    	_formatFilesHover: function _formatFilesHover(path, files) {
    	    if(path) {
    	        var title = null;
    	        if(files && files.length > 1) {
    	            '###Open file for \''+path+'\'###';
    	        }
    	        var hover = '';
    	        if(Array.isArray(files)) {  
        	        for(var i = 0; i < files.length; i++) {
        	            var file = files[i];
        	            if(file.name && file.path && file.contentType) {
        	                hover += '[';
        	                if(file.contentType.icon) {
        	                    hover += '!['+file.contentType.name+']('+file.contentType.icon+')';
        	                }
        	                var href = new URITemplate("#{,resource,params*}").expand(
        		                      {
        		                      resource: file.location, 
        		                      params: {}
        		                      });
        	                hover += file.name + ']('+href+') - '+file.path+'\n\n';
        	            }
        	            
        	        }
    	        } else {
    	            var site = false;
    	            var tmp = /^\s*http\s*:\s*\/\s*\/\s*(.*)/i.exec(path);
    	            if(tmp) {
    	                tmp = tmp[1];
    	                if(tmp.charAt(tmp.length-1) === '/') {
    	                   tmp = tmp.slice(0, tmp.length-1);
    	               }
    	               site = tmp.indexOf('/') === -1;
    	            }
    	            var name = path.slice(path.lastIndexOf('/')+1);
    	            if(site) {
    	                name = tmp;
    	            }
    	            if(site) {
    	               name = tmp;
    	               //title = '###Open site \''+name+'\'';
    	               hover += '[!['+name+'](../webtools/images/html.png)';
    	               hover += name + ']('+path+')\n\n';
    	            } else {
    	               //title = '###Open file for \''+name+'\'###';
        	            var img = null;
        	             if(/\.css$/i.test(path)) {
        		            img = '../webtools/images/css.png';
        		        } else if(/\.htm.*/i.test(path)) {
        		            img = '../webtools/images/html.png';
        		        } else {
        		            img = '../webtools/images/file.png';
        		        }
        		        hover += '[!['+name+']('+img+')';
    	                hover += name + ']('+path+') - '+path+'\n\n';
	                }
    	        }
    	        if(hover !== '') {
    	           return {title: title, content: hover, type:'markdown', allowFullWidth: true};
    	        }
    	    }
    	    return null;
    	},
		
		_getImageHover: function _getImageHover(editorContext, path, base64) {
		      if(path) {
		          if(/^http/i.test(path) || base64) {
    		          var html = '<html><body style="margin:1px;"><img src="'+path+'" style="width:100%;height:100%;"/></body></html>'; //$NON-NLS-0$  //$NON-NLS-1$
    			      return {type: "html", content: html, width: "100px", height: "100px"};  //$NON-NLS-0$  //$NON-NLS-1$  //$NON-NLS-2$
		          }
	              var idx = path.lastIndexOf('.');
	              if(idx > -1) {
	                  var ext = path.slice(idx+1);
	                  var that = this;
	                  return editorContext.getFileMetadata().then(function(meta) {
	                  	if(Array.isArray(meta.parents) && meta.parents.length > 0) {
							that.resolver.setSearchLocation(meta.parents[meta.parents.length - 1].Location);
						} else {
							that.resolver.setSearchLocation(null);	
						}
	                    return that.resolver.getWorkspaceFile(path, {ext:ext, type:'Image', icon:'../webtools/images/file.png'}).then(function(files) {
                			if(files && files.length > 0) {
                		    	var resolved = that.resolver.resolveRelativeFiles(path, files, meta);
                		        if(resolved && resolved.length > 0) {
                		        	var html = '<html><body style="margin:1px;"><img src="'+resolved[0].location+'" style="width:100%;height:100%;"/></body></html>'; //$NON-NLS-0$  //$NON-NLS-1$
    			                    return {type: "html", content: html, width: "100px", height: "100px"};  //$NON-NLS-0$  //$NON-NLS-1$  //$NON-NLS-2$
                		       	}
                		    }
            	        });
	              	});
	              }
		      }
		},
		
		_getColorHover: function _getColorHover(colorID){
			var html = '<html><body style=\"background-color: ' + colorID + ';\"></html>'; //$NON-NLS-0$  //$NON-NLS-1$
			return {type: "html", content: html, width: "50px", height: "25px"};  //$NON-NLS-0$  //$NON-NLS-1$  //$NON-NLS-2$
		},
		
		_getTagContentsHover: function _getTagContentsHover(editorContext, range){
			if (range){
				var self = this;
				return editorContext.getText().then(function(text) {
					if(range[0] >= 0 && range[0] < text.length && range[1] > range[0] && range[1] <= text.length){
						var start = self._findTagStart(text, range[0]);
						if (start === null){
							return null;
						}
						var html = "<body style=\"background-color:white\">" + text.substring(start, range[1]) + "</body>";  //$NON-NLS-0$  //$NON-NLS-1$
						return {type: "html", content: html};  //$NON-NLS-0$  //$NON-NLS-1$
					}
				});
			}
			return null;
		},
		
		/**
		 * @description Returns the offset that the tag starts with (location of '<');
		 * @param {String} contents The text to search for the tag start
		 * @param {Number} offset The offset in the contents to start the search
		 * @returns {Number} offset of the tag or <code>null</code>
		 * @since 8.0
		 */
		_findTagStart: function(contents, offset) {
			if(contents && offset) {
				var pos = offset;
				while(pos >= 0) {
					if(contents.charAt(pos) === '<') {
						return pos;
					}
					pos--;
				}
			}
			return null;
		}
		
	});

	HTMLHover.prototype.contructor = HTMLHover;
	
	return {
		HTMLHover: HTMLHover
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2010, 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 * 
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd*/
define("orion/editor/templates", [], function() { //$NON-NLS-0$

	/** 
	 * Removes prefix from string.
	 * @param {String} prefix
	 * @param {String} string
	 */
	function chop(prefix, string) {
		return string.substring(prefix.length);
	}
	
	var tabVar = "${tab}"; //$NON-NLS-0$
	var delimiterVar = "${delimiter}"; //$NON-NLS-0$
	var cursorVar = "${cursor}"; //$NON-NLS-0$
	
	function Template (prefix, description, template, name) {
		this.prefix = prefix;
		this.description = description;
		this.template = template;
		this.name = name;
		this._parse();
	}
	Template.prototype = /** @lends orion.editor.Template.prototype */ {
		getProposal: function(prefix, offset, context) {
			//any returned positions need to be offset based on current cursor position and length of prefix
			var startOffset = offset-prefix.length;
			var groups = {};
			var escapePosition;
			var delimiter = context.delimiter !== undefined ? context.delimiter : "\n"; //$NON-NLS-0$
			if (context.indentation) {
				delimiter += context.indentation;
			}
			var tab = context.tab !== undefined ? context.tab : "\t"; //$NON-NLS-0$
			var delta = 0;
			var variables = this.variables;
			var segments = this.segments, proposal = [];
			for (var i = 0; i < segments.length; i++) {
				var segment = segments[i];
				var variable = variables[segment];
				if (variable !== undefined) {
					switch (segment) {
						case tabVar:
							segment = tab;
							break;
						case delimiterVar:
							segment = delimiter;
							break;
						case cursorVar:
							segment = "";
							escapePosition = delta;
							break;
						default:
							var g = groups[segment];
							if (!g) {
								g = groups[segment] = {data: variable.data, positions: []};
							}
							segment = variable.substitution;
							if (g.data && g.data.values) { segment = g.data.values[0]; }
							g.positions.push({
								offset: startOffset + delta,
								length: segment.length
							});
					}
				}
				proposal.push(segment);
				delta += segment.length;
			}
			var newGroups = [];
			for (var p in groups) {
				if (groups.hasOwnProperty(p)) {
					newGroups.push(groups[p]);
				}
			}
			proposal = proposal.join("");
			if (escapePosition === undefined) {
				escapePosition = proposal.length;
			}
			return {
				proposal: proposal,
				name: this.name,
				description: this.description,
				groups: newGroups,
				escapePosition: startOffset + escapePosition,
				style: 'noemphasis'
			};
		},
		match: function(prefix) {
			return this.prefix.indexOf(prefix) === 0;
		},
		_parse: function() {
			var template = this.template;
			var segments = [], variables = {}, segment, start = 0;
			template = template.replace(/\n/g, delimiterVar);
			template = template.replace(/\t/g, tabVar);
			template.replace(/\$\{((?:[^\\}]+|\\.))*\}/g, function(group, text1, index) {
				var text = group.substring(2,group.length-1);
				var variable = group, substitution = text, data = null;
				var colon = substitution.indexOf(":"); //$NON-NLS-0$
				if (colon !== -1) {
					substitution = substitution.substring(0, colon);
					variable = "${"+ substitution + "}"; //$NON-NLS-1$ //$NON-NLS-0$
					data = JSON.parse(text.substring(colon + 1).replace("\\}", "}").trim()); //$NON-NLS-1$ //$NON-NLS-0$
				}
				var v = variables[variable];
				if (!v) { v = variables[variable] = {}; }
				v.substitution = substitution;
				if (data) {
					v.data = data;
				}
				segment = template.substring(start, index);
				if (segment) { segments.push(segment); }
				segments.push(variable);
				start = index + group.length;
				return substitution;
			});
			segment = template.substring(start, template.length);
			if (segment) { segments.push(segment); }
			this.segments = segments;
			this.variables = variables;
		}
	};
	
	function TemplateContentAssist (keywords, templates) {
		this._keywords = keywords || [];
		this._templates = [];
		this.addTemplates(templates || []);
	}
	TemplateContentAssist.prototype = /** @lends orion.editor.TemplateContentAssist.prototype */ {
		addTemplates: function(json) {
			var templates = this.getTemplates();
			for (var j = 0; j < json.length; j++) {
				templates.push(new Template(json[j].prefix, json[j].description, json[j].template, json[j].name));
			}
		},
		/**
		 * Called by the content assist engine to initialize this provider before any <tt>computeProposals()</tt> calls.
		 * This implementation does nothing; subclasses may override.
		 */
		initialize: function() {
		},
		computeProposals: function(buffer, offset, context) {
			var prefix = this.getPrefix(buffer, offset, context);
			var proposals = [];
			if (this.isValid(prefix, buffer, offset, context)) {
				proposals = proposals.concat(this.getTemplateProposals(prefix, offset, context));
				proposals = proposals.concat(this.getKeywordProposals(prefix));
			}
			return proposals;
		},
		getKeywords: function() {
			return this._keywords;
		},
		getKeywordProposals: function(prefix) {
			var proposals = [];
			var keywords = this.getKeywords();
			if (keywords) {
				for (var i = 0; i < keywords.length; i++) {
					if (keywords[i].indexOf(prefix) === 0) {
						proposals.push({proposal: chop(prefix, keywords[i]), 
							description: keywords[i], 
							style: 'noemphasis_keyword'//$NON-NLS-0$
						});
					}
				}
				
				if (0 < proposals.length) {
					proposals.splice(0, 0,{
						proposal: '',
						description: 'Keywords', //$NON-NLS-0$
						style: 'noemphasis_title_keywords', //$NON-NLS-0$
						unselectable: true
					});	
				}
			}
			return proposals;
		},
		getPrefix: function(buffer, offset, context) {
			return context.prefix;
		},
		getTemplates: function() {
			return this._templates;
		},
		getTemplateProposals: function(prefix, offset, context) {
			var proposals = [];
			var templates = this.getTemplates();
			for (var t = 0; t < templates.length; t++) {
				var template = templates[t];
				if (template.match(prefix)) {
					var proposal = template.getProposal(prefix, offset, context);
					this.removePrefix(prefix, proposal);
					proposals.push(proposal);
				}
			}
			
			if (0 < proposals.length) {
				//sort the proposals by name
				proposals.sort(function(p1, p2) {
					if (p1.name < p2.name) return -1;
					if (p1.name > p2.name) return 1;
					return 0;
				});
				// if any templates were added to the list of 
				// proposals, add a title as the first element
				proposals.splice(0, 0, {
					proposal: '',
					description: 'Templates', //$NON-NLS-0$
					style: 'noemphasis_title', //$NON-NLS-0$
					unselectable: true
				});
			}
			
			return proposals;
		},
		removePrefix: function(prefix, proposal) {
			var overwrite = proposal.overwrite = proposal.proposal.substring(0, prefix.length) !== prefix;
			if (!overwrite) {
				proposal.proposal = chop(prefix, proposal.proposal);
			}
		},
		isValid: function(prefix, buffer, offset, context) {
			return true;
		}
	};
	
	return {
		Template: Template,
		TemplateContentAssist: TemplateContentAssist
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2015, 2016 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env amd*/
define('javascript/util',[
], function() {
	/**
	 * @description Returns if the given character is upper case or not considering the locale
	 * @param {String} string A string of at least one char14acter
	 * @return {Boolean} True iff the first character of the given string is uppercase
	 */
	 function isUpperCase(string) {
		if (string.length < 1) {
		return false;
		}
		if (isNaN(string.charCodeAt(0))) {
			return false;
		}
		return string.toLocaleUpperCase().charAt(0) === string.charAt(0);
	}
	
	/**
	 * @description Match ignoring case and checking camel case.
	 * @param {String} prefix
	 * @param {String} target
	 * @returns {Boolean} If the two strings match
	 */
	function looselyMatches(prefix, target) {
		if (typeof prefix !== "string" || typeof target !== "string") {
			return false;
		}

		// Zero length string matches everything.
		if (prefix.length === 0) {
			return true;
		}

		// Exclude a bunch right away
		if (prefix.charAt(0).toLowerCase() !== target.charAt(0).toLowerCase()) {
			return false;
		}

		if (startsWith(target, prefix)) {
			return true;
		}

		var lowerCase = target.toLowerCase();
		if (startsWith(lowerCase, prefix)) {
			return true;
		}
		
		var _prefix = prefix.toLowerCase();

		var equalIndex = prefix.indexOf("=");
		if (equalIndex !== -1) {
			if (startsWith(target, prefix.substring(0, equalIndex))) {
				return true;
			}
		}
		// Test for camel characters in the prefix.
		if (prefix === _prefix) {
			return false;
		}
		//https://bugs.eclipse.org/bugs/show_bug.cgi?id=473777
		if(startsWith(lowerCase, _prefix)) {
			return true;
		}
		var prefixParts = toCamelCaseParts(prefix);
		var targetParts = toCamelCaseParts(target);

		if (prefixParts.length > targetParts.length) {
			return false;
		}

		for (var i = 0; i < prefixParts.length; ++i) {
			if (!startsWith(targetParts[i], prefixParts[i])) {
				return false;
			}
		}

		return true;
	}
	
	/**
	 * @description Returns if the string starts with the given prefix
	 * @param {String} s The string to check
	 * @param {String} pre The prefix 
	 * @returns {Boolean} True if the string starts with the prefix
	 */
	function startsWith(s, pre) {
		return s.slice(0, pre.length) === pre;
	}
	
	/**
	 * @description Convert an input string into parts delimited by upper case characters. Used for camel case matches.
	 * e.g. GroClaL = ['Gro','Cla','L'] to match say 'GroovyClassLoader'.
	 * e.g. mA = ['m','A']
	 * @function
	 * @public
	 * @param {String} str
	 * @return Array.<String>
	 */
	function toCamelCaseParts(str) {
		var parts = [];
		for (var i = str.length - 1; i >= 0; --i) {
			if (isUpperCase(str.charAt(i))) {
				parts.push(str.substring(i));
				str = str.substring(0, i);
			}
		}
		if (str.length !== 0) {
			parts.push(str);
		}
		return parts.reverse();
	}
	
	var emptyAST = {
		type: "Program", //$NON-NLS-0$
		body: [],
		comments: [],
		tokens: [],
		range: [0, 0],
		loc: {
			start: {},
			end: {}
		}
	};
	
	/**
	 * @description Creates a new empty AST for the fatal thrown error case
	 * @param {Object} error The fatal error thrown while trying to parse
	 * @param {String} name The name of the file we tried to parse
	 * @param {String} text The text we tried to parse
	 * @returns {Object} An empty AST with the fatal error attached in the errors array
	 * @since 11.0
	 */
	function errorAST(error, name, text) {
		var ast = emptyAST;
		ast.range[1] = typeof text === 'string' ? text.length : 0;
		ast.loc.start.line = error.lineNumber;
		ast.loc.start.column = 0;
		ast.loc.end.line = error.lineNumber;
		ast.loc.end.column = error.column;
		ast.errors = [error];
        ast.sourceFile  = Object.create(null);
        ast.sourceFile.text = text;
        ast.sourceFile.name = name;
        return ast;
	}
	
	/**
	 * @description Makes the errors from the given AST safe to transport (using postMessage for example)
	 * @param {Object} ast The AST to serialize errors for
	 * @returns {Array.<Object>} The searialized errors
	 * @since 11.0
	 */
	function serializeAstErrors(ast) {
		var errors = [];
		if(ast && ast.errors) {
			ast.errors.forEach(function(error) {
				var result = error ? JSON.parse(JSON.stringify(error)) : error; // sanitizing Error object
				if (error instanceof Error) {
					result.__isError = true;
					result.lineNumber = typeof result.lineNumber === 'number' ? result.lineNumber : error.lineNumber; //FF fails to include the line number from JSON.stringify
					result.message = result.message || error.message;
					result.name = result.name || error.name;
					result.stack = result.stack || error.stack;
				}
				var msg = error.message;
				result.message = msg = msg.replace(/^Line \d+: /, '');
				errors.push(result);
			});
		}
		return errors;
	}

	return {
		isUpperCase: isUpperCase,
		looselyMatches: looselyMatches,
		startsWith: startsWith,
		toCamelCaseParts: toCamelCaseParts,
		errorAST: errorAST,
		serializeAstErrors: serializeAstErrors
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2015 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/* eslint-env amd */
define('webtools/attributes',[

], function() {
	/* eslint-disable missing-nls */
	var attributes = Object.create(null);
	/*
	 * ADDED MISSING TAGS
	 * - xmlns
	 */
	attributes.globals = [
			{
				name: "accesskey",
				url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Global_attributes/accesskey",
				doc: "Provides a hint for generating a keyboard shortcut for the current element. This attribute consists of a space-separated list of characters. The browser should use the first one that exists on the computer keyboard layout."
			},
			{
 				name: "class",
 				url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Global_attributes/class",
 				doc: "Is a space-separated list of the classes of the element. Classes allows CSS and JavaScript to select and access specific elements via the class selectors or functions like the method Document.getElementsByClassName()."
 			},
 			{
 				name: "contenteditable",
 				url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Global_attributes/contenteditable",
 				doc: "Is an enumerated attribute indicating if the element should be editable by the user. If so, the browser modifies its widget to allow editing.",
				values: [
					{name: "true", doc: "(or empty string) indicates that the element must be editable"},
					{name: "false", doc: "the element must not be editable"}
				]
			},
 			{
 				name: "contextmenu",
 				url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Global_attributes/contextmenu",
 				doc: "Is the id of an <menu> to use as the contextual menu for this element."
 			},
 			{
 				name: "dataset",
 				url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Global_attributes/data-*",
 				doc: "Forms a class of attributes, called custom data attributes, that allow proprietary information to be exchanged between the HTML and its DOM representation that may be used by scripts. All such custom data are available via the HTMLElement interface of the element the attribute is set on. The HTMLElement.dataset property gives access to them."
 			},
 			{
 				name: "dir",
 				url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Global_attributes/dir",
 				doc: "Is an enumerated attribute indicating the directionality of the element's text.",
				values: [
					{name: "ltr", doc: "means left to right and is to be used for languages that are written from the left to the right (like English)"},
					{name: "rtl", doc: "means right to left and is to be used for languages that are written from the right to the left (like Arabic)"},
					{name: "auto", doc: "lets the user agent decide. It uses a basic algorithm as it parses the characters inside the element until it finds a character with a strong directionality, then apply that directionality to the whole element"}
				]
			},
 			{
 				name: "draggable",
 				url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Global_attributes/draggable",
 				doc: "Is an enumerated attribute indicating whether the element can be dragged, using the Drag and Drop API.",
				values: [
					{name: "true", doc: "the element may be dragged"},
					{name: "false", doc: "the element may not be dragged"}
				]
 			},
 			{
 				name: "dropzone",
 				url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Global_attributes/dropzone",
 				doc: "Is an enumerated attribute indicating what types of content can be dropped on an element, using the Drag and Drop API.",
				values: [
					{name: "copy", doc: "dropping will create a copy of the element that was dragged"},
					{name: "move", doc: "the element that was dragged will be moved to this new location"},
					{name: "link", doc: "create a link to the dragged data"}
				]
 			},
 			{
 				name: "hidden",
 				url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Global_attributes/hidden",
 				doc: "Is a Boolean attribute indicates that the element is not yet, or is no longer, relevant. For example, it can be used to hide elements of the page that can't be used until the login process has been completed. The browser won't render such elements. This attribute must not be used to hide content that could legitimately be shown."
 			},
 			{
 				name: "id",
 				url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Global_attributes/id",
 				doc: "Defines a unique identifier (ID) which must be unique in the whole document. Its purpose is to identify the element when linking (using a fragment identifier), scripting, or styling (with CSS)."
 			},
// 			{
// 				name: "itemid",
// 				url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Global_attributes/itemid",
// 				doc: "This attribute is related to the WHATWG HTML Microdata feature.",
// 			},
// 			{
// 				name: "itemprop",
// 				url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Global_attributes/itemprop",
// 				doc: "This attribute is related to the WHATWG HTML Microdata feature.",
// 			},
// 			{
// 				name: "itemref",
// 				url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Global_attributes/itemref",
// 				doc: "This attribute is related to the WHATWG HTML Microdata feature.",
// 			},
// 			{
// 				name: "itemscope",
// 				url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Global_attributes/itemscope",
// 				doc: "This attribute is related to the WHATWG HTML Microdata feature.",
// 			},
// 			{
// 				name: "itemtype",
// 				url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Global_attributes/itemtype",
// 				doc: "This attribute is related to the WHATWG HTML Microdata feature.",
// 			},
 			{
 				name: "lang",
 				url: "/en-US/docs/Web/HTML/Global_attributes/lang",
 				doc: "Participates in defining the language of the element, the language that non-editable elements are written in or the language that editable elements should be written in. The tag contains one single entry value in the format defines in the Tags for Identifying Languages (BCP47) IETF document. xml:lang has priority over it."
 			},
 			{
 				name: "spellcheck",
				url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Global_attributes/spellcheck",
				doc: "Is an enumerated attribute that defines whether the element may be checked for spelling errors.",
				values: [
					{name: "true", doc: "the element should be, if possible, checked for spelling errors"},
  					{name: "false", doc: "the element should not be checked for spelling errors"}
  				]
 			},
 			{
 				name: "style",
 				url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Global_attributes/style",
 				doc: "Contains CSS styling declarations to be applied to the element. Note that it is recommended for styles to be defined in a separate file or files. This attribute and the <style> element have mainly the purpose of allowing for quick styling, for example for testing purposes."
 			},
 			{
 				name: "tabindex",
 				url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Global_attributes/tabindex",
 				doc: "Is an integer attribute that indicates if the element can take input focus (is focusable), if it should participate in sequential keyboard navigation, and if so, at what position.",
				values: [
					{name: "-1", doc: "a negative value means that the element should be focusable, but should not be reachable via sequential keyboard navigation"},
					{name: "0", doc: "the element should be focusable and reachable via sequential keyboard navigation, but its relative order is defined by the platform convention"},
					{name: "positiveInteger", doc: "a positive value means that the element should be focusable and reachable via sequential keyboard navigation; its relative order is defined by the value of the attribute: the sequential follow the increasing number of the tabindex. If several elements share the same tabindex, their relative order follows their relative position in the document"}
				]
			},
			{
	 			name: "title",
	 			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Global_attributes/title",
	 			doc: "Contains a text representing advisory information related to the element it belongs to. Such information can typically, but not necessarily, be presented to the user as a tooltip."
 			},
 			{
 				name: "translate",
 				url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Global_attributes/translate",
 				doc: "Is an enumerated attribute that is used to specify whether an element's attribute values and the values of its Text node children are to be translated when the page is localized, or whether to leave them unchanged.",
				values: [
					{name: "yes", doc: "empty string and \"yes\" indicate that the element will be translated"},
					{name: "no", doc: "the element will not be translated"}
				]
 			}
	];
	
	attributes.windowevents = [
		//Window events
		{
			name: "onafterprint",
			url: "https://developer.mozilla.org/en-US/docs/Web/API/WindowEventHandlers/onafterprint",
			doc: "The WindowEventHandlers.onafterprint property sets and returns the onafterprint EventHandler for the current window."
		},
		{
			name: "onbeforeprint",
			url: "https://developer.mozilla.org/en-US/docs/Web/API/WindowEventHandlers.onbeforeprint",
			doc: "The onbeforeprint property sets and returns the onbeforeprint event handler code for the current window."
		},
		{
			name: "onbeforeunload",
			url: "https://developer.mozilla.org/en-US/docs/Web/API/WindowEventHandlers/onbeforeunload",
			doc: "An event that fires when a window is about to unload its resources. The document is still visible and the event is still cancelable."
		},
		{
			name : "onerror",
			url: "https://developer.mozilla.org/en-US/docs/Web/API/GlobalEventHandlers/onerror",
			doc: "An event handler for runtime script errors."
		},
		{
			name: "onhashchange",
			url: "https://developer.mozilla.org/en-US/docs/Web/API/WindowEventHandlers/onhashchange",
			doc: "The hashchange event fires when a window's hash changes (see location.hash)."
		},
		{
			name: "onload",
			url: "https://developer.mozilla.org/en-US/docs/Web/API/GlobalEventHandlers/onload",
			doc: "An event handler for the load event of a window."
		},
		{
			name: "onmessage",
			url: "https://developer.mozilla.org/en-US/docs/Web/API/WebSocket/onmessage",
			doc: "An event handler for message events"
		},
		{
			name: "onoffline",
			url: "https://developer.mozilla.org/en-US/docs/Web/API/Document/onoffline",
			doc: "This event handler is called when an offline is fired on body and bubbles up, when navigator.onLine property changes and becomes false."
		},
		{
			name: "ononline",
			url: "https://developer.mozilla.org/en-US/docs/Web/API/Document/ononline",
			doc: "\"online\" event is fired on the <body> of each page when the browser switches between online and offline mode. Additionally, the events bubble up from document.body, to document, ending at window. Both events are non-cancellable (you can't prevent the user from coming online, or going offline)."
		},
		{
			name: "onpagehide",
			url: "https://developer.mozilla.org/en-US/docs/Web/API/WindowEventHandlers/onpagehide",
			doc: "Is an EventHandler representing the code to be called when the pagehide event is raised."
		},
		{
			name: "onpageshow",
			url: "https://developer.mozilla.org/en-US/docs/Web/API/WindowEventHandlers/onpageshow",
			doc: "Is an EventHandler representing the code to be called when the pageshow event is raised."
		},
		{
			name: "onpopstate",
			url: "https://developer.mozilla.org/en-US/docs/Web/API/WindowEventHandlers/onpopstate",
			doc: "An event handler for the popstate event on the window."
		},
		{
			name: "onresize",
			url: "https://developer.mozilla.org/en-US/docs/Web/API/WindowEventHandlers/onresize",
			doc: "Is an EventHandler representing the code to be called when the resize event is raised."
		},
		{
			name: "onstorage",
			url: "https://developer.mozilla.org/en-US/docs/Web/API/WindowEventHandlers/onstorage",
			doc: "Is an EventHandler representing the code to be called when the storage event is raised."
		},
		{
			name: "onunload",
			url: "https://developer.mozilla.org/en-US/docs/Web/API/WindowEventHandlers/onunload",
			doc: "Is an EventHandler representing the code to be called when the unload event is raised."
		},
	];
	
	attributes.formevents = [
		//Form events
		{
			name: "onblur",
			url: "https://developer.mozilla.org/en-US/docs/Web/API/GlobalEventHandlers/onblur",
			doc: "The onblur property returns the onBlur event handler code, if any, that exists on the current element."
		},
		{
			name: "onchange",
			url: "https://developer.mozilla.org/en-US/docs/Web/API/GlobalEventHandlers/onchange",
			doc: "The onchange property sets and returns the event handler for the change event."
		},
		{
			name: "oncontextmenu",
			url: "https://developer.mozilla.org/en-US/docs/Web/API/GlobalEventHandlers/oncontextmenu",
			doc: "An event handler property for right-click events on the window. Unless the default behavior is prevented (see examples below on how to do this), the browser context menu will activate (though IE8 has a bug with this and will not activate the context menu if a contextmenu event handler is defined). Note that this event will occur with any non-disabled right-click event and does not depend on an element possessing the \"contextmenu\" attribute."
		},
		{
			name: "onfocus",
			url: "https://developer.mozilla.org/en-US/docs/Web/API/GlobalEventHandlers/onfocus",
			doc: "The onfocus property returns the onFocus event handler code on the current element."
		},
		{
			name: "oninput",
			url: "https://developer.mozilla.org/en-US/docs/Web/API/GlobalEventHandlers/oninput",
			doc: "An event handler for the input event on the window. The input event is raised when an <input> element value changes. "
		},
		{
			name: "oninvalid",
			url: "",
			doc: "The oninvalid event occurs when a submittable <input> element is invalid."
		},
		{
			name: "onreset",
			url: "https://developer.mozilla.org/en-US/docs/Web/API/GlobalEventHandlers/onreset",
			doc: "The GlobalEventHandlers.onreset property contains an EventHandler triggered when a reset event is received."
		},
		{
			name: "onsearch",
			url: "",
			doc: "The onsearch attribute fires when a user presses the \"ENTER\" key or clicks the \"x\" button in an <input> element with type=\"search\"."
		},
		{
			name: "onselect",
			url: "https://developer.mozilla.org/en-US/docs/Web/API/GlobalEventHandlers/onselect",
			doc: "An event handler for the select event on the window."
		},
		{
			name: "onsubmit",
			url: "https://developer.mozilla.org/en-US/docs/Web/API/GlobalEventHandlers/onsubmit",
			doc: "An event handler for the submit event on the window."
		}
	];

	attributes.keyboardevents = [
		//Keyboard events
		{
			name: "onkeydown",
			url: "https://developer.mozilla.org/en-US/docs/Web/API/GlobalEventHandlers/onkeydown",
			doc: "The onkeydown property returns the onKeyDown event handler code on the current element."
		},
		{
			name: "onkeypress",
			url: "https://developer.mozilla.org/en-US/docs/Web/API/GlobalEventHandlers/onkeypress",
			doc: "The onkeypress property sets and returns the onKeyPress event handler code for the current element."
		},
		{
			name: "onkeyup",
			url: "https://developer.mozilla.org/en-US/docs/Web/API/GlobalEventHandlers/onkeyup",
			doc: "The onkeyup property returns the onKeyUp event handler code for the current element."
		},
	];
	attributes.mouseevents = [
		//Mouse events
		{
			name: "onclick",
			url: "https://developer.mozilla.org/en-US/docs/Web/API/GlobalEventHandlers/onclick",
			doc: "The onclick property returns the click event handler code on the current element."
		},
		{
			name: "ondblclick",
			url: "https://developer.mozilla.org/en-US/docs/Web/API/GlobalEventHandlers/ondblclick",
			doc: "The ondblclick property returns the onDblClick event handler code on the current element."
		},
		{
			name: "ondrag",
			url: "http://www.w3schools.com/tags/ev_ondrag.asp",
			doc: "The ondrag attribute fires when an element or text selection is being dragged."
		},
		{
			name: "ondragend",
			url: "http://www.w3schools.com/tags/ev_ondragend.asp",
			doc: "The ondragend attribute fires when the user has finished dragging an element or text selection."
		},
		{
			name: "ondragenter",
			url: "http://www.w3schools.com/tags/ev_ondragenter.asp",
			doc: "The ondragenter attribute fires when a draggable element or text selection enters a valid drop target."
		},
		{
			name: "ondragleave",
			url: "http://www.w3schools.com/tags/ev_ondragleave.asp",
			doc: "The ondragleave attribute fires when a draggable element or text selection leaves a valid drop target."
		},
		{
			name: "ondragover",
			url: "http://www.w3schools.com/tags/ev_ondragover.asp",
			doc: "The ondragover attribute fires when a draggable element or text selection is being dragged over a valid drop target."
		},
		{
			name: "ondragstart",
			url: "http://www.w3schools.com/tags/ev_ondragstart.asp",
			doc: "The ondragstart attribute fires when the user starts to drag an element or text selection."
		},
		{
			name: "ondrop",
			url: "http://www.w3schools.com/tags/ev_ondrop.asp",
			doc: "The ondrop attribute fires when a draggable element or text selection is dropped on a valid drop target."
		},
		{
			name: "onmousedown",
			url: "https://developer.mozilla.org/en-US/docs/Web/API/GlobalEventHandlers/onmousedown",
			doc: "The onmousedown property returns the onmousedown event handler code on the current element."
		},
		{
			name: "onmousemove",
			url: "https://developer.mozilla.org/en-US/docs/Web/API/GlobalEventHandlers/onmousemove",
			doc: "The onmousemove property returns the mousemove event handler code on the current element."
		},
		{
			name: "onmouseout",
			url: "https://developer.mozilla.org/en-US/docs/Web/API/GlobalEventHandlers/onmouseout",
			doc: "The onmouseout property returns the onMouseOut event handler code on the current element."
		},
		{
			name: "onmouseover",
			url: "https://developer.mozilla.org/en-US/docs/Web/API/GlobalEventHandlers/onmouseover",
			doc: "The onmouseover property returns the onMouseOver event handler code on the current element."
		},
		{
			name: "onmouseup",
			url: "https://developer.mozilla.org/en-US/docs/Web/API/GlobalEventHandlers/onmouseup",
			doc: "The onmouseup property returns the onMouseUp event handler code on the current element."
		},
		{
			name: "onmousewheel",
			url: "",
			doc: "Deprecated. Use the onwheel attribute instead"
		},
		{
			name: "onscroll",
			url: "https://developer.mozilla.org/en-US/docs/Web/API/GlobalEventHandlers/onscroll",
			doc: "An event handler for scroll events on element."
		},
		{
			name: "onwheel",
			url: "http://www.w3schools.com/tags/ev_onwheel.asp",
			doc: "The onwheel attribute fires when the wheel of a pointing device is rolled up or down over an element."
		}
	];
	
	attributes.tags = Object.create(null);
	attributes.tags.a = [
		{
			name: "download",
			doc: "This attribute, if present, indicates that the author intends the hyperlink to be used for downloading a resource so that when the user clicks on the link they will be prompted to save it as a local file. If the attribute has a value, the value will be used as the pre-filled file name in the Save prompt that opens when the user clicks on the link (the user can change the name before actually saving the file of course). There are no restrictions on allowed values (though / and \ will be converted to underscores, preventing specific path hints), but you should consider that most file systems have limitations with regard to what punctuation is supported in file names, and browsers are likely to adjust file names accordingly.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/a#attr-download"
		},
		{
			name: "href",
			doc: "This was the single required attribute for anchors defining a hypertext source link, but is no longer required in HTML5. Omitting this attribute creates a placeholder link. The href attribute indicates the link target, either a URL or a URL fragment. A URL fragment is a name preceded by a hash mark (#), which specifies an internal target location (an ID) within the current document. URLs are not restricted to Web (HTTP)-based documents. URLs might use any protocol supported by the browser. For example, file, ftp, and mailto work in most user agents.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/a#attr-href"
		},
		{
			name: "hreflang",
			doc: "This attribute indicates the language of the linked resource. It is purely advisory. Allowed values are determined by BCP47 for HTML5 and by RFC1766 for HTML4. Use this attribute only if the href attribute is present.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/a#attr-hreflang"
		},
		{
			name: "media",
			doc: "This attribute specifies the media which the linked resource applies to. Its value must be a media query. This attribute is mainly useful when linking to external stylesheets by allowing the user agent to pick the best adapted one for the device it runs on.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/a#attr-media"
		},
		{
			name: "ping",
			doc: "The 'ping' attribute, if present, sends the URLs of the resources a notification/ping if the user follows the hyperlink.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/a#attr-ping"
		},
		{
			name: "rel",
			doc: "For anchors containing the href attribute, this attribute specifies the relationship of the target object to the link object. The value is a comma-separated list of link types values. The values and their semantics will be registered by some authority that might have meaning to the document author. The default relationship, if no other is given, is void. Use this attribute only if the href attribute is present.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/a#attr-rel"
		},
		{
			name: "shape",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/a#attr-shape"
		},
		{
			name: "target",
			doc: "This attribute specifies where to display the linked resource. In HTML4, this is the name of, or a keyword for, a frame. In HTML5, it is a name of, or keyword for, a browsing context (for example, tab, window, or inline frame). The following keywords have special meanings:",
			values: [
				{name: "_self", doc: "Load the response into the same HTML4 frame (or HTML5 browsing context) as the current one. This value is the default if the attribute is not specified"},
				{name: "_blank", doc: "Load the response into a new unnamed HTML4 window or HTML5 browsing context"},
				{name: "_parent", doc: "Load the response into the HTML4 frameset parent of the current frame or HTML5 parent browsing context of the current one. If there is no parent, this option behaves the same way as _self"},
				{name: "_top", doc: "In HTML4: Load the response into the full, original window, canceling all other frames. In HTML5: Load the response into the top-level browsing context (that is, the browsing context that is an ancestor of the current one, and has no parent). If there is no parent, this option behaves the same way as _self"}
			],
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/a#attr-target"
		},
		{
			name: "type",
			doc: "This attribute specifies the media type in the form of a MIME type for the link target. Generally, this is provided strictly as advisory information; however, in the future a browser might add a small icon for multimedia types. For example, a browser might add a small speaker icon when type is set to audio/wav. For a complete list of recognized MIME types, see http://www.w3.org/TR/html4/references.html#ref-MIMETYPES. Use this attribute only if the href attribute is present.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/a#attr-type"
		}
		// Obsolete attributes
//		{
//			name: "charset",
//			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/a#Obsolete",
//			doc: "This attribute defines the character encoding of the linked resource. The value is a space- and/or comma-delimited list of character sets as defined in RFC 2045. The default value is ISO-8859-1.",
//		},
//		{
//			name: "coords",
//			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/a#Obsolete",
//			doc: "For use with object shapes, this attribute uses a comma-separated list of numbers to define the coordinates of the object on the page.",
//		},
//		{
//			name: "name",
//			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/a#Obsolete",
//			doc: "This attribute is required in an anchor defining a target location within a page. A value for name is similar to a value for the id core attribute and should be an alphanumeric identifier unique to the document. Under the HTML 4.01 specification, id and name both can be used with the <a> element as long as they have identical values.",
//		},
//		{
//			name: "rev",
//			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/a#Obsolete",
//			doc: "This attribute specifies a reverse link, the inverse relationship of the rel attribute. It is useful for indicating where an object came from, such as the author of a document.",
//		},
//		{
//			name: "shape",
//			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/a#Obsolete",
//			doc: "This attribute is used to define a selectable region for hypertext source links associated with a figure to create an image map. The values for the attribute are circle, default, polygon, and rect. The format of the coords attribute depends on the value of shape. For circle, the value is x,y,r where x and y are the pixel coordinates for the center of the circle and r is the radius value in pixels. For rect, the coords attribute should be x,y,w,h. The x,y values define the upper-left-hand corner of the rectangle, while w and h define the width and height respectively. A value of polygon for shape requires x1,y1,x2,y2,... values for coords. Each of the x,y pairs defines a point in the polygon, with successive points being joined by straight lines and the last point joined to the first. The value default for shape requires that the entire enclosed area, typically an image, be used.",
//		}
	];

	attributes.tags.applet = [
		{
			name: "align",
			doc: "Specifies the horizontal alignment of the element.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/applet#attr-align"
		},
		{
			name: "code",
			doc: "Specifies the URL of the applet's class file to be loaded and executed.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/applet#attr-code"
		},
		{
			name: "codebase",
			doc: "This attribute gives the absolute or relative URL of the directory where applets' .class files referenced by the code attribute are stored.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/applet#attr-codebase"
		}
	];

	attributes.tags.area = [
		{
			name: "coords",
			doc: "A set of values specifying the coordinates of the hot-spot region.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/area#attr-coords"
		},
		{
			name: "download",
			doc: "Indicates that the hyperlink is to be used for downloading a resource.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/area#attr-download"
		},
		{
			name: "href",
			doc: "The URL of a linked resource.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/area#attr-href"
		},
		{
			name: "hreflang",
			doc: "Specifies the language of the linked resource.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/area#attr-hreflang"
		},
		{
			name: "media",
			doc: "Specifies a hint of the media for which the linked resource was designed.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/area#attr-media"
		},
		{
			name: "ping",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/area#attr-ping"
		},
		{
			name: "rel",
			doc: "Specifies the relationship of the target object to the link object.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/area#attr-rel"
		},
		{
			name: "shape",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/area#attr-shape"
		},
		{
			name: "target",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/area#attr-target"
		}
	];

	attributes.tags.audio = [
		{
			name: "autoplay",
			doc: "The audio or video should play as soon as possible.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/audio#attr-autoplay"
		},
		{
			name: "buffered",
			doc: "Contains the time range of already buffered media.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/audio#attr-buffered"
		},
		{
			name: "controls",
			doc: "Indicates whether the browser should show playback controls to the user.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/audio#attr-controls"
		},
		{
			name: "loop",
			doc: "Indicates whether the media should start playing from the start when it's finished.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/audio#attr-loop"
		},
		{
			name: "preload",
			doc: "Indicates whether the whole resource, parts of it or nothing should be preloaded.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/audio#attr-preload"
		},
		{
			name: "src",
			doc: "The URL of the embeddable content.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/audio#attr-src"
		}
	];

	attributes.tags.base = [
		{
			name: "href",
			doc: "The URL of a linked resource.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/base#attr-href"
		},
		{
			name: "target",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/base#attr-target"
		}
	];

	attributes.tags.basefont = [
		{
			name: "color",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/basefont#attr-color"
		}
	];

	attributes.tags.bgsound = [
		{
			name: "loop",
			doc: "Indicates whether the media should start playing from the start when it's finished.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/bgsound#attr-loop"
		}
	];

	attributes.tags.blockquote = [
		{
			name: "cite",
			doc: "Contains a URI which points to the source of the quote or change.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/blockquote#attr-cite"
		}
	];

	attributes.tags.body = [
		{
			name: "bgcolor",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/body#attr-bgcolor"
		}
		// Obsolete attributes
//		{
//			name: "alink",
//			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/body#Attributes",
//			doc: "Color of text for hyperlinks when selected. This method is non-conforming, use CSS color property in conjunction with the :active pseudo-class instead.",
//		},
//		{
//			name: "background",
//			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/body#Attributes",
//			doc: "URI of a image to use as a background. This method is non-conforming, use CSS background property on the element instead.",
//		},
//		{
//			name: "bgcolor",
//			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/body#Attributes",
//			doc: "Background color for the document. This method is non-conforming, use CSS background-color property on the element instead.",
//		},
//		{
//			name: "bottommargin",
//			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/body#Attributes",
//			doc: "The margin of the bottom of the body. This method is non-conforming, use CSS margin-bottom property on the element instead.",
//		},
//		{
//			name: "leftmargin",
//			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/body#Attributes",
//			doc: "The margin of the left of the body. This method is non-conforming, use CSS margin-left property on the element instead.",
//		},
//		{
//			name: "link",
//			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/body#Attributes",
//			doc: "Color of text for unvisited hypertext links. This method is non-conforming, use CSS color property in conjunction with the :link pseudo-class instead.",
//		},
//		{
//			name: "rightmargin",
//			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/body#Attributes",
//			doc: "The margin of the right of the body. This method is non-conforming, use CSS margin-right property on the element instead.",
//		},
//		{
//			name: "text",
//			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/body#Attributes",
//			doc: "Foreground color of text. This method is non-conforming, use CSS color property on the element instead.",
//		},
//		{
//			name: "topmargin",
//			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/body#Attributes",
//			doc: "The margin of the top of the body. This method is non-conforming, use CSS margin-top property on the element instead.",
//		},
//		{
//			name: "vlink",
//			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/body#Attributes",
//			doc: "Color of text for visited hypertext links. This method is non-conforming, use CSS color property in conjunction with the :visited pseudo-class instead.",
//		}
	];
	
	// Obsolete tags
//	attributes.tags.br = [
//		{
//			name: "clear",
//			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/br#Attributes",
//			doc: "Indicates where to begin the next line after the break.",
//		}
//	];

	attributes.tags.button = [
		{
			name: "autofocus",
			doc: "The element should be automatically focused after the page loaded.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/button#attr-autofocus"
		},
		{
			name: "disabled",
			doc: "Indicates whether the user can interact with the element.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/button#attr-disabled"
		},
		{
			name: "form",
			doc: "Indicates the form that is the owner of the element.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/button#attr-form"
		},
		{
			name: "formaction",
			doc: "Indicates the action of the element, overriding the action defined in the <form>.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/button#attr-formaction"
		},
		{
			name: "name",
			doc: "Name of the element. For example used by the server to identify the fields in form submits.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/button#attr-name"
		},
		{
			name: "type",
			doc: "Defines the type of the element.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/button#attr-type"
		},
		{
			name: "value",
			doc: "Defines a default value which will be displayed in the element on page load.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/button#attr-value"
		}
	];

	attributes.tags.canvas = [
		{
			name: "height",
			doc: "Note: In some instances, such as <div>, this is a legacy attribute, in which case the CSS height property should be used instead. In other cases, such as <canvas>, the height must be specified with this attribute.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/canvas#attr-height"
		},
		{
			name: "width",
			doc: "Note: In some instances, such as <div>, this is a legacy attribute, in which case the CSS width property should be used instead. In other cases, such as <canvas>, the width must be specified with this attribute.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/canvas#attr-width"
		}
	];

	attributes.tags.caption = [
		{
			name: "align",
			doc: "Specifies the horizontal alignment of the element.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/caption#attr-align"
		}
	];

	attributes.tags.col = [
		{
			name: "align",
			doc: "Specifies the horizontal alignment of the element.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/col#attr-align"
		},
		{
			name: "bgcolor",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/col#attr-bgcolor"
		},
		{
			name: "span",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/col#attr-span"
		}
	];

	attributes.tags.colgroup = [
		{
			name: "align",
			doc: "Specifies the horizontal alignment of the element.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/colgroup#attr-align"
		},
		{
			name: "bgcolor",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/colgroup#attr-bgcolor"
		},
		{
			name: "span",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/colgroup#attr-span"
		}
	];

	attributes.tags.command = [
		{
			name: "checked",
			doc: "Indicates whether the element should be checked on page load.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/command#attr-checked"
		},
		{
			name: "disabled",
			doc: "Indicates whether the user can interact with the element.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/command#attr-disabled"
		},
		{
			name: "icon",
			doc: "Specifies a picture which represents the command.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/command#attr-icon"
		},
		{
			name: "radiogroup",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/command#attr-radiogroup"
		},
		{
			name: "type",
			doc: "Defines the type of the element.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/command#attr-type"
		}
	];

	attributes.tags.del = [
		{
			name: "cite",
			doc: "Contains a URI which points to the source of the quote or change.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/del#attr-cite"
		},
		{
			name: "datetime",
			doc: "Indicates the date and time associated with the element.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/del#attr-datetime"
		}
	];

	attributes.tags.details = [
		{
			name: "open",
			doc: "Indicates whether the details will be shown on page load.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/details#attr-open"
		}
	];

	attributes.tags.embed = [
		{
			name: "height",
			doc: "Note: In some instances, such as <div>, this is a legacy attribute, in which case the CSS height property should be used instead. In other cases, such as <canvas>, the height must be specified with this attribute.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/embed#attr-height"
		},
		{
			name: "src",
			doc: "The URL of the embeddable content.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/embed#attr-src"
		},
		{
			name: "type",
			doc: "Defines the type of the element.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/embed#attr-type"
		},
		{
			name: "width",
			doc: "Note: In some instances, such as <div>, this is a legacy attribute, in which case the CSS width property should be used instead. In other cases, such as <canvas>, the width must be specified with this attribute.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/embed#attr-width"
		}
	];

	attributes.tags.fieldset = [
		{
			name: "disabled",
			doc: "Indicates whether the user can interact with the element.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/fieldset#attr-disabled"
		},
		{
			name: "form",
			doc: "Indicates the form that is the owner of the element.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/fieldset#attr-form"
		},
		{
			name: "name",
			doc: "Name of the element. For example used by the server to identify the fields in form submits.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/fieldset#attr-name"
		}
	];

	attributes.tags.font = [
		{
			name: "color",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/font#attr-color"
		}
	];

	attributes.tags.form = [
		{
			name: "accept",
			doc: "List of types the server accepts, typically a file type.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/form#attr-accept"
		},
		{
			name: "accept-charset",
			doc: "List of supported charsets.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/form#attr-accept-charset"
		},
		{
			name: "action",
			doc: "The URI of a program that processes the information submitted via the form.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/form#attr-action"
		},
		{
			name: "autocomplete",
			doc: "Indicates whether controls in this form can by default have their values automatically completed by the browser.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/form#attr-autocomplete"
		},
		{
			name: "enctype",
			doc: "Defines the content type of the form date when the method is POST.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/form#attr-enctype"
		},
		{
			name: "method",
			doc: "Defines which HTTP method to use when submitting the form. Can be GET (default) or POST.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/form#attr-method"
		},
		{
			name: "name",
			doc: "Name of the element. For example used by the server to identify the fields in form submits.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/form#attr-name"
		},
		{
			name: "novalidate",
			doc: "This attribute indicates that the form shouldn't be validated when submitted.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/form#attr-novalidate"
		},
		{
			name: "target",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/form#attr-target"
		}
	];

	attributes.tags.hr = [
		// Deprecated attributes
//		{
//			name: "align",
//			doc: "Specifies the horizontal alignment of the element.",
//			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/hr#attr-align"
//		},
//		{
//			name: "color",
//			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/hr#attr-color"
//		},
//		{
//			name: "noshade",
//			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/hr#Attributes",
//			doc: "Sets the rule to have no shading.",
//		},
//		{
//			name: "size",
//			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/hr#Attributes",
//			doc: "Sets the height, in pixels, of the rule.",
//		},
//		{
//			name: "width",
//			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/hr#Attributes",
//			doc: "Sets the length of the rule on the page through a pixel or percentage value.",
//		}
	];

	attributes.tags.html = [
		{
			name: "manifest",
			doc: "Specifies the URI of a resource manifest indicating resources that should be cached locally. See Using the application cache for details.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/html#attr-manifest"
		},
		// Deprecated attributes
//		{
//			name: "version",
//			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/html#Attributes",
//			doc: "Specifies the version of the HTML Document Type Definition that governs the current document. This attribute is not needed, because it is redundant with the version information in the document type declaration.",
//		},
		{
			name: "xmlns",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/html#attr-xmlns",
			doc: "Specifies the XML Namespace of the document. Default value is http://www.w3.org/1999/xhtml. This is required in XHTML, and optional in HTML5."
		}
	];

	attributes.tags.iframe = [
		{
			name: "align",
			doc: "The alignment of this element with respect to the surrounding context.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/iframe#attr-align"
		},
		{
			name: "height",
			doc: "Indicates the height of the frame HTML5 in CSS pixels, or HTML 4.01 in pixels or as a percentage.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/iframe#attr-height"
		},
		{
			name: "name",
			doc: "A name for the embedded browsing context (or frame). This can be used as the value of the target attribute of an <a> or <form> element, or the formtarget attribute of an <input> or <button> element.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/iframe#attr-name"
		},
		{
			name: "sandbox",
			doc: "If specified as an empty string, this attribute enables extra restrictions on the content that can appear in the inline frame. The value of the attribute can either be an empty string (all the restrictions are applied), or a space-separated list of tokens that lift particular restrictions.",
			values: [
				{name: "allow-same-origin", doc: "Allows the content to be treated as being from its normal origin. If this keyword is not used, the embedded content is treated as being from a unique origin"},
				{name: "allow-top-navigation", doc: "Allows the embedded browsing context to navigate (load) content to the top-level browsing context. If this keyword is not used, this operation is not allowed"},
				{name: "allow-forms", doc: "Allows the embedded browsing context to submit forms. If this keyword is not used, this operation is not allowed"},
				{name: "allow-popups", doc: "Allows popups (like from window.open, target=\"_blank\", showModalDialog). If this keyword is not used, that functionality will silently fail"},
				{name: "allow-scripts", doc: "Allows the embedded browsing context to run scripts (but not create pop-up windows). If this keyword is not used, this operation is not allowed"},
				{name: "allow-pointer-lock", doc: "Allows the embedded browsing context to use the Pointer Lock API"},
				{name: "allow-unsandboxed-auxiliary", doc: "(Chrome only) Allows a sandboxed document to open new windows without forcing the sandboxing flags upon them. This will allow, for example, a third-party advertisement to be safely sandboxed without forcing the same restrictions upon a landing page"}
			],
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/iframe#attr-sandbox"
		},
		{
			name: "seamless",
			doc: "This Boolean attribute indicates that the browser should render the inline frame in a way that makes it appear to be part of the containing document, for example by applying CSS styles that apply to the <iframe> to the contained document before styles specified in that document, and by opening links in the contained documents in the parent browsing context (unless another setting prevents this). In XHTML, attribute minimization is forbidden, and the seamless attribute must be defined as <iframe seamless=\"seamless\">.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/iframe#attr-seamless"
		},
		{
			name: "src",
			doc: "The URL of the embeddable content.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/iframe#attr-src"
		},
		{
			name: "srcdoc",
			doc:  "The content of the page that the embedded context is to contain. This attribute is expected to be used together with the sandbox and seamless attributes. If a browser supports the srcdoc attribute, it will override the content specified in the src attribute (if present). If a browser does NOT support the srcdoc attribute, it will show the file specified in the src attribute instead (if present).",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/iframe#attr-srcdoc"
		},
		{
			name: "width",
			doc: "Indicates the width of the frame HTML5 in CSS pixels, or HTML 4.01 in pixels or as a percentage.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/iframe#attr-width"
		}
	];

	attributes.tags.img = [
		{
			name: "align",
			doc: "The alignment of the image with respect to its surrounding context.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/img#attr-align"
		},
		{
			name: "alt",
			doc: "This attribute defines the alternative text describing the image. Users will see this displayed if the image URL is wrong, the image is not in one of the supported formats, or if the image is not yet downloaded.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/img#attr-alt"
		},
		{
			name: "border",
			doc: "The width of a border around the image.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/img#attr-border"
		},
		{
			name: "height",
			doc: "The intrinsic height of the image in HTML5 CSS pixels, or HTML 4 in pixels or as a percentage.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/img#attr-height"
		},
		{
			name: "ismap",
			doc: "This Boolean attribute indicates that the image is part of a server-side map. If so, the precise coordinates of a click are sent to the server.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/img#attr-ismap"
		},
		{
			name: "src",
			doc: "The image URL. This attribute is mandatory for the <img> element. On browsers supporting srcset, src is treated like a candidate image with a pixel density descriptor 1x unless an image with this pixel density descriptor is already defined in srcset or srcset contains 'w' descriptors.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/img#attr-src"
		},
		{
			name: "srcset",
			doc: "A list of one or more strings separated by commas indicating a set of possible image sources for the user agent to use. Each string is composed of:"
					+ "\n* a URL to an image"
					+ "\n* optionally, whitespace followed by one of:"
					+ "\n * a width descriptor, that is a positive integer directly followed by 'w'. The width descriptor is divided by the source size given in the sizes attribute to calculate the effective pixel density"
					+ "\n * a pixel density descriptor, that is a positive floating point number directly followed by 'x'"
					+ "\nIf no descriptor is specified, the source is assigned the default descriptor: 1x.\n"
					+ "\nIt is invalid to mix width descriptors and pixel density descriptors in the same srcset attribute. Duplicate descriptors (for instance, two sources in the same srcset which are both described with '2x') are invalid, too."
					+ "\nUser agents are given discretion to choose any one of the available sources. This provides them with significant leeway to tailor their selection based on things like user preferences or bandwidth conditions.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/img#attr-srcset"
		},
		{
			name: "usemap",
			doc: "The partial URL (starting with '#') of an image map associated with the element.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/img#attr-usemap"
		},
		{
			name: "width",
			doc: "The intrinsic width of the image in HTML5 CSS pixels, or HTML 4 in pixels or as a percentage.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/img#attr-width"
		}
	];

	attributes.tags.input = [
		{
			name: "accept",
			doc: "List of types the server accepts, typically a file type.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/input#attr-accept"
		},
		{
			name: "autocomplete",
			doc: "Indicates whether controls in this form can by default have their values automatically completed by the browser.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/input#attr-autocomplete"
		},
		{
			name: "autofocus",
			doc: "The element should be automatically focused after the page loaded.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/input#attr-autofocus"
		},
		{
			name: "autosave",
			doc: "Previous values should persist dropdowns of selectable values across page loads.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/input#attr-autosave"
		},
		{
			name: "checked",
			doc: "Indicates whether the element should be checked on page load.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/input#attr-checked"
		},
		{
			name: "dirname",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/input#attr-dirname"
		},
		{
			name: "disabled",
			doc: "Indicates whether the user can interact with the element.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/input#attr-disabled"
		},
		{
			name: "form",
			doc: "Indicates the form that is the owner of the element.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/input#attr-form"
		},
		{
			name: "formaction",
			doc: "Indicates the action of the element, overriding the action defined in the <form>.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/input#attr-formaction"
		},
		{
			name: "height",
			doc: "Note: In some instances, such as <div>, this is a legacy attribute, in which case the CSS height property should be used instead. In other cases, such as <canvas>, the height must be specified with this attribute.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/input#attr-height"
		},
		{
			name: "list",
			doc: "Identifies a list of pre-defined options to suggest to the user.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/input#attr-list"
		},
		{
			name: "max",
			doc: "Indicates the maximum value allowed.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/input#attr-max"
		},
		{
			name: "maxlength",
			doc: "Defines the maximum number of characters allowed in the element.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/input#attr-maxlength"
		},
		{
			name: "min",
			doc: "Indicates the minimum value allowed.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/input#attr-min"
		},
		{
			name: "multiple",
			doc: "Indicates whether multiple values can be entered in an input of the type email or file.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/input#attr-multiple"
		},
		{
			name: "name",
			doc: "Name of the element. For example used by the server to identify the fields in form submits.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/input#attr-name"
		},
		{
			name: "pattern",
			doc: "Defines a regular expression which the element's value will be validated against.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/input#attr-pattern"
		},
		{
			name: "placeholder",
			doc: "Provides a hint to the user of what can be entered in the field.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/input#attr-placeholder"
		},
		{
			name: "readonly",
			doc: "Indicates whether the element can be edited.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/input#attr-readonly"
		},
		{
			name: "required",
			doc: "Indicates whether this element is required to fill out or not.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/input#attr-required"
		},
		{
			name: "size",
			doc: "Defines the width of the element (in pixels). If the element's type attribute is text or password then it's the number of characters.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/input#attr-size"
		},
		{
			name: "src",
			doc: "The URL of the embeddable content.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/input#attr-src"
		},
		{
			name: "step",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/input#attr-step"
		},
		{
			name: "type",
			doc: "The type of control to display. The default type is text, if this attribute is not specified.",
			values: [
				{name: "button", doc: "A push button with no default behavior."},
				{name: "checkbox", doc: "A check box. You must use the value attribute to define the value submitted by this item. Use the checked attribute to indicate whether this item is selected. You can also use the indeterminate attribute to indicate that the checkbox is in an indeterminate state (on most platforms, this draws a horizontal line across the checkbox)."},
				{name: "color", doc: "A control for specifying a color. A color picker's UI has no required features other than accepting simple colors as text (more info)."},
				{name: "date", doc: "A control for entering a date (year, month, and day, with no time)."},
				{name: "datetime-local", doc: "A control for entering a date and time, with no time zone."},
				{name: "email", doc: "A field for editing an e-mail address. The input value is validated to contain either the empty string or a single valid e-mail address before submitting. The :valid and :invalid CSS pseudo-classes are applied as appropriate."},
				{name: "file", doc: "A control that lets the user select a file. Use the accept attribute to define the types of files that the control can select."},
				{name: "hidden", doc: "A control that is not displayed, but whose value is submitted to the server."},
				{name: "image", doc: "A graphical submit button. You must use the src attribute to define the source of the image and the alt attribute to define alternative text. You can use the height and width attributes to define the size of the image in pixels."},
				{name: "month", doc: "A control for entering a month and year, with no time zone."},
				{name: "number", doc: "A control for entering a floating point number."},
				{name: "password", doc: "A single-line text field whose value is obscured. Use the maxlength attribute to specify the maximum length of the value that can be entered."},
				{name: "radio", doc: "A radio button. You must use the value attribute to define the value submitted by this item. Use the checked attribute to indicate whether this item is selected by default. Radio buttons that have the same value for the name attribute are in the same \"radio button group\"; only one radio button in a group can be selected at a time."},
				{name: "range", doc: "A control for entering a number whose exact value is not important. This type control uses the following default values if the corresponding attributes are not specified:"
					+ "\n * min: 0"
					+ "\n * max: 100"
					+ "\n * value: min + (max-min)/2, or min if max is less than min"
					+ "\n * step: 1"},
				{name: "reset", doc: "A button that resets the contents of the form to default values."},
				{name: "search", doc: "A single-line text field for entering search strings; line-breaks are automatically removed from the input value."},
				{name: "submit", doc: "A button that submits the form."},
				{name: "tel", doc: "A control for entering a telephone number; line-breaks are automatically removed from the input value, but no other syntax is enforced. You can use attributes such as pattern and maxlength to restrict values entered in the control. The :valid and :invalid CSS pseudo-classes are applied as appropriate."},
				{name: "text", doc: "A single-line text field; line-breaks are automatically removed from the input value."},
				{name: "time", doc: "A control for entering a time value with no time zone."},
				{name: "url", doc: "A field for editing a URL. The input value is validated to contain either the empty string or a valid absolute URL before submitting. Line-breaks and leading or trailing whitespace are automatically removed from the input value. You can use attributes such as pattern and maxlength to restrict values entered in the control. The :valid and :invalid CSS pseudo-classes are applied as appropriate."},
				{name: "week", doc: "A control for entering a date consisting of a week-year number and a week number with no time zone."}
			],
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/input#attr-type"
		},
		{
			name: "usemap",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/input#attr-usemap"
		},
		{
			name: "value",
			doc: "Defines a default value which will be displayed in the element on page load.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/input#attr-value"
		},
		{
			name: "width",
			doc: "Note: In some instances, such as <div>, this is a legacy attribute, in which case the CSS width property should be used instead. In other cases, such as <canvas>, the width must be specified with this attribute.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/input#attr-width"
		}
	];

	attributes.tags.ins = [
		{
			name: "cite",
			doc: "Contains a URI which points to the source of the quote or change.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/ins#attr-cite"
		},
		{
			name: "datetime",
			doc: "Indicates the date and time associated with the element.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/ins#attr-datetime"
		}
	];

	attributes.tags.keygen = [
		{
			name: "autofocus",
			doc: "The element should be automatically focused after the page loaded.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/keygen#attr-autofocus"
		},
		{
			name: "challenge",
			doc: "A challenge string that is submitted along with the public key.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/keygen#attr-challenge"
		},
		{
			name: "disabled",
			doc: "Indicates whether the user can interact with the element.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/keygen#attr-disabled"
		},
		{
			name: "form",
			doc: "Indicates the form that is the owner of the element.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/keygen#attr-form"
		},
		{
			name: "keytype",
			doc: "Specifies the type of key generated.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/keygen#attr-keytype"
		},
		{
			name: "name",
			doc: "Name of the element. For example used by the server to identify the fields in form submits.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/keygen#attr-name"
		}
	];

	attributes.tags.label = [
		{
			name: "for",
			doc: "Describes elements which belongs to this one.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/label#attr-for"
		},
		{
			name: "form",
			doc: "Indicates the form that is the owner of the element.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/label#attr-form"
		}
	];

	attributes.tags.li = [
		{
			name: "value",
			doc: "Defines a default value which will be displayed in the element on page load.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/li#attr-value"
		}
	];

	attributes.tags.link = [
		{
			name: "href",
			doc: "The URL of a linked resource.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/link#attr-href"
		},
		{
			name: "hreflang",
			doc: "Specifies the language of the linked resource.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/link#attr-hreflang"
		},
		{
			name: "media",
			doc: "Specifies a hint of the media for which the linked resource was designed.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/link#attr-media"
		},
		{
			name: "rel",
			doc: "Specifies the relationship of the target object to the link object.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/link#attr-rel"
		},
		{
			name: "sizes",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/link#attr-sizes"
		}
	];

	attributes.tags.map = [
		{
			name: "name",
			doc: "Name of the element. For example used by the server to identify the fields in form submits.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/map#attr-name"
		}
	];

	attributes.tags.marquee = [
		{
			name: "bgcolor",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/marquee#attr-bgcolor"
		},
		{
			name: "loop",
			doc: "Indicates whether the media should start playing from the start when it's finished.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/marquee#attr-loop"
		}
	];

	attributes.tags.menu = [
		{
			name: "type",
			doc: "Defines the type of the element.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/menu#attr-type"
		}
	];

	attributes.tags.meta = [
		{
			name: "charset",
			doc: "Declares the character encoding of the page or script.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/meta#attr-charset"
		},
		{
			name: "content",
			doc: "A value associated with http-equiv or name depending on the context.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/meta#attr-content"
		},
		{
			name: "http-equiv",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/meta#attr-http-equiv"
		},
		{
			name: "name",
			doc: "Name of the element. For example used by the server to identify the fields in form submits.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/meta#attr-name"
		}
	];

	attributes.tags.meter = [
		{
			name: "form",
			doc: "Indicates the form that is the owner of the element.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/meter#attr-form"
		},
		{
			name: "high",
			doc: "Indicates the lower bound of the upper range.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/meter#attr-high"
		},
		{
			name: "low",
			doc: "Indicates the upper bound of the lower range.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/meter#attr-low"
		},
		{
			name: "max",
			doc: "Indicates the maximum value allowed.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/meter#attr-max"
		},
		{
			name: "min",
			doc: "Indicates the minimum value allowed.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/meter#attr-min"
		},
		{
			name: "optimum",
			doc: "Indicates the optimal numeric value.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/meter#attr-optimum"
		},
		{
			name: "value",
			doc: "Defines a default value which will be displayed in the element on page load.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/meter#attr-value"
		}
	];

	attributes.tags.object = [
		{
			name: "border",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/object#attr-border"
		},
		{
			name: "form",
			doc: "Indicates the form that is the owner of the element.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/object#attr-form"
		},
		{
			name: "height",
			doc: "Note: In some instances, such as <div>, this is a legacy attribute, in which case the CSS height property should be used instead. In other cases, such as <canvas>, the height must be specified with this attribute.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/object#attr-height"
		},
		{
			name: "name",
			doc: "Name of the element. For example used by the server to identify the fields in form submits.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/object#attr-name"
		},
		{
			name: "type",
			doc: "Defines the type of the element.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/object#attr-type"
		},
		{
			name: "usemap",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/object#attr-usemap"
		},
		{
			name: "width",
			doc: "Note: In some instances, such as <div>, this is a legacy attribute, in which case the CSS width property should be used instead. In other cases, such as <canvas>, the width must be specified with this attribute.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/object#attr-width"
		}
	];

	attributes.tags.ol = [
		{
			name: "reversed",
			doc: "Indicates whether the list should be displayed in a descending order instead of a ascending.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/ol#attr-reversed"
		},
		{
			name: "start",
			doc: "Defines the first number if other than 1.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/ol#attr-start"
		}
	];

	attributes.tags.optgroup = [
		{
			name: "disabled",
			doc: "Indicates whether the user can interact with the element.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/optgroup#attr-disabled"
		}
	];

	attributes.tags.option = [
		{
			name: "disabled",
			doc: "Indicates whether the user can interact with the element.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/option#attr-disabled"
		},
		{
			name: "selected",
			doc: "Defines a value which will be selected on page load.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/option#attr-selected"
		},
		{
			name: "value",
			doc: "Defines a default value which will be displayed in the element on page load.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/option#attr-value"
		}
	];

	attributes.tags.output = [
		{
			name: "for",
			doc: "Describes elements which belongs to this one.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/output#attr-for"
		},
		{
			name: "form",
			doc: "Indicates the form that is the owner of the element.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/output#attr-form"
		},
		{
			name: "name",
			doc: "Name of the element. For example used by the server to identify the fields in form submits.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/output#attr-name"
		}
	];

	attributes.tags.param = [
		{
			name: "name",
			doc: "Name of the element. For example used by the server to identify the fields in form submits.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/param#attr-name"
		},
		{
			name: "value",
			doc: "Defines a default value which will be displayed in the element on page load.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/param#attr-value"
		}
	];

	attributes.tags.progress = [
		{
			name: "form",
			doc: "Indicates the form that is the owner of the element.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/progress#attr-form"
		},
		{
			name: "max",
			doc: "Indicates the maximum value allowed.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/progress#attr-max"
		},
		{
			name: "value",
			doc: "Defines a default value which will be displayed in the element on page load.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/progress#attr-value"
		}
	];

	attributes.tags.q = [
		{
			name: "cite",
			doc: "Contains a URI which points to the source of the quote or change.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/q#attr-cite"
		}
	];

	attributes.tags.script = [
		{
			name: "async",
			doc: "Indicates that the script should be executed asynchronously.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/script#attr-async"
		},
		{
			name: "charset",
			doc: "Declares the character encoding of the page or script.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/script#attr-charset"
		},
		{
			name: "defer",
			doc: "Indicates that the script should be executed after the page has been parsed.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/script#attr-defer"
		},
		{
			name: "language",
			doc: "Defines the script language used in the element.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/script#attr-language"
		},
		{
			name: "src",
			doc: "The URL of the embeddable content.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/script#attr-src"
		},
		{
			name: "type",
			doc: "Defines the type of the element.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/script#attr-type"
		}
	];

	attributes.tags.select = [
		{
			name: "autofocus",
			doc: "The element should be automatically focused after the page loaded.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/select#attr-autofocus"
		},
		{
			name: "disabled",
			doc: "Indicates whether the user can interact with the element.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/select#attr-disabled"
		},
		{
			name: "form",
			doc: "Indicates the form that is the owner of the element.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/select#attr-form"
		},
		{
			name: "multiple",
			doc: "Indicates whether multiple values can be entered in an input of the type email or file.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/select#attr-multiple"
		},
		{
			name: "name",
			doc: "Name of the element. For example used by the server to identify the fields in form submits.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/select#attr-name"
		},
		{
			name: "required",
			doc: "Indicates whether this element is required to fill out or not.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/select#attr-required"
		},
		{
			name: "size",
			doc: "Defines the width of the element (in pixels). If the element's type attribute is text or password then it's the number of characters.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/select#attr-size"
		}
	];

	attributes.tags.source = [
		{
			name: "media",
			doc: "Specifies a hint of the media for which the linked resource was designed.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/source#attr-media"
		},
		{
			name: "src",
			doc: "The URL of the embeddable content.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/source#attr-src"
		},
		{
			name: "type",
			doc: "Defines the type of the element.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/source#attr-type"
		}
	];

	attributes.tags.style = [
		{
			name: "media",
			doc: "Specifies a hint of the media for which the linked resource was designed.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/style#attr-media"
		},
		{
			name: "scoped",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/style#attr-scoped"
		},
		{
			name: "type",
			doc: "Defines the type of the element.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/style#attr-type"
		}
	];

	attributes.tags.table = [
		{
			name: "align",
			doc: "Specifies the horizontal alignment of the element.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/table#attr-align"
		},
		{
			name: "bgcolor",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/table#attr-bgcolor"
		},
		{
			name: "border",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/table#attr-border"
		},
		{
			name: "summary",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/table#attr-summary"
		}
	];

	attributes.tags.tbody = [
		{
			name: "align",
			doc: "Specifies the horizontal alignment of the element.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/tbody#attr-align"
		},
		{
			name: "bgcolor",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/tbody#attr-bgcolor"
		}
	];

	attributes.tags.td = [
		{
			name: "align",
			doc: "Specifies the horizontal alignment of the element.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/td#attr-align"
		},
		{
			name: "bgcolor",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/td#attr-bgcolor"
		},
		{
			name: "colspan",
			doc: "The colspan attribute defines the number of columns a cell should span.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/td#attr-colspan"
		},
		{
			name: "headers",
			doc: "IDs of the <th> elements which applies to this element.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/td#attr-headers"
		},
		{
			name: "rowspan",
			doc: "Defines the number of rows a table cell should span over.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/td#attr-rowspan"
		}
	];

	attributes.tags.textarea = [
		{
			name: "autofocus",
			doc: "The element should be automatically focused after the page loaded.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/textarea#attr-autofocus"
		},
		{
			name: "cols",
			doc: "Defines the number of columns in a textarea.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/textarea#attr-cols"
		},
		{
			name: "dirname",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/textarea#attr-dirname"
		},
		{
			name: "disabled",
			doc: "Indicates whether the user can interact with the element.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/textarea#attr-disabled"
		},
		{
			name: "form",
			doc: "Indicates the form that is the owner of the element.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/textarea#attr-form"
		},
		{
			name: "maxlength",
			doc: "Defines the maximum number of characters allowed in the element.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/textarea#attr-maxlength"
		},
		{
			name: "name",
			doc: "Name of the element. For example used by the server to identify the fields in form submits.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/textarea#attr-name"
		},
		{
			name: "placeholder",
			doc: "Provides a hint to the user of what can be entered in the field.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/textarea#attr-placeholder"
		},
		{
			name: "readonly",
			doc: "Indicates whether the element can be edited.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/textarea#attr-readonly"
		},
		{
			name: "required",
			doc: "Indicates whether this element is required to fill out or not.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/textarea#attr-required"
		},
		{
			name: "rows",
			doc: "Defines the number of rows in a textarea.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/textarea#attr-rows"
		},
		{
			name: "wrap",
			doc: "Indicates whether the text should be wrapped.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/textarea#attr-wrap"
		}
	];

	attributes.tags.tfoot = [
		{
			name: "align",
			doc: "Specifies the horizontal alignment of the element.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/tfoot#attr-align"
		},
		{
			name: "bgcolor",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/tfoot#attr-bgcolor"
		}
	];

	attributes.tags.th = [
		{
			name: "align",
			doc: "Specifies the horizontal alignment of the element.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/th#attr-align"
		},
		{
			name: "bgcolor",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/th#attr-bgcolor"
		},
		{
			name: "colspan",
			doc: "The colspan attribute defines the number of columns a cell should span.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/th#attr-colspan"
		},
		{
			name: "headers",
			doc: "IDs of the <th> elements which applies to this element.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/th#attr-headers"
		},
		{
			name: "rowspan",
			doc: "Defines the number of rows a table cell should span over.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/th#attr-rowspan"
		},
		{
			name: "scope",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/th#attr-scope"
		}
	];

	attributes.tags.thead = [
		{
			name: "align",
			doc: "Specifies the horizontal alignment of the element.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/thead#attr-align"
		}
	];

	attributes.tags.time = [
		{
			name: "datetime",
			doc: "Indicates the date and time associated with the element.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/time#attr-datetime"
		},
		{
			name: "pubdate",
			doc: "Indicates whether this date and time is the date of the nearest <article> ancestor element.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/time#attr-pubdate"
		}
	];

	attributes.tags.tr = [
		{
			name: "align",
			doc: "Specifies the horizontal alignment of the element.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/tr#attr-align"
		},
		{
			name: "bgcolor",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/tr#attr-bgcolor"
		}
	];

	attributes.tags.track = [
		{
			name: "default",
			doc: "Indicates that the track should be enabled unless the user's preferences indicate something different.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/track#attr-default"
		},
		{
			name: "kind",
			doc: "Specifies the kind of text track.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/track#attr-kind"
		},
		{
			name: "label",
			doc: "Specifies a user-readable title of the text track.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/track#attr-label"
		},
		{
			name: "src",
			doc: "The URL of the embeddable content.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/track#attr-src"
		},
		{
			name: "srclang",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/track#attr-srclang"
		}
	];

	attributes.tags.video = [
		{
			name: "autoplay",
			doc: "A Boolean attribute; if specified, the video will automatically begin to play back as soon as it can do so without stopping to finish loading the data.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/video#attr-autoplay"
		},
		{
			name: "buffered",
			doc: "An attribute you can read to determine which time ranges of the media have been buffered. This attribute contains a TimeRanges object.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/video#attr-buffered"
		},
		{
			name: "controls",
			doc: "If this attribute is present, Gecko will offer controls to allow the user to control video playback, including volume, seeking, and pause/resume playback.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/video#attr-controls"
		},
		{
			name: "height",
			doc: "The height of the video's display area, in CSS pixels.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/video#attr-height"
		},
		{
			name: "loop",
			doc: "A Boolean attribute; if specified, we will, upon reaching the end of the video, automatically seek back to the start.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/video#attr-loop"
		},
		{
			name: "poster",
			doc: "A URL indicating a poster frame to show until the user plays or seeks. If this attribute isn't specified, nothing is displayed until the first frame is available; then the first frame is displayed as the poster frame.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/video#attr-poster"
		},
		{
			name: "preload",
			doc: "This enumerated attribute is intended to provide a hint to the browser about what the author thinks will lead to the best user experience.",
			values: [
				{name: "none", doc: "hints that either the author thinks that the user won't need to consult that video or that the server wants to minimize its traffic; in others terms this hint indicates that the video should not be cached"},
				{name: "metadata", doc: "hints that though the author thinks that the user won't need to consult that video, fetching the metadata (e.g. length) is reasonable"},
				{name: "auto", doc: "(or empty string) hints that the user needs have priority; in others terms this hint indicated that, if needed, the whole video could be downloaded, even if the user is not expected to use it"},
			],
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/video#attr-preload"
		},
		{
			name: "src",
			doc: "The URL of the video to embed. This is optional; you may instead use the <source> element within the video block to specify the video to embed.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/video#attr-src"
		},
		{
			name: "width",
			doc: "The width of the video's display area, in CSS pixels.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/video#attr-width"
		}
	];

	var aria = Object.create(null);
	aria.globalAttributes = [
		{
			name: "aria-atomic",
			doc: "Indicates whether assistive technologies will present all, or only parts of, the changed region based on the change notifications defined by the aria-relevant attribute. See related aria-relevant.",
			values: [
				{name: "true", doc: "Assistive technologies will present the entire region as a whole."},
				{name: "false", doc: "A change within the region may be processed by the assistive technologies on its own."}
			],
			url: "http://www.w3.org/TR/wai-aria/complete#aria-atomic"
		},
		{
			name: "aria-busy",
			doc: "Indicates whether an element, and its subtree, are currently being updated.",
			values: [
				{name: "true", doc: "The live region is still being updated."},
				{name: "false", doc: "There are no more expected updates for that live region."}
			],
			url: "http://www.w3.org/TR/wai-aria/complete#aria-busy"
		},
		{
			name: "aria-controls",
			doc: "Identifies the element (or elements) whose contents or presence are controlled by the current element. See related aria-owns.",
			url: "http://www.w3.org/TR/wai-aria/complete#aria-controls"
		},
		{
			name: "aria-describedby",
			doc: "Identifies the element (or elements) that describes the object. See related aria-labelledby.",
			url: "http://www.w3.org/TR/wai-aria/complete#aria-describedby"
		},
		{
			name: "aria-disabled",
			doc: "Indicates that the element is perceivable but disabled, so it is not editable or otherwise operable. See related aria-hidden and aria-readonly.",
			values: [
				{name: "true", doc: "The element and all focusable descendants are disabled and its value cannot be changed by the user."},
				{name: "false", doc: "The element is enabled."}
			],
			url: "http://www.w3.org/TR/wai-aria/complete#aria-disabled"
		},
		{
			name: "aria-dropeffect",
			doc: "Indicates what functions can be performed when the dragged object is released on the drop target. This allows assistive technologies to convey the possible drag options available to users, including whether a pop-up menu of choices is provided by the application. Typically, drop effect functions can only be provided once an object has been grabbed for a drag operation as the drop effect functions available are dependent on the object being dragged.",
			values: [
				{name: "copy", doc: "A duplicate of the source object will be dropped into the target."},
				{name: "move", doc: "The source object will be removed from its current location and dropped into the target."},
				{name: "link", doc: "A reference or shortcut to the dragged object will be created in the target object."},
				{name: "execute", doc: "A function supported by the drop target is executed, using the drag source as an input."},
				{name: "popup", doc: "There is a popup menu or dialog that allows the user to choose one of the drag operations (copy, move, link, execute) and any other drag functionality, such as cancel."},
				{name: "none", doc: "No operation can be performed; effectively cancels the drag operation if an attempt is made to drop on this object. Ignored if combined with any other token value. e.g. 'none copy' is equivalent to a 'copy' value."}
			],
			url: "http://www.w3.org/TR/wai-aria/complete#aria-dropeffect"
		},
		{
			name: "aria-flowto",
			doc: "Identifies the next element (or elements) in an alternate reading order of content which, at the user's discretion, allows assistive technology to override the general default of reading in document source order.",
			url: "http://www.w3.org/TR/wai-aria/complete#aria-flowto"
		},
		{
			name: "aria-grabbed",
			doc: "Indicates an element's 'grabbed' state in a drag-and-drop operation.",
			values: [
				{name: "true", doc: "Indicates that the element has been 'grabbed' for dragging."},
				{name: "false", doc: "Indicates that the element supports being dragged."},
				{name: "undefined", doc: "Indicates that the element does not support being dragged."}
			],
			url: "http://www.w3.org/TR/wai-aria/complete#aria-grabbed"
		},
		{
			name: "aria-haspopup",
			doc: "Indicates that the element has a popup context menu or sub-level menu.",
			values: [
				{name: "true", doc: "Indicates the object has a popup, either as a descendant or pointed to by aria-owns."},
				{name: "false", doc: "The object has no popup."}
			],
			url: "http://www.w3.org/TR/wai-aria/complete#aria-haspopup"
		},
		{
			name: "aria-hidden",
			doc: "Indicates that the element and all of its descendants are not visible or perceivable to any user as implemented by the author. See related aria-disabled.",
			values: [
				{name: "true", doc: "Indicates that this section of the document and its children are hidden from the rendered view."},
				{name: "false", doc: "Indicates that this section of the document is rendered."}
			],
			url: "http://www.w3.org/TR/wai-aria/complete#aria-hidden"
		},
		{
			name: "aria-invalid",
			doc: "Indicates the entered value does not conform to the format expected by the application.",
			values: [
				{name: "grammar", doc: "A grammatical error was detected."},
				{name: "false", doc: "There are no detected errors in the value."},
				{name: "spelling", doc: "A spelling error was detected."},
				{name: "true", doc: "The value entered by the user has failed validation."}
			],
			url: "http://www.w3.org/TR/wai-aria/complete#aria-invalid"
		},
		{
			name: "aria-label",
			doc: "Defines a string value that labels the current element. See related aria-labelledby.",
			url: "http://www.w3.org/TR/wai-aria/complete#aria-label"
		},
		{
			name: "aria-labelledby",
			doc: "Identifies the element (or elements) that labels the current element. See related aria-label and aria-describedby.",
			url: "http://www.w3.org/TR/wai-aria/complete#aria-labelledby"
		},
		{
			name: "aria-live",
			doc: "Indicates that an element will be updated, and describes the types of updates the user agents, assistive technologies, and user can expect from the live region.",
			values: [
				{name: "off", doc: "Updates to the region will not be presented to the user unless the assistive technology is currently focused on that region."},
				{name: "polite", doc: "(Background change) Assistive technologies SHOULD announce updates at the next graceful opportunity, such as at the end of speaking the current sentence or when the user pauses typing."},
				{name: "assertive", doc: "This information has the highest priority and assistive technologies SHOULD notify the user immediately. Because an interruption may disorient users or cause them to not complete their current task, authors SHOULD NOT use the assertive value unless the interruption is imperative."}
			],
			url: "http://www.w3.org/TR/wai-aria/complete#aria-live"
		},
		{
			name: "aria-owns",
			doc: "Identifies an element (or elements) in order to define a visual, functional, or contextual parent/child relationship between DOM elements where the DOM hierarchy cannot be used to represent the relationship. See related aria-controls.",
			url: "http://www.w3.org/TR/wai-aria/complete#aria-owns"
		},
		{
			name: "aria-relevant",
			doc: "Indicates what user agent change notifications (additions, removals, etc.) assistive technologies will receive within a live region. See related aria-atomic.",
			values: [
				{name: "additions", doc: "Element nodes are added to the DOM within the live region."},
				{name: "removals", doc: "Text or element nodes within the live region are removed from the DOM."},
				{name: "text", doc: "Text is added to any DOM descendant nodes of the live region."},
				{name: "all", doc: "Equivalent to the combination of all values, 'additions removals text'."},
				{name: "additions text", doc: "Equivalent to the combination of values, 'additions text'."}
			],
			url: "http://www.w3.org/TR/wai-aria/complete#aria-relevant"
		}
	];

	aria.tags = Object.create(null);
	aria.tags.a = ["button", "checkbox", "link", "menuitem", "menuitemcheckbox", "menuitemradio", 
		"tab", "treeitem"];
	aria.tags.abbr = ["all"];
	aria.tags.address = ["contentinfo"];
	aria.tags.area = ["link"];
	aria.tags.article = ["application", "article", "document", "main"];
	aria.tags.aside = ["complementary", "note", "presentation", "search"];
	aria.tags.audio = ["application"];
	aria.tags.b = ["all"];
	aria.tags.bdi = ["all"];
	aria.tags.bdo = ["all"];
	aria.tags.blockquote = ["all"];
	aria.tags.body = ["application", "document"];
	aria.tags.br = ["all"];
	aria.tags.button = ["button", "link", "menuitem", "menuitemcheckbox", "menuitemradio", "radio"];
	aria.tags.canvas = ["all"];
	aria.tags.caption = ["all"];
	aria.tags.cite = ["all"];
	aria.tags.code = ["all"];
	aria.tags.data = ["all"];
	aria.tags.datalist = ["listbox"];
	aria.tags.dd = ["all"];
	aria.tags.del = ["all"];
	aria.tags.details = ["alert", "alertdialog", "application", "article", "banner", "button", 
		"columnheader", "complementary", "contentinfo", "definition", "dialog", "directory", "document", "form", 
		"grid", "gridcell", "group", "heading", "img", "link", "list", "listbox", 
		"listitem", "log", "main", "marquee", "math", "menu", "menubar", "navigation", 
		"note", "radiogroup", "region", "row", "rowgroup", "rowheader", "search", "separator", 
		"status", "tab", "tablist", "tabpanel", "timer", "toolbar", "tooltip", "tree", 
		"treegrid", "treeitem"];
	aria.tags.dfn = ["all"];
	aria.tags.dialog = ["alert", "alertdialog", "application", "dialog", "log", "marquee", 
		"status"];
	aria.tags.div = ["all"];
	aria.tags.dl = ["all"];
	aria.tags.dt = ["all"];
	aria.tags.em = ["all"];
	aria.tags.embed = ["application", "document", "img", "presentation"];
	aria.tags.fieldset = ["group", "presentation"];
	aria.tags.figcaption = ["all"];
	aria.tags.figure = ["all"];
	aria.tags.footer = ["contentinfo", "presentation"];
	aria.tags.form = ["all"];
	aria.tags.h1 = ["heading", "presentation", "tab"];
	aria.tags.h2 = ["heading", "presentation", "tab"];
	aria.tags.h3 = ["heading", "presentation", "tab"];
	aria.tags.h4 = ["heading", "presentation", "tab"];
	aria.tags.h5 = ["heading", "presentation", "tab"];
	aria.tags.h6 = ["heading", "presentation", "tab"];
	aria.tags.header = ["banner", "presentation"];
	aria.tags.hr = ["presentation", "separator"];
	aria.tags.i = ["all"];
	aria.tags.iframe = ["application", "document", "img", "presentation"];
	aria.tags.img = ["all"];
	aria.tags.input = ["button", "checkbox", "combobox", "link", "menuitem", "menuitemcheckbox", 
		"menuitemradio", "radio", "searchbox", "slider", "spinbutton", "switch", "textbox"];
	aria.tags.ins = ["all"];
	aria.tags.kbd = ["all"];
	aria.tags.legend = ["all"];
	aria.tags.li = ["listitem", "menuitem", "menuitemcheckbox", "menuitemradio", "option", "presentation", 
		"radio", "tab", "treeitem"];
	aria.tags.link = ["link"];
	aria.tags.main = ["main", "presentation"];
	aria.tags.mark = ["all"];
	aria.tags.menu = ["directory", "list", "listbox", "menu", "menubar", "tablist", 
		"tabpanel", "tree"];
	aria.tags.menuitem = ["menuitem"];
	aria.tags.nav = ["navigation", "presentation"];
	aria.tags.object = ["application", "document", "img", "presentation"];
	aria.tags.ol = ["directory", "group", "list", "listbox", "menu", "menubar", 
		"presentation", "radiogroup", "tablist", "toolbar", "tree"];
	aria.tags.option = ["menuitem", "menuitemradio", "option", "separator"];
	aria.tags.output = ["all"];
	aria.tags.p = ["all"];
	aria.tags.pre = ["all"];
	aria.tags.progress = ["progressbar"];
	aria.tags.q = ["all"];
	aria.tags.rb = ["all"];
	aria.tags.rp = ["all"];
	aria.tags.rt = ["all"];
	aria.tags.rtc = ["all"];
	aria.tags.ruby = ["all"];
	aria.tags.s = ["all"];
	aria.tags.samp = ["all"];
	aria.tags.section = ["alert", "alertdialog", "application", "contentinfo", "dialog", "document", 
		"log", "main", "marquee", "presentation", "region", "search", "status"];
	aria.tags.select = ["listbox", "menu"];
	aria.tags.small = ["all"];
	aria.tags.span = ["all"];
	aria.tags.strong = ["all"];
	aria.tags.sub = ["all"];
	aria.tags.summary = ["button"];
	aria.tags.table = ["all"];
	aria.tags.tbody = ["all"];
	aria.tags.td = ["all"];
	aria.tags.textarea = ["textbox"];
	aria.tags.tfoot = ["all"];
	aria.tags.th = ["all"];
	aria.tags.thead = ["all"];
	aria.tags.time = ["all"];
	aria.tags.tr = ["all"];
	aria.tags.u = ["all"];
	aria.tags.ul = ["directory", "group", "list", "listbox", "menu", "menubar", 
		"presentation", "radiogroup", "tablist", "toolbar", "tree"];
	aria.tags.var = ["all"];
	aria.tags.video = ["application"];
	aria.tags.wbr = ["all"];

	aria.roles = Object.create(null);
	aria.roles.all = ["alert", "alertdialog", "application", "article", "banner", "button", 
		"checkbox", "columnheader", "combobox", "complementary", "contentinfo", "definition", "dialog", "directory", 
		"document", "form", "grid", "gridcell", "group", "heading", "img", "link", 
		"list", "listbox", "listitem", "log", "main", "marquee", "math", "menu", 
		"menubar", "menuitem", "menuitemcheckbox", "menuitemradio", "navigation", "note", "option", "presentation", 
		"progressbar", "radio", "radiogroup", "region", "row", "rowgroup", "rowheader", "scrollbar", 
		"search", "separator", "slider", "spinbutton", "status", "tab", "tablist", "tabpanel", 
		"textbox", "timer", "toolbar", "tooltip", "tree", "treegrid", "treeitem"];
	aria.roles.alert = ["aria-expanded"];
	aria.roles.alertdialog = ["aria-expanded"];
	aria.roles.application = ["aria-expanded"];
	aria.roles.article = ["aria-expanded"];
	aria.roles.banner = ["aria-expanded"];
	aria.roles.button = ["aria-expanded", "aria-pressed"];
	aria.roles.checkbox = ["aria-checked"];
	aria.roles.columnheader = ["aria-expanded", "aria-readonly", "aria-required", "aria-selected", "aria-sort"];
	aria.roles.combobox = ["aria-activedescendant", "aria-autocomplete", "aria-expanded", "aria-required"];
	aria.roles.complementary = ["aria-expanded"];
	aria.roles.contentinfo = ["aria-expanded"];
	aria.roles.definition = ["aria-expanded"];
	aria.roles.dialog = ["aria-expanded"];
	aria.roles.directory = ["aria-expanded"];
	aria.roles.document = ["aria-expanded"];
	aria.roles.form = ["aria-expanded"];
	aria.roles.grid = ["aria-activedescendant", "aria-expanded", "aria-level", "aria-multiselectable", "aria-readonly"];
	aria.roles.gridcell = ["aria-expanded", "aria-readonly", "aria-required", "aria-selected"];
	aria.roles.group = ["aria-activedescendant", "aria-expanded"];
	aria.roles.heading = ["aria-expanded", "aria-level"];
	aria.roles.img = ["aria-expanded"];
	aria.roles.link = ["aria-expanded"];
	aria.roles.list = ["aria-expanded"];
	aria.roles.listbox = ["aria-activedescendant", "aria-expanded", "aria-multiselectable", "aria-required"];
	aria.roles.listitem = ["aria-expanded", "aria-level", "aria-posinset", "aria-setsize"];
	aria.roles.log = ["aria-expanded"];
	aria.roles.main = ["aria-expanded"];
	aria.roles.marquee = ["aria-expanded"];
	aria.roles.math = ["aria-expanded"];
	aria.roles.menu = ["aria-activedescendant", "aria-expanded"];
	aria.roles.menubar = ["aria-activedescendant", "aria-expanded"];
	aria.roles.menuitemcheckbox = ["aria-checked"];
	aria.roles.menuitemradio = ["aria-checked", "aria-posinset", "aria-selected", "aria-setsize"];
	aria.roles.navigation = ["aria-expanded"];
	aria.roles.note = ["aria-expanded"];
	aria.roles.option = ["aria-checked", "aria-posinset", "aria-selected", "aria-setsize"];
	aria.roles.progressbar = ["aria-valuemax", "aria-valuemin", "aria-valuenow", "aria-valuetext"];
	aria.roles.radio = ["aria-checked", "aria-posinset", "aria-selected", "aria-setsize"];
	aria.roles.radiogroup = ["aria-activedescendant", "aria-expanded", "aria-required"];
	aria.roles.region = ["aria-expanded"];
	aria.roles.row = ["aria-activedescendant", "aria-expanded", "aria-level", "aria-selected"];
	aria.roles.rowgroup = ["aria-activedescendant", "aria-expanded"];
	aria.roles.rowheader = ["aria-expanded", "aria-readonly", "aria-required", "aria-selected", "aria-sort"];
	aria.roles.scrollbar = ["aria-orientation", "aria-valuemax", "aria-valuemin", "aria-valuenow", "aria-valuetext"];
	aria.roles.search = ["aria-expanded"];
	aria.roles.separator = ["aria-expanded", "aria-orientation"];
	aria.roles.slider = ["aria-orientation", "aria-valuemax", "aria-valuemin", "aria-valuenow", "aria-valuetext"];
	aria.roles.spinbutton = ["aria-required", "aria-valuemax", "aria-valuemin", "aria-valuenow", "aria-valuetext"];
	aria.roles.status = ["aria-expanded"];
	aria.roles.tab = ["aria-expanded", "aria-selected"];
	aria.roles.tablist = ["aria-activedescendant", "aria-expanded", "aria-level", "aria-multiselectable"];
	aria.roles.tabpanel = ["aria-expanded"];
	aria.roles.textbox = ["aria-activedescendant", "aria-autocomplete", "aria-multiline", "aria-readonly", "aria-required"];
	aria.roles.timer = ["aria-expanded"];
	aria.roles.toolbar = ["aria-activedescendant", "aria-expanded"];
	aria.roles.tooltip = ["aria-expanded"];
	aria.roles.tree = ["aria-activedescendant", "aria-expanded", "aria-multiselectable", "aria-required"];
	aria.roles.treegrid = ["aria-activedescendant", "aria-expanded", "aria-level", "aria-multiselectable", "aria-readonly", "aria-required"];
	aria.roles.treeitem = ["aria-checked", "aria-expanded", "aria-level", "aria-posinset", "aria-selected", "aria-setsize"];

	aria.attributes = Object.create(null);
	aria.attributes.activedescendant = {
		name: "aria-activedescendant",
		doc: "Identifies the currently active descendant of a composite widget.",
		url: "http://www.w3.org/TR/wai-aria/complete#aria-activedescendant"
	};
	aria.attributes.autocomplete = {
		name: "aria-autocomplete",
		doc: "Indicates whether user input completion suggestions are provided.",
		values: [
			{name: "inline", doc: "The system provides text after the caret as a suggestion for how to complete the field."},
			{name: "list", doc: "A list of choices appears from which the user can choose."},
			{name: "both", doc: "A list of choices appears and the currently selected suggestion also appears inline."},
			{name: "none", doc: "No input completion suggestions are provided."}
		],
		url: "http://www.w3.org/TR/wai-aria/complete#aria-autocomplete"
	};
	aria.attributes.checked = {
		name: "aria-checked",
		doc: "Indicates the current 'checked' state of checkboxes, radio buttons, and other widgets. See related aria-pressed and aria-selected.",
		values: [
			{name: "true", doc: "The element is checked."},
			{name: "false", doc: "The element supports being checked but is not currently checked."},
			{name: "mixed", doc: "Indicates a mixed mode value for a tri-state checkbox or menuitemcheckbox."},
			{name: "undefined", doc: "The element does not support being checked."}
		],
		url: "http://www.w3.org/TR/wai-aria/complete#aria-checked"
	};
	aria.attributes.expanded = {
		name: "aria-expanded",
		doc: "Indicates whether the element, or another grouping element it controls, is currently expanded or collapsed.",
		values: [
			{name: "true", doc: "The element, or another grouping element it controls, is expanded."},
			{name: "false", doc: "The element, or another grouping element it controls, is collapsed."},
			{name: "undefined", doc: "The element, or another grouping element it controls, is neither expandable nor collapsible; all its child elements are shown or there are no child elements."}
		],
		url: "http://www.w3.org/TR/wai-aria/complete#aria-expanded"
	};
	aria.attributes.level = {
		name: "aria-level",
		doc: "Defines the hierarchical level of an element within a structure.",
		url: "http://www.w3.org/TR/wai-aria/complete#aria-level"
	};
	aria.attributes.multiline = {
		name: "aria-multiline",
		doc: "Indicates whether a text box accepts multiple lines of input or only a single line.",
		values: [
			{name: "true", doc: "This is a multi-line text box."},
			{name: "false", doc: "This is a single-line text box."}
		],
		url: "http://www.w3.org/TR/wai-aria/complete#aria-multiline"
	};
	aria.attributes.multiselectable = {
		name: "aria-multiselectable",
		doc: "Indicates that the user may select more than one item from the current selectable descendants.",
		values: [
			{name: "true", doc: "More than one item in the widget may be selected at a time."},
			{name: "false", doc: "Only one item can be selected."}
		],
		url: "http://www.w3.org/TR/wai-aria/complete#aria-multiselectable"
	};
	aria.attributes.orientation = {
		name: "aria-orientation",
		doc: "Indicates whether the element and orientation is horizontal or vertical.",
		values: [
			{name: "vertical", doc: "The element is oriented vertically."},
			{name: "horizontal", doc: "The element is oriented horizontally."}
		],
		url: "http://www.w3.org/TR/wai-aria/complete#aria-orientation"
	};
	aria.attributes.posinset = {
		name: "aria-posinset",
		doc: "Defines an element's number or position in the current set of listitems or treeitems. Not required if all elements in the set are present in the DOM. See related aria-setsize.",
		url: "http://www.w3.org/TR/wai-aria/complete#aria-posinset"
	};
	aria.attributes.pressed = {
		name: "aria-pressed",
		doc: "Indicates the current 'pressed' state of toggle buttons. See related aria-checked and aria-selected.",
		values: [
			{name: "true", doc: "The element is pressed."},
			{name: "false", doc: "The element supports being pressed but is not currently pressed."},
			{name: "mixed", doc: "Indicates a mixed mode value for a tri-state toggle button."},
			{name: "undefined", doc: "The element does not support being pressed."}
		],
		url: "http://www.w3.org/TR/wai-aria/complete#aria-pressed"
	};
	aria.attributes.readonly = {
		name: "aria-readonly",
		doc: "Indicates that the element is not editable, but is otherwise operable. See related aria-disabled.",
		values: [
			{name: "true", doc: "The user cannot change the value of the element."},
			{name: "false", doc: "The user can set the value of the element."}
		],
		url: "http://www.w3.org/TR/wai-aria/complete#aria-readonly"
	};
	aria.attributes.required = {
		name: "aria-required",
		doc: "Indicates that user input is required on the element before a form may be submitted.",
		values: [
			{name: "true", doc: "Users need to provide input on an element before a form is submitted."},
			{name: "false", doc: "User input is not necessary to submit the form."}
		],
		url: "http://www.w3.org/TR/wai-aria/complete#aria-required"
	};
	aria.attributes.selected = {
		name: "aria-selected",
		doc: "Indicates the current 'selected' state of various widgets. See related aria-checked and aria-pressed.",
		values: [
			{name: "true", doc: "The selectable element is selected."},
			{name: "false", doc: "The selectable element is not selected."},
			{name: "undefined", doc: "The element is not selectable."}
		],
		url: "http://www.w3.org/TR/wai-aria/complete#aria-selected"
	};
	aria.attributes.setsize = {
		name: "aria-setsize",
		doc: "Defines the number of items in the current set of listitems or treeitems. Not required if all elements in the set are present in the DOM. See related aria-posinset.",
		url: "http://www.w3.org/TR/wai-aria/complete#aria-setsize"
	};
	aria.attributes.sort = {
		name: "aria-sort",
		doc: "Indicates if items in a table or grid are sorted in ascending or descending order.",
		values: [
			{name: "ascending", doc: "Items are sorted in ascending order by this column."},
			{name: "descending", doc: "Items are sorted in descending order by this column."},
			{name: "none", doc: "There is no defined sort applied to the column."},
			{name: "other", doc: "A sort algorithm other than ascending or descending has been applied."}
		],
		url: "http://www.w3.org/TR/wai-aria/complete#aria-sort"
	};
	aria.attributes.valuemax = {
		name: "aria-valuemax",
		doc: "Defines the maximum allowed value for a range widget.",
		url: "http://www.w3.org/TR/wai-aria/complete#aria-valuemax"
	};
	aria.attributes.valuemin = {
		name: "aria-valuemin",
		doc: "Defines the minimum allowed value for a range widget.",
		url: "http://www.w3.org/TR/wai-aria/complete#aria-valuemin"
	};
	aria.attributes.valuenow = {
		name: "aria-valuenow",
		doc: "Defines the current value for a range widget. See related aria-valuetext.",
		url: "http://www.w3.org/TR/wai-aria/complete#aria-valuenow"
	};
	aria.attributes.valuetext = {
		name: "aria-valuetext",
		doc: "Defines the human readable text alternative of aria-valuenow for a range widget.",
		url: "http://www.w3.org/TR/wai-aria/complete#aria-valuetext"
	};
	aria.attributes.role = {
		name: "role",
		doc: "Describes the role(s) the current element plays in the context of the document. This can be used, for example, by applications and assistive technologies to determine the purpose of an element.",
		values: [
			{name: "alert", doc: "A message with important, and usually time-sensitive, information. See related alertdialog and status.", url: "https://www.w3.org/TR/wai-aria/complete#alert"},
			{name: "alertdialog", doc: "A type of dialog that contains an alert message, where initial focus goes to an element within the dialog. See related alert and dialog.", url: "https://www.w3.org/TR/wai-aria/complete#alertdialog"},
			{name: "application", doc: "A region declared as a web application, as opposed to a web document.", url: "https://www.w3.org/TR/wai-aria/complete#application"},
			{name: "article", doc: "A section of a page that consists of a composition that forms an independent part of a document, page, or site.", url: "https://www.w3.org/TR/wai-aria/complete#article"},
			{name: "banner", doc: "A region that contains mostly site-oriented content, rather than page-specific content.", url: "https://www.w3.org/TR/wai-aria/complete#banner"},
			{name: "button", doc: "An input that allows for user-triggered actions when clicked or pressed. See related link.", url: "https://www.w3.org/TR/wai-aria/complete#button"},
			{name: "checkbox", doc: "A checkable input that has three possible values: true, false, or mixed.", url: "https://www.w3.org/TR/wai-aria/complete#checkbox"},
			{name: "columnheader", doc: "A cell containing header information for a column.", url: "https://www.w3.org/TR/wai-aria/complete#columnheader"},
			{name: "combobox", doc: "A presentation of a select; usually similar to a textbox where users can type ahead to select an option, or type to enter arbitrary text as a new item in the list. See related listbox.", url: "https://www.w3.org/TR/wai-aria/complete#combobox"},
			{name: "complementary", doc: "A supporting section of the document, designed to be complementary to the main content at a similar level in the DOM hierarchy, but remains meaningful when separated from the main content.", url: "https://www.w3.org/TR/wai-aria/complete#complementary"},
			{name: "contentinfo", doc: "A large perceivable region that contains information about the parent document.", url: "https://www.w3.org/TR/wai-aria/complete#contentinfo"},
			{name: "definition", doc: "A definition of a term or concept.", url: "https://www.w3.org/TR/wai-aria/complete#definition"},
			{name: "dialog", doc: "A dialog is an application window that is designed to interrupt the current processing of an application in order to prompt the user to enter information or require a response. See related alertdialog.", url: "https://www.w3.org/TR/wai-aria/complete#dialog"},
			{name: "directory", doc: "A list of references to members of a group, such as a static table of contents.", url: "https://www.w3.org/TR/wai-aria/complete#directory"},
			{name: "document", doc: "A region containing related information that is declared as document content, as opposed to a web application.", url: "https://www.w3.org/TR/wai-aria/complete#document"},
			{name: "form", doc: "A landmark region that contains a collection of items and objects that, as a whole, combine to create a form. See related search.", url: "https://www.w3.org/TR/wai-aria/complete#form"},
			{name: "grid", doc: "A grid is an interactive control which contains cells of tabular data arranged in rows and columns, like a table.", url: "https://www.w3.org/TR/wai-aria/complete#grid"},
			{name: "gridcell", doc: "A cell in a grid or treegrid.", url: "https://www.w3.org/TR/wai-aria/complete#gridcell"},
			{name: "group", doc: "A set of user interface objects which are not intended to be included in a page summary or table of contents by assistive technologies.", url: "https://www.w3.org/TR/wai-aria/complete#group"},
			{name: "heading", doc: "A heading for a section of the page.", url: "https://www.w3.org/TR/wai-aria/complete#heading"},
			{name: "img", doc: "A container for a collection of elements that form an image.", url: "https://www.w3.org/TR/wai-aria/complete#img"},
			{name: "link", doc: "An interactive reference to an internal or external resource that, when activated, causes the user agent to navigate to that resource. See related button.", url: "https://www.w3.org/TR/wai-aria/complete#link"},
			{name: "list", doc: "A group of non-interactive list items. See related listbox.", url: "https://www.w3.org/TR/wai-aria/complete#list"},
			{name: "listbox", doc: "A widget that allows the user to select one or more items from a list of choices. See related combobox and list.", url: "https://www.w3.org/TR/wai-aria/complete#listbox"},
			{name: "listitem", doc: "A single item in a list or directory.", url: "https://www.w3.org/TR/wai-aria/complete#listitem"},
			{name: "log", doc: "A type of live region where new information is added in meaningful order and old information may disappear. See related marquee.", url: "https://www.w3.org/TR/wai-aria/complete#log"},
			{name: "main", doc: "The main content of a document.", url: "https://www.w3.org/TR/wai-aria/complete#main"},
			{name: "marquee", doc: "A type of live region where non-essential information changes frequently. See related log.", url: "https://www.w3.org/TR/wai-aria/complete#marquee"},
			{name: "math", doc: "Content that represents a mathematical expression.", url: "https://www.w3.org/TR/wai-aria/complete#math"},
			{name: "menu", doc: "A type of widget that offers a list of choices to the user.", url: "https://www.w3.org/TR/wai-aria/complete#menu"},
			{name: "menubar", doc: "A presentation of menu that usually remains visible and is usually presented horizontally.", url: "https://www.w3.org/TR/wai-aria/complete#menubar"},
			{name: "menuitem", doc: "An option in a set of choices contained by a menu or menubar.", url: "https://www.w3.org/TR/wai-aria/complete#menuitem"},
			{name: "menuitemcheckbox", doc: "A menuitem with a checkable state whose possible values are true, false, or mixed.", url: "https://www.w3.org/TR/wai-aria/complete#menuitemcheckbox"},
			{name: "menuitemradio", doc: "A checkable menuitem in a set of elements with role menuitemradio, only one of which can be checked at a time.", url: "https://www.w3.org/TR/wai-aria/complete#menuitemradio"},
			{name: "navigation", doc: "A collection of navigational elements (usually links) for navigating the document or related documents.", url: "https://www.w3.org/TR/wai-aria/complete#navigation"},
			{name: "note", doc: "A section whose content is parenthetic or ancillary to the main content of the resource.", url: "https://www.w3.org/TR/wai-aria/complete#note"},
			{name: "option", doc: "A selectable item in a select list.", url: "https://www.w3.org/TR/wai-aria/complete#option"},
			{name: "presentation", doc: "An element whose implicit native role semantics will not be mapped to the accessibility API.", url: "https://www.w3.org/TR/wai-aria/complete#presentation"},
			{name: "progressbar", doc: "An element that displays the progress status for tasks that take a long time.", url: "https://www.w3.org/TR/wai-aria/complete#progressbar"},
			{name: "radio", doc: "A checkable input in a group of radio roles, only one of which can be checked at a time.", url: "https://www.w3.org/TR/wai-aria/complete#radio"},
			{name: "radiogroup", doc: "A group of radio buttons.", url: "https://www.w3.org/TR/wai-aria/complete#radiogroup"},
			{name: "region", doc: "A large perceivable section of a web page or document, that is important enough to be included in a page summary or table of contents, for example, an area of the page containing live sporting event statistics.", url: "https://www.w3.org/TR/wai-aria/complete#region"},
			{name: "row", doc: "A row of cells in a grid.", url: "https://www.w3.org/TR/wai-aria/complete#row"},
			{name: "rowgroup", doc: "A group containing one or more row elements in a grid.", url: "https://www.w3.org/TR/wai-aria/complete#rowgroup"},
			{name: "rowheader", doc: "A cell containing header information for a row in a grid.", url: "https://www.w3.org/TR/wai-aria/complete#rowheader"},
			{name: "search", doc: "A landmark region that contains a collection of items and objects that, as a whole, combine to create a search facility. See related form.", url: "https://www.w3.org/TR/wai-aria/complete#search"},
			{name: "separator", doc: "A divider that separates and distinguishes sections of content or groups of menuitems.", url: "https://www.w3.org/TR/wai-aria/complete#separator"},
			{name: "scrollbar", doc: "A graphical object that controls the scrolling of content within a viewing area, regardless of whether the content is fully displayed within the viewing area.", url: "https://www.w3.org/TR/wai-aria/complete#scrollbar"},
			{name: "slider", doc: "A user input where the user selects a value from within a given range.", url: "https://www.w3.org/TR/wai-aria/complete#slider"},
			{name: "spinbutton", doc: "A form of range that expects the user to select from among discrete choices.", url: "https://www.w3.org/TR/wai-aria/complete#spinbutton"},
			{name: "status", doc: "A container whose content is advisory information for the user but is not important enough to justify an alert, often but not necessarily presented as a status bar. See related alert.", url: "https://www.w3.org/TR/wai-aria/complete#status"},
			{name: "tab", doc: "A grouping label providing a mechanism for selecting the tab content that is to be rendered to the user.", url: "https://www.w3.org/TR/wai-aria/complete#tab"},
			{name: "tablist", doc: "A list of tab elements, which are references to tabpanel elements.", url: "https://www.w3.org/TR/wai-aria/complete#tablist"},
			{name: "tabpanel", doc: "A container for the resources associated with a tab, where each tab is contained in a tablist.", url: "https://www.w3.org/TR/wai-aria/complete#tabpanel"},
			{name: "textbox", doc: "Input that allows free-form text as its value.", url: "https://www.w3.org/TR/wai-aria/complete#textbox"},
			{name: "timer", doc: "A type of live region containing a numerical counter which indicates an amount of elapsed time from a start point, or the time remaining until an end point.", url: "https://www.w3.org/TR/wai-aria/complete#timer"},
			{name: "toolbar", doc: "A collection of commonly used function buttons or controls represented in compact visual form.", url: "https://www.w3.org/TR/wai-aria/complete#toolbar"},
			{name: "tooltip", doc: "A contextual popup that displays a description for an element.", url: "https://www.w3.org/TR/wai-aria/complete#tooltip"},
			{name: "tree", doc: "A type of list that may contain sub-level nested groups that can be collapsed and expanded.", url: "https://www.w3.org/TR/wai-aria/complete#tree"},
			{name: "treegrid", doc: "A grid whose rows can be expanded and collapsed in the same manner as for a tree.", url: "https://www.w3.org/TR/wai-aria/complete#treegrid"},
			{name: "treeitem", doc: "An option item of a tree. This is an element within a tree that may be expanded or collapsed if it contains a sub-level group of treeitem elements.", url: "https://www.w3.org/TR/wai-aria/complete#treeitem"}
		],
		url: "http://www.w3.org/TR/role-attribute/"
	};

	/**
	 * @description Returns the set of attributes that apply to the given DOM node
	 * @param {Object} node The DOM node to get attributes for
	 * @returns {Array.<Object>} The array of attributes for the given node or an empty array, never null
	 * @since 10.0
	 */
	function getAttributesForNode(node) {
		var ret = Object.create(null);
		var attrs = [].concat(attributes.globals);
		var ariaattrs = [];
		if (node){
			var testNode = node;
			if (node.type === 'attr'){
				testNode = node.parent;
			}
			if(testNode && testNode.type === 'tag') {
				var tags = attributes.tags[testNode.name];
				if(Array.isArray(tags) && tags.length > 0) {
					attrs = attrs.concat(tags);
				}
				// Determine ARIA attributes for node
				tags = aria.tags[testNode.name];
				if (Array.isArray(tags) && tags.length > 0) {
					var roles = [];
					if (tags[0] === "all") {
						roles = aria.roles.all;
					} else {
						roles = tags; // tag-specific roles
					}
					for (var i = 0; i < roles.length; i++) {
						var attrlist = aria.roles[roles[i]];
						if (Array.isArray(attrlist) && attrlist.length > 0) {
							for (var j = 0; j < attrlist.length; j++) {
								var attrkey = attrlist[j].substring(5); // strip off the "aria-"
								var attr = aria.attributes[attrkey];
								var found = false;
								for (var k = 0; k < ariaattrs.length; k++) {
									if (ariaattrs[k].name === attr.name) {
										found = true;
										break;
									}
								}
								if (!found) {
									ariaattrs.push(attr);
								}
							}
						}
					}
					ariaattrs.push(aria.attributes.role); // this tag allows role
				}
				ariaattrs = ariaattrs.concat(aria.globalAttributes); // add ARIA globals last to avoid looping through them (above)
			}
		}
		ret.global = attrs;
		ret.aria = ariaattrs;
		ret.formevents = attributes.formevents.slice(0);
		ret.keyboardevents = attributes.keyboardevents.slice(0);
		ret.mouseevents = attributes.mouseevents.slice(0);
		ret.windowevents = attributes.windowevents.slice(0);
		return ret;
	}
	
	function getValuesForAttribute(node) {
		var values = null;
		var url = null;
		if (node.type === 'attr') {
			var attr = null;
			var name = node.name;
			if (name.indexOf("aria-") === 0 || name === "role") {
				// ARIA attribute
				if (name === "role") {
					values = aria.attributes.role.values;
				} else {
					var key = name.substring(5); // strip off the "aria-"
					attr = aria.attributes[key];
					if (!attr) {
						attr = findAttribute(name, aria.globalAttributes);
					}
				}
			} else {
				// HTML attribute
				var tagAttrs = null;
				var testNode = node.parent;
				if (testNode && testNode.type === 'tag') {
					tagAttrs = attributes.tags[testNode.name];
				}
				if (Array.isArray(tagAttrs)) {
					attr = findAttribute(name, tagAttrs);
				} else {
					attr = findAttribute(name, attributes.globals);
				}
			}
			if (attr) {
				values = attr.values;
				url = attr.url;
			}
		}
		if (values && url) {
			for (var i = 0; i < values.length; i++) {
				if (!values[i].url) {
					values[i]["url"] = url;
				}
			}
		}
		return values;
	}
	
	function findAttribute(name, attributes) {
		var attr = null;
		for (var i = 0; i < attributes.length; i++) {
			if (attributes[i].name === name) {
				attr = attributes[i];
				break;
			}
		}
		return attr;
	}

	return {
		getAttributesForNode: getAttributesForNode,
		getValuesForAttribute: getValuesForAttribute
	};
});
/*******************************************************************************
 * @license
 * Copyright (c) 2015 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/* eslint-env amd */
define('webtools/deprecatedAttributes',[

], function() {
	/* eslint-disable missing-nls */
	// Scraped from http://www.w3.org/TR/html4/index/attributes.html
	// and http://www.w3.org/TR/html5/obsolete.html#non-conforming-features
	// Microdata deprecated attributes (itemid, itemprop, itemref, itemscope, and itemtype) determined from http://manu.sporny.org/2013/microdata-downward-spiral/
	var deprecated = Object.create(null);
	deprecated.global = Object.create(null);
	deprecated.global.itemid = "HTML 5";
	deprecated.global.itemprop = "HTML 5";
	deprecated.global.itemref = "HTML 5";
	deprecated.global.itemscope = "HTML 5";
	deprecated.global.itemtype = "HTML 5";
	deprecated.a = Object.create(null);
	deprecated.a.charset = "HTML 5";
	deprecated.a.coords = "HTML 5";
	deprecated.a.datafld = "HTML 5";
	deprecated.a.datasrc = "HTML 5";
	deprecated.a.methods = "HTML 5";
	deprecated.a.shape = "HTML 5";
	deprecated.a.urn = "HTML 5";
	deprecated.applet = Object.create(null);
	deprecated.applet.align = "HTML 4.01";
	deprecated.applet.alt = "HTML 4.01";
	deprecated.applet.archive = "HTML 4.01";
	deprecated.applet.code = "HTML 4.01";
	deprecated.applet.codebase = "HTML 4.01";
	deprecated.applet.datafld = "HTML 5";
	deprecated.applet.datasrc = "HTML 5";
	deprecated.applet.height = "HTML 4.01";
	deprecated.applet.hspace = "HTML 4.01";
	deprecated.applet.name = "HTML 4.01";
	deprecated.applet.object = "HTML 4.01";
	deprecated.applet.vspace = "HTML 4.01";
	deprecated.applet.width = "HTML 4.01";
	deprecated.area = Object.create(null);
	deprecated.area.nohref = "HTML 5";
	deprecated.basefont = Object.create(null);
	deprecated.basefont.color = "HTML 4.01";
	deprecated.basefont.face = "HTML 4.01";
	deprecated.basefont.size = "HTML 4.01";
	deprecated.body = Object.create(null);
	deprecated.body.alink = "HTML 5";
	deprecated.body.background = "HTML 5";
	deprecated.body.bgcolor = "HTML 5";
	deprecated.body.link = "HTML 5";
	deprecated.body.marginbottom = "HTML 5";
	deprecated.body.marginheight = "HTML 5";
	deprecated.body.marginleft = "HTML 5";
	deprecated.body.marginright = "HTML 5";
	deprecated.body.margintop = "HTML 5";
	deprecated.body.marginwidth = "HTML 5";
	deprecated.body.text = "HTML 5";
	deprecated.body.vlink = "HTML 5";
	deprecated.br = Object.create(null);
	deprecated.br.clear = "HTML 5";
	deprecated.button = Object.create(null);
	deprecated.button.datafld = "HTML 5";
	deprecated.button.dataformatas = "HTML 5";
	deprecated.button.datasrc = "HTML 5";
	deprecated.caption = Object.create(null);
	deprecated.caption.align = "HTML 5";
	deprecated.col = Object.create(null);
	deprecated.col.align = "HTML 5";
	deprecated.col.char = "HTML 5";
	deprecated.col.charoff = "HTML 5";
	deprecated.col.valign = "HTML 5";
	deprecated.col.width = "HTML 5";
	deprecated.dir = Object.create(null);
	deprecated.dir.compact = "HTML 4.01";
	deprecated.div = Object.create(null);
	deprecated.div.align = "HTML 5";
	deprecated.div.datafld = "HTML 5";
	deprecated.div.dataformatas = "HTML 5";
	deprecated.div.datasrc = "HTML 5";
	deprecated.dl = Object.create(null);
	deprecated.dl.compact = "HTML 5";
	deprecated.embed = Object.create(null);
	deprecated.embed.align = "HTML 5";
	deprecated.embed.hspace = "HTML 5";
	deprecated.embed.name = "HTML 5";
	deprecated.embed.vspace = "HTML 5";
	deprecated.fieldset = Object.create(null);
	deprecated.fieldset.datafld = "HTML 5";
	deprecated.font = Object.create(null);
	deprecated.font.color = "HTML 4.01";
	deprecated.font.face = "HTML 4.01";
	deprecated.font.size = "HTML 4.01";
	deprecated.form = Object.create(null);
	deprecated.form.accept = "HTML 5";
	deprecated.frame = Object.create(null);
	deprecated.frame.datafld = "HTML 5";
	deprecated.frame.datasrc = "HTML 5";
	deprecated.h1 = Object.create(null);
	deprecated.h1.align = "HTML 4.01";
	deprecated.h2 = Object.create(null);
	deprecated.h2.align = "HTML 4.01";
	deprecated.h3 = Object.create(null);
	deprecated.h3.align = "HTML 4.01";
	deprecated.h4 = Object.create(null);
	deprecated.h4.align = "HTML 4.01";
	deprecated.h5 = Object.create(null);
	deprecated.h5.align = "HTML 4.01";
	deprecated.h6 = Object.create(null);
	deprecated.h6.align = "HTML 4.01";
	deprecated.head = Object.create(null);
	deprecated.head.profile = "HTML 5";
	deprecated.hr = Object.create(null);
	deprecated.hr.align = "HTML 5";
	deprecated.hr.color = "HTML 5";
	deprecated.hr.noshade = "HTML 5";
	deprecated.hr.size = "HTML 5";
	deprecated.hr.width = "HTML 5";
	deprecated.html = Object.create(null);
	deprecated.html.version = "HTML 5";
	deprecated.iframe = Object.create(null);
	deprecated.iframe.align = "HTML 5";
	deprecated.iframe.allowtransparency = "HTML 5";
	deprecated.iframe.datafld = "HTML 5";
	deprecated.iframe.datasrc = "HTML 5";
	deprecated.iframe.frameborder = "HTML 5";
	deprecated.iframe.hspace = "HTML 5";
	deprecated.iframe.marginheight = "HTML 5";
	deprecated.iframe.marginwidth = "HTML 5";
	deprecated.iframe.scrolling = "HTML 5";
	deprecated.iframe.vspace = "HTML 5";
	deprecated.img = Object.create(null);
	deprecated.img.align = "HTML 5";
	deprecated.img.border = "HTML 4.01";
	deprecated.img.datafld = "HTML 5";
	deprecated.img.datasrc = "HTML 5";
	deprecated.img.hspace = "HTML 5";
	deprecated.img.lowsrc = "HTML 5";
	deprecated.img.name = "HTML 5";
	deprecated.img.vspace = "HTML 5";
	deprecated.input = Object.create(null);
	deprecated.input.align = "HTML 5";
	deprecated.input.datafld = "HTML 5";
	deprecated.input.dataformatas = "HTML 5";
	deprecated.input.datasrc = "HTML 5";
	deprecated.input.hspace = "HTML 5";
	deprecated.input.ismap = "HTML 5";
	deprecated.input.usemap = "HTML 5";
	deprecated.input.vspace = "HTML 5";
	deprecated.isindex = Object.create(null);
	deprecated.isindex.prompt = "HTML 4.01";
	deprecated.label = Object.create(null);
	deprecated.label.datafld = "HTML 5";
	deprecated.label.dataformatas = "HTML 5";
	deprecated.label.datasrc = "HTML 5";
	deprecated.legend = Object.create(null);
	deprecated.legend.align = "HTML 5";
	deprecated.legend.datafld = "HTML 5";
	deprecated.legend.dataformatas = "HTML 5";
	deprecated.legend.datasrc = "HTML 5";
	deprecated.li = Object.create(null);
	deprecated.li.type = "HTML 5";
	deprecated.li.value = "HTML 4.01";
	deprecated.link = Object.create(null);
	deprecated.link.charset = "HTML 5";
	deprecated.link.methods = "HTML 5";
	deprecated.link.target = "HTML 5";
	deprecated.link.urn = "HTML 5";
	deprecated.marquee = Object.create(null);
	deprecated.marquee.datafld = "HTML 5";
	deprecated.marquee.dataformatas = "HTML 5";
	deprecated.marquee.datasrc = "HTML 5";
	deprecated.menu = Object.create(null);
	deprecated.menu.compact = "HTML 4.01";
	deprecated.meta = Object.create(null);
	deprecated.meta.scheme = "HTML 5";
	deprecated.object = Object.create(null);
	deprecated.object.align = "HTML 5";
	deprecated.object.archive = "HTML 5";
	deprecated.object.border = "HTML 5";
	deprecated.object.classid = "HTML 5";
	deprecated.object.code = "HTML 5";
	deprecated.object.codebase = "HTML 5";
	deprecated.object.codetype = "HTML 5";
	deprecated.object.datafld = "HTML 5";
	deprecated.object.dataformatas = "HTML 5";
	deprecated.object.datasrc = "HTML 5";
	deprecated.object.declare = "HTML 5";
	deprecated.object.hspace = "HTML 5";
	deprecated.object.standby = "HTML 5";
	deprecated.object.vspace = "HTML 5";
	deprecated.ol = Object.create(null);
	deprecated.ol.compact = "HTML 5";
	deprecated.ol.start = "HTML 4.01";
	deprecated.ol.type = "HTML 4.01";
	deprecated.option = Object.create(null);
	deprecated.option.dataformatas = "HTML 5";
	deprecated.option.datasrc = "HTML 5";
	deprecated.option.name = "HTML 5";
	deprecated.p = Object.create(null);
	deprecated.p.align = "HTML 5";
	deprecated.param = Object.create(null);
	deprecated.param.datafld = "HTML 5";
	deprecated.param.type = "HTML 5";
	deprecated.param.valuetype = "HTML 5";
	deprecated.pre = Object.create(null);
	deprecated.pre.width = "HTML 5";
	deprecated.script = Object.create(null);
	deprecated.script.event = "HTML 5";
	deprecated.script.for = "HTML 5";
	deprecated.script.language = "HTML 4.01";
	deprecated.select = Object.create(null);
	deprecated.select.datafld = "HTML 5";
	deprecated.select.dataformatas = "HTML 5";
	deprecated.select.datasrc = "HTML 5";
	deprecated.span = Object.create(null);
	deprecated.span.datafld = "HTML 5";
	deprecated.span.dataformatas = "HTML 5";
	deprecated.span.datasrc = "HTML 5";
	deprecated.table = Object.create(null);
	deprecated.table.align = "HTML 5";
	deprecated.table.background = "HTML 5";
	deprecated.table.bgcolor = "HTML 5";
	deprecated.table.bordercolor = "HTML 5";
	deprecated.table.cellpadding = "HTML 5";
	deprecated.table.cellspacing = "HTML 5";
	deprecated.table.dataformatas = "HTML 5";
	deprecated.table.datapagesize = "HTML 5";
	deprecated.table.datasrc = "HTML 5";
	deprecated.table.frame = "HTML 5";
	deprecated.table.rules = "HTML 5";
	deprecated.table.summary = "HTML 5";
	deprecated.table.width = "HTML 5";
	deprecated.tbody = Object.create(null);
	deprecated.tbody.align = "HTML 5";
	deprecated.tbody.background = "HTML 5";
	deprecated.tbody.char = "HTML 5";
	deprecated.tbody.charoff = "HTML 5";
	deprecated.tbody.valign = "HTML 5";
	deprecated.td = Object.create(null);
	deprecated.td.align = "HTML 5";
	deprecated.td.axis = "HTML 5";
	deprecated.td.background = "HTML 5";
	deprecated.td.bgcolor = "HTML 5";
	deprecated.td.char = "HTML 5";
	deprecated.td.charoff = "HTML 5";
	deprecated.td.height = "HTML 5";
	deprecated.td.nowrap = "HTML 5";
	deprecated.td.scope = "HTML 5";
	deprecated.td.valign = "HTML 5";
	deprecated.td.width = "HTML 5";
	deprecated.textarea = Object.create(null);
	deprecated.textarea.datafld = "HTML 5";
	deprecated.textarea.datasrc = "HTML 5";
	deprecated.tfoot = Object.create(null);
	deprecated.tfoot.align = "HTML 5";
	deprecated.tfoot.background = "HTML 5";
	deprecated.tfoot.char = "HTML 5";
	deprecated.tfoot.charoff = "HTML 5";
	deprecated.tfoot.valign = "HTML 5";
	deprecated.th = Object.create(null);
	deprecated.th.align = "HTML 5";
	deprecated.th.axis = "HTML 5";
	deprecated.th.background = "HTML 5";
	deprecated.th.bgcolor = "HTML 5";
	deprecated.th.char = "HTML 5";
	deprecated.th.charoff = "HTML 5";
	deprecated.th.height = "HTML 5";
	deprecated.th.nowrap = "HTML 5";
	deprecated.th.valign = "HTML 5";
	deprecated.th.width = "HTML 5";
	deprecated.thead = Object.create(null);
	deprecated.thead.align = "HTML 5";
	deprecated.thead.background = "HTML 5";
	deprecated.thead.char = "HTML 5";
	deprecated.thead.charoff = "HTML 5";
	deprecated.thead.valign = "HTML 5";
	deprecated.tr = Object.create(null);
	deprecated.tr.align = "HTML 5";
	deprecated.tr.background = "HTML 5";
	deprecated.tr.bgcolor = "HTML 5";
	deprecated.tr.char = "HTML 5";
	deprecated.tr.charoff = "HTML 5";
	deprecated.tr.valign = "HTML 5";
	deprecated.ul = Object.create(null);
	deprecated.ul.compact = "HTML 5";
	deprecated.ul.type = "HTML 5";
	
	/**
	 * @description Returns the version in which the given attribute is deprecated for the given element tag or <code>null</code> if it is not deprecated.
	 * @param {String} tagName The name of the element tag
	 * @param {String} attributeName The name of the attribute
	 * @returns {String} The version in which the attribute is deprecated or <code>null</code> if it is not deprecated.
	 * @since 10.0
	 */
	function isAttributeDeprecated(tagName, attributeName) {
		var dep = deprecated[tagName];
		if (dep){
			dep = dep[attributeName];
			if (dep){
				return dep;
			}
		}
		dep = deprecated.global[attributeName];
		if (dep){
			return dep;
		}
		return null;
	}
	
	return {
		isAttributeDeprecated: isAttributeDeprecated
	};
});
/*******************************************************************************
 * @license
 * Copyright (c) 2015 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/* eslint-env amd */
define('webtools/tags',[

], function() {
	/* eslint-disable missing-nls */
	var tagTemplates = Object.create(null);
	tagTemplates = [
	/*
	 * The following data was scraped from https://developer.mozilla.org/en-US/docs/Web/HTML/Element
	 * These tags were missing from the scraped data and added manually:
	 * 		a, aside, blockquote, font, marquee, noframes
	 */
		{
			name: "a",
			category: "Text content",
			doc: "The HTML Anchor Element (<a>) defines a hyperlink",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/a"
		},
		{
			name: "abbr",
			category: "Inline text semantics",
			doc: "The HTML Abbreviation element (<abbr>) represents an abbreviation and optionally provides a full description for it. If present, the title attribute must contain this full description and nothing else.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/abbr"
		},
		{
			name: "acronym",
			category: "Obsolete and deprecated elements",
			doc: "The HTML Acronym Element (<acronym>) allows authors to clearly indicate a sequence of characters that compose an acronym or abbreviation for a word.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/acronym"
		},
		{
			name: "address",
			category: "Content sectioning",
			doc: "The HTML Address Element (<address>) should be used by authors to supply contact information for its nearest <article> or <body> ancestor; in the latter case, it applies to the whole document.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/address"
		},
		{
			name: "applet",
			category: "Obsolete and deprecated elements",
			doc: "The HTML Applet Element (<applet>) identifies the inclusion of a Java applet.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/applet"
		},
		{
			name: "area",
			category: "Image & multimedia",
			doc: "The HTML <area> element defines a hot-spot region on an image, and optionally associates it with a hypertext link. This element is used only within a <map> element.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/area"
		},
		
		{
			name: "article",
			category: "Content sectioning",
			doc: "The HTML Article Element (<article>) represents a self-contained composition in a document, page, application, or site, which is intended to be independently distributable or reusable, e.g., in syndication. This could be a forum post, a magazine or newspaper article, a blog entry, or any other independent item of content. Each <article> should be identified, typically by including a heading (h1-h6 element) as a child of the <article> element.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/article"
		},
		{
			name: "aside",
			category: "Text content",
			doc: "The HTML <aside> element represents a section of the page with content connected tangentially to the rest, which could be considered separate from that content.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/aside"
		},
		{
			name: "audio",
			category: "Image & multimedia",
			doc: "The HTML <audio> element is used to embed sound content in documents. It may contain several audio sources, represented using the src attribute or the <source> element; the browser will choose the most suitable one.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/audio"
		},
		{
			name: "b",
			category: "Inline text semantics",
			doc: "The HTML <b> Element represents a span of text stylistically different from normal text, without conveying any special importance or relevance. It is typically used for keywords in a summary, product names in a review, or other spans of text whose typical presentation would be boldfaced. Another example of its use is to mark the lead sentence of each paragraph of an article.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/b"
		},
		{
			name: "base",
			category: "Document metadata",
			doc: "The HTML Base Element (<base>) specifies the base URL to use for all relative URLs contained within a document. There can be only one <base> element in a document.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/base"
		},
		{
			name: "basefont",
			category: "Obsolete and deprecated elements",
			doc: "The HTML basefont element (<basefont>) establishes a default font size for a document. Font size then can be varied relative to the base font size using the <font> element.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/basefont"
		},
		{
			name: "bdi",
			category: "Inline text semantics",
			doc: "The HTML <bdi> Element (or Bi-Directional Isolation Element) isolates a span of text that might be formatted in a different direction from other text outside it.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/bdi"
		},
		{
			name: "bdo",
			category: "Inline text semantics",
			doc: "The HTML <bdo> Element (or HTML bidirectional override element) is used to override the current directionality of text. It causes the directionality of the characters to be ignored in favor of the specified directionality.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/bdo"
		},
		{
			name: "big",
			category: "Obsolete and deprecated elements",
			doc: "The HTML Big Element (<big>) makes the text font size one size bigger (for example, from small to medium, or from large to x-large) up to the browser's maximum font size.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/big"
		},
		{
			name: "blink",
			category: "Obsolete and deprecated elements",
			doc: "The HTML Blink Element (<blink>) is a non-standard element causing the enclosed text to flash slowly.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/blink"
		},
		{
			name: "blockquote",
			category: "Text content",
			doc: "The HTML <blockquote> Element (or HTML Block Quotation Element) indicates that the enclosed text is an extended quotation.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/blockquote"
		},
		{
			name: "body",
			category: "Content sectioning",
			doc: "The HTML Body Element (<body>) represents the content of an HTML document. There can be only one <body> element in a document.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/body"
		},
		{
			name: "br",
			category: "Inline text semantics",
			doc: "The HTML <br> Element (or HTML Line Break Element) produces a line break in text (carriage-return). It is useful for writing a poem or an address, where the division of lines is significant.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/br"
		},
		{
			name: "button",
			category: "Forms",
			doc: "The HTML <button> Element represents a clickable button.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/button"
		},
		{
			name: "canvas",
			category: "Scripting",
			doc: "The HTML <canvas> Element can be used to draw graphics via scripting (usually JavaScript). For example, it can be used to draw graphs, make photo compositions or even perform animations. You may (and should) provide alternate content inside the <canvas> block. That content will be rendered both on older browsers that don't support canvas and in browsers with JavaScript disabled.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/canvas"
		},
		{
			name: "caption",
			category: "Table content",
			doc: "The HTML <caption> Element (or HTML Table Caption Element) represents the title of a table. Though it is always the first descendant of a <table>, its styling, using CSS, may place it elsewhere, relative to the table.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/caption"
		},
		{
			name: "center",
			category: "Obsolete and deprecated elements",
			doc: "The HTML Center Element (<center>) is a block-level element that can contain paragraphs and other block-level and inline elements. The entire content of this element is centered horizontally within its containing element (typically, the <body>).",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/center"
		},
		{
			name: "cite",
			category: "Inline text semantics",
			doc: "The HTML Citation Element (<cite>) represents a reference to a creative work. It must include the title of a work or a URL reference, which may be in an abbreviated form according to the conventions used for the addition of citation metadata.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/cite"
		},
		{
			name: "code",
			category: "Inline text semantics",
			doc: "The HTML Code Element (<code>) represents a fragment of computer code. By default, it is displayed in the browser's default monospace font.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/code"
		},
		{
			name: "col",
			category: "Table content",
			doc: "The HTML Table Column Element (<col>) defines a column within a table and is used for defining common semantics on all common cells. It is generally found within a <colgroup> element.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/col"
		},
		{
			name: "colgroup",
			category: "Table content",
			doc: "The HTML Table Column Group Element (<colgroup>) defines a group of columns within a table.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/colgroup"
		},
		{
			name: "content",
			category: "Web Components",
			doc: "The HTML <content> element is used inside of Shadow DOM as an insertion point. It is not intended to be used in ordinary HTML. It is used with Web Components.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/content"
		},
		{
			name: "data",
			category: "Inline text semantics",
			doc: "The HTML <data> Element links a given content with a machine-readable translation. If the content is time- or date-related, the <time> must be used.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/data"
		},
		{
			name: "datalist",
			category: "Forms",
			doc: "The HTML Datalist Element (<datalist>) contains a set of <option> elements that represent the values available for other controls.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/datalist"
		},
		{
			name: "dd",
			category: "Text content",
			doc: "The HTML Description Element (<dd>) indicates the description of a term in a description list (<dl>) element. This element can occur only as a child element of a definition list and it must follow a <dt> element.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/dd"
		},
		{
			name: "del",
			category: "Edits",
			doc: "The HTML Deleted Text Element (<del>) represents a range of text that has been deleted from a document. This element is often (but need not be) rendered with strike-through text.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/del"
		},
		{
			name: "details",
			category: "Interactive elements",
			doc: "The HTML Details Element (<details>) is used as a disclosure widget from which the user can retrieve additional information.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/details"
		},
		{
			name: "dfn",
			category: "Inline text semantics",
			doc: "The HTML Definition Element (<dfn>) represents the defining instance of a term.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/dfn"
		},
		{
			name: "dialog",
			category: "Interactive elements",
			doc: "The HTML <dialog> element represents a dialog box or other interactive component, such as an inspector or window. <form> elements can be integrated within a dialog by specifying them with the attribute method=\"dialog\". When such a form is submitted, the dialog is closed with a returnValue attribute set to the value of the submit button used.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/dialog"
		},
		{
			name: "dir",
			category: "Obsolete and deprecated elements",
			doc: "The HTML directory element (<dir>) represents a directory, namely a collection of filenames.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/dir"
		},
		{
			name: "div",
			category: "Text content",
			doc: "The HTML <div> element (or HTML Document Division Element) is the generic container for flow content, which does not inherently represent anything. It can be used to group elements for styling purposes (using the class or id attributes), or because they share attribute values, such as lang. It should be used only when no other semantic element (such as <article> or <nav>) is appropriate.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/div"
		},
		{
			name: "dl",
			category: "Text content",
			doc: "The HTML <dl> Element (or HTML Description List Element) encloses a list of pairs of terms and descriptions. Common uses for this element are to implement a glossary or to display metadata (a list of key-value pairs).",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/dl"
		},
		{
			name: "dt",
			category: "Text content",
			doc: "The HTML <dt> element (or HTML Definition Term Element) identifies a term in a definition list. This element can occur only as a child element of a <dl>. It is usually followed by a <dd> element; however, multiple <dt> elements in a row indicate several terms that are all defined by the immediate next <dd> element.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/dt"
		},
		{
			name: "element",
			category: "<decorator>",
			doc: "The HTML <element> element is used to define new custom DOM elements.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/element"
		},
		{
			name: "em",
			category: "Inline text semantics",
			doc: "The HTML Emphasis Element (<em>) marks text that has stress emphasis. The <em> element can be nested, with each level of nesting indicating a greater degree of emphasis.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/em"
		},
		{
			name: "embed",
			category: "Embedded content",
			doc: "The HTML <embed> Element represents an integration point for an external application or interactive content (in other words, a plug-in).",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/embed"
		},
		{
			name: "fieldset",
			category: "Forms",
			doc: "The HTML <fieldset> element is used to group several controls as well as labels (<label>) within a web form.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/fieldset"
		},
		{
			name: "figcaption",
			category: "Text content",
			doc: "The HTML <figcaption> Element represents a caption or a legend associated with a figure or an illustration described by the rest of the data of the <figure> element which is its immediate ancestor which means <figcaption> can be the first or last element inside a <figure> block. Also, the HTML Figcaption Element is optional; if not provided, then the parent figure element will have no caption.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/figcaption"
		},
		{
			name: "figure",
			category: "Text content",
			doc: "The HTML <figure> Element represents self-contained content, frequently with a caption (<figcaption>), and is typically referenced as a single unit. While it is related to the main flow, its position is independent of the main flow. Usually this is an image, an illustration, a diagram, a code snippet, or a schema that is referenced in the main text, but that can be moved to another page or to an appendix without affecting the main flow.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/figure"
		},
		{
			name: "font",
			category: "Obsolete and deprecated elements",
			doc: "The HTML Font Element (<font>) defines the font size, color and face for its content.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/font"
		},
		{
			name: "footer",
			category: "Content sectioning",
			doc: "The HTML Footer Element (<footer>) represents a footer for its nearest sectioning content or sectioning root element. A footer typically contains information about the author of the section, copyright data or links to related documents.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/footer"
		},
		{
			name: "form",
			category: "Forms",
			doc: "The HTML <form> element represents a document section that contains interactive controls to submit information to a web server.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/form"
		},
		{
			name: "frame",
			category: "Obsolete and deprecated elements",
			doc: "<frame> is an HTML element which defines a particular area in which another HTML document can be displayed. A frame should be used within a <frameset>.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/frame"
		},
		{
			name: "frameset",
			category: "Obsolete and deprecated elements",
			doc: "<frameset> is an HTML element which is used to contain <frame> elements.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/frameset"
		},
		{
			name: "h1",
			category: "Content sectioning",
			doc: "Heading elements implement six levels of document headings, <h1> is the most important and <h6> is the least. A heading element briefly describes the topic of the section it introduces. Heading information may be used by user agents, for example, to construct a table of contents for a document automatically.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/h1"
		},
		{
			name: "h2",
			category: "Content sectioning",
			doc: "Heading elements implement six levels of document headings, <h1> is the most important and <h6> is the least. A heading element briefly describes the topic of the section it introduces. Heading information may be used by user agents, for example, to construct a table of contents for a document automatically.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/h2"
		},
		{
			name: "h3",
			category: "Content sectioning",
			doc: "Heading elements implement six levels of document headings, <h1> is the most important and <h6> is the least. A heading element briefly describes the topic of the section it introduces. Heading information may be used by user agents, for example, to construct a table of contents for a document automatically.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/h3"
		},
		{
			name: "h4",
			category: "Content sectioning",
			doc: "Heading elements implement six levels of document headings, <h1> is the most important and <h6> is the least. A heading element briefly describes the topic of the section it introduces. Heading information may be used by user agents, for example, to construct a table of contents for a document automatically.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/h4"
		},
		{
			name: "h5",
			category: "Content sectioning",
			doc: "Heading elements implement six levels of document headings, <h1> is the most important and <h6> is the least. A heading element briefly describes the topic of the section it introduces. Heading information may be used by user agents, for example, to construct a table of contents for a document automatically.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/h5"
		},
		{
			name: "h6",
			category: "Content sectioning",
			doc: "Heading elements implement six levels of document headings, <h1> is the most important and <h6> is the least. A heading element briefly describes the topic of the section it introduces. Heading information may be used by user agents, for example, to construct a table of contents for a document automatically.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/h6"
		},
		{
			name: "head",
			category: "Document metadata",
			doc: "The HTML Head Element (<head>) provides general information (metadata) about the document, including its title and links to or definitions of scripts and style sheets",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/head"
		},
		{
			name: "header",
			category: "Content sectioning",
			doc: "The HTML <header> Element represents a group of introductory or navigational aids. It may contain some heading elements but also other elements like a logo, wrapped section's header, a search form, and so on.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/header"
		},
		{
			name: "hgroup",
			category: "Content sectioning",
			doc: "The HTML <hgroup> Element (HTML Headings Group Element) represents the heading of a section. It defines a single title that participates in the outline of the document as the heading of the implicit or explicit section that it belongs to.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/hgroup"
		},
		{
			name: "hr",
			category: "Text content",
			doc: "The HTML <hr> element represents a thematic break between paragraph-level elements (for example, a change of scene in a story, or a shift of topic with a section). In previous versions of HTML, it represented a horizontal rule. It may still be displayed as a horizontal rule in visual browsers, but is now defined in semantic terms, rather than presentational terms.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/hr"
		},
		{
			name: "html",
			category: "Basic elements",
			doc: "The HTML Root Element (<html>) represents the root of an HTML document. All other elements must be descendants of this element.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/html"
		},
		{
			name: "i",
			category: "Inline text semantics",
			doc: "The HTML <i> Element represents a range of text that is set off from the normal text for some reason, for example, technical terms, foreign language phrases, or fictional character thoughts. It is typically displayed in italic type.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/i"
		},
		{
			name: "iframe",
			category: "Embedded content",
			doc: "The HTML Inline Frame Element (<iframe>) represents a nested browsing context, effectively embedding another HTML page into the current page. In HTML 4.01, a document may contain a head and a body or a head and a frame-set, but not both a body and a frame-set. However, an <iframe> can be used within a normal document body. Each browsing context has its own session history and active document. The browsing context that contains the embedded content is called the parent browsing context. The top-level browsing context (which has no parent) is typically the browser window.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/iframe"
		},
		{
			name: "img",
			category: "Image & multimedia",
			doc: "The HTML Image Element (<img>) represents an image of the document.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/img"
		},
		{
			name: "input",
			category: "Forms",
			doc: "The HTML <input> element is used to create interactive controls for web-based forms in order to accept data from the user. How an <input> works varies considerably depending on the value of its type attribute.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/input"
		},
		{
			name: "ins",
			category: "Edits",
			doc: "The HTML <ins> Element (or HTML Inserted Text) HTML represents a range of text that has been added to a document.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/ins"
		},
		{
			name: "isindex",
			category: "Obsolete and deprecated elements",
			doc: "<isindex> is an HTML element which is used for putting a text field in the document for querying document. <isindex> is intented to use inside of <head> element by W3C, however browsers provide support wherever it is used in the document.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/isindex"
		},
		{
			name: "kbd",
			category: "Inline text semantics",
			doc: "The HTML Keyboard Input Element (<kbd>) represents user input and produces an inline element displayed in the browser's default monospace font.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/kbd"
		},
		{
			name: "keygen",
			category: "Forms",
			doc: "The HTML <keygen> element exists to facilitate generation of key material, and submission of the public key as part of an HTML form. This mechanism is designed for use with Web-based certificate management systems. It is expected that the <keygen> element will be used in an HTML form along with other information needed to construct a certificate request, and that the result of the process will be a signed certificate.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/keygen"
		},
		{
			name: "label",
			category: "Forms",
			doc: "The HTML Label Element (<label>) represents a caption for an item in a user interface. It can be associated with a control either by placing the control element inside the <label> element, or by using the for attribute. Such a control is called the labeled control of the label element.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/label"
		},
		{
			name: "legend",
			category: "Forms",
			doc: "The HTML <legend> Element (or HTML Legend Field Element) represents a caption for the content of its parent <fieldset>.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/legend"
		},
		{
			name: "li",
			category: "Text content",
			doc: "The HTML List Item Element (<li>) is used to represent an item in a list. It must be contained in a parent element: an ordered list (<ol>), an unordered list (<ul>), or a menu (<menu>). In menus and unordered lists, list items are usually displayed using bullet points. In ordered lists, they are usually displayed with an ascending counter on the left, such as a number or letter.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/li"
		},
		{
			name: "link",
			category: "Document metadata",
			doc: "The HTML Link Element (<link>) specifies relationships between the current document and an external resource. Possible uses for this element include defining a relational framework for navigation. This Element is most used to link to style sheets.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/link"
		},
		{
			name: "listing",
			category: "Obsolete and deprecated elements",
			doc: "The HTML Listing Element (<listing>) renders text between the start and end tags without interpreting the HTML in between and using a monospaced font. The HTML 2 standard recommended that lines shouldn't be broken when not greater than 132 characters.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/listing"
		},
		{
			name: "main",
			category: "Text content",
			doc: "The HTML Main Element (<main>) can be used as a container for the dominant contents of another element. The main content area consists of content that is directly related to, or expands upon the central topic of a section or the central functionality of an application. This content should be unique to the document, excluding any content that is repeated across a set of documents such as sidebars, navigation links, copyright information, site logos, and search forms (unless, of course, the document's main function is as a search form). Unlike <article> and <section>, this element does not contribute to the document outline.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/main"
		},
		{
			name: "map",
			category: "Image & multimedia",
			doc: "The HTML <map> element is used with <area> elements to define an image map (a clickable link area).",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/map"
		},
		{
			name: "mark",
			category: "Inline text semantics",
			doc: "The HTML Mark Element (<mark>) represents highlighted text, i.e., a run of text marked for reference purpose, due to its relevance in a particular context. For example it can be used in a page showing search results to highlight every instance of the searched-for word.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/mark"
		},
				{
			name: "marquee",
			category: "Obsolete and deprecated elements",
			doc: "The HTML <marquee> element is used to insert a scrolling area of text.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/marquee"
		},
		{
			name: "menu",
			category: "Interactive elements",
			doc: "The HTML <menu> element represents a group of commands that a user can perform or activate. This includes both list menus, which might appear across the top of a screen, as well as context menus, such as those that might appear underneath a button after it has been clicked.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/menu"
		},
		{
			name: "menuitem",
			category: "Interactive elements",
			doc: "The HTML <menuitem> element represents a command that a user is able to invoke through a popup menu. This includes context menus, as well as menus that might be attached to a menu button.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/menuitem"
		},
		{
			name: "meta",
			category: "Document metadata",
			doc: "The HTML Meta Element (<meta>) represents any metadata information that cannot be represented by one of the other HTML meta-related elements (<base>, <link>, <script>, <style> or <title>).",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/meta"
		},
		{
			name: "meter",
			category: "Forms",
			doc: "The HTML <meter> Element represents either a scalar value within a known range or a fractional value.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/meter"
		},
		{
			name: "nav",
			category: "Content sectioning",
			doc: "The HTML Navigation Element (<nav>) represents a section of a page that links to other pages or to parts within the page: a section with navigation links.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/nav"
		},
		{
			name: "noembed",
			category: "Obsolete and deprecated elements",
			doc: "The <noembed> element is a deprecated and non-standard way to provide alternative, or \"fallback\", content for browsers that do not support the <embed> element or do not support embedded content an author wishes to use.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/noembed"
		},
		{
			name: "noframes",
			category: "Embedded content",
			doc: "<noframes> is an HTML element which is used to supporting browsers which are not able to support <frame> elements or configured to do so.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/noframes"
		},
		{
			name: "noscript",
			category: "Scripting",
			doc: "The HTML <noscript> Element defines a section of html to be inserted if a script type on the page is unsupported or if scripting is currently turned off in the browser.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/noscript"
		},
		{
			name: "object",
			category: "Embedded content",
			doc: "The HTML Embedded Object Element (<object>) represents an external resource, which can be treated as an image, a nested browsing context, or a resource to be handled by a plugin.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/object"
		},
		{
			name: "ol",
			category: "Text content",
			doc: "The HTML <ol> Element (or HTML Ordered List Element) represents an ordered list of items. Typically, ordered-list items are displayed with a preceding numbering, which can be of any form, like numerals, letters or Romans numerals or even simple bullets. This numbered style is not defined in the HTML description of the page, but in its associated CSS, using the list-style-type property.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/ol"
		},
		{
			name: "optgroup",
			category: "Forms",
			doc: "In a Web form, the HTML <optgroup> element  creates a grouping of options within a <select> element.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/optgroup"
		},
		{
			name: "option",
			category: "Forms",
			doc: "In a Web form, the HTML <option> element is used to create a control representing an item within a <select>, an <optgroup> or a <datalist> HTML5 element.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/option"
		},
		{
			name: "output",
			category: "Forms",
			doc: "The HTML <output> element represents the result of a calculation or user action.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/output"
		},
		{
			name: "p",
			category: "Text content",
			doc: "The HTML <p> element (or HTML Paragraph Element) represents a paragraph of text.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/p"
		},
		{
			name: "param",
			category: "Embedded content",
			doc: "The HTML <param> Element (or HTML Parameter Element) defines parameters for <object>.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/param"
		},
		{
			name: "plaintext",
			category: "Obsolete and deprecated elements",
			doc: "The HTML Plaintext Element (<plaintext>) renders everything following the start tag as raw text, without interpreting any HTML. There is no closing tag, since everything after it is considered raw text.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/plaintext"
		},
		{
			name: "pre",
			category: "Text content",
			doc: "The HTML Preformatted Text (<pre>) represents preformatted text. Text within this element is typically displayed in a non-proportional font exactly as it is laid out in the file. Whitespaces inside this element are displayed as typed.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/pre"
		},
		{
			name: "progress",
			category: "Forms",
			doc: "The HTML <progress> Element is used to view the completion progress of a task. While the specifics of how it's displayed is left up to the browser developer, it's typically displayed as a progress bar. Javascript can be used to manipulate the value of progress bar.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/progress"
		},
		{
			name: "q",
			category: "Inline text semantics",
			doc: "The HTML Quote Element (<q>) indicates that the enclosed text is a short inline quotation. This element is intended for short quotations that don't require paragraph breaks; for long quotations use <blockquote> element.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/q"
		},
		{
			name: "rp",
			category: "Inline text semantics",
			doc: "The HTML <rp> element is used to provide fall-back parenthesis for browsers non-supporting ruby annotations. Ruby annotations are for showing pronounciation of East Asian characters, like using Japanese furigana or Taiwainese bopomofo characters. The <rp> element is used in the case of lack of <ruby> element support its content has what should be displayed in order to indicate the presence of a ruby annotation, usually parentheses.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/rp"
		},
		{
			name: "rt",
			category: "Inline text semantics",
			doc: "Editorial review completed.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/rt"
		},
		{
			name: "rtc",
			category: "Inline text semantics",
			doc: "The HTML <rtc> Element embraces semantic annotations of characters presented in a ruby of <rb> elements used inside of <ruby> element. <rb> elements can have both pronunciation (<rt> and semantic (<rtc>) annotations.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/rtc"
		},
		{
			name: "ruby",
			category: "Inline text semantics",
			doc: "The HTML <ruby> Element represents a ruby annotation. Ruby annotations are for showing pronunciation of East Asian characters.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/ruby"
		},
		{
			name: "s",
			category: "Inline text semantics",
			doc: "The HTML Strikethrough Element (<s>) renders text with a strikethrough, or a line through it. Use the <s> element to represent things that are no longer relevant or no longer accurate. However, <s> is not appropriate when indicating document edits; for that, use the <del> and <ins> elements, as appropriate.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/s"
		},
		{
			name: "samp",
			category: "Inline text semantics",
			doc: "The HTML <samp> element is an element intended to identify sample output from a computer program. It is usually displayed in the browser's default monotype font (such as Lucida Console).",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/samp"
		},
		{
			name: "script",
			category: "Scripting",
			doc: "The HTML Script Element (<script>) is used to embed or reference an executable script within an HTML or XHTML document.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/script"
		},
		{
			name: "section",
			category: "Content sectioning",
			doc: "The HTML Section Element (<section>) represents a generic section of a document, i.e., a thematic grouping of content, typically with a heading. Each <section> should be identified, typically by including a heading (<h1>-<h6> element) as a child of the <section> element.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/section"
		},
		{
			name: "select",
			category: "Forms",
			doc: "The HTML select (<select>) element represents a control that presents a menu of options. The options within the menu are represented by <option> elements, which can be grouped by <optgroup> elements. Options can be pre-selected for the user.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/select"
		},
		{
			name: "shadow",
			category: "<decorator>",
			doc: "The HTML <shadow> element is used as a shadow DOM insertion point. You might use it if you have created multiple shadow roots under a shadow host. It is not useful in ordinary HTML. It is used with Web Components.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/shadow"
		},
		{
			name: "small",
			category: "Inline text semantics",
			doc: "The HTML Small Element (<small>) makes the text font size one size smaller (for example, from large to medium, or from small to x-small) down to the browser's minimum font size.  In HTML5, this element is repurposed to represent side-comments and small print, including copyright and legal text, independent of its styled presentation.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/small"
		},
		{
			name: "source",
			category: "Embedded content",
			doc: "The HTML <source> element is used to specify multiple media resources for <picture>, <audio> and <video> elements. It is an empty element. It is commonly used to serve the same media in multiple formats supported by different browsers.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/source"
		},
		{
			name: "spacer",
			category: "Obsolete and deprecated elements",
			doc: "<spacer> is an HTML element which is used for inserting white spaces to web pages. It was created by NetScape for achieving same effect as a single-pixel layout GIF image, which was something web designers used to use to add white spaces to web pages, without actually using a GIF. However <spacer> is not supported by any major browser and same effects can be created with various CSS rules. In Mozilla applications, support for this element was removed in Gecko 2.0. Therefore usage of <spacer> is unnecessary.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/spacer"
		},
		{
			name: "span",
			category: "Inline text semantics",
			doc: "The HTML <span> element is a generic inline container for phrasing content, which does not inherently represent anything. It can be used to group elements for styling purposes (using the class or id attributes), or because they share attribute values, such as lang. It should be used only when no other semantic element is appropriate. <span> is very much like a <div> element, but <div> is a block-level element whereas a <span> is an inline element.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/span"
		},
		{
			name: "strike",
			category: "Obsolete and deprecated elements",
			doc: "The HTML Strikethrough Element (<strike>) renders text with a strikethrough, or a line through it.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/strike"
		},
		{
			name: "strong",
			category: "Inline text semantics",
			doc: "The HTML Strong Element (<strong>) gives text strong importance, and is typically displayed in bold.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/strong"
		},
		{
			name: "style",
			category: "Document metadata",
			doc: "The HTML Style Element (<style>) contains style information for a document, or part of a document. By default, the style instructions written inside that element are expected to be CSS.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/style"
		},
		{
			name: "sub",
			category: "Inline text semantics",
			doc: "The HTML Subscript Element (<sub>) defines a span of text that should be displayed, for typographic reasons, lower, and often smaller, than the main span of text.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/sub"
		},
		{
			name: "summary",
			category: "Interactive elements",
			doc: "The HTML summary element (<summary>) is used as a summary, caption, or legend for the content of a <details> element.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/summary"
		},
		{
			name: "sup",
			category: "Inline text semantics",
			doc: "The HTML Superscript Element (<sup>) defines a span of text that should be displayed, for typographic reasons, higher, and often smaller, than the main span of text.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/sup"
		},
		{
			name: "table",
			category: "Table content",
			doc: "The HTML Table Element (<table>) represents data in two dimensions or more.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/table"
		},
		{
			name: "tbody",
			category: "Table content",
			doc: "The HTML Table Body Element (",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/tbody"
		},
		{
			name: "td",
			category: "Table content",
			doc: "The Table cell HTML element (<td>) defines a cell of a table that contains data. It participates in the table model.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/td"
		},
		{
			name: "template",
			category: "<decorator>",
			doc: "The HTML template element <template> is a mechanism for holding client-side content that is not to be rendered when a page is loaded but may subsequently be instantiated during runtime using JavaScript.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/template"
		},
		{
			name: "textarea",
			category: "Forms",
			doc: "The HTML <textarea> element represents a multi-line plain-text editing control.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/textarea"
		},
		{
			name: "tfoot",
			category: "Table content",
			doc: "The HTML Table Foot Element (<tfoot>) defines a set of rows summarizing the columns of the table.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/tfoot"
		},
		{
			name: "th",
			category: "Table content",
			doc: "The HTML Table Header Cell Element (<th>) defines a cell that is a header for a group of cells of a table. The group of cells that the header refers to is defined by the scope and headers attribute.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/th"
		},
		{
			name: "thead",
			category: "Table content",
			doc: "The HTML Table Head Element (<thead>) defines a set of rows defining the head of the columns of the table.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/thead"
		},
		{
			name: "time",
			category: "Inline text semantics",
			doc: "The HTML <time> element represents either a time on a 24-hour clock or a precise date in the Gregorian calendar (with optional time and timezone information).",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/time"
		},
		{
			name: "title",
			category: "Document metadata",
			doc: "The HTML Title Element (<title>) defines the title of the document, shown in a browser's title bar or on the page's tab. It can only contain text and any contained tags are not interpreted.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/title"
		},
		{
			name: "tr",
			category: "Table content",
			doc: "The HTML Table Row Element (<tr>) defines a row of cells in a table. Those can be a mix of <td> and <th> elements.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/tr"
		},
		{
			name: "track",
			category: "Image & multimedia",
			doc: "The HTML <track> element is used as a child of the media elements—<audio> and <video>. It lets you specify timed text tracks (or time-based data), for example to automatically handle subtitles. The tracks are formatted in WebVTT format (.vtt files) — Web Video Text Tracks.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/track"
		},
		{
			name: "tt",
			category: "Obsolete and deprecated elements",
			doc: "The HTML Teletype Text Element (<tt>) produces an inline element displayed in the browser's default monotype font. This element was intended to style text as it would display on a fixed width display, such as a teletype. It probably is more common to display fixed width type using the <code> element.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/tt"
		},
		{
			name: "u",
			category: "Inline text semantics",
			doc: "The HTML Underline Element (<u>) renders text with an underline, a line under the baseline of its content.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/u"
		},
		{
			name: "ul",
			category: "Text content",
			doc: "The HTML unordered list element (<ul>) represents an unordered list of items, namely a collection of items that do not have a numerical ordering, and their order in the list is meaningless. Typically, unordered-list items are displayed with a bullet, which can be of several forms, like a dot, a circle or a squared. The bullet style is not defined in the HTML description of the page, but in its associated CSS, using the list-style-type property.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/ul"
		},
		{
			name: "var",
			category: "Inline text semantics",
			doc: "The HTML Variable Element (<var>) represents a variable in a mathematical expression or a programming context.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/var"
		},
		{
			name: "video",
			category: "Image & multimedia",
			doc: "The HTML <video> element is used to embed video content. It may contain several video sources, represented using the src attribute or the <source> element; the browser will choose the most suitable one.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/video"
		},
		{
			name: "wbr",
			category: "Inline text semantics",
			doc: "The Word Break Opportunity (<wbr>) HTML element represents a position within text where the browser may optionally break a line, though its line-breaking rules would not otherwise create a break at that location.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/wbr"
		},
		{
			name: "xmp",
			category: "Obsolete and deprecated elements",
			doc: "The HTML Example Element (<xmp>) renders text between the start and end tags without interpreting the HTML in between and using a monospaced font. The HTML2 specification recommended that it should be rendered wide enough to allow 80 characters per line.",
			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/xmp"
		}
	];
	
	return {
		tagTemplates: tagTemplates
	};
});
/**
 * @license RequireJS i18n 2.0.4 Copyright (c) 2010-2012, The Dojo Foundation All Rights Reserved.
 * Available via the MIT or new BSD license.
 * see: http://github.com/requirejs/i18n for details
 */
/*jslint regexp: true */
/*global require: false, navigator: false, define: false */

/**
 * This plugin handles i18n! prefixed modules. It does the following:
 *
 * 1) A regular module can have a dependency on an i18n bundle, but the regular
 * module does not want to specify what locale to load. So it just specifies
 * the top-level bundle, like "i18n!nls/colors".
 *
 * This plugin will load the i18n bundle at nls/colors, see that it is a root/master
 * bundle since it does not have a locale in its name. It will then try to find
 * the best match locale available in that master bundle, then request all the
 * locale pieces for that best match locale. For instance, if the locale is "en-us",
 * then the plugin will ask for the "en-us", "en" and "root" bundles to be loaded
 * (but only if they are specified on the master bundle).
 *
 * Once all the bundles for the locale pieces load, then it mixes in all those
 * locale pieces into each other, then finally sets the context.defined value
 * for the nls/colors bundle to be that mixed in locale.
 *
 * 2) A regular module specifies a specific locale to load. For instance,
 * i18n!nls/fr-fr/colors. In this case, the plugin needs to load the master bundle
 * first, at nls/colors, then figure out what the best match locale is for fr-fr,
 * since maybe only fr or just root is defined for that locale. Once that best
 * fit is found, all of its locale pieces need to have their bundles loaded.
 *
 * Once all the bundles for the locale pieces load, then it mixes in all those
 * locale pieces into each other, then finally sets the context.defined value
 * for the nls/fr-fr/colors bundle to be that mixed in locale.
 */
(function () {
    'use strict';

    //regexp for reconstructing the master bundle name from parts of the regexp match
    //nlsRegExp.exec("foo/bar/baz/nls/en-ca/foo") gives:
    //["foo/bar/baz/nls/en-ca/foo", "foo/bar/baz/nls/", "/", "/", "en-ca", "foo"]
    //nlsRegExp.exec("foo/bar/baz/nls/foo") gives:
    //["foo/bar/baz/nls/foo", "foo/bar/baz/nls/", "/", "/", "foo", ""]
    //so, if match[5] is blank, it means this is the top bundle definition.
    var nlsRegExp = /(^.*(^|\/)nls(\/|$))([^\/]*)\/?([^\/]*)/;

    //Helper function to avoid repeating code. Lots of arguments in the
    //desire to stay functional and support RequireJS contexts without having
    //to know about the RequireJS contexts.
    function addPart(locale, master, needed, toLoad, prefix, suffix) {
        if (master[locale]) {
            needed.push(locale);
            if (master[locale] === true || master[locale] === 1) {
                toLoad.push(prefix + locale + '/' + suffix);
            }
        }
    }

    function addIfExists(req, locale, toLoad, prefix, suffix) {
        var fullName = prefix + locale + '/' + suffix;
        if (require._fileExists(req.toUrl(fullName + '.js'))) {
            toLoad.push(fullName);
        }
    }

    /**
     * Simple function to mix in properties from source into target,
     * but only if target does not already have a property of the same name.
     * This is not robust in IE for transferring methods that match
     * Object.prototype names, but the uses of mixin here seem unlikely to
     * trigger a problem related to that.
     */
    function mixin(target, source, force) {
        var prop;
        for (prop in source) {
            if (source.hasOwnProperty(prop) && (!target.hasOwnProperty(prop) || force)) {
                target[prop] = source[prop];
            } else if (typeof source[prop] === 'object') {
                if (!target[prop] && source[prop]) {
                    target[prop] = {};
                }
                mixin(target[prop], source[prop], force);
            }
        }
    }

    define('i18n',['module'], function (module) {
        var masterConfig = module.config ? module.config() : {};
        masterConfig = masterConfig || {};

        return {
            version: '2.0.4',
            /**
             * Called when a dependency needs to be loaded.
             */
            load: function (name, req, onLoad, config) {
                config = config || {};

                if (config.locale) {
                    masterConfig.locale = config.locale;
                }

                var masterName,
                    match = nlsRegExp.exec(name),
                    prefix = match[1],
                    locale = match[4],
                    suffix = match[5],
                    parts = locale.split("-"),
                    toLoad = [],
                    value = {},
                    i, part, current = "";

                //If match[5] is blank, it means this is the top bundle definition,
                //so it does not have to be handled. Locale-specific requests
                //will have a match[4] value but no match[5]
                if (match[5]) {
                    //locale-specific bundle
                    prefix = match[1];
                    masterName = prefix + suffix;
                } else {
                    //Top-level bundle.
                    masterName = name;
                    suffix = match[4];
                    locale = masterConfig.locale;
                    if (!locale) {
                        locale = masterConfig.locale =
                            typeof navigator === "undefined" ? "root" :
                            (navigator.language ||
                             navigator.userLanguage || "root").toLowerCase();
                    }
                    parts = locale.split("-");
                }

                if (config.isBuild) {
                    //Check for existence of all locale possible files and
                    //require them if exist.
                    toLoad.push(masterName);
                    addIfExists(req, "root", toLoad, prefix, suffix);
                    for (i = 0; i < parts.length; i++) {
                        part = parts[i];
                        current += (current ? "-" : "") + part;
                        addIfExists(req, current, toLoad, prefix, suffix);
                    }
                                        
                    if(config.locales) {
                    	var j, k; 
                    	for (j = 0; j < config.locales.length; j++) {
                    		locale = config.locales[j];
                    		parts = locale.split("-");
                    		current = "";
	                    	for (k = 0; k < parts.length; k++) {
		                        part = parts[k];
		                        current += (current ? "-" : "") + part;
		                        addIfExists(req, current, toLoad, prefix, suffix);
	                    	}
                    	}
                    }

                    req(toLoad, function () {
                        onLoad();
                    });
                } else {
                    //First, fetch the master bundle, it knows what locales are available.
                    req([masterName], function (master) {
                        //Figure out the best fit
                        var needed = [],
                            part;

                        //Always allow for root, then do the rest of the locale parts.
                        addPart("root", master, needed, toLoad, prefix, suffix);
                        for (i = 0; i < parts.length; i++) {
                            part = parts[i];
                            current += (current ? "-" : "") + part;
                            addPart(current, master, needed, toLoad, prefix, suffix);
                        }

                        //Load all the parts missing.
                        req(toLoad, function () {
                            var i, partBundle, part;
                            for (i = needed.length - 1; i > -1 && needed[i]; i--) {
                                part = needed[i];
                                partBundle = master[part];
                                if (partBundle === true || partBundle === 1) {
                                    partBundle = req(prefix + part + '/' + suffix);
                                }
                                mixin(value, partBundle);
                            }

                            //All done, notify the loader.
                            onLoad(value);
                        });
                    });
                }
            }
        };
    });
}());

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 * 
 ******************************************************************************/
/* eslint-env amd */
define('webtools/nls/messages',{
	root:true
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014, 2015 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 * 
 ******************************************************************************/
/* eslint-env amd */
define('webtools/nls/root/messages',{//Default message bundle
	'htmlOutline' : 'HTML Outline',   
	'htmlHover' : 'HTML Hover',   
	'htmlContentAssist' : 'HTML Content Assist',   
	'css' : 'CSS Validation',
	'cssOutline' : 'CSS Rule Outline',   
	'cssContentAssist' : 'CSS Content Assist',   
	'cssHover' : 'CSS Hover',   
	'csslintValidator' : 'CSS Validator',   
	'pluginName': 'Orion Web Tools Support',   
	'pluginDescription': 'This plug-in provides web language tools support for Orion, including HTML and CSS.',   
	'fontHoverExampleText': 'Lorem ipsum dolor...',   
	
	// Validator Severities
	'ignore' : 'Ignore',   
	'warning' : 'Warning',   
	'error' : 'Error',   
	
	// CSS Validator Settings
	'adjoining-classes': 'Disallow adjoining classes:',   
	'box-model': 'Beware of broken box size:',   
	'box-sizing': 'Disallow use of box-sizing:',   
	'bulletproof-font-face': 'Use the bulletproof @font-face syntax:',   
	'compatible-vendor-prefixes': 'Require compatible vendor prefixes:',   
	'display-property-grouping': 'Require properties appropriate for display:',   
	'duplicate-background-images': 'Disallow duplicate background images:',   
	'duplicate-properties': 'Disallow duplicate properties:',   
	'empty-rules': 'Disallow empty rules:',   
	'fallback-colors': 'Require fallback colors:',   
	'floats': 'Disallow too many floats:',   
	'font-faces': 'Don\'t use too many web fonts:',   
	'font-sizes': 'Disallow too many font sizes:',   
	'gradients': 'Require all gradient definitions:',   
	'ids': 'Disallow IDs in selectors:',   
	'import': 'Disallow @import:',   
	'important': 'Disallow !important:',   
	'known-properties': 'Require use of known properties:',   
	'outline-none': 'Disallow outline: none:',   
	'overqualified-elements': 'Disallow overqualified elements:',   
	'qualified-headings': 'Disallow qualified headings:',   
	'regex-selectors': 'Disallow selectors that look like regexs:',   
	'rules-count': 'Rules Count:',   
	'selector-max-approaching': 'Warn when approaching the 4095 selector limit for IE:',   
	'selector-max': 'Error when past the 4095 selector limit for IE:',   
	'shorthand': 'Require shorthand properties:',   
	'star-property-hack': 'Disallow properties with a star prefix:',   
	'text-indent': 'Disallow negative text-indent:',   
	'underscore-property-hack': 'Disallow properties with an underscore prefix:',   
	'unique-headings': 'Headings should only be defined once:',   
	'universal-selector': 'Disallow universal selector:',   
	'unqualified-attributes': 'Disallow unqualified attribute selectors:',   
	'vendor-prefix': 'Require standard property with vendor prefix:',   
	'zero-units': 'Disallow units for 0 values:',   
	
	// CSS Quick Fixes
	'quickfix-empty-rules': 'Remove the rule.',   
	'quickfix-important': 'Remove \'!important\' annotation.',   
	'quickfix-zero-units': 'Remove \'px\' qualifier.',   
	
	//HTML content assist
	'simpleDocDescription': 'Simple HTML document',
	'onlineDocumentation': '\n\n[Online documentation](${0})',
	'closeTagDescription': ' - Close the ${0} tag',
	'openCommentName': 'Open comment',
	'closeCommentName': 'Close comment',
	'obsoleteTag': '*Obsolete: This tag is obsolete. Its use is discouraged since it may not work in some browsers.*\n\n',
	'obsoleteTagDesc': ' [Obsolete]',
	'obsoleteAttr': '*Obsolete: This attribute is obsolete since ${0}. Its use is discouraged.*\n\n',
	'obsoleteAttrDesc': ' [Obsolete]',
	'possibleValues': ' Possible values are:',
	'valueNameDocMarkdown': '\n* ${0}: ${1}',
	'formeventsHeader': 'Form Events',
	'keyboardeventsHeader': 'Keyboard Events',
	'mouseeventsHeader': 'Mouse Events',
	'windoweventsHeader': 'Window Events',
	'addQuotesToAttributes': ' - Add quotes to the current attribute',
	
	//CSS content assist
	'ruleTemplateDescription': 'rule - class selector rule',
	'idSelectorTemplateDescription': 'rule - id selector rule',
	'outlineStyleTemplateDescription': 'outline - outline style',
	'backgroundImageTemplateDescription': 'background-image - image style',
	'urlImageTemplateDescription': 'url - url image',
	'rgbColourTemplateDescription': 'rgb - rgb color',
	'importTemplateDescription': 'import - import style sheet',
	'csslintTemplateDescription': 'csslint - add embedded rule severity',
	'keywordsAssistTitle': 'Keywords',
	'templateAssistHeader': 'Templates',
	'templateHoverHeader': 'Template source code:\n\n'
});


/*******************************************************************************
 * @license
 * Copyright (c) 2012, 2015 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors: IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
/*global requirejs*/
define('orion/i18nUtil',[], function() {
	/**
	 * Performs string substitution. Can be invoked in 2 ways:
	 *
	 * i) vargs giving numbered substition values:
	 *   formatMessage("${0} is ${1}", "foo", "bar")  // "foo is bar"
	 *
	 * ii) a map giving the substitutions:
	 *   formatMessage("${thing} is ${1}", {1: "bar", thing: "foo"})  // "foo is bar"
	 */
	function formatMessage(msg) {
		var pattern = /\$\{([^\}]+)\}/g, args = arguments;
		if (args.length === 2 && args[1] && typeof args[1] === "object") {
			return msg.replace(pattern, function(str, key) {
				return args[1][key];
			});
		}
		return msg.replace(pattern, function(str, index) {
			return args[(index << 0) + 1];
		});
	}
	return {
		formatMessage: formatMessage
	};
});
/*******************************************************************************
 * @license
 * Copyright (c) 2011, 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env amd */
define('webtools/htmlContentAssist',[
	'orion/editor/templates',
	'orion/objects',
	'webtools/util',
	'javascript/util',
	'webtools/attributes',
	'webtools/deprecatedAttributes',
	'webtools/tags',
	'i18n!webtools/nls/messages',
	'orion/i18nUtil'
], function(mTemplates, Objects, util, jsUtil, Attributes, Deprecated, Tags, Messages, i18nUtil) {

	var simpleDocTemplate = new mTemplates.Template("", Messages['simpleDocDescription'],
		"<!DOCTYPE html>\n" + //$NON-NLS-0$
		"<html lang=\"en\">\n" + //$NON-NLS-0$
		"<head>\n" + //$NON-NLS-0$
		"\t<meta charset=\"utf-8\">\n" + //$NON-NLS-0$
		"\t<title>${title}</title>\n" + //$NON-NLS-0$
		"</head>\n" + //$NON-NLS-0$
		"<body>\n" + //$NON-NLS-0$
		"\t<h1>${header}</h1>\n" + //$NON-NLS-0$
		"\t<p>\n" + //$NON-NLS-0$
		"\t\t${cursor}\n" + //$NON-NLS-0$
		"\t</p>\n" + //$NON-NLS-0$
		"</body>\n" + //$NON-NLS-0$
		"</html>"); //$NON-NLS-0$
		
//	var templates = [
//		{
//			tag: 'img',
//			prefix: "<img",
//			name: "<img>",
//			description: Messages['imageElementDescription'],
//			template: "<img src=\"${URI}\" alt=\"${Image}\"/>",
//			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/img"
//		},
//		{
//			tag: 'a',
//			prefix: "<a",
//			name: "<a>",
//			description: Messages['anchorElementDescription'],
//			template: "<a href=\"${cursor}\"></a>",
//			url: "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/a"
//		},
//		{
//			tag: 'ul',
//			prefix: "<ul",
//			name: "<ul>",
//			description: Messages['ulElementDescription'],
//			template: "<ul>\n\t<li>${cursor}</li>\n</ul>",
//			url: 'https://developer.mozilla.org/en/docs/Web/HTML/Element/ul'
//		},
//		{
//			tag: 'ol',
//			prefix: "<ol",
//			name: "<ol>",
//			description: Messages['olElementDescription'],
//			template: "<ol>\n\t<li>${cursor}</li>\n</ol>",
//			url: 'https://developer.mozilla.org/en/docs/Web/HTML/Element/ol'
//		},
//		{
//			tag: 'dl',
//			prefix: "<dl",
//			name: "<dl>",
//			description: Messages['dlElementDescription'],
//			template: "<dl>\n\t<dt>${cursor}</dt>\n\t<dd></dd>\n</dl>",
//			url: 'https://developer.mozilla.org/en/docs/Web/HTML/Element/dl'
//		},
//		{
//			tag: 'table',
//			prefix: "<table",
//			name: "<table>",
//			description: Messages['basicTableDescription'],
//			template: "<table>\n\t<tr>\n\t\t<td>${cursor}</td>\n\t</tr>\n</table>",
//			url: 'https://developer.mozilla.org/en/docs/Web/HTML/Element/table'
//		},
//	];
//
//	//elements that are typically placed on a single line (e.g., <b>, <h1>, etc)
//	var element, template, description, i;
//	var singleLineElements = [
//		"abbr","b","button","canvas","cite",
//		"command","dd","del","dfn","dt",
//		"em","embed","font","h1","h2",
//		"h3","h4","h5","h6","i",
//		"ins","kbd","label","li","mark",
//		"meter","object","option","output","progress",
//		"q","rp","rt","samp","small",
//		"strong","sub","sup","td","time",
//		"title","tt","u","var"
//	];
//	for (i = 0; i < singleLineElements.length; i++) {
//		element = singleLineElements[i];
//		description = "<" + element + "></" + element + ">";
//		template = "<" + element + ">${cursor}</" + element + ">";
//		templates.push({prefix: "<" + element, description: description, template: template});
//	}
//
//	//elements that typically start a block spanning multiple lines (e.g., <p>, <div>, etc)
//	var multiLineElements = [
//		"address","article","aside","audio","bdo",
//		"blockquote","body","caption","code","colgroup",
//		"datalist","details","div","fieldset","figure",
//		"footer","form","head","header","hgroup",
//		"iframe","legend","map","menu","nav",
//		"noframes","noscript","optgroup","p","pre",
//		"ruby","script","section","select","span",
//		"style","tbody","textarea","tfoot","th",
//		"thead","tr","video"
//	];
//	for (i = 0; i < multiLineElements.length; i++) {
//		element = multiLineElements[i];
//		description = "<" + element + "></" + element + ">";
//		template = "<" + element + ">\n\t${cursor}\n</" + element + ">";
//		templates.push({prefix: "<" + element, description: description, template: template});
//	}
//
//	//elements with no closing element (e.g., <hr>, <br>, etc)
//	var emptyElements = [
//		"area","base","br","col",
//		"hr","input","link","meta",
//		"param","keygen","source"
//	];
//	for (i = 0; i < emptyElements.length; i++) {
//		element = emptyElements[i];
//		template = description = "<" + element + "/>";
//		templates.push({prefix: "<" + element, description: description, template: template});
//	}

	/**
	 * @name orion.editor.HTMLContentAssistProvider
	 * @description Creates a new HTML content assist provider
	 * @class Provides content assist for HTML.
	 * @param {HtmlAstManager} htmlAstManager The backing AST manager
	 */
	function HTMLContentAssistProvider(htmlAstManager) {
		this.astmanager = htmlAstManager;
	}
	
	HTMLContentAssistProvider.prototype = new mTemplates.TemplateContentAssist([], []);
	
	Objects.mixin(HTMLContentAssistProvider.prototype, {
		/**
		 * @private
		 */
		_getPrefixStart: function(text, offset) {
			var index = offset;
			while (index > 0) {
				var char = text.substring(index - 1, index);
				if (/[A-Za-z0-9_-]/.test(char)) {
					index--;
				} else {
					break;
				}
			}
			return index;
		},
		/**
		 * @callback 
		 */
		computePrefix: function(editorContext, offset) {
			var that = this;
			return editorContext.getText().then(function (text) {
				return text.substring(that._getPrefixStart(text, offset), offset);
			});
		},
		/**
		 * @callback 
		 */
		computeContentAssist: function(editorContext, params) {
			var that = this;
			return this.astmanager.getAST(editorContext).then(function(ast) {
				// template - simple html document
				if (ast.source.length === 0) {
					return [simpleDocTemplate.getProposal("", params.offset, params)];
				}
				var proposals = that.computeProposalsFromAst(ast, params);
				if(proposals.length > 0 && proposals[0].unselectable) {
					//already sorted, only templates. hack until we break out template computation
					return proposals;
				}
				return proposals;
			});
		},
		/**
		 * Computes the completions from the given AST and parameter context
		 * @param {Object} ast The AST to inspect
		 * @param {Object} params The paramter context
		 * @returns {Array.<Object>} The array of proposal objects or an empty array, never null
		 * @since 10.0 
		 */
		computeProposalsFromAst: function(ast, params) {
			var node = util.findNodeAtOffset(ast, params.offset);
			if(node) {
				if(this.inScriptOrStyle(node, params.offset, ast.source) || this.inClosingTag(node, params.offset, ast.source)) {
					return [];
				}
				if (this.isCompletingCommentClose(node, params.offset)){
					return this.getComment(node, params.offset, ast.source, false);
				} else if (this.isCompletingAttributeValue(node, ast.source, params)) {
					return this.getValuesForAttribute(node, ast.source, params);
				} else if (this.isCompletingTagAttribute(node, ast.source, params)) {
					return this.getAttributesForNode(node, ast.source, params);
				} else if (this.isCompletingTag(node, params)){
					if (this.isCompletingCommentOpen(node)){
						return this.getComment(node, params.offset, ast.source, true);
					}
					if (this.isCompletingTagWithMatchingClose(node, ast.source)){
						return [];
					}
					return this.getTags(ast.source, params);
				}
				return this.getProposalsForTextContent(node, ast.source, params);
			}
			
			// If we have a non-empty source file with no valid tags, still offer tag completions
			return this.getTags(ast.source, params);
		},
		
		/**
		 * Returns if the tag block that we are in is a style of script block
		 * @param {Object} node The node
		 * @returns {Boolean} True if the current node context is style or script
		 * @since 10.0 
		 */
		inScriptOrStyle: function(node, offset, source) {
			if (node){
				if (node.type === 'tag'){
					var name = node.name ? node.name.toLowerCase() : '';
					if (name === 'script' || name === 'style') {
						if (node.openrange && node.endrange){
							// If we are in the tag itself we are not actually in the script block
							if (offset < node.openrange[1] || offset > node.endrange[0]){
								return false;
							}
						}
						return true;
					}
				}
			}
			return false;
		},  
		
		/**
		 * Returns if the offset is in a closing tag. A closing tag is determined as an 
		 * offset past the last child but before the closing range of the tag itself
		 * @param {Object} node The AST not context
		 * @param {Number} offset The curren offset
		 * @param {String} source The source 
		 */
		inClosingTag: function(node, offset, source) {
			if(node && source) {
				switch(node.type) {
					case 'tag': {
						// Smarter way now that we have end ranges
						if (node.endrange){
							return offset > node.endrange[0] && offset < node.endrange[1];
						}
						// TODO Delete the old way
						var _s = source.slice(node.range[0], node.range[1]);
						var _r = new RegExp("<\\s*\/\\s*"+node.name+"\\s*>$"); //$NON-NLS-1$ //$NON-NLS-2$
						var _m = _r.exec(_s);
						if(_m) {
							return offset > (_m.index+node.range[0]) && offset < node.range[1];
						}
						break;
					}
					default: {
						var _p = node.parent;
						if(_p && _p.type === 'tag') {
							return Array.isArray(_p.children) && _p.children.length > 0 && (offset > _p.children[_p.children.length-1].range[1]) && offset <= _p.range[1];
						}
						break;
					}
				}
			}
			return false;
		},
		
		/**
		 * Computes if we are trying to complete a comment start or end
		 * @param {Object} node The AST node to check with the offset
		 * @returns {Boolean} True if we are completing a comment, false otherwise 
		 * @since 10.0 
		 */
		isCompletingCommentClose: function(node, offset) {
			return node && node.type === 'comment' && offset >= node.range[0] && offset <= node.range[1];
		},
		
		/**
		 * Computes if we are trying to complete an open comment
		 * @param {Object} node The AST node to check with the offset
		 * @returns {Boolean} True if we are completing a comment, false otherwise 
		 * @since 10.0 
		 */
		isCompletingCommentOpen: function(node) {
			if(node 	&& node.type === 'tag' && node.name.match(/^!-{0,2}$/)){
				return true;
			}
			return false;
		},
		
		/**
		 * Computes if we are trying to complete tag attributes
		 * @param {Object} node The AST node to check with the offset
		 * @param {String} source The backing source
		 * @param {Object} params The parameters
		 * @returns {Boolean} True if we are completing the attributes of a tag, false otherwise 
		 * @since 10.0 
		 */
		isCompletingTagAttribute: function(node, source, params) {
			if(node) {
				var offset = params.offset;
				if(node.type === 'tag') {
					var tagNameEnd = node.range[0] + 1 + node.name.length;
					if(tagNameEnd < offset) {
						var idx = offset;
						while(idx < node.range[1]) {
							var char = source[idx];
							if(char === '<') {
								return false;
							} else if(char === '>') {
								return true;
							}
							idx++;
						}
					}
				} else if(node.type === 'attr') {
					return offset >= node.range[0] || offset <= node.range[1];
				}
			}
			return false;
		},
		
		/**
		 * Computes if we are trying to complete a tag
		 * @param {Object} node The AST node to check with the offset
		 * @param {Object} params The parameters
		 * @returns {Boolean} True if we are completing a tag, false otherwise 
		 * @since 10.0 
		 */
		isCompletingTag: function(node, params) {
			if(node) {
				var offset = params.offset;
				if(node.type === 'tag') {
					if (node.openrange){
						if (offset >= node.openrange[0] && offset <= node.openrange[1]){
							return true;
						}
					} else if (offset >= node.range[0] && offset <= node.range[1]){
						return true;
					}
 				}
			}
			return false;
		},
		
		/**
		 * Computes if we are completing a tag that already has a matching close tag
		 * @param {Object} node The AST node to check with the offset
		 * @param {String} source The source of the file
		 * @returns {Boolean} True if we are completing a tag with a matching close tag, false otherwise 
		 * @since 11.0 
		 */
		isCompletingTagWithMatchingClose: function(node, source) {
			if(node && node.type === 'tag' && node.name) {
				if (node.endrange && node.endrange.length === 2){
					// If the HTML is incomplete, the parser recovery sometimes uses the end range of the parent element
					return node.name === source.substring(node.endrange[0]+2, node.endrange[1]-1);
				}
 			}
		},
		
		/**
		 * Returns the tag proposals for the current offset
		 * @function
		 * @param {Object} source The source in the buffer
		 * @param {Object} params the parameters, including the offset in the source
		 * @returns returns {Array.<Object>} The array of proposals
		 */
		getTags: function(source, params) {
			var tags = Tags.tagTemplates;
			var proposals = [];
			var namePrefix = params.prefix ? params.prefix : "";
			var precedingChar = source.charAt(params.offset - namePrefix.length - 1);
			if (namePrefix.length === 0){
				while (precedingChar === '!' || precedingChar === '-'){
					namePrefix = precedingChar + namePrefix;
					precedingChar = source.charAt(params.offset - namePrefix.length - 1);
				}
				if (namePrefix.match(/^!-?$/)){
					var prefix = '<' + namePrefix;
					proposals.push(this.makeComputedProposal("<!-- ", Messages['openCommentName'], " - <!-- -->", null, prefix)); //$NON-NLS-1$ //$NON-NLS-2$
					return proposals;
				}
			}
			var leadingBracket = false;
			if (precedingChar === '<'){
				leadingBracket = true;
			}
			for(var j = 0; j < tags.length; j++) {
				var tag = tags[j];
				if(jsUtil.looselyMatches(namePrefix, tag.name)) {
					var hover = Object.create(null);
					hover.type = 'markdown'; //$NON-NLS-1$
					hover.content = "";
					if (tag.category === "Obsolete and deprecated elements"){
						hover.content += Messages['obsoleteTag'];
					}
					if (tag.doc){
						hover.content += tag.doc;
					}
					if(tag.url) {
						hover.content += i18nUtil.formatMessage(Messages['onlineDocumentation'], tag.url);
					}
					var proposalText = "";
					var desc = "";
					// TODO Allow tags to have custom templates
					tag.type = 'single'; //$NON-NLS-1$
					switch (tag.type) {
						case 'single':
							proposalText = "<" + tag.name + "></" + tag.name + ">"; //$NON-NLS-1$
//							desc = " - " + proposalText;
							if (leadingBracket){
								proposalText = proposalText.substring(1);
							}
							break;
						case 'multi':
							proposalText = "<" + tag.name + ">\n\n</" + tag.name + ">"; //$NON-NLS-1$
//							desc = " - " + proposalText;
							if (leadingBracket){
								proposalText = proposalText.substring(1);
							}
							break;
						case 'empty':
							proposalText = "<" + tag.name + "/>"; //$NON-NLS-1$
//							desc = " - " + proposalText;
							if (leadingBracket){
								proposalText = proposalText.substring(1);
							}
							break;
						default:
							proposalText = "<" + tag.name + ">";
//							desc = " - " + proposalText;
							if (leadingBracket){
								proposalText = proposalText.substring(1);
							}
							break;
					}
					if (tag.category === "Obsolete and deprecated elements"){
						desc += Messages['obsoleteTagDesc'];
					}
					var proposal = this.makeComputedProposal(proposalText, tag.name, desc, hover, params.prefix);
					// The prefix not being includes prevents content assist staying open while typing
//					if (source.charAt(params.offset - prefix.length - 1) === '<'){
//						prefix = '<' + prefix;
//						proposal.prefix = prefix;
//					}
					proposal.escapePosition = params.offset - namePrefix.length + tag.name.length + 2;
					if(leadingBracket){
						proposal.escapePosition--;
					}
					proposals.push(proposal);
				}
			}
			return proposals;	
		},
		
		/**
		 * Returns the attributes that can be used in the specified tag
		 * @param {Object} node The AST node for the tag we are completing within
		 * @param {Object} params The parameters
		 * @returns {Array.<Object>} The array of proposals
		 * @since 10.0 
		 */
		getAttributesForNode: function(node, source, params) {
			var proposals = [];
			var prefix = params.prefix ? params.prefix : "";
			// we need to check if we need to rebuild the prefix for completion that contains a '-'
			var index = params.offset - prefix.length - 1;
			if (index > 0 && index < source.length) {
				var precedingChar = source.charAt(index);
				if (precedingChar === '=' && prefix.length === 0 && (index - 1) > 0) {
					precedingChar = source.charAt(index - 1);
					if (/[A-Za-z0-9_]/.test(precedingChar)) {
						proposals.push(this.makeComputedProposal("\"\"",  Messages['addQuotesToAttributes'], " - \"\"", null, prefix)); //$NON-NLS-1$ //$NON-NLS-2$
						return proposals;
					}
				}
			}
			var attrs = Attributes.getAttributesForNode(node);
			if(Array.isArray(attrs.global)) {
				proposals = proposals.concat(this.addProposals(node, attrs.global, params));
			}
			if(Array.isArray(attrs.formevents)) {
				var arr = this.addProposals(node, attrs.formevents, params);
				if(arr.length > 0) {
					proposals.push({
							proposal: '',
							description: Messages['formeventsHeader'],
							style: 'noemphasis_title', //$NON-NLS-1$
							unselectable: true,
							kind: 'html' //$NON-NLS-1$
					});
					proposals = proposals.concat(arr);
				}
			}
			if(Array.isArray(attrs.keyboardevents)) {
				arr = this.addProposals(node, attrs.keyboardevents, params);
				if(arr.length > 0) {
					proposals.push({
							proposal: '',
							description: Messages['keyboardeventsHeader'],
							style: 'noemphasis_title', //$NON-NLS-1$
							unselectable: true,
							kind: 'html' //$NON-NLS-1$
					});
					proposals = proposals.concat(arr);
				}
			}
			if(Array.isArray(attrs.mouseevents)) {
				arr = this.addProposals(node, attrs.mouseevents, params);
				if(arr.length > 0) {
					proposals.push({
							proposal: '',
							description: Messages['mouseeventsHeader'],
							style: 'noemphasis_title', //$NON-NLS-1$
							unselectable: true,
							kind: 'html' //$NON-NLS-1$
						});
					proposals = proposals.concat(arr);
				}
			}
			if(Array.isArray(attrs.windowevents) && attrs.windowevents.length > 0) {
				arr = this.addProposals(node, attrs.windowevents, params);
				if(arr.length > 0) {
					proposals.push({
							proposal: '',
							description: Messages['windoweventsHeader'],
							style: 'noemphasis_title', //$NON-NLS-1$
							unselectable: true,
							kind: 'html' //$NON-NLS-1$
						});
					proposals = proposals.concat(arr);
				}
			}
			if(Array.isArray(attrs.aria)) {
				arr = this.addProposals(node, attrs.aria, params);
				if(arr.length > 0) {
					proposals.push({
							proposal: '',
							description: 'ARIA', //$NON-NLS-1$
							style: 'noemphasis_title', //$NON-NLS-1$
							unselectable: true,
							kind: 'html' //$NON-NLS-1$
					});
					proposals = proposals.concat(arr);
				}
			}
			return proposals;
		},
		
		addProposals: function addProposals(node, attrs, params) {
			// TODO Try adding spaces ahead of the attribute if previous attribute had no trailing whitespace
			var proposals = [];
			var tagNode = node;
			if (node.type === 'attr' && node.parent){
				tagNode = node.parent;
			}
			for(var j = 0; j < attrs.length; j++) {
				var attr = attrs[j];
				var prefix = params.prefix ? params.prefix : "";
				if(jsUtil.looselyMatches(prefix, attr.name) && !this._hasAttribute(tagNode, attr.name)) {
					
					var deprecated = Deprecated.isAttributeDeprecated(tagNode.name, attr.name);
					var hover = Object.create(null);
					var desc = "";
					hover.type = 'markdown'; //$NON-NLS-1$
					hover.content = "";
					if (deprecated){
						hover.content += i18nUtil.formatMessage(Messages['obsoleteAttr'], deprecated);
						desc += Messages['obsoleteAttrDesc'];
					}
					if (attr.doc){
						hover.content += attr.doc;
					}
					if (Array.isArray(attr.values)) {
						hover.content += Messages['possibleValues'];
						for(var v = 0; v < attr.values.length; v++) {
							var val = attr.values[v];
							hover.content += i18nUtil.formatMessage(Messages['valueNameDocMarkdown'], val.name, val.doc);
						}
					}
					if(attr.url) {
						hover.content += i18nUtil.formatMessage(Messages['onlineDocumentation'], attr.url);
					}
					var proposalText = attr.name;
					var caretOffset = 0;
					if (!Array.isArray(node.valueRange)
							|| (node.valueRange[0] > params.offset || node.valueRange[1] < params.offset)) {
						if (typeof node.value !== 'string') {
							proposalText += '=""'; //$NON-NLS-1$
							caretOffset = 2; // 2 to put the caret between the two quotes
						} else if (proposalText.indexOf(prefix) === -1) {
							proposalText += '=""'; //$NON-NLS-1$
							caretOffset = 2; // 2 to put the caret between the two quotes
						} else if (prefix.length === 0 || (prefix.length !== 0 && proposalText.indexOf(prefix) === -1)) {
							proposalText += '=""'; //$NON-NLS-1$
							caretOffset = 2; // 2 to put the caret between the two quotes
						}
					}
					var proposal = this.makeComputedProposal(proposalText, attr.name, desc, hover, prefix);
					proposal.escapePosition = params.offset - prefix.length + attr.name.length + caretOffset; 
					proposals.push(proposal);
				}
			}
			proposals.sort(function(l,r) {
				//sort by relevance and then by name
				if(typeof l.relevance === 'undefined') {
					l.relevance = 1;
				}
				if(typeof r.relevance === 'undefined') {
					r.relevance = 1;
				}
				if (l.relevance > r.relevance) {
					return -1;
				} else if (r.relevance > l.relevance) {
					return 1;
				}
				var ldesc = l.name;
				var rdesc = r.name;
				if (ldesc < rdesc) {
					return -1;
				} else if (rdesc < ldesc) {
					return 1;
				}
				return 0;
			});
			return proposals;
		},
		
		/**
		 * Returns a comment open/close proposal or an empty array
		 * @param {Object} node The AST node for the tag we are completing within
		 * @param {Number} offset The offset content assist was activated at
		 * @param {String} source The source of the file
		 * @param {Boolean} open If true will propose open comment proposals, otherwise return close comment
		 * @returns {Array.<Object>} The array of proposals
		 * @since 10.0 
		 */
		getComment: function(node, offset, source, open) {
			var proposals = [];
			if (open){
				var prefix = '<' + node.name;
				proposals.push(this.makeComputedProposal("<!-- ", Messages['openCommentName'], " - <!-- -->", null, prefix)); //$NON-NLS-1$ //$NON-NLS-2$
			} else {
				if (node.data.length > 0){
					prefix = "";
					// Check if user has typed dashes (not including the leading <!--)
					if (source.charAt(offset-1) === '-' && offset > node.range[0]+4){
						prefix += '-';
						if (source.charAt(offset-2) === '-'){
							prefix += '-';
						}
					}
					proposals.push(this.makeComputedProposal("-->", Messages['closeCommentName'], " - <!-- -->", null, prefix)); //$NON-NLS-1$ //$NON-NLS-2$
				}
			}
			return proposals;	
		},
		
		
		/**
		 * @description Returns true if the node has the given attribute already
		 * @param {Object} node The AST node to check
		 * @param {String} attribute the name of the attribute
		 * @returns {Boolean} True if the node has the given attribute, false otherwise
		 */
		_hasAttribute: function(node, attribute) {
			return node
					&& node.type === 'tag'
					&& typeof node.attributes === 'object'
					&& attribute
					&& !!node.attributes[attribute]
					&& node.attributes[attribute].value !== null; // a complete attribute needs a value
		},
		
		/**
		 * Returns the options (if any) that can be used in the specified attribute
		 * @param {Object} node The AST node for the attribute we are completing within
		 * @param {Object} params The parameters
		 * @returns {Array.<Object>} The array of proposals
		 * @since 10.0 
		 */
		getValuesForAttribute: function(node, source, params) {
			var proposals = [];
			var prefix = params.prefix ? params.prefix : "";
			// we need to check if we need to rebuild the prefix for completion that contains a '-'
			var index = params.offset - prefix.length - 1;
			if (index > 0 && index < source.length) {
				var precedingChar = source.charAt(index);
				if (precedingChar === '=' && prefix.length === 0 && (index - 1) > 0) {
					precedingChar = source.charAt(index - 1);
					if (/[A-Za-z0-9_]/.test(precedingChar)) {
						if (index + 1 >= source.length || 
								(source.charAt(index + 1) !== '\"' && source.charAt(index + 1) !== "'")) {
							proposals.push(this.makeComputedProposal("\"\"",  Messages['addQuotesToAttributes'], " - \"\"", null, prefix)); //$NON-NLS-1$ //$NON-NLS-2$
						}
					}
				}
			}
			var vals = Attributes.getValuesForAttribute(node);
			if(Array.isArray(vals)) {
				proposals = proposals.concat(this.addProposals(node, vals, params));
			}
			return proposals;	
		},
		/**
		 * Returns any proposals (if any) for when the user is editing text contents based upon
		 * state of the AST.  Templates are added to this list.
		 * @param {Object} node The AST node for the attribute we are completing within
		 * @param {String} source The backing source
		 * @param {Object} params The parameters
		 * @returns {Array.<Object>} The array of proposals
		 * @since 10.0 
		 */
		getProposalsForTextContent: function(node, source, params) {
			if (node){
				var startTag;
				// If text content is a '/' offer to close the tag
				// If we have an uncompleted tag '</' offer to complete the tag
				if (node.type === 'tag' && node.openrange && params.offset > node.openrange[1]){
					startTag = node;
				} else if (node.type === 'tag' && (params.offset > node.range[1] || params.offset < node.range[0])){
					startTag = node;
				}
				if (startTag){
					var preceding = source.substring(0, params.offset);
					var match = preceding.match(/<?\/$/);
					if (match){
						var name = '</' + startTag.name + '>'; //$NON-NLS-1$
						var desc = i18nUtil.formatMessage(Messages['closeTagDescription'], startTag.name);
						return [this.makeComputedProposal(name, name, desc, null, match[0])];
					}
				}
			}
			return this.getTags(source, params);	
		},
		
		/**
		 * Computes if we are trying to complete attributes
		 * @param {Object} node The AST node to check with the offset
		 * @param {String} source The backing source
		 * @param {Object} params The parameters
		 * @returns {Boolean} True if we are completing the attributes of a tag, false otherwise 
		 * @since 10.0 
		 */
		isCompletingAttributeValue: function(node, source, params) {
			// TODO We can do better with the new parser, handle no quotes cases too
			if(node && node.type === 'attr') {
				if (node.valueRange) {
					var range = node.valueRange;
					return range[0] <= params.offset && range[1] >= params.offset;
				}
				return this.within('"', '"', source, params.offset, node.range) || //$NON-NLS-1$ //$NON-NLS-2$
						this.within("'", "'", source, params.offset, node.range); //$NON-NLS-1$ //$NON-NLS-2$
			}
			return false;
		},
		
		/**
		 * Factory-like function to create proposal objects
		 * @param {String} proposal The proposal text
		 * @param {String} name The name for the proposal
		 * @param {String} description The description for the proposal
		 * @param {Object} hover The markdown hover object for the proposal
		 * @param {String} prefix The prefix for the proposal
		 * @since 10.0   
		 */
		makeComputedProposal: function(proposal, name, description, hover, prefix) {
			return {
				proposal: proposal,
				relevance: 100,
				name: name,
				description: description,
				hover: hover,
				prefix: prefix,
				style: 'emphasis', //$NON-NLS-1$
				overwrite: true,
				kind: 'html' //$NON-NLS-1$
		    };
		},
		
		/**
		 * A util method to determine if the caret is within two given chars, that are found within 
		 * the given bounds
		 * @param {String} start The char to the LHS
		 * @param {String} end The char to the RHS
		 * @param {String} source The source to check against
		 * @param {Number} offset   
		 */
		within: function(start, end, source, offset, bounds) {
			var idx = offset;
			var _c;
			var before = false;
			while(idx > bounds[0]) {
				_c = source[idx];
				if(_c === start) {
					before = true;
					break;
				}
				idx--;
			}
			if(before) {
				idx = offset;
				while(idx < bounds[1]) {
					_c = source[idx];
					if(_c === end) {
						return true;
					}
					idx++;
				}
			}
			return false;
		}
	});

	return {
		HTMLContentAssistProvider: HTMLContentAssistProvider
	};
});

 /*******************************************************************************
 * @license
 * Copyright (c) 2016 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env amd*/
define('webtools/htmlOccurrences',[
	'orion/objects',
	'webtools/util'
], function(Objects, util) {
	
	/**
	 * @name webtools.HtmlOccurrences
	 * @description creates a new instance of the occurrence finder
	 * @constructor
	 * @public
	 * @param {javascript.ASTManager} astManager
	 */
	function HTMLOccurrences(astManager) {
		this.astManager = astManager;
	}
	
	Objects.mixin(HTMLOccurrences.prototype, /** @lends webtools.HtmlOccurrences.prototype*/ {
		
		/**
		 * @name computeOccurrences
		 * @description Callback from the editor to compute the occurrences
		 * @function
		 * @public 
		 * @memberof javascript.JavaScriptOccurrences.prototype
		 * @param {Object} editorContext The current editor context
		 * @param {Object} ctxt The current selection context
		 */
		computeOccurrences: function(editorContext, ctxt) {
			var that = this;
	        return that.astManager.getAST(editorContext).then(function(ast) {
				return findOccurrences(ast, ctxt);
			});
		}
	});
	
	/**
	 * If the caret is on a tag, marks the matching end tag and vice versa
	 * @param ast
	 * @param ctxt
	 * @returns returns {[Object]} Array of occurrences to mark
	 */
	function findOccurrences(ast, ctxt) {
		if(ast && ctxt) {
			var start = ctxt.selection.start;
			var end = ctxt.selection.end;
			var source = ast.source;
			
			// If caret is beside a tag, mark it rather than the parent
			if (start === end){
				if ('<' === source[start]){
					start++;
					end++;
				} else if ('>' === source[start-1]){
					start--;
					end--;
				}
			}
			var node = util.findNodeAtOffset(ast, start);
			if(node) {
				if ((node.type === 'attr' || node.type === 'text') && node.parent){
					node = node.parent;
				}
				if (node.type === 'tag'){
					var occurrences = [];
					var tagName = node.name;
					var openTagStart = node.range[0];
					openTagStart++; // after the open bracket <
					var openTagEnd = openTagStart+tagName.length;
					var closeTagEnd = node.range[1] - 1;
					var closeTagStart = closeTagEnd;
					
					// Since the parser does not handle incorrect HTML well, we do some sanity checking here
					if(tagName !== source.substring(openTagStart, openTagEnd)){
						return []; // Unexpected open tag, abort
					}
					
					var char = source[closeTagStart];
					if (char === '>'){
						closeTagStart--;
						// Check for inline tag format <body/>
						if ('/' !== source[closeTagStart]){
							closeTagStart -= tagName.length;
							if ('/' + tagName !== source.substring(closeTagStart,closeTagEnd)){
								if (node.openrange && !node.endrange){
									// Void tag or some element with only an opening tag, only mark the open tag
									closeTagStart = closeTagEnd = -1;
								} else {
									return []; // Unexpected end tag, abort
								}
							}
						}
					} else {
						return []; // Unexpected character, abort
					}
					
					if (start >= node.range[0] && end <= node.range[1]){
						occurrences.push({start: openTagStart, end: openTagEnd});
						if (closeTagStart >= 0){
							occurrences.push({start: closeTagStart, end: closeTagEnd});
						}

					}
					// The following marks tags when caret is in the name
//					if(start <= openTagEnd && start >= openTagStart) {
//						occurrences.push({start: openTagStart, end: openTagEnd});
//						occurrences.push({start: closeTagStart, end: closeTagEnd});
//					}
//					if(start >= closeTagStart && start <= closeTagEnd) {
//						occurrences.push({start: openTagStart, end: openTagEnd});
//						occurrences.push({start: closeTagStart, end: closeTagEnd});
//					}
						
					return occurrences;
				}
			}
		}
		return [];
	}
	
	HTMLOccurrences.prototype.contructor = HTMLOccurrences;
	
	return {
			HTMLOccurrences: HTMLOccurrences
		};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*global Tautologistics*/
/*eslint-env amd */
define("webtools/htmlOutliner", [
	'orion/objects',
], function(Objects) {

	/**
	 * @description Creates a new validator
	 * @constructor
	 * @public
	 * @param {Object} htmlAstManager The back AST manager to provide shared HTML DOMs 
	 * @since 6.0
	 */
	function HtmlOutliner(htmlAstManager) {
	    this.htmlAstManager = htmlAstManager;
	}

	Objects.mixin(HtmlOutliner.prototype, /** @lends webtools.HtmlOutliner.prototype*/ {
		
		/**
		 * @callback
		 */
		computeOutline: function(editorContext, options) {
		    var that = this;
		    return that.htmlAstManager.getAST(editorContext).then(function(ast){
    			return that.domToOutline(ast);
		    });
		},
	
		/**
		 * Converts an HTML dom element into a label
		 * @param {Object} element The HTML element
		 * @return {String} A human readable label
		 */
	
		domToLabel: function(node) {
			var label = node.name;
			//include id if present
			var match = /id=['"]\S*["']/.exec(node.raw); //$NON-NLS-0$
			if (match) {
				label = label + " " + match[0]; //$NON-NLS-0$
			}
			//include class if present
			match = /class=['"]\S*["']/.exec(node.raw); //$NON-NLS-0$
			if (match) {
				label = label + " " + match[0]; //$NON-NLS-0$
			}
			return label;
		},
	
		/**
		 * Converts an HTML DOM node into an outline element
		 * @param {Object} An HTML DOM node as returned by the Tautologistics HTML parser
		 * @return {Object} A node in the outline tree
		 */
	
		domToOutline: function(dom) {
			//end recursion
			if (!dom) {
				return null;
			}
			var outline = [];
			for (var i = 0; i < dom.length; i++) {
				var node = dom[i];
				if(this.skip(node)) {
				    continue;
				}
				var element = {
					label: this.domToLabel(node),
					children: this.domToOutline(node.children),
					line: node.location.line,
					offset: node.location.col,
					length: node.name.length
				};
				outline.push(element);
			}
			if (outline.length > 0){
				return outline;
			}
			return null;
		},
	
	   /**
		 * Returns whether this HTML node should be omitted from the outline.
		 * @function 
		 * @private 
		 * @param {Object} node The HTML element
		 * @return {Boolean} true if the element should be skipped, and false otherwise
		 */
	
		skip: function(node) {
			//skip nodes with no name
			if (!node.name) {
				return true;
			}
	
			//skip formatting elements
			if (node.name === "b" || node.name === "i" || node.name === "em") { //$NON-NLS-0$ //$NON-NLS-1$ //$NON-NLS-2$
				return true;
			}
	
			//skip paragraphs and other blocks of formatted text
			if (node.name === "p" || node.name === "tt" || node.name === "code" || node.name === "blockquote") { //$NON-NLS-0$ //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
				return true;
			}
	
			//skip anchors
			if (node.name === "a") { //$NON-NLS-0$
				return true;
			}
	
			//include the element if we have no reason to skip it
			return false;
		},
	
		/**
		 * Returns the DOM node corresponding to the HTML body, 
		 * or null if no such node could be found.
		 */
		findBody: function(dom) {
			//recursively walk the dom looking for a body element
			for (var i = 0; i < dom.length; i++) {
				if (dom[i].name === "body") { //$NON-NLS-0$
					return dom[i].children;
				}
				if (dom[i].children) {
					var result = this.findBody(dom[i].children);
					if (result) {
						return result;
					}
				}
			}
			return null;
		}
	});
	
	return {
		HtmlOutliner : HtmlOutliner
	};
});
/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 * 
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd*/
define("orion/editor/stylers/lib/syntax", [], function() {
	return {
		id: "orion.lib",
		grammars: [{
			id: "orion.lib",
			repository: {
				brace_open: {
					match: "{",
					name: "punctuation.section.block.begin"
				},
				brace_close: {
					match: "}",
					name: "punctuation.section.block.end"
				},
				bracket_open: {
					match: "\\[",
					name: "punctuation.section.bracket.begin"
				},
				bracket_close: {
					match: "\\]",
					name: "punctuation.section.bracket.end"
				},
				parenthesis_open: {
					match: "\\(",
					name: "punctuation.section.parens.begin"
				},
				parenthesis_close: {
					match: "\\)",
					name: "punctuation.section.parens.end"
				},
				operator: {
					match: "(\\+|-|!|=|>|<|&|(\\|\\|))+",
					name: "punctuation.operator"
				},
				doc_block: {
					begin: "/\\*\\*",
					end: "\\*/",
					name: "comment.block.documentation",
					patterns: [
						{
							match: "@(?:(?!\\*/)\\S)*",
							name: "meta.documentation.annotation"
						}, {
							match: "<[^\\s>]*>",
							name: "meta.documentation.tag"
						}, {
							match: "(\\b)(TODO)(\\b)(((?!\\*/).)*)",
							name: "meta.annotation.task.todo",
							captures: {
								2: {name: "keyword.other.documentation.task"},
								4: {name: "comment.block"}
							}
						}
					]
				},
				number_decimal: {
					match: "\\b-?(?:\\.\\d+|\\d+\\.?\\d*)(?:[eE][+-]?\\d+)?\\b",
					name: "constant.numeric.number"
				},
				number_hex: {
					match: "\\b0[xX][0-9A-Fa-f]+\\b",
					name: "constant.numeric.hex"
				},
				string_doubleQuote: {
					match: '"(?:\\\\.|[^"])*"?',
					name: "string.quoted.double"
				},
				string_singleQuote: {
					match: "'(?:\\\\.|[^'])*'?",
					name: "string.quoted.single"
				},
				todo_comment_singleLine: {
					match: "(\\b)(TODO)(\\b)(.*)",
					name: "meta.annotation.task.todo",
					captures: {
						2: {name: "keyword.other.documentation.task"},
						4: {name: "comment.line"}
					}
				}
			}
		}, {
			id: "orion.c-like",
			repository: {
				comment_singleLine: {
					match: {match: "//.*", literal: "//"},
					name: "comment.line.double-slash",
					patterns: [
						{
							include: "orion.lib#todo_comment_singleLine"
						}
					]
				},
				comment_block: {
					begin: {match: "/\\*", literal: "/*"},
					end: {match: "\\*/", literal: "*/"}, 
					name: "comment.block",
					patterns: [
						{
							match: "(\\b)(TODO)(\\b)(((?!\\*/).)*)",
							name: "meta.annotation.task.todo",
							captures: {
								2: {name: "keyword.other.documentation.task"},
								4: {name: "comment.block"}
							}
						}
					]
				}
			}
		}],
		keywords: []
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 * 
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd*/

define("orion/editor/stylers/application_javascript/syntax", ["orion/editor/stylers/lib/syntax"], function(mLib) {
	var keywords = [
		"class", "const",
		"debugger", "delete",
		"enum", "export", "extends",
		"function",
		"implements", "import", "in", "instanceof", "interface",
		"let",
		"new",
		"package", "private", "protected", "public",
		"static", "super",
		"typeof",
		"var", "void",
		"with"
	];
	var controlKeywords = [
		"break",
		"case", "catch", "continue",
		"default", "do",
		"else",
		"finally", "for",
		"if",
		"return",
		"switch",
		"throw", "try",
		"while",
		"yield"
	];
	var languageVariables = ["this"];
	var constants = [
		"false", "null", "true", "undefined"
	];

	var grammars = [];
	grammars.push.apply(grammars, mLib.grammars);
	grammars.push({
		id: "orion.js",
		contentTypes: ["application/javascript"],
		patterns: [
			{
				begin: "'(?:\\\\.|[^\\\\'])*\\\\$",
				end: "^(?:$|(?:\\\\.|[^\\\\'])*('|[^\\\\]$))",
				name: "string.quoted.single.js"
			}, {
				begin: '"(?:\\\\.|[^\\\\"])*\\\\$',
				end: '^(?:$|(?:\\\\.|[^\\\\"])*("|[^\\\\]$))',
				name: "string.quoted.double.js"
			},
			{include: "orion.lib#string_doubleQuote"},
			{include: "orion.lib#string_singleQuote"},
			{include: "orion.c-like#comment_singleLine"},
			{
				match: "/(?![\\s\\*])(?:\\\\.|[^/])+/(?:[gim]{0,3})",
				name: "string.regexp.js"
			},
			{include: "orion.lib#doc_block"},
			{include: "orion.c-like#comment_block"},
			{include: "#jsFunctionDef"},
			{include: "orion.lib#brace_open"},
			{include: "orion.lib#brace_close"},
			{include: "orion.lib#bracket_open"},
			{include: "orion.lib#bracket_close"},
			{include: "orion.lib#parenthesis_open"},
			{include: "orion.lib#parenthesis_close"},
			{include: "orion.lib#operator"},
			{include: "orion.lib#number_decimal"},
			{include: "orion.lib#number_hex"},
			{
				match: "\\b(?:" + keywords.join("|") + ")\\b",
				name: "keyword.operator.js"
			},
			{
				match: "\\b(?:" + controlKeywords.join("|") + ")\\b",
				name: "keyword.control.js"
			},
			{
				match: "\\b(?:" + constants.join("|") + ")\\b",
				name: "constant.language.js"
			},
			{
				match: "\\b(?:" + languageVariables.join("|") + ")\\b",
				name: "variable.language.js"
			}
		],
		repository: {
			jsFunctionDef: {
				/*
				 * http://stackoverflow.com/questions/2008279/validate-a-javascript-function-name/2008444#2008444
				 * was referenced in the composition of the "begin" pattern below.
				 */
				begin: "(function)(\\s+[_$a-zA-Z\\xA0-\\uFFFF][_$a-zA-Z0-9\\xA0-\\uFFFF]*)?\\s*\\(",
				end: "\\)",
				captures: {
					1: {name: "keyword.operator.js"},
					2: {name: "entity.name.function.js"}
				},
				patterns: [
					{include: "orion.c-like#comment_singleLine"},
					{include: "orion.c-like#comment_block"},
					{
						match: "[^\\s,]+",
						name: "variable.parameter.js"
					}
				]
			}
		}
	});

	return {
		id: grammars[grammars.length - 1].id,
		grammars: grammars,
		keywords: keywords.concat(controlKeywords).concat(languageVariables).concat(constants)
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 * 
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd*/
define("orion/editor/stylers/text_css/syntax", ["orion/editor/stylers/lib/syntax"], function(mLib) {
	var keywords = [
		"alignment-adjust", "alignment-baseline", "animation-delay", "animation-direction", "animation-duration", "animation-iteration-count",
		"animation-name", "animation-play-state", "animation-timing-function", "animation", "appearance",
		"azimuth", "backface-visibility", "background-attachment", "background-clip", "background-color", "background-image",
		"background-origin", "background-position", "background-repeat", "background-size", "background", "baseline-shift",
		"binding", "bleed", "bookmark-label", "bookmark-level", "bookmark-state", "bookmark-target", "border-bottom-color",
		"border-bottom-left-radius", "border-bottom-right-radius", "border-bottom-style", "border-bottom-width", "border-bottom",
		"border-collapse", "border-color", "border-image-outset", "border-image-repeat", "border-image-slice", "border-image-source",
		"border-image-width", "border-image", "border-left-color", "border-left-style", "border-left-width", "border-left",
		"border-radius", "border-right-color", "border-right-style", "border-right-width", "border-right", "border-spacing", "border-style",
		"border-top-color", "border-top-left-radius", "border-top-right-radius", "border-top-style", "border-top-width", "border-top",
		"border-width", "border", "bottom", "box-align", "box-decoration-break", "box-direction", "box-flex-group", "box-flex", "box-lines",
		"box-ordinal-group", "box-orient", "box-pack", "box-shadow", "box-sizing", "break-after", "break-before", "break-inside",
		"caption-side", "clear", "clip", "color-profile", "color", "column-count", "column-fill", "column-gap", "column-rule-color",
		"column-rule-style", "column-rule-width", "column-rule", "column-span", "column-width", "columns", "content", "counter-increment",
		"counter-reset", "crop", "cue-after", "cue-before", "cue", "cursor", "direction", "display", "dominant-baseline",
		"drop-initial-after-adjust", "drop-initial-after-align", "drop-initial-before-adjust", "drop-initial-before-align", "drop-initial-size",
		"drop-initial-value", "elevation", "empty-cells", "fit-position", "fit", "flex-align", "flex-flow", "flex-inline-pack", "flex-order",
		"flex-pack", "float-offset", "float", "font-family", "font-size-adjust", "font-size", "font-stretch", "font-style", "font-variant",
		"font-weight", "font", "grid-columns", "grid-rows", "hanging-punctuation", "height", "hyphenate-after",
		"hyphenate-before", "hyphenate-character", "hyphenate-lines", "hyphenate-resource", "hyphens", "icon", "image-orientation",
		"image-rendering", "image-resolution", "inline-box-align", "left", "letter-spacing", "line-height", "line-stacking-ruby",
		"line-stacking-shift", "line-stacking-strategy", "line-stacking", "list-style-image", "list-style-position", "list-style-type",
		"list-style", "margin-bottom", "margin-left", "margin-right", "margin-top", "margin", "mark-after", "mark-before", "mark",
		"marker-offset", "marks", "marquee-direction", "marquee-loop", "marquee-play-count", "marquee-speed", "marquee-style", "max-height",
		"max-width", "min-height", "min-width", "move-to", "nav-down", "nav-index", "nav-left", "nav-right", "nav-up", "opacity", "orphans",
		"outline-color", "outline-offset", "outline-style", "outline-width", "outline", "overflow-style", "overflow-x", "overflow-y",
		"overflow", "padding-bottom", "padding-left", "padding-right", "padding-top", "padding", "page-break-after", "page-break-before", "page-break-inside",
		"page-policy", "page", "pause-after", "pause-before", "pause", "perspective-origin", "perspective", "phonemes", "pitch-range",
		"pitch", "play-during", "position", "presentation-level", "punctuation-trim", "quotes", "rendering-intent", "resize",
		"rest-after", "rest-before", "rest", "richness", "right", "rotation-point", "rotation", "ruby-align", "ruby-overhang", "ruby-position",
		"ruby-span", "size", "speak-header", "speak-numeral", "speak-punctuation", "speak", "speech-rate", "stress", "string-set", "table-layout",
		"target-name", "target-new", "target-position", "target", "text-align-last", "text-align", "text-decoration", "text-emphasis",
		"text-height", "text-indent", "text-justify", "text-outline", "text-shadow", "text-transform", "text-wrap", "top", "transform-origin",
		"transform-style", "transform", "transition-delay", "transition-duration", "transition-property", "transition-timing-function",
		"transition", "unicode-bidi", "vertical-align", "visibility", "voice-balance", "voice-duration", "voice-family",
		"voice-pitch-range", "voice-pitch", "voice-rate", "voice-stress", "voice-volume", "volume", "white-space-collapse", "white-space",
		"widows", "width", "word-break", "word-spacing", "word-wrap", "z-index"
	];
	var colors = [
		"AliceBlue", "AntiqueWhite", "Aquamarine", "Aqua", "Azure",
		"Beige", "Bisque", "Black", "BlanchedAlmond", "BlueViolet", "Blue", "Brown", "BurlyWood",
		"CadetBlue", "Chartreuse", "Chocolate", "Coral", "CornflowerBlue", "Cornsilk", "Crimson", "Cyan",
		"DarkBlue", "DarkCyan", "DarkGoldenRod", "DarkGray", "DarkGrey", "DarkGreen", "DarkKhaki", "DarkMagenta", "DarkOliveGreen",
		"DarkOrange", "DarkOrchid", "DarkRed", "DarkSalmon", "DarkSeaGreen", "DarkSlateBlue", "DarkSlateGray", "DarkSlateGrey",
		"DarkTurquoise", "DarkViolet", "DeepPink", "DeepSkyBlue", "DimGray", "DimGrey", "DodgerBlue",
		"FireBrick", "FloralWhite", "ForestGreen", "Fuchsia",
		"Gainsboro", "GhostWhite", "Gold", "GoldenRod", "Gray", "Grey", "GreenYellow", "Green",
		"HoneyDew", "HotPink",
		"IndianRed", "Indigo", "Ivory",
		"Khaki",
		"LavenderBlush", "Lavender", "LawnGreen", "LemonChiffon", "LightBlue", "LightCoral", "LightCyan", "LightGoldenRodYellow",
		"LightGray", "LightGrey", "LightGreen", "LightPink", "LightSalmon", "LightSeaGreen", "LightSkyBlue", "LightSlateGray",
		"LightSlateGrey", "LightSteelBlue", "LightYellow", "LimeGreen", "Lime", "Linen",
		"Magenta", "Maroon", "MediumAquaMarine", "MediumBlue", "MediumOrchid", "MediumPurple", "MediumSeaGreen", "MediumSlateBlue",
		"MediumSpringGreen", "MediumTurquoise", "MediumVioletRed", "MidnightBlue", "MintCream", "MistyRose", "Moccasin",
		"NavajoWhite", "Navy",
		"OldLace", "OliveDrab", "Olive", "OrangeRed", "Orange", "Orchid",
		"PaleGoldenRod", "PaleGreen", "PaleTurquoise", "PaleVioletRed", "PapayaWhip", "PeachPuff", "Peru", "Pink", "Plum", "PowderBlue", "Purple",
		"RebeccaPurple", "Red", "RosyBrown", "RoyalBlue",
		"SaddleBrown", "Salmon", "SandyBrown", "SeaGreen", "SeaShell", "Sienna", "Silver", "SkyBlue", "SlateBlue", "SlateGray", "SlateGrey", "Snow", "SpringGreen", "SteelBlue",
		"Tan", "Teal", "Thistle", "Tomato", "Turquoise",
		"Violet",
		"Wheat", "WhiteSmoke", "White",
		"YellowGreen", "Yellow"
	];
	var directives = ["charset", "document", "font-face", "import", "keyframes", "media", "namespace", "page", "supports"];

	var grammars = [];
	grammars.push.apply(grammars, mLib.grammars);
	grammars.push({
		id: "orion.css",
		contentTypes: ["text/css"],
		patterns: [
			{include: "#string_single_multiline"},
			{include: "#string_double_multiline"},
			{include: "orion.lib#string_doubleQuote"},
			{include: "orion.lib#string_singleQuote"},
			{include: "orion.c-like#comment_block"},
			{include: "orion.lib#brace_open"},
			{include: "orion.lib#brace_close"},
			{include: "orion.lib#bracket_open"},
			{include: "orion.lib#bracket_close"},
			{include: "orion.lib#parenthesis_open"},
			{include: "orion.lib#parenthesis_close"},
			{include: "orion.lib#number_decimal"},
			{include: "#number_hex"},
			{include: "#numeric_value"},
			{include: "#color"},
			{include: "#keyword"},
			{include: "#directive"}
		],
		repository: {
			color: {
				match: "(?i)\\b(?:" + colors.join("|") + ")\\b",
				name: "constant.other.color.css"
			},
			directive: {
				match: "(^|\\s)(@("  + directives.join("|") + "))\\b",
				captures: {
					2: {name: "keyword.other.directive.css"}
				}
			},
			keyword: {
				match: "(?:-webkit-|-moz-|-ms-|-o-|\\b)(?:" + keywords.join("|") + ")\\b",
				name: "support.type.propertyName.css"
			},
			number_hex: {
				match: "#[0-9A-Fa-f]+\\b",
				name: "constant.numeric.hex.css"
			},
			numeric_value: {
				match: "(?i)\\b-?(?:\\.\\d+|\\d+\\.?\\d*)(?:%|em|ex|ch|rem|vw|vh|vmin|vmax|in|cm|mm|pt|pc|px|deg|grad|rad|turn|s|ms|Hz|kHz|dpi|dpcm|dppx)?\\b",
				name: "constant.numeric.value.css"
			},
			string_double_multiline: {
				begin: '"(?:\\\\.|[^\\\\"])*\\\\$',
				end: '^(?:$|(?:\\\\.|[^\\\\"])*("|[^\\\\]$))',
				name: "string.quoted.double.css"
			},
			string_single_multiline: {
				begin: "'(?:\\\\.|[^\\\\'])*\\\\$",
				end: "^(?:$|(?:\\\\.|[^\\\\'])*('|[^\\\\]$))",
				name: "string.quoted.single.css"
			}
		}
	});
	return {
		id: grammars[grammars.length - 1].id,
		grammars: grammars,
		keywords: keywords
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 * 
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd*/
/* eslint-disable missing-nls */
define("orion/editor/stylers/application_xml/syntax", ["orion/editor/stylers/lib/syntax"], function(mLib) {

	var grammars = [];
	grammars.push.apply(grammars, mLib.grammars);
	grammars.push({
		id: "orion.xml",
		contentTypes: ["application/xml", "application/xhtml+xml"],
		patterns: [
			{include: "#comment"},
			{include: "#doctype"},
			{include: "#xmlDeclaration"},
			{include: "#tag"},
			{include: "#ampersandEscape"}
		],
		repository: {
			ampersandEscape: {
				match: "&lt;|&gt;|&amp;",
				name: "constant.character"
			},
			comment: {
				begin: {match: "<!--", literal: "<!--"},
				end: {match: "-->", literal: "-->"},
				name: "comment.block.xml",
				patterns: [
					{
						match: "(\\b)(TODO)(\\b)(((?!-->).)*)",
						name: "meta.annotation.task.todo",
						captures: {
							2: {name: "keyword.other.documentation.task"},
							4: {name: "comment.line"}
						}
					}
				]
			},
			doctype: {
				begin: "<!(?:doctype|DOCTYPE)",
				end: ">",
				name: "meta.tag.doctype.xml",
				captures: {
					0: {name: "meta.tag.doctype.xml"}
				},
				patterns: [
					{include: "#comment"},
					{include: "orion.lib#string_doubleQuote"},
					{include: "orion.lib#string_singleQuote"}
				]
			},
			tag: {
				// https://www.w3.org/TR/2006/REC-xml11-20060816/#sec-common-syn
				begin: "</?[A-Za-z:_][A-Za-z0-9:_\\-.]*",
				end: "/?>",
				captures: {
					0: {name: "meta.tag.xml"}
				},
				patterns: [
					{include: "#comment"},
					{include: "orion.lib#string_doubleQuote"},
					{include: "orion.lib#string_singleQuote"}
				]	
			},
			xmlDeclaration: {
				begin: "<\\?xml",
				end: "\\?>",
				captures: {
					0: {name: "meta.tag.declaration.xml"}
				},
				patterns: [
					{include: "#comment"},
					{include: "orion.lib#string_doubleQuote"},
					{include: "orion.lib#string_singleQuote"}
				],
				name: "meta.tag.declaration.xml"
			}
		}
	});
	return {
		id: grammars[grammars.length - 1].id,
		grammars: grammars,
		keywords: []
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014, 2015 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 * 
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd*/
define("orion/editor/stylers/text_html/syntax", ["orion/editor/stylers/application_javascript/syntax", "orion/editor/stylers/text_css/syntax", "orion/editor/stylers/application_xml/syntax"],
	/* eslint-disable missing-nls */
	function(mJS, mCSS, mXML) {
	var attributes = [
		"accept-charset", "accept", "accesskey", "action", "align", "alt",  
		"async", "autocomplete", "autoplay", "autosave",
		"bgcolor", "border", "buffered", 
		"challenge", "charset", "checked", "cite", "class", "codebase", "code", "color",
		"colspan", "cols", "contenteditable", "content", "contextmenu", "controls", "coords",
		"data-[A-Za-z_:][\\w.:-]*", "data", "datetime", "default", "defer", "dirname", "dir",
		"disabled", "download", "draggable", "dropzone",
		"enctype",
		"formaction", "form", "for", 
		"headers", "height", "hidden", "high", "hreflang", "href", "http-equiv",
		"icon", "id", "ismap", "itemprop",
		"keytype", "kind", 
		"label", "language", "lang", "list", "loop", "low",  
		"manifest", "maxlength", "max", "media", "method", "min", "multiple",
		"name", "novalidate", 
		"open", "optimum", 
		"pattern", "ping", "placeholder", "poster", "preload", "pubdate",  
		"radiogroup", "readonly", "rel", "required", "reversed", "rowspan", "rows",
		"sandbox", "scoped", "scope", "seamless", "selected", "shape", "sizes", "size", "span", "spellcheck",
		"srcdoc", "srclang","srcset", "src", "start", "step", "style", "summary",
		"tabindex", "target", "title", "type",
		"usemap",
		"value",
		"width", "wrap" 
	];

	var grammars = [];
	grammars.push.apply(grammars, mJS.grammars);
	grammars.push.apply(grammars, mCSS.grammars);
	grammars.push.apply(grammars, mXML.grammars);
	grammars.push({
		id: "orion.html",
		contentTypes: ["text/html"],
		patterns: [
			{
				begin: "(?i)(<style)([^>]*)(>)",
				end: "(?i)(</style>)",
				captures: {
					1: {name: "meta.tag.html"},
					3: {name: "meta.tag.html"}
				},
				contentName: "source.css.embedded.html",
				patterns: [
					{include: "orion.css"}
				]
			}, {
				begin: "(?i)<script\\s*>|<script\\s.*?(?:language\\s*=\\s*(['\"])javascript\\1|type\\s*=\\s*(['\"])(?:text|application)/(?:javascript|ecmascript)\\2).*?>",
				end: "(?i)</script>",
				captures: {
					0: {name: "meta.tag.html"}
				},
				contentName: "source.js.embedded.html",
				patterns: [
					{include: "orion.js"}
				]
			}, {
				begin: "</?[A-Za-z0-9]+",
				end: "/?>",
				captures: {
					0: {name: "meta.tag.html"}
				},
				patterns: [
					{include: "orion.xml#comment"},
					{include: "orion.lib#string_doubleQuote"},
					{include: "orion.lib#string_singleQuote"},
					{include: "#attribute"}
				]
			},
			{include: "orion.xml#comment"},
			{include: "orion.xml#doctype"},
			{include: "orion.xml#ampersandEscape"}
		],
		repository: {
			attribute:{
				match: "\\b(?:" + attributes.join("|") + ")\\b",  
				name: "meta.tag.attribute.html"
			}
		}
	});
	return {
		id: grammars[grammars.length - 1].id,
		grammars: grammars,
		keywords: [],
		attributes: attributes
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2015, 2016 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html).
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/* eslint-env amd */
define('javascript/compilationUnit',[
'orion/Deferred'
], function(Deferred) {

    /**
     * @name CompilationUnit
     * @description Creates a new CompilationUint instance. These instances should not be cached as they do
     * not respond to model change events.
     * @constructor
     * @param {Array.<String>} sourceblocks The blocks of source to combine into one unit
     * @param {Object} metadata The metadata describing the file this unit represents
     * @param {Object} editorContext Optional editor context for the source file. Delegated to for setText and to get line information
     * @returns {CompilationUnit} The new CompiationUnit instance
     * @since 8.0
     */
    function CompilationUnit(sourceblocks, metadata, editorContext) {
        this._blocks = sourceblocks;
        this._metadata = metadata;
        if (metadata){
        	// The context returned by getEditorContext only contains javascript text so set the content type to match
        	this._metadata.contentType = {id: 'application/javascript'}; //$NON-NLS-1$
        }
        this._ec = editorContext;
        this._deps = [];
    }

    /**
     * @description Builds the backing source for the compilation unit
     * @function
     * @private
     */
    CompilationUnit.prototype._init = function _init() {
		var wrappedFunctionCallPrefix = "";  // Previously to avoid ESLint warnings we prefixed function calls with 'this.' See Bug 481137
        var _cursor = 0;
        this._source = '';
        this._blocks.sort(function(a, b){
        	var _a = a.offset ? a.offset : 0;
        	var _b = b.offset ? b.offset : 0;
        	return _a - _b;
        });
        for(var i = 0; i < this._blocks.length; i++) {
            var block = this._blocks[i];
            if(block.dependencies) {
            	this._deps.push(block.dependencies);
            }
            var pad = block.offset - _cursor;
            if(block.isWrappedFunctionCall){
				if (pad < wrappedFunctionCallPrefix.length){
					continue;
				}
				pad -= wrappedFunctionCallPrefix.length;
				while(pad > 0) {
                	this._source += ' '; //$NON-NLS-1$
                	pad--;
            	}
            	this._source += wrappedFunctionCallPrefix;
            	this._source += block.text;
            	if (block.text && block.text.charAt(block.text.length-1) !== ';'){
            		this._source += ';';
            	}
            } else {
	            while(pad > 0) {
	                this._source += ' '; //$NON-NLS-1$
	                pad--;
	            }
	            this._source += block.text;
            }

            _cursor = this._source.length;
        }
    };

    /**
     * @description Returns the source of this compilation unit
     * @function
     * @returns {String} The source of the compilation unit
     */
    CompilationUnit.prototype.getSource = function getSource() {
    	if(!this._source) {
            this._init();
        }
        return this._source;
    };

    /**
     * @description Returns if the given offset is valid compared to the blocks of code
     * that make up this unit
     * @function
     * @param {Number} offset The offset to check
     * @returns {Boolean} If the given offset is within any of the backing code blocks
     */
    CompilationUnit.prototype.validOffset = function validOffset(offset) {
        if(!this._blocks || this._blocks.length < 1 || offset < 0) {
	        return false;
	    }
	    for(var i = 0; i < this._blocks.length; i++) {
	        var block = this._blocks[i];
	        var idx = block.offset;
	        if(offset >= idx && offset <= idx+block.text.length) {
	            return true;
	        }
	    }
	    return false;
    };

    /**
     * @description Returns an EditorContext-like object that can resolve promises for <code>getText</code> and <code>getFileMetadata</code>
     * @function
     * @returns {Object} The EditorContext object to use when parsing
     */
   CompilationUnit.prototype.getEditorContext = function getEditorContext() {
        var proxy = Object.create(null);
        var that = this;
        proxy.getText = function() {
            return new Deferred().resolve(that.getSource());
        };
        proxy.getFileMetadata = function() {
            return new Deferred().resolve(that._metadata);
        };
        proxy.setText = function(text, start, end) {
            if(that._ec) {
                return that._ec.setText(text, start, end);
            }
            return new Deferred().resolve(null);
        };
        // Our tooling needs access to other functions on the editorContext so copy them here
        if (that._ec){
	        	proxy.getSelections = that._ec.getSelections;
	        	proxy.setSelection = that._ec.setSelection;
	        	proxy.syntaxCheck = that._ec.syntaxCheck;
	        	proxy.setCaretOffset = that._ec.setCaretOffset;
	    	}
        return proxy;
    };

    /**
     * @description Returns the computed list of dependencies
     * @function
     * @returns {Array.<string>} Returns the array of dependencies, or an empty array, never null
     * @since 9.0
     */
    CompilationUnit.prototype.getDependencies = function getDependencies() {
    	return this._deps;
    };

    return CompilationUnit;
});
/* eslint-disable  */
/* eslint-env amd */
define('csslint/csslint',[
], function() {

/*!
CSSLint
Copyright (c) 2013 Nicole Sullivan and Nicholas C. Zakas. All rights reserved.
Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.

*/
/* Build: v0.10.0 15-August-2013 01:07:22 */
var exports = exports || {};
var cssLint = (function(){
/*!
Parser-Lib
Copyright (c) 2009-2011 Nicholas C. Zakas. All rights reserved.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.

*/
/* Version v0.2.3, Build time: 19-June-2013 11:16:15 */
var parserlib = {};
(function(){


/**
 * A generic base to inherit from for any object
 * that needs event handling.
 * @class EventTarget
 * @constructor
 */
function EventTarget(){

    /**
     * The array of listeners for various events.
     * @type Object
     * @property _listeners
     * @private
     */
    this._listeners = {};    
}

EventTarget.prototype = {

    //restore constructor
    constructor: EventTarget,

    /**
     * Adds a listener for a given event type.
     * @param {String} type The type of event to add a listener for.
     * @param {Function} listener The function to call when the event occurs.
     * @return {void}
     * @method addListener
     */
    addListener: function(type, listener){
        if (!this._listeners[type]){
            this._listeners[type] = [];
        }

        this._listeners[type].push(listener);
    },
    
    /**
     * Fires an event based on the passed-in object.
     * @param {Object|String} event An object with at least a 'type' attribute
     *      or a string indicating the event name.
     * @return {void}
     * @method fire
     */    
    fire: function(event){
        if (typeof event == "string"){
            event = { type: event };
        }
        if (typeof event.target != "undefined"){
            event.target = this;
        }
        
        if (typeof event.type == "undefined"){
            throw new Error("Event object missing 'type' property.");
        }
        
        if (this._listeners[event.type]){
        
            //create a copy of the array and use that so listeners can't chane
            var listeners = this._listeners[event.type].concat();
            for (var i=0, len=listeners.length; i < len; i++){
                listeners[i].call(this, event);
            }
        }            
    },

    /**
     * vs a listener for a given event type.
     * @param {String} type The type of event to remove a listener from.
     * @param {Function} listener The function to remove from the event.
     * @return {void}
     * @method removeListener
     */
    removeListener: function(type, listener){
        if (this._listeners[type]){
            var listeners = this._listeners[type];
            for (var i=0, len=listeners.length; i < len; i++){
                if (listeners[i] === listener){
                    listeners.splice(i, 1);
                    break;
                }
            }
            
            
        }            
    }
};
/**
 * Convenient way to read through strings.
 * @namespace parserlib.util
 * @class StringReader
 * @constructor
 * @param {String} text The text to read.
 */
function StringReader(text){

    /**
     * The input text with line endings normalized.
     * @property _input
     * @type String
     * @private
     */
    this._input = text;//.replace(/\r\n?/g, "\n");  ORION 8.0

    /**
     * The row for the character to be read next.
     * @property _line
     * @type int
     * @private
     */
    this._line = 1;


    /**
     * The column for the character to be read next.
     * @property _col
     * @type int
     * @private
     */
    this._col = 1;

    /**
     * The index of the character in the input to be read next.
     * @property _cursor
     * @type int
     * @private
     */
    this._cursor = 0;
}

StringReader.prototype = {

    //restore constructor
    constructor: StringReader,

    //-------------------------------------------------------------------------
    // Position info
    //-------------------------------------------------------------------------

    /**
     * Returns the column of the character to be read next.
     * @return {int} The column of the character to be read next.
     * @method getCol
     */
    getCol: function(){
        return this._col;
    },

    /**
     * Returns the row of the character to be read next.
     * @return {int} The row of the character to be read next.
     * @method getLine
     */
    getLine: function(){
        return this._line ;
    },
    
    /**
     * @returns {Number} Returns the current offset into the source string
     * ORION 8.0
     */
    getCursor: function() {
        return this._cursor;
    },
    
    /**
     * Determines if you're at the end of the input.
     * @return {Boolean} True if there's no more input, false otherwise.
     * @method eof
     */
    eof: function(){
        return (this._cursor == this._input.length);
    },

    //-------------------------------------------------------------------------
    // Basic reading
    //-------------------------------------------------------------------------

    /**
     * Reads the next character without advancing the cursor.
     * @param {int} count How many characters to look ahead (default is 1).
     * @return {String} The next character or null if there is no next character.
     * @method peek
     */
    peek: function(count){
        var c = null;
        count = (typeof count == "undefined" ? 1 : count);

        //if we're not at the end of the input...
        if (this._cursor < this._input.length){

            //get character and increment cursor and column
            c = this._input.charAt(this._cursor + count-1);
        }

        return c;
    },

    /**
     * Reads the next character from the input and adjusts the row and column
     * accordingly.
     * @return {String} The next character or null if there is no next character.
     * @method read
     */
    read: function(){
        var c = null;

        //if we're not at the end of the input...
        if (this._cursor < this._input.length){

            //if the last character was a newline, increment row count
            //and reset column count
            if (this._input.charAt(this._cursor) == "\n"){
                this._line++;
                this._col=1;
            } else {
                this._col++;
            }

            //get character and increment cursor and column
            var i = this._cursor++;
            c = this._input.charAt(i);
            var num = this._input.charCodeAt(i);
        }

        return c;
    },

    //-------------------------------------------------------------------------
    // Misc
    //-------------------------------------------------------------------------

    /**
     * Saves the current location so it can be returned to later.
     * @method mark
     * @return {void}
     */
    mark: function(){
        this._bookmark = {
            cursor: this._cursor,
            line:   this._line,
            col:    this._col
        };
    },

    reset: function(){
        if (this._bookmark){
            this._cursor = this._bookmark.cursor;
            this._line = this._bookmark.line;
            this._col = this._bookmark.col;
            delete this._bookmark;
        }
    },

    //-------------------------------------------------------------------------
    // Advanced reading
    //-------------------------------------------------------------------------

    /**
     * Reads up to and including the given string. Throws an error if that
     * string is not found.
     * @param {String} pattern The string to read.
     * @return {String} The string when it is found.
     * @throws Error when the string pattern is not found.
     * @method readTo
     */
    readTo: function(pattern){

        var buffer = "",
            c;

        /*
         * First, buffer must be the same length as the pattern.
         * Then, buffer must end with the pattern or else reach the
         * end of the input.
         */
        while (buffer.length < pattern.length || buffer.lastIndexOf(pattern) != buffer.length - pattern.length){
            c = this.read();
            if (c){
                buffer += c;
            } else {
                throw new Error("Expected \"" + pattern + "\" at line " + this._line  + ", col " + this._col + ".");
            }
        }

        return buffer;

    },

    /**
     * Reads characters while each character causes the given
     * filter function to return true. The function is passed
     * in each character and either returns true to continue
     * reading or false to stop.
     * @param {Function} filter The function to read on each character.
     * @return {String} The string made up of all characters that passed the
     *      filter check.
     * @method readWhile
     */
    readWhile: function(filter){

        var buffer = "",
            c = this.read();

        while(c !== null && filter(c)){
            buffer += c;
            c = this.read();
        }

        return buffer;

    },

    /**
     * Reads characters that match either text or a regular expression and
     * returns those characters. If a match is found, the row and column
     * are adjusted; if no match is found, the reader's state is unchanged.
     * reading or false to stop.
     * @param {String|RegExp} matchter If a string, then the literal string
     *      value is searched for. If a regular expression, then any string
     *      matching the pattern is search for.
     * @return {String} The string made up of all characters that matched or
     *      null if there was no match.
     * @method readMatch
     */
    readMatch: function(matcher){

        var source = this._input.substring(this._cursor),
            value = null;

        //if it's a string, just do a straight match
        if (typeof matcher == "string"){
            if (source.indexOf(matcher) === 0){
                value = this.readCount(matcher.length);
            }
        } else if (matcher instanceof RegExp){
            if (matcher.test(source)){
                value = this.readCount(RegExp.lastMatch.length);
            }
        }

        return value;
    },


    /**
     * Reads a given number of characters. If the end of the input is reached,
     * it reads only the remaining characters and does not throw an error.
     * @param {int} count The number of characters to read.
     * @return {String} The string made up the read characters.
     * @method readCount
     */
    readCount: function(count){
        var buffer = "";

        while(count--){
            buffer += this.read();
        }

        return buffer;
    }

};
/**
 * Type to use when a syntax error occurs.
 * @class SyntaxError
 * @namespace parserlib.util
 * @constructor
 * @param {String} message The error message.
 * @param {int} line The line at which the error occurred.
 * @param {int} col The column at which the error occurred.
 */
function SyntaxError(message, line, col){

    /**
     * The column at which the error occurred.
     * @type int
     * @property col
     */
    this.col = col;

    /**
     * The line at which the error occurred.
     * @type int
     * @property line
     */
    this.line = line;

    /**
     * The text representation of the unit.
     * @type String
     * @property text
     */
    this.message = message;

}

//inherit from Error
SyntaxError.prototype = new Error();
/**
 * Base type to represent a single syntactic unit.
 * @class SyntaxUnit
 * @namespace parserlib.util
 * @constructor
 * @param {String} text The text of the unit.
 * @param {int} line The line of text on which the unit resides.
 * @param {int} col The column of text on which the unit resides.
 */
function SyntaxUnit(text, line, col, type){


    /**
     * The column of text on which the unit resides.
     * @type int
     * @property col
     */
    this.col = col;

    /**
     * The line of text on which the unit resides.
     * @type int
     * @property line
     */
    this.line = line;

    /**
     * The text representation of the unit.
     * @type String
     * @property text
     */
    this.text = text;

    /**
     * The type of syntax unit.
     * @type int
     * @property type
     */
    this.type = type;
}

/**
 * Create a new syntax unit based solely on the given token.
 * Convenience method for creating a new syntax unit when
 * it represents a single token instead of multiple.
 * @param {Object} token The token object to represent.
 * @return {parserlib.util.SyntaxUnit} The object representing the token.
 * @static
 * @method fromToken
 */
SyntaxUnit.fromToken = function(token){
    return new SyntaxUnit(token.value, token.startLine, token.startCol);
};

SyntaxUnit.prototype = {

    //restore constructor
    constructor: SyntaxUnit,
    
    /**
     * Returns the text representation of the unit.
     * @return {String} The text representation of the unit.
     * @method valueOf
     */
    valueOf: function(){
        return this.toString();
    },
    
    /**
     * Returns the text representation of the unit.
     * @return {String} The text representation of the unit.
     * @method toString
     */
    toString: function(){
        return this.text;
    }

};
/*global StringReader, SyntaxError*/

/**
 * Generic TokenStream providing base functionality.
 * @class TokenStreamBase
 * @namespace parserlib.util
 * @constructor
 * @param {String|StringReader} input The text to tokenize or a reader from 
 *      which to read the input.
 */
function TokenStreamBase(input, tokenData){

    /**
     * The string reader for easy access to the text.
     * @type StringReader
     * @property _reader
     * @private
     */
    this._reader = input ? new StringReader(input.toString()) : null;
    
    /**
     * Token object for the last consumed token.
     * @type Token
     * @property _token
     * @private
     */
    this._token = null;    
    
    /**
     * The array of token information.
     * @type Array
     * @property _tokenData
     * @private
     */
    this._tokenData = tokenData;
    
    /**
     * Lookahead token buffer.
     * @type Array
     * @property _lt
     * @private
     */
    this._lt = [];
    
    /**
     * Lookahead token buffer index.
     * @type int
     * @property _ltIndex
     * @private
     */
    this._ltIndex = 0;
    
    this._ltIndexCache = [];
}

/**
 * Accepts an array of token information and outputs
 * an array of token data containing key-value mappings
 * and matching functions that the TokenStream needs.
 * @param {Array} tokens An array of token descriptors.
 * @return {Array} An array of processed token data.
 * @method createTokenData
 * @static
 */
TokenStreamBase.createTokenData = function(tokens){

    var nameMap     = [],
        typeMap     = {},
        tokenData     = tokens.concat([]),
        i            = 0,
        len            = tokenData.length+1;
    
    tokenData.UNKNOWN = -1;
    tokenData.unshift({name:"EOF"});

    for (; i < len; i++){
        nameMap.push(tokenData[i].name);
        tokenData[tokenData[i].name] = i;
        if (tokenData[i].text){
            typeMap[tokenData[i].text] = i;
        }
    }
    
    tokenData.name = function(tt){
        return nameMap[tt];
    };
    
    tokenData.type = function(c){
        return typeMap[c];
    };
    
    return tokenData;
};

TokenStreamBase.prototype = {

    //restore constructor
    constructor: TokenStreamBase,    
    
    tokens: [],
    //-------------------------------------------------------------------------
    // Matching methods
    //-------------------------------------------------------------------------
    
    /**
     * Determines if the next token matches the given token type.
     * If so, that token is consumed; if not, the token is placed
     * back onto the token stream. You can pass in any number of
     * token types and this will return true if any of the token
     * types is found.
     * @param {int|int[]} tokenTypes Either a single token type or an array of
     *      token types that the next token might be. If an array is passed,
     *      it's assumed that the token can be any of these.
     * @param {variant} channel (Optional) The channel to read from. If not
     *      provided, reads from the default (unnamed) channel.
     * @return {Boolean} True if the token type matches, false if not.
     * @method match
     */
    match: function(tokenTypes, channel){
    
        //always convert to an array, makes things easier
        if (!(tokenTypes instanceof Array)){
            tokenTypes = [tokenTypes];
        }
                
        var tt  = this.get(channel),
            i   = 0,
            len = tokenTypes.length;
            
        while(i < len){
            if (tt == tokenTypes[i++]){
                return true;
            }
        }
        
        //no match found, put the token back
        this.unget();
        return false;
    },    
    
    /**
     * Determines if the next token matches the given token type.
     * If so, that token is consumed; if not, an error is thrown.
     * @param {int|int[]} tokenTypes Either a single token type or an array of
     *      token types that the next token should be. If an array is passed,
     *      it's assumed that the token must be one of these.
     * @param {variant} channel (Optional) The channel to read from. If not
     *      provided, reads from the default (unnamed) channel.
     * @return {void}
     * @method mustMatch
     */    
    mustMatch: function(tokenTypes, channel){

        var token;

        //always convert to an array, makes things easier
        if (!(tokenTypes instanceof Array)){
            tokenTypes = [tokenTypes];
        }

        if (!this.match.apply(this, arguments)){    
            token = this.LT(1);
            throw new SyntaxError("Expected " + this._tokenData[tokenTypes[0]].name + 
                " at line " + token.startLine + ", col " + token.startCol + ".", token.startLine, token.startCol);
        }
    },
    
    //-------------------------------------------------------------------------
    // Consuming methods
    //-------------------------------------------------------------------------
    
    /**
     * Keeps reading from the token stream until either one of the specified
     * token types is found or until the end of the input is reached.
     * @param {int|int[]} tokenTypes Either a single token type or an array of
     *      token types that the next token should be. If an array is passed,
     *      it's assumed that the token must be one of these.
     * @param {variant} channel (Optional) The channel to read from. If not
     *      provided, reads from the default (unnamed) channel.
     * @return {void}
     * @method advance
     */
    advance: function(tokenTypes, channel){
        
        while(this.LA(0) !== 0 && !this.match(tokenTypes, channel)){
            this.get();
        }

        return this.LA(0);    
    },
    
    /**
     * Consumes the next token from the token stream. 
     * @return {int} The token type of the token that was just consumed.
     * @method get
     */      
    get: function(channel){
    
        var tokenInfo   = this._tokenData,
            reader      = this._reader,
            value,
            i           =0,
            len         = tokenInfo.length,
            found       = false,
            token,
            info;
            
        //check the lookahead buffer first
        if (this._lt.length && this._ltIndex >= 0 && this._ltIndex < this._lt.length){  
                           
            i++;
            this._token = this._lt[this._ltIndex++];
            info = tokenInfo[this._token.type];
            
            //obey channels logic
            while((info.channel !== undefined && channel !== info.channel) &&
                    this._ltIndex < this._lt.length){
                this._token = this._lt[this._ltIndex++];
                info = tokenInfo[this._token.type];
                i++;
            }
            
            //here be dragons
            if ((info.channel === undefined || channel === info.channel) &&
                    this._ltIndex <= this._lt.length){
                this._ltIndexCache.push(i);
                return this._token.type;
            }
        }
        
        //call token retriever method
        token = this._getToken();

        //if it should be hidden, don't save a token
        if (token.type > -1 && !tokenInfo[token.type].hide){
                     
            //apply token channel
            token.channel = tokenInfo[token.type].channel;
         
            //save for later
            this._token = token;
            this._lt.push(token);

            //save space that will be moved (must be done before array is truncated)
            this._ltIndexCache.push(this._lt.length - this._ltIndex + i);  
        
            //keep the buffer under 5 items
            if (this._lt.length > 5){
                this._lt.shift();                
            }
            
            //also keep the shift buffer under 5 items
            if (this._ltIndexCache.length > 5){
                this._ltIndexCache.shift();
            }
                
            //update lookahead index
            this._ltIndex = this._lt.length;
        }
            
        /*
         * Skip to the next token if:
         * 1. The token type is marked as hidden.
         * 2. The token type has a channel specified and it isn't the current channel.
         */
        info = tokenInfo[token.type];
        if (info && 
                (info.hide || 
                (info.channel !== undefined && channel !== info.channel))){
            return this.get(channel);
        } else {
            //return just the type
            return token.type;
        }
    },
    
    /**
     * Looks ahead a certain number of tokens and returns the token type at
     * that position. This will throw an error if you lookahead past the
     * end of input, past the size of the lookahead buffer, or back past
     * the first token in the lookahead buffer.
     * @param {int} The index of the token type to retrieve. 0 for the
     *      current token, 1 for the next, -1 for the previous, etc.
     * @return {int} The token type of the token in the given position.
     * @method LA
     */
    LA: function(index){
        var total = index,
            tt;
        if (index > 0){
            //TODO: Store 5 somewhere
            if (index > 5){
                throw new Error("Too much lookahead.");
            }
        
            //get all those tokens
            while(total){
                tt = this.get();   
                total--;                            
            }
            
            //unget all those tokens
            while(total < index){
                this.unget();
                total++;
            }
        } else if (index < 0){
        
            if(this._lt[this._ltIndex+index]){
                tt = this._lt[this._ltIndex+index].type;
            } else {
                throw new Error("Too much lookbehind.");
            }
        
        } else {
            tt = this._token.type;
        }
        
        return tt;
    
    },
    
    /**
     * Looks ahead a certain number of tokens and returns the token at
     * that position. This will throw an error if you lookahead past the
     * end of input, past the size of the lookahead buffer, or back past
     * the first token in the lookahead buffer.
     * @param {int} The index of the token type to retrieve. 0 for the
     *      current token, 1 for the next, -1 for the previous, etc.
     * @return {Object} The token of the token in the given position.
     * @method LA
     */    
    LT: function(index){
    
        //lookahead first to prime the token buffer
        this.LA(index);
        
        //now find the token, subtract one because _ltIndex is already at the next index
        return this._lt[this._ltIndex+index-1];    
    },
    
    /**
     * Returns the token type for the next token in the stream without 
     * consuming it.
     * @return {int} The token type of the next token in the stream.
     * @method peek
     */
    peek: function(){
        return this.LA(1);
    },
    
    /**
     * Returns the actual token object for the last consumed token.
     * @return {Token} The token object for the last consumed token.
     * @method token
     */
    token: function(){
        return this._token;
    },
    
    /**
     * Returns the name of the token for the given token type.
     * @param {int} tokenType The type of token to get the name of.
     * @return {String} The name of the token or "UNKNOWN_TOKEN" for any
     *      invalid token type.
     * @method tokenName
     */
    tokenName: function(tokenType){
        if (tokenType < 0 || tokenType > this._tokenData.length){
            return "UNKNOWN_TOKEN";
        } else {
            return this._tokenData[tokenType].name;
        }
    },
    
    /**
     * Returns the token type value for the given token name.
     * @param {String} tokenName The name of the token whose value should be returned.
     * @return {int} The token type value for the given token name or -1
     *      for an unknown token.
     * @method tokenName
     */    
    tokenType: function(tokenName){
        return this._tokenData[tokenName] || -1;
    },
    
    /**
     * Returns the last consumed token to the token stream.
     * @method unget
     */      
    unget: function(){
        //if (this._ltIndex > -1){
        if (this._ltIndexCache.length){
            this._ltIndex -= this._ltIndexCache.pop();//--;
            this._token = this._lt[this._ltIndex - 1];
        } else {
            throw new Error("Too much lookahead.");
        }
    }

};




parserlib.util = {
StringReader: StringReader,
SyntaxError : SyntaxError,
SyntaxUnit  : SyntaxUnit,
EventTarget : EventTarget,
TokenStreamBase : TokenStreamBase
};
})();


/*
Parser-Lib
Copyright (c) 2009-2011 Nicholas C. Zakas. All rights reserved.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.

*/
/* Version v0.2.3, Build time: 19-June-2013 11:16:15 */
(function(){
var EventTarget = parserlib.util.EventTarget,
TokenStreamBase = parserlib.util.TokenStreamBase,
StringReader = parserlib.util.StringReader,
SyntaxError = parserlib.util.SyntaxError,
SyntaxUnit  = parserlib.util.SyntaxUnit;


var Colors = {
    aliceblue       :"#f0f8ff",
    antiquewhite    :"#faebd7",
    aqua            :"#00ffff",
    aquamarine      :"#7fffd4",
    azure           :"#f0ffff",
    beige           :"#f5f5dc",
    bisque          :"#ffe4c4",
    black           :"#000000",
    blanchedalmond  :"#ffebcd",
    blue            :"#0000ff",
    blueviolet      :"#8a2be2",
    brown           :"#a52a2a",
    burlywood       :"#deb887",
    cadetblue       :"#5f9ea0",
    chartreuse      :"#7fff00",
    chocolate       :"#d2691e",
    coral           :"#ff7f50",
    cornflowerblue  :"#6495ed",
    cornsilk        :"#fff8dc",
    crimson         :"#dc143c",
    cyan            :"#00ffff",
    darkblue        :"#00008b",
    darkcyan        :"#008b8b",
    darkgoldenrod   :"#b8860b",
    darkgray        :"#a9a9a9",
    darkgreen       :"#006400",
    darkkhaki       :"#bdb76b",
    darkmagenta     :"#8b008b",
    darkolivegreen  :"#556b2f",
    darkorange      :"#ff8c00",
    darkorchid      :"#9932cc",
    darkred         :"#8b0000",
    darksalmon      :"#e9967a",
    darkseagreen    :"#8fbc8f",
    darkslateblue   :"#483d8b",
    darkslategray   :"#2f4f4f",
    darkturquoise   :"#00ced1",
    darkviolet      :"#9400d3",
    deeppink        :"#ff1493",
    deepskyblue     :"#00bfff",
    dimgray         :"#696969",
    dodgerblue      :"#1e90ff",
    firebrick       :"#b22222",
    floralwhite     :"#fffaf0",
    forestgreen     :"#228b22",
    fuchsia         :"#ff00ff",
    gainsboro       :"#dcdcdc",
    ghostwhite      :"#f8f8ff",
    gold            :"#ffd700",
    goldenrod       :"#daa520",
    gray            :"#808080",
    green           :"#008000",
    greenyellow     :"#adff2f",
    honeydew        :"#f0fff0",
    hotpink         :"#ff69b4",
    indianred       :"#cd5c5c",
    indigo          :"#4b0082",
    ivory           :"#fffff0",
    khaki           :"#f0e68c",
    lavender        :"#e6e6fa",
    lavenderblush   :"#fff0f5",
    lawngreen       :"#7cfc00",
    lemonchiffon    :"#fffacd",
    lightblue       :"#add8e6",
    lightcoral      :"#f08080",
    lightcyan       :"#e0ffff",
    lightgoldenrodyellow  :"#fafad2",
    lightgray       :"#d3d3d3",
    lightgreen      :"#90ee90",
    lightpink       :"#ffb6c1",
    lightsalmon     :"#ffa07a",
    lightseagreen   :"#20b2aa",
    lightskyblue    :"#87cefa",
    lightslategray  :"#778899",
    lightsteelblue  :"#b0c4de",
    lightyellow     :"#ffffe0",
    lime            :"#00ff00",
    limegreen       :"#32cd32",
    linen           :"#faf0e6",
    magenta         :"#ff00ff",
    maroon          :"#800000",
    mediumaquamarine:"#66cdaa",
    mediumblue      :"#0000cd",
    mediumorchid    :"#ba55d3",
    mediumpurple    :"#9370d8",
    mediumseagreen  :"#3cb371",
    mediumslateblue :"#7b68ee",
    mediumspringgreen   :"#00fa9a",
    mediumturquoise :"#48d1cc",
    mediumvioletred :"#c71585",
    midnightblue    :"#191970",
    mintcream       :"#f5fffa",
    mistyrose       :"#ffe4e1",
    moccasin        :"#ffe4b5",
    navajowhite     :"#ffdead",
    navy            :"#000080",
    oldlace         :"#fdf5e6",
    olive           :"#808000",
    olivedrab       :"#6b8e23",
    orange          :"#ffa500",
    orangered       :"#ff4500",
    orchid          :"#da70d6",
    palegoldenrod   :"#eee8aa",
    palegreen       :"#98fb98",
    paleturquoise   :"#afeeee",
    palevioletred   :"#d87093",
    papayawhip      :"#ffefd5",
    peachpuff       :"#ffdab9",
    peru            :"#cd853f",
    pink            :"#ffc0cb",
    plum            :"#dda0dd",
    powderblue      :"#b0e0e6",
    purple          :"#800080",
    red             :"#ff0000",
    rosybrown       :"#bc8f8f",
    royalblue       :"#4169e1",
    saddlebrown     :"#8b4513",
    salmon          :"#fa8072",
    sandybrown      :"#f4a460",
    seagreen        :"#2e8b57",
    seashell        :"#fff5ee",
    sienna          :"#a0522d",
    silver          :"#c0c0c0",
    skyblue         :"#87ceeb",
    slateblue       :"#6a5acd",
    slategray       :"#708090",
    snow            :"#fffafa",
    springgreen     :"#00ff7f",
    steelblue       :"#4682b4",
    tan             :"#d2b48c",
    teal            :"#008080",
    thistle         :"#d8bfd8",
    tomato          :"#ff6347",
    turquoise       :"#40e0d0",
    violet          :"#ee82ee",
    wheat           :"#f5deb3",
    white           :"#ffffff",
    whitesmoke      :"#f5f5f5",
    yellow          :"#ffff00",
    yellowgreen     :"#9acd32",
    //CSS2 system colors http://www.w3.org/TR/css3-color/#css2-system
    activeBorder        :"Active window border.",
    activecaption       :"Active window caption.",
    appworkspace        :"Background color of multiple document interface.",
    background          :"Desktop background.",
    buttonface          :"The face background color for 3-D elements that appear 3-D due to one layer of surrounding border.",
    buttonhighlight     :"The color of the border facing the light source for 3-D elements that appear 3-D due to one layer of surrounding border.",
    buttonshadow        :"The color of the border away from the light source for 3-D elements that appear 3-D due to one layer of surrounding border.",
    buttontext          :"Text on push buttons.",
    captiontext         :"Text in caption, size box, and scrollbar arrow box.",
    graytext            :"Grayed (disabled) text. This color is set to #000 if the current display driver does not support a solid gray color.",
    highlight           :"Item(s) selected in a control.",
    highlighttext       :"Text of item(s) selected in a control.",
    inactiveborder      :"Inactive window border.",
    inactivecaption     :"Inactive window caption.",
    inactivecaptiontext :"Color of text in an inactive caption.",
    infobackground      :"Background color for tooltip controls.",
    infotext            :"Text color for tooltip controls.",
    menu                :"Menu background.",
    menutext            :"Text in menus.",
    scrollbar           :"Scroll bar gray area.",
    threeddarkshadow    :"The color of the darker (generally outer) of the two borders away from the light source for 3-D elements that appear 3-D due to two concentric layers of surrounding border.",
    threedface          :"The face background color for 3-D elements that appear 3-D due to two concentric layers of surrounding border.",
    threedhighlight     :"The color of the lighter (generally outer) of the two borders facing the light source for 3-D elements that appear 3-D due to two concentric layers of surrounding border.",
    threedlightshadow   :"The color of the darker (generally inner) of the two borders facing the light source for 3-D elements that appear 3-D due to two concentric layers of surrounding border.",
    threedshadow        :"The color of the lighter (generally inner) of the two borders away from the light source for 3-D elements that appear 3-D due to two concentric layers of surrounding border.",
    window              :"Window background.",
    windowframe         :"Window frame.",
    windowtext          :"Text in windows."
};

parserlib.util.Colors = Colors;  //ORION 8.0

/*global SyntaxUnit, Parser*/
/**
 * Represents a selector combinator (whitespace, +, >).
 * @namespace parserlib.css
 * @class Combinator
 * @extends parserlib.util.SyntaxUnit
 * @constructor
 * @param {String} text The text representation of the unit. 
 * @param {int} line The line of text on which the unit resides.
 * @param {int} col The column of text on which the unit resides.
 */
function Combinator(text, line, col){
    
    SyntaxUnit.call(this, text, line, col, Parser.COMBINATOR_TYPE);

    /**
     * The type of modifier.
     * @type String
     * @property type
     */
    this.type = "unknown";
    
    //pretty simple
    if (/^\s+$/.test(text)){
        this.type = "descendant";
    } else if (text == ">"){
        this.type = "child";
    } else if (text == "+"){
        this.type = "adjacent-sibling";
    } else if (text == "~"){
        this.type = "sibling";
    }

}

Combinator.prototype = new SyntaxUnit();
Combinator.prototype.constructor = Combinator;


/*global SyntaxUnit, Parser*/
/**
 * Represents a media feature, such as max-width:500.
 * @namespace parserlib.css
 * @class MediaFeature
 * @extends parserlib.util.SyntaxUnit
 * @constructor
 * @param {SyntaxUnit} name The name of the feature.
 * @param {SyntaxUnit} value The value of the feature or null if none.
 */
function MediaFeature(name, value){
    
    SyntaxUnit.call(this, "(" + name + (value !== null ? ":" + value : "") + ")", name.startLine, name.startCol, Parser.MEDIA_FEATURE_TYPE);

    /**
     * The name of the media feature
     * @type String
     * @property name
     */
    this.name = name;

    /**
     * The value for the feature or null if there is none.
     * @type SyntaxUnit
     * @property value
     */
    this.value = value;
}

MediaFeature.prototype = new SyntaxUnit();
MediaFeature.prototype.constructor = MediaFeature;


/*global SyntaxUnit, Parser*/
/**
 * Represents an individual media query.
 * @namespace parserlib.css
 * @class MediaQuery
 * @extends parserlib.util.SyntaxUnit
 * @constructor
 * @param {String} modifier The modifier "not" or "only" (or null).
 * @param {String} mediaType The type of media (i.e., "print").
 * @param {Array} parts Array of selectors parts making up this selector.
 * @param {int} line The line of text on which the unit resides.
 * @param {int} col The column of text on which the unit resides.
 */
function MediaQuery(modifier, mediaType, features, line, col){
    
    SyntaxUnit.call(this, (modifier ? modifier + " ": "") + (mediaType ? mediaType : "") + (mediaType && features.length > 0 ? " and " : "") + features.join(" and "), line, col, Parser.MEDIA_QUERY_TYPE); 

    /**
     * The media modifier ("not" or "only")
     * @type String
     * @property modifier
     */
    this.modifier = modifier;

    /**
     * The mediaType (i.e., "print")
     * @type String
     * @property mediaType
     */
    this.mediaType = mediaType;    
    
    /**
     * The parts that make up the selector.
     * @type Array
     * @property features
     */
    this.features = features;

}

MediaQuery.prototype = new SyntaxUnit();
MediaQuery.prototype.constructor = MediaQuery;


/*global Tokens, TokenStream, SyntaxError, Properties, Validation, ValidationError, SyntaxUnit,
    PropertyValue, PropertyValuePart, SelectorPart, SelectorSubPart, Selector,
    PropertyName, Combinator, MediaFeature, MediaQuery, EventTarget */

/**
 * A CSS3 parser.
 * @namespace parserlib.css
 * @class Parser
 * @constructor
 * @param {Object} options (Optional) Various options for the parser:
 *      starHack (true|false) to allow IE6 star hack as valid,
 *      underscoreHack (true|false) to interpret leading underscores
 *      as IE6-7 targeting for known properties, ieFilters (true|false)
 *      to indicate that IE < 8 filters should be accepted and not throw
 *      syntax errors.
 */
function Parser(options){

    //inherit event functionality
    EventTarget.call(this);


    this.options = options || {};

    this._tokenStream = null;
}

//Static constants
Parser.DEFAULT_TYPE = 0;
Parser.COMBINATOR_TYPE = 1;
Parser.MEDIA_FEATURE_TYPE = 2;
Parser.MEDIA_QUERY_TYPE = 3;
Parser.PROPERTY_NAME_TYPE = 4;
Parser.PROPERTY_VALUE_TYPE = 5;
Parser.PROPERTY_VALUE_PART_TYPE = 6;
Parser.SELECTOR_TYPE = 7;
Parser.SELECTOR_PART_TYPE = 8;
Parser.SELECTOR_SUB_PART_TYPE = 9;

Parser.prototype = function(){

    var proto = new EventTarget(),  //new prototype
        prop,
        additions =  {
            //ORION 8.0 AST generation junk
            ast: Object.create(null),
            _aststack: [],
            startNode: function(name, start, parentprop) {
              var node = Object.create(null);
              node.type = name;
              node.range = [-1,-1];
              if(start !== null && start !== undefined && start > -1) {
                node.range[0] = start;
              }
              this.addToParent(node, parentprop);
              if(node.type !== 'StyleSheet') {
                this.setNodeStart(this.ast, start);
              }
              this._aststack.push(node);
              return node;
            },
            endNode: function(node, end) {
                node.range[1] = end;
                this._aststack.pop();
                return node;
            },
            setNodeStart: function(node, from) {
                if(node.range[0] < 0) {
                    node.range[0] = from;
                }  
            },
            addToParent: function(node, parentprop) {
                if(typeof parentprop === "string") {
                  if(this._aststack.length > 0) {
                        var p = this._aststack[this._aststack.length-1];
                        if(Array.isArray(p[parentprop])) {
                            p[parentprop].push(node);
                        } else {
                            p[parentprop] = node;
                        }
                  }
              }
            },
            idNode: function(name, token) {
               var node = Object.create(null);
               node.type = name;
               node.range = token.range;
               return node;
            },
            //restore constructor
            constructor: Parser,

            //instance constants - yuck
            DEFAULT_TYPE : 0,
            COMBINATOR_TYPE : 1,
            MEDIA_FEATURE_TYPE : 2,
            MEDIA_QUERY_TYPE : 3,
            PROPERTY_NAME_TYPE : 4,
            PROPERTY_VALUE_TYPE : 5,
            PROPERTY_VALUE_PART_TYPE : 6,
            SELECTOR_TYPE : 7,
            SELECTOR_PART_TYPE : 8,
            SELECTOR_SUB_PART_TYPE : 9,

            //-----------------------------------------------------------------
            // Grammar
            //-----------------------------------------------------------------

            _stylesheet: function(){
				ast = Object.create(null);
            	_aststack = [];
                /*
                 * stylesheet
                 *  : [ CHARSET_SYM S* STRING S* ';' ]?
                 *    [S|CDO|CDC]* [ import [S|CDO|CDC]* ]*
                 *    [ namespace [S|CDO|CDC]* ]*
                 *    [ [ ruleset | media | page | font_face | keyframes ] [S|CDO|CDC]* ]*
                 *  ;
                 */

                var tokenStream = this._tokenStream,
                    charset     = null,
                    count,
                    token,
                    tt;

                this.fire("startstylesheet");
                this.ast = this.startNode('StyleSheet', 0); //ORION 8.0
                this.ast.body = [];
                //try to read character set
                this._charset();
                
                this._skipCruft();
                
                //try to read imports - may be more than one
                while (tokenStream.peek() == Tokens.IMPORT_SYM){
                    this._import(); //ORION 8.0
                    this._skipCruft();
                }

                //try to read namespaces - may be more than one
                while (tokenStream.peek() == Tokens.NAMESPACE_SYM){
                    this._namespace(true); //ORION 8.0
                    this._skipCruft();
                }

                //get the next token
                tt = tokenStream.peek();
				
                //try to read the rest
                while(tt > Tokens.EOF){

                    try {

                        switch(tt){
                            case Tokens.MEDIA_SYM:
                                this._media();
                                this._skipCruft();
                                break;
                            case Tokens.PAGE_SYM:
                                this._page();
                                this._skipCruft();
                                break;
                            case Tokens.FONT_FACE_SYM:
                                this._font_face();
                                this._skipCruft();
                                break;
                            case Tokens.KEYFRAMES_SYM:
                                this._keyframes();
                                this._skipCruft();
                                break;
                            case Tokens.VIEWPORT_SYM:
                                this._viewport();
                                this._skipCruft();
                                break;
                            case Tokens.UNKNOWN_SYM:  //unknown @ rule
                                tokenStream.get();
                                if (!this.options.strict){

                                    //fire error event
                                    this.fire({
                                        type:       "error",
                                        error:      null,
                                        message:    "Unknown @ rule: " + tokenStream.LT(0).value + ".",
                                        line:       tokenStream.LT(0).startLine,
                                        col:        tokenStream.LT(0).startCol
                                    });

                                    //skip braces
                                    count=0;
                                    while (tokenStream.advance([Tokens.LBRACE, Tokens.RBRACE]) == Tokens.LBRACE){
                                        count++;    //keep track of nesting depth
                                    }

                                    while(count){
                                        tokenStream.advance([Tokens.RBRACE]);
                                        count--;
                                    }

                                } else {
                                    //not a syntax error, rethrow it
                                    throw new SyntaxError("Unknown @ rule.", tokenStream.LT(0).startLine, tokenStream.LT(0).startCol);
                                }
                                break;
                            case Tokens.S:
                                this._readWhitespace();
                                break;
                            default:
                                if(!this._ruleset()){

                                    //error handling for known issues
                                    switch(tt){
                                        case Tokens.CHARSET_SYM:
                                            token = tokenStream.LT(1);
                                            this._charset(false);
                                            throw new SyntaxError("@charset not allowed here.", token.startLine, token.startCol);
                                        case Tokens.IMPORT_SYM:
                                            token = tokenStream.LT(1);
                                            this._import(false);
                                            throw new SyntaxError("@import not allowed here.", token.startLine, token.startCol);
                                        case Tokens.NAMESPACE_SYM:
                                            token = tokenStream.LT(1);
                                            this._namespace(false);
                                            throw new SyntaxError("@namespace not allowed here.", token.startLine, token.startCol);
                                        default:
                                            tokenStream.get();  //get the last token
                                            this._unexpectedToken(tokenStream.token());
                                    }

                                }
                        }
                    } catch(ex) {
                        if (ex instanceof SyntaxError && !this.options.strict){
                            this.fire({
                                type:       "error",
                                error:      ex,
                                message:    ex.message,
                                line:       ex.line,
                                col:        ex.col
                            });
                        } else {
                            throw ex;
                        }
                    }

                    tt = tokenStream.peek();
                }

                if (tt != Tokens.EOF){
                    this._unexpectedToken(tokenStream.token());
                }
                this.endNode(this.ast, tokenStream.curr().range[1]); //ORION 8.0
                this.fire("endstylesheet");
            },

            _charset: function(emit){
                var tokenStream = this._tokenStream,
                    charset,
                    token,
                    line,
                    col;
                if (tokenStream.match(Tokens.CHARSET_SYM)){
                    var node = this.startNode('Charset', tokenStream.curr().range[0], 'body'); //ORION 8.0
                    line = tokenStream.token().startLine;
                    col = tokenStream.token().startCol;

                    this._readWhitespace();
                    tokenStream.mustMatch(Tokens.STRING);

                    token = tokenStream.token();
                    charset = token.value;

                    this._readWhitespace();
                    tokenStream.mustMatch(Tokens.SEMICOLON);
                    
                    this.endNode(node, tokenStream.curr().range[1]);
                    if (emit !== false){
                        this.fire({
                            type:   "charset",
                            charset:charset,
                            line:   line,
                            col:    col
                        });
                    }
                }
            },

            _import: function(emit){
                /*
                 * import
                 *   : IMPORT_SYM S*
                 *    [STRING|URI] S* media_query_list? ';' S*
                 */

                var tokenStream = this._tokenStream,
                    tt,
                    uri,
                    importToken,
                    mediaList   = [];
                //read import symbol
                tokenStream.mustMatch(Tokens.IMPORT_SYM);
                importToken = tokenStream.token();
                var node = this.startNode('Import', tokenStream.curr().range[0], 'body');
                this._readWhitespace();

                tokenStream.mustMatch([Tokens.STRING, Tokens.URI]);

                //grab the URI value
                tt = tokenStream.token(); //ORION 8.0
                node.uri = this.idNode('Uri', tokenStream.curr());
                uri = tt.value.replace(/(?:url\()?["']([^"']+)["']\)?/, "$1");
                
                this._readWhitespace();
                mediaList = this._media_query_list();
                
                //must end with a semicolon
                tokenStream.mustMatch(Tokens.SEMICOLON);
                this._readWhitespace();
                
                if (emit !== false){
                    this.fire({
                        type:   "import",
                        uri:    uri,
                        media:  mediaList,
                        line:   importToken.startLine,
                        col:    importToken.startCol
                    });
                }
                this.endNode(node, tokenStream.curr().range[1]); //ORION 8.0
            },

            _namespace: function(emit){
                /*
                 * namespace
                 *   : NAMESPACE_SYM S* [namespace_prefix S*]? [STRING|URI] S* ';' S*
                 */

                var tokenStream = this._tokenStream,
                    line,
                    col,
                    prefix,
                    uri;

                //read import symbol
                tokenStream.mustMatch(Tokens.NAMESPACE_SYM);
                var node = this.startNode('Namespace', tokenStream.curr().range[0], 'body'); //ORION 8.0
                line = tokenStream.token().startLine;
                col = tokenStream.token().startCol;
                this._readWhitespace();

                //it's a namespace prefix - no _namespace_prefix() method because it's just an IDENT
                if (tokenStream.match(Tokens.IDENT)){
                    prefix = tokenStream.token().value;
                    node.prefix = prefix;  //ORION 8.0
                    this._readWhitespace();
                }

                tokenStream.mustMatch([Tokens.STRING, Tokens.URI]);
                /*if (!tokenStream.match(Tokens.STRING)){
                    tokenStream.mustMatch(Tokens.URI);
                }*/

                //grab the URI value
                node.uri = this.idNode('Uri', tokenStream.curr());  //ORION 8.0
                uri = tokenStream.token().value.replace(/(?:url\()?["']([^"']+)["']\)?/, "$1");

                this._readWhitespace();

                //must end with a semicolon
                tokenStream.mustMatch(Tokens.SEMICOLON);
                this._readWhitespace();

                if (emit !== false){
                    this.fire({
                        type:   "namespace",
                        prefix: prefix,
                        uri:    uri,
                        line:   line,
                        col:    col
                    });
                }
                this.endNode(node, tokenStream.curr().range[1]); //ORION 8.0
            },

            _media: function(){
                /*
                 * media
                 *   : MEDIA_SYM S* media_query_list S* '{' S* ruleset* '}' S*
                 *   ;
                 */
                var tokenStream     = this._tokenStream,
                    line,
                    col,
                    mediaList;//       = [];

                //look for @media
                tokenStream.mustMatch(Tokens.MEDIA_SYM);
                var node = this.startNode('Media', tokenStream.curr().range[0], 'body'); //ORION 8.0
                line = tokenStream.token().startLine;
                col = tokenStream.token().startCol;

                this._readWhitespace();

                mediaList = this._media_query_list();

                tokenStream.mustMatch(Tokens.LBRACE);
                this._readWhitespace();

                this.fire({
                    type:   "startmedia",
                    media:  mediaList,
                    line:   line,
                    col:    col
                });

                while(true) {
                    if (tokenStream.peek() == Tokens.PAGE_SYM){
                        this._page();
                    } else   if (tokenStream.peek() == Tokens.FONT_FACE_SYM){
                        this._font_face();
                    } else if (!this._ruleset()){
                        break;
                    }
                }

                tokenStream.mustMatch(Tokens.RBRACE);
                this._readWhitespace();

                this.fire({
                    type:   "endmedia",
                    media:  mediaList,
                    line:   line,
                    col:    col
                });
                this.endNode(node, tokenStream.curr().range[1]); //ORION 8.0
            },


            //CSS3 Media Queries
            _media_query_list: function(){
                /*
                 * media_query_list
                 *   : S* [media_query [ ',' S* media_query ]* ]?
                 *   ;
                 */
                var tokenStream = this._tokenStream,
                    mediaList   = [];

                this._readWhitespace();
                var node = this.startNode('MediaQueryList'); //ORION 8.0
                node.queries = [];
                if (tokenStream.peek() == Tokens.IDENT || tokenStream.peek() == Tokens.LPAREN){
                    node.range[0] = tokenStream.curr().range[0];
                    mediaList.push(this._media_query());
                }

                while(tokenStream.match(Tokens.COMMA)){
                    this._readWhitespace();
                    mediaList.push(this._media_query());
                }
                this.endNode(node, tokenStream.curr().range[1])
                if(node.range[0] > -1) {
                    //only add the node if something was parsed
                    this.addToParent(node, 'mediaqueries'); //ORION 8.0
                }
                return mediaList;
            },

            /*
             * Note: "expression" in the grammar maps to the _media_expression
             * method.

             */
            _media_query: function(){
                /*
                 * media_query
                 *   : [ONLY | NOT]? S* media_type S* [ AND S* expression ]*
                 *   | expression [ AND S* expression ]*
                 *   ;
                 */
                var tokenStream = this._tokenStream,
                    type        = null,
                    ident       = null,
                    token       = null,
                    expressions = [];
                var node = this.startNode('MediaQuery');//ORION 8.0
                node.expressions = [];
                if (tokenStream.match(Tokens.IDENT)){
                    ident = tokenStream.token().value.toLowerCase();

                    //since there's no custom tokens for these, need to manually check
                    if (ident != "only" && ident != "not"){
                        tokenStream.unget();
                        ident = null;
                    } else {
                        token = tokenStream.token();
                        this.setNodeStart(node, tokenStream.curr().range[0]); //ORION 8.0
                    }
                }
               
                this._readWhitespace();

                if (tokenStream.peek() == Tokens.IDENT){
                    type = this._media_type();
                    if (token === null){
                        token = tokenStream.token();
                        this.setNodeStart(node, tokenStream.curr().range[0]); //ORION 8.0
                    }
                } else if (tokenStream.peek() == Tokens.LPAREN){
                    if (token === null){
                        token = tokenStream.LT(1);
                    }
                    expressions.push(this._media_expression());
                }

                if (type === null && expressions.length === 0){
                    this.endNode(node); //ORION 8.0 clear it off the stack
                    return null;
                } else {
                    this._readWhitespace();
                    while (tokenStream.match(Tokens.IDENT)){
                        if (tokenStream.token().value.toLowerCase() != "and"){
                            this._unexpectedToken(tokenStream.token());
                        }
                        this._readWhitespace();
                        expressions.push(this._media_expression());
                    }
                }
                this.addToParent(this.endNode(node, tokenStream.curr().range[1]), 'queries'); //ORION 8.0
                return new MediaQuery(ident, type, expressions, token.startLine, token.startCol);
            },

            //CSS3 Media Queries
            _media_type: function(){
                /*
                 * media_type
                 *   : IDENT
                 *   ;
                 */
                return this._media_feature();
            },

            /**
             * Note: in CSS3 Media Queries, this is called "expression".
             * Renamed here to avoid conflict with CSS3 Selectors
             * definition of "expression". Also note that "expr" in the
             * grammar now maps to "expression" from CSS3 selectors.
             * @method _media_expression
             * @private
             */
            _media_expression: function(){
                /*
                 * expression
                 *  : '(' S* media_feature S* [ ':' S* expr ]? ')' S*
                 *  ;
                 */
                var tokenStream = this._tokenStream,
                    feature     = null,
                    token,
                    expression  = null;

                tokenStream.mustMatch(Tokens.LPAREN);
                var node = this.startNode('MediaExpression', tokenStream.curr().range[0]); //ORION 8.0
                feature = this._media_feature();
                node.feature = this.idNode('MediaFeature', tokenStream.curr());
                this._readWhitespace();

                if (tokenStream.match(Tokens.COLON)){
                    this._readWhitespace();
                    token = tokenStream.LT(1);
                    expression = this._expression();
                }

                tokenStream.mustMatch(Tokens.RPAREN);
                this._readWhitespace();
                this.addToParent(this.endNode(node, tokenStream.curr().range[1]), 'expressions'); // ORION 8.0
                return new MediaFeature(feature, (expression ? new SyntaxUnit(expression, token.startLine, token.startCol) : null));
            },

            //CSS3 Media Queries
            _media_feature: function(){
                /*
                 * media_feature
                 *   : IDENT
                 *   ;
                 */
                var tokenStream = this._tokenStream;

                tokenStream.mustMatch(Tokens.IDENT);
                var tok = tokenStream.curr();
                var node = this.startNode('MediaFeature', tok.range[0], 'mediafeature'); //ORION 8.0
                node.value = tok.value;
                this.endNode(node, tok.range[1]);
                return SyntaxUnit.fromToken(tokenStream.token());
            },

            //CSS3 Paged Media
            _page: function(){
                /*
                 * page:
                 *    PAGE_SYM S* IDENT? pseudo_page? S*
                 *    '{' S* [ declaration | margin ]? [ ';' S* [ declaration | margin ]? ]* '}' S*
                 *    ;
                 */
                var tokenStream = this._tokenStream,
                    line,
                    col,
                    identifier  = null,
                    pseudoPage  = null;

                //look for @page
                tokenStream.mustMatch(Tokens.PAGE_SYM);
                var node = this.startNode('Page', tokenStream.curr().range[0], 'body'); //ORION 8.0
                node.declarations = [];
                line = tokenStream.token().startLine;
                col = tokenStream.token().startCol;

                this._readWhitespace();

                if (tokenStream.match(Tokens.IDENT)){
                    identifier = tokenStream.token().value;
                    node.id = this.idNode('id', tokenStream.curr()); //ORION 8.0
                    //The value 'auto' may not be used as a page name and MUST be treated as a syntax error.
                    if (identifier.toLowerCase() === "auto"){
                        this._unexpectedToken(tokenStream.token());
                    }
                }

                //see if there's a colon upcoming
                if (tokenStream.peek() == Tokens.COLON){
                    pseudoPage = this._pseudo_page();
                    node.pseudo = this.idNode('PseudoPage', tokenStream.curr()); //ORION 8.0
                }

                this._readWhitespace();

                this.fire({
                    type:   "startpage",
                    id:     identifier,
                    pseudo: pseudoPage,
                    line:   line,
                    col:    col
                });

                this._readDeclarations(true, true);

                this.fire({
                    type:   "endpage",
                    id:     identifier,
                    pseudo: pseudoPage,
                    line:   line,
                    col:    col
                });
                this.endNode(node, toeknStream.curr().range[1]); //ORION 8.0
            },

            //CSS3 Paged Media
            _margin: function(){
                /*
                 * margin :
                 *    margin_sym S* '{' declaration [ ';' S* declaration? ]* '}' S*
                 *    ;
                 */
                var tokenStream = this._tokenStream,
                    line,
                    col,
                    marginSym   = this._margin_sym();

                if (marginSym){
                    var node = this.startNode('Margin', tokenStream.curr().range[0], 'body'); //ORION 8.0
                    node.declarations = [];
                    line = tokenStream.token().startLine;
                    col = tokenStream.token().startCol;

                    this.fire({
                        type: "startpagemargin",
                        margin: marginSym,
                        line:   line,
                        col:    col
                    });

                    this._readDeclarations(true);

                    this.fire({
                        type: "endpagemargin",
                        margin: marginSym,
                        line:   line,
                        col:    col
                    });
                    this.endNode(node, tokenStream.curr().range[1]); //ORION 8.0
                    return true;
                } else {
                    return false;
                }
            },

            //CSS3 Paged Media
            _margin_sym: function(){

                /*
                 * margin_sym :
                 *    TOPLEFTCORNER_SYM |
                 *    TOPLEFT_SYM |
                 *    TOPCENTER_SYM |
                 *    TOPRIGHT_SYM |
                 *    TOPRIGHTCORNER_SYM |
                 *    BOTTOMLEFTCORNER_SYM |
                 *    BOTTOMLEFT_SYM |
                 *    BOTTOMCENTER_SYM |
                 *    BOTTOMRIGHT_SYM |
                 *    BOTTOMRIGHTCORNER_SYM |
                 *    LEFTTOP_SYM |
                 *    LEFTMIDDLE_SYM |
                 *    LEFTBOTTOM_SYM |
                 *    RIGHTTOP_SYM |
                 *    RIGHTMIDDLE_SYM |
                 *    RIGHTBOTTOM_SYM
                 *    ;
                 */

                var tokenStream = this._tokenStream;

                if(tokenStream.match([Tokens.TOPLEFTCORNER_SYM, Tokens.TOPLEFT_SYM,
                        Tokens.TOPCENTER_SYM, Tokens.TOPRIGHT_SYM, Tokens.TOPRIGHTCORNER_SYM,
                        Tokens.BOTTOMLEFTCORNER_SYM, Tokens.BOTTOMLEFT_SYM,
                        Tokens.BOTTOMCENTER_SYM, Tokens.BOTTOMRIGHT_SYM,
                        Tokens.BOTTOMRIGHTCORNER_SYM, Tokens.LEFTTOP_SYM,
                        Tokens.LEFTMIDDLE_SYM, Tokens.LEFTBOTTOM_SYM, Tokens.RIGHTTOP_SYM,
                        Tokens.RIGHTMIDDLE_SYM, Tokens.RIGHTBOTTOM_SYM]))
                {
                    return SyntaxUnit.fromToken(tokenStream.token());
                } else {
                    return null;
                }

            },

            _pseudo_page: function(){
                /*
                 * pseudo_page
                 *   : ':' IDENT
                 *   ;
                 */

                var tokenStream = this._tokenStream;

                tokenStream.mustMatch(Tokens.COLON);
                tokenStream.mustMatch(Tokens.IDENT);

                //TODO: CSS3 Paged Media says only "left", "center", and "right" are allowed

                return tokenStream.token().value;
            },

            _font_face: function(){
                /*
                 * font_face
                 *   : FONT_FACE_SYM S*
                 *     '{' S* declaration [ ';' S* declaration ]* '}' S*
                 *   ;
                 */
                var tokenStream = this._tokenStream,
                    line,
                    col;

                //look for @page
                tokenStream.mustMatch(Tokens.FONT_FACE_SYM);
                var node = this.startNode('FontFace', tokenStream.curr().range[0], 'body'); //ORION 8.0
                node.declarations = [];
                line = tokenStream.token().startLine;
                col = tokenStream.token().startCol;

                this._readWhitespace();

                this.fire({
                    type:   "startfontface",
                    line:   line,
                    col:    col
                });

                this._readDeclarations(true);
                
                this.endNode(node, tokenStream.curr().range[1]); //ORION 8.0
                this.fire({
                    type:   "endfontface",
                    line:   line,
                    col:    col
                });
            },

            _viewport: function(){
                /*
                 * viewport
                 *   : VIEWPORT_SYM S*
                 *     '{' S* declaration? [ ';' S* declaration? ]* '}' S*
                 *   ;
                 */
                 var tokenStream = this._tokenStream,
                    line,
                    col;

                    tokenStream.mustMatch(Tokens.VIEWPORT_SYM);
                    var node = this.startNode('Viewport', tokenStream.curr().range[0], 'body'); //ORION 8.0
                    node.declarations = [];
                    line = tokenStream.token().startLine;
                    col = tokenStream.token().startCol;

                    this._readWhitespace();

                    this.fire({
                        type:   "startviewport",
                        line:   line,
                        col:    col
                    });

                    this._readDeclarations(true);
                    
                    this.endNode(node, tokenStream.curr().range[1]); //ORION 8.0
                    this.fire({
                        type:   "endviewport",
                        line:   line,
                        col:    col
                    });

            },

            _operator: function(inFunction){

                /*
                 * operator (outside function)
                 *  : '/' S* | ',' S* | /( empty )/
                 * operator (inside function)
                 *  : '/' S* | '+' S* | '*' S* | '-' S* /( empty )/
                 *  ;
                 */

                var tokenStream = this._tokenStream,
                    token       = null;

                if (tokenStream.match([Tokens.SLASH, Tokens.COMMA]) ||
                    (inFunction && tokenStream.match([Tokens.PLUS, Tokens.STAR, Tokens.MINUS]))){
                    token =  tokenStream.token();
                    this._readWhitespace();
                }
                if(token) {
                    var node = this.startNode('Operator', tokenStream.curr().range[0], 'operator'); //ORION 8.0
                    node.value = PropertyValuePart.fromToken(token);
                    this.endNode(node, tokenStream.curr().range[1]);
                }
                return token ? PropertyValuePart.fromToken(token) : null;

            },

            _combinator: function(){

                /*
                 * combinator
                 *  : PLUS S* | GREATER S* | TILDE S* | S+
                 *  ;
                 */

                var tokenStream = this._tokenStream,
                    value       = null,
                    token;

                if(tokenStream.match([Tokens.PLUS, Tokens.GREATER, Tokens.TILDE])){
                    token = tokenStream.token();
                    value = new Combinator(token.value, token.startLine, token.startCol);
                    this._readWhitespace();
                }
                if(value !== null) {
                    var node = this.startNode('Combinator', tokenStream.curr().range[0], 'selector'); //ORION 8.0
                    node.value = value;
                    this.endNode(node, tokenStream.curr().range[0])
                }
                return value;
            },

            _unary_operator: function(){

                /*
                 * unary_operator
                 *  : '-' | '+'
                 *  ;
                 */

                var tokenStream = this._tokenStream;

                if (tokenStream.match([Tokens.MINUS, Tokens.PLUS])){
                    var node = this.startNode('UnaryOperator', tokenStream.curr().range[0], 'operator'); //ORION 8.0
                    node.value = tokenStream.token().value;
                    this.endNode(node, tokenStream.curr().range[1]);
                    return tokenStream.token().value;
                } else {
                    return null;
                }
            },

            _property: function(){
                /*
                 * property
                 *   : IDENT S*
                 *   ;
                 */

                var tokenStream = this._tokenStream,
                    value       = null,
                    hack        = null,
                    tokenValue,
                    token,
                    line,
                    col;
                var node = this.startNode('Property', tokenStream.curr().range[0]); //ORION 8.0
                //check for star hack - throws error if not allowed
                if (tokenStream.peek() == Tokens.STAR && this.options.starHack){
                    tokenStream.get();
                    token = tokenStream.token();
                    hack = token.value;
                    line = token.startLine;
                    col = token.startCol;
                    this.setNodeStart(node, tokenStream.curr().range[0]); //ORION 8.0
                }

                if(tokenStream.match(Tokens.IDENT)){
                    token = tokenStream.token();
                    tokenValue = token.value;
                    
                    this.setNodeStart(node, tokenStream.curr().range[0]); //ORION 8.0
                    //check for underscore hack - no error if not allowed because it's valid CSS syntax
                    if (tokenValue.charAt(0) == "_" && this.options.underscoreHack){
                        hack = "_";
                        tokenValue = tokenValue.substring(1);
                    }
                    node.hack = hack;
                    node.value = tokenValue;
                    value = new PropertyName(tokenValue, hack, (line||token.startLine), (col||token.startCol));
                    this._readWhitespace();
                }
				this.addToParent(this.endNode(node, tokenStream.curr().range[1]), 'properties'); //ORION 8.0
                return value;
            },

            //Augmented with CSS3 Selectors
            _ruleset: function(){
                /*
                 * ruleset
                 *   : selectors_group
                 *     '{' S* declaration? [ ';' S* declaration? ]* '}' S*
                 *   ;
                 */

                var tokenStream = this._tokenStream,
                    tt,
                    selectors;

                var node = this.startNode('Rule', tokenStream.curr().range[1]); //ORION 8.0
                node.declarations = [];
                /*
                 * Error Recovery: If even a single selector fails to parse,
                 * then the entire ruleset should be thrown away.
                 */
                try {
                    selectors = this._selectors_group();
                } catch (ex){
                    if (ex instanceof SyntaxError && !this.options.strict){

                        //fire error event
                        this.fire({
                            type:       "error",
                            error:      ex,
                            message:    ex.message,
                            line:       ex.line,
                            col:        ex.col
                        });

                        //skip over everything until closing brace
                        tt = tokenStream.advance([Tokens.RBRACE]);
                        if (tt == Tokens.RBRACE){
                            //if there's a right brace, the rule is finished so don't do anything
                        } else {
                            //otherwise, rethrow the error because it wasn't handled properly
                            throw ex;
                        }

                    } else {
                        //not a syntax error, rethrow it
                        throw ex;
                    }

                    //trigger parser to continue
                    return true;
                }

                //if it got here, all selectors parsed
                if (selectors){

                    this.fire({
                        type:       "startrule",
                        selectors:  selectors,
                        line:       selectors[0].line,
                        col:        selectors[0].col
                    });

                    this._readDeclarations(true);

                    this.fire({
                        type:       "endrule",
                        selectors:  selectors,
                        line:       selectors[0].line,
                        col:        selectors[0].col
                    });
                    this.addToParent(this.endNode(node, tokenStream.curr().range[1]), 'body');
                }

                return selectors;

            },

            //CSS3 Selectors
            _selectors_group: function(){

                /*
                 * selectors_group
                 *   : selector [ COMMA S* selector ]*
                 *   ;
                 */
                var tokenStream = this._tokenStream,
                    selectors   = [],
                    selector;

                selector = this._selector();
                if (selector !== null){

                    selectors.push(selector);
                    while(tokenStream.match(Tokens.COMMA)){
                        this._readWhitespace();
                        selector = this._selector();
                        if (selector !== null){
                            selectors.push(selector);
                        } else {
                            this._unexpectedToken(tokenStream.LT(1));
                        }
                    }
                }

                return selectors.length ? selectors : null;
            },

            //CSS3 Selectors
            _selector: function(){
                /*
                 * selector
                 *   : simple_selector_sequence [ combinator simple_selector_sequence ]*
                 *   ;
                 */

                var tokenStream = this._tokenStream,
                    selector    = [],
                    nextSelector = null,
                    combinator  = null,
                    ws          = null;

                //if there's no simple selector, then there's no selector
                nextSelector = this._simple_selector_sequence();
                if (nextSelector === null){
                    return null;
                }

                selector.push(nextSelector);

                do {

                    //look for a combinator
                    combinator = this._combinator();

                    if (combinator !== null){
                        selector.push(combinator);
                        nextSelector = this._simple_selector_sequence();

                        //there must be a next selector
                        if (nextSelector === null){
                            this._unexpectedToken(tokenStream.LT(1));
                        } else {

                            //nextSelector is an instance of SelectorPart
                            selector.push(nextSelector);
                        }
                    } else {

                        //if there's not whitespace, we're done
                        if (this._readWhitespace()){

                            //add whitespace separator
                            ws = new Combinator(tokenStream.token().value, tokenStream.token().startLine, tokenStream.token().startCol);

                            //combinator is not required
                            combinator = this._combinator();

                            //selector is required if there's a combinator
                            nextSelector = this._simple_selector_sequence();
                            if (nextSelector === null){
                                if (combinator !== null){
                                    this._unexpectedToken(tokenStream.LT(1));
                                }
                            } else {

                                if (combinator !== null){
                                    selector.push(combinator);
                                } else {
                                    selector.push(ws);
                                }

                                selector.push(nextSelector);
                            }
                        } else {
                            break;
                        }

                    }
                } while(true);

                return new Selector(selector, selector[0].line, selector[0].col);
            },

            //CSS3 Selectors
            _simple_selector_sequence: function(){
                /*
                 * simple_selector_sequence
                 *   : [ type_selector | universal ]
                 *     [ HASH | class | attrib | pseudo | negation ]*
                 *   | [ HASH | class | attrib | pseudo | negation ]+
                 *   ;
                 */

                var tokenStream = this._tokenStream,

                    //parts of a simple selector
                    elementName = null,
                    modifiers   = [],

                    //complete selector text
                    selectorText= "",

                    //the different parts after the element name to search for
                    components  = [
                        //HASH
                        function(){
                            return tokenStream.match(Tokens.HASH) ?
                                    new SelectorSubPart(tokenStream.token().value, "id", tokenStream.token().startLine, tokenStream.token().startCol) :
                                    null;
                        },
                        this._class,
                        this._attrib,
                        this._pseudo,
                        this._negation
                    ],
                    i           = 0,
                    len         = components.length,
                    component   = null,
                    found       = false,
                    line,
                    col;


                //get starting line and column for the selector
                line = tokenStream.LT(1).startLine;
                col = tokenStream.LT(1).startCol;

                elementName = this._type_selector();
                if (!elementName){
                    elementName = this._universal();
                }

                if (elementName !== null){
                    selectorText += elementName;
                }

                while(true){

                    //whitespace means we're done
                    if (tokenStream.peek() === Tokens.S){
                        break;
                    }

                    //check for each component
                    while(i < len && component === null){
                        component = components[i++].call(this);
                    }

                    if (component === null){

                        //we don't have a selector
                        if (selectorText === ""){
                            return null;
                        } else {
                            break;
                        }
                    } else {
                        i = 0;
                        modifiers.push(component);
                        selectorText += component.toString();
                        component = null;
                    }
                }


                return selectorText !== "" ?
                        new SelectorPart(elementName, modifiers, selectorText, line, col) :
                        null;
            },

            //CSS3 Selectors
            _type_selector: function(){
                /*
                 * type_selector
                 *   : [ namespace_prefix ]? element_name
                 *   ;
                 */

                var tokenStream = this._tokenStream,
                    ns          = this._namespace_prefix(),
                    elementName = this._element_name();

                if (!elementName){
                    /*
                     * Need to back out the namespace that was read due to both
                     * type_selector and universal reading namespace_prefix
                     * first. Kind of hacky, but only way I can figure out
                     * right now how to not change the grammar.
                     */
                    if (ns){
                        tokenStream.unget();
                        if (ns.length > 1){
                            tokenStream.unget();
                        }
                    }

                    return null;
                } else {
                    if (ns){
                        elementName.text = ns + elementName.text;
                        elementName.col -= ns.length;
                    }
                    return elementName;
                }
            },

            //CSS3 Selectors
            _class: function(){
                /*
                 * class
                 *   : '.' IDENT
                 *   ;
                 */

                var tokenStream = this._tokenStream,
                    token;

                if (tokenStream.match(Tokens.DOT)){
                    tokenStream.mustMatch(Tokens.IDENT);
                    token = tokenStream.token();
                    return new SelectorSubPart("." + token.value, "class", token.startLine, token.startCol - 1);
                } else {
                    return null;
                }

            },

            //CSS3 Selectors
            _element_name: function(){
                /*
                 * element_name
                 *   : IDENT
                 *   ;
                 */

                var tokenStream = this._tokenStream,
                    token;

                if (tokenStream.match(Tokens.IDENT)){
                    token = tokenStream.token();
                    return new SelectorSubPart(token.value, "elementName", token.startLine, token.startCol);

                } else {
                    return null;
                }
            },

            //CSS3 Selectors
            _namespace_prefix: function(){
                /*
                 * namespace_prefix
                 *   : [ IDENT | '*' ]? '|'
                 *   ;
                 */
                var tokenStream = this._tokenStream,
                    value       = "";

                //va that this is a namespace prefix
                if (tokenStream.LA(1) === Tokens.PIPE || tokenStream.LA(2) === Tokens.PIPE){

                    if(tokenStream.match([Tokens.IDENT, Tokens.STAR])){
                        value += tokenStream.token().value;
                    }

                    tokenStream.mustMatch(Tokens.PIPE);
                    value += "|";

                }

                return value.length ? value : null;
            },

            //CSS3 Selectors
            _universal: function(){
                /*
                 * universal
                 *   : [ namespace_prefix ]? '*'
                 *   ;
                 */
                var tokenStream = this._tokenStream,
                    value       = "",
                    ns;

                ns = this._namespace_prefix();
                if(ns){
                    value += ns;
                }

                if(tokenStream.match(Tokens.STAR)){
                    value += "*";
                }

                return value.length ? value : null;

           },

            //CSS3 Selectors
            _attrib: function(){
                /*
                 * attrib
                 *   : '[' S* [ namespace_prefix ]? IDENT S*
                 *         [ [ PREFIXMATCH |
                 *             SUFFIXMATCH |
                 *             SUBSTRINGMATCH |
                 *             '=' |
                 *             INCLUDES |
                 *             DASHMATCH ] S* [ IDENT | STRING ] S*
                 *         ]? ']'
                 *   ;
                 */

                var tokenStream = this._tokenStream,
                    value       = null,
                    ns,
                    token;

                if (tokenStream.match(Tokens.LBRACKET)){
                    token = tokenStream.token();
                    value = token.value;
                    value += this._readWhitespace();

                    ns = this._namespace_prefix();

                    if (ns){
                        value += ns;
                    }

                    tokenStream.mustMatch(Tokens.IDENT);
                    value += tokenStream.token().value;
                    value += this._readWhitespace();

                    if(tokenStream.match([Tokens.PREFIXMATCH, Tokens.SUFFIXMATCH, Tokens.SUBSTRINGMATCH,
                            Tokens.EQUALS, Tokens.INCLUDES, Tokens.DASHMATCH])){

                        value += tokenStream.token().value;
                        value += this._readWhitespace();

                        tokenStream.mustMatch([Tokens.IDENT, Tokens.STRING]);
                        value += tokenStream.token().value;
                        value += this._readWhitespace();
                    }

                    tokenStream.mustMatch(Tokens.RBRACKET);

                    return new SelectorSubPart(value + "]", "attribute", token.startLine, token.startCol);
                } else {
                    return null;
                }
            },

            //CSS3 Selectors
            _pseudo: function(){

                /*
                 * pseudo
                 *   : ':' ':'? [ IDENT | functional_pseudo ]
                 *   ;
                 */

                var tokenStream = this._tokenStream,
                    pseudo      = null,
                    colons      = ":",
                    line,
                    col;

                if (tokenStream.match(Tokens.COLON)){

                    if (tokenStream.match(Tokens.COLON)){
                        colons += ":";
                    }

                    if (tokenStream.match(Tokens.IDENT)){
                        pseudo = tokenStream.token().value;
                        line = tokenStream.token().startLine;
                        col = tokenStream.token().startCol - colons.length;
                    } else if (tokenStream.peek() == Tokens.FUNCTION){
                        line = tokenStream.LT(1).startLine;
                        col = tokenStream.LT(1).startCol - colons.length;
                        pseudo = this._functional_pseudo();
                    }

                    if (pseudo){
                        pseudo = new SelectorSubPart(colons + pseudo, "pseudo", line, col);
                    }
                }

                return pseudo;
            },

            //CSS3 Selectors
            _functional_pseudo: function(){
                /*
                 * functional_pseudo
                 *   : FUNCTION S* expression ')'
                 *   ;
                */

                var tokenStream = this._tokenStream,
                    value = null;

                if(tokenStream.match(Tokens.FUNCTION)){
                    value = tokenStream.token().value;
                    value += this._readWhitespace();
                    value += this._expression();
                    tokenStream.mustMatch(Tokens.RPAREN);
                    value += ")";
                }

                return value;
            },

            //CSS3 Selectors
            _expression: function(){
                /*
                 * expression
                 *   : [ [ PLUS | '-' | DIMENSION | NUMBER | STRING | IDENT ] S* ]+
                 *   ;
                 */

                var tokenStream = this._tokenStream,
                    value       = "";
                var node = this.startNode('Expression', tokenStream.curr().range[0]); //ORION 8.0
                while(tokenStream.match([Tokens.PLUS, Tokens.MINUS, Tokens.DIMENSION,
                        Tokens.NUMBER, Tokens.STRING, Tokens.IDENT, Tokens.LENGTH,
                        Tokens.FREQ, Tokens.ANGLE, Tokens.TIME,
                        Tokens.RESOLUTION, Tokens.SLASH])){

                    value += tokenStream.token().value;
                    value += this._readWhitespace();
                }
                var val = value.length ? value : null
                node.value = val
                this.addToParent(this.endNode(node, tokenStream.curr().range[1]), 'expression'); //ORION 8.0
                return val;

            },

            //CSS3 Selectors
            _negation: function(){
                /*
                 * negation
                 *   : NOT S* negation_arg S* ')'
                 *   ;
                 */

                var tokenStream = this._tokenStream,
                    line,
                    col,
                    value       = "",
                    arg,
                    subpart     = null;

                if (tokenStream.match(Tokens.NOT)){
                    value = tokenStream.token().value;
                    line = tokenStream.token().startLine;
                    col = tokenStream.token().startCol;
                    value += this._readWhitespace();
                    arg = this._negation_arg();
                    value += arg;
                    value += this._readWhitespace();
                    tokenStream.match(Tokens.RPAREN);
                    value += tokenStream.token().value;

                    subpart = new SelectorSubPart(value, "not", line, col);
                    subpart.args.push(arg);
                }

                return subpart;
            },

            //CSS3 Selectors
            _negation_arg: function(){
                /*
                 * negation_arg
                 *   : type_selector | universal | HASH | class | attrib | pseudo
                 *   ;
                 */

                var tokenStream = this._tokenStream,
                    args        = [
                        this._type_selector,
                        this._universal,
                        function(){
                            return tokenStream.match(Tokens.HASH) ?
                                    new SelectorSubPart(tokenStream.token().value, "id", tokenStream.token().startLine, tokenStream.token().startCol) :
                                    null;
                        },
                        this._class,
                        this._attrib,
                        this._pseudo
                    ],
                    arg         = null,
                    i           = 0,
                    len         = args.length,
                    elementName,
                    line,
                    col,
                    part;

                line = tokenStream.LT(1).startLine;
                col = tokenStream.LT(1).startCol;

                while(i < len && arg === null){

                    arg = args[i].call(this);
                    i++;
                }

                //must be a negation arg
                if (arg === null){
                    this._unexpectedToken(tokenStream.LT(1));
                }

                //it's an element name
                if (arg.type == "elementName"){
                    part = new SelectorPart(arg, [], arg.toString(), line, col);
                } else {
                    part = new SelectorPart(null, [arg], arg.toString(), line, col);
                }

                return part;
            },

            _declaration: function(){

                /*
                 * declaration
                 *   : property ':' S* expr prio?
                 *   | /( empty )/
                 *   ;
                 */

                var tokenStream = this._tokenStream,
                    property    = null,
                    expr        = null,
                    prio        = null,
                    error       = null,
                    invalid     = null,
                    propertyName= "";
				var node = this.startNode('Declaration', this._tokenStream.curr().range[0], 'declarations'); //ORION
                property = this._property();
                if (property !== null){
					node.value = property; //ORION
                    tokenStream.mustMatch(Tokens.COLON);
                    this._readWhitespace();

                    expr = this._expr();

                    //if there's no parts for the value, it's an error
                    if (!expr || expr.length === 0){
                        this._unexpectedToken(tokenStream.LT(1));
                    }

                    prio = this._prio();

                    /*
                     * If hacks should be allowed, then only check the root
                     * property. If hacks should not be allowed, treat
                     * _property or *property as invalid properties.
                     */
                    propertyName = property.toString();
                    if (this.options.starHack && property.hack == "*" ||
                            this.options.underscoreHack && property.hack == "_") {

                        propertyName = property.text;
                    }

                    try {
                        this._validateProperty(propertyName, expr);
                    } catch (ex) {
                        invalid = ex;
                    }

                    this.fire({
                        type:       "property",
                        property:   property,
                        value:      expr,
                        important:  prio,
                        line:       property.line,
                        col:        property.col,
                        invalid:    invalid
                    });
					this.endNode(node, this._tokenStream.curr().range[1]);
                    return true;
                } else {
                	this.endNode(node, this._tokenStream.curr().range[1]);
                    return false;
                }
            },

            _prio: function(){
                /*
                 * prio
                 *   : IMPORTANT_SYM S*
                 *   ;
                 */

                var tokenStream = this._tokenStream,
                    result      = tokenStream.match(Tokens.IMPORTANT_SYM);

                this._readWhitespace();
                return result;
            },

            _expr: function(inFunction){
                /*
                 * expr
                 *   : term [ operator term ]*
                 *   ;
                 */

                var tokenStream = this._tokenStream,
                    values      = [],
					//valueParts	= [],
                    value       = null,
                    operator    = null;

                value = this._term();
                if (value !== null){

                    values.push(value);

                    do {
                        operator = this._operator(inFunction);

                        //if there's an operator, keep building up the value parts
                        if (operator){
                            values.push(operator);
                        } /*else {
                            //if there's not an operator, you have a full value
							values.push(new PropertyValue(valueParts, valueParts[0].line, valueParts[0].col));
							valueParts = [];
						}*/

                        value = this._term();

                        if (value === null){
                            break;
                        } else {
                            values.push(value);
                        }
                    } while(true);
                }

				//cleanup
                /*if (valueParts.length){
                    values.push(new PropertyValue(valueParts, valueParts[0].line, valueParts[0].col));
                }*/

                return values.length > 0 ? new PropertyValue(values, values[0].line, values[0].col) : null;
            },

            _term: function(){

                /*
                 * term
                 *   : unary_operator?
                 *     [ NUMBER S* | PERCENTAGE S* | LENGTH S* | ANGLE S* |
                 *       TIME S* | FREQ S* | function | ie_function ]
                 *   | STRING S* | IDENT S* | URI S* | UNICODERANGE S* | hexcolor
                 *   ;
                 */

                var tokenStream = this._tokenStream,
                    unary       = null,
                    value       = null,
                    token,
                    line,
                    col;

                //returns the operator or null
                unary = this._unary_operator();
                if (unary !== null){
                    line = tokenStream.token().startLine;
                    col = tokenStream.token().startCol;
                }

                //exception for IE filters
                if (tokenStream.peek() == Tokens.IE_FUNCTION && this.options.ieFilters){

                    value = this._ie_function();
                    if (unary === null){
                        line = tokenStream.token().startLine;
                        col = tokenStream.token().startCol;
                    }

                //see if there's a simple match
                } else if (tokenStream.match([Tokens.NUMBER, Tokens.PERCENTAGE, Tokens.LENGTH,
                        Tokens.ANGLE, Tokens.TIME,
                        Tokens.FREQ, Tokens.STRING, Tokens.IDENT, Tokens.URI, Tokens.UNICODE_RANGE])){

                    value = tokenStream.token().value;
                    if (unary === null){
                        line = tokenStream.token().startLine;
                        col = tokenStream.token().startCol;
                    }
                    this._readWhitespace();
                } else {

                    //see if it's a color
                    token = this._hexcolor();
                    if (token === null){

                        //if there's no unary, get the start of the next token for line/col info
                        if (unary === null){
                            line = tokenStream.LT(1).startLine;
                            col = tokenStream.LT(1).startCol;
                        }

                        //has to be a function
                        if (value === null){

                            /*
                             * This checks for alpha(opacity=0) style of IE
                             * functions. IE_FUNCTION only presents progid: style.
                             */
                            if (tokenStream.LA(3) == Tokens.EQUALS && this.options.ieFilters){
                                value = this._ie_function();
                            } else {
                                value = this._function();
                            }
                        }

                        /*if (value === null){
                            return null;
                            //throw new Error("Expected identifier at line " + tokenStream.token().startLine + ", character " +  tokenStream.token().startCol + ".");
                        }*/

                    } else {
                        value = token.value;
                        if (unary === null){
                            line = token.startLine;
                            col = token.startCol;
                        }
                    }

                }

                return value !== null ?
                        new PropertyValuePart(unary !== null ? unary + value : value, line, col) :
                        null;

            },

            _function: function(){

                /*
                 * function
                 *   : FUNCTION S* expr ')' S*
                 *   ;
                 */

                var tokenStream = this._tokenStream,
                    functionText = null,
                    expr        = null,
                    lt;

                if (tokenStream.match(Tokens.FUNCTION)){
                    functionText = tokenStream.token().value;
                    this._readWhitespace();
                    expr = this._expr(true);
                    functionText += expr;

                    //START: Horrible hack in case it's an IE filter
                    if (this.options.ieFilters && tokenStream.peek() == Tokens.EQUALS){
                        do {

                            if (this._readWhitespace()){
                                functionText += tokenStream.token().value;
                            }

                            //might be second time in the loop
                            if (tokenStream.LA(0) == Tokens.COMMA){
                                functionText += tokenStream.token().value;
                            }

                            tokenStream.match(Tokens.IDENT);
                            functionText += tokenStream.token().value;

                            tokenStream.match(Tokens.EQUALS);
                            functionText += tokenStream.token().value;

                            //functionText += this._term();
                            lt = tokenStream.peek();
                            while(lt != Tokens.COMMA && lt != Tokens.S && lt != Tokens.RPAREN){
                                tokenStream.get();
                                functionText += tokenStream.token().value;
                                lt = tokenStream.peek();
                            }
                        } while(tokenStream.match([Tokens.COMMA, Tokens.S]));
                    }

                    //END: Horrible Hack

                    tokenStream.match(Tokens.RPAREN);
                    functionText += ")";
                    this._readWhitespace();
                }

                return functionText;
            },

            _ie_function: function(){

                /* (My own extension)
                 * ie_function
                 *   : IE_FUNCTION S* IDENT '=' term [S* ','? IDENT '=' term]+ ')' S*
                 *   ;
                 */

                var tokenStream = this._tokenStream,
                    functionText = null,
                    expr        = null,
                    lt;

                //IE function can begin like a regular function, too
                if (tokenStream.match([Tokens.IE_FUNCTION, Tokens.FUNCTION])){
                    functionText = tokenStream.token().value;

                    do {

                        if (this._readWhitespace()){
                            functionText += tokenStream.token().value;
                        }

                        //might be second time in the loop
                        if (tokenStream.LA(0) == Tokens.COMMA){
                            functionText += tokenStream.token().value;
                        }

                        tokenStream.match(Tokens.IDENT);
                        functionText += tokenStream.token().value;

                        tokenStream.match(Tokens.EQUALS);
                        functionText += tokenStream.token().value;

                        //functionText += this._term();
                        lt = tokenStream.peek();
                        while(lt != Tokens.COMMA && lt != Tokens.S && lt != Tokens.RPAREN){
                            tokenStream.get();
                            functionText += tokenStream.token().value;
                            lt = tokenStream.peek();
                        }
                    } while(tokenStream.match([Tokens.COMMA, Tokens.S]));

                    tokenStream.match(Tokens.RPAREN);
                    functionText += ")";
                    this._readWhitespace();
                }

                return functionText;
            },

            _hexcolor: function(){
                /*
                 * There is a constraint on the color that it must
                 * have either 3 or 6 hex-digits (i.e., [0-9a-fA-F])
                 * after the "#"; e.g., "#000" is OK, but "#abcd" is not.
                 *
                 * hexcolor
                 *   : HASH S*
                 *   ;
                 */

                var tokenStream = this._tokenStream,
                    token = null,
                    color;

                if(tokenStream.match(Tokens.HASH)){

                    //need to do some validation here

                    token = tokenStream.token();
                    color = token.value;
                    if (!/#[a-f0-9]{3,6}/i.test(color)){
                        throw new SyntaxError("Expected a hex color but found '" + color + "' at line " + token.startLine + ", col " + token.startCol + ".", token.startLine, token.startCol);
                    }
                    this._readWhitespace();
                }

                return token;
            },

            //-----------------------------------------------------------------
            // Animations methods
            //-----------------------------------------------------------------

            _keyframes: function(){

                /*
                 * keyframes:
                 *   : KEYFRAMES_SYM S* keyframe_name S* '{' S* keyframe_rule* '}' {
                 *   ;
                 */
                var tokenStream = this._tokenStream,
                    token,
                    tt,
                    name,
                    prefix = "";

                tokenStream.mustMatch(Tokens.KEYFRAMES_SYM);
                token = tokenStream.token();
                if (/^@\-([^\-]+)\-/.test(token.value)) {
                    prefix = RegExp.$1;
                }

                this._readWhitespace();
                name = this._keyframe_name();

                this._readWhitespace();
                tokenStream.mustMatch(Tokens.LBRACE);

                this.fire({
                    type:   "startkeyframes",
                    name:   name,
                    prefix: prefix,
                    line:   token.startLine,
                    col:    token.startCol
                });

                this._readWhitespace();
                tt = tokenStream.peek();

                //check for key
                while(tt == Tokens.IDENT || tt == Tokens.PERCENTAGE) {
                    this._keyframe_rule();
                    this._readWhitespace();
                    tt = tokenStream.peek();
                }

                this.fire({
                    type:   "endkeyframes",
                    name:   name,
                    prefix: prefix,
                    line:   token.startLine,
                    col:    token.startCol
                });

                this._readWhitespace();
                tokenStream.mustMatch(Tokens.RBRACE);

            },

            _keyframe_name: function(){

                /*
                 * keyframe_name:
                 *   : IDENT
                 *   | STRING
                 *   ;
                 */
                var tokenStream = this._tokenStream,
                    token;

                tokenStream.mustMatch([Tokens.IDENT, Tokens.STRING]);
                return SyntaxUnit.fromToken(tokenStream.token());
            },

            _keyframe_rule: function(){

                /*
                 * keyframe_rule:
                 *   : key_list S*
                 *     '{' S* declaration [ ';' S* declaration ]* '}' S*
                 *   ;
                 */
                var tokenStream = this._tokenStream,
                    token,
                    keyList = this._key_list();

                this.fire({
                    type:   "startkeyframerule",
                    keys:   keyList,
                    line:   keyList[0].line,
                    col:    keyList[0].col
                });

                this._readDeclarations(true);

                this.fire({
                    type:   "endkeyframerule",
                    keys:   keyList,
                    line:   keyList[0].line,
                    col:    keyList[0].col
                });

            },

            _key_list: function(){

                /*
                 * key_list:
                 *   : key [ S* ',' S* key]*
                 *   ;
                 */
                var tokenStream = this._tokenStream,
                    token,
                    key,
                    keyList = [];

                //must be least one key
                keyList.push(this._key());

                this._readWhitespace();

                while(tokenStream.match(Tokens.COMMA)){
                    this._readWhitespace();
                    keyList.push(this._key());
                    this._readWhitespace();
                }

                return keyList;
            },

            _key: function(){
                /*
                 * There is a restriction that IDENT can be only "from" or "to".
                 *
                 * key
                 *   : PERCENTAGE
                 *   | IDENT
                 *   ;
                 */

                var tokenStream = this._tokenStream,
                    token;

                if (tokenStream.match(Tokens.PERCENTAGE)){
                    return SyntaxUnit.fromToken(tokenStream.token());
                } else if (tokenStream.match(Tokens.IDENT)){
                    token = tokenStream.token();

                    if (/from|to/i.test(token.value)){
                        return SyntaxUnit.fromToken(token);
                    }

                    tokenStream.unget();
                }

                //if it gets here, there wasn't a valid token, so time to explode
                this._unexpectedToken(tokenStream.LT(1));
            },

            //-----------------------------------------------------------------
            // Helper methods
            //-----------------------------------------------------------------

            /**
             * Not part of CSS grammar, but useful for skipping over
             * combination of white space and HTML-style comments.
             * @return {void}
             * @method _skipCruft
             * @private
             */
            _skipCruft: function(){
                while(this._tokenStream.match([Tokens.S, Tokens.CDO, Tokens.CDC])){
                    //noop
                }
            },

            /**
             * Not part of CSS grammar, but this pattern occurs frequently
             * in the official CSS grammar. Split out here to eliminate
             * duplicate code.
             * @param {Boolean} checkStart Indicates if the rule should check
             *      for the left brace at the beginning.
             * @param {Boolean} readMargins Indicates if the rule should check
             *      for margin patterns.
             * @return {void}
             * @method _readDeclarations
             * @private
             */
            _readDeclarations: function(checkStart, readMargins){
                /*
                 * Reads the pattern
                 * S* '{' S* declaration [ ';' S* declaration ]* '}' S*
                 * or
                 * S* '{' S* [ declaration | margin ]? [ ';' S* [ declaration | margin ]? ]* '}' S*
                 * Note that this is how it is described in CSS3 Paged Media, but is actually incorrect.
                 * A semicolon is only necessary following a declaration is there's another declaration
                 * or margin afterwards.
                 */
                var tokenStream = this._tokenStream,
                    tt;
                this._readWhitespace();
				var node = this.startNode('Declarations', this._tokenStream.curr().range[0], 'declarations');
				node.declarations = [];
                if (checkStart){
                    tokenStream.mustMatch(Tokens.LBRACE);
                }

                this._readWhitespace();

                try {

                    while(true){

                        if (tokenStream.match(Tokens.SEMICOLON) || (readMargins && this._margin())){
                            //noop
                        } else if (this._declaration()){
                            if (!tokenStream.match(Tokens.SEMICOLON)){
                                break;
                            }
                        } else {
                            break;
                        }

                        //if ((!this._margin() && !this._declaration()) || !tokenStream.match(Tokens.SEMICOLON)){
                        //    break;
                        //}
                        this._readWhitespace();
                    }

                    tokenStream.mustMatch(Tokens.RBRACE);
                    this._readWhitespace();

                } catch (ex) {
                    if (ex instanceof SyntaxError && !this.options.strict){

                        //fire error event
                        this.fire({
                            type:       "error",
                            error:      ex,
                            message:    ex.message,
                            line:       ex.line,
                            col:        ex.col
                        });

                        //see if there's another declaration
                        tt = tokenStream.advance([Tokens.SEMICOLON, Tokens.RBRACE]);
                        if (tt == Tokens.SEMICOLON){
                            //if there's a semicolon, then there might be another declaration
                            this._readDeclarations(false, readMargins);
                        } else if (tt != Tokens.RBRACE){
                            //if there's a right brace, the rule is finished so don't do anything
                            //otherwise, rethrow the error because it wasn't handled properly
                            throw ex;
                        }

                    } else {
                        //not a syntax error, rethrow it
                        throw ex;
                    }
                }
				this.addToParent(node, this.endNode(node, this._tokenStream.curr().range[1]));
            },

            /**
             * In some cases, you can end up with two white space tokens in a
             * row. Instead of making a change in every function that looks for
             * white space, this function is used to match as much white space
             * as necessary.
             * @method _readWhitespace
             * @return {String} The white space if found, empty string if not.
             * @private
             */
            _readWhitespace: function(){

                var tokenStream = this._tokenStream,
                    ws = "";

                while(tokenStream.match(Tokens.S)){
                    ws += tokenStream.token().value;
                }

                return ws;
            },


            /**
             * Throws an error when an unexpected token is found.
             * @param {Object} token The token that was found.
             * @method _unexpectedToken
             * @return {void}
             * @private
             */
            _unexpectedToken: function(token){
                throw new SyntaxError("Unexpected token '" + token.value + "' at line " + token.startLine + ", col " + token.startCol + ".", token.startLine, token.startCol);
            },

            /**
             * Helper method used for parsing subparts of a style sheet.
             * @return {void}
             * @method _verifyEnd
             * @private
             */
            _verifyEnd: function(){
                if (this._tokenStream.LA(1) != Tokens.EOF){
                    this._unexpectedToken(this._tokenStream.LT(1));
                }
            },

            //-----------------------------------------------------------------
            // Validation methods
            //-----------------------------------------------------------------
            _validateProperty: function(property, value){
                Validation.validate(property, value);
            },

            //-----------------------------------------------------------------
            // Parsing methods
            //-----------------------------------------------------------------

            parse: function(input){
                this._tokenStream = new TokenStream(input, Tokens);
                this._stylesheet();
            },

            parseStyleSheet: function(input){
                //just passthrough
                return this.parse(input);
            },

            parseMediaQuery: function(input){
                this._tokenStream = new TokenStream(input, Tokens);
                var result = this._media_query();

                //if there's anything more, then it's an invalid selector
                this._verifyEnd();

                //otherwise return result
                return result;
            },

            /**
             * Parses a property value (everything after the semicolon).
             * @return {parserlib.css.PropertyValue} The property value.
             * @throws parserlib.util.SyntaxError If an unexpected token is found.
             * @method parserPropertyValue
             */
            parsePropertyValue: function(input){

                this._tokenStream = new TokenStream(input, Tokens);
                this._readWhitespace();

                var result = this._expr();

                //okay to have a trailing white space
                this._readWhitespace();

                //if there's anything more, then it's an invalid selector
                this._verifyEnd();

                //otherwise return result
                return result;
            },

            /**
             * Parses a complete CSS rule, including selectors and
             * properties.
             * @param {String} input The text to parser.
             * @return {Boolean} True if the parse completed successfully, false if not.
             * @method parseRule
             */
            parseRule: function(input){
                this._tokenStream = new TokenStream(input, Tokens);

                //skip any leading white space
                this._readWhitespace();

                var result = this._ruleset();

                //skip any trailing white space
                this._readWhitespace();

                //if there's anything more, then it's an invalid selector
                this._verifyEnd();

                //otherwise return result
                return result;
            },

            /**
             * Parses a single CSS selector (no comma)
             * @param {String} input The text to parse as a CSS selector.
             * @return {Selector} An object representing the selector.
             * @throws parserlib.util.SyntaxError If an unexpected token is found.
             * @method parseSelector
             */
            parseSelector: function(input){

                this._tokenStream = new TokenStream(input, Tokens);

                //skip any leading white space
                this._readWhitespace();

                var result = this._selector();

                //skip any trailing white space
                this._readWhitespace();

                //if there's anything more, then it's an invalid selector
                this._verifyEnd();

                //otherwise return result
                return result;
            },

            /**
             * Parses an HTML style attribute: a set of CSS declarations
             * separated by semicolons.
             * @param {String} input The text to parse as a style attribute
             * @return {void}
             * @method parseStyleAttribute
             */
            parseStyleAttribute: function(input){
                input += "}"; // for error recovery in _readDeclarations()
                this._tokenStream = new TokenStream(input, Tokens);
                this._readDeclarations();
            }
        };

    //copy over onto prototype
    for (prop in additions){
        if (additions.hasOwnProperty(prop)){
            proto[prop] = additions[prop];
        }
    }

    return proto;
}();


/*
nth
  : S* [ ['-'|'+']? INTEGER? {N} [ S* ['-'|'+'] S* INTEGER ]? |
         ['-'|'+']? INTEGER | {O}{D}{D} | {E}{V}{E}{N} ] S*
  ;
*/

/*global Validation, ValidationTypes, ValidationError*/
var Properties = {

    //A
    "alignment-adjust"              : "auto | baseline | before-edge | text-before-edge | middle | central | after-edge | text-after-edge | ideographic | alphabetic | hanging | mathematical | <percentage> | <length>",
    "alignment-baseline"            : "baseline | use-script | before-edge | text-before-edge | after-edge | text-after-edge | central | middle | ideographic | alphabetic | hanging | mathematical",
    "animation"                     : 1,
    "animation-delay"               : { multi: "<time>", comma: true },
    "animation-direction"           : { multi: "normal | alternate", comma: true },
    "animation-duration"            : { multi: "<time>", comma: true },
    "animation-iteration-count"     : { multi: "<number> | infinite", comma: true },
    "animation-name"                : { multi: "none | <ident>", comma: true },
    "animation-play-state"          : { multi: "running | paused", comma: true },
    "animation-timing-function"     : 1,
    
    //vendor prefixed
    "-moz-animation-delay"               : { multi: "<time>", comma: true },
    "-moz-animation-direction"           : { multi: "normal | alternate", comma: true },
    "-moz-animation-duration"            : { multi: "<time>", comma: true },
    "-moz-animation-iteration-count"     : { multi: "<number> | infinite", comma: true },
    "-moz-animation-name"                : { multi: "none | <ident>", comma: true },
    "-moz-animation-play-state"          : { multi: "running | paused", comma: true },
    
    "-ms-animation-delay"               : { multi: "<time>", comma: true },
    "-ms-animation-direction"           : { multi: "normal | alternate", comma: true },
    "-ms-animation-duration"            : { multi: "<time>", comma: true },
    "-ms-animation-iteration-count"     : { multi: "<number> | infinite", comma: true },
    "-ms-animation-name"                : { multi: "none | <ident>", comma: true },
    "-ms-animation-play-state"          : { multi: "running | paused", comma: true },
    
    "-webkit-animation-delay"               : { multi: "<time>", comma: true },
    "-webkit-animation-direction"           : { multi: "normal | alternate", comma: true },
    "-webkit-animation-duration"            : { multi: "<time>", comma: true },
    "-webkit-animation-iteration-count"     : { multi: "<number> | infinite", comma: true },
    "-webkit-animation-name"                : { multi: "none | <ident>", comma: true },
    "-webkit-animation-play-state"          : { multi: "running | paused", comma: true },
    
    "-o-animation-delay"               : { multi: "<time>", comma: true },
    "-o-animation-direction"           : { multi: "normal | alternate", comma: true },
    "-o-animation-duration"            : { multi: "<time>", comma: true },
    "-o-animation-iteration-count"     : { multi: "<number> | infinite", comma: true },
    "-o-animation-name"                : { multi: "none | <ident>", comma: true },
    "-o-animation-play-state"          : { multi: "running | paused", comma: true },        
    
    "appearance"                    : "icon | window | desktop | workspace | document | tooltip | dialog | button | push-button | hyperlink | radio-button | checkbox | menu-item | tab | menu | menubar | pull-down-menu | pop-up-menu | list-menu | radio-group | checkbox-group | outline-tree | range | field | combo-box | signature | password | normal | none | inherit",
    "azimuth"                       : function (expression) {
        var simple      = "<angle> | leftwards | rightwards | inherit",
            direction   = "left-side | far-left | left | center-left | center | center-right | right | far-right | right-side",
            behind      = false,
            valid       = false,
            part;
        
        if (!ValidationTypes.isAny(expression, simple)) {
            if (ValidationTypes.isAny(expression, "behind")) {
                behind = true;
                valid = true;
            }
            
            if (ValidationTypes.isAny(expression, direction)) {
                valid = true;
                if (!behind) {
                    ValidationTypes.isAny(expression, "behind");
                }
            }
        }
        
        if (expression.hasNext()) {
            part = expression.next();
            if (valid) {
                throw new ValidationError("Expected end of value but found '" + part + "'.", part.line, part.col);
            } else {
                throw new ValidationError("Expected (<'azimuth'>) but found '" + part + "'.", part.line, part.col);
            }
        }        
    },
    
    //B
    "backface-visibility"           : "visible | hidden",
    "background"                    : 1,
    "background-attachment"         : { multi: "<attachment>", comma: true },
    "background-clip"               : { multi: "<box>", comma: true },
    "background-color"              : "<color> | inherit",
    "background-image"              : { multi: "<bg-image>", comma: true },
    "background-origin"             : { multi: "<box>", comma: true },
    "background-position"           : { multi: "<bg-position>", comma: true },
    "background-repeat"             : { multi: "<repeat-style>" },
    "background-size"               : { multi: "<bg-size>", comma: true },
    "baseline-shift"                : "baseline | sub | super | <percentage> | <length>",
    "behavior"                      : 1,
    "binding"                       : 1,
    "bleed"                         : "<length>",
    "bookmark-label"                : "<content> | <attr> | <string>",
    "bookmark-level"                : "none | <integer>",
    "bookmark-state"                : "open | closed",
    "bookmark-target"               : "none | <uri> | <attr>",
    "border"                        : "<border-width> || <border-style> || <color>",
    "border-bottom"                 : "<border-width> || <border-style> || <color>",
    "border-bottom-color"           : "<color> | inherit",
    "border-bottom-left-radius"     :  "<x-one-radius>",
    "border-bottom-right-radius"    :  "<x-one-radius>",
    "border-bottom-style"           : "<border-style>",
    "border-bottom-width"           : "<border-width>",
    "border-collapse"               : "collapse | separate | inherit",
    "border-color"                  : { multi: "<color> | inherit", max: 4 },
    "border-image"                  : 1,
    "border-image-outset"           : { multi: "<length> | <number>", max: 4 },
    "border-image-repeat"           : { multi: "stretch | repeat | round", max: 2 },
    "border-image-slice"            : function(expression) {
        
        var valid   = false,
            numeric = "<number> | <percentage>",
            fill    = false,
            count   = 0,
            max     = 4,
            part;
        
        if (ValidationTypes.isAny(expression, "fill")) {
            fill = true;
            valid = true;
        }
        
        while (expression.hasNext() && count < max) {
            valid = ValidationTypes.isAny(expression, numeric);
            if (!valid) {
                break;
            }
            count++;
        }
        
        
        if (!fill) {
            ValidationTypes.isAny(expression, "fill");
        } else {
            valid = true;
        }
        
        if (expression.hasNext()) {
            part = expression.next();
            if (valid) {
                throw new ValidationError("Expected end of value but found '" + part + "'.", part.line, part.col);
            } else {
                throw new ValidationError("Expected ([<number> | <percentage>]{1,4} && fill?) but found '" + part + "'.", part.line, part.col);
            }
        }         
    },
    "border-image-source"           : "<image> | none",
    "border-image-width"            : { multi: "<length> | <percentage> | <number> | auto", max: 4 },
    "border-left"                   : "<border-width> || <border-style> || <color>",
    "border-left-color"             : "<color> | inherit",
    "border-left-style"             : "<border-style>",
    "border-left-width"             : "<border-width>",
    "border-radius"                 : function(expression) {
        
        var valid   = false,
            simple = "<length> | <percentage> | inherit",
            slash   = false,
            fill    = false,
            count   = 0,
            max     = 8,
            part;

        while (expression.hasNext() && count < max) {
            valid = ValidationTypes.isAny(expression, simple);
            if (!valid) {
            
                if (expression.peek() == "/" && count > 0 && !slash) {
                    slash = true;
                    max = count + 5;
                    expression.next();
                } else {
                    break;
                }
            }
            count++;
        }
        
        if (expression.hasNext()) {
            part = expression.next();
            if (valid) {
                throw new ValidationError("Expected end of value but found '" + part + "'.", part.line, part.col);
            } else {
                throw new ValidationError("Expected (<'border-radius'>) but found '" + part + "'.", part.line, part.col);
            }
        }         
    },
    "border-right"                  : "<border-width> || <border-style> || <color>",
    "border-right-color"            : "<color> | inherit",
    "border-right-style"            : "<border-style>",
    "border-right-width"            : "<border-width>",
    "border-spacing"                : { multi: "<length> | inherit", max: 2 },
    "border-style"                  : { multi: "<border-style>", max: 4 },
    "border-top"                    : "<border-width> || <border-style> || <color>",
    "border-top-color"              : "<color> | inherit",
    "border-top-left-radius"        : "<x-one-radius>",
    "border-top-right-radius"       : "<x-one-radius>",
    "border-top-style"              : "<border-style>",
    "border-top-width"              : "<border-width>",
    "border-width"                  : { multi: "<border-width>", max: 4 },
    "bottom"                        : "<margin-width> | inherit", 
    "box-align"                     : "start | end | center | baseline | stretch",        //http://www.w3.org/TR/2009/WD-css3-flexbox-20090723/
    "box-decoration-break"          : "slice |clone",
    "box-direction"                 : "normal | reverse | inherit",
    "box-flex"                      : "<number>",
    "box-flex-group"                : "<integer>",
    "box-lines"                     : "single | multiple",
    "box-ordinal-group"             : "<integer>",
    "box-orient"                    : "horizontal | vertical | inline-axis | block-axis | inherit",
    "box-pack"                      : "start | end | center | justify",
    "box-shadow"                    : function (expression) {
        var result      = false,
            part;

        if (!ValidationTypes.isAny(expression, "none")) {
            Validation.multiProperty("<shadow>", expression, true, Infinity);                       
        } else {
            if (expression.hasNext()) {
                part = expression.next();
                throw new ValidationError("Expected end of value but found '" + part + "'.", part.line, part.col);
            }   
        }
    },
    "box-sizing"                    : "content-box | border-box | inherit",
    "break-after"                   : "auto | always | avoid | left | right | page | column | avoid-page | avoid-column",
    "break-before"                  : "auto | always | avoid | left | right | page | column | avoid-page | avoid-column",
    "break-inside"                  : "auto | avoid | avoid-page | avoid-column",
    
    //C
    "caption-side"                  : "top | bottom | inherit",
    "clear"                         : "none | right | left | both | inherit",
    "clip"                          : 1,
    "color"                         : "<color> | inherit",
    "color-profile"                 : 1,
    "column-count"                  : "<integer> | auto",                      //http://www.w3.org/TR/css3-multicol/
    "column-fill"                   : "auto | balance",
    "column-gap"                    : "<length> | normal",
    "column-rule"                   : "<border-width> || <border-style> || <color>",
    "column-rule-color"             : "<color>",
    "column-rule-style"             : "<border-style>",
    "column-rule-width"             : "<border-width>",
    "column-span"                   : "none | all",
    "column-width"                  : "<length> | auto",
    "columns"                       : 1,
    "content"                       : 1,
    "counter-increment"             : 1,
    "counter-reset"                 : 1,
    "crop"                          : "<shape> | auto",
    "cue"                           : "cue-after | cue-before | inherit",
    "cue-after"                     : 1,
    "cue-before"                    : 1,
    "cursor"                        : 1,
    
    //D
    "direction"                     : "ltr | rtl | inherit",
    "display"                       : "inline | block | list-item | inline-block | table | inline-table | table-row-group | table-header-group | table-footer-group | table-row | table-column-group | table-column | table-cell | table-caption | box | inline-box | grid | inline-grid | none | inherit | -moz-box | -moz-inline-block | -moz-inline-box | -moz-inline-grid | -moz-inline-stack | -moz-inline-table | -moz-grid | -moz-grid-group | -moz-grid-line | -moz-groupbox | -moz-deck | -moz-popup | -moz-stack | -moz-marker | -webkit-box | -webkit-inline-box",
    "dominant-baseline"             : 1,
    "drop-initial-after-adjust"     : "central | middle | after-edge | text-after-edge | ideographic | alphabetic | mathematical | <percentage> | <length>",
    "drop-initial-after-align"      : "baseline | use-script | before-edge | text-before-edge | after-edge | text-after-edge | central | middle | ideographic | alphabetic | hanging | mathematical",
    "drop-initial-before-adjust"    : "before-edge | text-before-edge | central | middle | hanging | mathematical | <percentage> | <length>",
    "drop-initial-before-align"     : "caps-height | baseline | use-script | before-edge | text-before-edge | after-edge | text-after-edge | central | middle | ideographic | alphabetic | hanging | mathematical",
    "drop-initial-size"             : "auto | line | <length> | <percentage>",
    "drop-initial-value"            : "initial | <integer>",
    
    //E
    "elevation"                     : "<angle> | below | level | above | higher | lower | inherit",
    "empty-cells"                   : "show | hide | inherit",
    
    //F
    "filter"                        : 1,
    "fit"                           : "fill | hidden | meet | slice",
    "fit-position"                  : 1,
    "float"                         : "left | right | none | inherit",    
    "float-offset"                  : 1,
    "font"                          : 1,
    "font-family"                   : 1,
    "font-size"                     : "<absolute-size> | <relative-size> | <length> | <percentage> | inherit",
    "font-size-adjust"              : "<number> | none | inherit",
    "font-stretch"                  : "normal | ultra-condensed | extra-condensed | condensed | semi-condensed | semi-expanded | expanded | extra-expanded | ultra-expanded | inherit",
    "font-style"                    : "normal | italic | oblique | inherit",
    "font-variant"                  : "normal | small-caps | inherit",
    "font-weight"                   : "normal | bold | bolder | lighter | 100 | 200 | 300 | 400 | 500 | 600 | 700 | 800 | 900 | inherit",
    
    //G
    "grid-cell-stacking"            : "columns | rows | layer",
    "grid-column"                   : 1,
    "grid-columns"                  : 1,
    "grid-column-align"             : "start | end | center | stretch",
    "grid-column-sizing"            : 1,
    "grid-column-span"              : "<integer>",
    "grid-flow"                     : "none | rows | columns",
    "grid-layer"                    : "<integer>",
    "grid-row"                      : 1,
    "grid-rows"                     : 1,
    "grid-row-align"                : "start | end | center | stretch",
    "grid-row-span"                 : "<integer>",
    "grid-row-sizing"               : 1,
    
    //H
    "hanging-punctuation"           : 1,
    "height"                        : "<margin-width> | inherit",
    "hyphenate-after"               : "<integer> | auto",
    "hyphenate-before"              : "<integer> | auto",
    "hyphenate-character"           : "<string> | auto",
    "hyphenate-lines"               : "no-limit | <integer>",
    "hyphenate-resource"            : 1,
    "hyphens"                       : "none | manual | auto",
    
    //I
    "icon"                          : 1,
    "image-orientation"             : "angle | auto",
    "image-rendering"               : 1,
    "image-resolution"              : 1,
    "inline-box-align"              : "initial | last | <integer>",
    
    //L
    "left"                          : "<margin-width> | inherit",
    "letter-spacing"                : "<length> | normal | inherit",
    "line-height"                   : "<number> | <length> | <percentage> | normal | inherit",
    "line-break"                    : "auto | loose | normal | strict",
    "line-stacking"                 : 1,
    "line-stacking-ruby"            : "exclude-ruby | include-ruby",
    "line-stacking-shift"           : "consider-shifts | disregard-shifts",
    "line-stacking-strategy"        : "inline-line-height | block-line-height | max-height | grid-height",
    "list-style"                    : 1,
    "list-style-image"              : "<uri> | none | inherit",
    "list-style-position"           : "inside | outside | inherit",
    "list-style-type"               : "disc | circle | square | decimal | decimal-leading-zero | lower-roman | upper-roman | lower-greek | lower-latin | upper-latin | armenian | georgian | lower-alpha | upper-alpha | none | inherit",
    
    //M
    "margin"                        : { multi: "<margin-width> | inherit", max: 4 },
    "margin-bottom"                 : "<margin-width> | inherit",
    "margin-left"                   : "<margin-width> | inherit",
    "margin-right"                  : "<margin-width> | inherit",
    "margin-top"                    : "<margin-width> | inherit",
    "mark"                          : 1,
    "mark-after"                    : 1,
    "mark-before"                   : 1,
    "marks"                         : 1,
    "marquee-direction"             : 1,
    "marquee-play-count"            : 1,
    "marquee-speed"                 : 1,
    "marquee-style"                 : 1,
    "max-height"                    : "<length> | <percentage> | none | inherit",
    "max-width"                     : "<length> | <percentage> | none | inherit",
    "min-height"                    : "<length> | <percentage> | inherit",
    "min-width"                     : "<length> | <percentage> | inherit",
    "move-to"                       : 1,
    
    //N
    "nav-down"                      : 1,
    "nav-index"                     : 1,
    "nav-left"                      : 1,
    "nav-right"                     : 1,
    "nav-up"                        : 1,
    
    //O
    "opacity"                       : "<number> | inherit",
    "orphans"                       : "<integer> | inherit",
    "outline"                       : 1,
    "outline-color"                 : "<color> | invert | inherit",
    "outline-offset"                : 1,
    "outline-style"                 : "<border-style> | inherit",
    "outline-width"                 : "<border-width> | inherit",
    "overflow"                      : "visible | hidden | scroll | auto | inherit",
    "overflow-style"                : 1,
    "overflow-x"                    : 1,
    "overflow-y"                    : 1,
    
    //P
    "padding"                       : { multi: "<padding-width> | inherit", max: 4 },
    "padding-bottom"                : "<padding-width> | inherit",
    "padding-left"                  : "<padding-width> | inherit",
    "padding-right"                 : "<padding-width> | inherit",
    "padding-top"                   : "<padding-width> | inherit",
    "page"                          : 1,
    "page-break-after"              : "auto | always | avoid | left | right | inherit",
    "page-break-before"             : "auto | always | avoid | left | right | inherit",
    "page-break-inside"             : "auto | avoid | inherit",
    "page-policy"                   : 1,
    "pause"                         : 1,
    "pause-after"                   : 1,
    "pause-before"                  : 1,
    "perspective"                   : 1,
    "perspective-origin"            : 1,
    "phonemes"                      : 1,
    "pitch"                         : 1,
    "pitch-range"                   : 1,
    "play-during"                   : 1,
    "pointer-events"                : "auto | none | visiblePainted | visibleFill | visibleStroke | visible | painted | fill | stroke | all | inherit",
    "position"                      : "static | relative | absolute | fixed | inherit",
    "presentation-level"            : 1,
    "punctuation-trim"              : 1,
    
    //Q
    "quotes"                        : 1,
    
    //R
    "rendering-intent"              : 1,
    "resize"                        : 1,
    "rest"                          : 1,
    "rest-after"                    : 1,
    "rest-before"                   : 1,
    "richness"                      : 1,
    "right"                         : "<margin-width> | inherit",
    "rotation"                      : 1,
    "rotation-point"                : 1,
    "ruby-align"                    : 1,
    "ruby-overhang"                 : 1,
    "ruby-position"                 : 1,
    "ruby-span"                     : 1,
    
    //S
    "size"                          : 1,
    "speak"                         : "normal | none | spell-out | inherit",
    "speak-header"                  : "once | always | inherit",
    "speak-numeral"                 : "digits | continuous | inherit",
    "speak-punctuation"             : "code | none | inherit",
    "speech-rate"                   : 1,
    "src"                           : 1,
    "stress"                        : 1,
    "string-set"                    : 1,
    
    "table-layout"                  : "auto | fixed | inherit",
    "tab-size"                      : "<integer> | <length>",
    "target"                        : 1,
    "target-name"                   : 1,
    "target-new"                    : 1,
    "target-position"               : 1,
    "text-align"                    : "left | right | center | justify | inherit" ,
    "text-align-last"               : 1,
    "text-decoration"               : 1,
    "text-emphasis"                 : 1,
    "text-height"                   : 1,
    "text-indent"                   : "<length> | <percentage> | inherit",
    "text-justify"                  : "auto | none | inter-word | inter-ideograph | inter-cluster | distribute | kashida",
    "text-outline"                  : 1,
    "text-overflow"                 : 1,
    "text-rendering"                : "auto | optimizeSpeed | optimizeLegibility | geometricPrecision | inherit",
    "text-shadow"                   : 1,
    "text-transform"                : "capitalize | uppercase | lowercase | none | inherit",
    "text-wrap"                     : "normal | none | avoid",
    "top"                           : "<margin-width> | inherit",
    "transform"                     : 1,
    "transform-origin"              : 1,
    "transform-style"               : 1,
    "transition"                    : 1,
    "transition-delay"              : 1,
    "transition-duration"           : 1,
    "transition-property"           : 1,
    "transition-timing-function"    : 1,
    
    //U
    "unicode-bidi"                  : "normal | embed | bidi-override | inherit",
    "user-modify"                   : "read-only | read-write | write-only | inherit",
    "user-select"                   : "none | text | toggle | element | elements | all | inherit",
    
    //V
    "vertical-align"                : "auto | use-script | baseline | sub | super | top | text-top | central | middle | bottom | text-bottom | <percentage> | <length>",
    "visibility"                    : "visible | hidden | collapse | inherit",
    "voice-balance"                 : 1,
    "voice-duration"                : 1,
    "voice-family"                  : 1,
    "voice-pitch"                   : 1,
    "voice-pitch-range"             : 1,
    "voice-rate"                    : 1,
    "voice-stress"                  : 1,
    "voice-volume"                  : 1,
    "volume"                        : 1,
    
    //W
    "white-space"                   : "normal | pre | nowrap | pre-wrap | pre-line | inherit | -pre-wrap | -o-pre-wrap | -moz-pre-wrap | -hp-pre-wrap", //http://perishablepress.com/wrapping-content/
    "white-space-collapse"          : 1,
    "widows"                        : "<integer> | inherit",
    "width"                         : "<length> | <percentage> | auto | inherit" ,
    "word-break"                    : "normal | keep-all | break-all",
    "word-spacing"                  : "<length> | normal | inherit",
    "word-wrap"                     : 1,
    
    //Z
    "z-index"                       : "<integer> | auto | inherit",
    "zoom"                          : "<number> | <percentage> | normal"
};

/*global SyntaxUnit, Parser*/
/**
 * Represents a selector combinator (whitespace, +, >).
 * @namespace parserlib.css
 * @class PropertyName
 * @extends parserlib.util.SyntaxUnit
 * @constructor
 * @param {String} text The text representation of the unit. 
 * @param {String} hack The type of IE hack applied ("*", "_", or null).
 * @param {int} line The line of text on which the unit resides.
 * @param {int} col The column of text on which the unit resides.
 */
function PropertyName(text, hack, line, col){
    
    SyntaxUnit.call(this, text, line, col, Parser.PROPERTY_NAME_TYPE);

    /**
     * The type of IE hack applied ("*", "_", or null).
     * @type String
     * @property hack
     */
    this.hack = hack;

}

PropertyName.prototype = new SyntaxUnit();
PropertyName.prototype.constructor = PropertyName;
PropertyName.prototype.toString = function(){
    return (this.hack ? this.hack : "") + this.text;
};

/*global SyntaxUnit, Parser*/
/**
 * Represents a single part of a CSS property value, meaning that it represents
 * just everything single part between ":" and ";". If there are multiple values
 * separated by commas, this type represents just one of the values.
 * @param {String[]} parts An array of value parts making up this value.
 * @param {int} line The line of text on which the unit resides.
 * @param {int} col The column of text on which the unit resides.
 * @namespace parserlib.css
 * @class PropertyValue
 * @extends parserlib.util.SyntaxUnit
 * @constructor
 */
function PropertyValue(parts, line, col){

    SyntaxUnit.call(this, parts.join(" "), line, col, Parser.PROPERTY_VALUE_TYPE);
    
    /**
     * The parts that make up the selector.
     * @type Array
     * @property parts
     */
    this.parts = parts;
    
}

PropertyValue.prototype = new SyntaxUnit();
PropertyValue.prototype.constructor = PropertyValue;


/*global SyntaxUnit, Parser*/
/**
 * A utility class that allows for easy iteration over the various parts of a
 * property value.
 * @param {parserlib.css.PropertyValue} value The property value to iterate over.
 * @namespace parserlib.css
 * @class PropertyValueIterator
 * @constructor
 */
function PropertyValueIterator(value){

    /** 
     * Iterator value
     * @type int
     * @property _i
     * @private
     */
    this._i = 0;
    
    /**
     * The parts that make up the value.
     * @type Array
     * @property _parts
     * @private
     */
    this._parts = value.parts;
    
    /**
     * Keeps track of bookmarks along the way.
     * @type Array
     * @property _marks
     * @private
     */
    this._marks = [];
    
    /**
     * Holds the original property value.
     * @type parserlib.css.PropertyValue
     * @property value
     */
    this.value = value;
    
}

/**
 * Returns the total number of parts in the value.
 * @return {int} The total number of parts in the value.
 * @method count
 */
PropertyValueIterator.prototype.count = function(){
    return this._parts.length;
};

/**
 * Indicates if the iterator is positioned at the first item.
 * @return {Boolean} True if positioned at first item, false if not.
 * @method isFirst
 */
PropertyValueIterator.prototype.isFirst = function(){
    return this._i === 0;
};

/**
 * Indicates if there are more parts of the property value.
 * @return {Boolean} True if there are more parts, false if not.
 * @method hasNext
 */
PropertyValueIterator.prototype.hasNext = function(){
    return (this._i < this._parts.length);
};

/**
 * Marks the current spot in the iteration so it can be restored to
 * later on.
 * @return {void}
 * @method mark
 */
PropertyValueIterator.prototype.mark = function(){
    this._marks.push(this._i);
};

/**
 * Returns the next part of the property value or null if there is no next
 * part. Does not move the internal counter forward.
 * @return {parserlib.css.PropertyValuePart} The next part of the property value or null if there is no next
 * part.
 * @method peek
 */
PropertyValueIterator.prototype.peek = function(count){
    return this.hasNext() ? this._parts[this._i + (count || 0)] : null;
};

/**
 * Returns the next part of the property value or null if there is no next
 * part.
 * @return {parserlib.css.PropertyValuePart} The next part of the property value or null if there is no next
 * part.
 * @method next
 */
PropertyValueIterator.prototype.next = function(){
    return this.hasNext() ? this._parts[this._i++] : null;
};

/**
 * Returns the previous part of the property value or null if there is no
 * previous part.
 * @return {parserlib.css.PropertyValuePart} The previous part of the 
 * property value or null if there is no next part.
 * @method previous
 */
PropertyValueIterator.prototype.previous = function(){
    return this._i > 0 ? this._parts[--this._i] : null;
};

/**
 * Restores the last saved bookmark.
 * @return {void}
 * @method restore
 */
PropertyValueIterator.prototype.restore = function(){
    if (this._marks.length){
        this._i = this._marks.pop();
    }
};


/*global SyntaxUnit, Parser, Colors*/
/**
 * Represents a single part of a CSS property value, meaning that it represents
 * just one part of the data between ":" and ";".
 * @param {String} text The text representation of the unit.
 * @param {int} line The line of text on which the unit resides.
 * @param {int} col The column of text on which the unit resides.
 * @namespace parserlib.css
 * @class PropertyValuePart
 * @extends parserlib.util.SyntaxUnit
 * @constructor
 */
function PropertyValuePart(text, line, col){

    SyntaxUnit.call(this, text, line, col, Parser.PROPERTY_VALUE_PART_TYPE);
    
    /**
     * Indicates the type of value unit.
     * @type String
     * @property type
     */
    this.type = "unknown";

    //figure out what type of data it is
    
    var temp;
    
    //it is a measurement?
    if (/^([+\-]?[\d\.]+)([a-z]+)$/i.test(text)){  //dimension
        this.type = "dimension";
        this.value = +RegExp.$1;
        this.units = RegExp.$2;
        
        //try to narrow down
        switch(this.units.toLowerCase()){
        
            case "em":
            case "rem":
            case "ex":
            case "px":
            case "cm":
            case "mm":
            case "in":
            case "pt":
            case "pc":
            case "ch":
            case "vh":
            case "vw":
            case "vm":
                this.type = "length";
                break;
                
            case "deg":
            case "rad":
            case "grad":
                this.type = "angle";
                break;
            
            case "ms":
            case "s":
                this.type = "time";
                break;
            
            case "hz":
            case "khz":
                this.type = "frequency";
                break;
            
            case "dpi":
            case "dpcm":
                this.type = "resolution";
                break;
                
            //default
                
        }
        
    } else if (/^([+\-]?[\d\.]+)%$/i.test(text)){  //percentage
        this.type = "percentage";
        this.value = +RegExp.$1;
    } else if (/^([+\-]?[\d\.]+)%$/i.test(text)){  //percentage
        this.type = "percentage";
        this.value = +RegExp.$1;
    } else if (/^([+\-]?\d+)$/i.test(text)){  //integer
        this.type = "integer";
        this.value = +RegExp.$1;
    } else if (/^([+\-]?[\d\.]+)$/i.test(text)){  //number
        this.type = "number";
        this.value = +RegExp.$1;
    
    } else if (/^#([a-f0-9]{3,6})/i.test(text)){  //hexcolor
        this.type = "color";
        temp = RegExp.$1;
        if (temp.length == 3){
            this.red    = parseInt(temp.charAt(0)+temp.charAt(0),16);
            this.green  = parseInt(temp.charAt(1)+temp.charAt(1),16);
            this.blue   = parseInt(temp.charAt(2)+temp.charAt(2),16);            
        } else {
            this.red    = parseInt(temp.substring(0,2),16);
            this.green  = parseInt(temp.substring(2,4),16);
            this.blue   = parseInt(temp.substring(4,6),16);            
        }
    } else if (/^rgb\(\s*(\d+)\s*,\s*(\d+)\s*,\s*(\d+)\s*\)/i.test(text)){ //rgb() color with absolute numbers
        this.type   = "color";
        this.red    = +RegExp.$1;
        this.green  = +RegExp.$2;
        this.blue   = +RegExp.$3;
    } else if (/^rgb\(\s*(\d+)%\s*,\s*(\d+)%\s*,\s*(\d+)%\s*\)/i.test(text)){ //rgb() color with percentages
        this.type   = "color";
        this.red    = +RegExp.$1 * 255 / 100;
        this.green  = +RegExp.$2 * 255 / 100;
        this.blue   = +RegExp.$3 * 255 / 100;
    } else if (/^rgba\(\s*(\d+)\s*,\s*(\d+)\s*,\s*(\d+)\s*,\s*([\d\.]+)\s*\)/i.test(text)){ //rgba() color with absolute numbers
        this.type   = "color";
        this.red    = +RegExp.$1;
        this.green  = +RegExp.$2;
        this.blue   = +RegExp.$3;
        this.alpha  = +RegExp.$4;
    } else if (/^rgba\(\s*(\d+)%\s*,\s*(\d+)%\s*,\s*(\d+)%\s*,\s*([\d\.]+)\s*\)/i.test(text)){ //rgba() color with percentages
        this.type   = "color";
        this.red    = +RegExp.$1 * 255 / 100;
        this.green  = +RegExp.$2 * 255 / 100;
        this.blue   = +RegExp.$3 * 255 / 100;
        this.alpha  = +RegExp.$4;        
    } else if (/^hsl\(\s*(\d+)\s*,\s*(\d+)%\s*,\s*(\d+)%\s*\)/i.test(text)){ //hsl()
        this.type   = "color";
        this.hue    = +RegExp.$1;
        this.saturation = +RegExp.$2 / 100;
        this.lightness  = +RegExp.$3 / 100;        
    } else if (/^hsla\(\s*(\d+)\s*,\s*(\d+)%\s*,\s*(\d+)%\s*,\s*([\d\.]+)\s*\)/i.test(text)){ //hsla() color with percentages
        this.type   = "color";
        this.hue    = +RegExp.$1;
        this.saturation = +RegExp.$2 / 100;
        this.lightness  = +RegExp.$3 / 100;        
        this.alpha  = +RegExp.$4;        
    } else if (/^url\(["']?([^\)"']+)["']?\)/i.test(text)){ //URI
        this.type   = "uri";
        this.uri    = RegExp.$1;
    } else if (/^([^\(]+)\(/i.test(text)){
        this.type   = "function";
        this.name   = RegExp.$1;
        this.value  = text;
    } else if (/^["'][^"']*["']/.test(text)){    //string
        this.type   = "string";
        this.value  = eval(text);
    } else if (Colors[text.toLowerCase()]){  //named color
        this.type   = "color";
        temp        = Colors[text.toLowerCase()].substring(1);
        this.red    = parseInt(temp.substring(0,2),16);
        this.green  = parseInt(temp.substring(2,4),16);
        this.blue   = parseInt(temp.substring(4,6),16);         
    } else if (/^[\,\/]$/.test(text)){
        this.type   = "operator";
        this.value  = text;
    } else if (/^[a-z\-\u0080-\uFFFF][a-z0-9\-\u0080-\uFFFF]*$/i.test(text)){
        this.type   = "identifier";
        this.value  = text;
    }

}

PropertyValuePart.prototype = new SyntaxUnit();
PropertyValuePart.prototype.constructor = PropertyValuePart;

/**
 * Create a new syntax unit based solely on the given token.
 * Convenience method for creating a new syntax unit when
 * it represents a single token instead of multiple.
 * @param {Object} token The token object to represent.
 * @return {parserlib.css.PropertyValuePart} The object representing the token.
 * @static
 * @method fromToken
 */
PropertyValuePart.fromToken = function(token){
    return new PropertyValuePart(token.value, token.startLine, token.startCol);
};
var Pseudos = {
    ":first-letter": 1,
    ":first-line":   1,
    ":before":       1,
    ":after":        1
};

Pseudos.ELEMENT = 1;
Pseudos.CLASS = 2;

Pseudos.isElement = function(pseudo){
    return pseudo.indexOf("::") === 0 || Pseudos[pseudo.toLowerCase()] == Pseudos.ELEMENT;
};
/*global SyntaxUnit, Parser, Specificity*/
/**
 * Represents an entire single selector, including all parts but not
 * including multiple selectors (those separated by commas).
 * @namespace parserlib.css
 * @class Selector
 * @extends parserlib.util.SyntaxUnit
 * @constructor
 * @param {Array} parts Array of selectors parts making up this selector.
 * @param {int} line The line of text on which the unit resides.
 * @param {int} col The column of text on which the unit resides.
 */
function Selector(parts, line, col){
    
    SyntaxUnit.call(this, parts.join(" "), line, col, Parser.SELECTOR_TYPE);
    
    /**
     * The parts that make up the selector.
     * @type Array
     * @property parts
     */
    this.parts = parts;
    
    /**
     * The specificity of the selector.
     * @type parserlib.css.Specificity
     * @property specificity
     */
    this.specificity = Specificity.calculate(this);

}

Selector.prototype = new SyntaxUnit();
Selector.prototype.constructor = Selector;


/*global SyntaxUnit, Parser*/
/**
 * Represents a single part of a selector string, meaning a single set of
 * element name and modifiers. This does not include combinators such as
 * spaces, +, >, etc.
 * @namespace parserlib.css
 * @class SelectorPart
 * @extends parserlib.util.SyntaxUnit
 * @constructor
 * @param {String} elementName The element name in the selector or null
 *      if there is no element name.
 * @param {Array} modifiers Array of individual modifiers for the element.
 *      May be empty if there are none.
 * @param {String} text The text representation of the unit. 
 * @param {int} line The line of text on which the unit resides.
 * @param {int} col The column of text on which the unit resides.
 */
function SelectorPart(elementName, modifiers, text, line, col){
    
    SyntaxUnit.call(this, text, line, col, Parser.SELECTOR_PART_TYPE);

    /**
     * The tag name of the element to which this part
     * of the selector affects.
     * @type String
     * @property elementName
     */
    this.elementName = elementName;
    
    /**
     * The parts that come after the element name, such as class names, IDs,
     * pseudo classes/elements, etc.
     * @type Array
     * @property modifiers
     */
    this.modifiers = modifiers;

}

SelectorPart.prototype = new SyntaxUnit();
SelectorPart.prototype.constructor = SelectorPart;


/*global SyntaxUnit, Parser*/
/**
 * Represents a selector modifier string, meaning a class name, element name,
 * element ID, pseudo rule, etc.
 * @namespace parserlib.css
 * @class SelectorSubPart
 * @extends parserlib.util.SyntaxUnit
 * @constructor
 * @param {String} text The text representation of the unit. 
 * @param {String} type The type of selector modifier.
 * @param {int} line The line of text on which the unit resides.
 * @param {int} col The column of text on which the unit resides.
 */
function SelectorSubPart(text, type, line, col){
    
    SyntaxUnit.call(this, text, line, col, Parser.SELECTOR_SUB_PART_TYPE);

    /**
     * The type of modifier.
     * @type String
     * @property type
     */
    this.type = type;
    
    /**
     * Some subparts have arguments, this represents them.
     * @type Array
     * @property args
     */
    this.args = [];

}

SelectorSubPart.prototype = new SyntaxUnit();
SelectorSubPart.prototype.constructor = SelectorSubPart;


/*global Pseudos, SelectorPart*/
/**
 * Represents a selector's specificity.
 * @namespace parserlib.css
 * @class Specificity
 * @constructor
 * @param {int} a Should be 1 for inline styles, zero for stylesheet styles
 * @param {int} b Number of ID selectors
 * @param {int} c Number of classes and pseudo classes
 * @param {int} d Number of element names and pseudo elements
 */
function Specificity(a, b, c, d){
    this.a = a;
    this.b = b;
    this.c = c;
    this.d = d;
}

Specificity.prototype = {
    constructor: Specificity,
    
    /**
     * Compare this specificity to another.
     * @param {Specificity} other The other specificity to compare to.
     * @return {int} -1 if the other specificity is larger, 1 if smaller, 0 if equal.
     * @method compare
     */
    compare: function(other){
        var comps = ["a", "b", "c", "d"],
            i, len;
            
        for (i=0, len=comps.length; i < len; i++){
            if (this[comps[i]] < other[comps[i]]){
                return -1;
            } else if (this[comps[i]] > other[comps[i]]){
                return 1;
            }
        }
        
        return 0;
    },
    
    /**
     * Creates a numeric value for the specificity.
     * @return {int} The numeric value for the specificity.
     * @method valueOf
     */
    valueOf: function(){
        return (this.a * 1000) + (this.b * 100) + (this.c * 10) + this.d;
    },
    
    /**
     * Returns a string representation for specificity.
     * @return {String} The string representation of specificity.
     * @method toString
     */
    toString: function(){
        return this.a + "," + this.b + "," + this.c + "," + this.d;
    }

};

/**
 * Calculates the specificity of the given selector.
 * @param {parserlib.css.Selector} The selector to calculate specificity for.
 * @return {parserlib.css.Specificity} The specificity of the selector.
 * @static
 * @method calculate
 */
Specificity.calculate = function(selector){

    var i, len,
        part,
        b=0, c=0, d=0;
        
    function updateValues(part){
    
        var i, j, len, num,
            elementName = part.elementName ? part.elementName.text : "",
            modifier;
    
        if (elementName && elementName.charAt(elementName.length-1) != "*") {
            d++;
        }    
    
        for (i=0, len=part.modifiers.length; i < len; i++){
            modifier = part.modifiers[i];
            switch(modifier.type){
                case "class":
                case "attribute":
                    c++;
                    break;
                    
                case "id":
                    b++;
                    break;
                    
                case "pseudo":
                    if (Pseudos.isElement(modifier.text)){
                        d++;
                    } else {
                        c++;
                    }                    
                    break;
                    
                case "not":
                    for (j=0, num=modifier.args.length; j < num; j++){
                        updateValues(modifier.args[j]);
                    }
            }    
         }
    }
    
    for (i=0, len=selector.parts.length; i < len; i++){
        part = selector.parts[i];
        
        if (part instanceof SelectorPart){
            updateValues(part);                
        }
    }
    
    return new Specificity(0, b, c, d);
};

/*global Tokens, TokenStreamBase*/

var h = /^[0-9a-fA-F]$/,
    nonascii = /^[\u0080-\uFFFF]$/,
    nl = /\n|\r\n|\r|\f/;

//-----------------------------------------------------------------------------
// Helper functions
//-----------------------------------------------------------------------------


function isHexDigit(c){
    return c !== null && h.test(c);
}

function isDigit(c){
    return c !== null && /\d/.test(c);
}

function isWhitespace(c){
    return c !== null && /\s/.test(c);
}

function isNewLine(c){
    return c !== null && nl.test(c);
}

function isNameStart(c){
    return c !== null && (/[a-z_\u0080-\uFFFF\\]/i.test(c));
}

function isNameChar(c){
    return c !== null && (isNameStart(c) || /[0-9\-\\]/.test(c));
}

function isIdentStart(c){
    return c !== null && (isNameStart(c) || /\-\\/.test(c));
}

function mix(receiver, supplier){
	for (var prop in supplier){
		if (supplier.hasOwnProperty(prop)){
			receiver[prop] = supplier[prop];
		}
	}
	return receiver;
}

//-----------------------------------------------------------------------------
// CSS Token Stream
//-----------------------------------------------------------------------------


/**
 * A token stream that produces CSS tokens.
 * @param {String|Reader} input The source of text to tokenize.
 * @constructor
 * @class TokenStream
 * @namespace parserlib.css
 */
function TokenStream(input){
	TokenStreamBase.call(this, input, Tokens);
	this.tokens = [];  //reset the cached stream ORION 8.0
}

TokenStream.prototype = mix(new TokenStreamBase(), {

    /**
     * Overrides the TokenStreamBase method of the same name
     * to produce CSS tokens.
     * @param {variant} channel The name of the channel to use
     *      for the next token.
     * @return {Object} A token object representing the next token.
     * @method _getToken
     * @private
     */
    _getToken: function(channel){

        var c,
            reader = this._reader,
            token   = null,
            startLine   = reader.getLine(),
            startCol    = reader.getCol();
            start       = reader.getCursor(); //ORION 8.0
        c = reader.read();


        while(c){
            switch(c){

                /*
                 * Potential tokens:
                 * - COMMENT
                 * - SLASH
                 * - CHAR
                 */
                case "/":

                    if(reader.peek() == "*"){
                        token = this.commentToken(c, startLine, startCol);
                    } else {
                        token = this.charToken(c, startLine, startCol);
                    }
                    break;

                /*
                 * Potential tokens:
                 * - DASHMATCH
                 * - INCLUDES
                 * - PREFIXMATCH
                 * - SUFFIXMATCH
                 * - SUBSTRINGMATCH
                 * - CHAR
                 */
                case "|":
                case "~":
                case "^":
                case "$":
                case "*":
                    if(reader.peek() == "="){
                        token = this.comparisonToken(c, startLine, startCol);
                    } else {
                        token = this.charToken(c, startLine, startCol);
                    }
                    break;

                /*
                 * Potential tokens:
                 * - STRING
                 * - INVALID
                 */
                case "\"":
                case "'":
                    token = this.stringToken(c, startLine, startCol);
                    break;

                /*
                 * Potential tokens:
                 * - HASH
                 * - CHAR
                 */
                case "#":
                    if (isNameChar(reader.peek())){
                        token = this.hashToken(c, startLine, startCol);
                    } else {
                        token = this.charToken(c, startLine, startCol);
                    }
                    break;

                /*
                 * Potential tokens:
                 * - DOT
                 * - NUMBER
                 * - DIMENSION
                 * - PERCENTAGE
                 */
                case ".":
                    if (isDigit(reader.peek())){
                        token = this.numberToken(c, startLine, startCol);
                    } else {
                        token = this.charToken(c, startLine, startCol);
                    }
                    break;

                /*
                 * Potential tokens:
                 * - CDC
                 * - MINUS
                 * - NUMBER
                 * - DIMENSION
                 * - PERCENTAGE
                 */
                case "-":
                    if (reader.peek() == "-"){  //could be closing HTML-style comment
                        token = this.htmlCommentEndToken(c, startLine, startCol);
                    } else if (isNameStart(reader.peek())){
                        token = this.identOrFunctionToken(c, startLine, startCol);
                    } else {
                        token = this.charToken(c, startLine, startCol);
                    }
                    break;

                /*
                 * Potential tokens:
                 * - IMPORTANT_SYM
                 * - CHAR
                 */
                case "!":
                    token = this.importantToken(c, startLine, startCol);
                    break;

                /*
                 * Any at-keyword or CHAR
                 */
                case "@":
                    token = this.atRuleToken(c, startLine, startCol);
                    break;

                /*
                 * Potential tokens:
                 * - NOT
                 * - CHAR
                 */
                case ":":
                    token = this.notToken(c, startLine, startCol);
                    break;

                /*
                 * Potential tokens:
                 * - CDO
                 * - CHAR
                 */
                case "<":
                    token = this.htmlCommentStartToken(c, startLine, startCol);
                    break;

                /*
                 * Potential tokens:
                 * - UNICODE_RANGE
                 * - URL
                 * - CHAR
                 */
                case "U":
                case "u":
                    if (reader.peek() == "+"){
                        token = this.unicodeRangeToken(c, startLine, startCol);
                        break;
                    }
                    /* falls through */
                default:

                    /*
                     * Potential tokens:
                     * - NUMBER
                     * - DIMENSION
                     * - LENGTH
                     * - FREQ
                     * - TIME
                     * - EMS
                     * - EXS
                     * - ANGLE
                     */
                    if (isDigit(c)){
                        token = this.numberToken(c, startLine, startCol);
                    } else
                    /*
                     * Potential tokens:
                     * - S
                     */
                    if (isWhitespace(c)){
                        token = this.whitespaceToken(c, startLine, startCol);
                    } else

                    /*
                     * Potential tokens:
                     * - IDENT
                     */
                    if (isIdentStart(c)){
                        token = this.identOrFunctionToken(c, startLine, startCol);
                    } else

                    /*
                     * Potential tokens:
                     * - CHAR
                     * - PLUS
                     */
                    {
                        token = this.charToken(c, startLine, startCol);
                    }
            }

            //make sure this token is wanted
            //TODO: check channel
            break;
        }
        if (!token && c === null){
            token = this.createToken(Tokens.EOF,null,startLine,startCol);
        }
        if(token.type !== Tokens.S) {
            var smalltoken = Object.create(null);
            smalltoken.type = Tokens.name(token.type);
            smalltoken.value = token.value;
            smalltoken.range = [start, reader.getCursor()];
            this.tokens.push(smalltoken);
        }
        return token;
    },

    /**
     * @description ORION 8.0 returns the last read token
     */
    curr: function() {
        if(this.tokens.length > 0) {
            return this.tokens[this.tokens.length-1];
        }
        return null;
    },

    //-------------------------------------------------------------------------
    // Methods to create tokens
    //-------------------------------------------------------------------------

    /**
     * Produces a token based on available data and the current
     * reader position information. This method is called by other
     * private methods to create tokens and is never called directly.
     * @param {int} tt The token type.
     * @param {String} value The text value of the token.
     * @param {int} startLine The beginning line for the character.
     * @param {int} startCol The beginning column for the character.
     * @param {Object} options (Optional) Specifies a channel property
     *      to indicate that a different channel should be scanned
     *      and/or a hide property indicating that the token should
     *      be hidden.
     * @return {Object} A token object.
     * @method createToken
     */
    createToken: function(tt, value, startLine, startCol, options){
        var reader = this._reader;
        options = options || {};

        return {
            value:      value,
            type:       tt,
            channel:    options.channel,
            hide:       options.hide || false,
            startLine:  startLine,
            startCol:   startCol,
            endLine:    reader.getLine(),
            endCol:     reader.getCol()
        };
    },

    //-------------------------------------------------------------------------
    // Methods to create specific tokens
    //-------------------------------------------------------------------------

    /**
     * Produces a token for any at-rule. If the at-rule is unknown, then
     * the token is for a single "@" character.
     * @param {String} first The first character for the token.
     * @param {int} startLine The beginning line for the character.
     * @param {int} startCol The beginning column for the character.
     * @return {Object} A token object.
     * @method atRuleToken
     */
    atRuleToken: function(first, startLine, startCol){
        var rule    = first,
            reader  = this._reader,
            tt      = Tokens.CHAR,
            valid   = false,
            ident,
            c;

        /*
         * First, mark where we are. There are only four @ rules,
         * so anything else is really just an invalid token.
         * Basically, if this doesn't match one of the known @
         * rules, just return '@' as an unknown token and allow
         * parsing to continue after that point.
         */
        reader.mark();

        //try to find the at-keyword
        ident = this.readName();
        rule = first + ident;
        tt = Tokens.type(rule.toLowerCase());

        //if it's not valid, use the first character only and reset the reader
        if (tt == Tokens.CHAR || tt == Tokens.UNKNOWN){
            if (rule.length > 1){
                tt = Tokens.UNKNOWN_SYM;                
            } else {
                tt = Tokens.CHAR;
                rule = first;
                reader.reset();
            }
        }

        return this.createToken(tt, rule, startLine, startCol);
    },

    /**
     * Produces a character token based on the given character
     * and location in the stream. If there's a special (non-standard)
     * token name, this is used; otherwise CHAR is used.
     * @param {String} c The character for the token.
     * @param {int} startLine The beginning line for the character.
     * @param {int} startCol The beginning column for the character.
     * @return {Object} A token object.
     * @method charToken
     */
    charToken: function(c, startLine, startCol){
        var tt = Tokens.type(c);

        if (tt == -1){
            tt = Tokens.CHAR;
        }

        return this.createToken(tt, c, startLine, startCol);
    },

    /**
     * Produces a character token based on the given character
     * and location in the stream. If there's a special (non-standard)
     * token name, this is used; otherwise CHAR is used.
     * @param {String} first The first character for the token.
     * @param {int} startLine The beginning line for the character.
     * @param {int} startCol The beginning column for the character.
     * @return {Object} A token object.
     * @method commentToken
     */
    commentToken: function(first, startLine, startCol){
        var reader  = this._reader,
            comment = this.readComment(first);

        return this.createToken(Tokens.COMMENT, comment, startLine, startCol);
    },

    /**
     * Produces a comparison token based on the given character
     * and location in the stream. The next character must be
     * read and is already known to be an equals sign.
     * @param {String} c The character for the token.
     * @param {int} startLine The beginning line for the character.
     * @param {int} startCol The beginning column for the character.
     * @return {Object} A token object.
     * @method comparisonToken
     */
    comparisonToken: function(c, startLine, startCol){
        var reader  = this._reader,
            comparison  = c + reader.read(),
            tt      = Tokens.type(comparison) || Tokens.CHAR;

        return this.createToken(tt, comparison, startLine, startCol);
    },

    /**
     * Produces a hash token based on the specified information. The
     * first character provided is the pound sign (#) and then this
     * method reads a name afterward.
     * @param {String} first The first character (#) in the hash name.
     * @param {int} startLine The beginning line for the character.
     * @param {int} startCol The beginning column for the character.
     * @return {Object} A token object.
     * @method hashToken
     */
    hashToken: function(first, startLine, startCol){
        var reader  = this._reader,
            name    = this.readName(first);

        return this.createToken(Tokens.HASH, name, startLine, startCol);
    },

    /**
     * Produces a CDO or CHAR token based on the specified information. The
     * first character is provided and the rest is read by the function to determine
     * the correct token to create.
     * @param {String} first The first character in the token.
     * @param {int} startLine The beginning line for the character.
     * @param {int} startCol The beginning column for the character.
     * @return {Object} A token object.
     * @method htmlCommentStartToken
     */
    htmlCommentStartToken: function(first, startLine, startCol){
        var reader      = this._reader,
            text        = first;

        reader.mark();
        text += reader.readCount(3);

        if (text == "<!--"){
            return this.createToken(Tokens.CDO, text, startLine, startCol);
        } else {
            reader.reset();
            return this.charToken(first, startLine, startCol);
        }
    },

    /**
     * Produces a CDC or CHAR token based on the specified information. The
     * first character is provided and the rest is read by the function to determine
     * the correct token to create.
     * @param {String} first The first character in the token.
     * @param {int} startLine The beginning line for the character.
     * @param {int} startCol The beginning column for the character.
     * @return {Object} A token object.
     * @method htmlCommentEndToken
     */
    htmlCommentEndToken: function(first, startLine, startCol){
        var reader      = this._reader,
            text        = first;

        reader.mark();
        text += reader.readCount(2);

        if (text == "-->"){
            return this.createToken(Tokens.CDC, text, startLine, startCol);
        } else {
            reader.reset();
            return this.charToken(first, startLine, startCol);
        }
    },

    /**
     * Produces an IDENT or FUNCTION token based on the specified information. The
     * first character is provided and the rest is read by the function to determine
     * the correct token to create.
     * @param {String} first The first character in the identifier.
     * @param {int} startLine The beginning line for the character.
     * @param {int} startCol The beginning column for the character.
     * @return {Object} A token object.
     * @method identOrFunctionToken
     */
    identOrFunctionToken: function(first, startLine, startCol){
        var reader  = this._reader,
            ident   = this.readName(first),
            tt      = Tokens.IDENT;

        //if there's a left paren immediately after, it's a URI or function
        if (reader.peek() == "("){
            ident += reader.read();
            if (ident.toLowerCase() == "url("){
                tt = Tokens.URI;
                ident = this.readURI(ident);

                //didn't find a valid URL or there's no closing paren
                if (ident.toLowerCase() == "url("){
                    tt = Tokens.FUNCTION;
                }
            } else {
                tt = Tokens.FUNCTION;
            }
        } else if (reader.peek() == ":"){  //might be an IE function

            //IE-specific functions always being with progid:
            if (ident.toLowerCase() == "progid"){
                ident += reader.readTo("(");
                tt = Tokens.IE_FUNCTION;
            }
        }

        return this.createToken(tt, ident, startLine, startCol);
    },

    /**
     * Produces an IMPORTANT_SYM or CHAR token based on the specified information. The
     * first character is provided and the rest is read by the function to determine
     * the correct token to create.
     * @param {String} first The first character in the token.
     * @param {int} startLine The beginning line for the character.
     * @param {int} startCol The beginning column for the character.
     * @return {Object} A token object.
     * @method importantToken
     */
    importantToken: function(first, startLine, startCol){
        var reader      = this._reader,
            important   = first,
            tt          = Tokens.CHAR,
            temp,
            c;

        reader.mark();
        c = reader.read();

        while(c){

            //there can be a comment in here
            if (c == "/"){

                //if the next character isn't a star, then this isn't a valid !important token
                if (reader.peek() != "*"){
                    break;
                } else {
                    temp = this.readComment(c);
                    if (temp === ""){    //broken!
                        break;
                    }
                }
            } else if (isWhitespace(c)){
                important += c + this.readWhitespace();
            } else if (/i/i.test(c)){
                temp = reader.readCount(8);
                if (/mportant/i.test(temp)){
                    important += c + temp;
                    tt = Tokens.IMPORTANT_SYM;

                }
                break;  //we're done
            } else {
                break;
            }

            c = reader.read();
        }

        if (tt == Tokens.CHAR){
            reader.reset();
            return this.charToken(first, startLine, startCol);
        } else {
            return this.createToken(tt, important, startLine, startCol);
        }


    },

    /**
     * Produces a NOT or CHAR token based on the specified information. The
     * first character is provided and the rest is read by the function to determine
     * the correct token to create.
     * @param {String} first The first character in the token.
     * @param {int} startLine The beginning line for the character.
     * @param {int} startCol The beginning column for the character.
     * @return {Object} A token object.
     * @method notToken
     */
    notToken: function(first, startLine, startCol){
        var reader      = this._reader,
            text        = first;

        reader.mark();
        text += reader.readCount(4);

        if (text.toLowerCase() == ":not("){
            return this.createToken(Tokens.NOT, text, startLine, startCol);
        } else {
            reader.reset();
            return this.charToken(first, startLine, startCol);
        }
    },

    /**
     * Produces a number token based on the given character
     * and location in the stream. This may return a token of
     * NUMBER, EMS, EXS, LENGTH, ANGLE, TIME, FREQ, DIMENSION,
     * or PERCENTAGE.
     * @param {String} first The first character for the token.
     * @param {int} startLine The beginning line for the character.
     * @param {int} startCol The beginning column for the character.
     * @return {Object} A token object.
     * @method numberToken
     */
    numberToken: function(first, startLine, startCol){
        var reader  = this._reader,
            value   = this.readNumber(first),
            ident,
            tt      = Tokens.NUMBER,
            c       = reader.peek();

        if (isIdentStart(c)){
            ident = this.readName(reader.read());
            value += ident;

            if (/^em$|^ex$|^px$|^gd$|^rem$|^vw$|^vh$|^vm$|^ch$|^cm$|^mm$|^in$|^pt$|^pc$/i.test(ident)){
                tt = Tokens.LENGTH;
            } else if (/^deg|^rad$|^grad$/i.test(ident)){
                tt = Tokens.ANGLE;
            } else if (/^ms$|^s$/i.test(ident)){
                tt = Tokens.TIME;
            } else if (/^hz$|^khz$/i.test(ident)){
                tt = Tokens.FREQ;
            } else if (/^dpi$|^dpcm$/i.test(ident)){
                tt = Tokens.RESOLUTION;
            } else {
                tt = Tokens.DIMENSION;
            }

        } else if (c == "%"){
            value += reader.read();
            tt = Tokens.PERCENTAGE;
        }

        return this.createToken(tt, value, startLine, startCol);
    },

    /**
     * Produces a string token based on the given character
     * and location in the stream. Since strings may be indicated
     * by single or double quotes, a failure to match starting
     * and ending quotes results in an INVALID token being generated.
     * The first character in the string is passed in and then
     * the rest are read up to and including the final quotation mark.
     * @param {String} first The first character in the string.
     * @param {int} startLine The beginning line for the character.
     * @param {int} startCol The beginning column for the character.
     * @return {Object} A token object.
     * @method stringToken
     */
    stringToken: function(first, startLine, startCol){
        var delim   = first,
            string  = first,
            reader  = this._reader,
            prev    = first,
            tt      = Tokens.STRING,
            c       = reader.read();

        while(c){
            string += c;

            //if the delimiter is found with an escapement, we're done.
            if (c == delim && prev != "\\"){
                break;
            }

            //if there's a newline without an escapement, it's an invalid string
            if (isNewLine(reader.peek()) && c != "\\"){
                tt = Tokens.INVALID;
                break;
            }

            //save previous and get next
            prev = c;
            c = reader.read();
        }

        //if c is null, that means we're out of input and the string was never closed
        if (c === null){
            tt = Tokens.INVALID;
        }

        return this.createToken(tt, string, startLine, startCol);
    },

    unicodeRangeToken: function(first, startLine, startCol){
        var reader  = this._reader,
            value   = first,
            temp,
            tt      = Tokens.CHAR;

        //then it should be a unicode range
        if (reader.peek() == "+"){
            reader.mark();
            value += reader.read();
            value += this.readUnicodeRangePart(true);

            //ensure there's an actual unicode range here
            if (value.length == 2){
                reader.reset();
            } else {

                tt = Tokens.UNICODE_RANGE;

                //if there's a ? in the first part, there can't be a second part
                if (value.indexOf("?") == -1){

                    if (reader.peek() == "-"){
                        reader.mark();
                        temp = reader.read();
                        temp += this.readUnicodeRangePart(false);

                        //if there's not another value, back up and just take the first
                        if (temp.length == 1){
                            reader.reset();
                        } else {
                            value += temp;
                        }
                    }

                }
            }
        }

        return this.createToken(tt, value, startLine, startCol);
    },

    /**
     * Produces a S token based on the specified information. Since whitespace
     * may have multiple characters, this consumes all whitespace characters
     * into a single token.
     * @param {String} first The first character in the token.
     * @param {int} startLine The beginning line for the character.
     * @param {int} startCol The beginning column for the character.
     * @return {Object} A token object.
     * @method whitespaceToken
     */
    whitespaceToken: function(first, startLine, startCol){
        var reader  = this._reader,
            value   = first + this.readWhitespace();
        return this.createToken(Tokens.S, value, startLine, startCol);
    },




    //-------------------------------------------------------------------------
    // Methods to read values from the string stream
    //-------------------------------------------------------------------------

    readUnicodeRangePart: function(allowQuestionMark){
        var reader  = this._reader,
            part = "",
            c       = reader.peek();

        //first read hex digits
        while(isHexDigit(c) && part.length < 6){
            reader.read();
            part += c;
            c = reader.peek();
        }

        //then read question marks if allowed
        if (allowQuestionMark){
            while(c == "?" && part.length < 6){
                reader.read();
                part += c;
                c = reader.peek();
            }
        }

        //there can't be any other characters after this point

        return part;
    },

    readWhitespace: function(){
        var reader  = this._reader,
            whitespace = "",
            c       = reader.peek();

        while(isWhitespace(c)){
            reader.read();
            whitespace += c;
            c = reader.peek();
        }

        return whitespace;
    },
    readNumber: function(first){
        var reader  = this._reader,
            number  = first,
            hasDot  = (first == "."),
            c       = reader.peek();


        while(c){
            if (isDigit(c)){
                number += reader.read();
            } else if (c == "."){
                if (hasDot){
                    break;
                } else {
                    hasDot = true;
                    number += reader.read();
                }
            } else {
                break;
            }

            c = reader.peek();
        }

        return number;
    },
    readString: function(){
        var reader  = this._reader,
            delim   = reader.read(),
            string  = delim,
            prev    = delim,
            c       = reader.peek();

        while(c){
            c = reader.read();
            string += c;

            //if the delimiter is found with an escapement, we're done.
            if (c == delim && prev != "\\"){
                break;
            }

            //if there's a newline without an escapement, it's an invalid string
            if (isNewLine(reader.peek()) && c != "\\"){
                string = "";
                break;
            }

            //save previous and get next
            prev = c;
            c = reader.peek();
        }

        //if c is null, that means we're out of input and the string was never closed
        if (c === null){
            string = "";
        }

        return string;
    },
    readURI: function(first){
        var reader  = this._reader,
            uri     = first,
            inner   = "",
            c       = reader.peek();

        reader.mark();

        //skip whitespace before
        while(c && isWhitespace(c)){
            reader.read();
            c = reader.peek();
        }

        //it's a string
        if (c == "'" || c == "\""){
            inner = this.readString();
        } else {
            inner = this.readURL();
        }

        c = reader.peek();

        //skip whitespace after
        while(c && isWhitespace(c)){
            reader.read();
            c = reader.peek();
        }

        //if there was no inner value or the next character isn't closing paren, it's not a URI
        if (inner === "" || c != ")"){
            uri = first;
            reader.reset();
        } else {
            uri += inner + reader.read();
        }

        return uri;
    },
    readURL: function(){
        var reader  = this._reader,
            url     = "",
            c       = reader.peek();

        //TODO: Check for escape and nonascii
        while (/^[!#$%&\\*-~]$/.test(c)){
            url += reader.read();
            c = reader.peek();
        }

        return url;

    },
    readName: function(first){
        var reader  = this._reader,
            ident   = first || "",
            c       = reader.peek();

        while(true){
            if (c == "\\"){
                ident += this.readEscape(reader.read());
                c = reader.peek();
            } else if(c && isNameChar(c)){
                ident += reader.read();
                c = reader.peek();
            } else {
                break;
            }
        }

        return ident;
    },
    
    readEscape: function(first){
        var reader  = this._reader,
            cssEscape = first || "",
            i       = 0,
            c       = reader.peek();    
    
        if (isHexDigit(c)){
            do {
                cssEscape += reader.read();
                c = reader.peek();
            } while(c && isHexDigit(c) && ++i < 6);
        }
        
        if (cssEscape.length == 3 && /\s/.test(c) ||
            cssEscape.length == 7 || cssEscape.length == 1){
                reader.read();
        } else {
            c = "";
        }
        
        return cssEscape + c;
    },
    
    readComment: function(first){
        var reader  = this._reader,
            comment = first || "",
            c       = reader.read();

        if (c == "*"){
            while(c){
                comment += c;

                //look for end of comment
                if (comment.length > 2 && c == "*" && reader.peek() == "/"){
                    comment += reader.read();
                    break;
                }

                c = reader.read();
            }

            return comment;
        } else {
            return "";
        }

    }
});


var Tokens  = [

    /*
     * The following token names are defined in CSS3 Grammar: http://www.w3.org/TR/css3-syntax/#lexical
     */

    //HTML-style comments
    { name: "CDO"},
    { name: "CDC"},

    //ignorables
    { name: "S", whitespace: true/*, channel: "ws"*/},
    { name: "COMMENT", comment: true, hide: true, channel: "comment" },

    //attribute equality
    { name: "INCLUDES", text: "~="},
    { name: "DASHMATCH", text: "|="},
    { name: "PREFIXMATCH", text: "^="},
    { name: "SUFFIXMATCH", text: "$="},
    { name: "SUBSTRINGMATCH", text: "*="},

    //identifier types
    { name: "STRING"},
    { name: "IDENT"},
    { name: "HASH"},

    //at-keywords
    { name: "IMPORT_SYM", text: "@import"},
    { name: "PAGE_SYM", text: "@page"},
    { name: "MEDIA_SYM", text: "@media"},
    { name: "FONT_FACE_SYM", text: "@font-face"},
    { name: "CHARSET_SYM", text: "@charset"},
    { name: "NAMESPACE_SYM", text: "@namespace"},
    { name: "VIEWPORT_SYM", text: "@viewport"},
    { name: "UNKNOWN_SYM" },
    //{ name: "ATKEYWORD"},

    //CSS3 animations
    { name: "KEYFRAMES_SYM", text: [ "@keyframes", "@-webkit-keyframes", "@-moz-keyframes", "@-o-keyframes" ] },

    //important symbol
    { name: "IMPORTANT_SYM"},

    //measurements
    { name: "LENGTH"},
    { name: "ANGLE"},
    { name: "TIME"},
    { name: "FREQ"},
    { name: "DIMENSION"},
    { name: "PERCENTAGE"},
    { name: "NUMBER"},

    //functions
    { name: "URI"},
    { name: "FUNCTION"},

    //Unicode ranges
    { name: "UNICODE_RANGE"},

    /*
     * The following token names are defined in CSS3 Selectors: http://www.w3.org/TR/css3-selectors/#selector-syntax
     */

    //invalid string
    { name: "INVALID"},

    //combinators
    { name: "PLUS", text: "+" },
    { name: "GREATER", text: ">"},
    { name: "COMMA", text: ","},
    { name: "TILDE", text: "~"},

    //modifier
    { name: "NOT"},

    /*
     * Defined in CSS3 Paged Media
     */
    { name: "TOPLEFTCORNER_SYM", text: "@top-left-corner"},
    { name: "TOPLEFT_SYM", text: "@top-left"},
    { name: "TOPCENTER_SYM", text: "@top-center"},
    { name: "TOPRIGHT_SYM", text: "@top-right"},
    { name: "TOPRIGHTCORNER_SYM", text: "@top-right-corner"},
    { name: "BOTTOMLEFTCORNER_SYM", text: "@bottom-left-corner"},
    { name: "BOTTOMLEFT_SYM", text: "@bottom-left"},
    { name: "BOTTOMCENTER_SYM", text: "@bottom-center"},
    { name: "BOTTOMRIGHT_SYM", text: "@bottom-right"},
    { name: "BOTTOMRIGHTCORNER_SYM", text: "@bottom-right-corner"},
    { name: "LEFTTOP_SYM", text: "@left-top"},
    { name: "LEFTMIDDLE_SYM", text: "@left-middle"},
    { name: "LEFTBOTTOM_SYM", text: "@left-bottom"},
    { name: "RIGHTTOP_SYM", text: "@right-top"},
    { name: "RIGHTMIDDLE_SYM", text: "@right-middle"},
    { name: "RIGHTBOTTOM_SYM", text: "@right-bottom"},

    /*
     * The following token names are defined in CSS3 Media Queries: http://www.w3.org/TR/css3-mediaqueries/#syntax
     */
    /*{ name: "MEDIA_ONLY", state: "media"},
    { name: "MEDIA_NOT", state: "media"},
    { name: "MEDIA_AND", state: "media"},*/
    { name: "RESOLUTION", state: "media"},

    /*
     * The following token names are not defined in any CSS specification but are used by the lexer.
     */

    //not a real token, but useful for stupid IE filters
    { name: "IE_FUNCTION" },

    //part of CSS3 grammar but not the Flex code
    { name: "CHAR" },

    //TODO: Needed?
    //Not defined as tokens, but might as well be
    {
        name: "PIPE",
        text: "|"
    },
    {
        name: "SLASH",
        text: "/"
    },
    {
        name: "MINUS",
        text: "-"
    },
    {
        name: "STAR",
        text: "*"
    },

    {
        name: "LBRACE",
        text: "{"
    },
    {
        name: "RBRACE",
        text: "}"
    },
    {
        name: "LBRACKET",
        text: "["
    },
    {
        name: "RBRACKET",
        text: "]"
    },
    {
        name: "EQUALS",
        text: "="
    },
    {
        name: "COLON",
        text: ":"
    },
    {
        name: "SEMICOLON",
        text: ";"
    },

    {
        name: "LPAREN",
        text: "("
    },
    {
        name: "RPAREN",
        text: ")"
    },
    {
        name: "DOT",
        text: "."
    }
];

(function(){

    var nameMap = [],
        typeMap = {};

    Tokens.UNKNOWN = -1;
    Tokens.unshift({name:"EOF"});
    for (var i=0, len = Tokens.length; i < len; i++){
        nameMap.push(Tokens[i].name);
        Tokens[Tokens[i].name] = i;
        if (Tokens[i].text){
            if (Tokens[i].text instanceof Array){
                for (var j=0; j < Tokens[i].text.length; j++){
                    typeMap[Tokens[i].text[j]] = i;
                }
            } else {
                typeMap[Tokens[i].text] = i;
            }
        }
    }

    Tokens.name = function(tt){
        return nameMap[tt];
    };

    Tokens.type = function(c){
        return typeMap[c] || -1;
    };

})();




//This file will likely change a lot! Very experimental!
/*global Properties, ValidationTypes, ValidationError, PropertyValueIterator */
var Validation = {

    validate: function(property, value){
    
        //normalize name
        var name        = property.toString().toLowerCase(),
            parts       = value.parts,
            expression  = new PropertyValueIterator(value),
            spec        = Properties[name],
            part,
            valid,            
            j, count,
            msg,
            types,
            last,
            literals,
            max, multi, group;
            
        if (!spec) {
            if (name.indexOf("-") !== 0){    //vendor prefixed are ok
                throw new ValidationError("Unknown property '" + property + "'.", property.line, property.col);
            }
        } else if (typeof spec != "number"){
        
            //initialization
            if (typeof spec == "string"){
                if (spec.indexOf("||") > -1) {
                    this.groupProperty(spec, expression);
                } else {
                    this.singleProperty(spec, expression, 1);
                }

            } else if (spec.multi) {
                this.multiProperty(spec.multi, expression, spec.comma, spec.max || Infinity);
            } else if (typeof spec == "function") {
                spec(expression);
            }

        }

    },
    
    singleProperty: function(types, expression, max, partial) {

        var result      = false,
            value       = expression.value,
            count       = 0,
            part;
         
        while (expression.hasNext() && count < max) {
            result = ValidationTypes.isAny(expression, types);
            if (!result) {
                break;
            }
            count++;
        }
        
        if (!result) {
            if (expression.hasNext() && !expression.isFirst()) {
                part = expression.peek();
                throw new ValidationError("Expected end of value but found '" + part + "'.", part.line, part.col);
            } else {
                 throw new ValidationError("Expected (" + types + ") but found '" + value + "'.", value.line, value.col);
            }        
        } else if (expression.hasNext()) {
            part = expression.next();
            throw new ValidationError("Expected end of value but found '" + part + "'.", part.line, part.col);
        }          
                 
    },    
    
    multiProperty: function (types, expression, comma, max) {

        var result      = false,
            value       = expression.value,
            count       = 0,
            sep         = false,
            part;
            
        while(expression.hasNext() && !result && count < max) {
            if (ValidationTypes.isAny(expression, types)) {
                count++;
                if (!expression.hasNext()) {
                    result = true;

                } else if (comma) {
                    if (expression.peek() == ",") {
                        part = expression.next();
                    } else {
                        break;
                    }
                }
            } else {
                break;

            }
        }
        
        if (!result) {
            if (expression.hasNext() && !expression.isFirst()) {
                part = expression.peek();
                throw new ValidationError("Expected end of value but found '" + part + "'.", part.line, part.col);
            } else {
                part = expression.previous();
                if (comma && part == ",") {
                    throw new ValidationError("Expected end of value but found '" + part + "'.", part.line, part.col); 
                } else {
                    throw new ValidationError("Expected (" + types + ") but found '" + value + "'.", value.line, value.col);
                }
            }
        
        } else if (expression.hasNext()) {
            part = expression.next();
            throw new ValidationError("Expected end of value but found '" + part + "'.", part.line, part.col);
        }  

    },
    
    groupProperty: function (types, expression, comma) {

        var result      = false,
            value       = expression.value,
            typeCount   = types.split("||").length,
            groups      = { count: 0 },
            partial     = false,
            name,
            part;
            
        while(expression.hasNext() && !result) {
            name = ValidationTypes.isAnyOfGroup(expression, types);
            if (name) {
            
                //no dupes
                if (groups[name]) {
                    break;
                } else {
                    groups[name] = 1;
                    groups.count++;
                    partial = true;
                    
                    if (groups.count == typeCount || !expression.hasNext()) {
                        result = true;
                    }
                }
            } else {
                break;
            }
        }
        
        if (!result) {        
            if (partial && expression.hasNext()) {
                    part = expression.peek();
                    throw new ValidationError("Expected end of value but found '" + part + "'.", part.line, part.col);
            } else {
                throw new ValidationError("Expected (" + types + ") but found '" + value + "'.", value.line, value.col);
            }
        } else if (expression.hasNext()) {
            part = expression.next();
            throw new ValidationError("Expected end of value but found '" + part + "'.", part.line, part.col);
        }           
    }

    

};
/**
 * Type to use when a validation error occurs.
 * @class ValidationError
 * @namespace parserlib.util
 * @constructor
 * @param {String} message The error message.
 * @param {int} line The line at which the error occurred.
 * @param {int} col The column at which the error occurred.
 */
function ValidationError(message, line, col){

    /**
     * The column at which the error occurred.
     * @type int
     * @property col
     */
    this.col = col;

    /**
     * The line at which the error occurred.
     * @type int
     * @property line
     */
    this.line = line;

    /**
     * The text representation of the unit.
     * @type String
     * @property text
     */
    this.message = message;

}

//inherit from Error
ValidationError.prototype = new Error();
//This file will likely change a lot! Very experimental!
/*global Properties, Validation, ValidationError, PropertyValueIterator, console*/
var ValidationTypes = {

    isLiteral: function (part, literals) {
        var text = part.text.toString().toLowerCase(),
            args = literals.split(" | "),
            i, len, found = false;
        
        for (i=0,len=args.length; i < len && !found; i++){
            if (text == args[i].toLowerCase()){
                found = true;
            }
        }
        
        return found;    
    },
    
    isSimple: function(type) {
        return !!this.simple[type];
    },
    
    isComplex: function(type) {
        return !!this.complex[type];
    },
    
    /**
     * Determines if the next part(s) of the given expression
     * are any of the given types.
     */
    isAny: function (expression, types) {
        var args = types.split(" | "),
            i, len, found = false;
        
        for (i=0,len=args.length; i < len && !found && expression.hasNext(); i++){
            found = this.isType(expression, args[i]);
        }
        
        return found;    
    },
    
    /**
     * Determines if the next part(s) of the given expression
     * are one of a group.
     */
    isAnyOfGroup: function(expression, types) {
        var args = types.split(" || "),
            i, len, found = false;
        
        for (i=0,len=args.length; i < len && !found; i++){
            found = this.isType(expression, args[i]);
        }
        
        return found ? args[i-1] : false;
    },
    
    /**
     * Determines if the next part(s) of the given expression
     * are of a given type.
     */
    isType: function (expression, type) {
        var part = expression.peek(),
            result = false;
            
        if (type.charAt(0) != "<") {
            result = this.isLiteral(part, type);
            if (result) {
                expression.next();
            }
        } else if (this.simple[type]) {
            result = this.simple[type](part);
            if (result) {
                expression.next();
            }
        } else {
            result = this.complex[type](expression);
        }
        
        return result;
    },
    
    
    
    simple: {

        "<absolute-size>": function(part){
            return ValidationTypes.isLiteral(part, "xx-small | x-small | small | medium | large | x-large | xx-large");
        },
        
        "<attachment>": function(part){
            return ValidationTypes.isLiteral(part, "scroll | fixed | local");
        },
        
        "<attr>": function(part){
            return part.type == "function" && part.name == "attr";
        },
                
        "<bg-image>": function(part){
            return this["<image>"](part) || this["<gradient>"](part) ||  part == "none";
        },        
        
        "<gradient>": function(part) {
            return part.type == "function" && /^(?:\-(?:ms|moz|o|webkit)\-)?(?:repeating\-)?(?:radial\-|linear\-)?gradient/i.test(part);
        },
        
        "<box>": function(part){
            return ValidationTypes.isLiteral(part, "padding-box | border-box | content-box");
        },
        
        "<content>": function(part){
            return part.type == "function" && part.name == "content";
        },        
        
        "<relative-size>": function(part){
            return ValidationTypes.isLiteral(part, "smaller | larger");
        },
        
        //any identifier
        "<ident>": function(part){
            return part.type == "identifier";
        },
        
        "<length>": function(part){
            if (part.type == "function" && /^(?:\-(?:ms|moz|o|webkit)\-)?calc/i.test(part)){
                return true;
            }else{
                return part.type == "length" || part.type == "number" || part.type == "integer" || part == "0";
            }
        },
        
        "<color>": function(part){
            return part.type == "color" || part == "transparent";
        },
        
        "<number>": function(part){
            return part.type == "number" || this["<integer>"](part);
        },
        
        "<integer>": function(part){
            return part.type == "integer";
        },
        
        "<line>": function(part){
            return part.type == "integer";
        },
        
        "<angle>": function(part){
            return part.type == "angle";
        },        
        
        "<uri>": function(part){
            return part.type == "uri";
        },
        
        "<image>": function(part){
            return this["<uri>"](part);
        },
        
        "<percentage>": function(part){
            return part.type == "percentage" || part == "0";
        },

        "<border-width>": function(part){
            return this["<length>"](part) || ValidationTypes.isLiteral(part, "thin | medium | thick");
        },
        
        "<border-style>": function(part){
            return ValidationTypes.isLiteral(part, "none | hidden | dotted | dashed | solid | double | groove | ridge | inset | outset");
        },
        
        "<margin-width>": function(part){
            return this["<length>"](part) || this["<percentage>"](part) || ValidationTypes.isLiteral(part, "auto");
        },
        
        "<padding-width>": function(part){
            return this["<length>"](part) || this["<percentage>"](part);
        },
        
        "<shape>": function(part){
            return part.type == "function" && (part.name == "rect" || part.name == "inset-rect");
        },
        
        "<time>": function(part) {
            return part.type == "time";
        }
    },
    
    complex: {

        "<bg-position>": function(expression){
            var types   = this,
                result  = false,
                numeric = "<percentage> | <length>",
                xDir    = "left | right",
                yDir    = "top | bottom",
                count = 0,
                hasNext = function() {
                    return expression.hasNext() && expression.peek() != ",";
                };

            while (expression.peek(count) && expression.peek(count) != ",") {
                count++;
            }
            
/*
<position> = [
  [ left | center | right | top | bottom | <percentage> | <length> ]
|
  [ left | center | right | <percentage> | <length> ]
  [ top | center | bottom | <percentage> | <length> ]
|
  [ center | [ left | right ] [ <percentage> | <length> ]? ] &&
  [ center | [ top | bottom ] [ <percentage> | <length> ]? ]
]
*/

            if (count < 3) {
                if (ValidationTypes.isAny(expression, xDir + " | center | " + numeric)) {
                        result = true;
                        ValidationTypes.isAny(expression, yDir + " | center | " + numeric);
                } else if (ValidationTypes.isAny(expression, yDir)) {
                        result = true;
                        ValidationTypes.isAny(expression, xDir + " | center");
                }
            } else {
                if (ValidationTypes.isAny(expression, xDir)) {
                    if (ValidationTypes.isAny(expression, yDir)) {
                        result = true;
                        ValidationTypes.isAny(expression, numeric);
                    } else if (ValidationTypes.isAny(expression, numeric)) {
                        if (ValidationTypes.isAny(expression, yDir)) {
                            result = true;
                            ValidationTypes.isAny(expression, numeric);
                        } else if (ValidationTypes.isAny(expression, "center")) {
                            result = true;
                        }
                    }
                } else if (ValidationTypes.isAny(expression, yDir)) {
                    if (ValidationTypes.isAny(expression, xDir)) {
                        result = true;
                        ValidationTypes.isAny(expression, numeric);
                    } else if (ValidationTypes.isAny(expression, numeric)) {
                        if (ValidationTypes.isAny(expression, xDir)) {
                                result = true;
                                ValidationTypes.isAny(expression, numeric);
                        } else if (ValidationTypes.isAny(expression, "center")) {
                            result = true;
                        }
                    }
                } else if (ValidationTypes.isAny(expression, "center")) {
                    if (ValidationTypes.isAny(expression, xDir + " | " + yDir)) {
                        result = true;
                        ValidationTypes.isAny(expression, numeric);
                    }
                }
            }
            
            return result;
        },

        "<bg-size>": function(expression){
            //<bg-size> = [ <length> | <percentage> | auto ]{1,2} | cover | contain
            var types   = this,
                result  = false,
                numeric = "<percentage> | <length> | auto",
                part,
                i, len;      
      
            if (ValidationTypes.isAny(expression, "cover | contain")) {
                result = true;
            } else if (ValidationTypes.isAny(expression, numeric)) {
                result = true;                
                ValidationTypes.isAny(expression, numeric);
            }
            
            return result;
        },
        
        "<repeat-style>": function(expression){
            //repeat-x | repeat-y | [repeat | space | round | no-repeat]{1,2}
            var result  = false,
                values  = "repeat | space | round | no-repeat",
                part;
            
            if (expression.hasNext()){
                part = expression.next();
                
                if (ValidationTypes.isLiteral(part, "repeat-x | repeat-y")) {
                    result = true;                    
                } else if (ValidationTypes.isLiteral(part, values)) {
                    result = true;

                    if (expression.hasNext() && ValidationTypes.isLiteral(expression.peek(), values)) {
                        expression.next();
                    }
                }
            }
            
            return result;
            
        },
        
        "<shadow>": function(expression) {
            //inset? && [ <length>{2,4} && <color>? ]
            var result  = false,
                count   = 0,
                inset   = false,
                color   = false,
                part;
                
            if (expression.hasNext()) {            
                
                if (ValidationTypes.isAny(expression, "inset")){
                    inset = true;
                }
                
                if (ValidationTypes.isAny(expression, "<color>")) {
                    color = true;
                }                
                
                while (ValidationTypes.isAny(expression, "<length>") && count < 4) {
                    count++;
                }
                
                
                if (expression.hasNext()) {
                    if (!color) {
                        ValidationTypes.isAny(expression, "<color>");
                    }
                    
                    if (!inset) {
                        ValidationTypes.isAny(expression, "inset");
                    }

                }
                
                result = (count >= 2 && count <= 4);
            
            }
            
            return result;
        },
        
        "<x-one-radius>": function(expression) {
            //[ <length> | <percentage> ] [ <length> | <percentage> ]?
            var result  = false,
                simple = "<length> | <percentage> | inherit";
                
            if (ValidationTypes.isAny(expression, simple)){
                result = true;
                ValidationTypes.isAny(expression, simple);
            }                
            
            return result;
        }
    }
};



parserlib.css = {
Colors              :Colors,
Combinator          :Combinator,
Parser              :Parser,
PropertyName        :PropertyName,
PropertyValue       :PropertyValue,
PropertyValuePart   :PropertyValuePart,
MediaFeature        :MediaFeature,
MediaQuery          :MediaQuery,
Selector            :Selector,
SelectorPart        :SelectorPart,
SelectorSubPart     :SelectorSubPart,
Specificity         :Specificity,
TokenStream         :TokenStream,
Tokens              :Tokens,
ValidationError     :ValidationError
};
})();




(function(){
for(var prop in parserlib){
exports[prop] = parserlib[prop];
}
})();


/**
 * Main CSSLint object.
 * @class CSSLint
 * @static
 * @extends parserlib.util.EventTarget
 */
/*global parserlib, Reporter*/
var CSSLint = (function(){

    var rules           = [],
        formatters      = [],
        embeddedRuleset = /\/\*\s*csslint([^\*]*)\*\//, // Edited to allow whitespace before csslint, see CSSLint issue #549
        api             = new parserlib.util.EventTarget();

    api.version = "0.10.0";

    //-------------------------------------------------------------------------
    // Rule Management
    //-------------------------------------------------------------------------

    /**
     * Adds a new rule to the engine.
     * @param {Object} rule The rule to add.
     * @method addRule
     */
    api.addRule = function(rule){
        rules.push(rule);
        rules[rule.id] = rule;
    };

    /**
     * Clears all rule from the engine.
     * @method clearRules
     */
    api.clearRules = function(){
        rules = [];
    };

    /**
     * Returns the rule objects.
     * @return An array of rule objects.
     * @method getRules
     */
    api.getRules = function(){
        return [].concat(rules).sort(function(a,b){
            return a.id > b.id ? 1 : 0;
        });
    };

    /**
     * Returns a ruleset configuration object with all current rules.
     * @return A ruleset object.
     * @method getRuleset
     */
    api.getRuleset = function() {
        var ruleset = {},
            i = 0,
            len = rules.length;

        while (i < len){
            ruleset[rules[i++].id] = 1;    //by default, everything is a warning
        }

        return ruleset;
    };

    /**
     * Returns a ruleset object based on embedded rules.
     * @param {String} text A string of css containing embedded rules.
     * @param {Object} ruleset A ruleset object to modify.
     * @return {Object} A ruleset object.
     * @method getEmbeddedRuleset
     */
    function applyEmbeddedRuleset(text, ruleset){
        var valueMap,
            embedded = text && text.match(embeddedRuleset),
            rules = embedded && embedded[1];

        if (rules) {
            valueMap = {
                "true": 2,  // true is error
                "": 1,      // blank is warning
                "false": 0, // false is ignore

                "2": 2,     // explicit error
                "1": 1,     // explicit warning
                "0": 0      // explicit ignore
            };

            rules.toLowerCase().split(",").forEach(function(rule){
                var pair = rule.split(":"),
                    property = pair[0] || "",
                    value = pair[1] || "";

                ruleset[property.trim()] = valueMap[value.trim()];
            });
        }

        return ruleset;
    }

    //-------------------------------------------------------------------------
    // Formatters
    //-------------------------------------------------------------------------

    /**
     * Adds a new formatter to the engine.
     * @param {Object} formatter The formatter to add.
     * @method addFormatter
     */
    api.addFormatter = function(formatter) {
        // formatters.push(formatter);
        formatters[formatter.id] = formatter;
    };

    /**
     * Retrieves a formatter for use.
     * @param {String} formatId The name of the format to retrieve.
     * @return {Object} The formatter or undefined.
     * @method getFormatter
     */
    api.getFormatter = function(formatId){
        return formatters[formatId];
    };

    /**
     * Formats the results in a particular format for a single file.
     * @param {Object} result The results returned from CSSLint.verify().
     * @param {String} filename The filename for which the results apply.
     * @param {String} formatId The name of the formatter to use.
     * @param {Object} options (Optional) for special output handling.
     * @return {String} A formatted string for the results.
     * @method format
     */
    api.format = function(results, filename, formatId, options) {
        var formatter = this.getFormatter(formatId),
            result = null;

        if (formatter){
            result = formatter.startFormat();
            result += formatter.formatResults(results, filename, options || {});
            result += formatter.endFormat();
        }

        return result;
    };

    /**
     * Indicates if the given format is supported.
     * @param {String} formatId The ID of the format to check.
     * @return {Boolean} True if the format exists, false if not.
     * @method hasFormat
     */
    api.hasFormat = function(formatId){
        return formatters.hasOwnProperty(formatId);
    };

    //-------------------------------------------------------------------------
    // Verification
    //-------------------------------------------------------------------------

    /**
     * Starts the verification process for the given CSS text.
     * @param {String} text The CSS text to verify.
     * @param {Object} ruleset (Optional) List of rules to apply. If null, then
     *      all rules are used. If a rule has a value of 1 then it's a warning,
     *      a value of 2 means it's an error.
     * @return {Object} Results of the verification.
     * @method verify
     */
    api.verify = function(text, ruleset){

        var i       = 0,
            len     = rules.length,
            reporter,
            lines,
            report,
            parser = new parserlib.css.Parser({ starHack: true, ieFilters: true,
                                                underscoreHack: true, strict: false });

        // normalize line endings
        lines = text.replace(/\n\r?/g, "$split$").split('$split$');

        if (!ruleset){
            ruleset = this.getRuleset();
        }

        if (embeddedRuleset.test(text)){
            ruleset = applyEmbeddedRuleset(text, ruleset);
        }

        reporter = new Reporter(lines, ruleset);

        ruleset.errors = 2;       //always report parsing errors as errors
        for (i in ruleset){
            if(ruleset.hasOwnProperty(i) && ruleset[i]){
                if (rules[i]){
                    rules[i].init(parser, reporter);
                }
            }
        }


        //capture most horrible error type
        try {
            parser.parse(text);
        } catch (ex) {
            reporter.error("Fatal error, cannot continue: " + ex.message, ex.line, ex.col, {});
        }
        //ORION 8.0
        var tokens = (parser._tokenStream && parser._tokenStream.tokens) ? parser._tokenStream.tokens : [];
        report = {
            messages    : reporter.messages,
            stats       : reporter.stats,
            ruleset     : reporter.ruleset,
            tokens      : tokens,   //add the token array to the result ORION 8.0
            ast         : parser.ast //add in the AST from the  parser ORION 8.0
        };

        //sort by line numbers, rollups at the bottom
        report.messages.sort(function (a, b){
            if (a.rollup && !b.rollup){
                return 1;
            } else if (!a.rollup && b.rollup){
                return -1;
            } else {
                return a.line - b.line;
            }
        });

        return report;
    };

    //-------------------------------------------------------------------------
    // Publish the API
    //-------------------------------------------------------------------------

    return api;

})();

/*global CSSLint*/
/**
 * An instance of Report is used to report results of the
 * verification back to the main API.
 * @class Reporter
 * @constructor
 * @param {String[]} lines The text lines of the source.
 * @param {Object} ruleset The set of rules to work with, including if
 *      they are errors or warnings.
 */
function Reporter(lines, ruleset){

    /**
     * List of messages being reported.
     * @property messages
     * @type String[]
     */
    this.messages = [];

    /**
     * List of statistics being reported.
     * @property stats
     * @type String[]
     */
    this.stats = [];

    /**
     * Lines of code being reported on. Used to provide contextual information
     * for messages.
     * @property lines
     * @type String[]
     */
    this.lines = lines;
    
    /**
     * Information about the rules. Used to determine whether an issue is an
     * error or warning.
     * @property ruleset
     * @type Object
     */
    this.ruleset = ruleset;
}

Reporter.prototype = {

    //restore constructor
    constructor: Reporter,

    /**
     * Report an error.
     * @param {String} message The message to store.
     * @param {int} line The line number.
     * @param {int} col The column number.
     * @param {Object} rule The rule this message relates to.
     * @method error
     */
    error: function(message, line, col, rule){
        this.messages.push({
            type    : "error",
            line    : line,
            col     : col,
            message : message,
            evidence: this.lines[line-1],
            rule    : rule || {}
        });
    },

    /**
     * Report an warning.
     * @param {String} message The message to store.
     * @param {int} line The line number.
     * @param {int} col The column number.
     * @param {Object} rule The rule this message relates to.
     * @method warn
     * @deprecated Use report instead.
     */
    warn: function(message, line, col, rule){
        this.report(message, line, col, rule);
    },

    /**
     * Report an issue.
     * @param {String} message The message to store.
     * @param {int} line The line number.
     * @param {int} col The column number.
     * @param {Object} rule The rule this message relates to.
     * @method report
     */
    report: function(message, line, col, rule){
        this.messages.push({
            type    : this.ruleset[rule.id] == 2 ? "error" : "warning",
            line    : line,
            col     : col,
            message : message,
            evidence: this.lines[line-1],
            rule    : rule
        });
    },

    /**
     * Report some informational text.
     * @param {String} message The message to store.
     * @param {int} line The line number.
     * @param {int} col The column number.
     * @param {Object} rule The rule this message relates to.
     * @method info
     */
    info: function(message, line, col, rule){
        this.messages.push({
            type    : "info",
            line    : line,
            col     : col,
            message : message,
            evidence: this.lines[line-1],
            rule    : rule
        });
    },

    /**
     * Report some rollup error information.
     * @param {String} message The message to store.
     * @param {Object} rule The rule this message relates to.
     * @method rollupError
     */
    rollupError: function(message, rule){
        this.messages.push({
            type    : "error",
            rollup  : true,
            message : message,
            rule    : rule
        });
    },

    /**
     * Report some rollup warning information.
     * @param {String} message The message to store.
     * @param {Object} rule The rule this message relates to.
     * @method rollupWarn
     */
    rollupWarn: function(message, rule){
        this.messages.push({
            type    : "warning",
            rollup  : true,
            message : message,
            rule    : rule
        });
    },

    /**
     * Report a statistic.
     * @param {String} name The name of the stat to store.
     * @param {Variant} value The value of the stat.
     * @method stat
     */
    stat: function(name, value){
        this.stats[name] = value;
    }
};

//expose for testing purposes
CSSLint._Reporter = Reporter;

CSSLint.Colors = parserlib.util.Colors;   //ORION 8.0
/*global CSSLint*/

/*
 * Utility functions that make life easier.
 */
CSSLint.Util = {
    /*
     * Adds all properties from supplier onto receiver,
     * overwriting if the same name already exists on
     * reciever.
     * @param {Object} The object to receive the properties.
     * @param {Object} The object to provide the properties.
     * @return {Object} The receiver
     */
    mix: function(receiver, supplier){
        var prop;

        for (prop in supplier){
            if (supplier.hasOwnProperty(prop)){
                receiver[prop] = supplier[prop];
            }
        }

        return prop;
    },

    /*
     * Polyfill for array indexOf() method.
     * @param {Array} values The array to search.
     * @param {Variant} value The value to search for.
     * @return {int} The index of the value if found, -1 if not.
     */
    indexOf: function(values, value){
        if (values.indexOf){
            return values.indexOf(value);
        } else {
            for (var i=0, len=values.length; i < len; i++){
                if (values[i] === value){
                    return i;
                }
            }
            return -1;
        }
    },

    /*
     * Polyfill for array forEach() method.
     * @param {Array} values The array to operate on.
     * @param {Function} func The function to call on each item.
     * @return {void}
     */
    forEach: function(values, func) {
        if (values.forEach){
            return values.forEach(func);
        } else {
            for (var i=0, len=values.length; i < len; i++){
                func(values[i], i, values);
            }
        }
    }
};

/*global CSSLint*/
/*
 * Rule: Don't use adjoining classes (.foo.bar).
 */
CSSLint.addRule({

    //rule information
    id: "adjoining-classes",
    name: "Disallow adjoining classes",
    desc: "Don't use adjoining classes.",
    browsers: "IE6",

    //initialization
    init: function(parser, reporter){
        var rule = this;
        parser.addListener("startrule", function(event){
            var selectors = event.selectors,
                selector,
                part,
                modifier,
                classCount,
                i, j, k;

            for (i=0; i < selectors.length; i++){
                selector = selectors[i];
                for (j=0; j < selector.parts.length; j++){
                    part = selector.parts[j];
                    if (part.type == parser.SELECTOR_PART_TYPE){
                        classCount = 0;
                        for (k=0; k < part.modifiers.length; k++){
                            modifier = part.modifiers[k];
                            if (modifier.type == "class"){
                                classCount++;
                            }
                            if (classCount > 1){
                                reporter.report("Don't use adjoining classes.", part.line, part.col, rule);
                            }
                        }
                    }
                }
            }
        });
    }

});
/*global CSSLint*/

/*
 * Rule: Don't use width or height when using padding or border. 
 */
CSSLint.addRule({

    //rule information
    id: "box-model",
    name: "Beware of broken box size",
    desc: "Don't use width or height when using padding or border.",
    browsers: "All",

    //initialization
    init: function(parser, reporter){
        var rule = this,
            widthProperties = {
                border: 1,
                "border-left": 1,
                "border-right": 1,
                padding: 1,
                "padding-left": 1,
                "padding-right": 1
            },
            heightProperties = {
                border: 1,
                "border-bottom": 1,
                "border-top": 1,
                padding: 1,
                "padding-bottom": 1,
                "padding-top": 1
            },
            properties,
            boxSizing = false;

        function startRule(){
            properties = {};
            boxSizing = false;
        }

        function endRule(){
            var prop, value;
            
            if (!boxSizing) {
                if (properties.height){
                    for (prop in heightProperties){
                        if (heightProperties.hasOwnProperty(prop) && properties[prop]){
                            value = properties[prop].value;
                            //special case for padding
                            if (!(prop == "padding" && value.parts.length === 2 && value.parts[0].value === 0)){
                                reporter.report("Using height with " + prop + " can sometimes make elements larger than you expect.", properties[prop].line, properties[prop].col, rule);
                            }
                        }
                    }
                }

                if (properties.width){
                    for (prop in widthProperties){
                        if (widthProperties.hasOwnProperty(prop) && properties[prop]){
                            value = properties[prop].value;
                            
                            if (!(prop == "padding" && value.parts.length === 2 && value.parts[1].value === 0)){
                                reporter.report("Using width with " + prop + " can sometimes make elements larger than you expect.", properties[prop].line, properties[prop].col, rule);
                            }
                        }
                    }
                }   
            }     
        }

        parser.addListener("startrule", startRule);
        parser.addListener("startfontface", startRule);
        parser.addListener("startpage", startRule);
        parser.addListener("startpagemargin", startRule);
        parser.addListener("startkeyframerule", startRule); 

        parser.addListener("property", function(event){
            var name = event.property.text.toLowerCase();
            
            if (heightProperties[name] || widthProperties[name]){
                if (!/^0\S*$/.test(event.value) && !(name == "border" && event.value == "none")){
                    properties[name] = { line: event.property.line, col: event.property.col, value: event.value };
                }
            } else {
                if (/^(width|height)/i.test(name) && /^(length|percentage)/.test(event.value.parts[0].type)){
                    properties[name] = 1;
                } else if (name == "box-sizing") {
                    boxSizing = true;
                }
            }
            
        });

        parser.addListener("endrule", endRule);
        parser.addListener("endfontface", endRule);
        parser.addListener("endpage", endRule);
        parser.addListener("endpagemargin", endRule);
        parser.addListener("endkeyframerule", endRule);         
    }

});
/*global CSSLint*/

/*
 * Rule: box-sizing doesn't work in IE6 and IE7.
 */
CSSLint.addRule({

    //rule information
    id: "box-sizing",
    name: "Disallow use of box-sizing",
    desc: "The box-sizing properties isn't supported in IE6 and IE7.",
    browsers: "IE6, IE7",
    tags: ["Compatibility"],

    //initialization
    init: function(parser, reporter){
        var rule = this;

        parser.addListener("property", function(event){
            var name = event.property.text.toLowerCase();
   
            if (name == "box-sizing"){
                reporter.report("The box-sizing property isn't supported in IE6 and IE7.", event.line, event.col, rule);
            }
        });       
    }

});
/*
 * Rule: Use the bulletproof @font-face syntax to avoid 404's in old IE
 * (http://www.fontspring.com/blog/the-new-bulletproof-font-face-syntax)
 */
/*global CSSLint*/
CSSLint.addRule({

    //rule information
    id: "bulletproof-font-face",
    name: "Use the bulletproof @font-face syntax",
    desc: "Use the bulletproof @font-face syntax to avoid 404's in old IE (http://www.fontspring.com/blog/the-new-bulletproof-font-face-syntax).",
    browsers: "All",

    //initialization
    init: function(parser, reporter){
        var rule = this,
            count = 0,
            fontFaceRule = false,
            firstSrc     = true,
            ruleFailed    = false,
            line, col;

        // Mark the start of a @font-face declaration so we only test properties inside it
        parser.addListener("startfontface", function(event){
            fontFaceRule = true;
        });

        parser.addListener("property", function(event){
            // If we aren't inside an @font-face declaration then just return
            if (!fontFaceRule) {
                return;
            }

            var propertyName = event.property.toString().toLowerCase(),
                value        = event.value.toString();

            // Set the line and col numbers for use in the endfontface listener
            line = event.line;
            col  = event.col;

            // This is the property that we care about, we can ignore the rest
            if (propertyName === 'src') {
                var regex = /^\s?url\(['"].+\.eot\?.*['"]\)\s*format\(['"]embedded-opentype['"]\).*$/i;

                // We need to handle the advanced syntax with two src properties
                if (!value.match(regex) && firstSrc) {
                    ruleFailed = true;
                    firstSrc = false;
                } else if (value.match(regex) && !firstSrc) {
                    ruleFailed = false;
                }
            }


        });

        // Back to normal rules that we don't need to test
        parser.addListener("endfontface", function(event){
            fontFaceRule = false;

            if (ruleFailed) {
                reporter.report("@font-face declaration doesn't follow the fontspring bulletproof syntax.", line, col, rule);
            }
        });
    }
});
/*
 * Rule: Include all compatible vendor prefixes to reach a wider
 * range of users.
 */
/*global CSSLint*/ 
CSSLint.addRule({

    //rule information
    id: "compatible-vendor-prefixes",
    name: "Require compatible vendor prefixes",
    desc: "Include all compatible vendor prefixes to reach a wider range of users.",
    browsers: "All",

    //initialization
    init: function (parser, reporter) {
        var rule = this,
            compatiblePrefixes,
            properties,
            prop,
            variations,
            prefixed,
            i,
            len,
            inKeyFrame = false,
            arrayPush = Array.prototype.push,
            applyTo = [];

        // See http://peter.sh/experiments/vendor-prefixed-css-property-overview/ for details
        compatiblePrefixes = {
            "animation"                  : "webkit moz",
            "animation-delay"            : "webkit moz",
            "animation-direction"        : "webkit moz",
            "animation-duration"         : "webkit moz",
            "animation-fill-mode"        : "webkit moz",
            "animation-iteration-count"  : "webkit moz",
            "animation-name"             : "webkit moz",
            "animation-play-state"       : "webkit moz",
            "animation-timing-function"  : "webkit moz",
            "appearance"                 : "webkit moz",
            "border-end"                 : "webkit moz",
            "border-end-color"           : "webkit moz",
            "border-end-style"           : "webkit moz",
            "border-end-width"           : "webkit moz",
            "border-image"               : "webkit moz o",
            "border-radius"              : "webkit",
            "border-start"               : "webkit moz",
            "border-start-color"         : "webkit moz",
            "border-start-style"         : "webkit moz",
            "border-start-width"         : "webkit moz",
            "box-align"                  : "webkit moz ms",
            "box-direction"              : "webkit moz ms",
            "box-flex"                   : "webkit moz ms",
            "box-lines"                  : "webkit ms",
            "box-ordinal-group"          : "webkit moz ms",
            "box-orient"                 : "webkit moz ms",
            "box-pack"                   : "webkit moz ms",
            "box-sizing"                 : "webkit moz",
            "box-shadow"                 : "webkit moz",
            "column-count"               : "webkit moz ms",
            "column-gap"                 : "webkit moz ms",
            "column-rule"                : "webkit moz ms",
            "column-rule-color"          : "webkit moz ms",
            "column-rule-style"          : "webkit moz ms",
            "column-rule-width"          : "webkit moz ms",
            "column-width"               : "webkit moz ms",
            "hyphens"                    : "epub moz",
            "line-break"                 : "webkit ms",
            "margin-end"                 : "webkit moz",
            "margin-start"               : "webkit moz",
            "marquee-speed"              : "webkit wap",
            "marquee-style"              : "webkit wap",
            "padding-end"                : "webkit moz",
            "padding-start"              : "webkit moz",
            "tab-size"                   : "moz o",
            "text-size-adjust"           : "webkit ms",
            "transform"                  : "webkit moz ms o",
            "transform-origin"           : "webkit moz ms o",
            "transition"                 : "webkit moz o",
            "transition-delay"           : "webkit moz o",
            "transition-duration"        : "webkit moz o",
            "transition-property"        : "webkit moz o",
            "transition-timing-function" : "webkit moz o",
            "user-modify"                : "webkit moz",
            "user-select"                : "webkit moz ms",
            "word-break"                 : "epub ms",
            "writing-mode"               : "epub ms"
        };


        for (prop in compatiblePrefixes) {
            if (compatiblePrefixes.hasOwnProperty(prop)) {
                variations = [];
                prefixed = compatiblePrefixes[prop].split(' ');
                for (i = 0, len = prefixed.length; i < len; i++) {
                    variations.push('-' + prefixed[i] + '-' + prop);
                }
                compatiblePrefixes[prop] = variations;
                arrayPush.apply(applyTo, variations);
            }
        }
                
        parser.addListener("startrule", function () {
            properties = [];
        });

        parser.addListener("startkeyframes", function (event) {
            inKeyFrame = event.prefix || true;
        });

        parser.addListener("endkeyframes", function (event) {
            inKeyFrame = false;
        });

        parser.addListener("property", function (event) {
            var name = event.property;
            if (CSSLint.Util.indexOf(applyTo, name.text) > -1) {
            
                // e.g., -moz-transform is okay to be alone in @-moz-keyframes
                if (!inKeyFrame || typeof inKeyFrame != "string" || 
                        name.text.indexOf("-" + inKeyFrame + "-") !== 0) {
                    properties.push(name);
                }
            }
        });

        parser.addListener("endrule", function (event) {
            if (!properties.length) {
                return;
            }

            var propertyGroups = {},
                i,
                len,
                name,
                prop,
                variations,
                value,
                full,
                actual,
                item,
                propertiesSpecified;

            for (i = 0, len = properties.length; i < len; i++) {
                name = properties[i];

                for (prop in compatiblePrefixes) {
                    if (compatiblePrefixes.hasOwnProperty(prop)) {
                        variations = compatiblePrefixes[prop];
                        if (CSSLint.Util.indexOf(variations, name.text) > -1) {
                            if (!propertyGroups[prop]) {
                                propertyGroups[prop] = {
                                    full : variations.slice(0),
                                    actual : [],
                                    actualNodes: []
                                };
                            }
                            if (CSSLint.Util.indexOf(propertyGroups[prop].actual, name.text) === -1) {
                                propertyGroups[prop].actual.push(name.text);
                                propertyGroups[prop].actualNodes.push(name);
                            }
                        }
                    }
                }
            }

            for (prop in propertyGroups) {
                if (propertyGroups.hasOwnProperty(prop)) {
                    value = propertyGroups[prop];
                    full = value.full;
                    actual = value.actual;

                    if (full.length > actual.length) {
                        for (i = 0, len = full.length; i < len; i++) {
                            item = full[i];
                            if (CSSLint.Util.indexOf(actual, item) === -1) {
                                propertiesSpecified = (actual.length === 1) ? actual[0] : (actual.length == 2) ? actual.join(" and ") : actual.join(", ");
                                reporter.report("The property " + item + " is compatible with " + propertiesSpecified + " and should be included as well.", value.actualNodes[0].line, value.actualNodes[0].col, rule); 
                            }
                        }

                    }
                }
            }
        });
    }
});
/*
 * Rule: Certain properties don't play well with certain display values. 
 * - float should not be used with inline-block
 * - height, width, margin-top, margin-bottom, float should not be used with inline
 * - vertical-align should not be used with block
 * - margin, float should not be used with table-*
 */
/*global CSSLint*/
CSSLint.addRule({

    //rule information
    id: "display-property-grouping",
    name: "Require properties appropriate for display",
    desc: "Certain properties shouldn't be used with certain display property values.",
    browsers: "All",

    //initialization
    init: function(parser, reporter){
        var rule = this;

        var propertiesToCheck = {
                display: 1,
                "float": "none",
                height: 1,
                width: 1,
                margin: 1,
                "margin-left": 1,
                "margin-right": 1,
                "margin-bottom": 1,
                "margin-top": 1,
                padding: 1,
                "padding-left": 1,
                "padding-right": 1,
                "padding-bottom": 1,
                "padding-top": 1,
                "vertical-align": 1
            },
            properties;

        function reportProperty(name, display, msg){
            if (properties[name]){
                if (typeof propertiesToCheck[name] != "string" || properties[name].value.toLowerCase() != propertiesToCheck[name]){
                    reporter.report(msg || name + " can't be used with display: " + display + ".", properties[name].line, properties[name].col, rule);
                }
            }
        }
        
        function startRule(){
            properties = {};
        }

        function endRule(){

            var display = properties.display ? properties.display.value : null;
            if (display){
                switch(display){

                    case "inline":
                        //height, width, margin-top, margin-bottom, float should not be used with inline
                        reportProperty("height", display);
                        reportProperty("width", display);
                        reportProperty("margin", display);
                        reportProperty("margin-top", display);
                        reportProperty("margin-bottom", display);              
                        reportProperty("float", display, "display:inline has no effect on floated elements (but may be used to fix the IE6 double-margin bug).");
                        break;

                    case "block":
                        //vertical-align should not be used with block
                        reportProperty("vertical-align", display);
                        break;

                    case "inline-block":
                        //float should not be used with inline-block
                        reportProperty("float", display);
                        break;

                    default:
                        //margin, float should not be used with table
                        if (display.indexOf("table-") === 0){
                            reportProperty("margin", display);
                            reportProperty("margin-left", display);
                            reportProperty("margin-right", display);
                            reportProperty("margin-top", display);
                            reportProperty("margin-bottom", display);
                            reportProperty("float", display);
                        }

                        //otherwise do nothing
                }
            }
          
        }

        parser.addListener("startrule", startRule);
        parser.addListener("startfontface", startRule);
        parser.addListener("startkeyframerule", startRule);
        parser.addListener("startpagemargin", startRule);
        parser.addListener("startpage", startRule);

        parser.addListener("property", function(event){
            var name = event.property.text.toLowerCase();

            if (propertiesToCheck[name]){
                properties[name] = { value: event.value.text, line: event.property.line, col: event.property.col };                    
            }
        });

        parser.addListener("endrule", endRule);
        parser.addListener("endfontface", endRule);
        parser.addListener("endkeyframerule", endRule);
        parser.addListener("endpagemargin", endRule);
        parser.addListener("endpage", endRule);

    }

});
/*
 * Rule: Disallow duplicate background-images (using url).
 */
/*global CSSLint*/
CSSLint.addRule({

    //rule information
    id: "duplicate-background-images",
    name: "Disallow duplicate background images",
    desc: "Every background-image should be unique. Use a common class for e.g. sprites.",
    browsers: "All",

    //initialization
    init: function(parser, reporter){
        var rule = this,
            stack = {};

        parser.addListener("property", function(event){
            var name = event.property.text,
                value = event.value,
                i, len;

            if (name.match(/background/i)) {
                for (i=0, len=value.parts.length; i < len; i++) {
                    if (value.parts[i].type == 'uri') {
                        if (typeof stack[value.parts[i].uri] === 'undefined') {
                            stack[value.parts[i].uri] = event;
                        }
                        else {
                            reporter.report("Background image '" + value.parts[i].uri + "' was used multiple times, first declared at line " + stack[value.parts[i].uri].line + ", col " + stack[value.parts[i].uri].col + ".", event.line, event.col, rule);
                        }
                    }
                }
            }
        });
    }
});
/*
 * Rule: Duplicate properties must appear one after the other. If an already-defined
 * property appears somewhere else in the rule, then it's likely an error.
 */
/*global CSSLint*/
CSSLint.addRule({

    //rule information
    id: "duplicate-properties",
    name: "Disallow duplicate properties",
    desc: "Duplicate properties must appear one after the other.",
    browsers: "All",

    //initialization
    init: function(parser, reporter){
        var rule = this,
            properties,
            lastProperty;            
            
        function startRule(event){
            properties = {};        
        }
        
        parser.addListener("startrule", startRule);
        parser.addListener("startfontface", startRule);
        parser.addListener("startpage", startRule);
        parser.addListener("startpagemargin", startRule);
        parser.addListener("startkeyframerule", startRule);        
        
        parser.addListener("property", function(event){
            var property = event.property,
                name = property.text.toLowerCase();
            
            if (properties[name] && (lastProperty != name || properties[name] == event.value.text)){
                reporter.report("Duplicate property '" + event.property + "' found.", event.line, event.col, rule);
            }
            
            properties[name] = event.value.text;
            lastProperty = name;
                        
        });
            
        
    }

});
/*
 * Rule: Style rules without any properties defined should be removed.
 */
/*global CSSLint*/
CSSLint.addRule({

    //rule information
    id: "empty-rules",
    name: "Disallow empty rules",
    desc: "Rules without any properties specified should be removed.",
    browsers: "All",

    //initialization
    init: function(parser, reporter){
        var rule = this,
            count = 0;

        parser.addListener("startrule", function(){
            count=0;
        });

        parser.addListener("property", function(){
            count++;
        });

        parser.addListener("endrule", function(event){
            var selectors = event.selectors;
            if (count === 0){
                reporter.report("Rule is empty.", selectors[0].line, selectors[0].col, rule);
            }
        });
    }

});
/*
 * Rule: There should be no syntax errors. (Duh.)
 */
/*global CSSLint*/
CSSLint.addRule({

    //rule information
    id: "errors",
    name: "Parsing Errors",
    desc: "This rule looks for recoverable syntax errors.",
    browsers: "All",

    //initialization
    init: function(parser, reporter){
        var rule = this;

        parser.addListener("error", function(event){
            reporter.error(event.message, event.line, event.col, rule);
        });

    }

});

/*global CSSLint*/
CSSLint.addRule({

    //rule information
    id: "fallback-colors",
    name: "Require fallback colors",
    desc: "For older browsers that don't support RGBA, HSL, or HSLA, provide a fallback color.",
    browsers: "IE6,IE7,IE8",

    //initialization
    init: function(parser, reporter){
        var rule = this,
            lastProperty,
            propertiesToCheck = {
                color: 1,
                background: 1,
                "border-color": 1,
                "border-top-color": 1,
                "border-right-color": 1,
                "border-bottom-color": 1,
                "border-left-color": 1,
                border: 1,
                "border-top": 1,
                "border-right": 1,
                "border-bottom": 1,
                "border-left": 1,
                "background-color": 1
            },
            properties;
        
        function startRule(event){
            properties = {};    
            lastProperty = null;    
        }
        
        parser.addListener("startrule", startRule);
        parser.addListener("startfontface", startRule);
        parser.addListener("startpage", startRule);
        parser.addListener("startpagemargin", startRule);
        parser.addListener("startkeyframerule", startRule);        
        
        parser.addListener("property", function(event){
            var property = event.property,
                name = property.text.toLowerCase(),
                parts = event.value.parts,
                i = 0, 
                colorType = "",
                len = parts.length;                
                        
            if(propertiesToCheck[name]){
                while(i < len){
                    if (parts[i].type == "color"){
                        if ("alpha" in parts[i] || "hue" in parts[i]){
                            
                            if (/([^\)]+)\(/.test(parts[i])){
                                colorType = RegExp.$1.toUpperCase();
                            }
                            
                            if (!lastProperty || (lastProperty.property.text.toLowerCase() != name || lastProperty.colorType != "compat")){
                                reporter.report("Fallback " + name + " (hex or RGB) should precede " + colorType + " " + name + ".", event.line, event.col, rule);
                            }
                        } else {
                            event.colorType = "compat";
                        }
                    }
                    
                    i++;
                }
            }

            lastProperty = event;
        });        
         
    }

});
/*
 * Rule: You shouldn't use more than 10 floats. If you do, there's probably
 * room for some abstraction.
 */
/*global CSSLint*/
CSSLint.addRule({

    //rule information
    id: "floats",
    name: "Disallow too many floats",
    desc: "This rule tests if the float property is used too many times",
    browsers: "All",

    //initialization
    init: function(parser, reporter){
        var rule = this;
        var count = 0;

        //count how many times "float" is used
        parser.addListener("property", function(event){
            if (event.property.text.toLowerCase() == "float" &&
                    event.value.text.toLowerCase() != "none"){
                count++;
            }
        });

        //report the results
        parser.addListener("endstylesheet", function(){
            reporter.stat("floats", count);
            if (count >= 10){
                reporter.rollupWarn("Too many floats (" + count + "), you're probably using them for layout. Consider using a grid system instead.", rule);
            }
        });
    }

});
/*
 * Rule: Avoid too many @font-face declarations in the same stylesheet.
 */
/*global CSSLint*/
CSSLint.addRule({

    //rule information
    id: "font-faces",
    name: "Don't use too many web fonts",
    desc: "Too many different web fonts in the same stylesheet.",
    browsers: "All",

    //initialization
    init: function(parser, reporter){
        var rule = this,
            count = 0;


        parser.addListener("startfontface", function(){
            count++;
        });

        parser.addListener("endstylesheet", function(){
            if (count > 5){
                reporter.rollupWarn("Too many @font-face declarations (" + count + ").", rule);
            }
        });
    }

});
/*
 * Rule: You shouldn't need more than 9 font-size declarations.
 */

/*global CSSLint*/
CSSLint.addRule({

    //rule information
    id: "font-sizes",
    name: "Disallow too many font sizes",
    desc: "Checks the number of font-size declarations.",
    browsers: "All",

    //initialization
    init: function(parser, reporter){
        var rule = this,
            count = 0;

        //check for use of "font-size"
        parser.addListener("property", function(event){
            if (event.property == "font-size"){
                count++;
            }
        });

        //report the results
        parser.addListener("endstylesheet", function(){
            reporter.stat("font-sizes", count);
            if (count >= 10){
                reporter.rollupWarn("Too many font-size declarations (" + count + "), abstraction needed.", rule);
            }
        });
    }

});
/*
 * Rule: When using a vendor-prefixed gradient, make sure to use them all.
 */
/*global CSSLint*/
CSSLint.addRule({

    //rule information
    id: "gradients",
    name: "Require all gradient definitions",
    desc: "When using a vendor-prefixed gradient, make sure to use them all.",
    browsers: "All",

    //initialization
    init: function(parser, reporter){
        var rule = this,
            gradients;

        parser.addListener("startrule", function(){
            gradients = {
                moz: 0,
                webkit: 0,
                oldWebkit: 0,
                o: 0
            };
        });

        parser.addListener("property", function(event){

            if (/\-(moz|o|webkit)(?:\-(?:linear|radial))\-gradient/i.test(event.value)){
                gradients[RegExp.$1] = 1;
            } else if (/\-webkit\-gradient/i.test(event.value)){
                gradients.oldWebkit = 1;
            }

        });

        parser.addListener("endrule", function(event){
            var missing = [];

            if (!gradients.moz){
                missing.push("Firefox 3.6+");
            }

            if (!gradients.webkit){
                missing.push("Webkit (Safari 5+, Chrome)");
            }
            
            if (!gradients.oldWebkit){
                missing.push("Old Webkit (Safari 4+, Chrome)");
            }

            if (!gradients.o){
                missing.push("Opera 11.1+");
            }

            if (missing.length && missing.length < 4){            
                reporter.report("Missing vendor-prefixed CSS gradients for " + missing.join(", ") + ".", event.selectors[0].line, event.selectors[0].col, rule); 
            }

        });

    }

});

/*
 * Rule: Don't use IDs for selectors.
 */
/*global CSSLint*/
CSSLint.addRule({

    //rule information
    id: "ids",
    name: "Disallow IDs in selectors",
    desc: "Selectors should not contain IDs.",
    browsers: "All",

    //initialization
    init: function(parser, reporter){
        var rule = this;
        parser.addListener("startrule", function(event){
            var selectors = event.selectors,
                selector,
                part,
                modifier,
                idCount,
                i, j, k;

            for (i=0; i < selectors.length; i++){
                selector = selectors[i];
                idCount = 0;

                for (j=0; j < selector.parts.length; j++){
                    part = selector.parts[j];
                    if (part.type == parser.SELECTOR_PART_TYPE){
                        for (k=0; k < part.modifiers.length; k++){
                            modifier = part.modifiers[k];
                            if (modifier.type == "id"){
                                idCount++;
                            }
                        }
                    }
                }

                if (idCount == 1){
                    reporter.report("Don't use IDs in selectors.", selector.line, selector.col, rule);
                } else if (idCount > 1){
                    reporter.report(idCount + " IDs in the selector, really?", selector.line, selector.col, rule);
                }
            }

        });
    }

});
/*
 * Rule: Don't use @import, use <link> instead.
 */
/*global CSSLint*/
CSSLint.addRule({

    //rule information
    id: "import",
    name: "Disallow @import",
    desc: "Don't use @import, use <link> instead.",
    browsers: "All",

    //initialization
    init: function(parser, reporter){
        var rule = this;
        
        parser.addListener("import", function(event){        
            reporter.report("@import prevents parallel downloads, use <link> instead.", event.line, event.col, rule);
        });

    }

});
/*
 * Rule: Make sure !important is not overused, this could lead to specificity
 * war. Display a warning on !important declarations, an error if it's
 * used more at least 10 times.
 */
/*global CSSLint*/
CSSLint.addRule({

    //rule information
    id: "important",
    name: "Disallow !important",
    desc: "Be careful when using !important declaration",
    browsers: "All",

    //initialization
    init: function(parser, reporter){
        var rule = this,
            count = 0;

        //warn that important is used and increment the declaration counter
        parser.addListener("property", function(event){
            if (event.important === true){
                count++;
                reporter.report("Use of !important", event.line, event.col, rule);
            }
        });

        //if there are more than 10, show an error
        parser.addListener("endstylesheet", function(){
            reporter.stat("important", count);
            if (count >= 10){
                reporter.rollupWarn("Too many !important declarations (" + count + "), try to use less than 10 to avoid specificity issues.", rule);
            }
        });
    }

});
/*
 * Rule: Properties should be known (listed in CSS3 specification) or
 * be a vendor-prefixed property.
 */
/*global CSSLint*/
CSSLint.addRule({

    //rule information
    id: "known-properties",
    name: "Require use of known properties",
    desc: "Properties should be known (listed in CSS3 specification) or be a vendor-prefixed property.",
    browsers: "All",

    //initialization
    init: function(parser, reporter){
        var rule = this;

        parser.addListener("property", function(event){
            var name = event.property.text.toLowerCase();

            // the check is handled entirely by the parser-lib (https://github.com/nzakas/parser-lib)
            if (event.invalid) {
                reporter.report(event.invalid.message, event.line, event.col, rule);
            }

        });
    }

});
/*
 * Rule: outline: none or outline: 0 should only be used in a :focus rule
 *       and only if there are other properties in the same rule.
 */
/*global CSSLint*/
CSSLint.addRule({

    //rule information
    id: "outline-none",
    name: "Disallow outline: none",
    desc: "Use of outline: none or outline: 0 should be limited to :focus rules.",
    browsers: "All",
    tags: ["Accessibility"],

    //initialization
    init: function(parser, reporter){
        var rule = this,
            lastRule;

        function startRule(event){
            if (event.selectors){
                lastRule = {
                    line: event.line,
                    col: event.col,
                    selectors: event.selectors,
                    propCount: 0,
                    outline: false
                };
            } else {
                lastRule = null;
            }
        }
        
        function endRule(event){
            if (lastRule){
                if (lastRule.outline){
                    if (lastRule.selectors.toString().toLowerCase().indexOf(":focus") == -1){
                        reporter.report("Outlines should only be modified using :focus.", lastRule.line, lastRule.col, rule);
                    } else if (lastRule.propCount == 1) {
                        reporter.report("Outlines shouldn't be hidden unless other visual changes are made.", lastRule.line, lastRule.col, rule);                        
                    }
                }
            }
        }

        parser.addListener("startrule", startRule);
        parser.addListener("startfontface", startRule);
        parser.addListener("startpage", startRule);
        parser.addListener("startpagemargin", startRule);
        parser.addListener("startkeyframerule", startRule); 

        parser.addListener("property", function(event){
            var name = event.property.text.toLowerCase(),
                value = event.value;                
                
            if (lastRule){
                lastRule.propCount++;
                if (name == "outline" && (value == "none" || value == "0")){
                    lastRule.outline = true;
                }            
            }
            
        });
        
        parser.addListener("endrule", endRule);
        parser.addListener("endfontface", endRule);
        parser.addListener("endpage", endRule);
        parser.addListener("endpagemargin", endRule);
        parser.addListener("endkeyframerule", endRule); 

    }

});
/*
 * Rule: Don't use classes or IDs with elements (a.foo or a#foo).
 */
/*global CSSLint*/
CSSLint.addRule({

    //rule information
    id: "overqualified-elements",
    name: "Disallow overqualified elements",
    desc: "Don't use classes or IDs with elements (a.foo or a#foo).",
    browsers: "All",

    //initialization
    init: function(parser, reporter){
        var rule = this,
            classes = {};
            
        parser.addListener("startrule", function(event){
            var selectors = event.selectors,
                selector,
                part,
                modifier,
                i, j, k;

            for (i=0; i < selectors.length; i++){
                selector = selectors[i];

                for (j=0; j < selector.parts.length; j++){
                    part = selector.parts[j];
                    if (part.type == parser.SELECTOR_PART_TYPE){
                        for (k=0; k < part.modifiers.length; k++){
                            modifier = part.modifiers[k];
                            if (part.elementName && modifier.type == "id"){
                                reporter.report("Element (" + part + ") is overqualified, just use " + modifier + " without element name.", part.line, part.col, rule);
                            } else if (modifier.type == "class"){
                                
                                if (!classes[modifier]){
                                    classes[modifier] = [];
                                }
                                classes[modifier].push({ modifier: modifier, part: part });
                            }
                        }
                    }
                }
            }
        });
        
        parser.addListener("endstylesheet", function(){
        
            var prop;
            for (prop in classes){
                if (classes.hasOwnProperty(prop)){
                
                    //one use means that this is overqualified
                    if (classes[prop].length == 1 && classes[prop][0].part.elementName){
                        reporter.report("Element (" + classes[prop][0].part + ") is overqualified, just use " + classes[prop][0].modifier + " without element name.", classes[prop][0].part.line, classes[prop][0].part.col, rule);
                    }
                }
            }        
        });
    }

});
/*
 * Rule: Headings (h1-h6) should not be qualified (namespaced).
 */
/*global CSSLint*/
CSSLint.addRule({

    //rule information
    id: "qualified-headings",
    name: "Disallow qualified headings",
    desc: "Headings should not be qualified (namespaced).",
    browsers: "All",

    //initialization
    init: function(parser, reporter){
        var rule = this;

        parser.addListener("startrule", function(event){
            var selectors = event.selectors,
                selector,
                part,
                i, j;

            for (i=0; i < selectors.length; i++){
                selector = selectors[i];

                for (j=0; j < selector.parts.length; j++){
                    part = selector.parts[j];
                    if (part.type == parser.SELECTOR_PART_TYPE){
                        if (part.elementName && /h[1-6]/.test(part.elementName.toString()) && j > 0){
                            reporter.report("Heading (" + part.elementName + ") should not be qualified.", part.line, part.col, rule);
                        }
                    }
                }
            }
        });
    }

});
/*
 * Rule: Selectors that look like regular expressions are slow and should be avoided.
 */
/*global CSSLint*/
CSSLint.addRule({

    //rule information
    id: "regex-selectors",
    name: "Disallow selectors that look like regexs",
    desc: "Selectors that look like regular expressions are slow and should be avoided.",
    browsers: "All",

    //initialization
    init: function(parser, reporter){
        var rule = this;

        parser.addListener("startrule", function(event){
            var selectors = event.selectors,
                selector,
                part,
                modifier,
                i, j, k;

            for (i=0; i < selectors.length; i++){
                selector = selectors[i];
                for (j=0; j < selector.parts.length; j++){
                    part = selector.parts[j];
                    if (part.type == parser.SELECTOR_PART_TYPE){
                        for (k=0; k < part.modifiers.length; k++){
                            modifier = part.modifiers[k];
                            if (modifier.type == "attribute"){
                                if (/([\~\|\^\$\*]=)/.test(modifier)){
                                    reporter.report("Attribute selectors with " + RegExp.$1 + " are slow!", modifier.line, modifier.col, rule);
                                }
                            }

                        }
                    }
                }
            }
        });
    }

});
/*
 * Rule: Total number of rules should not exceed x.
 */
/*global CSSLint*/
CSSLint.addRule({

    //rule information
    id: "rules-count",
    name: "Rules Count",
    desc: "Track how many rules there are.",
    browsers: "All",

    //initialization
    init: function(parser, reporter){
        var rule = this,
            count = 0;

        //count each rule
        parser.addListener("startrule", function(){
            count++;
        });

        parser.addListener("endstylesheet", function(){
            reporter.stat("rule-count", count);
        });
    }

});
/*
 * Rule: Warn people with approaching the IE 4095 limit 
 */
/*global CSSLint*/
CSSLint.addRule({

    //rule information
    id: "selector-max-approaching",
    name: "Warn when approaching the 4095 selector limit for IE",
    desc: "Will warn when selector count is >= 3800 selectors.",
    browsers: "IE",

    //initialization
    init: function(parser, reporter) {
        var rule = this, count = 0;

        parser.addListener('startrule', function(event) {
            count += event.selectors.length;
        });

        parser.addListener("endstylesheet", function() {
            if (count >= 3800) {
                reporter.report("You have " + count + " selectors. Internet Explorer supports a maximum of 4095 selectors per stylesheet. Consider refactoring.",0,0,rule); 
            }
        });
    }

});

/*
 * Rule: Warn people past the IE 4095 limit 
 */
/*global CSSLint*/
CSSLint.addRule({

    //rule information
    id: "selector-max",
    name: "Error when past the 4095 selector limit for IE",
    desc: "Will error when selector count is > 4095.",
    browsers: "IE",

    //initialization
    init: function(parser, reporter){
        var rule = this, count = 0;

        parser.addListener('startrule',function(event) {
            count += event.selectors.length;
        });

        parser.addListener("endstylesheet", function() {
            if (count > 4095) {
                reporter.report("You have " + count + " selectors. Internet Explorer supports a maximum of 4095 selectors per stylesheet. Consider refactoring.",0,0,rule); 
            }
        });
    }

});
/*
 * Rule: Use shorthand properties where possible.
 * 
 */
/*global CSSLint*/
CSSLint.addRule({

    //rule information
    id: "shorthand",
    name: "Require shorthand properties",
    desc: "Use shorthand properties where possible.",
    browsers: "All",
    
    //initialization
    init: function(parser, reporter){
        var rule = this,
            prop, i, len,
            propertiesToCheck = {},
            properties,
            mapping = {
                "margin": [
                    "margin-top",
                    "margin-bottom",
                    "margin-left",
                    "margin-right"
                ],
                "padding": [
                    "padding-top",
                    "padding-bottom",
                    "padding-left",
                    "padding-right"
                ]              
            };
            
        //initialize propertiesToCheck 
        for (prop in mapping){
            if (mapping.hasOwnProperty(prop)){
                for (i=0, len=mapping[prop].length; i < len; i++){
                    propertiesToCheck[mapping[prop][i]] = prop;
                }
            }
        }
            
        function startRule(event){
            properties = {};
        }
        
        //event handler for end of rules
        function endRule(event){
            
            var prop, i, len, total;
            
            //check which properties this rule has
            for (prop in mapping){
                if (mapping.hasOwnProperty(prop)){
                    total=0;
                    
                    for (i=0, len=mapping[prop].length; i < len; i++){
                        total += properties[mapping[prop][i]] ? 1 : 0;
                    }
                    
                    if (total == mapping[prop].length){
                        reporter.report("The properties " + mapping[prop].join(", ") + " can be replaced by " + prop + ".", event.line, event.col, rule);
                    }
                }
            }
        }        
        
        parser.addListener("startrule", startRule);
        parser.addListener("startfontface", startRule);
    
        //check for use of "font-size"
        parser.addListener("property", function(event){
            var name = event.property.toString().toLowerCase(),
                value = event.value.parts[0].value;

            if (propertiesToCheck[name]){
                properties[name] = 1;
            }
        });

        parser.addListener("endrule", endRule);
        parser.addListener("endfontface", endRule);     

    }

});
/*
 * Rule: Don't use properties with a star prefix.
 *
 */
/*global CSSLint*/
CSSLint.addRule({

    //rule information
    id: "star-property-hack",
    name: "Disallow properties with a star prefix",
    desc: "Checks for the star property hack (targets IE6/7)",
    browsers: "All",

    //initialization
    init: function(parser, reporter){
        var rule = this;

        //check if property name starts with "*"
        parser.addListener("property", function(event){
            var property = event.property;

            if (property.hack == "*") {
                reporter.report("Property with star prefix found.", event.property.line, event.property.col, rule);
            }
        });
    }
});
/*
 * Rule: Don't use text-indent for image replacement if you need to support rtl.
 *
 */
/*global CSSLint*/
CSSLint.addRule({

    //rule information
    id: "text-indent",
    name: "Disallow negative text-indent",
    desc: "Checks for text indent less than -99px",
    browsers: "All",

    //initialization
    init: function(parser, reporter){
        var rule = this,
            textIndent,
            direction;


        function startRule(event){
            textIndent = false;
            direction = "inherit";
        }

        //event handler for end of rules
        function endRule(event){
            if (textIndent && direction != "ltr"){
                reporter.report("Negative text-indent doesn't work well with RTL. If you use text-indent for image replacement explicitly set direction for that item to ltr.", textIndent.line, textIndent.col, rule);
            }
        }

        parser.addListener("startrule", startRule);
        parser.addListener("startfontface", startRule);

        //check for use of "font-size"
        parser.addListener("property", function(event){
            var name = event.property.toString().toLowerCase(),
                value = event.value;

            if (name == "text-indent" && value.parts[0].value < -99){
                textIndent = event.property;
            } else if (name == "direction" && value == "ltr"){
                direction = "ltr";
            }
        });

        parser.addListener("endrule", endRule);
        parser.addListener("endfontface", endRule);

    }

});
/*
 * Rule: Don't use properties with a underscore prefix.
 *
 */
/*global CSSLint*/
CSSLint.addRule({

    //rule information
    id: "underscore-property-hack",
    name: "Disallow properties with an underscore prefix",
    desc: "Checks for the underscore property hack (targets IE6)",
    browsers: "All",

    //initialization
    init: function(parser, reporter){
        var rule = this;

        //check if property name starts with "_"
        parser.addListener("property", function(event){
            var property = event.property;

            if (property.hack == "_") {
                reporter.report("Property with underscore prefix found.", event.property.line, event.property.col, rule);
            }
        });
    }
});
/*
 * Rule: Headings (h1-h6) should be defined only once.
 */
/*global CSSLint*/
CSSLint.addRule({

    //rule information
    id: "unique-headings",
    name: "Headings should only be defined once",
    desc: "Headings should be defined only once.",
    browsers: "All",

    //initialization
    init: function(parser, reporter){
        var rule = this;

        var headings =  {
                h1: 0,
                h2: 0,
                h3: 0,
                h4: 0,
                h5: 0,
                h6: 0
            };

        parser.addListener("startrule", function(event){
            var selectors = event.selectors,
                selector,
                part,
                pseudo,
                i, j;

            for (i=0; i < selectors.length; i++){
                selector = selectors[i];
                part = selector.parts[selector.parts.length-1];

                if (part.elementName && /(h[1-6])/i.test(part.elementName.toString())){
                    
                    for (j=0; j < part.modifiers.length; j++){
                        if (part.modifiers[j].type == "pseudo"){
                            pseudo = true;
                            break;
                        }
                    }
                
                    if (!pseudo){
                        headings[RegExp.$1]++;
                        if (headings[RegExp.$1] > 1) {
                            reporter.report("Heading (" + part.elementName + ") has already been defined.", part.line, part.col, rule);
                        }
                    }
                }
            }
        });
        
        parser.addListener("endstylesheet", function(event){
            var prop,
                messages = [];
                
            for (prop in headings){
                if (headings.hasOwnProperty(prop)){
                    if (headings[prop] > 1){
                        messages.push(headings[prop] + " " + prop + "s");
                    }
                }
            }
            
            if (messages.length){
                reporter.rollupWarn("You have " + messages.join(", ") + " defined in this stylesheet.", rule);
            }
        });        
    }

});
/*
 * Rule: Don't use universal selector because it's slow.
 */
/*global CSSLint*/
CSSLint.addRule({

    //rule information
    id: "universal-selector",
    name: "Disallow universal selector",
    desc: "The universal selector (*) is known to be slow.",
    browsers: "All",

    //initialization
    init: function(parser, reporter){
        var rule = this;

        parser.addListener("startrule", function(event){
            var selectors = event.selectors,
                selector,
                part,
                modifier,
                i, j, k;

            for (i=0; i < selectors.length; i++){
                selector = selectors[i];
                
                part = selector.parts[selector.parts.length-1];
                if (part.elementName == "*"){
                    reporter.report(rule.desc, part.line, part.col, rule);
                }
            }
        });
    }

});
/*
 * Rule: Don't use unqualified attribute selectors because they're just like universal selectors.
 */
/*global CSSLint*/
CSSLint.addRule({

    //rule information
    id: "unqualified-attributes",
    name: "Disallow unqualified attribute selectors",
    desc: "Unqualified attribute selectors are known to be slow.",
    browsers: "All",

    //initialization
    init: function(parser, reporter){
        var rule = this;

        parser.addListener("startrule", function(event){
            
            var selectors = event.selectors,
                selector,
                part,
                modifier,
                i, j, k;

            for (i=0; i < selectors.length; i++){
                selector = selectors[i];
                
                part = selector.parts[selector.parts.length-1];
                if (part.type == parser.SELECTOR_PART_TYPE){
                    for (k=0; k < part.modifiers.length; k++){
                        modifier = part.modifiers[k];
                        if (modifier.type == "attribute" && (!part.elementName || part.elementName == "*")){
                            reporter.report(rule.desc, part.line, part.col, rule);                               
                        }
                    }
                }
                
            }            
        });
    }

});
/*
 * Rule: When using a vendor-prefixed property, make sure to
 * include the standard one.
 */
/*global CSSLint*/
CSSLint.addRule({

    //rule information
    id: "vendor-prefix",
    name: "Require standard property with vendor prefix",
    desc: "When using a vendor-prefixed property, make sure to include the standard one.",
    browsers: "All",

    //initialization
    init: function(parser, reporter){
        var rule = this,
            properties,
            num,
            propertiesToCheck = {
                "-webkit-border-radius": "border-radius",
                "-webkit-border-top-left-radius": "border-top-left-radius",
                "-webkit-border-top-right-radius": "border-top-right-radius",
                "-webkit-border-bottom-left-radius": "border-bottom-left-radius",
                "-webkit-border-bottom-right-radius": "border-bottom-right-radius",
                
                "-o-border-radius": "border-radius",
                "-o-border-top-left-radius": "border-top-left-radius",
                "-o-border-top-right-radius": "border-top-right-radius",
                "-o-border-bottom-left-radius": "border-bottom-left-radius",
                "-o-border-bottom-right-radius": "border-bottom-right-radius",
                
                "-moz-border-radius": "border-radius",
                "-moz-border-radius-topleft": "border-top-left-radius",
                "-moz-border-radius-topright": "border-top-right-radius",
                "-moz-border-radius-bottomleft": "border-bottom-left-radius",
                "-moz-border-radius-bottomright": "border-bottom-right-radius",                
                
                "-moz-column-count": "column-count",
                "-webkit-column-count": "column-count",
                
                "-moz-column-gap": "column-gap",
                "-webkit-column-gap": "column-gap",
                
                "-moz-column-rule": "column-rule",
                "-webkit-column-rule": "column-rule",
                
                "-moz-column-rule-style": "column-rule-style",
                "-webkit-column-rule-style": "column-rule-style",
                
                "-moz-column-rule-color": "column-rule-color",
                "-webkit-column-rule-color": "column-rule-color",
                
                "-moz-column-rule-width": "column-rule-width",
                "-webkit-column-rule-width": "column-rule-width",
                
                "-moz-column-width": "column-width",
                "-webkit-column-width": "column-width",
                
                "-webkit-column-span": "column-span",
                "-webkit-columns": "columns",
                
                "-moz-box-shadow": "box-shadow",
                "-webkit-box-shadow": "box-shadow",
                
                "-moz-transform" : "transform",
                "-webkit-transform" : "transform",
                "-o-transform" : "transform",
                "-ms-transform" : "transform",
                
                "-moz-transform-origin" : "transform-origin",
                "-webkit-transform-origin" : "transform-origin",
                "-o-transform-origin" : "transform-origin",
                "-ms-transform-origin" : "transform-origin",
                
                "-moz-box-sizing" : "box-sizing",
                "-webkit-box-sizing" : "box-sizing",
                
                "-moz-user-select" : "user-select",
                "-khtml-user-select" : "user-select",
                "-webkit-user-select" : "user-select"                
            };

        //event handler for beginning of rules
        function startRule(){
            properties = {};
            num=1;        
        }
        
        //event handler for end of rules
        function endRule(event){
            var prop,
                i, len,
                standard,
                needed,
                actual,
                needsStandard = [];

            for (prop in properties){
                if (propertiesToCheck[prop]){
                    needsStandard.push({ actual: prop, needed: propertiesToCheck[prop]});
                }
            }

            for (i=0, len=needsStandard.length; i < len; i++){
                needed = needsStandard[i].needed;
                actual = needsStandard[i].actual;

                if (!properties[needed]){               
                    reporter.report("Missing standard property '" + needed + "' to go along with '" + actual + "'.", properties[actual][0].name.line, properties[actual][0].name.col, rule);
                } else {
                    //make sure standard property is last
                    if (properties[needed][0].pos < properties[actual][0].pos){
                        reporter.report("Standard property '" + needed + "' should come after vendor-prefixed property '" + actual + "'.", properties[actual][0].name.line, properties[actual][0].name.col, rule);
                    }
                }
            }

        }        
        
        parser.addListener("startrule", startRule);
        parser.addListener("startfontface", startRule);
        parser.addListener("startpage", startRule);
        parser.addListener("startpagemargin", startRule);
        parser.addListener("startkeyframerule", startRule);         

        parser.addListener("property", function(event){
            var name = event.property.text.toLowerCase();

            if (!properties[name]){
                properties[name] = [];
            }

            properties[name].push({ name: event.property, value : event.value, pos:num++ });
        });

        parser.addListener("endrule", endRule);
        parser.addListener("endfontface", endRule);
        parser.addListener("endpage", endRule);
        parser.addListener("endpagemargin", endRule);
        parser.addListener("endkeyframerule", endRule);         
    }

});
/*
 * Rule: You don't need to specify units when a value is 0.
 */
/*global CSSLint*/
CSSLint.addRule({

    //rule information
    id: "zero-units",
    name: "Disallow units for 0 values",
    desc: "You don't need to specify units when a value is 0.",
    browsers: "All",

    //initialization
    init: function(parser, reporter){
        var rule = this;

        //count how many times "float" is used
        parser.addListener("property", function(event){
            var parts = event.value.parts,
                i = 0, 
                len = parts.length;

            while(i < len){
                if ((parts[i].units || parts[i].type == "percentage") && parts[i].value === 0 && parts[i].type != "time"){
                    reporter.report("Values of 0 shouldn't have units specified.", parts[i].line, parts[i].col, rule);
                }
                i++;
            }

        });

    }

});
/*global CSSLint*/
(function() {

    /**
     * Replace special characters before write to output.
     *
     * Rules:
     *  - single quotes is the escape sequence for double-quotes
     *  - &amp; is the escape sequence for &
     *  - &lt; is the escape sequence for <
     *  - &gt; is the escape sequence for >
     *
     * @param {String} message to escape
     * @return escaped message as {String}
     */
    var xmlEscape = function(str) {
        if (!str || str.constructor !== String) {
            return "";
        }
        
        return str.replace(/[\"&><]/g, function(match) {
            switch (match) {
                case "\"":
                    return "&quot;";
                case "&":
                    return "&amp;";
                case "<":
                    return "&lt;";
                case ">":
                    return "&gt;";            
            }
        });
    };

    CSSLint.addFormatter({
        //format information
        id: "checkstyle-xml",
        name: "Checkstyle XML format",

        /**
         * Return opening root XML tag.
         * @return {String} to prepend before all results
         */
        startFormat: function(){
            return "<?xml version=\"1.0\" encoding=\"utf-8\"?><checkstyle>";
        },

        /**
         * Return closing root XML tag.
         * @return {String} to append after all results
         */
        endFormat: function(){
            return "</checkstyle>";
        },
        
        /**
         * Returns message when there is a file read error.
         * @param {String} filename The name of the file that caused the error.
         * @param {String} message The error message
         * @return {String} The error message.
         */
        readError: function(filename, message) {
            return "<file name=\"" + xmlEscape(filename) + "\"><error line=\"0\" column=\"0\" severty=\"error\" message=\"" + xmlEscape(message) + "\"></error></file>";
        },

        /**
         * Given CSS Lint results for a file, return output for this format.
         * @param results {Object} with error and warning messages
         * @param filename {String} relative file path
         * @param options {Object} (UNUSED for now) specifies special handling of output
         * @return {String} output for results
         */
        formatResults: function(results, filename, options) {
            var messages = results.messages,
                output = [];

            /**
             * Generate a source string for a rule.
             * Checkstyle source strings usually resemble Java class names e.g
             * net.csslint.SomeRuleName
             * @param {Object} rule
             * @return rule source as {String}
             */
            var generateSource = function(rule) {
                if (!rule || !('name' in rule)) {
                    return "";
                }
                return 'net.csslint.' + rule.name.replace(/\s/g,'');
            };



            if (messages.length > 0) {
                output.push("<file name=\""+filename+"\">");
                CSSLint.Util.forEach(messages, function (message, i) {
                    //ignore rollups for now
                    if (!message.rollup) {
                      output.push("<error line=\"" + message.line + "\" column=\"" + message.col + "\" severity=\"" + message.type + "\"" +
                          " message=\"" + xmlEscape(message.message) + "\" source=\"" + generateSource(message.rule) +"\"/>");
                    }
                });
                output.push("</file>");
            }

            return output.join("");
        }
    });

}());
/*global CSSLint*/
CSSLint.addFormatter({
    //format information
    id: "compact",
    name: "Compact, 'porcelain' format",

    /**
     * Return content to be printed before all file results.
     * @return {String} to prepend before all results
     */
    startFormat: function() {
        return "";
    },

    /**
     * Return content to be printed after all file results.
     * @return {String} to append after all results
     */
    endFormat: function() {
        return "";
    },

    /**
     * Given CSS Lint results for a file, return output for this format.
     * @param results {Object} with error and warning messages
     * @param filename {String} relative file path
     * @param options {Object} (Optional) specifies special handling of output
     * @return {String} output for results
     */
    formatResults: function(results, filename, options) {
        var messages = results.messages,
            output = "";
        options = options || {};

        /**
         * Capitalize and return given string.
         * @param str {String} to capitalize
         * @return {String} capitalized
         */
        var capitalize = function(str) {
            return str.charAt(0).toUpperCase() + str.slice(1);
        };

        if (messages.length === 0) {
            return options.quiet ? "" : filename + ": Lint Free!";
        }

        CSSLint.Util.forEach(messages, function(message, i) {
            if (message.rollup) {
                output += filename + ": " + capitalize(message.type) + " - " + message.message + "\n";
            } else {
                output += filename + ": " + "line " + message.line + 
                    ", col " + message.col + ", " + capitalize(message.type) + " - " + message.message + "\n";
            }
        });
    
        return output;
    }
});
/*global CSSLint*/
CSSLint.addFormatter({
    //format information
    id: "csslint-xml",
    name: "CSSLint XML format",

    /**
     * Return opening root XML tag.
     * @return {String} to prepend before all results
     */
    startFormat: function(){
        return "<?xml version=\"1.0\" encoding=\"utf-8\"?><csslint>";
    },

    /**
     * Return closing root XML tag.
     * @return {String} to append after all results
     */
    endFormat: function(){
        return "</csslint>";
    },

    /**
     * Given CSS Lint results for a file, return output for this format.
     * @param results {Object} with error and warning messages
     * @param filename {String} relative file path
     * @param options {Object} (UNUSED for now) specifies special handling of output
     * @return {String} output for results
     */
    formatResults: function(results, filename, options) {
        var messages = results.messages,
            output = [];

        /**
         * Replace special characters before write to output.
         *
         * Rules:
         *  - single quotes is the escape sequence for double-quotes
         *  - &amp; is the escape sequence for &
         *  - &lt; is the escape sequence for <
         *  - &gt; is the escape sequence for >
         * 
         * @param {String} message to escape
         * @return escaped message as {String}
         */
        var escapeSpecialCharacters = function(str) {
            if (!str || str.constructor !== String) {
                return "";
            }
            return str.replace(/\"/g, "'").replace(/&/g, "&amp;").replace(/</g, "&lt;").replace(/>/g, "&gt;");
        };

        if (messages.length > 0) {
            output.push("<file name=\""+filename+"\">");
            CSSLint.Util.forEach(messages, function (message, i) {
                if (message.rollup) {
                    output.push("<issue severity=\"" + message.type + "\" reason=\"" + escapeSpecialCharacters(message.message) + "\" evidence=\"" + escapeSpecialCharacters(message.evidence) + "\"/>");
                } else {
                    output.push("<issue line=\"" + message.line + "\" char=\"" + message.col + "\" severity=\"" + message.type + "\"" +
                        " reason=\"" + escapeSpecialCharacters(message.message) + "\" evidence=\"" + escapeSpecialCharacters(message.evidence) + "\"/>");
                }
            });
            output.push("</file>");
        }

        return output.join("");
    }
});
/*global CSSLint*/
CSSLint.addFormatter({
    //format information
    id: "junit-xml",
    name: "JUNIT XML format",

    /**
     * Return opening root XML tag.
     * @return {String} to prepend before all results
     */
    startFormat: function(){
        return "<?xml version=\"1.0\" encoding=\"utf-8\"?><testsuites>";
    },

    /**
     * Return closing root XML tag.
     * @return {String} to append after all results
     */
    endFormat: function() {
        return "</testsuites>";
    },

    /**
     * Given CSS Lint results for a file, return output for this format.
     * @param results {Object} with error and warning messages
     * @param filename {String} relative file path
     * @param options {Object} (UNUSED for now) specifies special handling of output
     * @return {String} output for results
     */
    formatResults: function(results, filename, options) {

        var messages = results.messages,
            output = [],
            tests = {
                'error': 0,
                'failure': 0
            };

        /**
         * Generate a source string for a rule.
         * JUNIT source strings usually resemble Java class names e.g
         * net.csslint.SomeRuleName
         * @param {Object} rule
         * @return rule source as {String}
         */
        var generateSource = function(rule) {
            if (!rule || !('name' in rule)) {
                return "";
            }
            return 'net.csslint.' + rule.name.replace(/\s/g,'');
        };

        /**
         * Replace special characters before write to output.
         *
         * Rules:
         *  - single quotes is the escape sequence for double-quotes
         *  - &lt; is the escape sequence for <
         *  - &gt; is the escape sequence for >
         *
         * @param {String} message to escape
         * @return escaped message as {String}
         */
        var escapeSpecialCharacters = function(str) {

            if (!str || str.constructor !== String) {
                return "";
            }

            return str.replace(/\"/g, "'").replace(/</g, "&lt;").replace(/>/g, "&gt;");

        };

        if (messages.length > 0) {

            messages.forEach(function (message, i) {

                // since junit has no warning class
                // all issues as errors
                var type = message.type === 'warning' ? 'error' : message.type;

                //ignore rollups for now
                if (!message.rollup) {

                    // build the test case seperately, once joined
                    // we'll add it to a custom array filtered by type
                    output.push("<testcase time=\"0\" name=\"" + generateSource(message.rule) + "\">");
                    output.push("<" + type + " message=\"" + escapeSpecialCharacters(message.message) + "\"><![CDATA[" + message.line + ':' + message.col + ':' + escapeSpecialCharacters(message.evidence)  + "]]></" + type + ">");
                    output.push("</testcase>");

                    tests[type] += 1;

                }

            });

            output.unshift("<testsuite time=\"0\" tests=\"" + messages.length + "\" skipped=\"0\" errors=\"" + tests.error + "\" failures=\"" + tests.failure + "\" package=\"net.csslint\" name=\"" + filename + "\">");
            output.push("</testsuite>");

        }

        return output.join("");

    }
});
/*global CSSLint*/
CSSLint.addFormatter({
    //format information
    id: "lint-xml",
    name: "Lint XML format",

    /**
     * Return opening root XML tag.
     * @return {String} to prepend before all results
     */
    startFormat: function(){
        return "<?xml version=\"1.0\" encoding=\"utf-8\"?><lint>";
    },

    /**
     * Return closing root XML tag.
     * @return {String} to append after all results
     */
    endFormat: function(){
        return "</lint>";
    },

    /**
     * Given CSS Lint results for a file, return output for this format.
     * @param results {Object} with error and warning messages
     * @param filename {String} relative file path
     * @param options {Object} (UNUSED for now) specifies special handling of output
     * @return {String} output for results
     */
    formatResults: function(results, filename, options) {
        var messages = results.messages,
            output = [];

        /**
         * Replace special characters before write to output.
         *
         * Rules:
         *  - single quotes is the escape sequence for double-quotes
         *  - &amp; is the escape sequence for &
         *  - &lt; is the escape sequence for <
         *  - &gt; is the escape sequence for >
         * 
         * @param {String} message to escape
         * @return escaped message as {String}
         */
        var escapeSpecialCharacters = function(str) {
            if (!str || str.constructor !== String) {
                return "";
            }
            return str.replace(/\"/g, "'").replace(/&/g, "&amp;").replace(/</g, "&lt;").replace(/>/g, "&gt;");
        };

        if (messages.length > 0) {
        
            output.push("<file name=\""+filename+"\">");
            CSSLint.Util.forEach(messages, function (message, i) {
                if (message.rollup) {
                    output.push("<issue severity=\"" + message.type + "\" reason=\"" + escapeSpecialCharacters(message.message) + "\" evidence=\"" + escapeSpecialCharacters(message.evidence) + "\"/>");
                } else {
                    output.push("<issue line=\"" + message.line + "\" char=\"" + message.col + "\" severity=\"" + message.type + "\"" +
                        " reason=\"" + escapeSpecialCharacters(message.message) + "\" evidence=\"" + escapeSpecialCharacters(message.evidence) + "\"/>");
                }
            });
            output.push("</file>");
        }

        return output.join("");
    }
});
/*global CSSLint*/
CSSLint.addFormatter({
    //format information
    id: "text",
    name: "Plain Text",

    /**
     * Return content to be printed before all file results.
     * @return {String} to prepend before all results
     */
    startFormat: function() {
        return "";
    },

    /**
     * Return content to be printed after all file results.
     * @return {String} to append after all results
     */
    endFormat: function() {
        return "";
    },

    /**
     * Given CSS Lint results for a file, return output for this format.
     * @param results {Object} with error and warning messages
     * @param filename {String} relative file path
     * @param options {Object} (Optional) specifies special handling of output
     * @return {String} output for results
     */
    formatResults: function(results, filename, options) {
        var messages = results.messages,
            output = "";
        options = options || {};

        if (messages.length === 0) {
            return options.quiet ? "" : "\n\ncsslint: No errors in " + filename + ".";
        }

        output = "\n\ncsslint: There are " + messages.length  +  " problems in " + filename + ".";
        var pos = filename.lastIndexOf("/"),
            shortFilename = filename;

        if (pos === -1){
            pos = filename.lastIndexOf("\\");       
        }
        if (pos > -1){
            shortFilename = filename.substring(pos+1);
        }

        CSSLint.Util.forEach(messages, function (message, i) {
            output = output + "\n\n" + shortFilename;
            if (message.rollup) {
                output += "\n" + (i+1) + ": " + message.type;
                output += "\n" + message.message;
            } else {
                output += "\n" + (i+1) + ": " + message.type + " at line " + message.line + ", col " + message.col;
                output += "\n" + message.message;
                output += "\n" + message.evidence;
            }
        });
    
        return output;
    }
});

return CSSLint;
})();

return cssLint;
});
/*******************************************************************************
 * @license
 * Copyright (c) 2015 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env amd*/
define('webtools/cssVisitor',[
], function() {

	var Visitor = {
        
        BREAK: 1,
        SKIP: 2,
        
        /**
         * Visit the given AST top-down node -> attributes -> value -> end visit
         * @param {Object} dom	The AST to visit
         * @param {Function} callback The visitor callback 
         */
        visit: function visit(ast, callback) {
            if(Array.isArray(ast.body) && callback && typeof(callback.visitNode) === 'function') {
                for(var i = 0; i < ast.body.length; i++) {
                    var ret = visitNode(callback, ast.body[i], null);
                    endVisitNode(callback, ast.body[i]);
                    if(ret === this.BREAK) {
                        return;
                    }
                }    
            }
        }
    };

	/**
	 * Visits a given node with a given callback function. if the function does not implement #visitNode this function does nothing
	 * @param {Function} callback The visitor callback. Can optionally implement #visitNode
	 * @param {Object} node The current node being visited
	 * @param {Object} last the last node that was visited
	 * @returns {Number} Returns #Break or #SKIP or nothing   
	 */
    function visitNode(callback, node, last) {
    	if(typeof(callback.visitNode) === 'function') {
	        node.parent = last;
	        var ret = callback.visitNode(node);
	        if(ret === Visitor.BREAK || ret === Visitor.SKIP) {
	            return ret;
	        } 
	        if(Array.isArray(node.body)) {
	            ret = visitNodes(node, node.body, callback);
	            if(ret === Visitor.BREAK) {
		            return ret;
		        }
	        } else if(Array.isArray(node.declarations)) {
				ret = visitNodes(node, node.declarations, callback);
	            if(ret === Visitor.BREAK) {
		            return ret;
		        }	        	
	        } else if(Array.isArray(node.properties)) {
				ret = visitNodes(node, node.properties, callback);
	            if(ret === Visitor.BREAK) {
		            return ret;
		        }	        	
	        }
        }
    }
    
    function visitNodes(node, arr, callback) {
    	for(var i = 0; i < arr.length; i++) {
	        var _n = arr[i];
	        var ret = callback.visitNode(_n, node);
	        if(typeof(callback.endVisitNode) === 'function') {
	    		callback.endVisitNode(_n);
	    	}
	        if(ret === Visitor.SKIP) {
	        	continue;
	        } else if(ret === Visitor.BREAK) {
	            return ret;
	        }
	    }
    }
    
    /**
     * Ends the visit on the given node with the given callback. Allows for post-processing when we are going to leave a node. If the callback does not implement
     * #endVisitNode this function does nothing
     * @param {Function} callback The visitor callback. Can optionally implement #endVisitNode
     * @param {Object} node The node we are ending the visit for
     */
    function endVisitNode(callback, node) {
    	if(typeof(callback.endVisitNode) === 'function') {
    		callback.endVisitNode(node);
    	}
    }

    return Visitor;    
});
/*******************************************************************************
 * @license
 * Copyright (c) 2011, 2015 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
define("webtools/cssContentAssist", [
	'orion/Deferred',
	'orion/editor/templates',
	'orion/editor/stylers/text_css/syntax',
	'orion/objects',
	'webtools/util',
	'javascript/compilationUnit',
	'csslint/csslint',
	'webtools/cssVisitor',
	'i18n!webtools/nls/messages'
], function(Deferred, mTemplates, mCSS, Objects, Util, CU, CSSLint, Visitor, Messages) {

/* eslint-disable missing-nls */
	var overflowValues = {
		type: "link",
		values: ["visible", "hidden", "scroll", "auto", "no-display", "no-content"]
	};
	var fontStyleValues = {
		type: "link",
		values: ["italic", "normal", "oblique", "inherit"]
	};
	var fontWeightValues = {
		type: "link",
		values: [
			"bold", "normal", "bolder", "lighter",
			"100", "200", "300", "400", "500", "600",
			"700", "800", "900", "inherit"
		]
	};
	var displayValues = {
		type: "link",
		values: [
			"none", "block", "box", "flex", "inline", "inline-block", "inline-flex", "inline-table",
			"list-item", "table", "table-caption", "table-cell", "table-column", "table-column-group",
			"table-footer-group", "table-header-group", "table-row", "table-row-group", "inherit"
		]
	};
	var visibilityValues = {
		type: "link",
		values: ["hidden", "visible", "collapse", "inherit"]
	};
	var positionValues = {
		type: "link",
		values: ["absolute", "fixed", "relative", "static", "inherit"]
	};
	var whitespaceValues = {
		type: "link",
		values: ["pre", "pre-line", "pre-wrap", "nowrap", "normal", "inherit"]
	};
	var wordwrapValues = {
		type: "link",
		values: ["normal", "break-word"] 
	};
	var floatValues = {
		type: "link",
		values: ["left", "right", "none", "inherit"] 
	};
	var borderStyles = {
		type: "link",
		values: ["solid", "dashed", "dotted", "double", "groove", "ridge", "inset", "outset"]
	};
	var widths = {
		type: "link",
		values: []
	};
	widths.values.push('0');
	for (var i=1; i<10; i++) {
		widths.values.push(i.toString() + 'px');
	}
	var colorValues = {
		type: "link",
		values: Object.keys(CSSLint.Colors)
	};
	var cursorValues = {
		type: "link",
		values: [
			"auto",
			"crosshair",
			"default",
			"e-resize",
			"help",
			"move",
			"n-resize",
			"ne-resize",
			"nw-resize",
			"pointer",
			"progress",
			"s-resize",
			"se-resize",
			"sw-resize",
			"text",
			"w-resize",
			"wait",
			"inherit"
		]
	};
	var csslintRules = {
		type: "link",
		values: [
			"adjoining-classes",
			"box-model",
			"box-sizing",
			"bulletproof-font-face",
			"compatible-vendor-prefixes",
			"display-property-grouping",
			"duplicate-background-images",
			"duplicate-properties",
			"empty-rules",
			"fallback-colors",
			"floats",
			"font-faces",
			"font-sizes",
			"gradients",
			"ids",
			"import",
			"important",
			"known-properties",
			"outline-none",
			"overqualified-elements",
			"qualified-headings",
			"regex-selectors",
			"rules-count",
			"selector-max-approaching",
			"selector-max",
			"shorthand",
			"star-property-hack",
			"text-indent",
			"underscore-property-hack",
			"unique-headings",
			"universal-selector",
			"unqualified-attributes",
			"vendor-prefix",
			"zero-units"
		]
	};
	var severityValues = {
		type: "link",
		values: [
			"false",
			"true",
			"0",
			"1",
			"2"
		]
	};
	
	var valuesProperties = [
		{prop: "display", values: displayValues},
		{prop: "overflow", values: overflowValues},
		{prop: "overflow-x", values: overflowValues},
		{prop: "overflow-y", values: overflowValues},
		{prop: "float", values: floatValues},
		{prop: "position", values: positionValues},
		{prop: "cursor", values: cursorValues},
		{prop: "color", values: colorValues},
		{prop: "border-top-color", values: colorValues},
		{prop: "border-bottom-color", values: colorValues},
		{prop: "border-right-color", values: colorValues},
		{prop: "border-left-color", values: colorValues},
		{prop: "background-color", values: colorValues},
		{prop: "font-style", values: fontStyleValues},
		{prop: "font-weight", values: fontWeightValues},
		{prop: "white-space", values: whitespaceValues},
		{prop: "word-wrap", values: wordwrapValues},
		{prop: "visibility", values: visibilityValues}
	];
	
	var pixelProperties = [
		"width", "height", "top", "bottom", "left", "right",
		"min-width", "min-height", "max-width", "max-height",
		"margin", "padding", "padding-left", "padding-right",
		"padding-top", "padding-bottom", "margin-left", "margin-top",
		"margin-bottom", "margin-right"
	];
	
	var borderProperties = ["border", "border-top", "border-bottom", "border-left", "border-right"];
/* eslint-enable missing-nls */

	function fromJSON(o) {
		return JSON.stringify(o).replace("}", "\\}");  //$NON-NLS-1$
	}
	
	var templates = [
		{
			prefix: "rule", //$NON-NLS-1$
			description: Messages['ruleTemplateDescription'],
			template: ".${class} {\n\t${cursor}\n}" //$NON-NLS-1$
		},
		{
			prefix: "rule", //$NON-NLS-1$
			description: Messages['idSelectorTemplateDescription'],
			template: "#${id} {\n\t${cursor}\n}" //$NON-NLS-1$
		},
		{
			prefix: "outline", //$NON-NLS-1$
			description: Messages['outlineStyleTemplateDescription'],
			template: "outline: ${color:" + fromJSON(colorValues) + "} ${style:" + fromJSON(borderStyles) + "} ${width:" + fromJSON(widths) + "};" //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
		},
		{
			prefix: "background-image", //$NON-NLS-1$
			description: Messages['backgroundImageTemplateDescription'],
			template: "background-image: url(\"${uri}\");" //$NON-NLS-1$
		},
		{
			prefix: "url", //$NON-NLS-1$
			description: Messages['urlImageTemplateDescription'],
			template: "url(\"${uri}\");" //$NON-NLS-1$
		},
		{
			prefix: "rgb", //$NON-NLS-1$
			description: Messages['rgbColourTemplateDescription'],
			template: "rgb(${red},${green},${blue});" //$NON-NLS-1$
		},
		{
			prefix: "@import", //$NON-NLS-1$
			description: Messages['importTemplateDescription'],
			template: "@import \"${uri}\";" //$NON-NLS-1$
		},
		{
			prefix: "csslint", //$NON-NLS-1$
			description: Messages['csslintTemplateDescription'],
			template: "\/*csslint ${:" + fromJSON(csslintRules) + "}: ${a:" + fromJSON(severityValues) + "} *\/" //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
		}
	];
	
	var prop;
	for (i=0; i<valuesProperties.length; i++) {
		prop = valuesProperties[i];
		templates.push({
			prefix: prop.prop,
			description: prop.prop + " - " + prop.prop + " style", //$NON-NLS-1$ //$NON-NLS-2$
			template: prop.prop + ": ${value:" + fromJSON(prop.values) + "};" //$NON-NLS-1$ //$NON-NLS-2$
		});
	}	
	
	for (i=0; i<pixelProperties.length; i++) {
		prop = pixelProperties[i];
		templates.push({
			prefix: prop,
			description: prop + " - " + prop + " pixel style", //$NON-NLS-1$ //$NON-NLS-2$
			template: prop  + ": ${value}px;" //$NON-NLS-1$
		});
	}
	var fourWidthsProperties = ["padding", "margin"]; //$NON-NLS-1$ //$NON-NLS-2$
	for (i=0; i<fourWidthsProperties.length; i++) {
		prop = fourWidthsProperties[i];
		templates.push({
			prefix: prop,
			description: prop + " - " + prop + " top right bottom left style", //$NON-NLS-1$ //$NON-NLS-2$
			template: prop  + ": ${top}px ${left}px ${bottom}px ${right}px;" //$NON-NLS-1$
		});
		templates.push({
			prefix: prop,
			description: prop + " - " + prop + " top right,left bottom style", //$NON-NLS-1$ //$NON-NLS-2$
			template: prop  + ": ${top}px ${right_left}px ${bottom}px;" //$NON-NLS-1$
		});
		templates.push({
			prefix: prop,
			description: prop + " - " + prop + " top,bottom right,left style", //$NON-NLS-1$ //$NON-NLS-2$
			template: prop  + ": ${top_bottom}px ${right_left}px" //$NON-NLS-1$
		});
	}
	
	for (i=0; i<borderProperties.length; i++) {
		prop = borderProperties[i];
		templates.push({
			prefix: prop,
			description: prop + " - " + prop + " style", //$NON-NLS-1$ //$NON-NLS-2$
			template: prop + ": ${width:" + fromJSON(widths) + "} ${style:" + fromJSON(borderStyles) + "} ${color:" + fromJSON(colorValues) + "};" //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
		});
	}
	
	/**
	 * @description Create a proxy template provider
	 * @returns {TemplateProvider} A new TemplateProvider
	 * @since 10.0
	 */
	function TemplateProvider() {
	}
	
	TemplateProvider.prototype = new mTemplates.TemplateContentAssist(mCSS.keywords, templates);
	
	Objects.mixin(TemplateProvider.prototype, {
		/**
         * @callback 
         */
        getKeywordProposals: function(prefix) {
			var proposals = [];
			var keywords = this.getKeywords();
			if (keywords) {
				for (i = 0; i < keywords.length; i++) {
					if (keywords[i].indexOf(prefix) === 0) {
						var _p = keywords[i].substring(prefix.length);
						proposals.push({
							proposal: _p, 
							description: keywords[i], 
							style: 'emphasis', //$NON-NLS-1$
							kind: 'css' //$NON-NLS-1$
						});
					}
				}
				
				if (0 < proposals.length) {
					proposals.splice(0, 0,{
						proposal: '',
						description: Messages['keywordsAssistTitle'],
						style: 'noemphasis_title_keywords', //$NON-NLS-1$
						unselectable: true,
						kind: 'css' //$NON-NLS-1$
					});	
				}
			}
			return proposals;
		},
		/**
		 * @callback 
		 */
		getTemplateProposals: function(prefix, offset, context) {
			var proposals = [];
			var _tmplates = this.getTemplates();
			for (var t = 0; t < _tmplates.length; t++) {
				var template = _tmplates[t];
				if (template.match(prefix)) {
					var proposal = template.getProposal(prefix, offset, context);
					proposal.style = 'emphasis'; //$NON-NLS-1$
					var obj = Object.create(null);
			        obj.type = 'markdown'; //$NON-NLS-1$
			        obj.content = Messages['templateHoverHeader'];
			        obj.content += proposal.proposal;
			        proposal.hover = obj;
			        proposal.kind = 'css'; //$NON-NLS-1$
			        this.removePrefix(prefix, proposal);
					proposals.push(proposal);
				}
			}
			
			if (0 < proposals.length) {
				//sort the proposals by name
				proposals.sort(function(p1, p2) {
					if (p1.name < p2.name) return -1;
					if (p1.name > p2.name) return 1;
					return 0;
				});
				// if any templates were added to the list of 
				// proposals, add a title as the first element
				proposals.splice(0, 0, {
					proposal: '',
					description: Messages['templateAssistHeader'],
					style: 'noemphasis_title', //$NON-NLS-0$
					unselectable: true
				});
			}
			
			return proposals;
		}
	});
	
	/**
	 * @name orion.editor.CssContentAssistProvider
	 * @class Provides content assist for CSS keywords.
	 * @param {CssResultManager} resultManager The backing reult manager
	 */
	function CssContentAssistProvider(resultManager) {
	    this._resultManager = resultManager;
	}
	var templateAssist = new TemplateProvider();
	
	Objects.mixin(CssContentAssistProvider.prototype, {
	   /**
	    * @callback
	    */
	   getPrefix: function getPrefix(buffer, offset, context) {
	       var index = offset;
    		while (index && /[A-Za-z\-\@]/.test(buffer.charAt(index - 1))) {
    			index--;
    		}
    		return index >= 0 ? buffer.substring(index, offset) : "";
        },
        
        /**
         * @callback
         * @since 8.0
         */
        computeContentAssist: function computeContentAssist(editorContext, params) {
            var that = this;
            return Deferred.when(editorContext.getFileMetadata(), function(meta) {
               if(meta.contentType.id === 'text/html') {
                  return editorContext.getText().then(function(text) {
                     var blocks = Util.findStyleBlocks(text, params.offset);
                     if(blocks && blocks.length > 0) {
                         var cu = new CU(blocks, meta);
                         return that._computeProposals(cu.getEditorContext(), text, params);
                     }
                  });
               } else {
                   return editorContext.getText().then(function(text) {
                      return that._computeProposals(editorContext, text, params);
                   });
               }
            });
        },
        
        /**
         * @description Computes the proposals from the given offset, also returns all keyword and template proposals
         * @since 8.0
         * @callback
         */
        _computeProposals: function _computeProposals(editorContext, buffer, context) {
            var _ctxt = this._getCompletionContext(editorContext, context);
            if(_ctxt) {
            	context.kind = _ctxt.value;
            }
            return [].concat(templateAssist.computeProposals(buffer, context.offset, context));
        },
        
        /**
         * @description Computes the kind of completion we are attempting. For example 'color: <assist>' would return 'color'
         * @function
         * @private
         * @param {Object} context The completion contest from #computeProposals
         * @returns {String} The completion context or <code>null</code>
         * @since 8.0
         */
        _getCompletionContext: function _getCompletionContext(editorContext, context) {
            if(this._resultManager) {
            	var that = this;
                return this._resultManager.getResult(editorContext).then(function(results) {
                   if(results && results.ast) {
                       var node = that.findNodeAtOffset(results.ast, context.offset);
                       if(node && node.type === 'Rule') {
                       		return 'rule'; //$NON-NLS-1$
                       }
                   }
                   return null;
                });
            }
            return null;
        },
        
        /**
		 * Returns the ast node at the given offset or the parent node enclosing it
		 * @param {Object} ast The AST to inspect
		 * @param {Number} offset The offset into the source 
		 * @returns {Object} The AST node at the given offset or null 
		 * @since 10.0
		 */
		findNodeAtOffset: function(ast, offset) {
			var found = null;
			 Visitor.visit(ast, {
	            visitNode: function(node) {
					if(node.range[0] <= offset) {
						found = node;
					} else {
					    return Visitor.BREAK;
					}      
	            },
	            endVisitNode: function(node) {
	            	if(found && offset > found.range[1] && offset > node.range[0]) {
	            		found = node;
	            	}
	            }
	        });
	        return found;
		}
	});

	return {
		CssContentAssistProvider: CssContentAssistProvider
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2015 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/* eslint-env amd */

/*
 * Extends the Compilation Unit in Javascript to pad the source file with newlines rather than just spaces.  This is needed for the cssLint validator
 * to run inside html style blocks as it is line/col based.
 */
define('webtools/compilationUnit',[
'orion/objects',
'orion/Deferred',
'javascript/compilationUnit'
], function(Objects, Deferred, CU) {
    
    Objects.mixin(CU.prototype, {
        
        /**
         * @description Returns a promise to build the backing source text for the compilation unit, padding the blocks with spaces and newlines
         * @function
         * @private
         * @returns Returns a promise that resolves to the source
         */
        getSource: function getSource() {
        	var promises = [];
        	if (this._ec && this._ec.getLineAtOffset){  // Tests use a fake editorContext with no line/offset functions
	            for (var i=0; i<this._blocks.length; i++) {
	            	var offset = this._blocks[i].offset;
	            	var length = this._blocks[i].text.length;
	            	promises.push(this._ec.getLineAtOffset(offset));
	            	promises.push(this._ec.getLineAtOffset(offset+length));
	            }
            }
            
        	var self = this;
            return Deferred.all(promises).then(function (lines){
            	var totalLength = 0;
            	var totalLines = 0;
            	var source = "";
            	for(var i = 0; i < self._blocks.length; i++) {
	                var block = self._blocks[i];
	    	        var pad = block.offset - totalLength;
	                var linePad = lines && lines.length > (i*2) ? lines[i*2] : 0;
	                linePad -= totalLines;
	    	        while(pad > 0 && linePad > 0){
	    	        	source += '\n'; //$NON-NLS-1$
	    	        	pad--;
	    	        	linePad --;
	    	        }
	                while(pad > 0) {
	                    source += ' '; //$NON-NLS-1$
	                    pad--;
	                }
	                source += block.text;
	                totalLength = source.length;
	                totalLines = lines && lines.length > (i*2) ? lines[(i*2)+1] : 0;
	            }
	            return source;
            });
        }
    });
    
    return CU;
});
/*******************************************************************************
 * @license
 * Copyright (c) 2014, 2015 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env amd*/
define("webtools/cssValidator", [
	'orion/objects',
	'webtools/compilationUnit',
	'webtools/util'
], function(Objects, CU, Util) {

	// TODO How to keep this list up to date with rules definitions, settings options and content assist
	var config = {
		// Define the default values for the rules
		// 0:off, 1:warning, 2:error
		rules: {
			"adjoining-classes" : 1, //$NON-NLS-0$
			"box-model" : 1, //$NON-NLS-0$
			"box-sizing" : 1, //$NON-NLS-0$
			"bulletproof-font-face" : 1, //$NON-NLS-0$
			"compatible-vendor-prefixes" : 1, //$NON-NLS-0$
			"display-property-grouping" : 1, //$NON-NLS-0$
			"duplicate-background-images" : 1, //$NON-NLS-0$
			"duplicate-properties" : 1, //$NON-NLS-0$
			"empty-rules" : 1, //$NON-NLS-0$
			"fallback-colors" : 1, //$NON-NLS-0$
			"floats" : 1, //$NON-NLS-0$
			"font-faces" : 1, //$NON-NLS-0$
			"font-sizes" : 1, //$NON-NLS-0$
			"gradients" : 1, //$NON-NLS-0$
			"ids" : 1, //$NON-NLS-0$
			"import" : 1, //$NON-NLS-0$
			"important" : 1, //$NON-NLS-0$
			"known-properties" : 1, //$NON-NLS-0$
			"outline-none" : 1, //$NON-NLS-0$
			"overqualified-elements" : 1, //$NON-NLS-0$
			"qualified-headings" : 1, //$NON-NLS-0$
			"regex-selectors" : 1, //$NON-NLS-0$
			"rules-count" : 1, //$NON-NLS-0$
			"selector-max-approaching" : 1, //$NON-NLS-0$
			"selector-max" : 1, //$NON-NLS-0$
			"shorthand" : 1, //$NON-NLS-0$
			"star-property-hack" : 1, //$NON-NLS-0$
			"text-indent" : 1, //$NON-NLS-0$
			"underscore-property-hack" : 1, //$NON-NLS-0$
			"unique-headings" : 1, //$NON-NLS-0$
			"universal-selector" : 1, //$NON-NLS-0$
			"unqualified-attributes" : 1, //$NON-NLS-0$
			"vendor-prefix" : 1, //$NON-NLS-0$
			"zero-units" : 1 //$NON-NLS-0$
		},
		
		/**
		 * @name getRuleSet
		 * @description Returns an editable ruleset to pass into verify() based on values set in the config settings
		 * @function
		 * @returns {Object} A ruleset based on the config settings
		 */
		getRuleSet: function(){
			// TODO Versions of CSSLint >0.10.0 create a copy of the ruleset inside verify (CSSLint Issue 458)
			return JSON.parse( JSON.stringify( this.rules ) );
		},
		
		/**
		 * @description Sets the given rule to the given enabled value
		 * @function
		 * @private
		 * @param {String} ruleId The id of the rule to change
		 * @param {Number} value The value to set the rule to
		 * @param {Object} [key] Optional key to use for complex rule configuration.
		 */
		setOption: function(ruleId, value, key) {
			if (typeof value === "number") { //$NON-NLS-0$
				if(Array.isArray(this.rules[ruleId])) {
					var ruleConfig = this.rules[ruleId];
					if (key) {
						ruleConfig[1] = ruleConfig[1] || {};
						ruleConfig[1][key] = value;
					} else {
						ruleConfig[0] = value;
					}
				}
				else {
					this.rules[ruleId] = value;
				}
			}
		}
	};
	
	/**
	 * @description Creates a new validator
	 * @constructor
	 * @public
	 * @param {Object} cssResultManager The back result manager
	 * @since 6.0
	 */
	function CssValidator(cssResultManager) {
	    this.cssResultManager = cssResultManager;
	}

	Objects.mixin(CssValidator.prototype, /** @lends webtools.CssValidator.prototype*/ {
		
		/**
		 * @description Callback to create problems from orion.edit.validator
		 * @function
		 * @public
		 * @param {orion.edit.EditorContext} editorContext The editor context
		 * @param {Object} context The in-editor context (selection, offset, etc)
		 * @returns {orion.Promise} A promise to compute some problems
		 * @callback
		 */
		computeProblems: function(editorContext, context) {
			var that = this;
			return editorContext.getFileMetadata().then(function(meta) {
			    if(meta && meta.contentType.id === 'text/html') {
			    	// Only run the validator if we have line information because cssLint uses line num and column, not offsets
			    	if (!editorContext.getLineAtOffset){
			    		return null;
			    	}
			        return editorContext.getText().then(function(text) {
    			         var blocks = Util.findStyleBlocks(text, context.offset);
    			         if(blocks && blocks.length > 0) {
    			             var cu = new CU(blocks, meta, editorContext);
    			             return that.cssResultManager.getResult(cu.getEditorContext(), config).then(function(results) {
                			    if(results) {
                			         return that._computeProblems(results);
                			    }
                			    return null;
        			         });
    			         }
			         });
			    } else {
    			    return that.cssResultManager.getResult(editorContext, config).then(function(results) {
        			    if(results) {
        			         return that._computeProblems(results);
        			    }
        			    return null;
        			});
    			}
			});
		},
		
		/**
		 * @description Create the problems 
		 * @function
		 * @private
		 * @param {String} contents The file contents
		 * @returns {Array} The problem array
		 */
		_computeProblems: function(results) {
			    var messages = results.messages,
			    problems = [];
			for (var i=0; i < messages.length; i++) {
				var message = messages[i];
				var range = null;
				if(message.token) {
				    range = [message.token.range[0]+1, message.token.range[1]+1];
				} else if (message.line) {
					range = this._getProblemRange(message);
				}
				if(range) {
    				problems.push({
						id: this._getProblemId(message),
						description: message.message,
						line: message.line,
						start: range[0],
						end: range[1],
						severity: message.type
					});
				}
			}
			return {problems: problems};
		},
		
		/**
		 * @description Computes the problem id to use in the framework from the cssLint message
		 * @param {Object} message The original CSSLint problem message
		 * @returns {String} The problem id to pass into the framework
		 * @since 8.0
		 */
		_getProblemId: function(message) {
		    if(message.rule) {
		        if(message.rule.id) {
		            return message.rule.id;
		        }
		    }
		    return null;
		},
		
		/**
		 * @description Computes the problem range (within the line) for the problem annotation
		 * @param {Object} message The original CSSLint problem message
		 * @returns {Object} Object containing start and end properties to pass into the framework
		 * @since 8.0
		 */
		_getProblemRange: function(message) {
			if (!message.rule || !message.rule.id || message.rule.id === "errors"){ //$NON-NLS-0$
				// Parsing errors often don't have a token to select, so instead select the line
				return [1, message.evidence.length + 1];
			}
		    var token = this._findToken(message.evidence, message.col);
		    var end = message.col + (token ? token.length : 1);
		    return [message.col, end];
		},
		
		_punc: '\n\t\r (){}[]:;,',  //$NON-NLS-0$
		
		/**
		 * @description Returns the token or word found at the given offset
		 * @param {String} contents The text to search for the token
		 * @param {Number} offset The offset in the contents to start the search
		 * @returns {String} Returns the computed token from the given string and offset or <code>null</code>
		 * @since 8.0
		 */
		_findToken: function(contents, offset) {
			if(contents && offset) {
				var ispunc = this._punc.indexOf(contents.charAt(offset)) > -1;
				var pos = ispunc ? offset-1 : offset;
				while(pos >= 0) {
					if(this._punc.indexOf(contents.charAt(pos)) > -1) {
						break;
					}
					pos--;
				}
				var s = pos;
				pos = offset;
				while(pos <= contents.length) {
					if(this._punc.indexOf(contents.charAt(pos)) > -1) {
						break;
					}
					pos++;
				}
				if((s === offset || (ispunc && (s === offset-1))) && pos === offset) {
					return null;
				}
				else if(s === offset) {
					return contents.substring(s, pos);
				}
				else {
					return contents.substring(s+1, pos);
				}
			}
			return null;
		},
		
		/**
		 * @description Callback from orion.cm.managedservice
		 * @function
		 * @public
		 * @param {Object} properties The properties that have been changed
		 */
		updated: function(properties) {
			if (!properties) {
				return;
			}
			// TODO these option -> setting mappings are becoming hard to manage
			// And they must be kept in sync with webToolsPlugin.js
			config.setOption("adjoining-classes", properties.validate_adjoining_classes); //$NON-NLS-0$
			config.setOption("box-model", properties.validate_box_model); //$NON-NLS-0$
			config.setOption("box-sizing", properties.validate_box_sizing); //$NON-NLS-0$
			config.setOption("compatible-vendor-prefixes", properties.validate_compatible_vendor_prefixes); //$NON-NLS-0$
			config.setOption("display-property-grouping", properties.validate_display_property_grouping); //$NON-NLS-0$
			config.setOption("duplicate-background-images", properties.validate_duplicate_background_images); //$NON-NLS-0$
			config.setOption("duplicate-properties", properties.validate_duplicate_properties); //$NON-NLS-0$
			config.setOption("empty-rules", properties.validate_empty_rules); //$NON-NLS-0$
			config.setOption("fallback-colors", properties.validate_fallback_colors); //$NON-NLS-0$
			config.setOption("floats", properties.validate_floats); //$NON-NLS-0$
			config.setOption("font-faces", properties.validate_font_faces); //$NON-NLS-0$
			config.setOption("font-sizes", properties.validate_font_sizes); //$NON-NLS-0$
			config.setOption("gradients", properties.validate_gradients); //$NON-NLS-0$
			config.setOption("ids", properties.validate_ids); //$NON-NLS-0$
			config.setOption("import", properties.validate_imports); //$NON-NLS-0$ // import is restricted key word
			config.setOption("important", properties.validate_important); //$NON-NLS-0$
			config.setOption("known-properties", properties.validate_known_properties); //$NON-NLS-0$
			config.setOption("outline-none", properties.validate_outline_none); //$NON-NLS-0$
			config.setOption("overqualified-elements", properties.validate_overqualified_elements); //$NON-NLS-0$
			config.setOption("qualified-headings", properties.validate_qualified_headings); //$NON-NLS-0$
			config.setOption("regex-selectors", properties.validate_regex_selectors); //$NON-NLS-0$
			config.setOption("rules-count", properties.validate_rules_count); //$NON-NLS-0$
			config.setOption("selector-max-approaching", properties.validate_selector_max_approaching); //$NON-NLS-0$
			config.setOption("selector-max", properties.validate_selector_max); //$NON-NLS-0$
			config.setOption("shorthand", properties.validate_shorthand); //$NON-NLS-0$
			config.setOption("star-property-hack", properties.validate_star_property_hack); //$NON-NLS-0$
			config.setOption("text-indent", properties.validate_text_indent); //$NON-NLS-0$
			config.setOption("underscore-property-hack", properties.validate_underscore_property_hack); //$NON-NLS-0$
			config.setOption("unique-headings", properties.validate_unique_headings); //$NON-NLS-0$
			config.setOption("universal-selector", properties.validate_universal_selector); //$NON-NLS-0$
			config.setOption("unqualified-attributes", properties.validate_unqualified_attributes); //$NON-NLS-0$
			config.setOption("vendor-prefix", properties.validate_vendor_prefix); //$NON-NLS-0$
			config.setOption("zero-units", properties.validate_zero_units); //$NON-NLS-0$
		},
		
		/**
		 * @description Hook for the test suite to enable only the given rule, or set all rules to a certain severity
		 * @function
		 * @private
		 * @param {String} ruleid The id for the rule, if null all rules will be set to the given severity
		 * @param {Number} severity The desired severity or null
		 * @since 8.0
		 */
		_enableOnly: function _enableOnly(ruleid, severity) {
			config.archivedRules = {};
		    var keys = Object.keys(config.rules);
		    for(var i = 0; i < keys.length; i++) {
		    	if (!ruleid){
		    		config.archivedRules[keys[i]] = config.rules[ruleid];
			        config.setOption(keys[i], severity ? severity : 2);
		    	} else {
			        if(keys[i] === ruleid) {
			        	config.archivedRules[ruleid] = config.rules[ruleid];
			            config.setOption(ruleid, severity ? severity : 2);
			        } else {
			        	config.archivedRules[keys[i]] = config.rules[ruleid];
			            config.setOption(keys[i], 0);
			        }
		        }
		    }
		},
		
		/**
		 * @description Hook for the test suite to restore the rule settings after 
		 * calling _enableOnly.  Does not support complex rules (csslint doesn't have any currently)
		 * @function
		 * @private
		 * @since 8.0
		 */
		_restoreRules: function _enableOnly() {
			if (config.archivedRules){
				config.rules = config.archivedRules;
				config.archivedRules = undefined;
			}
		},
	   
	   /**
	    * @description Hook for the parser test suite
	    * @function 
	    * @private 
	    * @since 8.0
	    */
		_defaultRuleSet: function _defaultConfig() {
		    return config.getRuleSet();
		}
	});
	
	return CssValidator;
});
/*******************************************************************************
 * @license
 * Copyright (c) 2014, 2015 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
define("webtools/cssOutliner", [
	'csslint/csslint',
	'orion/objects'
], function(csslint, Objects) {

	/**
	 * @description Creates a new validator
	 * @constructor
	 * @public
	 * @since 6.0
	 */
	function CssOutliner() {
		csslint.addRule(CssOutliner.prototype._outlineRule);
	}

	Objects.mixin(CssOutliner.prototype, /** @lends webtools.CssOutliner.prototype*/ {
		
		/**
		 * @descripton API callback to compute the outline
		 * @callback
		 */
		getOutline: function(contents, title) {
			csslint.verify(contents);
			return this._outlineRule.outline;
		},
		
		/**
		 * @description The CSS linting rule for creating the outline
		 * @private
		 */
		_outlineRule: {
			id: "css-outline", //$NON-NLS-0$
			name: "CSS outline", //$NON-NLS-0$
			desc: "CSS outline helper rule", //$NON-NLS-0$
			browsers: "All", //$NON-NLS-0$
			outline: [],
			/**
			 * @description API callback to start verifying
			 * @callback
			 */
			init: function(parser, reporter) {
				this.outline = [];
				// Pushes selector info into the outline
				var that = this;
				parser.addListener("startrule", function(event) { //$NON-NLS-0$
					var selectors = event.selectors;
					if (selectors && selectors.length) {
						var selectorText = [], line = null, col = null, length = null;
						for (var i=0; i < selectors.length; i++) {
							var sel = selectors[i];
							if (sel.parts && sel.parts.length > 0){
								var part = sel.parts[0]; // We want to check instanceof SelectorPart, but it is not API
								if (line === null) { line = part.line; }
								if (col === null) { col = part.col-1; }
								if (length === null){
									length = part.text.length;
								}
							}
							// If no valid parts found, just use the entire selector text
							if (line === null) { line = sel.line; }
							if (col === null) { col = sel.col-1; }
							if (length === null) {
								length = sel.text.length;
							}
							selectorText.push(sel.text);
						}
						that.outline.push({
							label: selectorText.join(", "), //$NON-NLS-0$
							line: line,
							offset: col,
							length: length
						});
					}
				});
			}
		}
	});
	
	return {
		CssOutliner : CssOutliner
	};
});
 /*******************************************************************************
 * @license
 * Copyright (c) 2014, 2015 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env amd*/
define('webtools/cssHover',[
'orion/objects',
'orion/URITemplate',
'webtools/util',
'javascript/compilationUnit',
'i18n!webtools/nls/messages',
'csslint/csslint' //for colour object
], function(Objects, URITemplate, Util, CU, messages, CSSLint) {
	
	/**
	 * @name webtools.CSSHover
	 * @description creates a new instance of the hover support
	 * @constructor
	 * @public
	 * @param {Object} resolver The backing file resolver
	 * @param {Object} cssResultManager The back result manager
	 * @since 8.0
	 */
	function CSSHover(resolver, cssResultManager) {
	    this.resolver = resolver;
	    this.cssResultManager = cssResultManager;
	}
	
	Objects.mixin(CSSHover.prototype, /** @lends webtools.CSSHover.prototype*/ {
		
		/**
		 * @name computeHover
		 * @description Callback from the editor to compute the hover
		 * @function
		 * @public 
		 * @memberof webtools.CSSHover.prototype
		 * @param {Object} editorContext The current editor context
		 * @param {Object} ctxt The current selection context
		 */
		computeHoverInfo: function computeHover(editorContext, ctxt) {
			if(ctxt.proposal && ctxt.proposal.kind === 'css') {
				return ctxt.proposal.hover;
			}
			var that = this;
			return editorContext.getFileMetadata().then(function(meta) {
			   if(meta.contentType.id === 'text/html') {
			       return editorContext.getText().then(function(text) {
			           var blocks = Util.findStyleBlocks(text);
			           if(blocks && blocks.length > 0) {
			               var cu = new CU(blocks, meta);
			               if(cu.validOffset(ctxt.offset)) {
    			               return that.cssResultManager.getResult(cu.getEditorContext(), that._emptyRuleSet()).then(function(results) {
                    			   return that._doHover(results, ctxt, meta);
                               });
                           }
			           }
			       });
			   } else {
			       return that.cssResultManager.getResult(editorContext, that._emptyRuleSet()).then(function(results) {
        			   return that._doHover(results, ctxt, meta);
                   });
			   }
			});
		},
		
		_doHover: function _doHover(results, ctxt, metadata) {
		    if(results) {
			    var token = Util.findToken(ctxt.offset, results.tokens);
				if (token){
				    //TODO, investigate creating an AST in the CSS parser, walking tokens can be expensive
				    if(this.hasPreviousToken(token, results.tokens, 'IMPORT_SYM')) {//$NON-NLS-0$
				        return this._getFileHover(token, metadata);
				    }
				    if(this.hasPreviousToken(token, results.tokens, 'IDENT', ['background', 'background-image', '-webkit-border-image', '-o-border-image', 'border-image', 'border-image-source', 'icon'])) {
				        return this._getImageHover(token, metadata);
				    }
				    var tok = this._isRgbLike(token, results.tokens);
				    if(tok) {
				        var color = this._collectColorId(tok, results.tokens);
		                if(color) {
		                    return this._getColorHover(color);    
		                }
				    }
					if (CSSLint.Colors[token.value]){
						return this._getColorHover(token.value);
					}
					if (/\#[0-9A-Fa-f]{1,6}/.test(token.value)){
						return this._getColorHover(token.value);	
					}
					tok = this._isFontLike(token, results.tokens);
					if(tok) {
					    var font = this._collectFontId(tok, results.tokens);
					    if(font) {
					        return this._getFontHover(tok.value, font);
					    }
					}
				}
			}
			return null;
		},
		
		_emptyRuleSet: function() {
		    var config = Object.create(null);
		    config.getRuleSet = function() {return null;};
		    return config;
		},
		
		fontLikeNames: ['font-family', 'font', 'font-size', 'font-size-adjust', 'font-stretch', 'font-style', 'font-variant', 'font-weight', 
		                  'text-decoration', 'text-shadow', 'text-transform'],
		
		_isFontLike: function _isFontLike(token, tokens) {
		    if(token && tokens) {
		        for(var i = token.index; i > -1; i--) {
		            var tok = tokens[i];
		            if(tok.type === 'IDENT' || tok.type === 'COMMA' || tok.type === 'STRING' || tok.type === 'LENGTH' || tok.type === 'NUMBER' || tok.type === 'HASH') {//$NON-NLS-0$  //$NON-NLS-1$  //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$ //$NON-NLS-5$
		                continue;
		            } else if(tok.type === 'COLON') {   //$NON-NLS-0$
		                //the next one would have to be IDENT and 'font-family'
		                tok = tokens[i-1];
		                if(tok.type === 'IDENT' && this.fontLikeNames.indexOf(tok.value.toLowerCase()) > -1) {   //$NON-NLS-0$
		                    tok.index = i-1;
		                    return tok;
		                } else {
		                    return null;
		                }
		            } else {
		                break;
		            }
		        }
		    }
		    return null;
		},
		
		_collectFontId: function _collectFontId(token, tokens) {
		    if(token && tokens) {
		        var id = '';
		        var next = null;
		        var idx = token.index;
		        //skip the colon
		        if(tokens[idx+1].type !== 'COLON') {   //$NON-NLS-0$
		            return null;
		        }
		        ++idx;
		        for(var i = idx+1; i < tokens.length; i++) {
		            next = tokens[i];
		            if(next.type === 'IDENT' || next.type === 'COMMA' || next.type === 'STRING' || next.type === 'NUMBER' || next.type === 'LENGTH' || next.type === 'HASH') {   //$NON-NLS-0$ //$NON-NLS-1$  //$NON-NLS-2$ //$NON-NLS-3$  //$NON-NLS-4$ //$NON-NLS-5$
		                id += next.value;
		                if(i < tokens.length-1) {
		                    id += ' ';
		                }
		                continue;
		            }
		            if(next.type === 'RBRACE' || next.type === 'SEMICOLON' || next.type === 'RPAREN') {   //$NON-NLS-0$ //$NON-NLS-1$  //$NON-NLS-2$
		                return id;
		            } else {
		                break;
		            }
		        }
		    }
		    return null;
		},
		
		_getFontHover: function _getFontHover(prop, font){
			var html = '<html><body style=\"background-color:white\"><div style="'+prop+':'+font+';margin:0px">'+messages['fontHoverExampleText']+'</div></body></html>';  //$NON-NLS-0$  //$NON-NLS-1$  //$NON-NLS-2$ //$NON-NLS-3$
			return {type: "html", content: html, height: '42px', width: '235px'};  //$NON-NLS-0$  //$NON-NLS-1$  //$NON-NLS-2$ //$NON-NLS-3$
		},
		
		_isColorFnName: function _isColorFnName(name) {
		    var val = name.toLowerCase();
		    return val === 'rgba(' || val === 'rgb(' || val === 'hsl(' || val === 'hsla(';   //$NON-NLS-0$ //$NON-NLS-1$  //$NON-NLS-2$ //$NON-NLS-3$
		},
 		
		_isRgbLike: function _isRgbLike(token, tokens) {
		    if(token.type === 'FUNCTION') {  //$NON-NLS-0$
		        if(this._isColorFnName(token.value.toLowerCase())) {
		            return token;
		        }
		    } 
		    var tok = this._isRgbLikeBody(token, tokens);
		    if(tok) {
		        return tok;
		    }
		    return null;
		},
		
		_isRgbLikeBody: function _isRgbLikeBody(token, tokens) {
		    if(token && tokens) {
		        for(var i = token.index; i > -1; i--) {
		            var tok = tokens[i];
		            if(tok.type === 'NUMBER' || tok.type === 'COMMA' || tok.type === 'PERCENTAGE') {  //$NON-NLS-0$  //$NON-NLS-1$  //$NON-NLS-2$
		                continue;
		            } else if(tok.type === 'FUNCTION') {  //$NON-NLS-0$
		                if(this._isColorFnName(tok.value)) {
		                    tok.index = i;
		                    return tok;
		                } else {
		                    return null;
		                }
		            } else {
		                break;
		            }
		        }
		    }
		    return null;
		},
		
		_collectColorId: function _collectColorId(token, tokens) {
		    if(token && tokens) {
		        var id = token.value;
		        var next = null;
		        var idx = token.index;
		        for(var i = idx+1; i < tokens.length; i++) {
		            next = tokens[i];
		            if(next.type === 'COMMA' || next.type === 'NUMBER' || next.type === 'PERCENTAGE') {  //$NON-NLS-0$  //$NON-NLS-1$  //$NON-NLS-2$
		                id += next.value;
		                continue;
		            }
		            if(next.type === 'RPAREN') {  //$NON-NLS-0$
		                id += next.value;
		                return id;
		            } else {
		                break;
		            }
		        }
		    }
		    return null;
		},
		
		hasPreviousToken: function hasPreviousToken(token, tokens, name, id) {
		    if(token && tokens) {
		        switch(token.type) {
		            case 'URI':   //$NON-NLS-0$
		            case 'STRING': {  //$NON-NLS-0$
		                if(token.index > 0) {
		                    var prev = null;
		                    for(var i = token.index-1; i >= 0; i--) {
		                        prev = tokens[i];
		                        if(prev.type === 'COLON' || prev.type === 'STRING' || prev.type === 'URI' || prev.type === 'COMMA') {  //$NON-NLS-0$  //$NON-NLS-1$  //$NON-NLS-2$  //$NON-NLS-3$
		                            continue;
		                        } else {
		                            break;
		                        }
		                    }
		                    if(Array.isArray(id) && prev && id.indexOf(prev.value) > -1) {
		                        return true;
		                    } else if(id && prev && prev.type === name) {
		                       return id === prev.value;
    		                } else {
    		                  return prev && prev.type === name;
    		                }
		                }
		            }
		        }
            }
            return false;
		},
		
		_getFileHover: function _getFileHover(token, metadata) {
		    var path = this._getPathFromToken(token);
		    if(path) {
    	        if(/^http/i.test(path)) {
    	            return this._formatFilesHover(path);
    	        } else {
    	            var that = this;
        	        return that.resolver.getWorkspaceFile(path, {ext:'css', type:'CSS', icon:'../webtools/images/css.png'}).then(function(files) {  //$NON-NLS-0$  //$NON-NLS-1$  //$NON-NLS-2$
        		        if(files && files.length > 0) {
        		            var resolved = that.resolver.resolveRelativeFiles(path, files, metadata);
        		            if(resolved.length > 0) {
        		              return that._formatFilesHover(path, resolved);
        		            }
        		        }
        	        });
    	        }
	        }
		    return null;
		},
		
		_getPathFromToken: function _getPathFromToken(token) {
		    var path = token.value;
		    switch(token.type) {
		        case 'STRING': {  //$NON-NLS-0$
		            path = token.value.slice(1, token.value.length-1); //peel off the quotes
		            break;
		        }
		        case 'URI': {  //$NON-NLS-0$
		            var val = /^\s*(?:url)\s*\(\s*(.*)\s*\)/i.exec(token.value);
    		        if(val) {
    		            path = val[1];
    		            var c = path.charAt(0);
    		            if(c === '\'' || c === '"') {  //$NON-NLS-0$  //$NON-NLS-1$
    		                path = path.slice(1);
    		            }
    		            c = path.charAt(path.length-1);
    		            if(c === '\'' || c === '"') {  //$NON-NLS-0$  //$NON-NLS-1$
    		                path = path.slice(0, path.length-1);
    		            }
    		        } else {
    		            return null;
    		        }
		        }
		    }
		    return path;
		},
		
		/**
    	 * @description Formats the list of files as links for the hover
    	 * @function
    	 * @private
    	 * @param {String} path The path we are navigating to
    	 * @param {Array.<javascript.ScriptResolver.File>} files The array of files to linkify
    	 * @returns {String} The mardown to show in the hover
    	 */
    	_formatFilesHover: function _formatFilesHover(path, files) {
    	    if(path) {
    	        var title = null; 
    	        if(files.length > 1) {
    	            title = '###Open file for \''+path+'\'###';  //$NON-NLS-0$  //$NON-NLS-1$
    	        }
    	        var hover = '';
    	        if(Array.isArray(files)) {  
        	        for(var i = 0; i < files.length; i++) {
        	            var file = files[i];
        	            if(file.name && file.path && file.contentType) {
        	                hover += '[';
        	                if(file.contentType.icon) {
        	                    hover += '!['+file.contentType.name+']('+file.contentType.icon+')';  //$NON-NLS-0$  //$NON-NLS-1$  //$NON-NLS-2$
        	                }
        	                var href = new URITemplate("#{,resource,params*}").expand(  //$NON-NLS-0$
        		                      {
        		                      resource: file.location, 
        		                      params: {}
        		                      }); //$NON-NLS-0$
        	                hover += file.name + ']('+href+') - '+file.path+'\n\n';  //$NON-NLS-0$  //$NON-NLS-1$  //$NON-NLS-2$
        	            }
        	            
        	        }
    	        } else {
    	            var name = path.slice(path.lastIndexOf('/')+1);  //$NON-NLS-0$
    	            title = '###Open file for \''+name+'\'###';  //$NON-NLS-0$  //$NON-NLS-1$
	                hover += '[!['+name+'](../webtools/images/css.png)';  //$NON-NLS-0$  //$NON-NLS-1$
	                hover += name + ']('+path+') - '+path+'\n\n';  //$NON-NLS-0$  //$NON-NLS-1$  //$NON-NLS-2$
    	        }
    	        if(hover !== '') {
    	           return {title: title, content: hover, type:'markdown', allowFullWidth: true};  //$NON-NLS-0$
    	        }
    	    }
    	    return null;
    	},
		
		_getImageHover: function _getImageHover(token, metadata) {
		      var path = this._getPathFromToken(token);
		      var that = this;
		      if(path) {
		          if(/^http/i.test(path) || /^data:image.*;base64/i.test(path)) {
    		          var html = '<html><body style="margin:1px;"><img src="'+path+'" style="width:100%;height:100%;"/></body></html>'; //$NON-NLS-0$  //$NON-NLS-1$
    			      return {type: "html", content: html, width: "100px", height: "100px"};  //$NON-NLS-0$  //$NON-NLS-1$  //$NON-NLS-2$
		          } else {
		              var idx = path.lastIndexOf('.');  //$NON-NLS-0$
		              if(idx > -1) {
		                  var ext = path.slice(idx+1);
    		              return that.resolver.getWorkspaceFile(path, {ext:ext, type:'Image', icon:'../webtools/images/file.png'}).then(function(files) {  //$NON-NLS-0$
                		        if(files) {
                		            //TODO we have to resolve each time as same-named files could be referenced from different locations
                		            //and the resolver caches all hits for the name
                		            var resolved = that.resolver.resolveRelativeFiles(path, files, metadata);
                		            if(resolved.length > 0) {
                		                 var html = '<html><body style="margin:1px;"><img src="'+resolved[0].location+'" style="width:100%;height:100%;"/></body></html>'; //$NON-NLS-0$  //$NON-NLS-1$
    			                         return {type: "html", content: html, width: "100px", height: "100px"};  //$NON-NLS-0$  //$NON-NLS-1$  //$NON-NLS-2$
                		            }
                		        }
                	        });
        	          }
		          }
		      }
		},
		
		_getColorHover: function _getColorHover(colorID){
			var html = '<html><body style=\"background-color: ' + colorID + ';\"></html>'; //$NON-NLS-0$  //$NON-NLS-1$
			return {type: "html", content: html, width: "50px", height: "25px"};  //$NON-NLS-0$  //$NON-NLS-1$  //$NON-NLS-2$
		}
		
	});

	CSSHover.prototype.contructor = CSSHover;
	
	return {
		CSSHover: CSSHover
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014, 2016 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html).
 *
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/
/*eslint-env browser, amd*/
define('orion/metrics',[
], function() {

	var _services = [];
	var timingVars = Object.create(null);

	/**
	 * @name Metrics
	 * @description Creates a new instance of the metrics service
	 * @param {Object} serviceRegistry The backing service registry to register the new service with
	 * @param {Object} args An object of additional arguments
	 * @returns {Metrics} A new Metrics instance
	 * @since 12.0
	 */
	function Metrics(serviceRegistry, args) {
		var refs = serviceRegistry.getServiceReferences("orion.metrics"); //$NON-NLS-0$
		var services = [];
		refs.forEach(function(current) {
			services.push(serviceRegistry.getService(current));
		});
		/* the following definitions are from https://developers.google.com/analytics/devguides/collection/analyticsjs/pages */
		var href = window.location.protocol + '//' + window.location.hostname + window.location.pathname + window.location.search; //$NON-NLS-0$
		var page = window.location.pathname + window.location.search;
		var title = document.title;

		_services = services;
		_services.forEach(function(current) {
			current.pageLoad(href, page, title, args);
		});
		serviceRegistry.registerService("orion.core.metrics.client", this); //$NON-NLS-1$
	}
	
	/** @callback */
	function _logTiming(timingCategory, timingVar, timingValue, timingLabel) {
		_services.forEach(function(current) {
			current.logTiming(timingCategory, timingVar, timingValue, timingLabel);
		});
	}
	/** @callback */
	function _logEvent(category, action, label, value, details) {
		_services.forEach(function(current) {
			current.logEvent(category, action, label || "", value, details);
		});
	}
	/** @callback */
	function _logPageLoadTiming(timingVar, timingLabel) {
		/* 
		 * The level of window.performance implementation varies across the browsers,
		 * so check for the existence of all utilized functions up-front.
		 */
		if (window.performance) {
			 /* ensure that no more timings of this type are logged for this page */
			if (window.performance.getEntriesByName && window.performance.mark) {
				if (window.performance.getEntriesByName(timingVar).length) {
					return;
				}
				window.performance.mark(timingVar);
			} else {
				if (timingVars[timingVar]) {
					return;
				}
				timingVars[timingVar] = Date.now();				
			}
			_logTiming("page", timingVar, window.performance.now(), timingLabel); //$NON-NLS-0$
		}
	}
	
	Metrics.prototype = {
		/**
		 * @description Log a timing
		 * @function
		 * @param {String} timingCategory The name of the category to log to
		 * @param {String} timingVar The name of the variable to log to
		 * @param {Number} timingValue The timing to log
		 * @param {String} timingLabel A label for the new timing
		 */
		logTiming: function(timingCategory, timingVar, timingValue, timingLabel) {
			_logTiming(timingCategory, timingVar, timingValue, timingLabel);
		},
		/**
		 * @description Log an event
		 * @function
		 * @param {String} category The name of the category to log to
		 * @param {String} action The name of the action logged
		 * @param {String} label A label for the event
		 * @param {String} value The event value to log
		 * @param {String} details Additional details about the event being logged
		 */
		logEvent: function(category, action, label, value, details) {
			_logEvent(category, action, label, value, details);
		},
		/**
		 * @description Log how long it took to load a page
		 * @function
		 * @param {Number} timingVar The timing to log
		 * @param {String} timingLabel A label for the new timing
		 */
		logPageLoadTiming: function(timingVar, timingLabel) {
			_logPageLoadTiming(timingVar, timingLabel);
		}
	};
	
	return {
		Metrics: Metrics,
		logTiming: _logTiming,
		logEvent: _logEvent,
		logPageLoadTiming: _logPageLoadTiming
	};
});

 /*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env amd*/
define('webtools/cssQuickFixes',[
'orion/objects',
'orion/metrics'
], function(Objects, Metrics) {
	
	/**
	 * @description Creates a new CSS quick fix computer
	 * @returns {webtools.CssQuickFixes} The new quick fix computer instance
	 * @since 8.0
	 */
	function CssQuickFixes() {
	}
	
	Objects.mixin(CssQuickFixes.prototype, /** @lends webtools.CssQuickFixes.prototype*/ {
		/**
		 * @description Editor command callback
		 * @function
		 * @param {orion.edit.EditorContext} editorContext The editor context
		 * @param {Object} context The context params
		 */
		execute: function(editorContext, context) {
			var id = context.annotation.id;
			Metrics.logEvent('language tools', 'quickfix', id, 'text/css');
		    var fixes = this[id];
	        if(fixes) {
	            return fixes(editorContext, context.annotation);
	        }
		    return null;
		},
		"empty-rules": function(editorContext, annotation) { //$NON-NLS-0$
			return editorContext.getText().then(function(text) {
				// Remove leading space
				var start = annotation.start;
				while (start >= 0 && (text[start-1] === ' ' || text[start-1] === '\t')){ //$NON-NLS-0$ //$NON-NLS-1$
					start--;
				}
				var contents = text.substring(annotation.start);
				contents = contents.match(/.*{\s*}\s*/,''); //$NON-NLS-0$
				if (contents){
					return editorContext.setText("", start, start+contents[0].length);
				}
				return null;
            });
		},
		"important": function(editorContext, annotation){ //$NON-NLS-0$
			return editorContext.getText().then(function(text) {
				// The annotation will select the property name. Get the complete property.
				var contents = text.substring(annotation.start);
				var startRange = contents.search(/\s*\!important/i);
				var endRange = contents.search(/[;}]/);
				if (startRange !== 1 && endRange !== -1 && startRange < endRange){
					contents = contents.substring(startRange, endRange);
					contents = contents.replace(/\s*\!important/gi, "");
					return editorContext.setText(contents, annotation.start+startRange, annotation.start+endRange);
				}
				return null;
            });
		},
		"zero-units": function(editorContext, annotation) { //$NON-NLS-0$
			return editorContext.getText().then(function(text) {
				var contents = text.substring(annotation.start, annotation.end);
				contents = contents.replace(/0px/gi,'0'); //$NON-NLS-0$
				return editorContext.setText(contents, annotation.start, annotation.end);
            });
		}
	});
	
	CssQuickFixes.prototype.contructor = CssQuickFixes;
	
	return {
		CssQuickFixes: CssQuickFixes
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2015, 2016 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html).
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env amd*/
/*globals Tautologistics */
define('webtools/cssResultManager',[
	'orion/Deferred',
	'orion/objects',
	'javascript/lru',
	'csslint/csslint'
], function(Deferred, Objects, LRU, CSSLint) {

	var registry;
	
	/**
	 * Provides a shared AST.
	 * @class Provides a shared parsed AST.
	 * @param {Object} serviceRegistry The platform service registry 
	 * @since 8.0
	 */
	function CssResultManager(serviceRegistry) {
		this.cache = new LRU(10);
		registry = serviceRegistry;
	}

	/**
	 * @description Delegate to log timings to the metrics service
	 * @param {Number} end The end time
	 * @since 12.0
	 */
	function logTiming(end) {
		if(registry) {
			var metrics = registry.getService("orion.core.metrics.client"); //$NON-NLS-1$
			if(metrics) {
				metrics.logTiming('language tools', 'parse', end, 'text/css'); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
			}
		}
	}

	Objects.mixin(CssResultManager.prototype, /** @lends webtools.CssResultManager.prototype */ {
		/**
		 * @param {orion.editor.EditorContext} editorContext
		 * @returns {orion.Promise} A promise resolving to the CSS parse / checking result or null if called
		 * with an incomplete config
		 */
		getResult: function(editorContext, config) {
		    if(typeof config === 'undefined') {
		        config = Object.create(null);
		    }
		    if(typeof config.getRuleSet === 'undefined') {
		    	/** @callback */
		        config.getRuleSet = function() {return null;};
			}
			var _self = this;
			return editorContext.getFileMetadata().then(function(metadata) {
				metadata = metadata || {};
				var loc = _self._getKey(metadata);
				var result = _self.cache.get(loc);
				if (result) {
					return new Deferred().resolve(result);
				}
				return editorContext.getText().then(function(text) {
				    var start = Date.now();
					result = CSSLint.verify(text, config.getRuleSet());
					var end = Date.now() - start;
					logTiming(end);
					_self.cache.put(loc, result);
					if(metadata.location) {
					    //only set this if the original metadata has a real location
					    result.fileLocation = metadata.location;
					}
					return result;
				});
			});
		},
		/**
		 * Returns the key to use when caching
		 * @param {Object} metadata The file infos
		 */
		_getKey: function _getKey(metadata) {
		      if(!metadata.location) {
		          return 'unknown'; //$NON-NLS-1$
		      }
		      return metadata.location;
		},

		/**
		 * Callback from the orion.edit.model service
		 * @param {Object} event An <tt>orion.edit.model</tt> event.
		 * @see https://wiki.eclipse.org/Orion/Documentation/Developer_Guide/Plugging_into_the_editor#orion.edit.model
		 */
		onModelChanging: function(event) {
		    if(this.inputChanged) {
		        //TODO haxxor, eat the first model changing event which immediately follows
		        //input changed
		        this.inputChanged = null;
		    } else {
		        this.cache.remove(this._getKey(event.file));
		    }
		},
		/**
		 * Callback from the orion.edit.model service
		 * @param {Object} event An <tt>orion.edit.model</tt> event.
		 * @see https://wiki.eclipse.org/Orion/Documentation/Developer_Guide/Plugging_into_the_editor#orion.edit.model
		 */
		onInputChanged: function(event) {
		    this.inputChanged = event;
		    //TODO will add to mult-env
		}
	});
	return CssResultManager;
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014, 2016 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html).
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
define('webtools/plugins/webToolsPlugin',['orion/plugin',
'orion/serviceregistry',
'javascript/scriptResolver',
'webtools/htmlAstManager',
'webtools/htmlHover',
'webtools/htmlContentAssist',
'webtools/htmlOccurrences',
'webtools/htmlOutliner',
'orion/editor/stylers/text_html/syntax',
'webtools/cssContentAssist',
'webtools/cssValidator',
'webtools/cssOutliner',
'webtools/cssHover',
'webtools/cssQuickFixes',
'webtools/cssResultManager',
'orion/editor/stylers/text_css/syntax',
'i18n!webtools/nls/messages'
], function(PluginProvider, mServiceRegistry, ScriptResolver, HtmlAstManager, htmlHover, htmlContentAssist, htmlOccurrences, htmlOutliner,
            mHTML, cssContentAssist, mCssValidator, mCssOutliner, cssHover, cssQuickFixes, cssResultManager, mCSS, messages) {

	/**
	 * Plug-in headers
	 */
	var headers = {
		name: messages["pluginName"],
		version: "1.0", //$NON-NLS-1$
		description: messages["pluginDescription"]
	};
	var serviceRegistry = new mServiceRegistry.ServiceRegistry();
	var provider = new PluginProvider(headers, serviceRegistry);

    	/**
    	 * Register the content types: HTML, CSS
    	 */
    	provider.registerServiceProvider("orion.core.contenttype", {}, { //$NON-NLS-1$
    		contentTypes: [
    			{	id: "text/html", //$NON-NLS-1$
    				"extends": "text/plain", //$NON-NLS-1$ //$NON-NLS-1$
    				name: "HTML", //$NON-NLS-1$
    				extension: ["html", "htm"], //$NON-NLS-1$ //$NON-NLS-2$
    				imageClass: "file-sprite-html modelDecorationSprite" //$NON-NLS-1$
    			},
    			{	id: "text/css", //$NON-NLS-1$
    				"extends": "text/plain", //$NON-NLS-1$ //$NON-NLS-1$
    				name: "CSS", //$NON-NLS-1$
    				extension: ["css"], //$NON-NLS-1$
    				imageClass: "file-sprite-css modelDecorationSprite" //$NON-NLS-1$
    			}
    		]
    	});
        var cssResultMgr = new cssResultManager(serviceRegistry);

    	/**
    	 * Register result manager as model changed listener
    	 */
    	provider.registerService("orion.edit.model", {  //$NON-NLS-1$
    		onModelChanging: cssResultMgr.onModelChanging.bind(cssResultMgr),
    		onInputChanged: cssResultMgr.onInputChanged.bind(cssResultMgr)
    	},
    	{
    		contentType: ["text/css", "text/html"],  //$NON-NLS-1$ //$NON-NLS-2$
    		types: ["ModelChanging", 'Destroy', 'onSaving', 'onInputChanged']  //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
    	});

        provider.registerService("orion.edit.contentassist", //$NON-NLS-1$
    		new cssContentAssist.CssContentAssistProvider(cssResultMgr),
    		{	name: messages["cssContentAssist"],
    			contentType: ["text/css", "text/html"] //$NON-NLS-1$ //$NON-NLS-2$
    		});

    	/**
    	 * Register validators
    	 */
    	provider.registerService(["orion.edit.validator", "orion.cm.managedservice"], new mCssValidator(cssResultMgr), //$NON-NLS-1$ //$NON-NLS-2$
    		{
    			contentType: ["text/css", "text/html"], //$NON-NLS-1$ //$NON-NLS-2$
    			pid: 'csslint.config'  //$NON-NLS-1$
    		});

    	var htmlAstManager = new HtmlAstManager.HtmlAstManager(serviceRegistry);

    	/**
    	 * Register content assist providers
    	 */
    	provider.registerService("orion.edit.contentassist", //$NON-NLS-1$
    		new htmlContentAssist.HTMLContentAssistProvider(htmlAstManager),
    		{	name: messages['htmlContentAssist'],
    			contentType: ["text/html"], //$NON-NLS-1$
    			charTriggers: "<",
    			excludedStyles: "(comment.*|string.*)" //$NON-NLS-1$
    		}
    	);
    	
	  	/**
    	 * Register occurrence providers
    	 */
    	provider.registerService("orion.edit.occurrences", //$NON-NLS-1$
    		new htmlOccurrences.HTMLOccurrences(htmlAstManager),
    		{
    			contentType: ["text/html"] //$NON-NLS-1$
    		}
    	);

    	/**
    	 * Register AST manager as Model Change listener
    	 */
    	provider.registerService("orion.edit.model", {  //$NON-NLS-1$
    		onModelChanging: htmlAstManager.onModelChanging.bind(htmlAstManager),
    		onInputChanged: htmlAstManager.onInputChanged.bind(htmlAstManager)
    	},
    	{
    		contentType: ["text/html"],  //$NON-NLS-1$
    		types: ["ModelChanging", 'Destroy', 'onSaving', 'onInputChanged']  //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
    	});

    	/**
    	* Register outliners
    	*/
    	provider.registerService("orion.edit.outliner", new htmlOutliner.HtmlOutliner(htmlAstManager), //$NON-NLS-1$
    		{
    			id: "orion.webtools.html.outliner", //$NON-NLS-1$
    			name: messages["htmlOutline"],
    			contentType: ["text/html"] //$NON-NLS-1$
    		});

    	provider.registerService("orion.edit.outliner", new mCssOutliner.CssOutliner(),  //$NON-NLS-1$
    		{
    			id: "orion.outline.css.outliner", //$NON-NLS-1$
    			name: messages["cssOutline"],
    			contentType: ["text/css"] //$NON-NLS-1$
    		});

    	/**
    	 * Register syntax styling
    	 */
    	var newGrammars = {};
    	mCSS.grammars.forEach(function(current){
    		newGrammars[current.id] = current;
    	});
    	mHTML.grammars.forEach(function(current){
    		newGrammars[current.id] = current;
    	});
    	for (var current in newGrammars) {
    	    if (newGrammars.hasOwnProperty(current)) {
       			provider.registerService("orion.edit.highlighter", {}, newGrammars[current]); //$NON-NLS-1$
      		}
        }

        var resolver = new ScriptResolver.ScriptResolver(serviceRegistry);

        /**
    	 * Register the hover support
    	 */
    	provider.registerService("orion.edit.hover", new cssHover.CSSHover(resolver, cssResultMgr),  //$NON-NLS-1$
    		{
    		    name: messages['cssHover'],
    			contentType: ["text/css", "text/html"]	//$NON-NLS-1$ //$NON-NLS-2$
    	});

    	/**
    	 * Register the hover support
    	 */
    	provider.registerService("orion.edit.hover", new htmlHover.HTMLHover(htmlAstManager, resolver),  //$NON-NLS-1$
    		{
    		    name: messages['htmlHover'],
    			contentType: ["text/html"]	//$NON-NLS-1$
    	});

    	/**
    	 * Register quick fixes as editor commands
    	 */
    	var cssQuickFixComputer = new cssQuickFixes.CssQuickFixes();

    	provider.registerServiceProvider("orion.edit.command",  //$NON-NLS-1$
    		cssQuickFixComputer,
    		{
	    		name: messages["quickfix-empty-rules"],
	    		scopeId: "orion.edit.quickfix", //$NON-NLS-1$
	    		id : "quickfix-empty-rules",  //$NON-NLS-1$
	    		contentType: ['text/css','text/html'],  //$NON-NLS-1$ //$NON-NLS-2$
	    		validationProperties: [
    				{source: "annotation:id", match: "empty-rules"} //$NON-NLS-1$ //$NON-NLS-2$
    		    ]
    		}
    	);
    	provider.registerServiceProvider("orion.edit.command",  //$NON-NLS-1$
    		cssQuickFixComputer,
    		{
	    		name: messages["quickfix-important"],
	    		scopeId: "orion.edit.quickfix", //$NON-NLS-1$
	    		id : "quickfix-important",  //$NON-NLS-1$
	    		contentType: ['text/css','text/html'],  //$NON-NLS-1$ //$NON-NLS-2$
	    		validationProperties: [
    		    	{source: "annotation:id", match: "important"} //$NON-NLS-1$ //$NON-NLS-2$
    		    ]
    		}
    	);
    	provider.registerServiceProvider("orion.edit.command",  //$NON-NLS-1$
    		cssQuickFixComputer,
    		{
	    		name: messages["quickfix-zero-units"],
	    		scopeId: "orion.edit.quickfix", //$NON-NLS-1$
	    		id : "quickfix-zero-units",  //$NON-NLS-1$
	    		contentType: ['text/css','text/html'],  //$NON-NLS-1$ //$NON-NLS-2$
	    		validationProperties: [
 		        	{source: "annotation:id", match: "zero-units"} //$NON-NLS-1$ //$NON-NLS-2$
    		    ]
    		}
    	);

        /**
    	 * CSSLint settings
    	 */
    	var ignore = 0, warning = 1, error = 2, severities = [
    		{label: messages.ignore,  value: ignore},
    		{label: messages.warning, value: warning},
    		{label: messages.error,   value: error}
    	];
    	provider.registerService("orion.core.setting",  //$NON-NLS-1$
    		{},
    		{	settings: [
    				{	pid: "csslint.config",  //$NON-NLS-1$
    					name: messages["csslintValidator"],
    					tags: "validation webtools css csslint".split(" "),  //$NON-NLS-1$  //$NON-NLS-1$
    					category: "css",  //$NON-NLS-1$
    					categoryLabel: messages["css"],
    					properties: [
    						{
    							id: "validate_adjoining_classes", //$NON-NLS-1$
    							name: messages["adjoining-classes"],
    							type: "number", //$NON-NLS-1$
    							defaultValue: warning,
    							options: severities
    						},
    						{
    							id: "validate_box_model", //$NON-NLS-1$
    							name: messages["box-model"],
    							type: "number", //$NON-NLS-1$
    							defaultValue: warning,
    							options: severities
    						},
    						{
    							id: "validate_box_sizing", //$NON-NLS-1$
    							name: messages["box-sizing"],
    							type: "number", //$NON-NLS-1$
    							defaultValue: warning,
    							options: severities
    						},
    						{
    							id: "validate_bulletproof_font_face", //$NON-NLS-1$
    							name: messages["bulletproof-font-face"],
    							type: "number", //$NON-NLS-1$
    							defaultValue: warning,
    							options: severities
    						},
    						{
    							id: "validate_compatible_vendor_prefixes", //$NON-NLS-1$
    							name: messages["compatible-vendor-prefixes"],
    							type: "number", //$NON-NLS-1$
    							defaultValue: warning,
    							options: severities
    						},
    						{
    							id: "validate_display_property_grouping", //$NON-NLS-1$
    							name: messages["display-property-grouping"],
    							type: "number", //$NON-NLS-1$
    							defaultValue: warning,
    							options: severities
    						},{
    							id: "validate_duplicate_background_images", //$NON-NLS-1$
    							name: messages["duplicate-background-images"],
    							type: "number", //$NON-NLS-1$
    							defaultValue: warning,
    							options: severities
    						},
    						{
    							id: "validate_duplicate_properties", //$NON-NLS-1$
    							name: messages["duplicate-properties"],
    							type: "number", //$NON-NLS-1$
    							defaultValue: warning,
    							options: severities
    						},
    						{
    							id: "validate_empty_rules", //$NON-NLS-1$
    							name: messages["empty-rules"],
    							type: "number", //$NON-NLS-1$
    							defaultValue: warning,
    							options: severities
    						},
    						{
    							id: "validate_fallback_colors", //$NON-NLS-1$
    							name: messages["fallback-colors"],
    							type: "number", //$NON-NLS-1$
    							defaultValue: warning,
    							options: severities
    						},
    						{
    							id: "validate_floats", //$NON-NLS-1$
    							name: messages["floats"],
    							type: "number", //$NON-NLS-1$
    							defaultValue: warning,
    							options: severities
    						},
    						{
    							id: "validate_font_faces", //$NON-NLS-1$
    							name: messages["font-faces"],
    							type: "number", //$NON-NLS-1$
    							defaultValue: warning,
    							options: severities
    						},
    						{
    							id: "validate_font_sizes", //$NON-NLS-1$
    							name: messages["font-sizes"],
    							type: "number", //$NON-NLS-1$
    							defaultValue: warning,
    							options: severities
    						},
    						{
    							id: "validate_gradients", //$NON-NLS-1$
    							name: messages["gradients"],
    							type: "number", //$NON-NLS-1$
    							defaultValue: warning,
    							options: severities
    						},
    						{
    							id: "validate_ids", //$NON-NLS-1$
    							name: messages["ids"],
    							type: "number", //$NON-NLS-1$
    							defaultValue: warning,
    							options: severities
    						},
    						{
    							id: "validate_imports", //$NON-NLS-1$
    							name: messages["import"],
    							type: "number", //$NON-NLS-1$
    							defaultValue: warning,
    							options: severities
    						},
    						{
    							id: "validate_important", //$NON-NLS-1$
    							name: messages["important"],
    							type: "number", //$NON-NLS-1$
    							defaultValue: warning,
    							options: severities
    						},
    						{
    							id: "validate_known_properties", //$NON-NLS-1$
    							name: messages["known-properties"],
    							type: "number", //$NON-NLS-1$
    							defaultValue: warning,
    							options: severities
    						},
    						{
    							id: "validate_outline_none", //$NON-NLS-1$
    							name: messages["outline-none"],
    							type: "number", //$NON-NLS-1$
    							defaultValue: warning,
    							options: severities
    						},
    						{
    							id: "validate_overqualified_elements", //$NON-NLS-1$
    							name: messages["overqualified-elements"],
    							type: "number", //$NON-NLS-1$
    							defaultValue: warning,
    							options: severities
    						},
    						{
    							id: "validate_qualified_headings", //$NON-NLS-1$
    							name: messages["qualified-headings"],
    							type: "number", //$NON-NLS-1$
    							defaultValue: warning,
    							options: severities
    						},
    						{
    							id: "validate_regex_selectors", //$NON-NLS-1$
    							name: messages["regex-selectors"],
    							type: "number", //$NON-NLS-1$
    							defaultValue: warning,
    							options: severities
    						},
    						{
    							id: "validate_rules_count", //$NON-NLS-1$
    							name: messages["rules-count"],
    							type: "number", //$NON-NLS-1$
    							defaultValue: warning,
    							options: severities
    						},
    						{
    							id: "validate_selector_max_approaching", //$NON-NLS-1$
    							name: messages["selector-max-approaching"],
    							type: "number", //$NON-NLS-1$
    							defaultValue: warning,
    							options: severities
    						},
    						{
    							id: "validate_selector_max", //$NON-NLS-1$
    							name: messages["selector-max"],
    							type: "number", //$NON-NLS-1$
    							defaultValue: warning,
    							options: severities
    						},
    						{
    							id: "validate_shorthand", //$NON-NLS-1$
    							name: messages["shorthand"],
    							type: "number", //$NON-NLS-1$
    							defaultValue: warning,
    							options: severities
    						},
    						{
    							id: "validate_star_property_hack", //$NON-NLS-1$
    							name: messages["star-property-hack"],
    							type: "number", //$NON-NLS-1$
    							defaultValue: warning,
    							options: severities
    						},
    						{
    							id: "validate_text_indent", //$NON-NLS-1$
    							name: messages["text-indent"],
    							type: "number", //$NON-NLS-1$
    							defaultValue: warning,
    							options: severities
    						},
    						{
    							id: "validate_underscore_property_hack", //$NON-NLS-1$
    							name: messages["underscore-property-hack"],
    							type: "number", //$NON-NLS-1$
    							defaultValue: warning,
    							options: severities
    						},
    						{
    							id: "validate_unique_headings", //$NON-NLS-1$
    							name: messages["unique-headings"],
    							type: "number", //$NON-NLS-1$
    							defaultValue: warning,
    							options: severities
    						},
    						{
    							id: "validate_universal_selector", //$NON-NLS-1$
    							name: messages["universal-selector"],
    							type: "number", //$NON-NLS-1$
    							defaultValue: warning,
    							options: severities
    						},
    						{
    							id: "validate_unqualified_attributes", //$NON-NLS-1$
    							name: messages["unqualified-attributes"],
    							type: "number", //$NON-NLS-1$
    							defaultValue: warning,
    							options: severities
    						},
    						{
    							id: "validate_vendor_prefix", //$NON-NLS-1$
    							name: messages["vendor-prefix"],
    							type: "number", //$NON-NLS-1$
    							defaultValue: warning,
    							options: severities
    						},
    						{
    							id: "validate_zero_units", //$NON-NLS-1$
    							name: messages["zero-units"],
    							type: "number", //$NON-NLS-1$
    							defaultValue: warning,
    							options: severities
    						}]
    				}]
    		}
    	);

    	provider.connect(function() {
			var fc = serviceRegistry.getService("orion.core.file.client"); //$NON-NLS-1$
	    	fc.addEventListener("Changed", htmlAstManager.onFileChanged.bind(htmlAstManager));
 		});
});


//# sourceMappingURL=webToolsPlugin.js.map