/**
 * Copyright (c) 2014,2019 Contributors to the Eclipse Foundation
 * 
 * See the NOTICE file(s) distributed with this work for additional
 * information regarding copyright ownership.
 * 
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0
 * 
 * SPDX-License-Identifier: EPL-2.0
 */
package org.eclipse.smarthome.model.thing.internal;

import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import java.math.BigDecimal;
import java.net.URI;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.CopyOnWriteArraySet;
import java.util.function.Consumer;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.smarthome.config.core.ConfigDescription;
import org.eclipse.smarthome.config.core.ConfigDescriptionParameter;
import org.eclipse.smarthome.config.core.ConfigDescriptionRegistry;
import org.eclipse.smarthome.config.core.Configuration;
import org.eclipse.smarthome.core.common.registry.AbstractProvider;
import org.eclipse.smarthome.core.i18n.LocaleProvider;
import org.eclipse.smarthome.core.service.ReadyMarker;
import org.eclipse.smarthome.core.service.ReadyMarkerFilter;
import org.eclipse.smarthome.core.service.ReadyService;
import org.eclipse.smarthome.core.thing.Channel;
import org.eclipse.smarthome.core.thing.ChannelUID;
import org.eclipse.smarthome.core.thing.Thing;
import org.eclipse.smarthome.core.thing.ThingProvider;
import org.eclipse.smarthome.core.thing.ThingTypeUID;
import org.eclipse.smarthome.core.thing.ThingUID;
import org.eclipse.smarthome.core.thing.binding.ThingHandlerFactory;
import org.eclipse.smarthome.core.thing.binding.builder.BridgeBuilder;
import org.eclipse.smarthome.core.thing.binding.builder.ChannelBuilder;
import org.eclipse.smarthome.core.thing.binding.builder.ThingBuilder;
import org.eclipse.smarthome.core.thing.type.AutoUpdatePolicy;
import org.eclipse.smarthome.core.thing.type.ChannelDefinition;
import org.eclipse.smarthome.core.thing.type.ChannelKind;
import org.eclipse.smarthome.core.thing.type.ChannelType;
import org.eclipse.smarthome.core.thing.type.ChannelTypeRegistry;
import org.eclipse.smarthome.core.thing.type.ChannelTypeUID;
import org.eclipse.smarthome.core.thing.type.ThingType;
import org.eclipse.smarthome.core.thing.type.ThingTypeRegistry;
import org.eclipse.smarthome.core.thing.util.ThingHelper;
import org.eclipse.smarthome.core.util.BundleResolver;
import org.eclipse.smarthome.model.core.ModelRepository;
import org.eclipse.smarthome.model.core.ModelRepositoryChangeListener;
import org.eclipse.smarthome.model.thing.thing.ModelBridge;
import org.eclipse.smarthome.model.thing.thing.ModelChannel;
import org.eclipse.smarthome.model.thing.thing.ModelProperty;
import org.eclipse.smarthome.model.thing.thing.ModelPropertyContainer;
import org.eclipse.smarthome.model.thing.thing.ModelThing;
import org.eclipse.smarthome.model.thing.thing.ThingModel;
import org.eclipse.xtend.lib.annotations.Data;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.Pure;
import org.eclipse.xtext.xbase.lib.util.ToStringBuilder;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ReferenceCardinality;
import org.osgi.service.component.annotations.ReferencePolicy;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * {@link ThingProvider} implementation which computes *.things files.
 * 
 * @author Oliver Libutzki - Initial contribution
 * @author niehues - Fix ESH Bug 450236
 *         https://bugs.eclipse.org/bugs/show_bug.cgi?id=450236 - Considering
 *         ThingType Description
 * @author Simon Kaufmann - Added asynchronous retry in case the handler
 *         factory cannot load a thing yet (bug 470368),
 *         added delay until ThingTypes are fully loaded
 * @author Markus Rathgeb - Add locale provider support
 */
@Component(immediate = true, service = ThingProvider.class)
@SuppressWarnings("all")
public class GenericThingProvider extends AbstractProvider<Thing> implements ThingProvider, ModelRepositoryChangeListener, ReadyService.ReadyTracker {
  @Data
  private static final class QueueContent {
    private final ThingTypeUID thingTypeUID;
    
    private final String label;
    
    private final Configuration configuration;
    
    private final ThingUID thingUID;
    
    private final ThingUID bridgeUID;
    
    private final ThingHandlerFactory thingHandlerFactory;
    
    public QueueContent(final ThingTypeUID thingTypeUID, final String label, final Configuration configuration, final ThingUID thingUID, final ThingUID bridgeUID, final ThingHandlerFactory thingHandlerFactory) {
      super();
      this.thingTypeUID = thingTypeUID;
      this.label = label;
      this.configuration = configuration;
      this.thingUID = thingUID;
      this.bridgeUID = bridgeUID;
      this.thingHandlerFactory = thingHandlerFactory;
    }
    
    @Override
    @Pure
    public int hashCode() {
      final int prime = 31;
      int result = 1;
      result = prime * result + ((this.thingTypeUID== null) ? 0 : this.thingTypeUID.hashCode());
      result = prime * result + ((this.label== null) ? 0 : this.label.hashCode());
      result = prime * result + ((this.configuration== null) ? 0 : this.configuration.hashCode());
      result = prime * result + ((this.thingUID== null) ? 0 : this.thingUID.hashCode());
      result = prime * result + ((this.bridgeUID== null) ? 0 : this.bridgeUID.hashCode());
      return prime * result + ((this.thingHandlerFactory== null) ? 0 : this.thingHandlerFactory.hashCode());
    }
    
    @Override
    @Pure
    public boolean equals(final Object obj) {
      if (this == obj)
        return true;
      if (obj == null)
        return false;
      if (getClass() != obj.getClass())
        return false;
      GenericThingProvider.QueueContent other = (GenericThingProvider.QueueContent) obj;
      if (this.thingTypeUID == null) {
        if (other.thingTypeUID != null)
          return false;
      } else if (!this.thingTypeUID.equals(other.thingTypeUID))
        return false;
      if (this.label == null) {
        if (other.label != null)
          return false;
      } else if (!this.label.equals(other.label))
        return false;
      if (this.configuration == null) {
        if (other.configuration != null)
          return false;
      } else if (!this.configuration.equals(other.configuration))
        return false;
      if (this.thingUID == null) {
        if (other.thingUID != null)
          return false;
      } else if (!this.thingUID.equals(other.thingUID))
        return false;
      if (this.bridgeUID == null) {
        if (other.bridgeUID != null)
          return false;
      } else if (!this.bridgeUID.equals(other.bridgeUID))
        return false;
      if (this.thingHandlerFactory == null) {
        if (other.thingHandlerFactory != null)
          return false;
      } else if (!this.thingHandlerFactory.equals(other.thingHandlerFactory))
        return false;
      return true;
    }
    
    @Override
    @Pure
    public String toString() {
      ToStringBuilder b = new ToStringBuilder(this);
      b.add("thingTypeUID", this.thingTypeUID);
      b.add("label", this.label);
      b.add("configuration", this.configuration);
      b.add("thingUID", this.thingUID);
      b.add("bridgeUID", this.bridgeUID);
      b.add("thingHandlerFactory", this.thingHandlerFactory);
      return b.toString();
    }
    
    @Pure
    public ThingTypeUID getThingTypeUID() {
      return this.thingTypeUID;
    }
    
    @Pure
    public String getLabel() {
      return this.label;
    }
    
    @Pure
    public Configuration getConfiguration() {
      return this.configuration;
    }
    
    @Pure
    public ThingUID getThingUID() {
      return this.thingUID;
    }
    
    @Pure
    public ThingUID getBridgeUID() {
      return this.bridgeUID;
    }
    
    @Pure
    public ThingHandlerFactory getThingHandlerFactory() {
      return this.thingHandlerFactory;
    }
  }
  
  private final static String XML_THING_TYPE = "esh.xmlThingTypes";
  
  private LocaleProvider localeProvider;
  
  private ModelRepository modelRepository;
  
  private ThingTypeRegistry thingTypeRegistry;
  
  private ChannelTypeRegistry channelTypeRegistry;
  
  private BundleResolver bundleResolver;
  
  private Map<String, Collection<Thing>> thingsMap = new ConcurrentHashMap<String, Collection<Thing>>();
  
  private List<ThingHandlerFactory> thingHandlerFactories = new CopyOnWriteArrayList<ThingHandlerFactory>();
  
  private ConfigDescriptionRegistry configDescriptionRegistry;
  
  private final List<GenericThingProvider.QueueContent> queue = new CopyOnWriteArrayList<GenericThingProvider.QueueContent>();
  
  private Thread lazyRetryThread = null;
  
  private final static Logger logger = LoggerFactory.getLogger(GenericThingProvider.class);
  
  private final Set<String> loadedXmlThingTypes = new CopyOnWriteArraySet<String>();
  
  public void activate() {
    final Consumer<String> _function = (String it) -> {
      this.createThingsFromModel(it);
    };
    this.modelRepository.getAllModelNamesOfType("things").forEach(_function);
  }
  
  @Override
  public Collection<Thing> getAll() {
    return IterableExtensions.<Thing>toList(Iterables.<Thing>concat(this.thingsMap.values()));
  }
  
  private void createThingsFromModel(final String modelName) {
    GenericThingProvider.logger.debug("Read things from model \'{}\'", modelName);
    Collection<Thing> _get = this.thingsMap.get(modelName);
    boolean _tripleEquals = (_get == null);
    if (_tripleEquals) {
      this.thingsMap.put(modelName, CollectionLiterals.<Thing>newArrayList());
    }
    if ((this.modelRepository != null)) {
      EObject _model = this.modelRepository.getModel(modelName);
      final ThingModel model = ((ThingModel) _model);
      if ((model == null)) {
        return;
      }
      final Function1<ModelThing, ThingHandlerFactory> _function = (ModelThing it) -> {
        final ThingUID thingUID = this.constructThingUID(it);
        if ((thingUID != null)) {
          final ThingTypeUID thingTypeUID = this.constructThingTypeUID(it, thingUID);
          final Function1<ThingHandlerFactory, Boolean> _function_1 = (ThingHandlerFactory it_1) -> {
            return Boolean.valueOf(it_1.supportsThingType(thingTypeUID));
          };
          return IterableExtensions.<ThingHandlerFactory>findFirst(this.thingHandlerFactories, _function_1);
        } else {
          return null;
        }
      };
      Iterable<ThingHandlerFactory> _map = IterableExtensions.<ModelThing, ThingHandlerFactory>map(this.flattenModelThings(model.getThings()), _function);
      Iterable<ThingHandlerFactory> _filter = null;
      if (_map!=null) {
        final Function1<ThingHandlerFactory, Boolean> _function_1 = (ThingHandlerFactory it) -> {
          return Boolean.valueOf((it != null));
        };
        _filter=IterableExtensions.<ThingHandlerFactory>filter(_map, _function_1);
      }
      Set<ThingHandlerFactory> _set = null;
      if (_filter!=null) {
        _set=IterableExtensions.<ThingHandlerFactory>toSet(_filter);
      }
      if (_set!=null) {
        final Consumer<ThingHandlerFactory> _function_2 = (ThingHandlerFactory it) -> {
          this.createThingsFromModelForThingHandlerFactory(modelName, it);
        };
        _set.forEach(_function_2);
      }
    }
  }
  
  private ThingUID constructThingUID(final ModelThing modelThing) {
    String _id = modelThing.getId();
    boolean _tripleNotEquals = (_id != null);
    if (_tripleNotEquals) {
      String _id_1 = modelThing.getId();
      return new ThingUID(_id_1);
    } else {
      String _bridgeUID = modelThing.getBridgeUID();
      boolean _tripleNotEquals_1 = (_bridgeUID != null);
      if (_tripleNotEquals_1) {
        String _bridgeUID_1 = modelThing.getBridgeUID();
        final String bindingId = new ThingUID(_bridgeUID_1).getBindingId();
        String _thingTypeId = modelThing.getThingTypeId();
        String _thingId = modelThing.getThingId();
        return new ThingUID(bindingId, _thingTypeId, _thingId);
      } else {
        GenericThingProvider.logger.warn("Thing {} does not have a bridge so it needs to be defined in full notation like <bindingId>:{}:{}", modelThing.getThingTypeId(), modelThing.getThingTypeId(), modelThing.getThingId());
        return null;
      }
    }
  }
  
  private ThingTypeUID constructThingTypeUID(final ModelThing modelThing, final ThingUID thingUID) {
    String _thingTypeId = modelThing.getThingTypeId();
    boolean _tripleNotEquals = (_thingTypeId != null);
    if (_tripleNotEquals) {
      String _bindingId = thingUID.getBindingId();
      String _thingTypeId_1 = modelThing.getThingTypeId();
      return new ThingTypeUID(_bindingId, _thingTypeId_1);
    } else {
      String _bindingId_1 = thingUID.getBindingId();
      String _thingTypeId_2 = thingUID.getThingTypeId();
      return new ThingTypeUID(_bindingId_1, _thingTypeId_2);
    }
  }
  
  private Iterable<ModelThing> flattenModelThings(final Iterable<ModelThing> things) {
    if (((things == null) || (((Object[])Conversions.unwrapArray(things, Object.class)).length == 0))) {
      return Collections.<ModelThing>unmodifiableList(CollectionLiterals.<ModelThing>newArrayList());
    }
    final Consumer<ModelThing> _function = (ModelThing it) -> {
      String _elvis = null;
      String _thingId = it.getThingId();
      if (_thingId != null) {
        _elvis = _thingId;
      } else {
        String _string = this.constructThingUID(it).toString();
        _elvis = _string;
      }
      it.setThingId(_elvis);
    };
    things.forEach(_function);
    final Consumer<ModelBridge> _function_1 = (ModelBridge it) -> {
      final ModelBridge bridge = it;
      final Consumer<ModelThing> _function_2 = (ModelThing it_1) -> {
        final ModelThing thing = it_1;
        thing.setBridgeUID(bridge.getId());
        String _elvis = null;
        String _id = it_1.getId();
        if (_id != null) {
          _elvis = _id;
        } else {
          String _id_1 = bridge.getId();
          ThingUID _thingUID = new ThingUID(_id_1);
          String _string = this.getThingUID(it_1, _thingUID).toString();
          _elvis = _string;
        }
        thing.setId(_elvis);
      };
      bridge.getThings().forEach(_function_2);
    };
    Iterables.<ModelBridge>filter(things, ModelBridge.class).forEach(_function_1);
    final Function1<ModelBridge, EList<ModelThing>> _function_2 = (ModelBridge b) -> {
      return b.getThings();
    };
    Iterable<ModelThing> _flattenModelThings = this.flattenModelThings(Iterables.<ModelThing>concat(IterableExtensions.<ModelBridge, EList<ModelThing>>map(Iterables.<ModelBridge>filter(things, ModelBridge.class), _function_2)));
    return Iterables.<ModelThing>concat(things, _flattenModelThings);
  }
  
  private void createThing(final ModelThing modelThing, final Collection<Thing> thingList, final ThingHandlerFactory thingHandlerFactory) {
    final ThingUID thingUID = this.getThingUID(modelThing, null);
    if ((thingUID == null)) {
      return;
    }
    final ThingTypeUID thingTypeUID = this.constructThingTypeUID(modelThing, thingUID);
    boolean _isSupportedByThingHandlerFactory = this.isSupportedByThingHandlerFactory(thingTypeUID, thingHandlerFactory);
    boolean _not = (!_isSupportedByThingHandlerFactory);
    if (_not) {
      return;
    }
    GenericThingProvider.logger.trace("Creating thing for type \'{}\' with UID \'{}.", thingTypeUID, thingUID);
    final Configuration configuration = this.createConfiguration(modelThing);
    final ThingUID uid = thingUID;
    final Function1<Thing, Boolean> _function = (Thing it) -> {
      return Boolean.valueOf(it.getUID().equals(uid));
    };
    boolean _exists = IterableExtensions.<Thing>exists(thingList, _function);
    if (_exists) {
      GenericThingProvider.logger.debug("Thing already exists {}", uid.toString());
      return;
    }
    final ThingType thingType = this.getThingType(thingTypeUID);
    String _xifexpression = null;
    String _label = modelThing.getLabel();
    boolean _tripleNotEquals = (_label != null);
    if (_tripleNotEquals) {
      _xifexpression = modelThing.getLabel();
    } else {
      String _label_1 = null;
      if (thingType!=null) {
        _label_1=thingType.getLabel();
      }
      _xifexpression = _label_1;
    }
    final String label = _xifexpression;
    final String location = modelThing.getLocation();
    ThingUID _xifexpression_1 = null;
    String _bridgeUID = modelThing.getBridgeUID();
    boolean _tripleNotEquals_1 = (_bridgeUID != null);
    if (_tripleNotEquals_1) {
      String _bridgeUID_1 = modelThing.getBridgeUID();
      _xifexpression_1 = new ThingUID(_bridgeUID_1);
    }
    final ThingUID bridgeUID = _xifexpression_1;
    final Thing thingFromHandler = this.getThingFromThingHandlerFactories(thingTypeUID, label, configuration, thingUID, bridgeUID, thingHandlerFactory);
    ThingBuilder _xifexpression_2 = null;
    if ((modelThing instanceof ModelBridge)) {
      _xifexpression_2 = BridgeBuilder.create(thingTypeUID, thingUID);
    } else {
      _xifexpression_2 = ThingBuilder.create(thingTypeUID, thingUID);
    }
    final ThingBuilder thingBuilder = _xifexpression_2;
    thingBuilder.withConfiguration(configuration);
    thingBuilder.withBridge(bridgeUID);
    thingBuilder.withLabel(label);
    thingBuilder.withLocation(location);
    EList<ModelChannel> _channels = modelThing.getChannels();
    List<ChannelDefinition> _elvis = null;
    List<ChannelDefinition> _channelDefinitions = null;
    if (thingType!=null) {
      _channelDefinitions=thingType.getChannelDefinitions();
    }
    if (_channelDefinitions != null) {
      _elvis = _channelDefinitions;
    } else {
      ArrayList<ChannelDefinition> _newArrayList = CollectionLiterals.<ChannelDefinition>newArrayList();
      _elvis = _newArrayList;
    }
    final List<Channel> channels = this.createChannels(thingTypeUID, thingUID, _channels, _elvis);
    thingBuilder.withChannels(channels);
    Thing thing = thingBuilder.build();
    if ((thingFromHandler != null)) {
      this.merge(thingFromHandler, thing);
    }
    Thing _elvis_1 = null;
    if (thingFromHandler != null) {
      _elvis_1 = thingFromHandler;
    } else {
      _elvis_1 = thing;
    }
    thingList.add(_elvis_1);
  }
  
  private boolean isSupportedByThingHandlerFactory(final ThingTypeUID thingTypeUID, final ThingHandlerFactory specific) {
    if ((specific != null)) {
      return specific.supportsThingType(thingTypeUID);
    }
    for (final ThingHandlerFactory thingHandlerFactory : this.thingHandlerFactories) {
      boolean _supportsThingType = thingHandlerFactory.supportsThingType(thingTypeUID);
      if (_supportsThingType) {
        return true;
      }
    }
    return false;
  }
  
  private Thing getThingFromThingHandlerFactories(final ThingTypeUID thingTypeUID, final String label, final Configuration configuration, final ThingUID thingUID, final ThingUID bridgeUID, final ThingHandlerFactory specific) {
    Object _xblockexpression = null;
    {
      if (((specific != null) && specific.supportsThingType(thingTypeUID))) {
        GenericThingProvider.logger.trace("Creating thing from specific ThingHandlerFactory {} for thingType {}", specific, thingTypeUID);
        return this.getThingFromThingHandlerFactory(thingTypeUID, label, configuration, thingUID, bridgeUID, specific);
      }
      for (final ThingHandlerFactory thingHandlerFactory : this.thingHandlerFactories) {
        {
          GenericThingProvider.logger.trace("Searching thingHandlerFactory for thingType: {}", thingTypeUID);
          boolean _supportsThingType = thingHandlerFactory.supportsThingType(thingTypeUID);
          if (_supportsThingType) {
            return this.getThingFromThingHandlerFactory(thingTypeUID, label, configuration, thingUID, bridgeUID, thingHandlerFactory);
          }
        }
      }
      _xblockexpression = null;
    }
    return ((Thing)_xblockexpression);
  }
  
  private Thing getThingFromThingHandlerFactory(final ThingTypeUID thingTypeUID, final String label, final Configuration configuration, final ThingUID thingUID, final ThingUID bridgeUID, final ThingHandlerFactory thingHandlerFactory) {
    final Thing thing = thingHandlerFactory.createThing(thingTypeUID, configuration, thingUID, bridgeUID);
    if ((thing == null)) {
      GenericThingProvider.logger.debug(
        "ThingHandlerFactory \'{}\' claimed it can handle \'{}\' type but actually did not. Queued for later refresh.", 
        thingHandlerFactory.getClass().getSimpleName(), thingTypeUID.getAsString());
      GenericThingProvider.QueueContent _queueContent = new GenericThingProvider.QueueContent(thingTypeUID, label, configuration, thingUID, bridgeUID, thingHandlerFactory);
      this.queue.add(_queueContent);
      if (((this.lazyRetryThread == null) || (!this.lazyRetryThread.isAlive()))) {
        Thread _thread = new Thread(this.lazyRetryRunnable);
        this.lazyRetryThread = _thread;
        this.lazyRetryThread.start();
      }
    } else {
      thing.setLabel(label);
    }
    return thing;
  }
  
  protected void _merge(final Thing targetThing, final Thing sourceThing) {
    targetThing.setBridgeUID(sourceThing.getBridgeUID());
    this.merge(targetThing.getConfiguration(), sourceThing.getConfiguration());
    this.merge(targetThing, sourceThing.getChannels());
    targetThing.setLocation(sourceThing.getLocation());
    targetThing.setLabel(sourceThing.getLabel());
  }
  
  protected void _merge(final Configuration target, final Configuration source) {
    final Consumer<String> _function = (String it) -> {
      target.put(it, source.get(it));
    };
    source.keySet().forEach(_function);
  }
  
  protected void _merge(final Thing targetThing, final List<Channel> source) {
    final List<Channel> channelsToAdd = CollectionLiterals.<Channel>newArrayList();
    final Consumer<Channel> _function = (Channel sourceChannel) -> {
      final Function1<Channel, Boolean> _function_1 = (Channel it) -> {
        return Boolean.valueOf(it.getUID().equals(sourceChannel.getUID()));
      };
      final Iterable<Channel> targetChannels = IterableExtensions.<Channel>filter(targetThing.getChannels(), _function_1);
      final Consumer<Channel> _function_2 = (Channel it) -> {
        this.merge(it, sourceChannel);
      };
      targetChannels.forEach(_function_2);
      boolean _isEmpty = IterableExtensions.isEmpty(targetChannels);
      if (_isEmpty) {
        channelsToAdd.add(sourceChannel);
      }
    };
    source.forEach(_function);
    ThingHelper.addChannelsToThing(targetThing, channelsToAdd);
  }
  
  protected void _merge(final Channel target, final Channel source) {
    this.merge(target.getConfiguration(), source.getConfiguration());
  }
  
  private ArrayList<String> getParentPath(final ThingUID bridgeUID) {
    final ArrayList<String> bridgeIds = CollectionLiterals.<String>newArrayList();
    bridgeIds.addAll(bridgeUID.getBridgeIds());
    bridgeIds.add(bridgeUID.getId());
    return bridgeIds;
  }
  
  private List<Channel> createChannels(final ThingTypeUID thingTypeUID, final ThingUID thingUID, final List<ModelChannel> modelChannels, final List<ChannelDefinition> channelDefinitions) {
    List<Channel> _xblockexpression = null;
    {
      final Set<String> addedChannelIds = CollectionLiterals.<String>newHashSet();
      final List<Channel> channels = CollectionLiterals.<Channel>newArrayList();
      final Consumer<ModelChannel> _function = (ModelChannel it) -> {
        boolean _add = addedChannelIds.add(it.getId());
        if (_add) {
          ChannelKind parsedKind = ChannelKind.STATE;
          ChannelTypeUID channelTypeUID = null;
          String itemType = null;
          String label = it.getLabel();
          final Configuration configuration = this.createConfiguration(it);
          AutoUpdatePolicy autoUpdatePolicy = null;
          String _channelType = it.getChannelType();
          boolean _tripleNotEquals = (_channelType != null);
          if (_tripleNotEquals) {
            String _bindingId = thingUID.getBindingId();
            String _channelType_1 = it.getChannelType();
            ChannelTypeUID _channelTypeUID = new ChannelTypeUID(_bindingId, _channelType_1);
            channelTypeUID = _channelTypeUID;
            final ChannelType resolvedChannelType = this.getChannelType(channelTypeUID);
            if ((resolvedChannelType != null)) {
              itemType = resolvedChannelType.getItemType();
              parsedKind = resolvedChannelType.getKind();
              if ((label == null)) {
                label = resolvedChannelType.getLabel();
              }
              autoUpdatePolicy = resolvedChannelType.getAutoUpdatePolicy();
              this.applyDefaultConfiguration(configuration, resolvedChannelType);
            } else {
              GenericThingProvider.logger.error("Channel type {} could not be resolved.", channelTypeUID.getAsString());
            }
          } else {
            itemType = it.getType();
            String _xifexpression = null;
            String _channelKind = it.getChannelKind();
            boolean _tripleEquals = (_channelKind == null);
            if (_tripleEquals) {
              _xifexpression = "State";
            } else {
              _xifexpression = it.getChannelKind();
            }
            final String kind = _xifexpression;
            parsedKind = ChannelKind.parse(kind);
          }
          String _id = it.getId();
          ChannelUID _channelUID = new ChannelUID(thingUID, _id);
          ChannelBuilder channel = ChannelBuilder.create(_channelUID, itemType).withKind(parsedKind).withConfiguration(configuration).withType(channelTypeUID).withLabel(label).withAutoUpdatePolicy(autoUpdatePolicy);
          Channel _build = channel.build();
          channels.add(_build);
        }
      };
      modelChannels.forEach(_function);
      final Consumer<ChannelDefinition> _function_1 = (ChannelDefinition it) -> {
        boolean _add = addedChannelIds.add(it.getId());
        if (_add) {
          final ChannelType channelType = this.getChannelType(it.getChannelTypeUID());
          if ((channelType != null)) {
            String _id = it.getId();
            ChannelUID _channelUID = new ChannelUID(thingTypeUID, thingUID, _id);
            Channel _build = ChannelBuilder.create(_channelUID, channelType.getItemType()).withType(it.getChannelTypeUID()).withAutoUpdatePolicy(channelType.getAutoUpdatePolicy()).build();
            channels.add(_build);
          } else {
            GenericThingProvider.logger.warn(
              "Could not create channel \'{}\' for thing \'{}\', because channel type \'{}\' could not be found.", 
              it.getId(), thingUID, it.getChannelTypeUID());
          }
        }
      };
      channelDefinitions.forEach(_function_1);
      _xblockexpression = channels;
    }
    return _xblockexpression;
  }
  
  private void applyDefaultConfiguration(final Configuration configuration, final ChannelType channelType) {
    if (((this.configDescriptionRegistry != null) && (configuration != null))) {
      URI _configDescriptionURI = channelType.getConfigDescriptionURI();
      boolean _tripleNotEquals = (_configDescriptionURI != null);
      if (_tripleNotEquals) {
        final ConfigDescription configDescription = this.configDescriptionRegistry.getConfigDescription(channelType.getConfigDescriptionURI());
        if ((configDescription != null)) {
          final Function1<ConfigDescriptionParameter, Boolean> _function = (ConfigDescriptionParameter it) -> {
            return Boolean.valueOf(((it.getDefault() != null) && (configuration.get(it.getName()) == null)));
          };
          final Consumer<ConfigDescriptionParameter> _function_1 = (ConfigDescriptionParameter it) -> {
            final Object value = this.getDefaultValueAsCorrectType(it.getType(), it.getDefault());
            if ((value != null)) {
              configuration.put(it.getName(), value);
            }
          };
          IterableExtensions.<ConfigDescriptionParameter>filter(configDescription.getParameters(), _function).forEach(_function_1);
        }
      }
    }
  }
  
  private Object getDefaultValueAsCorrectType(final ConfigDescriptionParameter.Type parameterType, final String defaultValue) {
    try {
      if (parameterType != null) {
        switch (parameterType) {
          case TEXT:
            return defaultValue;
          case BOOLEAN:
            return Boolean.valueOf(Boolean.parseBoolean(defaultValue));
          case INTEGER:
            return new BigDecimal(defaultValue);
          case DECIMAL:
            return new BigDecimal(defaultValue);
          default:
            return null;
        }
      } else {
        return null;
      }
    } catch (final Throwable _t) {
      if (_t instanceof NumberFormatException) {
        final NumberFormatException ex = (NumberFormatException)_t;
        GenericThingProvider.logger.warn("Could not parse default value \'{}\' as type \'{}\': {}", defaultValue, parameterType, ex.getMessage(), ex);
        return null;
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
  }
  
  private Configuration createConfiguration(final ModelPropertyContainer propertyContainer) {
    Configuration _xblockexpression = null;
    {
      final Configuration configuration = new Configuration();
      final Consumer<ModelProperty> _function = (ModelProperty it) -> {
        int _size = it.getValue().size();
        boolean _tripleEquals = (_size == 1);
        if (_tripleEquals) {
          configuration.put(it.getKey(), it.getValue().get(0));
        } else {
          configuration.put(it.getKey(), it.getValue());
        }
      };
      propertyContainer.getProperties().forEach(_function);
      _xblockexpression = configuration;
    }
    return _xblockexpression;
  }
  
  private ThingType getThingType(final ThingTypeUID thingTypeUID) {
    ThingType _thingType = null;
    if (this.thingTypeRegistry!=null) {
      _thingType=this.thingTypeRegistry.getThingType(thingTypeUID, this.localeProvider.getLocale());
    }
    return _thingType;
  }
  
  private ChannelType getChannelType(final ChannelTypeUID channelTypeUID) {
    ChannelType _channelType = null;
    if (this.channelTypeRegistry!=null) {
      _channelType=this.channelTypeRegistry.getChannelType(channelTypeUID, this.localeProvider.getLocale());
    }
    return _channelType;
  }
  
  @Reference
  protected void setModelRepository(final ModelRepository modelRepository) {
    this.modelRepository = modelRepository;
    modelRepository.addModelRepositoryChangeListener(this);
  }
  
  protected void unsetModelRepository(final ModelRepository modelRepository) {
    modelRepository.removeModelRepositoryChangeListener(this);
    this.modelRepository = null;
  }
  
  @Reference
  protected void setBundleResolver(final BundleResolver bundleResolver) {
    this.bundleResolver = bundleResolver;
  }
  
  protected void unsetBundleResolver(final BundleResolver bundleResolver) {
    this.bundleResolver = null;
  }
  
  @Override
  public void modelChanged(final String modelName, final org.eclipse.smarthome.model.core.EventType type) {
    boolean _endsWith = modelName.endsWith("things");
    if (_endsWith) {
      if (type != null) {
        switch (type) {
          case ADDED:
            this.createThingsFromModel(modelName);
            break;
          case MODIFIED:
            Collection<Thing> _elvis = null;
            Collection<Thing> _get = this.thingsMap.get(modelName);
            if (_get != null) {
              _elvis = _get;
            } else {
              ArrayList<Thing> _newArrayList = CollectionLiterals.<Thing>newArrayList();
              _elvis = _newArrayList;
            }
            final Collection<Thing> oldThings = _elvis;
            EObject _model = this.modelRepository.getModel(modelName);
            final ThingModel model = ((ThingModel) _model);
            if ((model != null)) {
              final Set<ThingUID> newThingUIDs = this.getAllThingUIDs(model);
              final Function1<Thing, Boolean> _function = (Thing it) -> {
                boolean _contains = newThingUIDs.contains(it.getUID());
                return Boolean.valueOf((!_contains));
              };
              final Iterable<Thing> removedThings = IterableExtensions.<Thing>filter(oldThings, _function);
              final Consumer<Thing> _function_1 = (Thing it) -> {
                this.notifyListenersAboutRemovedElement(it);
              };
              removedThings.forEach(_function_1);
              this.createThingsFromModel(modelName);
            }
            break;
          case REMOVED:
            Collection<Thing> _elvis_1 = null;
            Collection<Thing> _remove = this.thingsMap.remove(modelName);
            if (_remove != null) {
              _elvis_1 = _remove;
            } else {
              ArrayList<Thing> _newArrayList_1 = CollectionLiterals.<Thing>newArrayList();
              _elvis_1 = _newArrayList_1;
            }
            final Collection<Thing> things = _elvis_1;
            final Consumer<Thing> _function_2 = (Thing it) -> {
              this.notifyListenersAboutRemovedElement(it);
            };
            things.forEach(_function_2);
            break;
          default:
            break;
        }
      }
    }
  }
  
  private Set<ThingUID> getAllThingUIDs(final ThingModel model) {
    return this.getAllThingUIDs(model.getThings(), null);
  }
  
  private Set<ThingUID> getAllThingUIDs(final List<ModelThing> thingList, final ThingUID parentUID) {
    final HashSet<ThingUID> ret = new HashSet<ThingUID>();
    final Consumer<ModelThing> _function = (ModelThing it) -> {
      final ThingUID thingUID = this.getThingUID(it, parentUID);
      ret.add(thingUID);
      if ((it instanceof ModelBridge)) {
        ret.addAll(this.getAllThingUIDs(((ModelBridge)it).getThings(), thingUID));
      }
    };
    thingList.forEach(_function);
    return ret;
  }
  
  private ThingUID getThingUID(final ModelThing modelThing, final ThingUID parentUID) {
    if (((parentUID != null) && (modelThing.getId() == null))) {
      String _bindingId = parentUID.getBindingId();
      String _thingTypeId = modelThing.getThingTypeId();
      final ThingTypeUID thingTypeUID = new ThingTypeUID(_bindingId, _thingTypeId);
      String _thingId = modelThing.getThingId();
      ArrayList<String> _parentPath = this.getParentPath(parentUID);
      return new ThingUID(thingTypeUID, _thingId, ((String[])Conversions.unwrapArray(_parentPath, String.class)));
    } else {
      return this.constructThingUID(modelThing);
    }
  }
  
  @Reference
  protected void setLocaleProvider(final LocaleProvider localeProvider) {
    this.localeProvider = localeProvider;
  }
  
  protected void unsetLocaleProvider(final LocaleProvider localeProvider) {
    this.localeProvider = null;
  }
  
  @Reference(cardinality = ReferenceCardinality.OPTIONAL, policy = ReferencePolicy.DYNAMIC)
  protected void setThingTypeRegistry(final ThingTypeRegistry thingTypeRegistry) {
    this.thingTypeRegistry = thingTypeRegistry;
  }
  
  protected void unsetThingTypeRegistry(final ThingTypeRegistry thingTypeRegistry) {
    this.thingTypeRegistry = null;
  }
  
  @Reference
  protected void setChannelTypeRegistry(final ChannelTypeRegistry channelTypeRegistry) {
    this.channelTypeRegistry = channelTypeRegistry;
  }
  
  protected void unsetChannelTypeRegistry(final ChannelTypeRegistry channelTypeRegistry) {
    this.channelTypeRegistry = null;
  }
  
  @Reference(cardinality = ReferenceCardinality.MULTIPLE, policy = ReferencePolicy.DYNAMIC)
  protected void addThingHandlerFactory(final ThingHandlerFactory thingHandlerFactory) {
    GenericThingProvider.logger.debug("ThingHandlerFactory added {}", thingHandlerFactory);
    this.thingHandlerFactories.add(thingHandlerFactory);
    this.thingHandlerFactoryAdded(thingHandlerFactory);
  }
  
  protected void removeThingHandlerFactory(final ThingHandlerFactory thingHandlerFactory) {
    this.thingHandlerFactories.remove(thingHandlerFactory);
    this.thingHandlerFactoryRemoved();
  }
  
  private Object thingHandlerFactoryRemoved() {
    return null;
  }
  
  public void thingHandlerFactoryAdded(final ThingHandlerFactory thingHandlerFactory) {
    final Consumer<String> _function = (String it) -> {
      this.createThingsFromModelForThingHandlerFactory(it, thingHandlerFactory);
    };
    this.thingsMap.keySet().forEach(_function);
  }
  
  @Reference
  public void setReadyService(final ReadyService readyService) {
    readyService.registerTracker(this, new ReadyMarkerFilter().withType(GenericThingProvider.XML_THING_TYPE));
  }
  
  public void unsetReadyService(final ReadyService readyService) {
    readyService.unregisterTracker(this);
  }
  
  @Override
  public void onReadyMarkerAdded(final ReadyMarker readyMarker) {
    final String bsn = readyMarker.getIdentifier();
    this.loadedXmlThingTypes.add(bsn);
    this.handleXmlThingTypesLoaded(bsn);
  }
  
  private String getBundleName(final ThingHandlerFactory thingHandlerFactory) {
    return this.bundleResolver.resolveBundle(thingHandlerFactory.getClass()).getSymbolicName();
  }
  
  private void handleXmlThingTypesLoaded(final String bsn) {
    final Function1<ThingHandlerFactory, Boolean> _function = (ThingHandlerFactory it) -> {
      return Boolean.valueOf(this.getBundleName(it).equals(bsn));
    };
    final Consumer<ThingHandlerFactory> _function_1 = (ThingHandlerFactory thingHandlerFactory) -> {
      this.thingHandlerFactoryAdded(thingHandlerFactory);
    };
    IterableExtensions.<ThingHandlerFactory>filter(this.thingHandlerFactories, _function).forEach(_function_1);
  }
  
  @Override
  public void onReadyMarkerRemoved(final ReadyMarker readyMarker) {
    final String bsn = readyMarker.getIdentifier();
    this.loadedXmlThingTypes.remove(bsn);
  }
  
  private void createThingsFromModelForThingHandlerFactory(final String modelName, final ThingHandlerFactory factory) {
    boolean _contains = this.loadedXmlThingTypes.contains(this.getBundleName(factory));
    boolean _not = (!_contains);
    if (_not) {
      return;
    }
    final Thing[] oldThings = ((Thing[])Conversions.unwrapArray(this.thingsMap.get(modelName), Thing.class)).clone();
    final ArrayList<Thing> newThings = CollectionLiterals.<Thing>newArrayList();
    if ((this.modelRepository != null)) {
      EObject _model = this.modelRepository.getModel(modelName);
      final ThingModel model = ((ThingModel) _model);
      if ((model != null)) {
        final Consumer<ModelThing> _function = (ModelThing it) -> {
          this.createThing(it, newThings, factory);
        };
        this.flattenModelThings(model.getThings()).forEach(_function);
      }
    }
    final Consumer<Thing> _function_1 = (Thing newThing) -> {
      final Function1<Thing, Boolean> _function_2 = (Thing it) -> {
        ThingUID _uID = it.getUID();
        ThingUID _uID_1 = newThing.getUID();
        return Boolean.valueOf(Objects.equal(_uID, _uID_1));
      };
      final Thing oldThing = IterableExtensions.<Thing>findFirst(((Iterable<Thing>)Conversions.doWrapArray(oldThings)), _function_2);
      if ((oldThing != null)) {
        boolean _equals = ThingHelper.equals(oldThing, newThing);
        boolean _not_1 = (!_equals);
        if (_not_1) {
          GenericThingProvider.logger.debug("Updating thing \'{}\' from model \'{}\'.", newThing.getUID(), modelName);
          this.notifyListenersAboutUpdatedElement(oldThing, newThing);
        }
      } else {
        GenericThingProvider.logger.debug("Adding thing \'{}\' from model \'{}\'.", newThing.getUID(), modelName);
        this.thingsMap.get(modelName).add(newThing);
        this.notifyListenersAboutAddedElement(newThing);
      }
    };
    newThings.forEach(_function_1);
  }
  
  private final Runnable lazyRetryRunnable = new Runnable() {
    @Override
    public void run() {
      try {
        GenericThingProvider.logger.debug("Starting lazy retry thread");
        while ((!GenericThingProvider.this.queue.isEmpty())) {
          {
            boolean _isEmpty = GenericThingProvider.this.queue.isEmpty();
            boolean _not = (!_isEmpty);
            if (_not) {
              final ArrayList<Thing> newThings = new ArrayList<Thing>();
              final Consumer<GenericThingProvider.QueueContent> _function = (GenericThingProvider.QueueContent qc) -> {
                GenericThingProvider.logger.trace("Searching thingHandlerFactory for thingType: {}", qc.thingTypeUID);
                final Thing thing = qc.thingHandlerFactory.createThing(qc.thingTypeUID, qc.configuration, qc.thingUID, 
                  qc.bridgeUID);
                if ((thing != null)) {
                  GenericThingProvider.this.queue.remove(qc);
                  GenericThingProvider.logger.debug("Successfully loaded \'{}\' during retry", qc.thingUID);
                  newThings.add(thing);
                }
              };
              GenericThingProvider.this.queue.forEach(_function);
              boolean _isEmpty_1 = newThings.isEmpty();
              boolean _not_1 = (!_isEmpty_1);
              if (_not_1) {
                final Consumer<Thing> _function_1 = (Thing newThing) -> {
                  final Function1<String, Boolean> _function_2 = (String mName) -> {
                    final Function1<Thing, Boolean> _function_3 = (Thing it) -> {
                      ThingUID _uID = it.getUID();
                      ThingUID _uID_1 = newThing.getUID();
                      return Boolean.valueOf(Objects.equal(_uID, _uID_1));
                    };
                    boolean _isEmpty_2 = IterableExtensions.isEmpty(IterableExtensions.<Thing>filter(GenericThingProvider.this.thingsMap.get(mName), _function_3));
                    return Boolean.valueOf((!_isEmpty_2));
                  };
                  final String modelName = IterableExtensions.<String>findFirst(GenericThingProvider.this.thingsMap.keySet(), _function_2);
                  final Function1<Thing, Boolean> _function_3 = (Thing it) -> {
                    ThingUID _uID = it.getUID();
                    ThingUID _uID_1 = newThing.getUID();
                    return Boolean.valueOf(Objects.equal(_uID, _uID_1));
                  };
                  final Thing oldThing = IterableExtensions.<Thing>findFirst(GenericThingProvider.this.thingsMap.get(modelName), _function_3);
                  if ((oldThing != null)) {
                    GenericThingProvider.this.merge(newThing, oldThing);
                    GenericThingProvider.this.thingsMap.get(modelName).remove(oldThing);
                    GenericThingProvider.this.thingsMap.get(modelName).add(newThing);
                    GenericThingProvider.logger.debug("Refreshing thing \'{}\' after successful retry", newThing.getUID());
                    boolean _equals = ThingHelper.equals(oldThing, newThing);
                    boolean _not_2 = (!_equals);
                    if (_not_2) {
                      GenericThingProvider.this.notifyListenersAboutUpdatedElement(oldThing, newThing);
                    }
                  } else {
                    String _format = String.format("Item %s not yet known", newThing.getUID());
                    throw new IllegalStateException(_format);
                  }
                };
                newThings.forEach(_function_1);
              }
            }
            boolean _isEmpty_2 = GenericThingProvider.this.queue.isEmpty();
            boolean _not_2 = (!_isEmpty_2);
            if (_not_2) {
              Thread.sleep(1000);
            }
          }
        }
        GenericThingProvider.logger.debug("Lazy retry thread ran out of work. Good bye.");
      } catch (Throwable _e) {
        throw Exceptions.sneakyThrow(_e);
      }
    }
  };
  
  @Reference
  protected void setConfigDescriptionRegistry(final ConfigDescriptionRegistry configDescriptionRegistry) {
    this.configDescriptionRegistry = configDescriptionRegistry;
  }
  
  protected void unsetConfigDescriptionRegistry(final ConfigDescriptionRegistry configDescriptionRegistry) {
    this.configDescriptionRegistry = null;
  }
  
  public void merge(final Object targetThing, final Object source) {
    if (targetThing instanceof Thing
         && source instanceof List) {
      _merge((Thing)targetThing, (List<Channel>)source);
      return;
    } else if (targetThing instanceof Thing
         && source instanceof Thing) {
      _merge((Thing)targetThing, (Thing)source);
      return;
    } else if (targetThing instanceof Configuration
         && source instanceof Configuration) {
      _merge((Configuration)targetThing, (Configuration)source);
      return;
    } else if (targetThing instanceof Channel
         && source instanceof Channel) {
      _merge((Channel)targetThing, (Channel)source);
      return;
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(targetThing, source).toString());
    }
  }
}
