/* ====================================================================
 * Copyright (c) 2002 Johnny Shelley.  All rights reserved.
 *
 * Bcrypt is licensed under the BSD software license. See the file
 * called 'LICENSE' that you should have received with this software
 * for details
 * ====================================================================
 */

#include "includes.h"
#include "defines.h"
#include "functions.h"

/* 256 KiB, as recommended by http://www.zlib.net/zlib_how.html */
#define ZCHUNK (256 * 1024)
#define BLOCK_SIZE (sizeof(uint32_t) * 2)

off_t BFEncrypt(FILE *infd, FILE *outfd, char *key, BCoptions *options) {
  BLOWFISH_CTX ctx;
  unsigned char *myEndian = NULL;
  uLong sz = 0;

  /* zlib variables */
  int ret, flush;
  unsigned have;
  z_stream strm;
  uint8_t in[ZCHUNK];
  uint8_t out[ZCHUNK];

  strm.zalloc = Z_NULL;
  strm.zfree = Z_NULL;
  strm.opaque = Z_NULL;
  ret = deflateInit(&strm, Z_DEFAULT_COMPRESSION);
  if (ret != Z_OK)
    exit(2);


  getEndian(&myEndian);

  if (fwrite(myEndian, sizeof(uint8_t), 1, outfd) != 1) {
    exit(EIO);
  }
  if (fwrite(&options->compression, sizeof(uint8_t), 1, outfd) != 1) {
    exit(EIO);
  }

  Blowfish_Init(&ctx, (unsigned char*)key, MAXKEYBYTES);

  int n;
  /* 2 * ZCHUNK for the overhead when compressing */
  uint8_t rest[2 * ZCHUNK];
  int restidx = 0;
  uint32_t *rwalk = (uint32_t*)rest;
  do {
    if ((n = fread(in, 1, ZCHUNK, infd)) < ZCHUNK) {
      if (!feof(infd)) {
        exit(3);
      }
    }
    strm.avail_in = n;
    strm.next_in = in;
    flush = ((n == 0) ? Z_FINISH : Z_NO_FLUSH);
    if (options->compression) {
      do {
        strm.avail_out = ZCHUNK;
        strm.next_out = out;
        ret = deflate(&strm, flush);

        have = ZCHUNK - strm.avail_out;
        memcpy(rest + restidx, out, have);
        restidx += have;
        sz += have;
      } while (strm.avail_out == 0);
    } else {
      memcpy(rest + restidx, in, n);
      restidx += n;
      sz += n;
    }

    /* Encrypt as many blocks (each BLOCK_SIZE bytes big) as we can */
    int i;
    /* i adresses sizeof(uint32_t) = 4 byte blocks,
     * but we use (restidx / BLOCK_SIZE) loop up to full blocks */
    for (i = 0; i < 2 * (restidx / BLOCK_SIZE); i += 2) {
      Blowfish_Encrypt(&ctx, &rwalk[i], &rwalk[i + 1]);
      if (fwrite(&rwalk[i], BLOCK_SIZE, 1, outfd) != 1) {
        exit(EIO);
      }
    }

    if (i > 0) {
      int remaining = restidx - (i * sizeof(uint32_t));
      memmove(rest, &rwalk[i], remaining);
      restidx = remaining;
    }
  } while (flush != Z_FINISH);

  /* Now append the key (MAXKEYBYTES bytes) */
  memcpy(rest + restidx, key, MAXKEYBYTES);
  restidx += MAXKEYBYTES;
  sz += MAXKEYBYTES;

  /* Pad the rest with 0-bytes */
  int r;
  if (sz >= BLOCK_SIZE)
    r = getremain(sz, BLOCK_SIZE);
  else
    r = BLOCK_SIZE - sz;

  memset(rest + restidx, '\0', r);
  restidx += r;
  sz += r;

  /* Encrypt the rest */
  int i;
  for (i = 0; i < (restidx / sizeof(uint32_t)); i += 2) {
    Blowfish_Encrypt(&ctx, &rwalk[i], &rwalk[i+1]);
    if (fwrite(&rwalk[i], BLOCK_SIZE, 1, outfd) != 1) {
      exit(EIO);
    }
  }

  if (options->compression) {
    if (fwrite(&options->origsize, sizeof(uint32_t), 1, outfd) != 1) {
      exit(EIO);
    }
  }

  free(myEndian);
  return(sz);
}


off_t BFDecrypt(FILE *infd, FILE *outfd, char *key, char *key2, BCoptions *options) {
  BLOWFISH_CTX ctx;
  char *mykey = NULL;
  int swap = 0;
  uLong sz = 0;

  /* zlib variables */
  int ret;
  unsigned have;
  z_stream strm;
  uint8_t in[ZCHUNK];
  uint8_t out[ZCHUNK];
  uint8_t last_chunk[BLOCK_SIZE + MAXKEYBYTES];
  int last_chunk_length;

  strm.zalloc = Z_NULL;
  strm.zfree = Z_NULL;
  strm.opaque = Z_NULL;
  strm.avail_in = 0;
  strm.next_in = Z_NULL;
  ret = inflateInit(&strm);
  if (ret != Z_OK)
    exit(2);


  if ((mykey = calloc(MAXKEYBYTES + 1, 1)) == NULL)
    memerror();

  uint8_t endianness, compressed;
  if (fread(&endianness, sizeof(uint8_t), 1, infd) != 1) {
      exit(EIO);
  }
  if (fread(&compressed, sizeof(uint8_t), 1, infd) != 1) {
      exit(EIO);
  }

  if ((swap = testEndian(&endianness)) == 1)
    memcpy(mykey, key2, MAXKEYBYTES);
  else
    memcpy(mykey, key, MAXKEYBYTES);

  Blowfish_Init(&ctx, (unsigned char*)mykey, MAXKEYBYTES);

  int n;
  uint32_t *inwalk = (uint32_t*)in;
  memset(last_chunk, 0, BLOCK_SIZE);
  last_chunk_length = 0;
  do {
    if ((n = fread(in, 1, ZCHUNK, infd)) < ZCHUNK) {
      if (!feof(infd)) {
        exit(EIO);
      }
    }

    /* Decrypt as many blocks (each BLOCK_SIZE bytes big) as we can */
    int i;
    /* i adresses sizeof(uint32_t) = 4 byte blocks,
     * but we use (n / BLOCK_SIZE) loop up to full blocks */
    for (i = 0; i < 2 * (n / BLOCK_SIZE); i += 2) {
      if (swap) {
        inwalk[i] = swapEndian(inwalk[i]);
        inwalk[i + 1] = swapEndian(inwalk[i + 1]);
      }
      Blowfish_Decrypt(&ctx, &inwalk[i], &inwalk[i + 1]);
      if (swap) {
        inwalk[i] = swapEndian(inwalk[i]);
        inwalk[i + 1] = swapEndian(inwalk[i + 1]);
      }
    }

    strm.avail_in = n;
    strm.next_in = in;
    if (compressed) {
      do {
        strm.avail_out = ZCHUNK;
        strm.next_out = out;
        ret = inflate(&strm, Z_NO_FLUSH);
        switch (ret) {
          case Z_NEED_DICT:
            ret = Z_DATA_ERROR;
            /* fall-through */
          case Z_DATA_ERROR:
            fprintf(stderr, "zlib: invalid data.\nThis probably means that the provided decryption key is wrong\nor the file is corrupted.\n");
            exit(1);
          case Z_MEM_ERROR:
            fprintf(stderr, "zlib: memory error\n");
            exit(ENOMEM);
        }

        have = ZCHUNK - strm.avail_out;
        if (fwrite(out, have, 1, outfd) != 1) {
          exit(EIO);
        }
        sz += have;
      } while (strm.avail_out == 0);

      /*
       * If zlib is done and we still have data in our buffer, that extra data
       * needs to be written out to file. It contains the key + padding +
       * original filesize.
       *
       * We use a buffer to strip the padding from the output before writing it
       * to the final FD (may be stdout that can't be rewind)
       */
      if (strm.avail_in > sizeof(uint32_t)) {
        /* We don't write the original file size since it is not stored for
         * uncompressed files either. Actually, we don't need it at all and
         * just support it for compatibility with earlier versions. */
        last_chunk_length = strm.avail_in - sizeof(uint32_t);
        memcpy(last_chunk, strm.next_in, last_chunk_length);
      }
    } else {
      /* Abort the loop on EOF when the file is not compressed */
      if (n < ZCHUNK)
        ret = Z_STREAM_END;

      if (n < BLOCK_SIZE + MAXKEYBYTES) {
        /* The remaining data is smaller than our buffer */
        if (n <= BLOCK_SIZE + MAXKEYBYTES - last_chunk_length) {
          // the last data is smaller than the remaining space in the buffer
          memcpy(last_chunk + last_chunk_length, in, n);
          last_chunk_length += n;
        } else {
          // not enough space in the buffer, we have to write down the first part to make some room
          int to_write = n - (BLOCK_SIZE + MAXKEYBYTES - last_chunk_length);
          if (fwrite(last_chunk, to_write, 1, outfd) != 1) {
            exit(EIO);
          }
          sz += to_write;
          memmove(last_chunk, last_chunk + to_write, last_chunk_length - to_write);
          memcpy(last_chunk + to_write, in, n);
          last_chunk_length = last_chunk_length - to_write + n; /* should be BLOCK_SIZE + MAXKEYBYTES */
        }
      } else {
        /* enough room in the comming data do flush our buffer from the previous loop */
        /* write the previous chunk */
        if (last_chunk_length > 0) {
          if (fwrite(last_chunk, last_chunk_length, 1, outfd) != 1) {
            exit(EIO);
          }
          sz += last_chunk_length;
        }
        /* write data except the last chunk we store in the buffer for the end/next loop */
        if (fwrite(in, n - BLOCK_SIZE - MAXKEYBYTES, 1, outfd) != 1) {
          exit(EIO);
        }
        sz += n - BLOCK_SIZE - MAXKEYBYTES;
        memcpy(last_chunk, in + n - BLOCK_SIZE - MAXKEYBYTES, BLOCK_SIZE + MAXKEYBYTES);
        last_chunk_length = BLOCK_SIZE + MAXKEYBYTES;

      }
    }
  } while (ret != Z_STREAM_END);

  /* Strip off the padding from the buffer */
  int cur_pos = last_chunk_length;
  do {
    cur_pos--;
    if (cur_pos < 0) {
      fprintf(stderr, "No non-NULL char in buffer\n");
      exit(EIO);
    }
  } while (last_chunk[cur_pos] == '\0');

  /* Something goes wrong, this should not be negative */
  if (cur_pos + 1 - MAXKEYBYTES < 0) {
    fprintf(stderr, "Missing data in buffer\n");
    exit(EIO);
  }

  /* write the last piece of data (without the KEY !)*/
  if (cur_pos + 1 - MAXKEYBYTES > 0) {
    if (fwrite(last_chunk, cur_pos + 1 - MAXKEYBYTES, 1, outfd) != 1) {
      exit(EIO);
    }
  }
  sz += cur_pos + 1 - MAXKEYBYTES;

  /* Verify that the stored key matches our key */
  if (memcmp(last_chunk + cur_pos + 1 - MAXKEYBYTES, mykey, MAXKEYBYTES) != 0) {
    memset(mykey, 0, strlen(mykey));
    free(mykey);
    return 0;
  }

  memset(mykey, 0, strlen(mykey));
  free(mykey);
  return sz;
}
