/*-*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*-

Emoticons Plugin for ChatZilla
A javascript plugin that allows ChatZilla to use Gaim emoticon themes.
Copyright (C) 2004  John J. Foerch <jjfoerch@earthlink.net>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/


const LF = "\n";


var State = {
	emoticons: new Object(),
	theme: {
		'Name=': null,
		'Author=': null,
		'Icon=': null,
		'Description=': null,
		'Sections': {},
		'Precedence': []
	},
	mungeMatch: '',
	pluginLocation: String(this.plugin.url).replace(/\/[^\/]*$/,'/'),
	defaultTheme: 'kosem_utz',
	localTheme: undefined,
	themeURLObject: null,
	remoteRequest: null,
	remoteRequestCallback: null,
	error: false,
}

function initPlugin(glob) {
	//make required variables
	plugin.id = 'emoticons';
	plugin.major = 0;  // Major version number.
	plugin.minor = 46;  // Minor version number.
	plugin.version = plugin.major + '.' + plugin.minor;
	plugin.description = 'Use Gaim emoticon themes in ChatZilla!';

	//register preferences
	var defaultURL = (State.pluginLocation + State.defaultTheme + '/');
	plugin.prefAry = [
		['emoticons.url', defaultURL,,urlSetter],
		];
	client.prefManager.addPrefs(plugin.prefAry);
	//add an Observer for my preference. needed to ensure
	//theme loading when pref is set to default.
	client.prefManager.addObserver({onPrefChanged: prefObserver});

	//Initialize State properties and read theme
	State.themeURLObject = client.prefs['emoticons.url'];
	importTheme();

	//register Commands
	plugin.cmdlist = [
		['emoticons',cmdEmoticons,CMD_CONSOLE,'[<url>]'],
	];
	client.commandManager.defineCommands(plugin.cmdlist);
	
	//register Commands Help
	client.commandManager.commands.emoticons.help = emoticonsHelp();
	
	display (GPLNotice(),MT_INFO);
}

function importTheme () {
	if (! plugin.enabled) { return; }
	var theme_content;
	
	//check for callback
	if (State.remoteRequestCallback) {
		//be sure to set remoteRequestCallback to false both
		//here and on request failure
		theme_content = State.remoteRequest.responseText;
		State.remoteRequestCallback = false;
	} else {
		//check URL scheme
		var re = /^file:/;
		var localScheme = re.test(client.prefs['emoticons.url']);
		if (localScheme) {
			State.localTheme = true;
			//local? try to load
			//create nsLocalFile object
			var localName = getFileFromURLSpec (State.themeURLObject);
			var f = new nsLocalFile(localName.path);
			f.append('theme');
			//does the file exist?
			if (! f.exists()) {
				//do error and return
				State.error = true;
				return;
			}
			f = fopen (f.path,'<');
			theme_content = f.read();
			f.close();
		} else {
			//remote? setup callback and make request
			State.localTheme = false;
			State.remoteRequestCallback = true;
			var uri = State.themeURLObject + ('theme');
			callAsync(uri, '', importTheme, themeReqFail);
			return;
		}
	}
	//check well formedness of file
	if (theme_content == null) {
		//display error message
		State.error = true;
		return;
	}
	
	var content_lines = theme_content.split (/\n+/);
	
	var section_regexp = /^\[(.+)\]$/;
	//match:
	//ignore possible ^!\s+
	//capture file name
	//capture all remaining terms as emoticons
	var re = /^(?:\!\s)?(\S+)\s+(?:(\S+)\s?){1,}$/;
	
	State.mungeMatch = '';
	State.emoticons = new Object();
	State.theme.Precedence = new Array();
	State.theme.Sections = new Object();
	
	var current_section = null;
	
	content_loop :
	for (var i in content_lines) {
		if (content_lines[i].substr(0,1) == '#') { continue content_loop; }
		for (var j in State.theme) {
			if (content_lines[i].substr(0,j.length) == j) {
				State.theme[j] = content_lines[i].substr(j.length);
				continue content_loop;
			}
		}
		//check for Section
		if (section_regexp.test(content_lines[i])) {
			current_section =
				content_lines[i].substring(1,content_lines[i].length-2);
			makeSection(current_section);
		}
		//check for entry
		if (re.test(content_lines[i])) {
			if (current_section == null) {
				current_section = 'Default';
				makeSection (current_section);
			}
			var ar = content_lines[i].split (/\s+/);
			var adder = 0;
			if (ar[0] == '!') { adder = 1; }
			for (var a = 1 + adder; a < ar.length; a++) {
				if (ar[a] == null) { next; }
				//add emoticon entries to current section
				State.theme.Sections[current_section][ar[a]] = ar[0 + adder];
				///old method:
				//State.emoticons[ar[a]] = ar[0 + adder];
			}
		}
	}
	
	//at this point, we have a data structure in State.theme.Sections
	//containing all of our emoticons.
	//iterate through that structure to put emoticons into a global list.
	//the reason we store the emoticons in a data structure instead of
	//immediately dumping them into one giant list is that themes often
	//have duplicate entries in different sections.  For now, the order
	//of the sections in the theme will be their precedence.  In a future
	//version of this plugin, perhaps we can let the user choose the
	//precedence of the sections.
	for (var p in State.theme.Precedence) {
		var section = State.theme.Precedence[p];
		for (var s in State.theme.Sections[section]) {
			State.emoticons[s] = State.theme.Sections[section][s];
		}
	}
	
	
	//sort emoticons keys by length (longest first)
	//and build mungeMatch string
	var sorted_emoticons = keys(State.emoticons);
	sorted_emoticons.sort(function (a,b) {return b.length - a.length;});
	for (v in sorted_emoticons) {
		if (State.mungeMatch != '') { State.mungeMatch += '|'; }
		State.mungeMatch += regexEscape(sorted_emoticons[v]);
	}
	
	State.mungeMatch = '(?:^|\\s)(' + State.mungeMatch + ')(?:\\s|$)';

	//register Munger
	//if ('face' in client.munger.entries) {
	//	client.munger.delRule('face');
	//}
	//munger.addRule will change an existing munger entry,
	//so deletion is not necessary
	client.munger.addRule('face', RegExp(State.mungeMatch),mungeEmoticons);
	
	displayThemeInfo();
		
	function regexEscape (s) {
		var re = /\\|\^|\$|\*|\+|\?|\(|\)|\||\{|\}|\[|\]|\./g;
		s=s.replace(re,"\\$&");
		return s;
	}

	function makeSection (section) {	
		if (section in State.theme.Sections) { return; }
		State.theme.Sections[section] = new Object();
		State.theme.Precedence.unshift(section);
	}

}


function mungeEmoticons (matchText, containerTag){

	//test: is the plugin enabled?
	//does the plugin have a valid theme?
	var enabled = plugin.enabled;
	var havetheme = (State.remoteRequestCallback === true ? false : true);
	var validtheme = (! State.error);

	if (enabled && havetheme && validtheme) {
		var imgsrc = client.prefs['emoticons.url'] +
			State.emoticons[matchText];
		//is the theme local?
		if (State.localTheme === true) {
			//test for the existence of the image
			var f = getFileFromURLSpec(imgsrc);
			if (f.exists()) {
				//exists: create image tag
				emoticonImage(imgsrc);
			} else {
				//does not exist: display text emoticon
				emoticonText();
			}
		} else {
			//create image tag in faith
			emoticonImage(imgsrc);
		}
	} else {
		//display text emoticon
		emoticonText();
	}

	function emoticonImage (imgsrc) {
		var ns = 'http://www.w3.org/1999/xhtml';
		var anchor = document.createElementNS(ns, 'html:img');
		anchor.setAttribute('class', 'emoticon');
		anchor.setAttribute('src', imgsrc);
		anchor.setAttribute('alt', matchText);
		anchor.setAttribute('title', matchText);
		containerTag.appendChild(anchor);
	}
	
	function emoticonText () {
		containerTag.appendChild(document.createTextNode(matchText));
	}
}


//On the subtle relationship between the prefSetter and the Observer.
//The prefSetter's job is to validate data to ensure that the pref
//will always have a legal value.  The Observer is executed after
//the pref has been set, and its value has passed validation.  The
//prefSetter will not be called when the pref is reset to default.
//The Observer will be called in that instance.
function prefObserver (prefName, newValue, oldValue) {
	if (prefName != 'emoticons.url') { return; }
	State.themeURLObject = newValue;
	importTheme();
}


function urlSetter (value) {
	//test for known scheme
	var u;
	var re = /^file:|http:|ftp:/;
	var isKnownScheme = re.test(value);
	if (isKnownScheme) {
		//known: make URL and set pref to string URL
		u = value;
		State.error = false;
	} else {
		//unknown: try as local file
		var isLocal = false;
		try {
			var f = nsLocalFile (value);
			if (f.exists()) {
				isLocal = true;
				State.error = false;
			} else {
				//error message
				State.error = true;
				return;
			}
		} catch (e) {
			State.error = true;
			return;
		}
		u = getURLSpecFromFile (value);
	}

	if (u.substr(u.length - 1) != '/') { u += '/'; }
	
	return client.prefManager.setPref('emoticons.url', u);//is this important?
}


function themeReqFail () {
	State.error = true;
	State.remoteRequestCallback = false;
	display ('emoticons: request of remote theme failed.',MT_ERROR);
}

//asynchronous file request as explained by fur,
//who also suggests that this part of the plugin
//be dubbed "remoticons". I note, that is a recursive
//portmanteau word.
//multi: multipart form encoding.
//	don't know what this is.
//	advised to pass an empty string.
function callAsync(uri, multi, success, failure) {
	State.remoteRequest = new XMLHttpRequest();
	State.remoteRequest.multipart = multi;
	State.remoteRequest.open('GET', uri, true);
	State.remoteRequest.onload=success;
	State.remoteRequest.onerror=failure;
	State.remoteRequest.send('');
}


////////////////////////////////////////
// COMMANDS
//
//enablePlugin is called by /enable-plugin emoticons
function enablePlugin () {
	plugin.enabled = true;
	display ("emoticons plugin enabled.",MT_INFO);
	importTheme();
}

//disablePlugin in called by /disable-plugin emoticons
function disablePlugin () {
	plugin.enabled = false;
	display ("emoticons plugin disabled.",MT_INFO);
}

function cmdEmoticons (e) {
	if (! 'url' in e || e.url === null) {
		displayThemeInfo();
		displayEmoticons();
		return;
	}
	urlSetter(e.url);
	display ("emoticons: new theme " + client.prefs['emoticons.url'],MT_INFO);
	//enable plugin if disabled
}

function displayEmoticons () {
	client.munger.enabled = false;

	var tr = makeMessageRow();
	tr.setAttribute ('class','msg');
	tr.setAttribute ('msg-type',MT_INFO);

	var ns = 'http://www.w3.org/1999/xhtml';
	var table1 = document.createElementNS (ns,'html:table');
	tr.firstChild.appendChild (table1);

	var emoticon_chart = new Object();
	for (var i in State.emoticons) {
		//i = symbol
		//State.emoticons[i] = image
		var imgsrc = client.prefs['emoticons.url'] + State.emoticons[i];
		var noimage = false;
		//is theme local?
		if(State.localTheme) {
			var f = getFileFromURLSpec(imgsrc);
			if (! f.exists()) {
				noimage = true;
			}
		}
		if (! noimage) {
			//put an entry in emoticon_chart or append to entry
			if (!(State.emoticons[i] in emoticon_chart)) {
				emoticon_chart[State.emoticons[i]] = '';
			}
			emoticon_chart[State.emoticons[i]] +=
				i + "RRRRRRR".replace(/R/g,String.fromCharCode(160));
		}
	}
	
	
	//insert rows
	for (i in emoticon_chart) {
		//var q = emoticon_chart[i].join("\t");
		var q = emoticon_chart[i];
		var imgsrc = client.prefs['emoticons.url'] + i;
		var img = document.createElementNS(ns,'html:img');
		img.setAttribute('src',imgsrc);
		
		var ttr = document.createElementNS(ns,'html:tr');
		ttr.appendChild	(document.createElementNS(ns,'html:td'));
		ttr.appendChild	(document.createElementNS(ns,'html:td'));
		ttr.firstChild.appendChild(img);
		ttr.firstChild.setAttribute('width','20%');
		ttr.firstChild.nextSibling.appendChild(document.createTextNode(q));
		table1.appendChild(ttr);
	}
	
	
	addHistory (client.currentObject, tr, false);
	
	client.munger.enabled = true;
}

function displayThemeInfo() {
	var ns = 'http://www.w3.org/1999/xhtml';
	var icon = client.prefs['emoticons.url'] + State.theme['Icon='];
	var noimage = false;
	var img = null;
	//is theme local?
	if(State.localTheme) {
		var f = getFileFromURLSpec(icon);
		if (! f.exists()) {
			noimage = true;
		}
	}
	if (! noimage) {
		img = document.createElementNS (ns,'html:img');
		img.setAttribute ('src',icon);
	}
	
	var tr = makeMessageRow();
	tr.setAttribute ('class','msg');
	tr.setAttribute ('msg-type',MT_INFO);
	var table1 = makeTable (1,2);
	tr.firstChild.appendChild (table1);
	table1.firstChild.firstChild.setAttribute('width','20%');
	if (img != null) {
		table1.firstChild.firstChild.appendChild(img);
	}
	var table2 = makeTable (3,1);
	table1.firstChild.firstChild.nextSibling.appendChild(table2);
	
	table2.firstChild.firstChild.appendChild
		(document.createTextNode("Name: " + State.theme['Name=']));
	table2.firstChild.nextSibling.firstChild.appendChild
		(document.createTextNode("Author: " + State.theme['Author=']));
	table2.firstChild.nextSibling.nextSibling.firstChild.appendChild
		(document.createTextNode("Description: " + 
								 State.theme['Description=']));

	addHistory (client.currentObject, tr, false);
}

function makeMessageRow() {
	var ns = 'http://www.w3.org/1999/xhtml';
	var tr = document.createElementNS (ns,'html:tr');
	tr.appendChild(document.createElementNS (ns,'html:td'));
	tr.firstChild.setAttribute ('colspan','3');
	return tr;
}

function makeTable (rows, cols) {
	var ns = 'http://www.w3.org/1999/xhtml';
	var table = document.createElementNS (ns,'html:table');
	table.setAttribute ('rows',rows);
	table.setAttribute ('cols',cols);
	for (var j = 0; j < rows; j ++) {
		var tr = document.createElementNS (ns,'html:tr');
		table.appendChild(tr);
		for (var k = 0; k < cols; k ++) {
			var td = document.createElementNS (ns,'html:td');
			tr.appendChild(td);
		}
	}
	return table;
}


function emoticonsHelp() {
	return "To display the current theme, do /emoticons"+LF+
	"To set the theme, do /emoticons <url>"+LF+
	"where <url> is a directory containing a Gaim"+LF+
	"emoticon theme. Obtain themes from gaim.sourceforge.net.";
}


function GPLNotice() {
    return "Emoticons Plugin version "+ plugin.version +
		", Copyright (C) 2004 John J Foerch"+LF+
		"Emoticons Plugin comes with ABSOLUTELY NO WARRANTY."+LF+
		"This is free software, and you are welcome to redistribute it"+LF+
		"under certain conditions; read the included file LICENSE "+
		"for details.";
}
