#!/usr/bin/python2.2
# 
# Copyright 2002, 2003 Zuza Software Foundation
# 
# This file is part of mozpotools.
#
# mozpotools is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
# 
# mozpotools is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with mozpotools; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

"""classes that hold elements of .po files (poelement) or entire files (pofile)
gettext-style .po (or .pot) files are used in translations for KDE et al (see kbabel)
FIXME: add simple test which reads in a file and writes it out again"""

from __future__ import generators
import extract

class poelement:
  # sourcecomments = []     #   #: sourcefile.xxx:35
  # typecomments = []       #   #, fuzzy
  # visiblecomments = []    #   #_ note to translator
  # othercomments = []      #   # this is another comment
  # msgid = []
  # msgstr = []
  # isheader = 0            # indicates if this is the header of the file
  
  def __init__(self,isheader=0):
    self.sourcecomments = []
    self.typecomments = []
    self.visiblecomments = []
    self.othercomments = []
    self.msgidcomments = []
    self.msgid = []
    self.msgstr = []
    self.isheader = isheader

  def isblank(self):
    return (len("".join(self.msgid).strip()) == 0) and (len("".join(self.msgid).strip()) == 0)

  def fromlines(self,lines):
    inmsgid = 0
    inmsgstr = 0
    linesprocessed = 0
    for line in lines:
      linesprocessed += 1
      if line[0] == '#':
        if inmsgstr:
          # if we're already in the message string, this is from the next element
          break
        if line[1] == ':':
          self.sourcecomments.append(line)
        elif line[1] == ',':
          self.typecomments.append(line)
        elif line[1] == '_':
          self.visiblecomments.append(line)
        else:
          self.othercomments.append(line)
      else:
        if line[:5] == 'msgid':
          inmsgid = 1
          inmsgstr = 0
        elif line[:6] == 'msgstr':
          inmsgstr = 1
          inmsgid = 0
      str = extract.extractstr(line)
      if not str is None:
        if inmsgid:
          # self.othercomments.append("# msgid=["+repr(str)+","+repr(str[:2])+"]\n")
          if str.find('_:') != -1:
            self.msgidcomments.append(str)
          else:
            self.msgid.append(str)
        elif inmsgstr:
          self.msgstr.append(str)
    return linesprocessed

  def tolines(self):
    for sourcecomment in self.sourcecomments:
      yield sourcecomment
    for typecomment in self.typecomments:
      yield typecomment
    for visiblecomment in self.visiblecomments:
      yield visiblecomment
    for othercomment in self.othercomments:
      yield othercomment
    # if there's no msgid don't do msgid and string, unless we're the header
    if (len(self.msgid) == 0) or ((len(self.msgid) == 1) and (self.msgid[0] == '""')):
      if not self.isheader:
        return
    msgidstr = "msgid "
    msgidstartline = 0
    if len(self.msgid) > 0 and len(self.msgidcomments) == 0:
      msgidstr += self.msgid[0]
      msgidstartline = 1
    elif len(self.msgidcomments) > 0:
      # comments first, no blank leader line needed
      for msgidcomment in self.msgidcomments:
        msgidstr += msgidcomment # + '\n'
      msgidstr = extract.rstripeol(msgidstr)
    else:
      msgidstr += '""'
    yield msgidstr + '\n'
    # add the rest
    for msgidline in self.msgid[msgidstartline:]:
      yield msgidline + '\n'
    msgstrstr = "msgstr "
    if len(self.msgstr) > 0:
      msgstrstr += self.msgstr[0]
    else:
      msgstrstr = '""'
    yield msgstrstr + '\n'
    # add the rest
    for msgstrline in self.msgstr[1:]:
      yield msgstrline + '\n'

class pofile:
  def __init__(self):
    self.poelements = []

  def fromlines(self,lines):
    start = 0
    end = 0
    # make only the first one the header
    isheader = 1
    linesprocessed = 0
    while end <= len(lines):
      if (end == len(lines)) or (lines[end] == '\n'):   # end of lines or just a carriage return
        finished = 0
        while not finished:
          newpe = poelement(isheader)
          linesprocessed = newpe.fromlines(lines[start:end])
          start += linesprocessed
          if linesprocessed > 1:
            self.poelements.append(newpe)
            isheader = 0
          else:
            finished = 1
      end = end+1

  def removeblanks(self):
    """remove any poelements which say they are blank"""
    self.poelements = filter(poelement.isblank, self.poelements)

  def tolines(self):
    lines = []
    for pe in self.poelements:
      pelines = pe.tolines()
      lines.extend(pelines)
      # add a line break
      lines.append('\n')
    return lines

if __name__ == '__main__':
  import sys
  pf = pofile()
  pf.fromlines(sys.stdin.readlines())
  sys.stdout.writelines(pf.tolines())

