/* runtex -- run (pdf)(e)(La)TeX and various utils to process given document
	and related files
	Copyright (c) 2005 by Bernd Becker.
	This program is public domain. Use at your own risk ! */

#include "runtex.h"
#include "io_stuff.h"
#include "strstuff.h"
#include "config.h"

/* free up all data used by program */
void free_all_data(void)
{
	int i;

	if (runtex_data.config_file) fclose(runtex_data.config_file);
	free_if(runtex_data.config_buffer);
	for(i=0; i<NUMBER_RUNTEX_TEX_VARIANTS; i++)
	{
		free_if(runtex_data.tex[i]);
		free_if(runtex_data.tex_opt[i]);
	}
/* file names of utilities */
	for(i=0; i<NUMBER_RUNTEX_UTILS; i++)
	{
		free_if(runtex_data.utils[i]);
		free_if(runtex_data.utils_opt[i]);
	}
	free_if(runtex_data.local_texmf);
	free_if(runtex_data.local_packages);
	free_if(runtex_data.local_doc);
	free_if(runtex_data.arg_name);
	free_if(runtex_data.job_name);
	free_if(runtex_data.job_dir);
}

/* displays message and error, frees up all data
	args: message = format string
			... = variables used */
void exit_on_error(const char* message, ...)
{
	va_list args;

	fflush(stdout);
	free_all_data();
	fprintf(stderr,"%s: ",PROD_NAME);
	va_start(args,message);
	vfprintf(stderr,message,args);
	va_end(args);
	if (errno) fprintf(stderr,"(errno: %s)",strerror(errno));
	putc('\n',stderr);
	fflush(stderr);
	exit(EXIT_FAILURE);
}
/* print version info */
void print_version(void)
{
	printf("%s %s (beta)\tCopyright (c) 2005 Bernd Becker\n\
This program(me) is public domain. Use at your own risk!\n",
		PROD_NAME,PROD_VERSION);
}
/* explain switches */
void usage(void)
{
	print_version();
	printf("\n\
  Tries to guess the number of times the document needs to be processed\n\
with TeX or a variant, and if other programs have to be executed after\n\
the first TeX run.\n\
It can also be used to install packages, but it may not work correctly\n\
all the time.\n");
	printf("\nUsage: %s [OPTIONS] FILE\n",PROD_NAME);
	printf("Options:\n\
  -d      delete temporary files created by packages/utilities\n\
  -h      print this help\n\
  -i      try to install a package and create its documentation\n\
  -l      print performed actions\n\
  -n      do not execute utilities\n\
  -r 1-4  don't guess but process document this number of times\n\
  -t STR  use TeX variant instead of TeX, where STR may be one of:\n\
\te=eTeX\n\
\tl=LaTeX\n\
\tp=pdfTeX\n\
\tel=eLaTeX\n\
\tpe=pdfeTeX\n\
\tpl=pdfLaTeX\n\
\tpel=pdfeLaTeX\n\
  -v      print program version\n\
\nReport bugs to munin@munin.inka.de.\n");
	exit(EXIT_FAILURE);
}

/* string-related functions */
#include"strstuff.c"

/* .ini-related functions */
#include"config.c"

/* files-related functions */
#include"io_stuff.c"

/* run TeX (variant), check for created files, utils to run,
	if TeX (variant) needs to be run again */
void process_tex_document(char* arg_name)
{
	char* tex_name=runtex_data.tex[runtex_data.tex_variant];

	if (runtex_data.print_actions)
		printf("Processing %s with %s\n",arg_name,tex_name);
	do
	{
		if (!run_external_program(0,runtex_data.tex_variant,arg_name))
			break;
	/* was # runs set by user - decrease counter */
		if (runtex_data.tex_runs) runtex_data.tex_runs--;
		process_document_related_files(runtex_data.job_dir);
		while (1)
		{
			if (!runtex_data.tex_runs)
			{/* only delete temporary files if everything else went fine */
				if (!runtex_data.keep_temps)
					delete_temporary_tex_files(runtex_data.job_dir);
				return;
			}
			if (!run_external_program(0,runtex_data.tex_variant,arg_name))
				break;
			runtex_data.tex_runs--;
		}
	} while (0);
	exit_on_error("\n\n%s exited abnormally.",tex_name);
}
/* we create a dir in the temp dir into which to move the .ins/.dtx file,
	execute TeX on the .ins, move all generated files to
	local_texmf/local_packages and then run TeX on .dtx as if it was
	a .tex file, move resulting .dvi/pdf to local_texmf/local_doc,
	and delete all files and directory */
int install_tex_package(void)
{
	char buffer[MAX_PATH];
	char src_name[MAX_PATH];
	char dest_name[MAX_PATH];
	char install_dir[MAX_PATH];
	char* tmp_src_name;
	char* tmp_dest_name;
	char* tmp_str;
	char* temp_dir;
#ifdef __WIN32__
	struct _finddata_t dir_list;
#else
	struct ffblk dir_list;
#endif
	int find_file_handle;
	int find_error;
	int str_size;

	if (runtex_data.print_actions)
		printf("Trying to install package %s\n",runtex_data.job_name);
	temp_dir=create_temp_dir();
	do
	{
		if (!temp_dir) break;
		tmp_str=install_dir;
		tmp_str+=sprintf(install_dir,"%s%s%s",runtex_data.local_texmf,
			runtex_data.local_packages,runtex_data.job_name);
	/* we overwrite existing package as if it was an update */
		if ((mkdir(buffer)) && (errno!=EEXIST)) break;
		*tmp_str=NATIVE_DIRSEP;
	/* create source name */
		tmp_src_name=src_name;
		tmp_src_name+=sprintf(src_name,"%s%s.",runtex_data.job_dir,runtex_data.job_name);
	/* create destination name */
		tmp_dest_name=dest_name;
		tmp_dest_name+=sprintf(dest_name,"%s%s.",temp_dir,runtex_data.job_name);
		memcpy(tmp_src_name,"dtx",4);
		memcpy(tmp_dest_name,"dtx",4);
	/* copy .ins & .dtx file into dest dir */
		if (!copy_file_to(src_name,dest_name)) break;
		memcpy(tmp_src_name,"ins",4);
		memcpy(tmp_dest_name,"ins",4);
		if (!copy_file_to(src_name,dest_name)) break;
	/* re-set dir var because we change directory */
		free_pntr(runtex_data.job_dir)
		str_size=strlen(temp_dir);
	/* add dirsep char to size */
		new_string(runtex_data.job_dir,str_size+1);
		if (!runtex_data.job_dir) break;
		memcpy(runtex_data.job_dir,temp_dir,str_size);
		free_pntr(temp_dir)
	/* add dirsep and NULL pad to string */
		tmp_str=runtex_data.job_dir+str_size;
		*tmp_str=NATIVE_DIRSEP;
		tmp_str++;
		*tmp_str=0;
#ifdef __WIN32__
		_chdir(runtex_data.job_dir);
#else
		chdir(runtex_data.job_dir);
#endif
	/* execute TeX on .ins to generate files */
		tmp_str=buffer;
		tmp_str+=sprintf(buffer,"%s.",runtex_data.job_name);
		memcpy(tmp_str,"ins",4);
		if (!run_external_program(0,runtex_data.tex_variant,buffer))
			break;
	/* always delete temporaries */
		runtex_data.keep_temps=0;
	/* may have to add "-include-dir=runtex_data.job_dir" to the cmd-line */
	/* *.drv exist - process them instead of .dtx */
#ifdef __WIN32__
		find_file_handle=_findfirst("*.drv",&dir_list);
		if (find_file_handle<0)
#else
		find_file_handle=findfirst(buffer,&dir_list,FA_ARCH);
		if (find_file_handle)
#endif
		{/* not found - process \jobname.dtx */
			memcpy(tmp_str,"dtx",3);
			process_tex_document(buffer);
		}
		else
		{/* save original jobname */
			free(runtex_data.arg_name);
			runtex_data.arg_name=runtex_data.job_name;
			while (1)
			{/* .drv may be different from \jobname */
				runtex_data.job_name=get_file_name(dir_list.name);
				if (!runtex_data.job_name) break;
			/* process as TeX file */
				process_tex_document(dir_list.name);
				free(runtex_data.job_name);
#ifdef __WIN32__
				find_error=_findnext(find_file_handle,&dir_list);
#else
				find_error=findnext(&dir_list);
#endif
				if (find_error) break;
			}
#ifdef __WIN32__
			_findclose(find_file_handle);
#endif
			if (!runtex_data.job_name) break;
		/* get back original jobname */
			runtex_data.job_name=runtex_data.arg_name;
			runtex_data.arg_name=NULL;
		}
		sprintf(dest_name,"%s%s",runtex_data.local_texmf,runtex_data.local_doc);
	/* if dvi not found move pdfs */
		if (!move_files(NULL,"*.dvi",dest_name))
			(void)move_files(NULL,"*.pdf",dest_name);
	/* delete .dtx .ins, *.drv */
		tmp_str=buffer;
		tmp_str+=sprintf(buffer,"%s.",runtex_data.job_name);
		memcpy(tmp_str,"ins",4);
		(void)unlink(buffer);
		memcpy(tmp_str,"dtx",3);
		(void)unlink(buffer);
		(void)delete_files("*.drv");
	/* move rest of files to package dir */
		move_files(NULL,"*.*",install_dir);
	/* NULL file name since we want dir name only */
		tmp_src_name=get_file_name_pointer(src_name);
		*tmp_src_name=0;
#ifdef __WIN32__
		_chdir(src_name);
#else
		chdir(src_name);
#endif
	/* delete temporary dir */
		rmdir(runtex_data.job_dir);
	/* delete original files as well */
	/*	memcpy(tmp_str,"dtx",3);*/
		(void)unlink(buffer);
		memcpy(tmp_str,"ins",3);
		(void)unlink(buffer);
		if (runtex_data.print_actions)
			printf("It seemed to have worked !?\n");
	/* refresh file database */
		if (!run_external_program(1,_REFRESH_FILE_DB,NULL))
			printf("Could not refresh file database\n");
		return 1;
	} while (0);
	if (temp_dir)
	{
		printf("\nCheck '%s' for temporary files that may have been left\n",
			temp_dir);
		free(temp_dir);
	}
	return 0;
}

void set_tex_variant(char* tex_opt)
{
	int opt_size=strlen(tex_opt);

	switch (opt_size)
	{
		case 1:
			if (*tex_opt=='l') runtex_data.tex_variant=_LATEX;
			else if (*tex_opt=='e') runtex_data.tex_variant=_ETEX;
			else if (*tex_opt=='p') runtex_data.tex_variant=_PDFTEX;
		break;
		case 2:
			if (*tex_opt=='e')
			{
				if (*(tex_opt+1)=='l') runtex_data.tex_variant=_ELATEX;
			}
			else if (*tex_opt=='p')
			{
				tex_opt++;
				if (*tex_opt=='e') runtex_data.tex_variant=_PDFETEX;
				else if (*tex_opt=='l') runtex_data.tex_variant=_PDFLATEX;
			}
		break;
		case 3:
			if ((*tex_opt=='p') && (*(tex_opt+1)=='e') && (*(tex_opt+2)=='l'))
				runtex_data.tex_variant=_PDFELATEX;
		break;
	}
	if (!runtex_data.tex_variant)
		exit_on_error("TeX variant switch used with unknown combination.");
}

int main(int argc, char* const argv[])
{
	int option_char;
	int ret_val;
	char* tmp_str;

	set_runtex_defaults();
/* get arguments */
	while (option_char=getopt(argc,argv,"dhilnr:t:v"),
	option_char!=EOF)
	{
		switch (option_char)
		{
			case 'd':
				runtex_data.keep_temps=0;
			break;
			case 'h':
				usage();
			case 'i':
				runtex_data.install=1;
			break;
			case 'l':
				runtex_data.print_actions=1;
			break;
			case 'n':
				runtex_data.exec_utils=0;
			break;
			case 'r':
				ret_val=atoi(optarg);
			/* number is between 1 and 4 set value */
				if ((ret_val>0) && (ret_val<5))
					runtex_data.tex_runs=ret_val;
			break;
			case 't':
				set_tex_variant(optarg);
			break;
			case 'v':
				print_version();
				exit(EXIT_SUCCESS);
			break;
			default:
				usage();
		}
	}
/* was file passed ? */
	if (optind+1==argc)
	{/* yes - check for suffix and set dir of passed */
		tmp_str=argv[optind];
		do
		{/* to save the 'many' bytes needed to do this everytime */
			if (!get_file_suffix_pointer(tmp_str))
			{/* no suffix given */
				if (!runtex_data.install)
				{/* process document - try to find .tex*/
					runtex_data.arg_name=create_full_file_name(NULL,
						tmp_str,".tex");
					runtex_data.job_dir=get_true_path(runtex_data.job_name);
					if (!runtex_data.job_dir)
					{/* now try .ltx */
						free(runtex_data.arg_name);
						runtex_data.arg_name=create_full_file_name(NULL,
							tmp_str,".ltx");
					}
				/* found file - go on */
					else break;
				}
				else
				{/* install package */
					runtex_data.arg_name=create_full_file_name(NULL,
						tmp_str,".ins");
				}
			}
		/* suffix given - need to save the argument for the TeX runs */
			else runtex_data.arg_name=create_full_file_name(NULL,
				tmp_str,NULL);
			runtex_data.job_dir=get_true_path(runtex_data.arg_name);
		/* no file of that name - error */
			if (!runtex_data.job_dir)
				exit_on_error("Could not resolve path for file %s.",tmp_str);
		} while (0);
	}
/* no - error */
	else usage();
	while (1)
	{/* get file name without path and suffix */
		runtex_data.job_name=get_file_name(runtex_data.job_dir);
		if (runtex_data.job_name)
		{/* get directory name */
			tmp_str=get_dir_name(runtex_data.job_dir);
			if (tmp_str)
			{/* that the full path only contains the dir should not happen */
				free(runtex_data.job_dir);
				runtex_data.job_dir=tmp_str;
				break;
			}
		}
		exit_on_error(
"Could not dissect the full path of %s into its parts.",argv[optind]);
	}
	read_runtex_inis(argv[0]);
	if (!runtex_data.tex[runtex_data.tex_variant])
		exit_on_error(
"Requested TeX variant's executable not given in any runtex.ini.");
/* run tex, analyze files in directory of job, run utils,
	run tex again/a number of times */
	if (!runtex_data.install) process_tex_document(runtex_data.arg_name);
	else install_tex_package();
	free_all_data();
	exit(EXIT_SUCCESS);
}

