/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.brooklyn.core.test.entity;

import static com.google.common.base.Preconditions.checkNotNull;

import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicInteger;

import org.apache.brooklyn.api.entity.Entity;
import org.apache.brooklyn.api.entity.EntitySpec;
import org.apache.brooklyn.api.location.Location;
import org.apache.brooklyn.core.entity.AbstractEntity;
import org.apache.brooklyn.core.entity.lifecycle.Lifecycle;
import org.apache.brooklyn.core.entity.lifecycle.ServiceStateLogic;
import org.apache.brooklyn.core.feed.ConfigToAttributes;
import org.apache.brooklyn.util.collections.MutableMap;
import org.apache.brooklyn.util.core.flags.SetFromFlag;
import org.apache.brooklyn.util.time.Duration;
import org.apache.brooklyn.util.time.Time;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.collect.Lists;

/**
 * Mock entity for testing.
 */
public class TestEntityImpl extends AbstractEntity implements TestEntity {
    private static final Logger LOG = LoggerFactory.getLogger(TestEntityImpl.class);

    @SetFromFlag
    private String myField;
    
    @SetFromFlag("myField2Alias")
    private String myField2;
    
    protected int sequenceValue = 0;
    protected AtomicInteger counter = new AtomicInteger(0);
    protected Map<?,?> constructorProperties;
    protected Map<?,?> configureProperties;
    protected int configureCount;
    protected int configureDuringConstructionCount;
    protected List<String> callHistory = Collections.synchronizedList(Lists.<String>newArrayList());

    public TestEntityImpl() {
        super();
    }
    public TestEntityImpl(Entity parent) {
        this(MutableMap.of(), parent);
    }
    public TestEntityImpl(Map properties, Entity parent) {
        super(properties, parent);
        this.constructorProperties = properties;
        this.configureDuringConstructionCount = configureCount;
    }
    
    @Override
    public AbstractEntity configure(Map flags) {
        this.configureProperties = flags;
        configureCount++;
        return super.configure(flags);
    }
    
    @Override // made public for testing
    public boolean isLegacyConstruction() {
        return super.isLegacyConstruction();
    }
    
    @Override
    public void myEffector() {
        if (LOG.isTraceEnabled()) LOG.trace("In myEffector for {}", this);
        callHistory.add("myEffector");
    }
    
    @Override
    public Object identityEffector(Object arg) {
        if (LOG.isTraceEnabled()) LOG.trace("In identityEffector for {}", this);
        callHistory.add("identityEffector");
        return checkNotNull(arg, "arg");
    }
    
    @Override
    public void sleepEffector(Duration duration) {
        if (LOG.isTraceEnabled()) LOG.trace("In sleepEffector for {}", this);
        callHistory.add("sleepEffector");
        Time.sleep(duration);
    }

    @Override
    public String getMyField() {
        return myField;
    }
    
    @Override
    public String getMyField2() {
        return myField2;
    }

    @Override
    public AtomicInteger getCounter() {
        return counter;
    }
    
    @Override
    public int getCount() {
        return counter.get();
    }

    @Override
    public Map<?,?> getConstructorProperties() {
        return constructorProperties;
    }
    
    @Override
    public Map<?,?> getConfigureProperties() {
        return configureProperties;
    }
    
    @Override
    public int getConfigureCount() {
        return configureCount;
    }

    @Override
    public int getConfigureDuringConstructionCount() {
        return configureDuringConstructionCount;
    }

    @Override
    public synchronized int getSequenceValue() {
        return sequenceValue;
    }

    @Override
    public synchronized void setSequenceValue(int value) {
        sequenceValue = value;
        sensors().set(SEQUENCE, value);
    }

    @Override
    public void start(Collection<? extends Location> locs) {
        ConfigToAttributes.apply(this);
        
        LOG.trace("Starting {}", this);
        callHistory.add("start");
        ServiceStateLogic.setExpectedState(this, Lifecycle.STARTING);
        counter.incrementAndGet();
        addLocations(locs);
        sensors().set(SERVICE_UP, true);
        ServiceStateLogic.setExpectedState(this, Lifecycle.RUNNING);
    }

    @Override
    public void stop() { 
        LOG.trace("Stopping {}", this);
        callHistory.add("stop");
        ServiceStateLogic.setExpectedState(this, Lifecycle.STOPPING);
        counter.decrementAndGet();
        sensors().set(SERVICE_UP, false);
        ServiceStateLogic.setExpectedState(this, Lifecycle.STOPPED);
    }

    @Override
    public void restart() {
        LOG.trace("Restarting {}", this);
        callHistory.add("restart");
    }
    
    /**
     * TODO Rename to addChild
     */
    @Override
    public <T extends Entity> T createChild(EntitySpec<T> spec) {
        return addChild(spec);
    }

    @Override
    public <T extends Entity> T createAndManageChild(EntitySpec<T> spec) {
        if (!getManagementSupport().isDeployed()) throw new IllegalStateException("Entity "+this+" not managed");
        T child = createChild(spec);
        return child;
    }

    @Override
    public Entity createAndManageChildFromConfig() {
        return createAndManageChild(checkNotNull(getConfig(CHILD_SPEC), "childSpec"));
    }

//    @Override
//    public String toString() {
//        String id = getId();
//        return getEntityType().getSimpleName()+"["+id.substring(Math.max(0, id.length()-8))+"]";
//    }

    @Override
    public List<String> getCallHistory() {
        return callHistory;
    }
    
    @Override
    public void clearCallHistory() {
        callHistory.clear();
    }
    
    public static class TestEntityWithoutEnrichers extends TestEntityImpl {
        @Override
        protected void initEnrichers() {}
    }
}
