// Licensed to the Apache Software Foundation (ASF) under one
// or more contributor license agreements.  See the NOTICE file
// distributed with this work for additional information
// regarding copyright ownership.  The ASF licenses this file
// to you under the Apache License, Version 2.0 (the
// "License"); you may not use this file except in compliance
// with the License.  You may obtain a copy of the License at
//
//   http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing,
// software distributed under the License is distributed on an
// "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
// KIND, either express or implied.  See the License for the
// specific language governing permissions and limitations
// under the License.

package org.apache.doris.nereids.trees.expressions.functions.agg;

import org.apache.doris.catalog.FunctionSignature;
import org.apache.doris.nereids.exceptions.AnalysisException;
import org.apache.doris.nereids.trees.expressions.Expression;
import org.apache.doris.nereids.trees.expressions.functions.ExplicitlyCastableSignature;
import org.apache.doris.nereids.trees.expressions.visitor.ExpressionVisitor;
import org.apache.doris.nereids.types.StringType;
import org.apache.doris.nereids.types.coercion.AnyDataType;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;

import java.util.List;

/**
 * AggregateFunction 'approx_top_k'. This class is generated by GenerateFunction.
 */
public class ApproxTopK extends NullableAggregateFunction
        implements ExplicitlyCastableSignature {

    public static final List<FunctionSignature> SIGNATURES = ImmutableList.of(
            FunctionSignature.ret(StringType.INSTANCE)
                    .varArgs(AnyDataType.INSTANCE_WITHOUT_INDEX)
    );

    public ApproxTopK(Expression... varArgs) {
        this(false, varArgs);
    }

    public ApproxTopK(boolean distinct, Expression... varArgs) {
        this(distinct, false, varArgs);
    }

    public ApproxTopK(boolean distinct, boolean alwaysNullable, Expression... varArgs) {
        super("approx_top_k", distinct, alwaysNullable, varArgs);
    }

    @Override
    public void checkLegalityBeforeTypeCoercion() {
        if (arity() < 3) {
            throw new AnalysisException(
                    "Function requires at least 3 parameters: " + this.toSql());
        }

        if (!getArgument(arity() - 2).isConstant() || !getArgumentType(arity() - 2).isIntegerLikeType()) {
            throw new AnalysisException(
                    "The second to last parameter must be a constant Integer Type: " + this.toSql());
        }

        if (!getArgument(arity() - 1).isConstant() || !getArgumentType(arity() - 1).isIntegerLikeType()) {
            throw new AnalysisException(
                    "The last parameter must be a constant Integer Type: " + this.toSql());
        }
    }

    @Override
    public ApproxTopK withDistinctAndChildren(boolean distinct, List<Expression> children) {
        Preconditions.checkArgument(children.size() >= 1);
        return new ApproxTopK(distinct, alwaysNullable, children.toArray(new Expression[0]));
    }

    @Override
    public NullableAggregateFunction withAlwaysNullable(boolean alwaysNullable) {
        return new ApproxTopK(distinct, alwaysNullable, children.toArray(new Expression[0]));
    }

    @Override
    public <R, C> R accept(ExpressionVisitor<R, C> visitor, C context) {
        return visitor.visitApproxTopK(this, context);
    }

    @Override
    public List<FunctionSignature> getSignatures() {
        return SIGNATURES;
    }
}
