/* Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*
 * Original Copyright (c) 2005 Covalent Technologies
 *
 * FTP Protocol module for Apache 2.0
 */

#ifndef FTP_INTERNAL_H
#define FTP_INTERNAL_H

#include "mod_ftp.h"

#ifdef FTP_APXS_BUILD
/* Not defined on Win32/Netware since #undef's aren't really needed */
#include "ftp_config.h"
#endif
#ifndef CORE_PRIVATE
#define CORE_PRIVATE
#endif

/* FTP header has a couple of defines... on win32
   rather than including it, we'll just hack em in
   ourselves
 */
#ifdef HAVE_ARPA_FTP_H
#include <arpa/ftp.h>
#else
#define	TYPE_A		1	/* ASCII */
#define	TYPE_I		3	/* image */
#endif

/* Apache headers */
#include "http_log.h"
#include "http_main.h"
#include "http_core.h"
#include "http_connection.h"
#include "http_request.h"
#include "http_protocol.h"

#include "ap_config.h"
#include "ap_mmn.h"

#include "util_filter.h"
#include "util_ebcdic.h"
#include "scoreboard.h"

#include "apr_buckets.h"
#include "apr_strings.h"
#include "apr_hooks.h"
#include "apr_hash.h"
#include "apr_user.h"
#include "apr_lib.h"
#include "apr_inherit.h"
#include "apr_poll.h"

#ifdef FTP_APXS_BUILD
#include "mod_log_config.h"
#else
#include "../../modules/loggers/mod_log_config.h"
#endif

#if APR_HAVE_UNISTD_H
#include <unistd.h>
#endif

#if APR_HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif

#ifdef HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif

/* SSL Filter name */
#define FTP_SSL_FILTER        "SSL/TLS Filter"

/*
 * FTP_NEED_* - Values used for filter bitmasks
 *
 */

#define FTP_NEED_BYTERANGE      (1 << 1)
#define FTP_NEED_CONTENT_LENGTH (1 << 2)
#define FTP_NEED_CRLF           (1 << 4)
#define FTP_NEED_DATA_OUT       (1 << 5)
/*
 * FTP_MSG_* - Text for sending back to the client.  These include 
 *             metacharacters to enforce rigidity of layout.
 */

#define FTP_MSG_PERM_DENIED  "%s: Permission denied"
#define FTP_MSG_NOT_A_DIR    "%s: Not a directory"
#define FTP_MSG_NOSUCHFILE   "%s: No such file or directory"
#define FTP_MSG_OPENASCII    "Opening ASCII mode data connection for file list"
#define FTP_MSG_SUCCESS      "%s: Command successful"
#define FTP_MSG_NOTPLAIN     "%s: Not a plain file"
#define FTP_MSG_DIR_CREAT    "\"%s\" new directory created"
#define FTP_MSG_DIR_CUR      "\"%s\" is current directory"
#define FTP_MSG_HELP_SYNTAX  "Syntax: %s %s"
#define FTP_MSG_HELP         "Direct comments to %s"
#define FTP_MSG_NOTIMPL      "%s: Command not implemented"
#define FTP_MSG_NOTALLOWED   "Permission denied: %s not allowed here"
#define FTP_MSG_SESSIONLIMIT "Maximum number of concurrent sessions reached," \
                             " closing connection."

#define FTP_DEFAULT_DBFILE   "logs/ftplogins"
/*
 * FTP_OPT_* - FTP server options
 *
 */
#define FTP_OPT_NONE             0
#define FTP_OPT_REQUIRESSL      (1 << 0) /* Require AUTH before USER/PASS */
#define FTP_OPT_CHECKMAXCLIENTS (1 << 1) /* Check if the server is full */
#define FTP_OPT_REMOVEUSERGROUP (1 << 2) /* Fill in user/group information
                                          * on directory listings */
#define FTP_OPT_NLSTSHOWDIRS    (1 << 3) /* Show directories on NLST 
                                          * requests */
#define FTP_OPT_NLSTISLIST      (1 << 4) /* Maps NLST directly to LIST */
#define FTP_OPT_CREATEHOMEDIRS  (1 << 5) /* Create home dirs on the fly */
#define FTP_OPT_SHOWUNAUTH      (1 << 6) /* Show unauthorized files in LIST */
#define FTP_OPT_LISTISNLST      (1 << 7) /* Maps LiST directly to NLST */
#define FTP_OPT_ALLOWPROXYPORT  (1 << 8) /* Data connect to other than client */
#define FTP_OPT_ALLOWPROXYPASV  (1 << 9) /* Data listens to other than client */
#define FTP_OPT_VHOST_BY_USER  (1 << 10) /* Parse USER for @hostname vhost */
#define FTP_OPT_STRIP_HOSTNAME (1 << 11) /* Strip @hostname from username */
#define FTP_OPT_NO_UTF8_FEAT   (1 << 12) /* Hide UTF8 from FEAT results */

/* Supported FTP AUTH mechanisms. */
#define FTP_AUTH_NONE         0
#define FTP_AUTH_SSL          1
#define FTP_AUTH_TLS          2

/* Data protection levels */
#define FTP_PROT_CLEAR        0
#define FTP_PROT_SAFE         1
#define FTP_PROT_CONFIDENTIAL 2
#define FTP_PROT_PRIVATE      3

typedef enum {
    FTP_LIMIT_OK,
    FTP_LIMIT_HIT_PERUSER,
    FTP_LIMIT_HIT_PERIP,
    FTP_LIMIT_HIT_PERSERVER,
    FTP_LIMIT_ERROR
} ftp_loginlimit_t;

/* Directory entry structure.  Used for directory listings */
typedef struct ftp_direntry
{
    const char *name;           /* Just the name of the file (foo.c) */
    const char *modestring;
    char datestring[13];        /* Date srings have a fixed length */
    char *username;
    char *groupname;
    apr_off_t size; 
    apr_off_t csize;
    apr_int32_t  nlink;
    struct ftp_direntry *child; /* For descending */
    struct ftp_direntry *next;
} ftp_direntry;

#define FTP_DEFAULT_UMASK (APR_GWRITE | APR_WWRITE)
#define FTP_UNSPEC        -2
#define FTP_TIMEOUT_LOGIN 120
#define FTP_TIMEOUT_IDLE  600
#define FTP_TIMEOUT_DATA  300
#define FTP_MAX_LOGINS    3
#define FTP_MAX_TRIES     10 /* Maximum number of times to retry on
                                failed bind() and connect() */

#define FTP_DATA_BLOCK_SIZE  48000 /* Size in bytes to read at a time
                                      on the data channel before looking
                                      aside for an ABOR */

/* Wrappers for module_config
 *
 * mod_ftp.c
 */

extern ap_filter_rec_t *ftp_input_filter_handle;

/* Routines for handling incoming connections
 * 
 * ftp_connection.c
 */
int ftp_process_connection(conn_rec *c);

/* Placeholder of "unknown" username (not valid for ftp_cmd_pass)
 */
extern const char ftp_unknown_username[];

/* Routines for handling FTP requests 
 *
 * ftp_request.c 
 */
void ftp_process_request(request_rec *r);

/* Routines specific to the FTP Protocol 
 * 
 * ftp_protocol.c 
 */
request_rec *ftp_read_request(ftp_connection *fc);
apr_status_t ftp_read_ahead_request(ftp_connection *fc);
apr_status_t ftp_reply(ftp_connection *fc, ap_filter_t *out_filter, 
                       apr_pool_t *p, int n, int l, const char *fmt, ...);
apr_status_t ftp_show_file(ap_filter_t *out_filter, apr_pool_t *p, int code, 
                           ftp_connection *fc, const char *file);
void ftp_send_response(request_rec *r, int res);
apr_status_t ftp_ssl_init(conn_rec *c);
int ftp_have_ssl(void);

/* Routines specific to the FTP data connections
 * 
 * ftp_data_connection.c 
 */
void ftp_reset_dataconn(ftp_connection *fc);
conn_rec *ftp_open_dataconn(request_rec *r, int write_not_read);

/* FTP command handlers registration.
 */
void ftp_register_core_cmds(apr_pool_t *pool);

/* Routines for handling implementation of login limits.
 * Note that the actual Mutexing functions and DB file
 * functions are non-public.
 *
 * ftp_limitlogin.c
 */
apr_status_t ftp_mutexdb_init(server_rec *s, apr_pool_t *p);
apr_status_t ftp_mutexdb_child_init(server_rec *s, apr_pool_t *p);
apr_status_t ftp_mutexdb_cleanup(void *dummy);
ftp_loginlimit_t ftp_limitlogin_check(const char *user, request_rec *r);
int ftp_limitlogin_loggedout(conn_rec *c);

/* FTP utilities
 *
 * ftp_util.c
 */
int ftp_eprt_decode(apr_int32_t *family, char **addr, apr_port_t *port,
                    char *arg);
struct ftp_direntry *ftp_direntry_get(request_rec *r, const char *pattern);

void ftp_set_authorization(request_rec *r);
int ftp_set_uri(request_rec *r, const char *arg);

#ifdef WIN32
#define mode_t int
#endif

mode_t ftp_unix_perms2mode(apr_fileperms_t perms);
apr_fileperms_t ftp_unix_mode2perms(mode_t mode);

char *ftp_toupper(apr_pool_t *p, const char *s);
int ftp_check_maxclients(request_rec *r);

/* FTP filters
 *
 * ftp_filters.c
 */
apr_status_t ftp_data_out_filter(ap_filter_t * f, apr_bucket_brigade * bb);
apr_status_t ftp_crlf_filter(ap_filter_t *f, apr_bucket_brigade *bb);
apr_status_t ftp_protocol_filter(ap_filter_t *f, apr_bucket_brigade *b,
                              ap_input_mode_t mode, apr_read_type_e block,
                              apr_off_t readbytes);
apr_bucket *ftp_bucket_datasock_create(ftp_connection *fc,
                                       apr_bucket_alloc_t *list);
extern const apr_bucket_type_t ftp_bucket_type_datasock;

/* FTP logging hooks into mod_log_config
 *
 * ftp_log.c
 */
const char *ftp_log_transfer_mode(request_rec *r, char *a);
const char *ftp_log_action_flags(request_rec *r, char *a);
const char *ftp_log_transfer_direction(request_rec *r, char *a);
const char *ftp_log_accessed_anonymously(request_rec *r, char *a);
const char *ftp_log_service_name(request_rec *r, char *a);
const char *ftp_log_auth_method(request_rec *r, char *a);
const char *ftp_log_auth_user_id(request_rec *r, char *a);

/* dirlist stuff */
#define FTP_MODESTRING_LEN 11  /* # of chars in a modestring, with \0 */
#define FTP_UNKNOWN_MODESTRING "??????????"
#define FTP_MAX_MODESTRING ((APR_UREAD  | APR_UWRITE | APR_UEXECUTE | \
                             APR_GREAD  | APR_GWRITE | APR_GEXECUTE | \
                             APR_WREAD  | APR_WWRITE | APR_WEXECUTE | \
                             APR_USETID | APR_GSETID | APR_WSTICKY) + 1)

/* FTP message functions for dealing with client messages
 *
 * ftp_message.c
 */
void ftp_message_generate(ftp_connection *fc, const char *inbuf,
                          char *outbuf, size_t outlen);

/* inet_pton functionallity ripped straight out of APR
 *
 * ftp_inet_pton.c
 */
int ftp_inet_pton(int af, const char *src, void *dst);

/* FTP hooks */

/*
 * This is the function that translates ftp aliased methods.
 * It will retreive the base command from the cmd hash.
 */
const char *ftp_get_cmd_alias(const char *key);

/*
 * This is the function that runs the ftp handler.  It will iterate through
 * the linked list of entries that returned from the cmd hash.
 */
int ftp_run_cmd(request_rec *r, const char *key);

/*
 * This function reports (true|false) if a given command should
 * abort an active data transfer
 */
int ftp_cmd_abort_data(const char *key);

/* Finalizes ftp_cmd_help and ftp_cmd_feat messages */
void ftp_cmd_finalize(apr_pool_t *p, apr_pool_t *ptemp);

/* FTP low-numbered-port allocation daemon
 *
 * ftp_lowportd.c
 */
/* Lone configuration option */
const char *lowportd_set_socket(cmd_parms *cmd, void *dummy, const char *arg);

/* The module is enabled by the presence of unix domain sockets */
#if APR_HAVE_SYS_UN_H && (defined(HAVE_STRUCT_MSGHDR_MSG_ACCRIGHTS) \
                       || defined(HAVE_STRUCT_MSGHDR_MSG_CONTROL))

#define HAVE_FTP_LOWPORTD

/* Initialization */
int lowportd_pre_config(apr_pool_t *pconf, apr_pool_t *plog, apr_pool_t *ptemp);
int lowportd_post_config(apr_pool_t *pconf, apr_pool_t *plog, apr_pool_t *ptemp,
                         server_rec *main_server);

/* Query */
apr_status_t ftp_request_lowport(apr_socket_t **sock, request_rec *r,
                                 apr_sockaddr_t *sa, apr_pool_t *p);
#endif

#endif
