/*******************************************************************************
 * Copyright (c) 2010 Engineering Group.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Marco Cortella (Engineering Group) - initial API and implementation and/or initial documentation
 *******************************************************************************/
package org.eclipse.ebam.emf;

import java.util.List;

import javax.persistence.EntityManager;
import javax.persistence.FlushModeType;
import javax.persistence.NoResultException;
import javax.persistence.Query;

import org.eclipse.ebam.model.entities.SbiAlarm;
import org.eclipse.ebam.model.entities.SbiAlarmEvent;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class AlarmTransactions extends EMF{

	/**
	 * Logger.
	 */
	private static Logger logger = 
							LoggerFactory.getLogger(AlarmTransactions.class);
	private EntityManager em = null;
	
	/**Method that executes database query to extract <code>SbiAlarmEvent</code>
	 * all active alarms.
	 * @return <code>SbiAlarmEvent</code>
	 * @throws Exception error on select
	 */
	@SuppressWarnings("finally")
	public final List<SbiAlarmEvent> findActiveAlarmEvents() 
														throws Exception {
		List<SbiAlarmEvent> alarmEvents = null;
		EntityManager em = getEMF().createEntityManager();
		try {
	        Query query = em.createQuery("select e from SbiAlarmEvent e " 
	        		+ "where e.active = :active");
	        query.setParameter("active", "T");
	        alarmEvents = (List<SbiAlarmEvent>) query.getResultList();	        

		} catch (Exception e) {
			logger.error(e.getMessage(), e);
		} finally {
			em.close();			
		}
		return alarmEvents;
	}
	
	/**Method that executes database query to extract <code>SbiAlarmEvent</code>
	 * all active alarms.
	 * @return <code>SbiAlarmEvent</code>
	 * @throws Exception error on select
	 */
	@SuppressWarnings("finally")
	public final void updateSbiAlarmEvent(SbiAlarmEvent ev) 
														throws Exception {

		EntityManager em = getEMF().createEntityManager();
		try {
			em.getTransaction().begin();

			// Set the flush-mode so the manager query does not force a
			// write before the read.
			em.setFlushMode(FlushModeType.COMMIT);

			em.merge(ev);

			em.getTransaction().commit();

		} catch (Exception e) {
			if (em.getTransaction() != null && em.getTransaction().isActive()) {
				em.getTransaction().rollback();
			}
			logger.error(e.getMessage(), e);
			throw e;
		} finally {
			em.close();
		}
	}
	
	/**Method that executes database query to insert a new <code>SbiAlarmEvent</code>.
	 * 
	 * @param ev SbiAlarmEvent
	 * @return boolean
	 * @throws Exception error on select
	 */
	public final boolean createNewSbiAlarmEvent(final SbiAlarmEvent ev) throws Exception {
		boolean isSaved = false;
		EntityManager em = getEMF().createEntityManager();
		try {
			em.getTransaction().begin();
	
			// Set the flush-mode so the manager query does not force a 
			// write before the read.
			em.setFlushMode(FlushModeType.COMMIT);

			SbiAlarm alarm=loadSbiAlarmByLabel(em,ev.getSbiAlarm().getLabel());
			ev.setSbiAlarm(alarm);
			em.persist(ev);
			
			em.getTransaction().commit();

			isSaved = true;

		} catch (Exception e) {
			if (em.getTransaction() != null && em.getTransaction().isActive()) {
				em.getTransaction().rollback();
			}			
			logger.error(e.getMessage(), e);
			throw e;
		} finally {			
			em.close();			
		}
		return isSaved;
	}
	
	public final boolean mergeSbiAlarm(final SbiAlarm sbiAlarm) throws Exception {
		boolean isSaved = false;
		EntityManager em = getEMF().createEntityManager();
		try {
			em.getTransaction().begin();
	
			// Set the flush-mode so the manager query does not force a 
			// write before the read.
			em.setFlushMode(FlushModeType.COMMIT);

			em.merge(sbiAlarm);
			
			em.getTransaction().commit();

			isSaved = true;

		} catch (Exception e) {
			if (em.getTransaction() != null && em.getTransaction().isActive()) {
				em.getTransaction().rollback();
			}			
			logger.error(e.getMessage(), e);
			throw e;
		} finally {			
			em.close();			
		}
		return isSaved;
	}
	
	/**Method that executes database query to extract <code>SbiAlarmEvent</code>
	 * all active alarms.
	 * @return <code>SbiAlarmEvent</code>
	 * @throws Exception error on select
	 */
	public final SbiAlarm loadSbiAlarmByLabel(String alarmLabel) 
														throws Exception {
		SbiAlarm a = null;
		EntityManager em = getEMF().createEntityManager();
		try {
	        Query query = em.createQuery("select e from SbiAlarm e " 
	        		+ "where e.label = :label");
	        query.setParameter("label", alarmLabel);
	        a = (SbiAlarm) query.getSingleResult();	        

		} catch (NoResultException e) {
			logger.debug("No alarm with this label exists:"+alarmLabel);
		} catch (Exception e) {
			logger.error(e.getMessage(), e);
			throw e;
		} finally {
			em.close();			
		}
		return a;
	}

    private final SbiAlarm loadSbiAlarmByLabel(EntityManager em,String alarmLabel) throws Exception {
	SbiAlarm a = null;
	try {
	    Query query = em.createQuery("select e from SbiAlarm e "
		    + "where e.label = :label");
	    query.setParameter("label", alarmLabel);
	    a = (SbiAlarm) query.getSingleResult();

	} catch (NoResultException e) {
	    logger.debug("No alarm with this label exists:"+alarmLabel, e);
	} catch (Exception e) {
	    logger.error(e.getMessage(), e);
	    throw e;
	} 
	return a;
    }
	
	/**Method that executes database query to extract <code>List<SbiAlarm></code>
	 * with specified EventId
	 * @return <code>List<SbiAlarm></code>
	 * @throws Exception error on select
	 */
	public final List<SbiAlarm> loadSbiAlarmListByEventId(Double eventId) 
														throws Exception {
		List<SbiAlarm> a = null;
		EntityManager em = getEMF().createEntityManager();
		try {
	        Query query = em.createQuery("select e from SbiAlarm e " 
	        		+ "where e.event.eventId = :eventId");
	        query.setParameter("eventId", eventId);
	        a =(List<SbiAlarm>) query.getResultList();	        

		} catch (NoResultException e) {
			logger.debug("No alarm with this eventId exists:"+eventId.toString(), e);
		} catch (Exception e) {
			logger.error(e.getMessage(), e);
			throw e;
		} finally {
			em.close();			
		}
		return a;
	}
	
}
