
package org.mandas.docker.client.messages;

import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.databind.annotation.JsonDeserialize;
import com.fasterxml.jackson.databind.annotation.JsonSerialize;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import org.mandas.docker.Nullable;
import org.mandas.docker.client.jackson.UnixTimestampDeserializer;
import org.mandas.docker.client.jackson.UnixTimestampSerializer;

/**
 * Immutable implementation of {@link Event}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableEvent.builder()}.
 */
@SuppressWarnings({"all"})
final class ImmutableEvent implements Event {
  private final @Nullable Event.Type type;
  private final @Nullable String action;
  private final @Nullable Event.Actor actor;
  private final Date time;
  private final @Nullable Long timeNano;

  private ImmutableEvent(
      @Nullable Event.Type type,
      @Nullable String action,
      @Nullable Event.Actor actor,
      Date time,
      @Nullable Long timeNano) {
    this.type = type;
    this.action = action;
    this.actor = actor;
    this.time = time;
    this.timeNano = timeNano;
  }

  /**
   * @return The value of the {@code type} attribute
   */
  @JsonProperty("Type")
  @Override
  public @Nullable Event.Type type() {
    return type;
  }

  /**
   * @return The value of the {@code action} attribute
   */
  @JsonProperty("Action")
  @Override
  public @Nullable String action() {
    return action;
  }

  /**
   * @return The value of the {@code actor} attribute
   */
  @JsonProperty("Actor")
  @Override
  public @Nullable Event.Actor actor() {
    return actor;
  }

  /**
   * @return The value of the {@code time} attribute
   */
  @JsonProperty("time")
  @JsonDeserialize(using = UnixTimestampDeserializer.class)
  @JsonSerialize(using = UnixTimestampSerializer.class)
  @Override
  public Date time() {
    return time;
  }

  /**
   * @return The value of the {@code timeNano} attribute
   */
  @JsonProperty("timeNano")
  @Override
  public @Nullable Long timeNano() {
    return timeNano;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Event#type() type} attribute.
   * A value equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for type (can be {@code null})
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableEvent withType(@Nullable Event.Type value) {
    if (this.type == value) return this;
    return new ImmutableEvent(value, this.action, this.actor, this.time, this.timeNano);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Event#action() action} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for action (can be {@code null})
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableEvent withAction(@Nullable String value) {
    if (Objects.equals(this.action, value)) return this;
    return new ImmutableEvent(this.type, value, this.actor, this.time, this.timeNano);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Event#actor() actor} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for actor (can be {@code null})
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableEvent withActor(@Nullable Event.Actor value) {
    if (this.actor == value) return this;
    return new ImmutableEvent(this.type, this.action, value, this.time, this.timeNano);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Event#time() time} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for time
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableEvent withTime(Date value) {
    if (this.time == value) return this;
    Date newValue = Objects.requireNonNull(value, "time");
    return new ImmutableEvent(this.type, this.action, this.actor, newValue, this.timeNano);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Event#timeNano() timeNano} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for timeNano (can be {@code null})
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableEvent withTimeNano(@Nullable Long value) {
    if (Objects.equals(this.timeNano, value)) return this;
    return new ImmutableEvent(this.type, this.action, this.actor, this.time, value);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableEvent} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableEvent
        && equalsByValue((ImmutableEvent) another);
  }

  private boolean equalsByValue(ImmutableEvent another) {
    return Objects.equals(type, another.type)
        && Objects.equals(action, another.action)
        && Objects.equals(actor, another.actor)
        && time.equals(another.time)
        && Objects.equals(timeNano, another.timeNano);
  }

  /**
   * Computes a hash code from attributes: {@code type}, {@code action}, {@code actor}, {@code time}, {@code timeNano}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + Objects.hashCode(type);
    h += (h << 5) + Objects.hashCode(action);
    h += (h << 5) + Objects.hashCode(actor);
    h += (h << 5) + time.hashCode();
    h += (h << 5) + Objects.hashCode(timeNano);
    return h;
  }

  /**
   * Prints the immutable value {@code Event} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "Event{"
        + "type=" + type
        + ", action=" + action
        + ", actor=" + actor
        + ", time=" + time
        + ", timeNano=" + timeNano
        + "}";
  }

  /**
   * Creates an immutable copy of a {@link Event} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable Event instance
   */
  public static ImmutableEvent copyOf(Event instance) {
    if (instance instanceof ImmutableEvent) {
      return (ImmutableEvent) instance;
    }
    return ImmutableEvent.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableEvent ImmutableEvent}.
   * <pre>
   * ImmutableEvent.builder()
   *    .type(org.mandas.docker.client.messages.Event.Type | null) // nullable {@link Event#type() type}
   *    .action(String | null) // nullable {@link Event#action() action}
   *    .actor(org.mandas.docker.client.messages.Event.Actor | null) // nullable {@link Event#actor() actor}
   *    .time(Date) // required {@link Event#time() time}
   *    .timeNano(Long | null) // nullable {@link Event#timeNano() timeNano}
   *    .build();
   * </pre>
   * @return A new ImmutableEvent builder
   */
  public static ImmutableEvent.Builder builder() {
    return new ImmutableEvent.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableEvent ImmutableEvent}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  static final class Builder implements Event.Builder {
    private static final long INIT_BIT_TIME = 0x1L;
    private long initBits = 0x1L;

    private Event.Type type;
    private String action;
    private Event.Actor actor;
    private Date time;
    private Long timeNano;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code Event} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(Event instance) {
      Objects.requireNonNull(instance, "instance");
      @Nullable Event.Type typeValue = instance.type();
      if (typeValue != null) {
        type(typeValue);
      }
      @Nullable String actionValue = instance.action();
      if (actionValue != null) {
        action(actionValue);
      }
      @Nullable Event.Actor actorValue = instance.actor();
      if (actorValue != null) {
        actor(actorValue);
      }
      this.time(instance.time());
      @Nullable Long timeNanoValue = instance.timeNano();
      if (timeNanoValue != null) {
        timeNano(timeNanoValue);
      }
      return this;
    }

    /**
     * Initializes the value for the {@link Event#type() type} attribute.
     * @param type The value for type (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Type")
    public final Builder type(@Nullable Event.Type type) {
      this.type = type;
      return this;
    }

    /**
     * Initializes the value for the {@link Event#action() action} attribute.
     * @param action The value for action (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Action")
    public final Builder action(@Nullable String action) {
      this.action = action;
      return this;
    }

    /**
     * Initializes the value for the {@link Event#actor() actor} attribute.
     * @param actor The value for actor (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Actor")
    public final Builder actor(@Nullable Event.Actor actor) {
      this.actor = actor;
      return this;
    }

    /**
     * Initializes the value for the {@link Event#time() time} attribute.
     * @param time The value for time 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("time")
    @JsonDeserialize(using = UnixTimestampDeserializer.class)
    @JsonSerialize(using = UnixTimestampSerializer.class)
    public final Builder time(Date time) {
      this.time = Objects.requireNonNull(time, "time");
      initBits &= ~INIT_BIT_TIME;
      return this;
    }

    /**
     * Initializes the value for the {@link Event#timeNano() timeNano} attribute.
     * @param timeNano The value for timeNano (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("timeNano")
    public final Builder timeNano(@Nullable Long timeNano) {
      this.timeNano = timeNano;
      return this;
    }

    /**
     * Builds a new {@link ImmutableEvent ImmutableEvent}.
     * @return An immutable instance of Event
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableEvent build() {
      if (initBits != 0) {
        throw new IllegalStateException(formatRequiredAttributesMessage());
      }
      return new ImmutableEvent(type, action, actor, time, timeNano);
    }

    private String formatRequiredAttributesMessage() {
      List<String> attributes = new ArrayList<>();
      if ((initBits & INIT_BIT_TIME) != 0) attributes.add("time");
      return "Cannot build Event, some of required attributes are not set " + attributes;
    }
  }

  /**
   * Immutable implementation of {@link Event.Actor}.
   * <p>
   * Use the builder to create immutable instances:
   * {@code ImmutableEvent.Actor.builder()}.
   */
  static final class Actor implements Event.Actor {
    private final String id;
    private final @Nullable Map<String, String> attributes;

    private Actor(
        String id,
        @Nullable Map<String, String> attributes) {
      this.id = id;
      this.attributes = attributes;
    }

    /**
     * @return The value of the {@code id} attribute
     */
    @JsonProperty("ID")
    @Override
    public String id() {
      return id;
    }

    /**
     * @return The value of the {@code attributes} attribute
     */
    @JsonProperty("Attributes")
    @Override
    public @Nullable Map<String, String> attributes() {
      return attributes;
    }

    /**
     * Copy the current immutable object by setting a value for the {@link Event.Actor#id() id} attribute.
     * An equals check used to prevent copying of the same value by returning {@code this}.
     * @param value A new value for id
     * @return A modified copy or the {@code this} object
     */
    public final ImmutableEvent.Actor withId(String value) {
      String newValue = Objects.requireNonNull(value, "id");
      if (this.id.equals(newValue)) return this;
      return new ImmutableEvent.Actor(newValue, this.attributes);
    }

    /**
     * Copy the current immutable object by replacing the {@link Event.Actor#attributes() attributes} map with the specified map.
     * Nulls are not permitted as keys or values.
     * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
     * @param entries The entries to replace the attributes map
     * @return A modified copy or {@code this} if not changed
     */
    public final ImmutableEvent.Actor withAttributes(@Nullable Map<String, ? extends String> entries) {
      if (this.attributes == entries) return this;
      @Nullable Map<String, String> newValue = entries == null ? null : createUnmodifiableMap(true, false, entries);
      return new ImmutableEvent.Actor(this.id, newValue);
    }

    /**
     * This instance is equal to all instances of {@code Actor} that have equal attribute values.
     * @return {@code true} if {@code this} is equal to {@code another} instance
     */
    @Override
    public boolean equals(Object another) {
      if (this == another) return true;
      return another instanceof ImmutableEvent.Actor
          && equalsByValue((ImmutableEvent.Actor) another);
    }

    private boolean equalsByValue(ImmutableEvent.Actor another) {
      return id.equals(another.id)
          && Objects.equals(attributes, another.attributes);
    }

    /**
     * Computes a hash code from attributes: {@code id}, {@code attributes}.
     * @return hashCode value
     */
    @Override
    public int hashCode() {
      int h = 5381;
      h += (h << 5) + id.hashCode();
      h += (h << 5) + Objects.hashCode(attributes);
      return h;
    }

    /**
     * Prints the immutable value {@code Actor} with attribute values.
     * @return A string representation of the value
     */
    @Override
    public String toString() {
      return "Actor{"
          + "id=" + id
          + ", attributes=" + attributes
          + "}";
    }

    /**
     * Creates an immutable copy of a {@link Event.Actor} value.
     * Uses accessors to get values to initialize the new immutable instance.
     * If an instance is already immutable, it is returned as is.
     * @param instance The instance to copy
     * @return A copied immutable Actor instance
     */
    public static ImmutableEvent.Actor copyOf(Event.Actor instance) {
      if (instance instanceof ImmutableEvent.Actor) {
        return (ImmutableEvent.Actor) instance;
      }
      return ImmutableEvent.Actor.builder()
          .from(instance)
          .build();
    }

    /**
     * Creates a builder for {@link ImmutableEvent.Actor Actor}.
     * <pre>
     * ImmutableEvent.Actor.builder()
     *    .id(String) // required {@link Event.Actor#id() id}
     *    .attributes(Map&lt;String, String&gt; | null) // nullable {@link Event.Actor#attributes() attributes}
     *    .build();
     * </pre>
     * @return A new Actor builder
     */
    public static ImmutableEvent.Actor.Builder builder() {
      return new ImmutableEvent.Actor.Builder();
    }

    /**
     * Builds instances of type {@link ImmutableEvent.Actor Actor}.
     * Initialize attributes and then invoke the {@link #build()} method to create an
     * immutable instance.
     * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
     * but instead used immediately to create instances.</em>
     */
    static final class Builder {
      private static final long INIT_BIT_ID = 0x1L;
      private long initBits = 0x1L;

      private String id;
      private Map<String, String> attributes = null;

      private Builder() {
      }

      /**
       * Fill a builder with attribute values from the provided {@code Actor} instance.
       * Regular attribute values will be replaced with those from the given instance.
       * Absent optional values will not replace present values.
       * Collection elements and entries will be added, not replaced.
       * @param instance The instance from which to copy values
       * @return {@code this} builder for use in a chained invocation
       */
      public final Builder from(Event.Actor instance) {
        Objects.requireNonNull(instance, "instance");
        this.id(instance.id());
        @Nullable Map<String, String> attributesValue = instance.attributes();
        if (attributesValue != null) {
          putAllAttributes(attributesValue);
        }
        return this;
      }

      /**
       * Initializes the value for the {@link Event.Actor#id() id} attribute.
       * @param id The value for id 
       * @return {@code this} builder for use in a chained invocation
       */
      @JsonProperty("ID")
      public final Builder id(String id) {
        this.id = Objects.requireNonNull(id, "id");
        initBits &= ~INIT_BIT_ID;
        return this;
      }

      /**
       * Put one entry to the {@link Event.Actor#attributes() attributes} map.
       * @param key The key in the attributes map
       * @param value The associated value in the attributes map
       * @return {@code this} builder for use in a chained invocation
       */
      public final Builder addAttribute(String key, String value) {
        if (this.attributes == null) {
          this.attributes = new LinkedHashMap<String, String>();
        }
        this.attributes.put(
            Objects.requireNonNull(key, "attributes key"),
            Objects.requireNonNull(value, value == null ? "attributes value for key: " + key : null));
        return this;
      }

      /**
       * Put one entry to the {@link Event.Actor#attributes() attributes} map. Nulls are not permitted
       * @param entry The key and value entry
       * @return {@code this} builder for use in a chained invocation
       */
      public final Builder addAttribute(Map.Entry<String, ? extends String> entry) {
        if (this.attributes == null) {
          this.attributes = new LinkedHashMap<String, String>();
        }
        String k = entry.getKey();
        String v = entry.getValue();
        this.attributes.put(
            Objects.requireNonNull(k, "attributes key"),
            Objects.requireNonNull(v, v == null ? "attributes value for key: " + k : null));
        return this;
      }

      /**
       * Sets or replaces all mappings from the specified map as entries for the {@link Event.Actor#attributes() attributes} map. Nulls are not permitted as keys or values, but parameter itself can be null
       * @param entries The entries that will be added to the attributes map
       * @return {@code this} builder for use in a chained invocation
       */
      @JsonProperty("Attributes")
      public final Builder attributes(@Nullable Map<String, ? extends String> entries) {
        if (entries == null) {
          this.attributes = null;
          return this;
        }
        this.attributes = new LinkedHashMap<String, String>();
        return putAllAttributes(entries);
      }

      /**
       * Put all mappings from the specified map as entries to {@link Event.Actor#attributes() attributes} map. Nulls are not permitted
       * @param entries The entries that will be added to the attributes map
       * @return {@code this} builder for use in a chained invocation
       */
      public final Builder putAllAttributes(Map<String, ? extends String> entries) {
        if (this.attributes == null) {
          this.attributes = new LinkedHashMap<String, String>();
        }
        for (Map.Entry<String, ? extends String> e : entries.entrySet()) {
          String k = e.getKey();
          String v = e.getValue();
          this.attributes.put(
              Objects.requireNonNull(k, "attributes key"),
              Objects.requireNonNull(v, v == null ? "attributes value for key: " + k : null));
        }
        return this;
      }

      /**
       * Builds a new {@link ImmutableEvent.Actor Actor}.
       * @return An immutable instance of Actor
       * @throws java.lang.IllegalStateException if any required attributes are missing
       */
      public ImmutableEvent.Actor build() {
        if (initBits != 0) {
          throw new IllegalStateException(formatRequiredAttributesMessage());
        }
        return new ImmutableEvent.Actor(id, attributes == null ? null : createUnmodifiableMap(false, false, attributes));
      }

      private String formatRequiredAttributesMessage() {
        List<String> attributes = new ArrayList<>();
        if ((initBits & INIT_BIT_ID) != 0) attributes.add("id");
        return "Cannot build Actor, some of required attributes are not set " + attributes;
      }
    }
  }

  private static <K, V> Map<K, V> createUnmodifiableMap(boolean checkNulls, boolean skipNulls, Map<? extends K, ? extends V> map) {
    switch (map.size()) {
    case 0: return Collections.emptyMap();
    case 1: {
      Map.Entry<? extends K, ? extends V> e = map.entrySet().iterator().next();
      K k = e.getKey();
      V v = e.getValue();
      if (checkNulls) {
        Objects.requireNonNull(k, "key");
        Objects.requireNonNull(v, v == null ? "value for key: " + k : null);
      }
      if (skipNulls && (k == null || v == null)) {
        return Collections.emptyMap();
      }
      return Collections.singletonMap(k, v);
    }
    default: {
      Map<K, V> linkedMap = new LinkedHashMap<>(map.size() * 4 / 3 + 1);
      if (skipNulls || checkNulls) {
        for (Map.Entry<? extends K, ? extends V> e : map.entrySet()) {
          K k = e.getKey();
          V v = e.getValue();
          if (skipNulls) {
            if (k == null || v == null) continue;
          } else if (checkNulls) {
            Objects.requireNonNull(k, "key");
            Objects.requireNonNull(v, v == null ? "value for key: " + k : null);
          }
          linkedMap.put(k, v);
        }
      } else {
        linkedMap.putAll(map);
      }
      return Collections.unmodifiableMap(linkedMap);
    }
    }
  }
}
