/*
 * Copyright (C) 2007 The Guava Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except
 * in compliance with the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License
 * is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express
 * or implied. See the License for the specific language governing permissions and limitations under
 * the License.
 */

package com.google.common.base;

import com.google.common.annotations.GwtCompatible;
import java.util.Arrays;
import org.jspecify.annotations.Nullable;

/**
 * Helper functions that can operate on any {@code Object}.
 *
 * <p>See the Guava User Guide on <a
 * href="https://github.com/google/guava/wiki/CommonObjectUtilitiesExplained">writing {@code Object}
 * methods with {@code Objects}</a>.
 *
 * @author Laurence Gonsalves
 * @since 2.0
 */
@GwtCompatible
public final class Objects extends ExtraObjectsMethodsForWeb {

  /**
   * Determines whether two possibly-null objects are equal. Returns:
   *
   * <ul>
   *   <li>{@code true} if {@code a} and {@code b} are both null.
   *   <li>{@code true} if {@code a} and {@code b} are both non-null and they are equal according to
   *       {@link Object#equals(Object)}.
   *   <li>{@code false} in all other situations.
   * </ul>
   *
   * <p>This assumes that any non-null objects passed to this function conform to the {@code
   * equals()} contract.
   *
   * <p><b>Note:</b> this method is now unnecessary and should be treated as deprecated; use {@link
   * java.util.Objects#equals} instead.
   */
  @SuppressWarnings("InlineMeSuggester") // would introduce fully qualified references to Objects
  public static boolean equal(@Nullable Object a, @Nullable Object b) {
    return java.util.Objects.equals(a, b);
  }

  /**
   * Generates a hash code for multiple values. The hash code is generated by calling {@link
   * Arrays#hashCode(Object[])}. Note that array arguments to this method, with the exception of a
   * single Object array, do not get any special handling; their hash codes are based on identity
   * and not contents.
   *
   * <p>This is useful for implementing {@link Object#hashCode()}. For example, in an object that
   * has three properties, {@code x}, {@code y}, and {@code z}, one could write:
   *
   * {@snippet :
   * public int hashCode() {
   *   return Objects.hashCode(getX(), getY(), getZ());
   * }
   * }
   *
   * <p><b>Warning:</b> When a single object is supplied, the returned hash code does not equal the
   * hash code of that object.
   *
   * <p><b>Note:</b> this method is now unnecessary and should be treated as deprecated; use {@link
   * java.util.Objects#hash} instead.
   */
  @SuppressWarnings("InlineMeSuggester") // would introduce fully qualified references to Objects
  public static int hashCode(@Nullable Object @Nullable ... objects) {
    return java.util.Objects.hash(objects);
  }

  private Objects() {}
}
