/*
 * Copyright 2002-2014 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.springframework.cache.interceptor;

import java.lang.reflect.Method;

import org.springframework.util.Assert;
import org.springframework.util.ObjectUtils;

/**
 * Represent a method on a particular {@link Class} and  is suitable as a key.
 * <p>Mainly for internal use within the framework.
 *
 * @author Costin Leau
 * @author Stephane Nicoll
 * @since 4.0.4
 */
public final class MethodCacheKey {

	private final Method method;

	private final Class<?> targetClass;

	public MethodCacheKey(Method method, Class<?> targetClass) {
		Assert.notNull(method, "method must be set.");
		this.method = method;
		this.targetClass = targetClass;
	}

	@Override
	public boolean equals(Object other) {
		if (this == other) {
			return true;
		}
		if (!(other instanceof MethodCacheKey)) {
			return false;
		}
		MethodCacheKey otherKey = (MethodCacheKey) other;
		return (this.method.equals(otherKey.method) && ObjectUtils.nullSafeEquals(this.targetClass,
				otherKey.targetClass));
	}

	@Override
	public int hashCode() {
		return this.method.hashCode() * 29 + (this.targetClass != null ? this.targetClass.hashCode() : 0);
	}

}