/**
 * <copyright>
 * 
 * Copyright (c) 2016 itemis and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     itemis - Initial API and implementation
 * 
 * </copyright>
 */
package org.eclipse.sphinx.emf.util;

import com.google.common.base.Objects;
import org.eclipse.core.runtime.Assert;
import org.eclipse.emf.common.util.URI;

/**
 * Useful API extensions for {@link URI}.
 */
@SuppressWarnings("all")
public class URIExtensions {
  public final static String URI_SEGMENT_SEPARATOR = "/";
  
  public final static String URI_AUTHORITY_SEPARATOR = (URIExtensions.URI_SEGMENT_SEPARATOR + URIExtensions.URI_SEGMENT_SEPARATOR);
  
  public final static String URI_QUERY_SEPARATOR = "?";
  
  public final static String URI_FRAGMENT_SEPARATOR = "#";
  
  /**
   * Retrieves the {@link URI#fragment() fragment} of the given {@link URI} and converted into a URI of its own.
   * 
   * @param uri The URI whose fragment is to be returned as URI.
   * @return The given URI's fragment as URI, or <code>null</code> if given URI has no fragment.
   */
  public static URI getFragment(final URI uri) {
    Assert.isNotNull(uri);
    boolean _hasFragment = uri.hasFragment();
    if (_hasFragment) {
      String fragment = uri.fragment();
      boolean _startsWith = fragment.startsWith(((URIExtensions.URI_SEGMENT_SEPARATOR + "") + URIExtensions.URI_SEGMENT_SEPARATOR));
      if (_startsWith) {
        final int queryIdx = fragment.indexOf(URIExtensions.URI_QUERY_SEPARATOR);
        String _xifexpression = null;
        if ((queryIdx != (-1))) {
          _xifexpression = fragment.substring(0, queryIdx);
        } else {
          _xifexpression = fragment;
        }
        final String path = _xifexpression;
        String _xifexpression_1 = null;
        boolean _and = false;
        if (!(queryIdx != (-1))) {
          _and = false;
        } else {
          int _length = fragment.length();
          boolean _greaterThan = (_length > (queryIdx + 1));
          _and = _greaterThan;
        }
        if (_and) {
          int _length_1 = fragment.length();
          _xifexpression_1 = fragment.substring((queryIdx + 1), _length_1);
        } else {
          _xifexpression_1 = null;
        }
        final String query = _xifexpression_1;
        String[] _split = path.split(URIExtensions.URI_SEGMENT_SEPARATOR);
        return URI.createHierarchicalURI(_split, query, null);
      } else {
        return URI.createURI(fragment, true);
      }
    }
    return null;
  }
  
  /**
   * Returns a copy of the given {@link URI} where the {@link URI#fragment() fragment} has been substituted
   * with the string value of provided fragment URI. The returned URI will have no fragment
   * if the provided fragment URI is <code>null</code>.
   * 
   * @param uri The URI to be manipulated.
   * @param fragment A URI specifying the fragment that the given URI's fragment is to be substituted with; may be <code>null</code>.
   * @return A URI formed by replacing the fragment of the given URI with the string value of provided fragment URI
   *         in case that the latter is not <code>null</code>; a copy of the given URI without its fragment otherwise.
   */
  public static URI substituteFragment(final URI uri, final URI fragment) {
    Assert.isNotNull(uri);
    URI result = uri.trimFragment();
    boolean _notEquals = (!Objects.equal(fragment, null));
    if (_notEquals) {
      String _string = fragment.toString();
      URI _appendFragment = result.appendFragment(_string);
      result = _appendFragment;
    }
    return result;
  }
  
  /**
   * Returns a copy of the given {@link URI} that has been converted into a {@link URI#isPrefix() prefix}
   * and can be used in {@link URI#replacePrefix(URI, URI)} operations.
   * 
   * @param uri The URI to be returned as prefix URI.
   * @return The given URI as prefix URI.
   * @see URI#isPrefix()
   * @see URI#replacePrefix(URI, URI)
   */
  public static URI asPrefix(final URI uri) {
    Assert.isNotNull(uri);
    URI result = uri.trimQuery();
    URI _trimFragment = result.trimFragment();
    result = _trimFragment;
    boolean _hasTrailingPathSeparator = result.hasTrailingPathSeparator();
    boolean _not = (!_hasTrailingPathSeparator);
    if (_not) {
      URI _appendSegment = result.appendSegment("");
      result = _appendSegment;
    }
    return result;
  }
  
  /**
   * If given old last segment matches the {@link URI#lastSegment() last segment} of given {@link URI},
   * this returns a copy of the given URI where the last segment has been replaced by the provided new last segment.
   * 
   * @param uri The URI to be manipulated.
   * @param oldLastSegment The expected current last segment of the URI's path.
   * @param newLastSegment The intended future last segment of the URI's path.
   * @return A URI formed by replacing the last segment of the given URI with the provided new last segment
   *         in case that the given URI's existing last segment matches the provided old last segment,
   *         or <code>null</code> otherwise.
   */
  public static URI replaceLastSegment(final URI uri, final String oldLastSegment, final String newLastSegment) {
    Assert.isNotNull(uri);
    int _segmentCount = uri.segmentCount();
    boolean _greaterThan = (_segmentCount > 0);
    Assert.isLegal(_greaterThan);
    String _lastSegment = uri.lastSegment();
    boolean _equals = _lastSegment.equals(oldLastSegment);
    if (_equals) {
      URI _trimSegments = uri.trimSegments(1);
      return _trimSegments.appendSegment(newLastSegment);
    }
    return null;
  }
  
  /**
   * If given old base URI matches the given {@link URI}, this returns a copy of the given URI
   * where all components included in the old base URI have been replaced by the provided new base URI.
   * The old and new base URI may not only include a {@link URI#path() path} but also
   * a {@link URI#query() query} and/or a {@link URI#fragment() fragment}.
   * 
   * @param uri The URI to be manipulated.
   * @param oldBaseURI The URI's expected current base URI.
   * @param newBaseURI The URI's intended future base URI.
   * @return A URI formed by replacing the provided old base URI in the given URI with the provided new base URI
   *         in case that the old base URI matches the given URI, or <code>null</code> otherwise.
   */
  public static URI replaceBaseURI(final URI uri, final URI oldBaseURI, final URI newBaseURI) {
    Assert.isNotNull(uri);
    Assert.isNotNull(oldBaseURI);
    Assert.isNotNull(newBaseURI);
    boolean _and = false;
    boolean _and_1 = false;
    boolean _hasFragment = uri.hasFragment();
    if (!_hasFragment) {
      _and_1 = false;
    } else {
      boolean _hasFragment_1 = oldBaseURI.hasFragment();
      _and_1 = _hasFragment_1;
    }
    if (!_and_1) {
      _and = false;
    } else {
      boolean _hasFragment_2 = newBaseURI.hasFragment();
      _and = _hasFragment_2;
    }
    if (_and) {
      URI _trimFragment = uri.trimFragment();
      URI _trimFragment_1 = oldBaseURI.trimFragment();
      boolean _equals = _trimFragment.equals(_trimFragment_1);
      if (_equals) {
        URI _fragment = URIExtensions.getFragment(uri);
        URI _fragment_1 = URIExtensions.getFragment(oldBaseURI);
        URI _asPrefix = URIExtensions.asPrefix(_fragment_1);
        URI _fragment_2 = URIExtensions.getFragment(newBaseURI);
        URI _asPrefix_1 = URIExtensions.asPrefix(_fragment_2);
        URI fragment = _fragment.replacePrefix(_asPrefix, _asPrefix_1);
        boolean _notEquals = (!Objects.equal(fragment, null));
        if (_notEquals) {
          return URIExtensions.substituteFragment(newBaseURI, fragment);
        }
      }
    } else {
      URI _asPrefix_2 = URIExtensions.asPrefix(oldBaseURI);
      URI _asPrefix_3 = URIExtensions.asPrefix(newBaseURI);
      return uri.replacePrefix(_asPrefix_2, _asPrefix_3);
    }
    return null;
  }
  
  /**
   * If given {@link URI} has a {@link URI#isHierarchical() hierarchical} {@link URI#fragment() fragment}
   * and given old last fragment segment matches the {@link URI#lastSegment() last segment} of that fragment,
   * this returns a copy of the given URI where the last segment of its fragment
   * has been replaced by the provided new last fragment segment.
   * 
   * @param uri The URI to be manipulated.
   * @param oldLastSegment The expected current last segment of the URI's fragment.
   * @param newLastSegment The intended future last segment of the URI's fragment.
   * @return A URI formed by replacing the last segment in the fragment of the given URI
   *         with the provided new last fragment segment in case that the existing last segment
   *         in the fragment of the given URI matches the provided old last fragment segment,
   *         or <code>null</code> otherwise.
   */
  public static URI replaceLastFragmentSegment(final URI uri, final String oldLastFragmentSegment, final String newLastFragmentSegment) {
    Assert.isNotNull(uri);
    URI fragment = URIExtensions.getFragment(uri);
    boolean _notEquals = (!Objects.equal(fragment, null));
    if (_notEquals) {
      URI _replaceLastSegment = URIExtensions.replaceLastSegment(fragment, oldLastFragmentSegment, newLastFragmentSegment);
      fragment = _replaceLastSegment;
      boolean _notEquals_1 = (!Objects.equal(fragment, null));
      if (_notEquals_1) {
        return URIExtensions.substituteFragment(uri, fragment);
      }
    }
    return null;
  }
}
