/*
    File: 3dbuzz.js
    
    Written to work with the 3DBuzz Toolbar for Mozilla.
    
    Author: Keith Athaide
    
    Notes:
            To add a button that can be hidden
                Add a line to the onInitToolbar() in options.js
                Add a line to save() in options.js
                Add a line to LoadSettings() in this file
*/

var VERSION = "0.9.0.0";
var NUMBER_OF_CUSTOM_FORUMS = 4;
var DEFAULT_SERVER = 1;

//all these indices are one based
var g_curCustomForum = 1;
var g_curForumJump = 1;
var g_curOnlineClass = 1;
var g_maxSearches = 10;
var g_autocompleteState = 1;

// These indices are used when the corresponding ones above are -1 : allow the use of custom menu items
// These indices have no meaning if the corresponding ones above are not -1.
var g_curCustomOnlineClass = -1;
var g_curCustomForumJump = -1;

//g_spacesToAdd - required for the buttons in 'modern' skin
var g_spacesToAdd = "    ";
var g_shouldAddSpaces = 1;


//menu states
var g_forumJumpMenuShowing = 0;
var g_onlineClassMenuShowing = 0;

//variables used to read and write to RDF datasources
/*
    nslRDFService - used to generate resource objects that can uniquely identify a resource in a data source
    nslRDFDataSource - represents a datasource. Has methods to get and set values
    nslRDFContainer - represents a container node in a datasource. Methods to add and remove resources
    nslRDFContainerUtils - methods to create Seq, Bag and Alt resources.
*/
var g_RDFC = '@mozilla.org/rdf/container;1'; //Note that g_RDFC, strictly, should not be global as it's state is not assured to be
                                             // valid for all situations.
g_RDFC = Components.classes[g_RDFC].createInstance();
g_RDFC = g_RDFC.QueryInterface(Components.interfaces.nsIRDFContainer);

var g_RDFCUtils = '@mozilla.org/rdf/container-utils;1';
g_RDFCUtils = Components.classes[g_RDFCUtils].getService();
g_RDFCUtils = g_RDFCUtils.QueryInterface(Components.interfaces.nsIRDFContainerUtils);

var g_RDF = '@mozilla.org/rdf/rdf-service;1';
g_RDF = Components.classes[g_RDF].getService();
g_RDF = g_RDF.QueryInterface(Components.interfaces.nsIRDFService);

var g_nameRDF;
var g_urlRDF;
var g_idRDF;
var g_xidRDF;

/*
    Allows a one-place editing for user alerts.
*/
function MsgBox(msg)
{
    var tempVar = g_RDF.GetDataSource("rdf:local-store")
                        .GetTarget(RDF.GetResource("http://www.3dbuzz.com/settings/showWarnings"), g_nameRDF, true);
    if(tempVar)
        tempVar = parseInt(tempVar.QueryInterface(Components.interfaces.nsIRDFLiteral).Value);
    else
        tempVar = 1;

    if(tempVar)
        alert(msg);
}

function MsgConfirm(msg)
{
    var tempVar = g_RDF.GetDataSource("rdf:local-store")
                        .GetTarget(RDF.GetResource("http://www.3dbuzz.com/settings/showWarnings"), g_nameRDF, true);
    if(tempVar)
        tempVar = parseInt(tempVar.QueryInterface(Components.interfaces.nsIRDFLiteral).Value);
    else
        tempVar = 1;
        
    if(tempVar)
        return confirm(msg);
        
    return 1;
}

/*
    Initializes a datasource from the database property in the component with the supplied name and initialises g_RDFC to be valid for
    this datasource.
*/
function initDataSource(comp)
{
    var ds;
    var element = document.getElementById(comp);
    var sources = element.database.GetDataSources();
    var rootNode = g_RDF.GetResource(element.ref);
    
    while(sources.hasMoreElements())
    {
        try
        {
            ds = sources.getNext();
            ds = ds.QueryInterface(Components.interfaces.nsIRDFDataSource);
            
            g_RDFC.Init(ds, rootNode);
        }
        catch(e)
        {
            g_RDFCUtils.MakeSeq(ds, rootNode);
            g_RDFC.Init(ds, rootNode);
        }
    }
    
    return ds;
}

/*
    Initialise various settings: The text of the custom forum, forum jump and online classes buttons.
                                 The maximum number of searches and the state of autocomplete
                                 Whether to add spaces to the labels of menu buttons and the number of spaces to add
                                 The visibility of the buttons
*/
function LoadSettings()
{
    //var dsource = initDataSource("settings");
    var dsource = g_RDF.GetDataSource("rdf:local-store");
    
    var res;
    
    // g_curCustomForum
    res = dsource.GetTarget(g_RDF.GetResource("http://www.3dbuzz.com/settings/curCustomForum"), g_nameRDF, true);
    if(res)
        g_curCustomForum = parseInt(res.QueryInterface(Components.interfaces.nsIRDFLiteral).Value);
    
    // g_curForumJump
    res = dsource.GetTarget(g_RDF.GetResource("http://www.3dbuzz.com/settings/curForumJump"), g_nameRDF, true);
    if(res)
        g_curForumJump = parseInt(res.QueryInterface(Components.interfaces.nsIRDFLiteral).Value);
    
    // g_curOnlineClass
    res = dsource.GetTarget(g_RDF.GetResource("http://www.3dbuzz.com/settings/curOnlineClass"), g_nameRDF, true);
    if(res)
        g_curOnlineClass = parseInt(res.QueryInterface(Components.interfaces.nsIRDFLiteral).Value);
    
    // g_maxSearches
    res = dsource.GetTarget(g_RDF.GetResource("http://www.3dbuzz.com/settings/maxSearches"), g_nameRDF, true);
    if(res)
        g_maxSearches = parseInt(res.QueryInterface(Components.interfaces.nsIRDFLiteral).Value);
    
    // g_autocompleteState    
    res = dsource.GetTarget(g_RDF.GetResource("http://www.3dbuzz.com/settings/autocomplete"), g_nameRDF, true);
    if(res)
        g_autocompleteState = parseInt(res.QueryInterface(Components.interfaces.nsIRDFLiteral).Value);
    
    // g_shouldAddSpaces
    res = dsource.GetTarget(g_RDF.GetResource("http://www.3dbuzz.com/settings/shouldAddSpaces"), g_nameRDF, true);
    if(res)
        g_shouldAddSpaces = parseInt(res.QueryInterface(Components.interfaces.nsIRDFLiteral).Value);
        
    // g_spacesToAdd
    res = dsource.GetTarget(g_RDF.GetResource("http://www.3dbuzz.com/settings/spaces"), g_nameRDF, true);
    if(res)
    {
        var spaces = parseInt(res.QueryInterface(Components.interfaces.nsIRDFLiteral).Value);
        g_spacesToAdd = "";
        for(var i = 0; i < spaces; i++)
            g_spacesToAdd = g_spacesToAdd + " ";
    }
    
    // g_curCustomOnlineClass
    res = dsource.GetTarget(g_RDF.GetResource("http://www.3dbuzz.com/settings/curCustomOnlineClass"), g_nameRDF, true);
    if(res)
        g_curCustomOnlineClass = parseInt(res.QueryInterface(Components.interfaces.nsIRDFLiteral).Value);
        
    // g_curCustomForumJump
    res = dsource.GetTarget(g_RDF.GetResource("http://www.3dbuzz.com/settings/curCustomForumJump"), g_nameRDF, true);
    if(res)
        g_curCustomForumJump = parseInt(res.QueryInterface(Components.interfaces.nsIRDFLiteral).Value);
    
    if(g_autocompleteState)
        document.getElementById("search-textbox").setAttribute("disableAutocomplete", "false");
    else
        document.getElementById("search-textbox").setAttribute("disableAutocomplete", "true");
        
    // Set the display of buttons
    // An alternative way would be to just directly set the class of the buttons in Options.js and to add the persist attribute to each
    // button.
    setButtonVisibility(dsource, "3dbuzz-button");
    setButtonVisibility(dsource, "openirc-button");
    setButtonVisibility(dsource, "customforum-button");
    setButtonVisibility(dsource, "forumjump-button");
    setButtonVisibility(dsource, "onlineclass-button");
    setButtonVisibility(dsource, "viewnewposts-button");
    setButtonVisibility(dsource, "search-box");
    setButtonVisibility(dsource, "reload-3dbuzzbutton");
    setButtonVisibility(dsource, "frameoperations-button");
    setButtonVisibility(dsource, "options-button");
}

function setButtonVisibility(dsource, buttonName)
{
    var res = dsource.GetTarget(g_RDF.GetResource("http://www.3dbuzz.com/buttonvisibility/"+buttonName), g_nameRDF, true);
    if(res)
    {
        var visible = parseInt(res.QueryInterface(Components.interfaces.nsIRDFLiteral).Value);
        if(visible)
            document.getElementById(buttonName).setAttribute("class", "tbbutton");
        else
            document.getElementById(buttonName).setAttribute("class", "hiddenbutton");
    }
}

function changeButtonText(buttonName, newValue)
{
    var elem = document.getElementById(buttonName);
    if(g_shouldAddSpaces)
        elem.label = newValue + g_spacesToAdd;
    else
        elem.label = newValue;
}

/*
    The weird prefix for this function was required otherwise it overloaded a function of the same name defined in 
    chrome://navigator/content/sessionUIHistory.js (resource://comm.jar) and caused the urlbar to behave funnily.
*/

function buzz3d_createMenuItem(menuPopup, lbl, insertBefore)
{
    var menuitem;
    
    if(lbl == "separator")
    {
        menuitem = document.createElement("menuseparator");
    }
    else
    {
        menuitem = document.createElement("menuitem");
        menuitem.setAttribute("tooltiptext", lbl);
    }
    
    menuitem.setAttribute("label", lbl);
    menuPopup.insertBefore(menuitem, insertBefore);
    return menuitem;
}

function deleteItems(menuPopup)
{
    var children = menuPopup.childNodes;
    
    for(var i = 0; i < children.length; i++)
    {
        deleteItems(children[i]);
        menuPopup.removeChild(children[i]);
    }
}

function createCustomForumMenu(menuPopup)
{
    deleteItems(menuPopup);
    var cf_dsource = g_RDF.GetDataSource("rdf:local-store");
    var disp;
    var itemsAdded = 0;
    
    for(var i = 1; i <= 4; i++)
    {
        var resource = RDF.GetResource("http://www.3dbuzz.com/customforum"+i);
        
        //first determine whether to display this item
        var res = cf_dsource.GetTarget(resource, RDF.GetResource("http://www.3dbuzz.com/rdf#display"), true);
        if(res)
            disp = parseInt(res.QueryInterface(Components.interfaces.nsIRDFLiteral).Value);
        else
            disp = 1;
            
        if(disp == 0) //the menuitem shouldn't be displayed
            continue;   //don't add the menuitem
        
        
        var text = cf_dsource.GetTarget(g_RDF.GetResource("http://www.3dbuzz.com/customforum"+i), g_nameRDF, true);
        
        if(!text)
            text = "Custom Forum "+i;
        else
            text = text.QueryInterface(Components.interfaces.nsIRDFLiteral).Value;
        
        var menuitem = buzz3d_createMenuItem(menuPopup, text);
        menuitem.setAttribute("id", "http://www.3dbuzz.com/customforum"+i);
        menuitem.setAttribute("onclick", "if(event.button)onCustomForumMenuPopup(this, event);");
        itemsAdded++;
    }
}

function findMenuItem(menuPopup, itemLabelText)
{
    if(itemLabelText == "")
        return null;
    
    var numSeparators = 0;
    
    for(var i = 0; i < menuPopup.childNodes.length; i++)
    {
        var child = menuPopup.childNodes[i];
        var lbl = child.getAttribute("label");
        
        if(lbl == "") // separator
        {
            numSeparators++;
            
            if(itemLabelText.indexOf("separator") != -1)
            {
                var num = itemLabelText.substring(9, itemLabelText.length);
                if(num == numSeparators)
                    return child;
            }
        }
        else if(lbl == itemLabelText && child.getAttribute("id").indexOf("additional") == -1)
            return child;
    }
            
    return null;
}

function modifyExistingMenu(dsource, rootNode, menuPopup, origPopup, sCommandHandler, sClickHandler)
{
    var entries;
    
    try {
        entries = g_RDFCUtils.MakeSeq(dsource, rootNode).GetElements();
    } catch(e) {
        return;
    }
    
    while(entries.hasMoreElements())
    {
        var entry = entries.getNext();
        entry = entry.QueryInterface(Components.interfaces.nsIRDFResource);
        
        var misc = dsource.GetTarget(entry, g_urlRDF, true);
        var sName = dsource.GetTarget(entry, g_nameRDF, true).QueryInterface(Components.interfaces.nsIRDFLiteral).Value;
        var insertBefore = dsource.GetTarget(entry, RDF.GetResource("http://www.3dbuzz.com/rdf#insertbefore"), true)
                                .QueryInterface(Components.interfaces.nsIRDFLiteral).Value;
        
        var ibCtrl = findMenuItem(origPopup, insertBefore);
        
        if(misc == null)
        {
            var menu = document.createElement("menu");
            var menuPopup2 = document.createElement("menupopup");
            
            menu.setAttribute("label", sName);
            menu.setAttribute("tooltiptext", sName);
            modifyExistingMenu(dsource, entry, menuPopup2, menuPopup, sCommandHandler, sClickHandler);
            
            menu.appendChild(menuPopup2);
            
            if(menuPopup == origPopup)
                menuPopup.insertBefore(menu,ibCtrl);
            else 
                menuPopup.appendChild(menu);
        }
        else
        {
            misc = misc.QueryInterface(Components.interfaces.nsIRDFLiteral).Value;
            
            var menuitem;
            if(menuPopup != origPopup)
                ibCtrl = null;
                
            menuitem = buzz3d_createMenuItem(menuPopup, sName, ibCtrl);
            menuitem.setAttribute("id", entry.Value);
            
            if(sName != "separator")
            {
                menuitem.setAttribute("oncommand", sCommandHandler);
                menuitem.setAttribute("onclick", sClickHandler);
            }
        }
    }
}

function onOnlineClassMenuShowing(menuPopup)
{
    g_onlineClassMenuShowing++;
    if(g_onlineClassMenuShowing > 1)
        return;
        
    modifyExistingMenu(RDF.GetDataSource("rdf:local-store"), RDF.GetResource("http://www.3dbuzz.com/additional/onlineclasses"),
                            menuPopup, menuPopup, "onOnlineClassMenu(this);", "if(event.button)onOnlineClassMenuPopup(this, event);");
}

function onOnlineClassMenuHidden(menuPopup)
{
    g_onlineClassMenuShowing--;
    
    if(g_onlineClassMenuShowing < 1)
        menuPopup.parentNode.builder.rebuild();
}

function onForumJumpMenuShowing(menuPopup)
{
    g_forumJumpMenuShowing++;
    if(g_forumJumpMenuShowing > 1)
        return;
    
    modifyExistingMenu(RDF.GetDataSource("rdf:local-store"), RDF.GetResource("http://www.3dbuzz.com/additional/forumjump"),
                            menuPopup, menuPopup, "onForumJumpMenu(this);", "if(event.button)onForumJumpMenuPopup(this, event);");
}

function onForumJumpMenuHidden(menuPopup)
{
    g_forumJumpMenuShowing--;
    
    if(g_forumJumpMenuShowing < 1)
        menuPopup.parentNode.builder.rebuild();
}

function changeCustomForumText()
{
    var cf_dsource = g_RDF.GetDataSource("rdf:local-store");
    var text = cf_dsource.GetTarget(g_RDF.GetResource("http://www.3dbuzz.com/customforum"+g_curCustomForum), g_nameRDF, true);
    
    if(!text)
        text = "Custom Forum "+g_curCustomForum;
    else
        text = text.QueryInterface(Components.interfaces.nsIRDFLiteral).Value;
        
    changeButtonText("customforum-button", text);
}

function changeForumJumpText()
{    
    var fj_dsource;
    var text;
                                    
    if(g_curForumJump != -1)
    {
        fj_dsource = initDataSource("forumjump-button");
        text = fj_dsource.GetTarget(g_RDF.GetResource("http://www.3dbuzz.com/forum"+g_curForumJump),
                                    g_nameRDF, true).QueryInterface(Components.interfaces.nsIRDFLiteral);
    }
    else
    {
        var node = g_RDF.GetResource("http://www.3dbuzz.com/additional/forum"+g_curCustomForumJump);
        
        fj_dsource = RDF.GetDataSource("rdf:local-store");
        text = fj_dsource.GetTarget(node, g_nameRDF, true);
        
        if(text == null || fj_dsource.GetTarget(node, g_urlRDF, true) == null)
        {
            saveSetting("http://www.3dbuzz.com/settings/curForumJump", 1, g_curForumJump);
            saveSetting("http://www.3dbuzz.com/settings/curCustomForumJump", -1, g_curCustomForumJump);
            g_curForumJump = 1;
            g_curCustomForumJump = -1;
            changeForumJumpText();
            return;
        }
        else
        {
            text = text.QueryInterface(Components.interfaces.nsIRDFLiteral);
            
            if(text.Value == "separator")
            {
                saveSetting("http://www.3dbuzz.com/settings/curForumJump", 1, g_curForumJump);
                saveSetting("http://www.3dbuzz.com/settings/curCustomForumJump", -1, g_curCustomForumJump);
                g_curForumJump = 1;
                g_curCustomForumJump = -1;
                changeForumJumpText();
                return;
            }
        }
    }
    
    changeButtonText("forumjump-button", text.Value);
}

function changeOnlineClassText()
{
    var oc_dsource;
    var text;
    
    if(g_curOnlineClass != - 1)
    {
        oc_dsource = initDataSource("onlineclass-button");
        text = oc_dsource.GetTarget(g_RDF.GetResource("http://www.3dbuzz.com/onlineclass"+g_curOnlineClass),
                                        g_nameRDF, true).QueryInterface(Components.interfaces.nsIRDFLiteral);
    }
    else
    {
        var node = g_RDF.GetResource("http://www.3dbuzz.com/additional/onlineclass"+g_curCustomOnlineClass);
        
        oc_dsource = RDF.GetDataSource("rdf:local-store");
        text = oc_dsource.GetTarget(node, g_nameRDF, true);
        
        if(text == null || oc_dsource.GetTarget(node, g_urlRDF, true) == null)
        {
            saveSetting("http://www.3dbuzz.com/settings/curOnlineClass", 1, g_curOnlineClass);
            saveSetting("http://www.3dbuzz.com/settings/curCustomOnlineClass", -1, g_curCustomOnlineClass);
            g_curOnlineClass = 1;
            g_curCustomOnlineClass = -1;
            changeOnlineClassText();
            return;
        }
        else
        {
            text = text.QueryInterface(Components.interfaces.nsIRDFLiteral);
            
            if(text.Value == "separator")
            {
                saveSetting("http://www.3dbuzz.com/settings/curOnlineClass", 1, g_curOnlineClass);
                saveSetting("http://www.3dbuzz.com/settings/curCustomOnlineClass", -1, g_curCustomOnlineClass);
                g_curOnlineClass = 1;
                g_curCustomOnlineClass = -1;
                changeOnlineClassText();
                return;
            }
        }
    }
    
    changeButtonText("onlineclass-button", text.Value);
}

function changeSearchText()
{
    changeButtonText("search3DBuzz-button", "Search");
}


function onInitToolbar()
{
    g_nameRDF = g_RDF.GetResource("http://www.3dbuzz.com/rdf#name");
    g_urlRDF = g_RDF.GetResource("http://www.3dbuzz.com/rdf#url");
    g_idRDF = g_RDF.GetResource("http://www.3dbuzz.com/rdf#id");
    g_xidRDF = g_RDF.GetResource("http://www.3dbuzz.com/rdf#xid");
    
    LoadSettings();
    changeCustomForumText();
    changeForumJumpText();
    changeOnlineClassText();
    changeSearchText();
    
    //set state of the menu item in the view menu which shows if the toolbar is displayed
    var comp = document.getElementById("3dbuzz-toolbar");
    var menuItem = document.getElementById("3DBuzzToolbar-showhide");
    var cl = comp.getAttribute("class");
    
    if(cl == "normalToolbar")
        menuItem.setAttribute("checked", "true");
    else
        menuItem.setAttribute("checked", "false");
}

function saveSetting(nodeName, nodeValue, curValue)
{
    var newNode = g_RDF.GetResource(nodeName);
    var newValue = g_RDF.GetLiteral(nodeValue);
    var dsource = g_RDF.GetDataSource("rdf:local-store");//initDataSource("settings");
    //var RDFC = Components.classes['@mozilla.org/rdf/container;1'].createInstance(Components.interfaces.nsIRDFContainer);

    try
    {
      //RDFC.Init(dsource, g_RDF.GetResource("http://www.3dbuzz.com/settings"));

      curValue = g_RDF.GetLiteral(curValue);
    
      dsource.Unassert(newNode, g_nameRDF, curValue, true);
      dsource.Assert(newNode, g_nameRDF, newValue, true);
    
      //RDFC.RemoveElement(newNode, true);
      //RDFC.InsertElementAt(newNode, 1, true);
    
      dsource.QueryInterface(Components.interfaces.nsIRDFRemoteDataSource).Flush();
    } catch(e) {}
}

function onCustomForumMenuPopup(menuItem, evnt)
{
    var commonResourceID = "http://www.3dbuzz.com/customforum";
    var num = menuItem.id.substring(commonResourceID.length, menuItem.id.length);
    var cf_dsource = g_RDF.GetDataSource("rdf:local-store");
    var url = cf_dsource.GetTarget(g_RDF.GetResource("http://www.3dbuzz.com/customforum"+num), 
                                    g_urlRDF, true);
                                    
    if(url)
        url = url.QueryInterface(Components.interfaces.nsIRDFLiteral).Value;
    else
        url = "http://www.3dbuzz.com";
        
    loadURL(url, evnt, false);
}

function onCustomForumMenu(menuItem)
{
    var commonResourceID = "http://www.3dbuzz.com/customforum";
    var num = menuItem.id.substring(commonResourceID.length, menuItem.id.length);
    saveSetting("http://www.3dbuzz.com/settings/curCustomForum", num, g_curCustomForum);
    
    g_curCustomForum = parseInt(num);
    changeCustomForumText();
}

function onCustomForum(evnt)
{
    var cf_dsource = g_RDF.GetDataSource("rdf:local-store");
    var url = cf_dsource.GetTarget(g_RDF.GetResource("http://www.3dbuzz.com/customforum"+g_curCustomForum), 
                                    g_urlRDF, true);

    if(url)
        url = url.QueryInterface(Components.interfaces.nsIRDFLiteral).Value;
    else
        url = "http://www.3dbuzz.com";
        
    loadURL(url, evnt, false);
}


/////////////////////////////////////////////////////////
//             FORUM JUMP MENU FUNCTIONS
/////////////////////////////////////////////////////////

function onForumJumpMenuPopup(menuItem, evnt)
{
    var commonResourceID;
    var num;
    var fj_dsource;
    var url = null;
    
    if(menuItem.id.indexOf("additional") == -1)
        commonResourceID = "http://www.3dbuzz.com/forum";
    else
        commonResourceID = "http://www.3dbuzz.com/additional/forum";
    
    num = parseInt(menuItem.id.substring(commonResourceID.length, menuItem.id.length));

    if(menuItem.id.indexOf("additional") == -1)
    {
        fj_dsource = initDataSource("forumjump-button");
        var id = fj_dsource.GetTarget(g_RDF.GetResource("http://www.3dbuzz.com/forum"+num),
                                         g_idRDF, true).QueryInterface(Components.interfaces.nsIRDFLiteral);
        url = "forumdisplay.php?s=&daysprune=&forumid=" + id.Value;
    }
    else
    {
        fj_dsource = g_RDF.GetDataSource("rdf:local-store");
        url = fj_dsource.GetTarget(g_RDF.GetResource("http://www.3dbuzz.com/additional/forum"+num),
                                         g_urlRDF, true);
        if(url)
            url = url.QueryInterface(Components.interfaces.nsIRDFLiteral).Value;
    }
    
    if(url)
        loadURL(url, evnt, false);
}

function onForumJumpMenu(menuItem)
{
    var commonResourceID;
    var num;
    
    if(menuItem.id.indexOf("additional") == -1)
        commonResourceID = "http://www.3dbuzz.com/forum";
    else
        commonResourceID = "http://www.3dbuzz.com/additional/forum";
    
    num = menuItem.id.substring(commonResourceID.length, menuItem.id.length);
    
    if(menuItem.id.indexOf("additional") == -1)
    {
        saveSetting("http://www.3dbuzz.com/settings/curForumJump", num, g_curForumJump);
        saveSetting("http://www.3dbuzz.com/settings/curCustomForumJump", -1 , g_curCustomForumJump);
        g_curForumJump = parseInt(num);
        g_curCustomForumJump = -1;
    }
    else
    {
        saveSetting("http://www.3dbuzz.com/settings/curForumJump", -1, g_curForumJump);
        saveSetting("http://www.3dbuzz.com/settings/curCustomForumJump", num, g_curCustomForumJump);
        g_curForumJump = -1;
        g_curCustomForumJump = parseInt(num);
    }
    
    changeForumJumpText();
}

function onForumJump(evnt)
{
    var fj_dsource;                 
    var url;
    
    if(g_curForumJump != -1)
    {
        fj_dsource = initDataSource("forumjump-button");
        var id = fj_dsource.GetTarget(g_RDF.GetResource("http://www.3dbuzz.com/forum"+g_curForumJump),
                                         g_idRDF, true).QueryInterface(Components.interfaces.nsIRDFLiteral);
        url = "forumdisplay.php?s=&daysprune=&forumid=" + id.Value;
    }
    else
    {
        fj_dsource = g_RDF.GetDataSource("rdf:local-store");
        url = fj_dsource.GetTarget(g_RDF.GetResource("http://www.3dbuzz.com/additional/forum"+g_curCustomForumJump),
                                         g_urlRDF, true);
        if(url)
            url = url.QueryInterface(Components.interfaces.nsIRDFLiteral).Value;
        else
        {
            saveSetting("http://www.3dbuzz.com/settings/curForumJump", 1, g_curForumJump);
            saveSetting("http://www.3dbuzz.com/settings/curCustomForumJump", -1, g_curCustomForumJump);
            g_curForumJump = 1;
            g_curCustomForumJump = -1;
            changeForumJumpText();
            return;
        }
    }
    
    loadURL(url, evnt, false);
}

//  END OF ONLINE CLASS MENU FUNCTIONS
/////////////////////////////////////////////////////////


/////////////////////////////////////////////////////////
//             ONLINE CLASS MENU FUNCTIONS
/////////////////////////////////////////////////////////

function onOnlineClassMenuPopup(menuItem, evnt)
{
    var commonResourceID;
    var num;
    var oc_dsource;                 
    var url = null;
    
    if(menuItem.id.indexOf("additional") == -1)
        commonResourceID = "http://www.3dbuzz.com/onlineclass";
    else
        commonResourceID = "http://www.3dbuzz.com/additional/onlineclass";
    
    num = parseInt(menuItem.id.substring(commonResourceID.length, menuItem.id.length));
    
    if(menuItem.id.indexOf("additional") == -1)
    {
        oc_dsource = initDataSource("onlineclass-button");
        var xid = oc_dsource.GetTarget(g_RDF.GetResource("http://www.3dbuzz.com/onlineclass"+num),
                                    g_xidRDF, true).QueryInterface(Components.interfaces.nsIRDFLiteral);
        if(num < 4)
            url = "ocreg.php?classid=" + xid.Value;
        else
            url = "unr_ocreg.php?classid=" + xid.Value;
    }
    else
    {
        oc_dsource = g_RDF.GetDataSource("rdf:local-store");
        url = oc_dsource.GetTarget(g_RDF.GetResource("http://www.3dbuzz.com/additional/onlineclass"+num),
                                    g_urlRDF, true);
                                    
        if(url)
            url = url.QueryInterface(Components.interfaces.nsIRDFLiteral).Value;
    }
    
    if(url)
        loadURL(url, evnt, false);
}

function onOnlineClassMenu(menuItem)
{
    var commonResourceID;
    var num;
    
    if(menuItem.id.indexOf("additional") == -1)
        commonResourceID = "http://www.3dbuzz.com/onlineclass";
    else
        commonResourceID = "http://www.3dbuzz.com/additional/onlineclass";
    
    num = menuItem.id.substring(commonResourceID.length, menuItem.id.length);
    
    if(menuItem.id.indexOf("additional") == -1)
    {
        saveSetting("http://www.3dbuzz.com/settings/curOnlineClass", num, g_curOnlineClass);
        saveSetting("http://www.3dbuzz.com/settings/curCustomOnlineClass", -1 , g_curCustomOnlineClass);
        g_curOnlineClass = parseInt(num);
        g_curCustomOnlineClass = -1;
    }
    else
    {
        saveSetting("http://www.3dbuzz.com/settings/curOnlineClass", -1, g_curOnlineClass);
        saveSetting("http://www.3dbuzz.com/settings/curCustomOnlineClass", num, g_curCustomOnlineClass);
        g_curOnlineClass = -1;
        g_curCustomOnlineClass = parseInt(num);
    }
    changeOnlineClassText();
}

function onOnlineClass(evnt)
{
    var oc_dsource;                 
    var url;
    
    if(g_curOnlineClass != -1)
    {
        oc_dsource = initDataSource("onlineclass-button");
        var xid = oc_dsource.GetTarget(g_RDF.GetResource("http://www.3dbuzz.com/onlineclass"+g_curOnlineClass),
                                    g_xidRDF, true).QueryInterface(Components.interfaces.nsIRDFLiteral);
        if(g_curOnlineClass < 4)
            url = "ocreg.php?classid=" + xid.Value;
        else
            url = "unr_ocreg.php?classid=" + xid.Value;
    }
    else
    {
        oc_dsource = g_RDF.GetDataSource("rdf:local-store");
        url = oc_dsource.GetTarget(g_RDF.GetResource("http://www.3dbuzz.com/additional/onlineclass"+g_curCustomOnlineClass),
                                    g_urlRDF, true);
                                    
        if(url)
            url = url.QueryInterface(Components.interfaces.nsIRDFLiteral).Value;
        else
        {
            saveSetting("http://www.3dbuzz.com/settings/curOnlineClass", 1, g_curOnlineClass);
            saveSetting("http://www.3dbuzz.com/settings/curCustomOnlineClass", -1, g_curCustomOnlineClass);
            g_curOnlineClass = 1;
            g_curCustomOnlineClass = -1;
            changeOnlineClassText();
            return;
        }
    }        
        
    loadURL(url, evnt, false);
}

//  END OF ONLINE CLASS MENU FUNCTIONS
/////////////////////////////////////////////////////////


//URL loading functions
function getServer()
{
    var server = 1;
    var url;// = getWebNavigation().currentURI.spec.toLowerCase();
    
    if(window._content.frames.length > 0)
    {
        if(window._content.main)
            url = window._content.main.location.href.toLowerCase();
        else if(window._content.frames[0].main)
            url = window._content.frames[0].main.location.href.toLowerCase();
    }
    else
        url = getWebNavigation().currentURI.spec.toLowerCase();

    if(url.indexOf("http://sv") == 0 && url.charAt(9) > '0' && url.charAt(9) <= '9')
        server = url.charAt(9)-'0';
    else
    {
        var dsource = RDF.GetDataSource("rdf:local-store");
        var res = dsource.GetTarget(g_RDF.GetResource("http://www.3dbuzz.com/settings/defaultServer"), g_nameRDF, true);
        if(res)
            server = parseInt(res.QueryInterface(Components.interfaces.nsIRDFLiteral).Value);
    }
        
    return server;
}

function isAt3DBuzz()
{
    var url = getWebNavigation().currentURI.spec.toLowerCase();
    
    if(url.indexOf("http://www.3dbuzz.com") == 0 ||
        url.indexOf("http://sv1.3dbuzz.com") == 0 ||
        url.indexOf("http://sv2.3dbuzz.com") == 0 ||
        url.indexOf("http://sv3.3dbuzz.com") == 0 ||
        url.indexOf("http://sv4.3dbuzz.com") == 0 
      )
      return true;
      
   return false;
}

function parseServer(url)
{
    if(url.indexOf("http://sv1.3dbuzz.com") != 0 &&
        url.indexOf("http://sv2.3dbuzz.com") != 0 && 
        url.indexOf("http://sv3.3dbuzz.com") != 0)
        return url;
        
    var urlStart = "http://sv";
    var urlEnd;
    urlEnd = url.substring(urlStart.length+1, url.length);
    url = urlStart + getServer() + urlEnd;
    
    return url;
}

function getRelativeURL(url)
{
    if(url.indexOf("http://sv1.3dbuzz.com/vbforum/") != 0 &&
        url.indexOf("http://sv2.3dbuzz.com/vbforum/") != 0 &&
        url.indexOf("http://sv3.3dbuzz.com/vbforum/") != 0)
        return parseServer(url);
    
    var generalStart = "http://sv3.3dbuzz.com/vbforum/";
        
    return url.substring(generalStart.length, url.length);
}

function areURLsEqual(url1, url2)
{
    if(url1.indexOf("http://sv") == 0)
        url1 = url1.substring(10, url1.length);
    if(url2.indexOf("http://sv") == 0)
        url2 = url2.substring(10, url2.length);
        
    return (url1 == url2);
}

function isFramedURL(url)
{
    url = url.toLowerCase();
    
    if(url == "http://www.3dbuzz.com" ||
        url == "http://www.3dbuzz.com/" ||
        url.indexOf("http://sv1.3dbuzz.com/index_sv.php?") == 0 ||
        url.indexOf("http://sv2.3dbuzz.com/index_sv.php?") == 0 ||
        url.indexOf("http://sv3.3dbuzz.com/index_sv.php?") == 0 ||
        url.indexOf("chrome://3dbuzz/content/frames.html") == 0 )
        return true;

    return false;
}

function loadURL(url, evnt, bLoadFrames)
{
    var browser = document.getElementById("content");
    var server = getServer();
    var urlToLoad = -1;
    var iLoadType = 0;
    var res;
    var mod1 = 0;
    var mod2 = 0;
    
    /*
        iLoadType:
            0 - Do nothing
            1 - Use current window
            2 - New foreground tab
            3 - New background tab
            
        See mouseConfig.xul and mouseConfig.js
        
        Also note that event.which - which returns the mouse button, returns the following numbers:
            0 - Left mouse button
            2 - Middle mouse button
            3 - Right mouse button
    */
    
    if(evnt != null)
    {
        if(evnt.altKey) //Don't do anything if ALT is pressed
            return;
        
        if(evnt.shiftKey && evnt.ctrlKey)
        {
            mod1 = 2;
            mod2 = 1;
        }
        else if(evnt.shiftKey)
            mod1 = 1;
        else if(evnt.ctrlKey)
            mod1 = 2;
        res = g_RDF.GetDataSource("rdf:local-store").GetTarget(
                                g_RDF.GetResource("http://www.3dbuzz.com/settings/mouse/button"+ 
                                                        evnt.which + mod1 + mod2
                                                  ),
                                g_nameRDF, true);
        if(res)
            iLoadType = parseInt(res.QueryInterface(Components.interfaces.nsIRDFLiteral).Value);
        else // Take care of "Ctrl + Shift" vs "Shift + Ctrl"
        {
            res = g_RDF.GetDataSource("rdf:local-store").GetTarget(
                                    g_RDF.GetResource("http://www.3dbuzz.com/settings/mouse/button"+ 
                                                            evnt.which + mod2 + mod1
                                                      ),
                                    g_nameRDF, true);
            if(res)
                iLoadType = parseInt(res.QueryInterface(Components.interfaces.nsIRDFLiteral).Value);
            else 
            { // use defaults
                if(evnt.shiftKey)
                    if(evnt.ctrlKey)
                        iLoadType = 2;
                    else
                        iLoadType = 3;
                else
                    iLoadType = 1;
            }
        }
    }
    else // evnt == null
        iLoadType = 1;
        
    if(iLoadType == 0)
        return; // Do Nothing

    url = parseServer(url.toLowerCase());
    
    // Longest I know of is gopher://.
    // Also give a berth of 2 characters
    if((url.indexOf("://") < 0 || url.indexOf("://") > 8) && url != "about:blank")
        url = "http://sv" + server + ".3dbuzz.com/vbforum/" + url;
        
    if(isFramedURL(url) || 
        url.indexOf("irc://") == 0) 
    {
        urlToLoad = url;
    }
    else if(!bLoadFrames && iLoadType == 1)
    {
        if(window._content.frames.length > 0)
        {
            urlToLoad = url;

            if(window._content.main)
            {
               window._content.main.location = urlToLoad;  //Special case
               return;
            }
            else if(window._content.frames[0].main)
            {
               window._content.frames[0].main.location = urlToLoad;
               return;
            }
            else
                urlToLoad = -1;
        }
    }/*
    else
    {
        var relURL = getRelativeURL(url);

        if(url.indexOf("http://") == 0 && areURLsEqual(relURL, url)) // URL is not relative to "http://svX.3dbuzz.com/vbforum/"
            urlToLoad = parseServer(url);
        else 
        {
            urlToLoad = -1;
            url = relURL;
        }
    }*/
    
    if(urlToLoad == -1)
    {
        urlToLoad = getFrameURL(url, bLoadFrames);
    }
    
    if(iLoadType == 3) //load in new background tab
    {
        var newTab = browser.addTab(urlToLoad);
    }
    else if(iLoadType == 2) // load in foreground tab
    {
        var newTab = browser.addTab(urlToLoad);
        browser.selectedTab = newTab;
    }
    else // load in current window
        browser.loadURI(urlToLoad);
}

function getFrameURL(url, bLoadFrames)
{
    var urlToLoad = url;
    var iBrowsingWithFrames = 1;
    
    // initialize iBrowsingWithFrames properly
    iBrowsingWithFrames = g_RDF.GetDataSource("rdf:local-store").GetTarget(
                                        RDF.GetResource("http://www.3dbuzz.com/settings/browseWithFrames"), g_nameRDF, true);
    if(iBrowsingWithFrames)
        iBrowsingWithFrames = parseInt(iBrowsingWithFrames.QueryInterface(Components.interfaces.nsIRDFLiteral).Value);
    else
        iBrowsingWithFrames = 1;
        
    //alert(iBrowsingWithFrames);
    
    /*
        If set to frameless browsing, and still asked to load frames, 
        use both side and both frames
    */
    if(bLoadFrames && iBrowsingWithFrames == 4)
        iBrowsingWithFrames = 1;    
    
    if(iBrowsingWithFrames != 4)
    {
        urlToLoad = "chrome://3dbuzz/content/frames.html?id=" + iBrowsingWithFrames + 
                        "&server=" + getServer() +
                        "&url=" + url;
    }

    return urlToLoad;
}

//searching related functions

function DoSearch(evnt)
{
    var txt = document.getElementById("search-textbox").value;
    if(!txt || txt.length < 1)
    {
        MsgBox(document.getElementById("stringRes").getString("alertNoTextToSearch"));
        return;
    }
        
    var url = 
         "search.php?forumchoice=-1&searchdate=-1&beforeafter=after&action=simplesearch&sortby=lastpost&sortorder=descending&query=" 
                + txt;
	loadURL(url,evnt,false);
}

function DoUsernameSearch(evnt)
{
    var txt = document.getElementById("search-textbox").value;
    if(!txt || txt.length < 1)
    {
        MsgBox(document.getElementById("stringRes").getString("alertNoTextToSearch"));
        return;
    }
    var url =
         "search.php?forumchoice=-1&searchdate=-1&beforeafter=after&action=simplesearch&sortby=lastpost&sortorder=descending&query=&exactname=yes&searchuser="
                + txt;
    loadURL(url, evnt, false);
}

function adjustRecentSearchList()
{
    var rs_dsource = initDataSource("search-popup");
    
    var entries = g_RDFC.GetElements();
    var totalSearchEntries = g_RDFC.GetCount();
    var es = new Array(totalSearchEntries);
    var values = new Array(totalSearchEntries);
    var i = 0;
    
    while(entries.hasMoreElements())
    {
        var entry = entries.getNext();
        entry = entry.QueryInterface(Components.interfaces.nsIRDFResource);
        var val = rs_dsource.GetTarget(entry, g_nameRDF, true);
        val = val.QueryInterface(Components.interfaces.nsIRDFLiteral);
        
        es[i] = entry;
        values[i] = val;
        i++;
    }
    
    //Now remove all the elements
    i = 0;
    while(i < totalSearchEntries)
    {
        rs_dsource.Unassert(es[i], g_nameRDF, values[i], true);
        g_RDFC.RemoveElementAt(1, true);
        i++;
    }
    
    //Add only the required number back
    for(i = 0; i < g_maxSearches && i < totalSearchEntries; i++)
    {
        var newnode = g_RDF.GetResource("http://www.3dbuzz.com/recentsearch"+(g_RDFC.GetCount()+1));
        rs_dsource.Assert(newnode, g_nameRDF, values[i], true);
        g_RDFC.InsertElementAt(newnode,i+1,true);
    }
    
    //Commit changes to disk
    rs_dsource.QueryInterface(Components.interfaces.nsIRDFRemoteDataSource).Flush();
}

function onSearchboxKeyPress(e)
{
    if(e.which == 13) //Enter was pressed
    {
        var rs_dsource = initDataSource("search-popup");
        
        var enteredValue = document.getElementById("search-textbox").value;
        
        var newnode=g_RDF.GetResource("http://www.3dbuzz.com/researchsearch"+(g_RDFC.GetCount()+1));
        var newvalue=g_RDF.GetLiteral(enteredValue);
  
        // Prevent adding duplicate search entries
        if(!hasSearchItem(rs_dsource, newvalue.Value))
        {
            rs_dsource.Assert(newnode,g_nameRDF,newvalue,true);
            g_RDFC.InsertElementAt(newnode,1,true);
  
            rs_dsource.QueryInterface(Components.interfaces.nsIRDFRemoteDataSource).Flush();
        }
        DoSearch(e);
        adjustRecentSearchList();
    }
}

/*
    The built in function HasAssertion() of nsIRDFDataSource would've been better, but the knowledge of using it doesn't rest with me.
*/
function hasSearchItem(rs_dsource, tVal)
{
    var entries = g_RDFC.GetElements();
    
    while(entries.hasMoreElements())
    {
        var entry = entries.getNext();
        entry = entry.QueryInterface(Components.interfaces.nsIRDFResource);
        var val = rs_dsource.GetTarget(entry, g_nameRDF, true);
        val = val.QueryInterface(Components.interfaces.nsIRDFLiteral);
        
        if(tVal == val.Value)
            return true;
    }
    
    return false;
}

function createSearchboxMenu(menuPopup)
{
    deleteItems(menuPopup);
    var dsource = g_RDF.GetDataSource("rdf:local-store");
    
    for(var i = 1; i <= g_maxSearches; i++)
    {
        var text = dsource.GetTarget(g_RDF.GetResource("http://www.3dbuzz.com/recentsearch"+i), g_nameRDF, true);
        
        if(!text)
            break;
            
        text = text.QueryInterface(Components.interfaces.nsIRDFLiteral).Value;
        
        var menuitem = buzz3d_createMenuItem(menuPopup, text);
        menuitem.setAttribute("id", "http://www.3dbuzz.com/recentsearch"+i);
    }
    if(i == 1) // No items were added
    {
        menuitem = buzz3d_createMenuItem(menuPopup, "No searches available");
        menuitem.setAttribute("disabled", "true");
    }
}

function on3DBuzz(evnt)
{
    loadURL("buzz_index.php", evnt, false);
}

/*
    Removes leading and trailing spaces from a string.
    
    It could perhaps be optimized more.
*/
function trim(str)
{
    var start = 0;
    var end = str.length;
    var i;
    
    for(i = 0; i < str.length; i++)
        if(str[i] == " ")
            start++;
        else
            break;
            
    for(i = str.length-1; i >= start; i--)
        if(str[i] == " ")
            end--;
        else
            break;
            
    return str.substring(start, end);
}

function onIRC(evnt)
{
    var dsource = g_RDF.GetDataSource("rdf:local-store");
    var opt = dsource.GetTarget(RDF.GetResource("http://www.3dbuzz.com/settings/ircOption"), g_nameRDF, true);
    if(opt)
        opt = parseInt(opt.QueryInterface(Components.interfaces.nsIRDFLiteral).Value);
    else
        opt = 0;
    
    switch(opt)
    {
        case 0:
            loadURL("irc_framesplit.php?index.php", evnt, false);
        break;
        
        case 1:
            var txt = dsource.GetTarget(RDF.GetResource("http://www.3dbuzz.com/settings/ircaddress"), g_nameRDF, true);
            if(txt)
                txt = txt.QueryInterface(Components.interfaces.nsIRDFLiteral).Value;
            else
                break;
                
            var URLs = txt.split(";");
            for(var i = 0; i < URLs.length; i++)
            {
                URLs[i] = trim(URLs[i]);
            }
            loadURL("irc://irc.3dbuzz.com/3dbuzz", evnt, false);
        break;
        
        case 2:
            var txt = dsource.GetTarget(RDF.GetResource("http://www.3dbuzz.com/settings/ircclient"), g_nameRDF, true);
            if(txt)
                txt = txt.QueryInterface(Components.interfaces.nsIRDFLiteral).Value;
            else
                txt = 0;

                
            try
            {
                var aFile = Components.classes["@mozilla.org/file/local;1"].createInstance(Components.interfaces.nsILocalFile);
                aFile.initWithPath(txt);
                aFile.launch();
            }
            catch(e)
            {
                MsgBox(e);
            }
        break;
    }
}

function onViewNewPosts(evnt)
{
    loadURL("search.php?s=&action=getnew", evnt, false);
}

function onAdvancedSearch(evnt)
{
    loadURL("search.php", evnt, false);
}

function onSearchboxListCmd(evnt)
{
    var txt = evnt.target.getAttribute("label");
    document.getElementById("search-textbox").setAttribute("value", txt);
    DoSearch(evnt);
}

function on3DBuzzReload()
{
    var t = getWebNavigation();
    
    if(window._content.frames.length > 0)
    {
        if(window._content.main)
            window._content.main.location.reload()
        else if(window._content.frames[0].main)
            window._content.frames[0].main.location.reload();
    }
    else
        getWebNavigation().document.location.reload();
}

function onUpdateFrames(evnt)
{
    var url = -1;
    
    if(evnt.altKey)
        return;
    
    if(window._content.frames.length > 0)
    {
        if(window._content.main)
            url = window._content.main.location.href.toLowerCase();
        else if(window._content.frames[0].main)
            url = window._content.frames[0].main.location.href.toLowerCase();
    }
    else
        url = getWebNavigation().currentURI.spec.toLowerCase();
        
    if(url != -1)
        document.getElementById("content").loadURI(getFrameURL(url, false));
}

function onAddFrames(evnt)
{
    var url = getWebNavigation().currentURI.spec;
      
    if(evnt.altKey || isFramedURL(url)) // already has frames
        return;

    loadURL(url, null, true);
}

function onRemoveFrames(evnt)
{
    var url = -1;
    
    if(evnt.altKey)
        return;

    if(window._content.frames.length > 0)
    {
        if(window._content.main)
            url = window._content.main.location.href.toLowerCase();
        else if(window._content.frames[0].main)
            url = window._content.frames[0].main.location.href.toLowerCase();
    }
        
    if(url != -1)
        document.getElementById("content").loadURI(url);
}


function onOptions()
{
    window.openDialog("chrome://3dbuzz/content/options.xul","options", "chrome,centerscreen,modal=yes,dialog=yes");
}

function toggleToolbar(menuItem)
{
    var comp = document.getElementById("3dbuzz-toolbar");
    var cl = comp.getAttribute("class");
    
    if(cl != "normalToolbar")
    {
        comp.setAttribute("class", "normalToolbar");
        menuItem.setAttribute("checked", "true");
    }
    else
    {
        comp.setAttribute("class", "hiddenbutton");
        menuItem.setAttribute("checked", "false");
    }
}

function onToolbarClicked(evnt)
{
    if(evnt.altKey)
    {
        if(evnt.shiftKey || evnt.ctrlKey)
            window.openDialog("chrome://3dbuzz/content/alternateOptions.xul","Options", 
                                "chrome,centerscreen,modal=yes,dialog=yes");
        else
            onOptions();
    }
}

window.addEventListener("load", onInitToolbar, true);
