package Netpkg; 

# Copyright Jean-Philippe Guillemin <jp.guillemin@free.fr>. This program is free software; you can redistribute
# it and/or modify it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or (at your option)
# any later version. Please take a look at http://www.gnu.org/copyleft/gpl.htm
#
# Netpkg is a tool for easily install or upgrade packages via the network. With Netpkg,
# you can make a minimal installation of Zenwalk Linux and install/upgrade just the
# packages you need most.
#
#

use strict;
use POSIX;
use LWP::UserAgent ();
use LWP::MediaTypes qw(guess_media_type media_suffix);
use URI ();
use HTTP::Date ();
use POSIX qw/strftime/;
use Locale::gettext;
use Encode;
use Unicode::Normalize; #for accent-insensitive sort
use Storable; 
use threads ('yield',
             'stack_size' => 64*4096,
             'exit' => 'threads_only',
             'stringify');
use threads::shared;
my $semaphorus : shared = 0;
my $ZenStyle = 1 ;

# Internationalization
setlocale(LC_ALL, "");
textdomain("netpkg");

####################################################################
############################ CONSTANTS  ############################
####################################################################

use constant CATEGORY => 0;
use constant AVAILABLE_PKG => 1;
use constant SORTSTRING => 2;
use constant SIZE_COMPRESSED => 3;
use constant SIZE_UNCOMPRESSED => 4;
use constant DEPENDENCIES => 5;
use constant AVAILABLE_VERSION => 6;
use constant AVAILABLE_BUILD => 7;
use constant DESCRIPTION => 8;
use constant INSTALLED_PKG => 9;
use constant INSTALLED_VERSION => 10;
use constant INSTALLED_BUILD => 11;
use constant STATUS => 12;
use constant SELECTED_AS_PKG => 13;
use constant SELECTED_AS_DEP => 14;
use constant BLACKLISTED => 15;
use constant PATH => 16;
use constant USED_BY => 17;
use constant INSTALLED_DEPENDENCIES => 18;
use constant MISSING_DEPENDENCIES => 19;
use constant CATEGORY_ICON => 20;
use constant LOCAL_PKG => 21;
use constant BUFFER => 30;

use constant STATUS_OK => 0;
use constant STATUS_DOWNGRADED => -1;
use constant STATUS_UPDATED => 1;
use constant STATUS_NEW => 2;
use constant STATUS_UNAVAILABLE => -2;

use constant PROCESS_PKGS_ONLY => 1;
use constant PROCESS_PKGS_AND_DEPS => 2;

use constant NONE => 0b00000000;
use constant ALL => 0b00001111;
use constant SELECTED => 0b00011111;
use constant NEW => 0b00001000;
use constant NOT_NEW => 0b00000111;
use constant UPDATED => 0b00000100;
use constant NOT_UPDATED => 0b00001011;
use constant INSTALLED => 0b00000010;
use constant NOT_INSTALLED => 0b00001101;
use constant DOWNGRADED => 0b00000001;
use constant NOT_DOWNGRADED => 0b00001110;
# MODIFIED = UPDATED | DOWNGRADED
use constant MODIFIED => 0b00000101;
use constant NOT_MODIFIED => 0b00001010;
use constant ORPHANS => 0b00010000;
use constant NOT_ORPHANS => 0b00001111;

use constant ICON_COLUMN => 0;
use constant SOFT_COLUMN => 1;
use constant AVAILABLE_PKG_COLUMN => 2;
use constant INSTALLED_PKG_COLUMN => 3;
use constant SELECTED_PKG_COLUMN => 4;

use constant WITH_PKG_INFOS => 0;
use constant WITH_SIZES => 1;

####################################################################
sub new { 
	my $self = {};
	bless ($self);	
	return $self;
}

####################################################################
# $Netpkg->GetMirrors($Hash,$configfile );
sub GetMirrors {
	my $self = shift; 
	my ($Hash,$configfile) = @_ ;
	
    open(FH, "<$configfile");
    
    while(<FH>) {
		chomp($_);    	
		if ( $_ =~ /(\s*Internet_mirror\s*=\s*)(.*)/ ){   
			next if ( $_ =~ /^[ \t]*$/ ) ;
			$Hash->{$2} = 1 ;
		} 			
	}
	close(FH);	
	
	open(FH, "<$self->{Netpkg_dir}/last_mirror");
	my $lastmirror;
	while(<FH>) {
		chomp($_);    
		next if ( $_ =~ /^[ \t]*$/ ) ;	
		$Hash->{$_} = 1 ;
		$lastmirror = $_ ;
	}
	$Hash->{"$lastmirror"} = 0 ;
	close(FH);	
	
	return 0; 
}

####################################################################
# $Netpkg->SaveMirrors($Hash);
sub SaveMirrors {
	my $self = shift; 
	my $Hash = shift; 
	open(FH, ">$self->{Netpkg_dir}/last_mirror");
	for $_ ( sort { $Hash->{$a} cmp $Hash->{$b} } keys %{$Hash} ) { 
		print FH "$_\n";
	}
	
	close(FH);	
	
	return 0; 
}

####################################################################
# $Netpkg->SaveDB(\%db, $dbname);
sub SaveDB {
	my $self = shift; 
	my %db = %{$_[0]};
	my $dbname = $_[1];
	sysopen(DF, "$self->{Netpkg_dir}/$dbname", O_RDWR|O_CREAT, 0666) ;
	store(\%db, "$self->{Netpkg_dir}/$dbname");
	
	return 0; 
}

####################################################################
# $Netpkg->LoadDB(\$result, $dbname);
sub LoadDB {
	my $self = shift; 
	my $result = shift; 
	my $dbname = shift; 
	my %newdb = [];
	my $key ;
	if (-e "$self->{Netpkg_dir}/$dbname" ) {
		%newdb = %{ retrieve("$self->{Netpkg_dir}/$dbname") } ; 
		$$result = 0 ;
	}else{
		$$result = -1;
	}
	return \%newdb;
}

# Will parse the repository data to fill %db related fields
# $href = $Netpkg->BuildDB
sub BuildDB {
	my $self = shift; 
	my %newdb = [];
	my $soft;
	my $desc;
	my $package;
	
	my @keys = split ( / / , $self->{Black_list}); 
	my %Blacklist;
	foreach my $key (@keys) { $Blacklist{$key} = 1 } ;

	open(FH, "$self->{Netpkg_dir}/PACKAGES.TXT") || return -1 ; 
	while (<FH>) {
		
		Gtk2->main_iteration while ( Gtk2->events_pending );
		
		chomp $_;
		# print "$_";
		if ($_ =~ /PACKAGE NAME:\s*([^\s]*)$/) { 
			$package="$1"; 
			$package =~ /^(.*)-([^-]*)-[^-]*-([^-]*).t[glx]z$/;
			$soft="$1"; 
			@{$newdb{$soft}}[AVAILABLE_PKG] = "$package";	
			@{$newdb{$soft}}[AVAILABLE_VERSION] = "$2";
			@{$newdb{$soft}}[AVAILABLE_BUILD] = "$3";		
			next;
		}
		if ($_ =~ /PACKAGE LOCATION:\s*\.?\/?([^\s]*)\s*$/) { 
			@{$newdb{$soft}}[PATH] = "$1";	

			if ( $ZenStyle = 1 ) {
				if ( 
				@{$newdb{$soft}}[PATH] =~ /^l$/ || 
				@{$newdb{$soft}}[PATH] =~ /.*\/l$/ 
				) {
					@{$newdb{$soft}}[CATEGORY] = decode('utf8',gettext("Runtime libraries"));
					@{$newdb{$soft}}[CATEGORY_ICON] = "gnome-other";
				}elsif ( 
				@{$newdb{$soft}}[PATH] =~ /^d$/ ||
				@{$newdb{$soft}}[PATH] =~ /.*\/d$/ 
				) {
					@{$newdb{$soft}}[CATEGORY] = decode('utf8',gettext("Development libraries"));
					@{$newdb{$soft}}[CATEGORY_ICON] = "development";
				}elsif ( 
				@{$newdb{$soft}}[PATH] =~ /^x$/ ||
				@{$newdb{$soft}}[PATH] =~ /.*\/x$/ 
				) {
					@{$newdb{$soft}}[CATEGORY] = decode('utf8',gettext("Xorg"));
					@{$newdb{$soft}}[CATEGORY_ICON] = "xorg";
				}elsif ( 
				@{$newdb{$soft}}[PATH] =~ /^ap$/ ||
				@{$newdb{$soft}}[PATH] =~ /.*\/ap$/
				) {
					@{$newdb{$soft}}[CATEGORY] = decode('utf8',gettext("Console Applications"));
					@{$newdb{$soft}}[CATEGORY_ICON] = "utilities-terminal";
				}elsif ( 
				@{$newdb{$soft}}[PATH] =~ /^xap$/ ||
				@{$newdb{$soft}}[PATH] =~ /.*\/xap$/
				) {
					@{$newdb{$soft}}[CATEGORY] = decode('utf8',gettext("Graphical Applications"));
					@{$newdb{$soft}}[CATEGORY_ICON] = "applications-graphics";
				}elsif ( 
				@{$newdb{$soft}}[PATH] =~ /.*\/g$/ 
				) {
					@{$newdb{$soft}}[CATEGORY] = decode('utf8',gettext("Gnome Desktop"));
					@{$newdb{$soft}}[CATEGORY_ICON] = "gnome-logo";
				}elsif ( 
				@{$newdb{$soft}}[PATH] =~ /.*\/kde$/ 
				) {
					@{$newdb{$soft}}[CATEGORY] = decode('utf8',gettext("KDE Desktop"));
					@{$newdb{$soft}}[CATEGORY_ICON] = "konqueror";
				}elsif ( 
				@{$newdb{$soft}}[PATH] =~ /^a$/ || 
				@{$newdb{$soft}}[PATH] =~ /.*\/a$/ 
				) {
					@{$newdb{$soft}}[CATEGORY] = decode('utf8',gettext("System"));
					@{$newdb{$soft}}[CATEGORY_ICON] = "applications-system";
				}elsif ( 
				@{$newdb{$soft}}[PATH] =~ /^locale$/ || 
				@{$newdb{$soft}}[PATH] =~ /.*\/locale$/ 
				) {
					@{$newdb{$soft}}[CATEGORY] = decode('utf8',gettext("Localization"));
					@{$newdb{$soft}}[CATEGORY_ICON] = "preferences-desktop-locale";
				}elsif ( 
				@{$newdb{$soft}}[PATH] =~ /^games$/ || 
				@{$newdb{$soft}}[PATH] =~ /.*\/games$/ 
				) {
					@{$newdb{$soft}}[CATEGORY] = decode('utf8',gettext("Games"));
					@{$newdb{$soft}}[CATEGORY_ICON] = "applications-games";
				}elsif ( 
				@{$newdb{$soft}}[PATH] =~ /^n$/ || 
				@{$newdb{$soft}}[PATH] =~ /.*\/n$/ ||
				@{$newdb{$soft}}[PATH] =~ /^s$/ || 
				@{$newdb{$soft}}[PATH] =~ /.*\/s$/ 
				) {
					@{$newdb{$soft}}[CATEGORY] = decode('utf8',gettext("Network"));
					@{$newdb{$soft}}[CATEGORY_ICON] = "applications-internet";
				}else{
					@{$newdb{$soft}}[CATEGORY] = decode('utf8',gettext("Misc"));
					@{$newdb{$soft}}[CATEGORY_ICON] = "applications-other";
				}
				
				#if ( @{$newdb{$soft}}[PATH] =~ /^extra.*/ ) {
				#	@{$newdb{$soft}}[CATEGORY] = "Extra ".@{$newdb{$soft}}[CATEGORY] ;
				#}
				
				
				@{$newdb{$soft}}[SORTSTRING] = @{$newdb{$soft}}[CATEGORY]."/".$package;
				
			}else{
				@{$newdb{$soft}}[CATEGORY] = @{$newdb{$soft}}[PATH] ;
				if ( @{$newdb{$soft}}[PATH] =~ /^extra.*/ ) {
					@{$newdb{$soft}}[SORTSTRING] = "y".@{$newdb{$soft}}[CATEGORY]."/".$package;
				}else{
					@{$newdb{$soft}}[SORTSTRING] = @{$newdb{$soft}}[CATEGORY]."/".$package;
				}
			}	
			
			next;
		}	
		if ($_ =~ /PACKAGE SIZE \(c.*:\s*(.*)\s*$/) { 
			@{$newdb{$soft}}[SIZE_COMPRESSED] = "$1";	
			next;						
		}			
		if ($_ =~ /PACKAGE SIZE \(u.*:\s*(.*)\s*$/) { 
			@{$newdb{$soft}}[SIZE_UNCOMPRESSED] = "$1";	
			next;							
		}														
		if ($_ =~ /PACKAGE REQUIRED:\s*([^\s]*)\s*$/) { 
			@{$newdb{$soft}}[DEPENDENCIES] = "$1";	
			@{$newdb{$soft}}[DEPENDENCIES] =~ s/,/ /g ;
			next;				
		}
		# @{$newdb{$soft}}[DESCRIPTION] = "" if ( undef @{$newdb{$soft}}[DESCRIPTION] ) ;	
		while ($_ =~ /\Q${soft}\E:(.*)/) {
			$desc = $1;
			chomp $desc;
			$desc =~ s/[\ \t]{2,}/ /g ;						
			@{$newdb{$soft}}[DESCRIPTION] = @{$newdb{$soft}}[DESCRIPTION]." ".$desc;
			$_ = <SH>;
		}
		
		# Until we check local packages, everything is NEW
		@{$newdb{$soft}}[INSTALLED_VERSION] = decode('utf8',gettext("not installed"));
		@{$newdb{$soft}}[STATUS] = STATUS_NEW ;
		
		# SELECTED PACKAGE BOOLEAN	
		@{$newdb{$soft}}[SELECTED_AS_PKG] = 0; 
		# SELECTED DEP BOOLEAN	
		@{$newdb{$soft}}[SELECTED_AS_DEP] = 0; 
		# DELETING REVERSE DEPS
		@{$newdb{$soft}}[USED_BY] = "";
		
		if ( defined $Blacklist{$soft} ) {
			@{$newdb{$soft}}[BLACKLISTED] = 1 ;
		}
			
	}
	close(FH); 	

	return \%newdb;
}


####################################################################
# Will check installed packages to fill %db related fields
# $Netpkg->CheckInstalledPkgs(\%db)
sub CheckInstalledPkgs {
	my $self = shift; 
	my %db = %{$_[0]};
	my $fh;
	my ($package,$soft,$version,$build,$desc,$meta);

	my %localdb = [];
	opendir(DH, "$self->{Package_logs}") || return -1 ;
	foreach (sort readdir(DH)) {
		
		$meta = $_;
		open(FH, "<$self->{Package_logs}/$meta") || return -1 ;		
		next if ( length ($meta) <= 3 );
		
		$meta =~ /^(.*)-([^-]*)-[^-]*-([^-]*)$/;
		$soft="$1"; 
		$version="$2";
		$build="$3";
		
		while(<FH>) {
			if ($_ =~ /^PACKAGE LOCATION:.*(\s|\/)([^\/]*t[glx]z)\s*$/) { 				
				@{ $db{$soft} }[INSTALLED_PKG] = "$2";
				last;							
			}					
		}	
		
		@{ $db{$soft} }[INSTALLED_VERSION] = "$version";	
		@{ $db{$soft} }[INSTALLED_BUILD] = "$build";			
		
		if (defined @{$db{$soft}}[AVAILABLE_PKG] ) {

			# STATUS ( -1:downgraded ; 0:installed ; 1:updated )
			@{$db{$soft}}[STATUS] = &ZW::Netpkg::PVfilter::test_package("@{$db{$soft}}[AVAILABLE_PKG]", "@{$db{$soft}}[INSTALLED_PKG]") ;					

		}else{
			@{$db{$soft}}[STATUS] = STATUS_OK;
			@{$db{$soft}}[AVAILABLE_VERSION] = "not on mirror";	
			@{$db{$soft}}[CATEGORY] = decode('utf8',gettext("Local"));	
			@{$db{$soft}}[CATEGORY_ICON] = "folder";
			@{$db{$soft}}[SORTSTRING] = "z"."/".@{$db{$soft}}[CATEGORY]."/".$package;
			
			seek(FH, 0, 0);
			while(<FH>) {
				if ($_ =~ /^COMPRESSED PACKAGE SIZE:\s*(.*)\s*$/) { 
					@{$db{$soft}}[SIZE_COMPRESSED] = "$1";	
					next;				
				}			
				if ($_ =~ /^UNCOMPRESSED PACKAGE SIZE:\s*(.*)\s*$/) { 
					@{$db{$soft}}[SIZE_UNCOMPRESSED] = "$1";
					next;							
				}	
				if ($_ =~ /^PACKAGE LOCATION:\s*(.*)\s*$/) { 
					my $fullpath = "$1";
					my $regex = '^'.$self->{Local_repository}.'/(.*)/[^/]*.t[glx]z\s*$';
					$regex =~ s/\//\\\//g ;
					if ($fullpath =~ /$regex/ ) {
						@{$db{$soft}}[PATH] = "$1";
					}else{
						@{$db{$soft}}[PATH] = "local";
					}
					next;							
				}					
				
				# @{$db{$soft}}[DESCRIPTION] = "" if ( undef @{$db{$soft}}[DESCRIPTION] ) ;	
				while ($_ =~ /\Q${soft}\E:(.*)/) {
					$desc = $1;
					chomp $desc;
					$desc =~ s/[\ \t]{2,}/ /g ;									
					@{$db{$soft}}[DESCRIPTION] = @{$db{$soft}}[DESCRIPTION]." ".$desc;
					$_ = <FH>;
				}
			}
		}	
		close(FH);	
	}
	closedir(DH); 
	return 0;
}


####################################################################
# $Netpkg->GetFilters();
# $filter = $Netpkg->GetFilters()
sub GetFilters {
	my $self = shift; 
	
    open(FH, "<$self->{Netpkg_dir}/last_filter") or return -1 ;
    binmode FH;	
	my $Filter = <FH> ;
	chomp($Filter);    
	close(FH);
	
	return $Filter;
}

####################################################################
# $Netpkg->SaveFilters($Filter);
sub SaveFilters {
	my $self = shift; 
	my $Filter = shift; 
	open(FH, ">$self->{Netpkg_dir}/last_filter");
	binmode FH;	
	print FH "$Filter";
	close(FH);	
	
	return 0; 
}

####################################################################
# $Netpkg->Configure($option, $default, $configfile);
sub Configure {
	my $self = shift; 
	
	# default value
	$self->{"$_[0]"} = $_[1];
	
    open(FH, "<$_[2]");
    while (<FH>) {	
		chomp($_);

		if ( $_ =~ /^\s*${_[0]}\s*=\s*(.*)$/ ){
			$self->{"$_[0]"} = $1;
			last;
		}
	}	
	close(FH);
	return 0; 
}

####################################################################
# $Netpkg->SaveConf($option, $newvalue, $configfile);
sub SaveConf {
	my $self = shift; 
	my $buffer;
	my $found = 0;
    open(FH, "<$_[2]");
    while (<FH>) {	
		chomp($_);

		if ( $_ =~ /^\s*${_[0]}\s*=\s*.*$/ ){
			$buffer = $buffer."$_[0] = $_[1]\n";
			$found = 1;
		}else{
			$buffer = $buffer."$_\n";
		}
	}	
	if ( $found == 0 ) {
		$buffer = $buffer."\n$_[0] = $_[1]\n";
	}
	close(FH);
	open(FH, ">$_[2]");
	print FH $buffer;
	close(FH);	
	return 0; 
}

####################################################################
# $Netpkg->Get ($Url, $Object, $Mode, $Data, $File, $ProgressBar, $Start, $Range);
sub Get {
	my $self = shift;
	my ($Url, $Object, $Mode, $Data, $File, $ProgressBar, $Start, $Range) = @_; 
	
	# Normalize URL
	$Url =~ s/\/\//\//g ;
	$Url =~ s/^[htp:]*\/*(.*)/http:\/\/$1/ ;

	# Create an LWP UserAgent
	my $Browser = LWP::UserAgent->new;

	# Setup authentication details, if necessary
	if($self->{Proxy_Socket} ne '') {
		my $ProxyUrl = "http://"."$self->{Proxy_User}".":"."$self->{Proxy_Password}"."@"."$self->{Proxy_Socket}";
		$Browser->proxy(['http', 'ftp'], $ProxyUrl);
	}
	# if ($Browser->is_protocol_supported("ftp")) {print "ftp ok\n"}
	my $result = $Browser->head($Url);
	if ( ! $result->is_success ) {
		return -1 ;
	}
	
	$ProgressBar->set_text(decode('utf8',gettext("Loading"))." ".$Object);
	my $remote_headers = $result->headers;
	my $total_size = $remote_headers->content_length;
	my $KBtotal_size = int($total_size / 1024) ;
	my $iterate = 0 ;
	if($Mode) {
		$Browser->timeout(20);
		open(FH, ">$File") or die "Cannot open file $File : $!\n";
		binmode FH;	
		my $size;	
		my $res = $Browser->request(HTTP::Request->new(GET => $Url), sub {
				my ($chunk, $res) = @_;
				print FH $chunk;
				$size += length($chunk);
				$iterate++ ;
				if ( $iterate % 20 == 0 ){
					# print "$iterate ";
					my $Progress = $Start + $Range * $size / $total_size ;
					my $percent = int($size / $total_size * 100) ;
					if ($Progress >= 1) { $Progress = 1 }
					$ProgressBar->set_fraction($Progress);
					$ProgressBar->set_text(decode('utf8',gettext("Loading"))." ".$Object." (".$KBtotal_size." kB) ".$percent." %");
					Gtk2->main_iteration while ( Gtk2->events_pending );  
				}			
			});		
		close FH;
	}else{
		$Browser->timeout(20);
		my $res = $Browser->request(HTTP::Request->new(GET => $Url), sub {
				my($chunk, $res) = @_;
				${$Data} .= $chunk;
				$iterate++ ;
				if ( $iterate % 10 == 0 ){	
					# print "$iterate ";			
					my $len= length(${$Data});
					my $Progress = $Start + $Range * ($len / $total_size) ;
					my $percent = int(length(${$Data}) / $total_size * 100) ;
					if ($Progress >= 1) { $Progress = 1 }
					$ProgressBar->set_fraction($Progress);
					$ProgressBar->set_text(decode('utf8',gettext("Loading"))." ".$Object." (".$KBtotal_size." kB) ".$percent." %");
					Gtk2->main_iteration while ( Gtk2->events_pending );  	
				}		
			});
	}
	
	return 0;	
}	

# Called inside a thread to execute a system command
# the shared semaphor is set to '1' when system processing ends
####################################################################
# $Netpkg->Execute($command)
sub Execute {
	my $command = shift;
	threads->yield();
	$semaphorus = 0;	
	`$command`;
    if ($? == -1) {
    	$semaphorus = 1;
		threads->exit(-1); 
	} else {
		$semaphorus = 1;
		threads->exit(0); 
	} 
}

####################################################################
# $Netpkg->Install( $File );
sub Install {
	my $self = shift; 
	my $package = shift;
	my $command = "/sbin/installpkg $package >/dev/null" ; 

    #print "/sbin/installpkg $package\n";
    
    # initialize the shared semaphor to '0', then wait for the thread to set it to '1'
    $semaphorus = 0;
	my $thr = threads->new(\&Execute, $command);
	while ($semaphorus == 0) {
		Gtk2->main_iteration while ( Gtk2->events_pending  );  
	} 
	my $date = strftime('%a %b %d %H:%M:%S %Y',localtime); 
	open(FH, ">>$self->{Logfile}");
	print FH "[I] $package $date\n";
	close(FH);
	return 0;
}

####################################################################
# $Netpkg->Update( $File );
sub Update {
	my $self = shift; 
	my $package = shift;
	my $command = "/sbin/upgradepkg $package > /dev/null";
    #print "/sbin/upgradepkg --reinstall $package\n";
    
    # initialize the shared semaphor to '0', then wait for the thread to set it to '1'
    $semaphorus = 0;
	my $thr = threads->new(\&Execute, $command);
	while ($semaphorus == 0) {
		Gtk2->main_iteration while ( Gtk2->events_pending  );  
	}
	my $date = strftime('%a %b %d %H:%M:%S %Y',localtime); 
	open(FH, ">>$self->{Logfile}");
	print FH "[U] $package $date\n";
	close(FH);
	return 0;
}

####################################################################
# $Netpkg->Reinstall( $File );
sub Reinstall {
	my $self = shift; 
	my $package = shift;
	my $command = "/sbin/upgradepkg --reinstall $package >/dev/null";
    #print "/sbin/upgradepkg --reinstall $package\n";
    
    # initialize the shared semaphor to '0', then wait for the thread to set it to '1'
    $semaphorus = 0;
	my $thr = threads->new(\&Execute, $command);
	while ($semaphorus == 0) {
		Gtk2->main_iteration while ( Gtk2->events_pending  );  
	}
	my $date = strftime('%a %b %d %H:%M:%S %Y',localtime); 
	open(FH, ">>$self->{Logfile}");
	print FH "[I] $package $date\n";
	close(FH);
	return 0;
}

####################################################################
# $Netpkg->Remove( $soft );
sub Remove {
	my $self = shift; 
	my $package = shift;
	my $command = "/sbin/removepkg $package >/dev/null";
    #print "/sbin/removepkg $package\n";
    
    # initialize the shared semaphor to '0', then wait for the thread to set it to '1'
    $semaphorus = 0;
	my $thr = threads->new(\&Execute, $command);
	while ($semaphorus == 0) {
		Gtk2->main_iteration while ( Gtk2->events_pending  );  
	}
	my $date = strftime('%a %b %d %H:%M:%S %Y',localtime); 
	open(FH, ">>$self->{Logfile}");
	print FH "[R] $package $date\n";
	close(FH);
	return 0;
}

####################################################################
# $Netpkg->ProcessDeps(\%db);
sub ProcessDeps {
	my $self = shift; 
	my %db = %{$_[0]};
	my $soft; 
	my $dep; 
	for $soft ( keys %db ) {
		if ( defined $db{$soft} ) {	
			@{$db{$soft}}[USED_BY] = "";
			@{$db{$soft}}[INSTALLED_DEPENDENCIES] = "";
			@{$db{$soft}}[MISSING_DEPENDENCIES] = "";			
		}
	}
	for $soft ( keys %db ) {
		if ( defined $db{$soft} ) {
			my @Deps = split (/ /, @{$db{$soft}}[DEPENDENCIES]);	
			foreach $dep (@Deps) {
				if ( defined $db{$dep} ) {			
					if ( @{$db{$soft}}[STATUS] != STATUS_NEW ) {
						@{$db{$dep}}[USED_BY] .= " $soft" ; 
					}
					if ( @{$db{$dep}}[STATUS] == STATUS_OK ) {
						@{$db{$soft}}[INSTALLED_DEPENDENCIES] .= " $dep" ;
					}else{
						@{$db{$soft}}[MISSING_DEPENDENCIES] .= " $dep" ;
					}
				}
			}
		}

		@{$db{$dep}}[USED_BY] =~ s/ {2,}/ /g ;
		@{$db{$soft}}[INSTALLED_DEPENDENCIES] =~ s/ {2,}/ /g ;
		@{$db{$soft}}[MISSING_DEPENDENCIES] =~ s/ {2,}/ /g ;
	}
}



####################################################################
# $Netpkg->SelectMissingDeps(\%db);
sub SelectMissingDeps {
	my $self = shift; 
	my %db = %{$_[0]};
	my $soft; 
	for $soft ( keys %db ) { 
		@{$db{$soft}}[SELECTED_AS_DEP] = 0 ;
	}
	for $soft ( keys %db ) {
		if ( @{$db{$soft}}[SELECTED_AS_PKG] ) {

			# array of missing deps
			my @Deps = split (/ /, @{$db{$soft}}[MISSING_DEPENDENCIES]);
			
			my $dep; 
			foreach $dep (@Deps) {

				@{$db{$dep}}[SELECTED_AS_DEP] = 1 ; 
				
				# one more level
				my @Deps2 = split (/ /, @{$db{$dep}}[MISSING_DEPENDENCIES]);					
				my $dep2; 
				foreach $dep2 (@Deps2) {
					@{$db{$dep2}}[SELECTED_AS_DEP] = 1 ; 					
				}
			}
		}
	}
}

####################################################################
# $Netpkg->SelectDeps(\%db);
sub SelectDeps {
	my $self = shift; 
	my %db = %{$_[0]};
	my $soft; 
	for $soft ( keys %db ) { 
		@{$db{$soft}}[SELECTED_AS_DEP] = 0 ;
	}
	for $soft ( keys %db ) {
		if ( @{$db{$soft}}[SELECTED_AS_PKG] ) {
				
			# array of missing deps
			my @Deps = split (/ /, @{$db{$soft}}[DEPENDENCIES]);
			
			my $dep; 
			foreach $dep (@Deps) {
				@{$db{$dep}}[SELECTED_AS_DEP] = 1 ; 
			}
		}
	}
}

####################################################################
# $Netpkg->AsyncRun( $command );
sub AsyncRun {
	my $self = shift; 
	my $command = shift;
	    
    # initialize the shared semaphor to '0', then wait for the thread to set it to '1'
    $semaphorus = 0;
	my $thr = threads->new(\&Execute, $command);
	while ($semaphorus == 0) {
		Gtk2->main_iteration while ( Gtk2->events_pending  );  
	}
	return 0;
}


1;

