/*
 * Copyright (c) 2007, Dennis M. Sosnoski All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided that the
 * following conditions are met:
 * 
 * Redistributions of source code must retain the above copyright notice, this list of conditions and the following
 * disclaimer. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the
 * following disclaimer in the documentation and/or other materials provided with the distribution. Neither the name of
 * JiBX nor the names of its contributors may be used to endorse or promote products derived from this software without
 * specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package org.jibx.binding.generator;

import java.lang.reflect.Modifier;

import org.jibx.binding.model.IClassItem;
import org.jibx.binding.util.StringArray;
import org.jibx.runtime.IUnmarshallingContext;
import org.jibx.runtime.JiBXException;

/**
 * Member property customization information.
 * 
 * @author Dennis M. Sosnoski
 */
public class MemberPropertyCustom extends MemberCustom
{
    /** Enumeration of allowed attribute names */
    public static final StringArray s_allowedAttributes =
        new StringArray(new String[] { "get-name", "set-name" }, MemberCustom.s_allowedAttributes);
    
    // values specific to member level
    private boolean m_isPrivate; // internal use, not included in binding
    
    private String m_getName;
    
    private String m_setName;
    
    /**
     * Constructor.
     * 
     * @param parent
     */
    protected MemberPropertyCustom(NestingBase parent) {
        super(parent);
    }
    
    /**
     * Constructor with name known.
     * 
     * @param parent
     * @param name
     */
    protected MemberPropertyCustom(NestingBase parent, String name) {
        super(parent, name);
    }
    
    /**
     * Make sure all attributes are defined.
     * 
     * @param uctx unmarshalling context
     */
    protected void preSet(IUnmarshallingContext uctx) {
        validateAttributes(uctx, s_allowedAttributes);
    }
    
    /**
     * Check if member represents a property.
     * 
     * @return <code>true</code>
     */
    public boolean isProperty() {
        return true;
    }
    
    /**
     * Check if a private member.
     *
     * @return <code>true</code> if private, <code>false</code> if not
     */
    public boolean isPrivate() {
        return m_isPrivate;
    }
    
    /**
     * Check if collection member.
     * 
     * @return <code>false</code>
     */
    public boolean isCollection() {
        return false;
    }
    
    /**
     * Get 'get' method name.
     * 
     * @return 'get' method name (<code>null</code> if none)
     */
    public String getGetName() {
        return m_getName;
    }
    
    /**
     * Get 'set' method name.
     * 
     * @return 'set' method name (<code>null</code> if none)
     */
    public String getSetName() {
        return m_setName;
    }
    
    /**
     * Post-set method that converts the get/set method names to a member name.
     * 
     * @throws JiBXException
     */
    protected void postSet() throws JiBXException {
        if (m_getName == null && m_setName == null) {
            throw new JiBXException("'get-name' or 'set-name' attribute is required for <property> element");
        } else {
            if (m_setName == null) {
                setBaseName(memberNameFromGetMethod(m_getName));
            } else {
                setBaseName(memberNameFromSetMethod(m_setName));
            }
        }
    }
    
    /**
     * Complete customization information based on access method information.
     * 
     * @param gmeth read access method (<code>null</code> if none)
     * @param smeth write access method (<code>null</code> if none)
     * @param type value type
     * @param req required member flag (<code>null</code> if unknown)
     * @param style representation style (<code>null</code> if unspecified)
     */
    /* package */void completeProperty(IClassItem gmeth, IClassItem smeth, String type, Boolean req, Integer style) {
        if (m_getName == null && gmeth != null) {
            m_getName = gmeth.getName();
        }
        if (m_setName == null && smeth != null) {
            m_setName = smeth.getName();
        }
        m_isPrivate = (gmeth == null || Modifier.isPrivate(gmeth.getAccessFlags())) &&
            (smeth == null || Modifier.isPrivate(smeth.getAccessFlags()));
        complete(type, req, style);
    }
    
    private static MemberPropertyCustom factory(IUnmarshallingContext ictx) {
        return new MemberPropertyCustom(getContainingClass(ictx));
    }
}