/*
 * Decompiled with CFR 0.152.
 */
package org.openscience.cdk.geometry;

import java.awt.Dimension;
import java.util.Vector;
import javax.vecmath.Point2d;
import javax.vecmath.Point3d;
import javax.vecmath.Vector2d;
import org.openscience.cdk.Atom;
import org.openscience.cdk.AtomContainer;
import org.openscience.cdk.AtomEnumeration;
import org.openscience.cdk.Bond;
import org.openscience.cdk.tools.LoggingTool;

public class GeometryTools {
    private static LoggingTool logger = null;

    public static void translateAllPositive(AtomContainer atomCon) {
        double minX = Double.MAX_VALUE;
        double minY = Double.MAX_VALUE;
        Atom[] atoms = atomCon.getAtoms();
        for (int i = 0; i < atoms.length; ++i) {
            if (atoms[i].getPoint2d() == null) continue;
            if (atoms[i].getPoint2d().x < minX) {
                minX = atoms[i].getPoint2d().x;
            }
            if (!(atoms[i].getPoint2d().y < minY)) continue;
            minY = atoms[i].getPoint2d().y;
        }
        logger.debug("Translating: minx=" + minX + ", minY=" + minY);
        GeometryTools.translate2D(atomCon, minX * -1.0, minY * -1.0);
    }

    public static void translate2D(AtomContainer atomCon, double transX, double transY) {
        GeometryTools.translate2D(atomCon, new Vector2d(transX, transY));
    }

    public static void scaleMolecule(AtomContainer atomCon, double scaleFactor) {
        for (int i = 0; i < atomCon.getAtomCount(); ++i) {
            if (atomCon.getAtomAt(i).getPoint2d() == null) continue;
            atomCon.getAtomAt((int)i).getPoint2d().x *= scaleFactor;
            atomCon.getAtomAt((int)i).getPoint2d().y *= scaleFactor;
        }
    }

    public static void rotate(AtomContainer atomCon, Point2d center, double angle) {
        Point2d p = null;
        Atom atom = null;
        for (int i = 0; i < atomCon.getAtomCount(); ++i) {
            atom = atomCon.getAtomAt(i);
            p = atom.getPoint2d();
            double offsetAngle = GeometryTools.getAngle(p.x - center.x, p.y - center.y);
            double distance = p.distance(center);
            p.x = center.x + Math.sin(angle + offsetAngle) * distance;
            p.y = center.y - Math.cos(angle + offsetAngle) * distance;
        }
    }

    public static void scaleMolecule(AtomContainer atomCon, Dimension areaDim, double fillFactor) {
        Dimension molDim = GeometryTools.get2DDimension(atomCon);
        double widthFactor = (double)areaDim.width / (double)molDim.width;
        double heightFactor = (double)areaDim.height / (double)molDim.height;
        double scaleFactor = Math.min(widthFactor, heightFactor) * fillFactor;
        GeometryTools.scaleMolecule(atomCon, scaleFactor);
    }

    public static Dimension get2DDimension(AtomContainer atomCon) {
        double[] minmax = GeometryTools.getMinMax(atomCon);
        double maxX = minmax[2];
        double maxY = minmax[3];
        double minX = minmax[0];
        double minY = minmax[1];
        return new Dimension((int)(maxX - minX + 1.0), (int)(maxY - minY + 1.0));
    }

    public static double[] getMinMax(AtomContainer container) {
        double maxX = Double.MIN_VALUE;
        double maxY = Double.MIN_VALUE;
        double minX = Double.MAX_VALUE;
        double minY = Double.MAX_VALUE;
        for (int i = 0; i < container.getAtomCount(); ++i) {
            Atom atom = container.getAtomAt(i);
            if (atom.getPoint2d() == null) continue;
            if (atom.getX2d() > maxX) {
                maxX = atom.getX2d();
            }
            if (atom.getX2d() < minX) {
                minX = atom.getX2d();
            }
            if (atom.getY2d() > maxY) {
                maxY = atom.getY2d();
            }
            if (!(atom.getY2d() < minY)) continue;
            minY = atom.getY2d();
        }
        double[] minmax = new double[]{minX, minY, maxX, maxY};
        return minmax;
    }

    public static void translate2D(AtomContainer atomCon, Vector2d vector) {
        Atom[] atoms = atomCon.getAtoms();
        for (int i = 0; i < atoms.length; ++i) {
            if (atoms[i].getPoint2d() != null) {
                atoms[i].getPoint2d().add(vector);
                continue;
            }
            logger.warn("Could not translate atom in 2D space");
        }
    }

    public static void translate2DCentreOfMassTo(AtomContainer atomCon, Point2d p) {
        Point2d com = GeometryTools.get2DCentreOfMass(atomCon);
        Vector2d translation = new Vector2d(p.x - com.x, p.y - com.y);
        Atom[] atoms = atomCon.getAtoms();
        for (int i = 0; i < atoms.length; ++i) {
            if (atoms[i].getPoint2d() == null) continue;
            atoms[i].getPoint2d().add(translation);
        }
    }

    public static void center(AtomContainer atomCon, Dimension areaDim) {
        Dimension molDim = GeometryTools.get2DDimension(atomCon);
        int transX = (areaDim.width - molDim.width) / 2;
        int transY = (areaDim.height - molDim.height) / 2;
        GeometryTools.translateAllPositive(atomCon);
        GeometryTools.translate2D(atomCon, new Vector2d(transX, transY));
    }

    public static Point2d get2DCenter(Vector atoms) {
        double x = 0.0;
        double y = 0.0;
        for (int f = 0; f < atoms.size(); ++f) {
            Atom atom = (Atom)atoms.elementAt(f);
            if (atom.getPoint2d() == null) continue;
            x += atom.getX2d();
            y += atom.getY2d();
        }
        return new Point2d(x / (double)atoms.size(), y / (double)atoms.size());
    }

    public static Point2d get2DCentreOfMass(AtomContainer ac) {
        double x = 0.0;
        double y = 0.0;
        double totalmass = 0.0;
        AtomEnumeration atoms = (AtomEnumeration)ac.atoms();
        while (atoms.hasMoreElements()) {
            Atom a = (Atom)atoms.nextElement();
            double mass = a.getExactMass();
            totalmass += mass;
            x += mass * a.getX2d();
            y += mass * a.getY2d();
        }
        return new Point2d(x / totalmass, y / totalmass);
    }

    public static Point3d get3DCentreOfMass(AtomContainer ac) {
        double x = 0.0;
        double y = 0.0;
        double z = 0.0;
        double totalmass = 0.0;
        AtomEnumeration atoms = (AtomEnumeration)ac.atoms();
        while (atoms.hasMoreElements()) {
            Atom a = (Atom)atoms.nextElement();
            double mass = a.getExactMass();
            totalmass += mass;
            x += mass * a.getX3d();
            y += mass * a.getY3d();
            z += mass * a.getZ3d();
        }
        return new Point3d(x / totalmass, y / totalmass, z / totalmass);
    }

    public static double getAngle(double xDiff, double yDiff) {
        double angle = 0.0;
        if (xDiff >= 0.0 && yDiff >= 0.0) {
            angle = Math.atan(yDiff / xDiff);
        } else if (xDiff < 0.0 && yDiff >= 0.0) {
            angle = Math.PI + Math.atan(yDiff / xDiff);
        } else if (xDiff < 0.0 && yDiff < 0.0) {
            angle = Math.PI + Math.atan(yDiff / xDiff);
        } else if (xDiff >= 0.0 && yDiff < 0.0) {
            angle = Math.PI * 2 + Math.atan(yDiff / xDiff);
        }
        return angle;
    }

    public static int[] distanceCalculator(int[] coords, double dist) {
        double angle = coords[2] - coords[0] == 0 ? 1.5707963267948966 : Math.atan(((double)coords[3] - (double)coords[1]) / ((double)coords[2] - (double)coords[0]));
        int begin1X = (int)(Math.cos(angle + 1.5707963267948966) * dist + (double)coords[0]);
        int begin1Y = (int)(Math.sin(angle + 1.5707963267948966) * dist + (double)coords[1]);
        int begin2X = (int)(Math.cos(angle - 1.5707963267948966) * dist + (double)coords[0]);
        int begin2Y = (int)(Math.sin(angle - 1.5707963267948966) * dist + (double)coords[1]);
        int end1X = (int)(Math.cos(angle - 1.5707963267948966) * dist + (double)coords[2]);
        int end1Y = (int)(Math.sin(angle - 1.5707963267948966) * dist + (double)coords[3]);
        int end2X = (int)(Math.cos(angle + 1.5707963267948966) * dist + (double)coords[2]);
        int end2Y = (int)(Math.sin(angle + 1.5707963267948966) * dist + (double)coords[3]);
        int[] newCoords = new int[]{begin1X, begin1Y, begin2X, begin2Y, end1X, end1Y, end2X, end2Y};
        return newCoords;
    }

    public static int[] getBondCoordinates(Bond bond) {
        if (bond.getAtomAt(0).getPoint2d() == null || bond.getAtomAt(1).getPoint2d() == null) {
            logger.error("getBondCoordinates() called on Bond without 2D coordinates!");
            return new int[0];
        }
        int beginX = (int)bond.getAtomAt((int)0).getPoint2d().x;
        int endX = (int)bond.getAtomAt((int)1).getPoint2d().x;
        int beginY = (int)bond.getAtomAt((int)0).getPoint2d().y;
        int endY = (int)bond.getAtomAt((int)1).getPoint2d().y;
        int[] coords = new int[]{beginX, beginY, endX, endY};
        return coords;
    }

    public static Atom getClosestAtom(int xPosition, int yPosition, AtomContainer atomCon) {
        Atom closestAtom = null;
        double smallestMouseDistance = -1.0;
        for (int i = 0; i < atomCon.getAtomCount(); ++i) {
            Atom currentAtom = atomCon.getAtomAt(i);
            double atomX = currentAtom.getX2d();
            double atomY = currentAtom.getY2d();
            double mouseDistance = Math.sqrt(Math.pow(atomX - (double)xPosition, 2.0) + Math.pow(atomY - (double)yPosition, 2.0));
            if (!(mouseDistance < smallestMouseDistance) && smallestMouseDistance != -1.0) continue;
            smallestMouseDistance = mouseDistance;
            closestAtom = currentAtom;
        }
        return closestAtom;
    }

    public static Bond getClosestBond(int xPosition, int yPosition, AtomContainer atomCon) {
        Bond closestBond = null;
        double smallestMouseDistance = -1.0;
        Bond[] bonds = atomCon.getBonds();
        for (int i = 0; i < bonds.length; ++i) {
            Bond currentBond = bonds[i];
            Point2d bondCenter = GeometryTools.get2DCenter(currentBond.getAtomsVector());
            double mouseDistance = Math.sqrt(Math.pow(bondCenter.x - (double)xPosition, 2.0) + Math.pow(bondCenter.y - (double)yPosition, 2.0));
            if (!(mouseDistance < smallestMouseDistance) && smallestMouseDistance != -1.0) continue;
            smallestMouseDistance = mouseDistance;
            closestBond = currentBond;
        }
        return closestBond;
    }

    public static void sortBy2DDistance(Atom[] atoms, Point2d point) {
        Atom atom1 = null;
        Atom atom2 = null;
        boolean doneSomething = false;
        do {
            doneSomething = false;
            for (int f = 0; f < atoms.length - 1; ++f) {
                atom1 = atoms[f];
                atom2 = atoms[f + 1];
                double distance1 = point.distance(atom1.getPoint2d());
                double distance2 = point.distance(atom2.getPoint2d());
                if (!(distance2 < distance1)) continue;
                atoms[f] = atom2;
                atoms[f + 1] = atom1;
                doneSomething = true;
            }
        } while (doneSomething);
    }

    public static double getScaleFactor(AtomContainer ac, double bondLength) {
        double currentAverageBondLength = GeometryTools.getBondLengthAverage(ac);
        return bondLength / currentAverageBondLength;
    }

    public static double getBondLengthAverage(AtomContainer ac) {
        double bondLengthSum = 0.0;
        Bond[] bonds = ac.getBonds();
        int bondCounter = 0;
        for (int f = 0; f < bonds.length; ++f) {
            Bond bond = bonds[f];
            Atom atom1 = bond.getAtomAt(0);
            Atom atom2 = bond.getAtomAt(1);
            if (atom1.getPoint2d() == null || atom2.getPoint2d() == null) continue;
            ++bondCounter;
            bondLengthSum += GeometryTools.getLength2D(bond);
        }
        return bondLengthSum / (double)bondCounter;
    }

    public static double getLength2D(Bond bond) {
        if (bond.getAtomAt(0) == null || bond.getAtomAt(1) == null) {
            return 0.0;
        }
        Point2d p1 = bond.getAtomAt(0).getPoint2d();
        Point2d p2 = bond.getAtomAt(1).getPoint2d();
        if (p1 == null || p2 == null) {
            return 0.0;
        }
        return p1.distance(p2);
    }

    public static boolean has2DCoordinates(AtomContainer m) {
        Atom[] atoms = m.getAtoms();
        for (int i = 0; i < atoms.length; ++i) {
            if (atoms[i].getPoint2d() != null) continue;
            return false;
        }
        return true;
    }

    public static boolean has2DCoordinates(Atom a) {
        return a.getPoint2d() != null;
    }

    public static boolean has2DCoordinates(Bond b) {
        Atom[] atoms = b.getAtoms();
        for (int i = 0; i < atoms.length; ++i) {
            if (atoms[i].getPoint2d() != null) continue;
            return false;
        }
        return true;
    }

    public static boolean has3DCoordinates(AtomContainer m) {
        boolean hasinfo = true;
        Atom[] atoms = m.getAtoms();
        for (int i = 0; i < atoms.length; ++i) {
            if (atoms[i].getPoint3d() != null) continue;
            return false;
        }
        return hasinfo;
    }

    public static Vector2d calculatePerpendicularUnitVector(Point2d p1, Point2d p2) {
        Vector2d v = new Vector2d();
        v.sub(p2, p1);
        v.normalize();
        return new Vector2d(-1.0 * v.y, v.x);
    }

    public static double getNormalizationFactor(AtomContainer container) {
        Bond[] bonds = container.getBonds();
        double bondlength = 0.0;
        double ratio = 0.0;
        double desiredBondLength = 1.5;
        int counter = 0;
        for (int f = 0; f < bonds.length; ++f) {
            if (bonds[f].getAtomCount() != 2) continue;
            ++counter;
            Atom atom1 = bonds[f].getAtomAt(0);
            Atom atom2 = bonds[f].getAtomAt(1);
            bondlength += Math.sqrt(Math.pow(atom1.getX2d() - atom2.getX2d(), 2.0) + Math.pow(atom1.getY2d() - atom2.getY2d(), 2.0));
        }
        ratio = desiredBondLength / (bondlength /= (double)counter);
        return ratio;
    }

    public static int getBestAlignmentForLabel(AtomContainer container, Atom atom) {
        Atom[] connectedAtoms = container.getConnectedAtoms(atom);
        int overallDiffX = 0;
        for (int i = 0; i < connectedAtoms.length; ++i) {
            Atom connectedAtom = connectedAtoms[i];
            overallDiffX += (int)(connectedAtom.getX2d() - atom.getX2d());
        }
        if (overallDiffX <= 0) {
            return 1;
        }
        return -1;
    }

    static {
        if (logger == null) {
            logger = new LoggingTool("org.openscience.cdk.geometry.GeometryTools");
        }
    }
}

