/* -----------------------------------------------------------------------
   DigiTemp v1.3
      
   Copyright 1996, 1997, 1998, 1999 by Brian C. Lane <bcl@brianlane.com>
   All rights reserved.

   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the Free
   Software Foundation; either version 2 of the License, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful, but WITHOUT
   ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
   FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
   more details.

   You should have received a copy of the GNU General Public License along
   with this program; if not, write to the Free Software Foundation, Inc.,
   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
   
     digitemp -i			Initalize .digitemprc file
     digitemp -s/dev/ttyS0		Set serial port (required)
     digitemp -f5			Set Fail timeout to 5 seconds
     digitemp -r500			Set Read timeout to 500mS
     digitemp -l/var/log/temperature	Send output to logfile
     digitemp -v			Verbose mode
     digitemp -t0			Read Temperature
     digitemp -a			Read all Temperatures
     digitemp -d5                       Delay between samples (in sec.)
     digitemp -n50                      Number of times to repeat
     digitemp -o1                       Output format for logfile
                                        See description below
     digitemp -o"output format string"  See description below

     Logfile formats:
     1 = (default) - 1 line per sensor, time, C, F
         1 line for each sample, elapsed time, sensor #1, #2, ... tab seperated
     2 = Reading in C
     3 = Reading in F
     The format string uses strftime tokens plus 3 special
     ones for digitemp - %s for sensor #, %C for centigrage,
     %F for fahrenheight the case of the token is important!

   =======================================================================
   12/29/99     Changing license to GNU Public License version 2
                Leaving the version number the same, no code changes.

   06/03/99	Finishing this thing up.

   06/02/99	Use 2 logging routines, log_time to log a single line
   		per sensor with optional time data and log_str to log
   		a single line for multiple sensors with elapsed time at
   		the beginning of the line.

   05/29/99	Added 2 new init file tags, LOG_FORMAT and LOG_TYPE
		New formatting works, need to clean up the logic now.
		
   05/27/99	Adding user specified format string. To use it, first
   		parse out the digitemp specific stuff and create a new
   		string to feed into strftime, with the sensor, temperature
   		already set.

   05/24/99	Adding a output specifier string, this will allow users
   		to configure the output to look however they want it to.
   		The format string uses strftime tokens plus 3 special
   		ones for digitemp - %s for sensor #, %C for centigrage,
   		%F for fahrenheight

   05/23/99	Adding Solaris support via -DSOLARIS define
   		Fixing the time problems once and for all, using localtime
   		in the log_line routine. The user should have TZ set
   		correctly for their timezone. This has been tested
   		and works correctly.

		Changed version nuber to v1.3

   01/14/99	A user in Sweden (and another in Finland) discovered a
   		long standing bug. In do_temp I should have been using
   		0x100 instead of 0xFF for the subtraction. This caused
   		temperatures below 0 degrees C to jump up 1 degree as
   		it decreased. This is fixed.
   		
		Changed version number to v1.2

   10/20/98	Adding new features from DOS version to keep things
   		consistent. Removing the debug command, not used anyway.
		Added a free() to error condition edit from read_rcfile()   		
		Set some cases of freeing to = NULL, also freed the rom
		list before doing a search rom (searchROM checks too, but
		this is the right place for it).

   08/31/98	Adding a check for family 0x10 so that we can read DS1820s
   		while they are on a network that includes other 1-wire
   		devices.
   		Fixed a problem with freeing uninitalized rom_list when
   		starting a SearchROM. Not sure why this never appeared
   		before.

   03/06/98     Adding a -d debug level to help figure out why this thing
                is no longer working.

   03/13/97     Error in CRC calculation. Wrong # of bytes.
   		Error with 3 sensors. Sometimes doesn't store correct ROM
   		data to .digitemprc -- need to malloc more memory dummy!

   03/08/97	Adding user defined timeouts for failure and for the
   		read delay.

   01/24/97	Changed over to correct baud rate and 6 bits @ 115.2k
   		ROM search function is now working. All low level code
   		is functioning except for Alarm Search. Starting to move
   		into a seperate object file with API for users to write
   		their own code with.
   		
   01/22/97	Working on ROM search routine, double cannot handle a full
   		64 bits for some reason, converting to 64 byte array for
   		each bit.

   01/19/97	Rewriting for new interface. This programs handles all the
   		low level communications with the temperature sensor using
   		the 115200k serial adapter.
   
   01/02/96 	Rewriting this code to be more user friendly
     
   -----------------------------------------------------------------------*/
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <termios.h>
#include <unistd.h>
#include <getopt.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/ioctl.h>
#ifndef SOLARIS
#include <linux/serial.h>
#endif
#include <fcntl.h>
#include <time.h>
#include <string.h>

#include "onewire.h"
#include "ds1820.h"

extern char 	*optarg;              
extern int	optind, opterr, optopt;

char	serial_port[40], log_file[1024], log_format[80];
int	fail_time;				/* Failure timeout	*/
int	read_time;				/* Pause during read	*/
int	log_type;				/* output format type	*/


/* ----------------------------------------------------------------------- *
   Print out the program usage
 * ----------------------------------------------------------------------- */
void usage()
{
  printf("\nUsage: digitemp -s<device> [-i -d -l -r -v -t -p -a]\n");
  printf("                -i                            Initalize .digitemprc file\n");
  printf("                -s/dev/ttyS0                  Set serial port\n");
  printf("                -l/var/log/temperature        Send output to logfile\n");
  printf("                -f5                           Fail delay in S\n");
  printf("                -r500                         Read delay in mS\n");
  printf("                -v                            Verbose output\n");
  printf("                -t0                           Read Sensor #\n");
  printf("                -a                            Read all Sensors\n");
  printf("                -d5                           Delay between samples (in sec.)\n");
  printf("                -n50                          Number of times to repeat\n");
  printf("                -o2                           Output format for logfile\n");
  printf("                -o\"output format string\"      See description below\n");
  printf("\nLogfile formats:  1 = One line per sensor, time, C, F (default)\n");
  printf("                  2 = One line per sample, elapsed time, temperature in C\n");
  printf("                  3 = Same as #2, except temperature is in F\n");
  printf("        #2 and #3 have the data seperated by tabs, suitable for import\n");
  printf("        into a spreadsheet or other graphing software.\n");
  printf("\n        The format string uses strftime tokens plus 3 special ones for\n");
  printf("        digitemp - %%s for sensor #, %%C for centigrage, %%F for fahrenheight.\n");
  printf("        The case of the token is important! The default format string is:\n");
  printf("        \"%%b %%d %%H:%%M:%%S Sensor %%s C: %%.2C F: %%.2F\" which gives you an\n");
  printf("        output of: May 24 21:25:43 Sensor 0 C: 23.66 F: 74.59\n\n");

}


/* -----------------------------------------------------------------------
   Return the high-precision temperature value

   Calculated using formula from DS1820 datasheet

   Temperature   = scratch[0]
   Sign          = scratch[1]
   TH            = scratch[2]
   TL            = scratch[3]
   Count Remain  = scratch[6]
   Count Per C   = scratch[7]
   CRC           = scratch[8]
   
                   count_per_C - count_remain
   (temp - 0.25) * --------------------------
                       count_per_C

   If Sign is not 0x00 then it is a negative (Centigrade) number, and
   the temperature must be subtracted from 0x100 and multiplied by -1
      
   ----------------------------------------------------------------------- */
float do_temp( unsigned char *scratch )
{
  float temp, hi_precision;

  if( scratch[1] == 0 )
  {
    temp = (int) scratch[0] >> 1;
  } else {
    temp = -1 * (int) (0x100-scratch[0]) >> 1;
  }
  
  temp -= 0.25;

  hi_precision = (int) scratch[7] - (int) scratch[6];

  hi_precision = hi_precision / (int) scratch[7];

  temp = temp + hi_precision;

  return temp;
}


/* -----------------------------------------------------------------------
   Convert degrees C to degrees F
   ----------------------------------------------------------------------- */
float c2f( float temp )
{
  return 32 + ((temp*9)/5);
}



/* -----------------------------------------------------------------------
   Take the log_format string and parse out the
   digitemp tags (%*s %*C and %*F) including any format
   specifiers to pass to sprintf. Build a new string
   with the strftime tokens and the temperatures mixed
   together
   ----------------------------------------------------------------------- */
int build_tf( char *time_format, char *format, int sensor, float temp_c, float temp_f )
{
  char	*tf_ptr,
  	*lf_ptr,
  	*lf_ptr2,
  	*tk_ptr,
  	token[80],
  	temp[80];
  	
  if( !time_format || !format )
    return 0;

  tf_ptr = time_format;
  lf_ptr = format;
  
  while( *lf_ptr )
  {
    if( *lf_ptr != '%' )
    {
      *tf_ptr++ = *lf_ptr++;
    } else {
      /* Found a token, decide if its one of ours... */
      /* save initial pointer, grab everything up to... */
      lf_ptr2 = lf_ptr;
      tk_ptr = token;
      
      /* Take numbers, astrix, period and letters */
      while( isalnum( *lf_ptr ) || (*lf_ptr == '.') ||
             (*lf_ptr == '*') || (*lf_ptr == '%') )
      {
        *tk_ptr++ = *lf_ptr++;
        *tk_ptr = 0;  
      }
      
      /* see if the format specifier is digitemp or strftime */
      switch( *(tk_ptr-1) )
      {
        case 's' :
        	/* Sensor number */
	        /* Change the specifier to a d */
	        *(tk_ptr-1) = 'd';
	        
	        /* Pass it through sprintf */
	        sprintf( temp, token, sensor );

		/* Insert this into the time format string */
		tk_ptr = temp;
		while( *tk_ptr )
		  *tf_ptr++ = *tk_ptr++;
        	break;
        	
        case 'F' :
        	/* Degrees Fahrenheight */
	        /* Change the specifier to a f */
	        *(tk_ptr-1) = 'f';
	        
	        /* Pass it through sprintf */
	        sprintf( temp, token, temp_f );

		/* Insert this into the time format string */
		tk_ptr = temp;
		while( *tk_ptr )
		  *tf_ptr++ = *tk_ptr++;
        
        	break;
        	
        case 'C' :
        	/* Degrees Centigrade */
                /* Change the specifier to a f */
	        *(tk_ptr-1) = 'f';
	        
	        /* Pass it through sprintf */
	        sprintf( temp, token, temp_c );

		/* Insert this into the time format string */
		tk_ptr = temp;
		while( *tk_ptr )
		  *tf_ptr++ = *tk_ptr++;        	
        	break;
        	
        default:
		/* Not something for us, copy it into the time format */
        	tk_ptr = token;
        	while( *tk_ptr )
        	  *tf_ptr++ = *tk_ptr++;
        	break;
      } 
    }
  
  }

  return 1;
}


/* -----------------------------------------------------------------------
   Print a string to the console or the logfile
   ----------------------------------------------------------------------- */
int log_string( char *line )
{
  int fd=0;
  

  if( log_file[0] != 0 )
  {  
    if( (fd = open( log_file, O_CREAT | O_WRONLY | O_APPEND,
                          S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH ) ) == -1 )
    {
      printf("Error opening logfile: %s\n", log_file );
      return -1;
    }
    if( write( fd, line, strlen( line ) ) == -1)
      perror("Error loging to logfile");
    close( fd );
  } else {
    printf( line );
  }
  return 0;
}  


/* -----------------------------------------------------------------------
   Log one line of text to the logfile with the current date and time
   ----------------------------------------------------------------------- */
int log_time( int sensor, float temp_c, float temp_f )
{
  unsigned char	temp[1024],
  		time_format[160];
  time_t	mytime;


  mytime = time(NULL);
  if( mytime )
  {
    /* Build the time format string from log_format */
    build_tf( time_format, log_format, sensor, temp_c, temp_f );

    /* Handle the time format tokens */
    strftime( temp, 1024, time_format, localtime( &mytime ) );

    strcat( temp, "\n" );
  } else {
    sprintf( temp, "Time Error\n" );
  }
  /* Log it to stdout, logfile or both */
  log_string( temp );

  return 0;
}
  

/* -----------------------------------------------------------------------
   Read the temperature from one sensor
   ----------------------------------------------------------------------- */
int read_temp( int fd, struct _roms *rom_list, int sensor, int opts )
{
  char		temp[1024];
  unsigned char	scratch[9];
  int		x;
  float		temp_c;
    
  x = 0;  
  
  MatchROM( fd, fail_time, rom_list, sensor );
  ReadTemperature( fd, fail_time, read_time );

  MatchROM( fd, fail_time, rom_list, sensor );
  if( ReadScratchpad( fd, fail_time, scratch ) < 0 )
  {
    printf("Error reading Scratchpad\n");
    return -1;
  }

  /* Convert data to temperature */
  temp_c = do_temp( scratch );

  switch( log_type )
  {
    /* Multiple Centigrade temps per line */
    case 2:     sprintf( temp, "\t%3.2f", temp_c );
		log_string( temp );
		break;

    /* Multiple Fahrenheight temps per line */
    case 3:	sprintf( temp, "\t%3.2f", c2f(temp_c) );
		log_string( temp );
		break;

    default:	log_time( sensor, temp_c, c2f(temp_c) );
		break;
  }

  /* If verbose mode is enabled, show the DS1820's internal registers */
  if( opts & 0x0004 )
  {
    if( log_file[0] != 0 )
    {
      sprintf( temp, "  Temperature   : 0x%02X\n", scratch[0] );
      sprintf( temp, "  Sign          : 0x%02X\n", scratch[1] );
      sprintf( temp, "  TH            : 0x%02X\n", scratch[2] );
      sprintf( temp, "  TL            : 0x%02X\n", scratch[3] );
      sprintf( temp, "  Remain        : 0x%02X\n", scratch[6] );
      sprintf( temp, "  Count Per C   : 0x%02X\n", scratch[7] );
      sprintf( temp, "  CRC           : 0x%02X\n", scratch[8] );
    } else {
      printf("  Temperature   : 0x%02X\n", scratch[0] );
      printf("  Sign          : 0x%02X\n", scratch[1] );
      printf("  TH            : 0x%02X\n", scratch[2] );
      printf("  TL            : 0x%02X\n", scratch[3] );
      printf("  Remain        : 0x%02X\n", scratch[6] );
      printf("  Count Per C   : 0x%02X\n", scratch[7] );
      printf("  CRC           : 0x%02X\n", scratch[8] );
    }
  }
  
  return 0;
}


/* -----------------------------------------------------------------------
   Read the temperaturess for all the connected sensors

   Step through all the sensors in the list of serial numbers
   ----------------------------------------------------------------------- */
int read_all( int fd, struct _roms *rom_list, int opts )
{
  int x;
  
  for( x = 0; x < rom_list->max; x++ )
  {
    read_temp( fd, rom_list, x, opts );  
  }
  
  return 0;
}


/* -----------------------------------------------------------------------
   Read a .digitemprc file from the current directory

   The rc file contains:
   
   TTY <serial>
   LOG <logfilepath>
   FAIL_TIME <time in seconds>
   READ_TIME <time in mS>
   LOG_TYPE <from -o>
   LOG_FORMAT <format string for logging and printing>
   SENSORS <number of ROM lines>
   Multiple ROM x <serial number in bytes> lines
   
   ----------------------------------------------------------------------- */
int read_rcfile( char *fname, struct _roms *rom_list )
{
  FILE	*fp;
  char	temp[80];
  char	*ptr;
  int	sensors, x;
  
  sensors = 0;
  
  if( ( fp = fopen( fname, "r" ) ) == NULL )
  {
    /* No rcfile to read */
    return 1;
  }
  
  while( fgets( temp, 80, fp ) != 0 )
  {
    if( (temp[0] == '\n') || (temp[0] == '#') )
      continue;
      
    ptr = strtok( temp, " \t\n" );
    
    if( strncasecmp( "TTY", ptr, 3 ) == 0 )
    {
      ptr = strtok( NULL, " \t\n" );
      strcpy( serial_port, ptr );
    } else if( strncasecmp( "LOG_TYPE", ptr, 8 ) == 0 ) {
      ptr = strtok( NULL, " \t\n");
      log_type = atoi( ptr );
    } else if( strncasecmp( "LOG_FORMAT", ptr, 10 ) == 0 ) {
      ptr = strtok( NULL, "\"\n");
      strcpy( log_format, ptr );
    } else if( strncasecmp( "LOG", ptr, 3 ) == 0 ) {
      ptr = strtok( NULL, " \t\n" );
      strcpy( log_file, ptr );
    } else if( strncasecmp( "FAIL_TIME", ptr, 9 ) == 0 ) {
      ptr = strtok( NULL, " \t\n");
      fail_time = atoi( ptr );
    } else if( strncasecmp( "READ_TIME", ptr, 9 ) == 0 ) {
      ptr = strtok( NULL, " \t\n");
      read_time = atoi( ptr );
    } else if( strncasecmp( "SENSORS", ptr, 7 ) == 0 ) {
      ptr = strtok( NULL, " \t\n" );
      sensors = atoi( ptr );
      
      /* Reserve some memory for the list */
      if( ( rom_list->roms = malloc( sensors * 8 ) ) == NULL )
      {
        printf("Error reserving memory for %d sensors\n", sensors );
        return -1;
      }
      rom_list->max = sensors;

    } else if( strncasecmp( "ROM", ptr, 3 ) == 0 ) {
      ptr = strtok( NULL, " \t\n" );
      sensors = atoi( ptr );
      
      /* Read the 8 byte ROM address */
      for( x = 0; x < 8; x++ )
      {
        ptr = strtok( NULL, " \t\n" );
        rom_list->roms[(sensors * 8) + x] = atoi( ptr );
      }
    } else {
      printf("Error reading .digitemprc file\n");
      free( rom_list->roms );
      fclose( fp );
      return -1;
    }
  }
  
  fclose( fp ); 

  return 0;
}


/* -----------------------------------------------------------------------
   Write a .digitemprc file, it contains:
   
   TTY <serial>
   LOG <logfilepath>
   FAIL_TIME <time in seconds>
   READ_TIME <time in mS>
   LOG_TYPE <from -o>
   LOG_FORMAT <format string for logging and printing>
   SENSORS <number of ROM lines>
   Multiple ROM x <serial number in bytes> lines

   ----------------------------------------------------------------------- */
int write_rcfile( char *fname, struct _roms *rom_list )
{
  FILE	*fp;
  int	x, y;

  if( ( fp = fopen( fname, "wb" ) ) == NULL )
  {
    return -1;
  }
  
  fprintf( fp, "TTY %s\n", serial_port );
  if( log_file[0] != 0 )
    fprintf( fp, "LOG %s\n", log_file );

  fprintf( fp, "FAIL_TIME %d\n", fail_time );		/* Seconds	*/
  fprintf( fp, "READ_TIME %d\n", read_time );		/* mSeconds	*/

  fprintf( fp, "LOG_TYPE %d\n", log_type );
  fprintf( fp, "LOG_FORMAT \"%s\"\n", log_format );
  
  fprintf( fp, "SENSORS %d\n", rom_list->max );

  for( x = 0; x < rom_list->max; x++ )
  {
    fprintf( fp, "ROM %d ", x );
    
    for( y = 0; y < 8; y++ )
    {
      fprintf( fp, "%d ", rom_list->roms[(x * 8) + y] );
    }
    fprintf( fp, "\n" );
  }

  fclose( fp );
  
  return 0;
}


/* ----------------------------------------------------------------------- *
   DigiTemp main routine
   
   Parse command line options, run functions
 * ----------------------------------------------------------------------- */
int main( int argc, char *argv[] )
{
  int		sensor;			/* Single sensor to read*/
  char		c,
  		temp[80];		/* Temporary strings */
  int		opts;			/* Bitmask of flags	*/
  int		fd;
  int		x, y, stat;
  struct _roms	rom_list;		/* Attached Roms	*/
  int		sample_delay = 0,	/* Delay between samples	*/
		num_samples = 1;	/* Number of samples 		*/
  time_t	last_time,		/* Last time we started samples */
		start_time;		/* Starting time		*/
  long int	elapsed_time;		/* Elapsed from start		*/

  /* Make sure the structure is erased */
  bzero( &rom_list, sizeof( struct _roms ) );

  fd = 0;

  printf("DigiTemp v1.3 Copyright 1997-99 by Nexus Computing\n\n");

  if( argc == 1 )
  {
    usage();
    return -1;
  }

  serial_port[0] = 0;			/* No default port		*/
  log_file[0] = 0;			/* No default log file		*/
  fail_time = 5;			/* 5 Second fail default	*/
  read_time = 500;			/* 500mS read delay		*/
  sensor = 0;				/* First sensor			*/
  rom_list.max = 0;
  rom_list.roms = NULL;
  log_type = 1;			/* Normal DigiTemp logfile	*/
  sample_delay = 0;			/* No delay			*/
  num_samples = 1;			/* Only do it once by default	*/

  /* May 24 21:25:43 Sensor 0 C: 23.66 F: 74.59 */
  strcpy( log_format, "%b %d %H:%M:%S Sensor %s C: %.2C F: %.2F" );
  
  /* Read the .digitemprc file first, then let the command line
     arguments override them. If no .digitemprc is found, set up for
     1 sensors (Increased by the SearchROM routine).
  */
  if( read_rcfile( ".digitemprc", &rom_list ) < 0 )
  {
    /* Fatal error reading .digitemprc, exit with an error code */
    exit(2);
  }

  /* Command line options override any .digitemprc options temporarily	*/
  opts = 0;
  c = 0;
  while( c != -1 )
  {
    c = getopt(argc, argv, "?hiavr:f:s:l:t:d:n:o:");

    /* No more options, bail out */
    if( c == -1 )
      break;

    /* Process the command line arguments */
    switch( c )
    {
      case 'i':	opts |= 0x0001;			/* Initalize the s#'s	*/
      		break;
      		
      case 'r':	read_time = atoi(optarg);	/* Read delay in mS	*/
      		break;
      		
      case 'f': fail_time = atoi(optarg);	/* Fail delay in S	*/
      		break;
      		
      case 'v': opts |= 0x0004;			/* Verbose		*/
      		break;
      		
      case 's': if(optarg)			/* Serial port		*/
      		{
      		  strcpy( serial_port, optarg );
      		}
      		break;
      		
      case 'l': if(optarg)			/* Log Filename		*/
      		{
      		  strcpy( log_file, optarg );
      		}
      		break;
      		
      case 't':	if(optarg)			/* Sensor #		*/
      		{
      		  sensor = atoi(optarg);
      		  opts |= 0x0020;
      		}
      		break;

      case 'a': opts |= 0x0040;			/* Read All sensors	*/
		break;

      case 'd': if(optarg)			/* Sample Delay		*/
		{
		  sample_delay = atoi(optarg);
		}
		break;

      case 'n': if(optarg)			/* Number of samples 	*/
		{
		  num_samples = atoi(optarg);
		}
		break;

      case 'o': if(optarg)			/* Logfile format	*/
		{
		  if( isdigit( optarg[0] ) )
		  {
		    /* Its a number, get it */
		    log_type = atoi(optarg);
		  } else {
		    /* Not a nuber, get the string */
                    if( strlen( optarg ) > 79 )
                      printf("Output specifier too long!\n");
                    else
                      strcpy( log_format, optarg );
		    log_type=0;
		  }
		}
		break;

      case ':':
      case 'h':
      case '?': usage();
      		exit(3);
      		break;
    
      default:	break;
    }
  }

  if( opts == 0 )				/* Need at least 1 command */
  {
    usage();
    return -1;
  }

  /* Initalize the serial interface to the DS1820 */
  if( ( fd = Setup( serial_port ) ) < 0 )
  {
    printf("Error initalizing %s\n", serial_port );
    exit(4);
  }

  /* First, should we initalize the sensors? */
  /* This should store the serial numbers to the .digitemprc file */
  if( opts & 0x0001 )
  {
    /* Free up any thing that was read from .digitemprc */
    if( rom_list.roms != NULL )
    {
      free( rom_list.roms );
      rom_list.roms = NULL;
    }
    rom_list.max = 0;

    if( ( stat = SearchROM( fd, fail_time, &rom_list ) ) == -1 )
      printf("Error Searching for ROMs\n");

    /*
        Did the search find any sensors? Even if there was an error it may
        have found some valid sensors
     */
    if( rom_list.max > 0 )
    {
      for( x = 0; x < rom_list.max; x++ )
      {
        printf("ROM #%d : ", x );
        for( y = 0; y < 8; y++ )
        {
          printf("%02X", rom_list.roms[(x * 8) + y] );
        }
        printf("\n");
      }

      /* Write the new list of sensors to the current directory */
      write_rcfile( ".digitemprc", &rom_list );
    }
  }
  
  /* Record the starting time */
  start_time = time(NULL);

  /* Sample the prescribed number of times */
  for( x = 0; x < num_samples; x++ )
  {
    last_time = time(NULL);
    elapsed_time = last_time - start_time;

    switch( log_type )
    {
      /* For this type of logging we print out the elapsed time at the
         start of the line
       */
      case 3:
      case 2:	sprintf(temp, "%ld", elapsed_time );
                log_string( temp );
		break;
      default:
		break;
    }

    /* Should we read just one sensor? */
    if( opts & 0x0020 )
    {
      read_temp( fd, &rom_list, sensor, opts );
    }

  
    /* Should we read all connected sensors? */
    if( opts & 0x0040 )
    {
      read_all( fd, &rom_list, opts );
    }
  
    /* Wait until we have passed last_time + sample_delay. We do it
       this way because reading the sensors takes a certain amount
       of time, and sample_delay may be less then the time needed
       to read all the sensors. We should complain about this.
    */
    if( (time(NULL) > last_time + sample_delay) && (sample_delay > 0) )
    {
      fprintf(stderr, "Warning: delay (-d) is less than the time needed to ");
      fprintf(stderr, "read all of the attached sensors. It took %ld seconds", (long int) time(NULL) - last_time );
      fprintf(stderr, " to read the sensors\n" );
    }

    switch( log_type )
    {
      /* For this type of logging we print out the elapsed time at the
         start of the line
       */
      case 3:
      case 2:	log_string( "\n" );
		break;
      default:
		break;
    }

    /* Should we delay before the next sample? */
    if( sample_delay > 0 )
    {
      while( time(NULL) < last_time + sample_delay );
    }
  }

  if( rom_list.roms != NULL )
    free( rom_list.roms );

  /* Close the serial port */
  close( fd );
  
  exit(0);
}
