/* Input method implementations */


#ifndef INPUT_H
#define INPUT_H


#define CURSES_INTERFACE

#include <stdio.h>
#include <unistd.h>

extern "C" {
#include <curses.h>
}


#include "hp67.h"
#include "progmem.h"
#include "hp_curses.h"
#include "layout.h"
#include "stack.h"

/* Define input method class for calculator */



class IOmethod {
public:

// The method might want to be informed of the names of calculator buttons
  virtual ~IOmethod(void) { }

  virtual void initialize_io(void) { }  // Initialize

// Get the next input line
  virtual input_codes get_input_line(char *linebuffer, int buffsize) = 0;

// Print the stack
  virtual void show_stack(VStack &from) = 0;
  virtual void show_stack_and_step(VStack &from, 
				   CalcProgMem const *pstep, int pnum) = 0;

// If running a program, do we want to break?
  virtual int break_run(void) const = 0;

// Shutdown function
  virtual void shutdown_io(void) { };
};


#ifdef CURSES_INTERFACE


class ClickableWin {
private:
  WINDOW *win;

  int ncols, nrows;   // The number of rows and columns of this window
  int offset_x, offset_y;  // The offset of the window with respect to the screen
  char const **stringlist;
  int numstrings;
  
  int fieldwidth;
  int fieldsperline;

public:
  ClickableWin(WINDOW *parent, char const **strings, int nstrings, int fieldw,
	       int numrows, int numcols, int off_x, int off_y);

  ~ClickableWin(void) { delwin(win); }

private: 
  int in_my_window(int x, int y) const
    {
      return (x >= offset_x && 
	      x < offset_x + ncols &&
	      y >= offset_y &&
	      y < offset_y + nrows);
    }

  void write_entry(int num) const
    {
      if (stringlist[num] == NULL) return;

      wmove(win, num / fieldsperline, (num % fieldsperline) * fieldwidth);
      waddstr(win, stringlist[num]);
    }

public: 
  char const *string_clicked(int x, int y, int *keynum) const
    {
      if (!in_my_window(x, y)) return NULL;

      x -= offset_x;
      y -= offset_y;

      if (x % fieldwidth > fieldwidth - 1 - KEYCAP_PADDING)
	return NULL;         // Too close to the edge

      *keynum = x / fieldwidth + y * fieldsperline;
      if (*keynum >= numstrings) {
	*keynum = -1;
	return NULL;
      }

      wattron(win, A_REVERSE);
      write_entry(*keynum);
      wrefresh(win);
      usleep((unsigned long) (MOUSE_FLASH_PAUSE * 1000000));
      wattroff(win, A_REVERSE);
      write_entry(*keynum);
      wrefresh(win);

      return stringlist[*keynum];
    }
};


enum curses_shot { IMMED_SCRN = 0, PROG_SCRN, IMMED_STEP_SCRN };


class CursesIO : public IOmethod {
private:
  WINDOW *mainwin;
  int can_use_mouse;

  KeyInfo *keys;
  int numkeys;

  char const **immedstrs;
  char const **progstrs;

  int fieldwidth;
  int keysperline;

  int numkeyrows;


  curses_shot currentview;

  char *buffer;  // An input buffer


  ClickableWin *keyswin;

  int numpadleft, numpadright, numpadtop, numpadbottom;
  ClickableWin *numpadwin;

  WINDOW *inputwin;

  int stackleft, stackright, stacktop, stackbottom;
  unsigned int stackwinsize;
  WINDOW *stackwin;

  int progleft, progright, progtop, progbottom;
  unsigned int progwinsize;
  WINDOW *progwin;
  WINDOW *credits;

  int memleft, memright, memtop, membottom;
  unsigned int memwinsize;
  WINDOW *memwin;

public:
  CursesIO(KeyInfo const **keyptrs, int nkeys);
  ~CursesIO(void);
  input_codes get_input_line(char *linebuffer, int buffsize);
  int break_run(void) const;
  void shutdown_io(void) {};
  void show_stack(VStack &from);
  void show_stack_and_step(VStack &from, CalcProgMem const *pstep, int pnum);
  void show_prog_space(CalcProgMem const *pstep, int pnum);
  void show_appropriate_thing(run_mode runmode, VStack &from, CalcProgMem const *pstep, int pnum)
    {
      switch(runmode) {
      case IMMED:
	show_stack(from);
	break;
      case STEPPING:
	show_stack_and_step(from, pstep, pnum);
	break;
      case ENTER_PROG:
	show_prog_space(pstep, pnum);
	break;
      default:
	;// Nothing
      }
    }
      
private:
  inline void destroy_subwindows(void);
  int get_a_key(void);
  int get_next_char(void);
  void draw_numpad(void);
  void squeeze_in_memory(void);
  void compute_positions(void);
  void clear_input_line(void);
  void write_input_line(char const *text);
  void append_input_line(char singlechar);
  void set_default_immed_view(void);
  void set_default_prog_view(void);
  void set_immed_step_view(void);
  void draw_stack_elements(VStack &from);
  void draw_program_elements(CalcProgMem const *pmem, int pnum);
  void draw_memory_elements(void);
  void pretty_draw(WINDOW *win, char *string);
  void redraw_credits(void) {
    if (credits != NULL) {
      mvwaddstr(credits, 0, 0, "HP67 emulator\n");
      waddstr(credits, "by\n");
      waddstr(credits, "Christopher\n");
      waddstr(credits, "Neufeld");
    }
  }
};



inline void CursesIO::destroy_subwindows(void)
{
  if (inputwin != NULL) delwin(inputwin);
  if (stackwin != NULL) delwin(stackwin);
  if (progwin != NULL) delwin(progwin);
  if (memwin != NULL) delwin(memwin);
  inputwin = stackwin = progwin = memwin = NULL;

  if (keyswin != NULL) delete keyswin;
  if (numpadwin != NULL) delete numpadwin;
  keyswin = numpadwin = NULL;
  werase(mainwin);
}


inline int CursesIO::get_a_key(void)
{
  int a;

  if ((a = getch()) == ESC)
    return getch() + META;
  else
    return a;
}



#endif  /* CURSES_INTERFACE */


#endif  /* !INPUT_H */
