#!/usr/bin/perl -w
# vim: set sw=4 ts=4 si et:
# Copyright: GPL, Author: Guido Socher
# $Revision: $, last changed: $Date: $
#
use strict;
use vars qw($opt_h $opt_d);
use Getopt::Std;
use Fcntl;
#
# prototypes:
sub idletask();
sub entermenu();
sub selectnum($$);
sub goright($);
sub setgw();
sub setip();
sub setnm();
sub netmaskmenu();
sub gwmenu();
sub ipmenu();
sub mainmenu($);
sub runshutdown();
sub shutdownmenu();
sub sysinfomenu();
sub exitmenu();
sub processqueue();
sub schedulecmd($$);
sub lcdsend($);
sub catch_sig();
sub check_for_button_press($);
sub getuptime();
sub zerounpad($);
sub zeropad($);
sub get_ifconfig_llp_ip();
sub get_ifconfig_llp_gw();
sub get_ifconfig_llp_netmask();
sub write_ifconfig_llp($$);
sub getswap();
sub help();
sub logf($);
#
my $ifconfig_llp="/etc/ifconfig_llp.txt";
my $log="/var/log/llp.log";
#
my $version="1.1";
getopts("hd")||die "ERROR: No such option. -h for help.\n";
help() if ($opt_h);
$ARGV[0] || help();
my $dev=$ARGV[0];
#
# called before program termination by init script:
#
$SIG{INT} = \&catch_sig;
$SIG{HUP} = 'IGNORE';
$SIG{TERM} = \&catch_sig;
$SIG{QUIT} = \&catch_sig;
$|=1;
#
my ($rin,$rout,$timeout,$nfound,$timeleft);
my ($str);
my $dbgprintstr='';
(-w "$ifconfig_llp")||die "ERROR: file $ifconfig_llp not writeable for this process (pid $$ = $0)\n";
# initialize the serial line:
print `ttydevinit \"$dev\" 2>&1`;
logf("llp started, pid: $$");
sysopen(LCD,$dev,Fcntl::O_RDWR(),0666)||die "ERROR: can not read/write $dev\n";
my @processqueue;
my %state=('blinkled'=>0,'menutimeout'=>0,'logo'=>'linuxfocus','currip'=>'000.000.000.000');
my %buttonaction=(0=> sub {entermenu();},1=> sub {});
$rin='';
$timeout=5; # seconds
#
lcdsend("b=1"); # enable button signals
lcdsend("b=1"); # do it twice to clear any garbage that might be on
                # the serial line after startup.
lcdsend("l=10"); # red off
lcdsend("w=?"); # check for timeout
lcdsend("w=2"); # autokick watchdog
lcdsend("w=0"); # autokick watchdog
lcdsend("c=n"); # normal cursor
idletask();
processqueue();
while(1){
    vec($rin,fileno(LCD),1)=1;
	($nfound,$timeleft) = select($rout=$rin, undef, undef, $timeout);
    #$dbgprintstr.="DBG timeleft=$timeleft nfound=$nfound timeout=$timeout\n" if ($opt_d);
	if ($nfound){
        # data is available
		sysread LCD,$str,40; # there will never be more than 40 bytes
        check_for_button_press($str);
        logf("watchdog did timeout") if ($str=~/w:3/);
		$dbgprintstr.="$str" if ($opt_d);
	}
    print $dbgprintstr;
    $dbgprintstr='';
    processqueue();
}
#----------------------------------
sub idletask(){
    if ($state{'blinkled'}){
        lcdsend("l=00");
        $state{'blinkled'}=0;
    }else{
        lcdsend("l=01");
        $state{'blinkled'}=1;
    }
    lcdsend("D=".$state{'logo'});
    lcdsend("g=10");
    lcdsend("d=".getuptime());
    schedulecmd('sleepifidle',0);
}
sub entermenu(){
    @processqueue=(); 
    mainmenu(0);
    $state{'menutimeout'}=10;
}
sub selectnum($$){
    my $pos=shift;
    my $char=shift;
    $state{'menutimeout'}=10;
    if ($char eq "."){
        return(0);
    }
    $char++;
    # only 0-255:
    if ($char > 2 && ($pos==0 || $pos==4 || $pos==8 || $pos==12)){
        $char=0;
    }
    if ($char == 10){
        $char=0;
    }
    substr($state{'currip'},$pos,1)=$char;
    lcdsend("g=1$pos");
    lcdsend("d=$char");
    lcdsend("g=1$pos");
    $buttonaction{1}=sub {selectnum($pos,$char);};
    $buttonaction{0}=sub {goright($pos+1);};
}
sub goright($){
    my $currpos=shift;
    $state{'menutimeout'}=10;
    if ($currpos==15){
        write_ifconfig_llp($state{'iptype'},$state{'currip'});
        lcdsend("c=n");
        lcdsend("D=".$state{'iptype'}." changed");
        $buttonaction{0}=sub {exitmenu();};
        $buttonaction{1}=sub {exitmenu();};
        $state{'menutimeout'}=3;
        return(0);
    }
    lcdsend("c=r");
    $buttonaction{0}=sub {goright($currpos+1);};
    $buttonaction{1}=sub {selectnum($currpos,substr($state{'currip'},$currpos,1));};
}
sub setgw(){
    $state{'menutimeout'}=10;
    $state{'currip'}=get_ifconfig_llp_gw();
    $state{'iptype'}="GATEWAY";
    lcdsend("D=gw addr      select");
    lcdsend("g=10");
    lcdsend("d=".$state{'currip'}." ->");
    lcdsend("g=10");
    lcdsend("c=b");
    $buttonaction{1}=sub {selectnum(0,substr($state{'currip'},0,1));};
    $buttonaction{0}=sub {goright(1);};
}
sub setip(){
    $state{'menutimeout'}=10;
    $state{'currip'}=get_ifconfig_llp_ip();
    $state{'iptype'}="IPADDR";
    lcdsend("D=ip addr      select");
    lcdsend("g=10");
    lcdsend("d=".$state{'currip'}." ->");
    lcdsend("g=10");
    lcdsend("c=b");
    $buttonaction{1}=sub {selectnum(0,substr($state{'currip'},0,1));};
    $buttonaction{0}=sub {goright(1);};
}
sub setnm(){
    $state{'menutimeout'}=10;
    $state{'currip'}=get_ifconfig_llp_netmask();
    $state{'iptype'}="NETMASK";
    lcdsend("D=netmask      select");
    lcdsend("g=10");
    lcdsend("d=".$state{'currip'}." ->");
    lcdsend("g=10");
    lcdsend("c=b");
    $buttonaction{1}=sub {selectnum(0,substr($state{'currip'},0,1));};
    $buttonaction{0}=sub {goright(1);};
}
sub netmaskmenu(){
    my $ip=get_ifconfig_llp_netmask();
    $state{'menutimeout'}=10;
    lcdsend("D=netmask        exit");
    lcdsend("g=10");
    lcdsend("d=$ip set");
    $buttonaction{1}=sub {exitmenu();};
    $buttonaction{0}=sub {setnm();};
}
sub gwmenu(){
    my $ip=get_ifconfig_llp_gw();
    $state{'menutimeout'}=10;
    lcdsend("D=gw addr        next");
    lcdsend("g=10");
    lcdsend("d=$ip set");
    $buttonaction{1}=sub {netmaskmenu();};
    $buttonaction{0}=sub {setgw();};
}
sub ipmenu(){
    my $ip=get_ifconfig_llp_ip();
    $state{'menutimeout'}=10;
    lcdsend("D=ip addr        next");
    lcdsend("g=10");
    lcdsend("d=$ip set");
    $buttonaction{1}=sub {gwmenu();};
    $buttonaction{0}=sub {setip();};
}
sub mainmenu($){
    my $state=shift;
    $state{'menutimeout'}=10;
    if ($state==0){
        lcdsend("D=sysinfo menu     ok");
        lcdsend("g=10");
        lcdsend("d=               next");
        $buttonaction{1}=sub {sysinfomenu();};
        $buttonaction{0}=sub {mainmenu(1);};
    }
    if ($state==1){
        lcdsend("g=00");
        lcdsend("d=shutdown menu    ok");
        $buttonaction{1}=sub {shutdownmenu();};
        $buttonaction{0}=sub {mainmenu(2);};
    }
    if ($state==2){
        lcdsend("g=00");
        lcdsend("d=ip config menu   ok");
        $buttonaction{1}=sub {ipmenu();};
        $buttonaction{0}=sub {mainmenu(3);};
    }
    if ($state==3){
        lcdsend("g=00");
        lcdsend("d=exit this menu   ok");
        $buttonaction{1}=sub {exitmenu();};
        $buttonaction{0}=sub {mainmenu(0);};
    }
}
sub runshutdown(){
    $state{'menutimeout'}=10;
    print "running shutdown\n" if ($opt_d);
    logf("shutdown");
    print `/sbin/shutdown -t2 -h now`;
    lcdsend("D=executed           ");
    $buttonaction{0}=sub {exitmenu();};
    $buttonaction{1}=sub {exitmenu();};
}
sub shutdownmenu(){
    $state{'menutimeout'}=10;
    lcdsend("D=          exit menu");
    lcdsend("g=10");
    lcdsend("d=           shutdown");
    $buttonaction{1}=sub {exitmenu();};
    $buttonaction{0}=sub {runshutdown();};
}
sub sysinfomenu(){
    my @ltime = localtime();
    $state{'menutimeout'}=5;
    lcdsend("D=t:".sprintf("%04d%02d%02d %02d:%02d",1900 + $ltime[5],$ltime[4] + 1,$ltime[3],$ltime[2],$ltime[1]));
    lcdsend("g=10");
    lcdsend("d=".getswap());
    $buttonaction{0}=sub {exitmenu();};
    $buttonaction{1}=sub {exitmenu();};
}
sub exitmenu(){
    $state{'menutimeout'}=4;
    lcdsend("c=n");
    $buttonaction{0}=sub {entermenu();};
    $buttonaction{1}=sub {};
    idletask();
}
sub processqueue(){
    my $cmd;
    if (@processqueue){
        $cmd=shift @processqueue;
        if ($cmd->{'cmd'} eq "sleepifidle"){
            $dbgprintstr.="sleepifidle\n" if($opt_d); # dbg
            return(0);
        }
        if ($cmd->{'cmd'} eq "lcd"){
            syswrite LCD,$cmd->{'val'}."\n";
            $dbgprintstr.=$cmd->{'val'}."\n" if($opt_d); # dbg
            # with a usb to serial converter it can happen
            # that the uC is not fast enough (or the usb does not
	    # send the data fast enough):
            select(undef, undef, undef, 0.05); # usleep 50 
        }
    }else{
        if ($state{'menutimeout'}==1){
            # reset the buttons
            lcdsend("c=n");
            $buttonaction{0}=sub {entermenu();};
            $buttonaction{1}=sub {};
        }
        if ($state{'menutimeout'}){
            $state{'menutimeout'}--;
        }
        idletask() unless($state{'menutimeout'});
    }
}

# wait for the $ready to become 1, or delay a bit and send it then
# we need the delay because this application is not multithreaded
sub schedulecmd($$){
    my $c=shift;
    my $v=shift;
    push(@processqueue,{'cmd'=>$c,'val'=>$v});
}
sub lcdsend($){
    my $s=shift;
    schedulecmd('lcd',$s);
}
sub catch_sig(){
    syswrite LCD,"c=n\n";
    logf("llp terminates on signal");
    syswrite LCD,"D=shutdown\n";
    syswrite LCD,"l=00\n"; # green led off
    syswrite LCD,"l=11\n"; # red led on
    exit 0;
}
sub check_for_button_press($){
    my $s=shift;
    if ($s=~/#0/){
        &{$buttonaction{0}};
    }
    if ($s=~/#1/){
        &{$buttonaction{1}};
    }
}
#
sub getuptime(){
    my ($ld,$ut,$uptime);
    $ut="?";
    $ld="?";
    open(UPT,"/proc/uptime")||die "ERROR in lcdwriter: while reading /proc/uptime\n";
    while(<UPT>){
        if (/(\d+)/){
            # uptime is in seconds:
            $ut=$1 / 360;
            $ut=int $ut;
            $ut=$ut / 10;
            last;
        }
    }
    close UPT;
    open(LDT,"/proc/loadavg")||die "ERROR in lcdwriter: while reading /proc/loadavg\n";
    while(<LDT>){
        if (/^([\.\d]+)/){
            $ld=$1;
            last;
        }
    }
    close LDT;
    "u:$ut l:$ld";
}
sub zerounpad($){
    # remove leading zero in ip addr
    my $ip=shift;
    my @ia=split(/\./,$ip);
    sprintf("%d.%d.%d.%d",$ia[0],$ia[1],$ia[2],$ia[3])
}
sub zeropad($){
    # pad ip addr with zeros. e.g 1.3.4.25 will be 001.003.004.025
    my $ip=shift;
    my @ia=split(/\./,$ip);
    sprintf("%03d.%03d.%03d.%03d",$ia[0],$ia[1],$ia[2],$ia[3])
}
#
# read the ip addr form ifconfig_llp file
sub get_ifconfig_llp_ip(){
    my $ip="000.000.000.000";
    if (open(FF,"$ifconfig_llp")){
        while(<FF>){
            if (/^\s*IPADDR\s*=\s*([\d\.]+)/){
                $ip=zeropad($1);
            }
        }
        close FF;
    }
    return($ip);
}
sub get_ifconfig_llp_gw(){
    my $ip="000.000.000.000";
    if (open(FF,"$ifconfig_llp")){
        while(<FF>){
            if (/^\s*GATEWAY\s*=\s*([\d\.]+)/){
                $ip=zeropad($1);
            }
        }
        close FF;
    }
    return($ip);
}
sub get_ifconfig_llp_netmask(){
    my $ip="255.255.255.000";
    if (open(FF,"$ifconfig_llp")){
        while(<FF>){
            if (/^\s*NETMASK\s*=\s*([\d\.]+)/){
                $ip=zeropad($1);
            }
        }
        close FF;
    }
    return($ip);
}
#
sub write_ifconfig_llp($$){
    my $param=shift; # netmask or ip or gw
    my $ip=shift;
    my $file="";
    my $foundparam=0;
    $ip=zerounpad($ip);
    if (open(FF,"$ifconfig_llp")){
        while(<FF>){
            if (index($_,$param)>=0){ # found in this line
                if (s/^\s*$param\s*=\s*([\d\.]+)/$param=$ip/){
                    $foundparam=1;
                }
            }
            $file.=$_;
        }
        close FF;
    }
    if ($foundparam==0){
        # not yet in file, append
        $file.="\n$param=$ip\n";
    }
    open(FF,">$ifconfig_llp")||die "ERROR: can not write $ifconfig_llp\n";
    print FF $file;
    close FF;
    logf("$param changed to $ip, $ifconfig_llp updated");
}
#
sub getswap(){
    my ($total,$used,$free);
    open(SWP,"/proc/meminfo")||die "ERROR in lcdwriter: while reading /proc/meminfo\n";
    while(<SWP>){
        if (/SwapTotal: *(\d+)/){
            $total=$1;
        }
        if (/SwapFree: *(\d+)/){
            $free=$1;
        }
        if ($free && $total){
            $used=$total-$free;
        }else{
            $used="";
        }
    }
    close SWP;
    return "swap:$used Kb";
}
# append to the logfile.
sub logf($){
    my $str=shift;
    my @lt=localtime;
    my $timestr;
    $timestr=sprintf("%04d-%02d-%02d %02d:%02d",1900+$lt[5],$lt[4],$lt[3],$lt[2],$lt[1]);
    if (open(LOGF,">>$log")){
        print LOGF "${timestr}:llp: $str\n";
        close LOGF;
    }else{
        warn "Warning can not write to $log, ${timestr}: $str\n";
    }
}
#
sub help(){
print "This is a small perl program that function as a \"driver\"
for the linux lcd panel. It controls buttons and menus. It
will read/write IP addresses from/to: 
$ifconfig_llp

USAGE: llp.pl [-dh] device

OPTIONS: -d print debug info about commands send to the LCD panel
         -h this help

EXAMPLE: llp.pl /dev/ttyS0

This program writes to the log file $log.
You can rotate that log without any postrotate action.

version: $version
\n";
exit 0;
}
__END__ 

