/*
Copyright (c) 2015 Red Hat, Inc.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package org.ovirt.engine.sdk4.services;

import java.io.IOException;
import org.ovirt.engine.sdk4.Request;
import org.ovirt.engine.sdk4.Response;
import org.ovirt.engine.sdk4.Service;
import org.ovirt.engine.sdk4.builders.StorageDomainBuilder;
import org.ovirt.engine.sdk4.builders.TemplateBuilder;
import org.ovirt.engine.sdk4.types.StorageDomain;
import org.ovirt.engine.sdk4.types.Template;

/**
 * Manages the virtual machine template and template versions.
 */
public interface TemplateService extends Service {
    /**
     * Exports a template to the data center export domain.
     * 
     * For example, the operation can be facilitated using the following request:
     * 
     * [source]
     * ----
     * POST /ovirt-engine/api/templates/123/export
     * ----
     * 
     * With a request body like this:
     * 
     * [source,xml]
     * ----
     * <action>
     *   <storage_domain id="456"/>
     *   <exclusive>true<exclusive/>
     * </action>
     * ----
     */
    public interface ExportRequest extends Request<ExportRequest, ExportResponse> {
        /**
         * Indicates if the existing templates with the same name should be overwritten.
         * 
         * The export action reports a failed action if a template of the same name exists in the destination domain.
         * Set this parameter to `true` to change this behavior and overwrite any existing template.
         */
        ExportRequest exclusive(Boolean exclusive);
        ExportRequest storageDomain(StorageDomain storageDomain);
        /**
         * Specifies the destination export storage domain.
         */
        ExportRequest storageDomain(StorageDomainBuilder storageDomain);
    }
    
    /**
     * Exports a template to the data center export domain.
     * 
     * For example, the operation can be facilitated using the following request:
     * 
     * [source]
     * ----
     * POST /ovirt-engine/api/templates/123/export
     * ----
     * 
     * With a request body like this:
     * 
     * [source,xml]
     * ----
     * <action>
     *   <storage_domain id="456"/>
     *   <exclusive>true<exclusive/>
     * </action>
     * ----
     */
    public interface ExportResponse extends Response {
    }
    
    /**
     * Exports a template to the data center export domain.
     * 
     * For example, the operation can be facilitated using the following request:
     * 
     * [source]
     * ----
     * POST /ovirt-engine/api/templates/123/export
     * ----
     * 
     * With a request body like this:
     * 
     * [source,xml]
     * ----
     * <action>
     *   <storage_domain id="456"/>
     *   <exclusive>true<exclusive/>
     * </action>
     * ----
     */
    ExportRequest export();
    
    /**
     * Returns the information about this template or template version.
     */
    public interface GetRequest extends Request<GetRequest, GetResponse> {
        /**
         * Indicates if the results should be filtered according to the permissions of the user.
         */
        GetRequest filter(Boolean filter);
    }
    
    /**
     * Returns the information about this template or template version.
     */
    public interface GetResponse extends Response {
        /**
         * The information about the template or template version.
         */
        Template template();
    }
    
    /**
     * Returns the information about this template or template version.
     */
    GetRequest get();
    
    /**
     * Removes a virtual machine template.
     * 
     * [source]
     * ----
     * DELETE /ovirt-engine/api/templates/123
     * ----
     */
    public interface RemoveRequest extends Request<RemoveRequest, RemoveResponse> {
        /**
         * Indicates if the remove should be performed asynchronously.
         */
        RemoveRequest async(Boolean async);
    }
    
    /**
     * Removes a virtual machine template.
     * 
     * [source]
     * ----
     * DELETE /ovirt-engine/api/templates/123
     * ----
     */
    public interface RemoveResponse extends Response {
    }
    
    /**
     * Removes a virtual machine template.
     * 
     * [source]
     * ----
     * DELETE /ovirt-engine/api/templates/123
     * ----
     */
    RemoveRequest remove();
    
    /**
     * Seal the template.
     * 
     * Sealing erases all host-specific configuration from the filesystem:
     * SSH keys, UDEV rules, MAC addresses, system ID, hostname etc.,
     * thus making easy to use the template to create multiple virtual
     * machines without manual intervention.
     * 
     * Currently sealing is supported only for Linux OS.
     */
    public interface SealRequest extends Request<SealRequest, SealResponse> {
    }
    
    /**
     * Seal the template.
     * 
     * Sealing erases all host-specific configuration from the filesystem:
     * SSH keys, UDEV rules, MAC addresses, system ID, hostname etc.,
     * thus making easy to use the template to create multiple virtual
     * machines without manual intervention.
     * 
     * Currently sealing is supported only for Linux OS.
     */
    public interface SealResponse extends Response {
    }
    
    /**
     * Seal the template.
     * 
     * Sealing erases all host-specific configuration from the filesystem:
     * SSH keys, UDEV rules, MAC addresses, system ID, hostname etc.,
     * thus making easy to use the template to create multiple virtual
     * machines without manual intervention.
     * 
     * Currently sealing is supported only for Linux OS.
     */
    SealRequest seal();
    
    /**
     * Updates the template.
     * 
     * The `name`, `description`, `type`, `memory`, `cpu`, `topology`, `os`, `high_availability`, `display`,
     * `stateless`, `usb` and `timezone` elements can be updated after a template has been created.
     * 
     * For example, to update a template to so that it has 1 GiB of memory send a request like this:
     * 
     * [source]
     * ----
     * PUT /ovirt-engine/api/templates/123
     * ----
     * 
     * With the following request body:
     * 
     * [source,xml]
     * ----
     * <template>
     *   <memory>1073741824</memory>
     * </template>
     * ----
     * 
     * The `version_name` name attribute is the only one that can be updated within the `version` attribute used for
     * template versions:
     * 
     * [source,xml]
     * ----
     * <template>
     *   <version>
     *     <version_name>mytemplate_2</version_name>
     *   </version>
     * </template>
     * ----
     */
    public interface UpdateRequest extends Request<UpdateRequest, UpdateResponse> {
        /**
         * Indicates if the update should be performed asynchronously.
         */
        UpdateRequest async(Boolean async);
        UpdateRequest template(Template template);
        UpdateRequest template(TemplateBuilder template);
    }
    
    /**
     * Updates the template.
     * 
     * The `name`, `description`, `type`, `memory`, `cpu`, `topology`, `os`, `high_availability`, `display`,
     * `stateless`, `usb` and `timezone` elements can be updated after a template has been created.
     * 
     * For example, to update a template to so that it has 1 GiB of memory send a request like this:
     * 
     * [source]
     * ----
     * PUT /ovirt-engine/api/templates/123
     * ----
     * 
     * With the following request body:
     * 
     * [source,xml]
     * ----
     * <template>
     *   <memory>1073741824</memory>
     * </template>
     * ----
     * 
     * The `version_name` name attribute is the only one that can be updated within the `version` attribute used for
     * template versions:
     * 
     * [source,xml]
     * ----
     * <template>
     *   <version>
     *     <version_name>mytemplate_2</version_name>
     *   </version>
     * </template>
     * ----
     */
    public interface UpdateResponse extends Response {
        Template template();
    }
    
    /**
     * Updates the template.
     * 
     * The `name`, `description`, `type`, `memory`, `cpu`, `topology`, `os`, `high_availability`, `display`,
     * `stateless`, `usb` and `timezone` elements can be updated after a template has been created.
     * 
     * For example, to update a template to so that it has 1 GiB of memory send a request like this:
     * 
     * [source]
     * ----
     * PUT /ovirt-engine/api/templates/123
     * ----
     * 
     * With the following request body:
     * 
     * [source,xml]
     * ----
     * <template>
     *   <memory>1073741824</memory>
     * </template>
     * ----
     * 
     * The `version_name` name attribute is the only one that can be updated within the `version` attribute used for
     * template versions:
     * 
     * [source,xml]
     * ----
     * <template>
     *   <version>
     *     <version_name>mytemplate_2</version_name>
     *   </version>
     * </template>
     * ----
     */
    UpdateRequest update();
    
    /**
     * Returns a reference to the service that manages the CDROMs that are associated with the template.
     */
    TemplateCdromsService cdromsService();
    /**
     * Reference to the service that manages a specific
     * disk attachment of the template.
     */
    TemplateDiskAttachmentsService diskAttachmentsService();
    /**
     * Returns a reference to the service that manages the graphical consoles that are associated with the template.
     */
    TemplateGraphicsConsolesService graphicsConsolesService();
    /**
     * Returns a reference to the service that manages the NICs that are associated with the template.
     */
    TemplateNicsService nicsService();
    /**
     * Returns a reference to the service that manages the permissions that are associated with the template.
     */
    AssignedPermissionsService permissionsService();
    /**
     * Returns a reference to the service that manages the tags that are associated with the template.
     */
    AssignedTagsService tagsService();
    /**
     * Returns a reference to the service that manages the _watchdogs_ that are associated with the template.
     */
    TemplateWatchdogsService watchdogsService();
    /**
     * Service locator method, returns individual service on which the URI is dispatched.
     */
    Service service(String path);
}

