#include <err.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>

#include "liblif.h"

#define DIR_START		2
#define DEFAULT_VOLSIZE		256 * 1024

static int usage(void);

extern char *__progname;

static struct lifvol lv;
static struct lifdir ld;
static struct lifislhdr lih;
static int volsize, dirsize = 0, dirstart = 0;
static int isl_entrypoint, isl_startaddr, isl_length;
static char *volname = NULL;
static int volsize_specified = 0;

int
main(int argc, char *argv[])
{
	int ch, fd, i;
	int hdrsize;
	time_t tm;

	while ((ch = getopt(argc, argv, "d:e:l:n:s:v:K:")) != -1) {
		switch (ch) {
		case 'd':
			dirsize = atoi(optarg);
			dirsize = (dirsize%8 ? ((dirsize/8)+1)*8 : dirsize);
			break;
		case 'e':
			isl_entrypoint = atoi(optarg);
			break;
		case 'K':
			dirstart = atoi(optarg) * 1024 / SECTOR_SIZE;
			break;
		case 'l':
			isl_length = atoi(optarg);
			break;
		case 'n':
			volname = optarg;
			break;
		case 's':
			isl_startaddr = atoi(optarg);
			break;
		case 'v':
			volsize = (atoi(optarg) / SECTOR_SIZE) * SECTOR_SIZE;
			volsize_specified = 1;
			break;
		case '?':
		default:
			usage();
			/* NOTREACHED */
		}
	}
	argc -= optind;
	argv += optind;

	if (argc != 1)
		usage();

	fd = open(argv[0], O_CREAT|O_WRONLY, 0666);
	if (fd < 0)
		err(1, "failed to open %s", argv[0]);

	if (dirstart == 0)
		dirstart = DIR_START;

	if (dirsize == 0)
		dirsize = SECTOR_SIZE / sizeof(struct lifdir);

	hdrsize = DIR_START * SECTOR_SIZE + dirsize * sizeof(struct lifdir);

	if (hdrsize > volsize)
		volsize = hdrsize;	

	if (!volsize_specified) {
		volsize = DEFAULT_VOLSIZE;
		/*
		 * See if we're operating on a raw device, and use the
		 * full disk size, if necessary.
		 */

		/* XXX */
	}

	memset(&lv, 0, sizeof(struct lifvol));
	lv.lv_magic = htobe16(VOL_MAGIC);
	lif_makename(volname == NULL ? argv[0] : volname, lv.lv_label);

	tm = time(NULL);
	lv.lv_dirstart = htobe32(dirstart);
	lv.lv_oct = htobe16(VOL_OCT);
	lv.lv_dirsize = htobe32(dirsize);
	lv.lv_version = htobe16(1);
	lv.lv_length =  htobe32(volsize);
	lif_puttime(&lv.lv_toc, gmtime(&tm));

	if (write(fd, &lv, sizeof(struct lifvol)) != sizeof(struct lifvol))
		err(1, "unable to write volume header");

	if (isl_startaddr != 0 || isl_entrypoint != 0 || isl_length != 0) {
		lih.lih_addr = isl_startaddr;
		lih.lih_size = isl_length;
		lih.lih_entry = isl_entrypoint;
		lseek(fd, SECTOR_SIZE - 16, SEEK_SET);
		if (write(fd, &lih, sizeof(struct lifislhdr)) !=
		    sizeof(struct lifislhdr))
			err(1, "unable to write ISL header");
	}

	memset(&ld, 0, sizeof(struct lifdir));
	lseek(fd, DIR_START * SECTOR_SIZE, SEEK_SET);
	for (i=0; i<dirsize; i++) {
		if (i == dirsize)
			ld.ld_type = 0xffff;
		if (write(fd, &ld, sizeof(struct lifdir)) !=
		     sizeof(struct lifdir))
			err(1, "unable to write directory entry");
	}

	if (volsize > hdrsize) {
		lseek(fd, volsize-1, SEEK_SET);
		i = 0; write(fd, &i, 1);
	}

	close(fd);

	exit(0);
	/* NOTREACHED */
}

static int
usage(void)
{

	fprintf(stderr, "Usage: %s [-d dirsize] [-e entry] [-l length] [-n volname] [-s startaddr] [-v volsize] [-K dirstart] name\n", __progname);
	exit(EXIT_FAILURE);
}
