/*
** Exception Library -- General exception handling for ANSI C programs
** 
** Copyright (C) 1992 Computational Vision and Active Perception Lab. (CVAP),
**                    Royal Institute of Technology, Stockholm.
**
** This library is free software; you can redistribute it and/or
** modify it under the terms of the GNU Library General Public
** License as published by the Free Software Foundation; either
** version 2 of the License, or (at your option) any later version.
** 
** This library is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
** Library General Public License for more details.
** 
** You should have received a copy of the GNU Library General Public
** License along with this library (see COPYING-LIB); if not, write to 
** the Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, 
** USA.
** 
**                            Written by
**
**                   Harald Winroth, Matti Rendahl
**         Computational Vision and Active Perception Laboratory
**		    Royal Institute of Technology
**			  S-100 44 Stockholm
**				Sweden
**
** Report bugs to candela-bug@bion.kth.se, and direct all inquiries to 
** candela@bion.kth.se.
**
*/

#ifndef _exception_exc_err_h
#define _exception_exc_err_h

#include <stdarg.h>

/*
 * excErr exception type
 */

typedef struct _excErr
{
    struct _excErr *self;
    struct _excErr *in_domain;
    char *name;
    char *str;

} excErr;

/* 
 * Place-holder representing the excErr type 
 */

typedef struct
{
    /* fatal should be the first entry here (for backward compatibility) */

    int fatal;    /* Handler will return non-zero status */
    int nonfatal; /* Handler will return zero status */

} excErrType;

extern excErrType exc_err_type;

/* 
 * exc_err_progname contains the program name if set, or "anonymous program". 
 * It is set if exc_err_progname != exc_err_anonymous_program.
 */
extern char *exc_err_progname;
extern char exc_err_anonymous_program[];

/* 
 * Access functions for excErr exception variables
 */

extern excErr *exc_err_domain (excErr *e);
extern char *exc_err_name (excErr *e);
extern char *exc_err_str (excErr *e);
extern excErr *exc_err_top_domain (excErr *e);
extern char *exc_err_package_name (excErr *e);
extern char *exc_err_full_name (excErr *e);

/*
 * Error message buffers
 */

typedef enum
{
    excErrUnderline = 0,
    excErrInverse,
    excErrStandOut,

    excErrStr,
    excErrTab,

    excErrNumFormatTags /* must be last */

} excErrFormatTag;

typedef union 
{
    char *str;
    unsigned int col;
    int on;

} excErrFormatArg;

typedef struct 
{
    excErrFormatTag tag;
    excErrFormatArg arg;

} excErrFormat;

typedef enum
{
    excErrBufArgNone = 0,
    excErrBufArgLeftDelimiters,
    excErrBufArgRightDelimiters,
    excErrBufArgWhitespace,
    excErrBufArgWidth,
    excErrBufArgMaxTab,
    excErrBufArgAutoBreakChar

} excErrBufArg;

typedef struct
{
    char *left_delim;
    char *right_delim;
    char *whitespace;
    char *auto_break_char;
    unsigned int width;
    unsigned int max_tab;

    char *str;
    unsigned int str_len; /* str allocated length */
    unsigned int str_pos; /* next free pos */

    unsigned int cur_tab;
    unsigned int cur_col;

    excErrFormat *format;
    unsigned int format_len; /* format allocated length */
    unsigned int format_pos; /* next free pos */

} excErrBuf;

/*
 * Iterators
 */

#define _EXC_ERR_EVAL_1A(OP, F1, F2, ARGS) OP(F1, ARGS, _EXC_ERR_EVAL_1B)
#define _EXC_ERR_EVAL_1B(OP, F1, F2, ARGS) OP(F1, ARGS, _EXC_ERR_EVAL_1A)

#define _EXC_ERR_EVAL_2A(OP, F1, F2, ARGS) OP(F2, ARGS, _EXC_ERR_EVAL_2B)
#define _EXC_ERR_EVAL_2B(OP, F1, F2, ARGS) OP(F2, ARGS, _EXC_ERR_EVAL_2A)

#define _EXC_ERR_EVAL_1(FORMS) _EXC_ERR_EVAL_1A FORMS
#define _EXC_ERR_EVAL_2(FORMS) _EXC_ERR_EVAL_2A FORMS

#define _EXC_ERR_APPLY(FUNC, ARGS, NEXT) FUNC ARGS NEXT
#define _EXC_ERR_APPLY_STOP(FUNC, ARGS, NEXT) FUNC ARGS

/* 
 * Primitive errors 
 */

#define _EXC_ERR_DCL(DOMAIN, NAME, STR)					      \
  excErr NAME ;

#define _EXC_ERR_DEF(DOMAIN, NAME, STR)					      \
  { & DOMAIN . NAME , (excErr *) & DOMAIN, # NAME , STR },

#define EXC_ERR(DOMAIN, NAME, STR)					      \
  (_EXC_ERR_APPLY, _EXC_ERR_DCL, _EXC_ERR_DEF, (DOMAIN, NAME, STR))

/* 
 * Error domains 
 */

#define _EXC_ERR_DOM_DCL1(DOMAIN, NAME)					      \
  struct {								      \
      excErr domain;

#define _EXC_ERR_DOM_DCL2(NAME)						      \
  } NAME ;

#define _EXC_ERR_DOM_DEF1(DOMAIN, NAME)					      \
  {{ (excErr *) & DOMAIN . NAME, (excErr *) & DOMAIN, # NAME , 0 },

#define _EXC_ERR_DOM_DEF2(NAME)						      \
  },

#define EXC_ERR_DOM(DOMAIN, NAME, REST)					      \
  (_EXC_ERR_APPLY, _EXC_ERR_DOM_DCL1, _EXC_ERR_DOM_DEF1, (DOMAIN, NAME))      \
  REST									      \
  (_EXC_ERR_APPLY, _EXC_ERR_DOM_DCL2, _EXC_ERR_DOM_DEF2, (NAME))

/*
 * Top domains
 */

#define _EXC_ERR_TYPE(NAME) excErrType_ ## NAME

#define _EXC_ERR_TOP_DOM_DCL1(NAME)					      \
  typedef struct {							      \
      excErr domain;

#define _EXC_ERR_TOP_DOM_DCL2(NAME)					      \
  } _EXC_ERR_TYPE(NAME);						      \
  extern _EXC_ERR_TYPE(NAME) NAME

#define _EXC_ERR_TOP_DOM_DEF1(NAME)					      \
  _EXC_ERR_TYPE(NAME) NAME = {{ (excErr *) & NAME, 0, # NAME , 0 },

#define _EXC_ERR_TOP_DOM_DEF2(NAME)					      \
  }

#define _EXC_ERR_FORMS(NAME, REST)					      \
  (_EXC_ERR_APPLY, _EXC_ERR_TOP_DOM_DCL1, _EXC_ERR_TOP_DOM_DEF1, (NAME))      \
  REST									      \
  (_EXC_ERR_APPLY_STOP, _EXC_ERR_TOP_DOM_DCL2, _EXC_ERR_TOP_DOM_DEF2, (NAME))

/*
 * Exported macros for declarations and definitions
 */

#define EXC_ERR_DECLARE(NAME, REST) _EXC_ERR_EVAL_1(_EXC_ERR_FORMS(NAME, REST))
#define EXC_ERR_DEFINE(NAME, REST)  _EXC_ERR_EVAL_2(_EXC_ERR_FORMS(NAME, REST))

/* This is redefined in exc-err-define.h */
#define EXC_ERR_DESCRIPTION(NAME, REST) EXC_ERR_DECLARE(NAME, REST)

/* 
 * Creating, destroying and clearing error message buffers 
 */

extern excErrBuf *exc_err_buf_new (excErrBufArg first, ...);
extern void exc_err_buf_delete (excErrBuf *buf);
extern void exc_err_buf_clear (excErrBuf *buf);

/* 
 * Formatting error message buffers 
 */

extern void exc_err_buf_print (excErrBuf *buf, const char *str);
extern void exc_err_buf_printf (excErrBuf *buf, const char *fmt, ...);
extern void exc_err_buf_vprintf (excErrBuf *buf, const char *fmt, va_list l);

extern void exc_err_buf_print_indivis (excErrBuf *buf, const char *str);
extern void exc_err_buf_printf_indivis (excErrBuf *buf, const char *fmt, ...);
extern void exc_err_buf_vprintf_indivis (excErrBuf *buf, const char *fmt, 
					 va_list l);

extern void exc_err_buf_print_buf (excErrBuf *buf, excErrBuf *src);
extern void exc_err_buf_tab (excErrBuf *buf, unsigned int col);
extern void exc_err_buf_underline (excErrBuf *buf, int on);
extern void exc_err_buf_inverse (excErrBuf *buf, int on);
extern void exc_err_buf_stand_out (excErrBuf *buf, int on);

/* 
 * Access functions for excErrBuf 
 */

extern char *exc_err_buf_str (excErrBuf *buf);

/* 
 * Global error message buffer 
 */

extern excErrBuf *exc_err_buf;

/* 
 * Formatting the global exc_err_buf error message buffer (syntatic sugar) 
 */

extern void exc_err_print (const char *str);
extern void exc_err_printf (const char *fmt, ...);
extern void exc_err_vprintf (const char *fmt, va_list l);

extern void exc_err_print_indivis (const char *str);
extern void exc_err_printf_indivis (const char *fmt, ...);
extern void exc_err_vprintf_indivis (const char *fmt, va_list l);

extern void exc_err_tab (unsigned int col);
extern void exc_err_underline (int on);
extern void exc_err_inverse (int on);
extern void exc_err_stand_out (int on);
extern void exc_err_clear (void);

/* 
 * Single line error messages 
 */

extern void exc_error (excErr e, const char *fmt, ...);
extern void exc_verror (excErr e, const char *fmt, va_list l);
extern void exc_perror (excErr e, const char *fmt, ...);
extern void exc_vperror (excErr e, const char *fmt, va_list l);

/* 
 * Throw excErr exceptions without adding text to the error message buffer
 */

void exc_err_throw_fatal (excErr e);    /* exit code 1 */
void exc_err_throw_nonfatal (excErr e); /* exit code 0 */

/* 
 * Displaying messages (used by exc_err_handler) 
 */

extern void (*exc_err_print_hook)(const char *str);
extern void exc_err_builtin_print (const char *str);

/* 
 * Default exception handler 
 */

extern int exc_err_handler (void *e, void *e_type, void *h_data);

/* 
 * Initialization 
 */

extern void exc_err_init (void);

#endif /* !_exception_exc_err_h */
