// OCT_QUAN.CPP - Octree Color Quantization Class

#include <stdio.h>
#include "oct_quan.h"

// Color palette
OctColor OctQuant::Palette[O_NumColor];

// Build octree
BOOL OctQuant::BuildTree()
{
  int row;              // Row counter
  int col;              // Column counter
  BOOL status = TRUE;   // Return status
  OctColor color;       // Pixel color

  // Allocate octree root node
  if ((proot = MakeNode(0)) == NULL)
    status = FALSE;

  // Build the octree
  if (status == TRUE)
    for (row = 0; row < height; row++)
    {
      for (col = 0; col < width; col++)
      {
        if (GetPixel(col, row, color) == FALSE)
        {
          status = FALSE;
          break;
        }

        // Insert pixel color into octree
        if (InsertNode(proot, color) == FALSE)
        {
          DeleteNode(proot);      // Delete the octree
          status =  FALSE;
          break;
        }

        // Reduce octree if too many colors
        if (num_leaf > O_NumColor)
          ReduceTree();
      }

      if (status == FALSE)
        break;
    }

  return status;
}

// Recursively delete octree nodes
void OctQuant::DeleteNode( OctNode *pn )
{
  int i;        // Loop index
  OctNode *pc;  // Child node pointer

  if (pn == NULL)
    return;

  if (pn->IsLeaf() == FALSE)
  {
    // Delete child nodes
    for (i = 0; i < 8; i++)
    {
      if ((pc = pn->GetChild(i)) != NULL)
      {
        DeleteNode(pc);
        pn->SetChild(i, NULL);
        pn->DecNumChild();
      }
    }
  }
  else
    num_leaf--;

  delete pn;
}

// Set color palette entries
void OctQuant::FillPalette( OctNode *pn, int *pindex)
{
  int i;    // Loop index

  // Perform recursive depth-first traversal of octree
  if (pn != NULL)
  {
    if ((pn->IsLeaf() == TRUE) || pn->GetLevel() ==
        leaf_level)
    {
      // Set color palette entry
      Palette[*pindex] = pn->GetColor();

      // Set node color palette index
      pn->SetIndex(*pindex);

      // Advance to next color palette entry
      *pindex = *pindex + 1;
    }
    else
    {
      // Visit child nodes
      for (i = 0; i < 8; i++)
        FillPalette(pn->GetChild(i), pindex);
    }
  }
}

// Get next reducible node pointer
OctNode *OctQuant::GetReducible()
{
  int new_level;        // New reducible node level
  OctNode *prn;         // Reducible node pointer
  OctNode *plcn = NULL; // Largest pixel count node
  OctNode *pnext;       // Next node
  OctNode *pprev;       // Previous node

  new_level = leaf_level - 1;

  // Find lowest reducible node level 
  while (prnl[new_level] == NULL)
    new_level--;

  // Find node with largest pixel count
  prn = prnl[new_level];
  while (prn != NULL)
  {
    if (plcn == NULL)
      plcn = prn;
    else if (prn->GetCount() < plcn->GetCount())
      plcn = prn;
    prn = prn->GetNext();
  }
  
  // Remove node from reducible list
  pnext = plcn->GetNext();
  pprev = plcn->GetPrev();
  if (pprev == NULL)
  {
    prnl[new_level] = pnext;
    if (pnext != NULL)
      pnext->SetPrev(NULL);
  }
  else
  {
    pprev->SetNext(pnext);
    if (pnext != NULL)
      pnext->SetPrev(pprev);
  }

  plcn->SetNext(NULL);
  plcn->SetPrev(NULL);
  plcn->SetMark(FALSE);
  
  return plcn;
}

void OctQuant::ReduceTree()     // Reduce octree
{
  int i;        // Loop index
  OctNode *pn;  // Node pointer
  OctNode *pc;  // Child node pointer

  pn = GetReducible();  // Get next reducible node

  // Delete children
  for (i = 0; i < 8; i++)
  {
    if ((pc = pn->GetChild(i)) != NULL)
    {
      DeleteNode(pc);
      pn->SetChild(i, NULL);
      pn->DecNumChild();
    }
  }

  pn->SetLeaf(TRUE);    // Mark node as leaf
  num_leaf++;           // Increment leaf count

  // Update reduction and leaf levels
  if (pn->GetLevel() < (leaf_level - 1))
    leaf_level = pn->GetLevel() + 1;
}

// Insert node into octree
BOOL OctQuant::InsertNode( OctNode *pn, OctColor &c)
{
  int c_index;          // Child index
  int level;            // Node level
  BOOL status = TRUE;   // Return status
  OctNode *pc;          // Child node pointer

  level = pn->GetLevel();       // Get node level
  pn->AddColor(c);              // Add RGB color to node

  if (pn->IsLeaf() == FALSE && level < leaf_level)
  {
    // Find child node
    c_index = pn->FindChild(c);
    if ((pc = pn->GetChild(c_index)) != NULL)
    {
      if ((pn->GetNumChild() > 1) && (pn->IsMark() ==
          FALSE))
      {
        // Mark node as candidate for reduction
        MakeReducible(pn);
      }
    }
    else
    {
      // Allocate child node
      if ((pc = MakeNode(level + 1)) == NULL)
        return FALSE;

      // Set child node pointer
      pn->SetChild(c_index, pc);

      pn->IncNumChild();      // Increment child count
    }

    // Insert child node into octree
    status = InsertNode(pc, c);
  }

  return status;
}

