#ifndef STLREADER_H
#define STLREADER_H

#include "GeoCommon.h"
#include "ModelReader.h"
#include <stdlib.h>
#include <stdio.h>

/**
 * Class for reading STL (binary) files
 * for STL specification, see http://rpdrc.ic.polyu.edu.hk/old_files/stl_binary_format.htm
 *
 * @author Tao Ju
 */
class STLReader : public ModelReader
{
public:
	// File handle
	FILE* fin ;

	// Number of triangles
	int numTrians, curtrian ;

	// Bounding box
	float min[3], max[3] ;

	// Size of box
	float maxsize ;

	// Offset
	long offset ;

	// For output points only
	Triangle* curT ;
	int curVert ;

public:
	/// Constructor
	STLReader( char* fname, float scl ) 
	{
		if ( ! ( fin = fopen( fname, "rb" ) ) )
		{
			printf("Unable to open file %s\n", fname) ;	
		};

		// Scan to get info
		// printf("Scanning file for bounding box...\n");
		
		// Parse header
		char header[84] ;
		fread( header, sizeof( char ), 80, fin ) ;

		// Face count
		fread( &numTrians, sizeof( int ), 1, fin ) ;
		curtrian = 0 ;
		this->offset = ftell( fin ) ;

		// Read each face
		for ( int i = 0 ; i < numTrians ; i ++ )
		{
			char temp[2] ;
			float normal[3], v[3][3] ;

			// Face normal
			fread( normal, sizeof( float ), 3, fin ) ;
			// Face vertices
			fread( v[0], sizeof( float ), 3, fin ) ;
			fread( v[1], sizeof( float ), 3, fin ) ;
			fread( v[2], sizeof( float ), 3, fin ) ;
			// Tail
			fread( temp, sizeof( char ), 2, fin ) ;

			if ( i == 0 )
			{
				for ( int j = 0 ; j < 3 ; j ++ )
				{
					min[j] = max[j] = v[0][j] ;
				}
			}

			for ( int j = 0 ; j < 3 ; j ++ )
			{
				for ( int k = 0 ; k < 3 ; k ++ )
				{
					if ( min[k] > v[j][k] )
					{
						min[k] = v[j][k] ;
					}
					if ( max[k] < v[j][k] )
					{
						max[k] = v[j][k] ;
					}
				}
			}
		}

		maxsize = max[0] - min[0] ;
		if ( max[1] - min[1] > maxsize )
		{
			maxsize = max[1] - min[1] ;
		}
		if ( max[2] - min[2] > maxsize )
		{
			maxsize = max[2] - min[2] ;
		}

		for ( i = 0 ; i < 3 ; i ++ )
		{
			min[i] = ( max[i] + min[i] ) / 2 - maxsize / 2 ;
			max[i] = ( max[i] + min[i] ) / 2 + maxsize / 2 ;
		}
		
		// printf( "Low corner: %f %f %f   High corner %f %f %f \n", min[0], min[1], min[2], max[0], max[1], max[2] );
		// printf( "Maxdif: %f \n", maxsize );

		// Scale
		for ( i = 0 ; i < 3 ; i ++ )
		{
			min[i] -= maxsize * ( 1 / scl - 1 ) / 2 ;
		}
		maxsize *= ( 1 / scl ) ;


		// Reset
		reset( ) ;
		
	};

	/// Get next triangle
	Triangle* getNextTriangle( )
	{
		if ( curtrian == numTrians )
		{
			return NULL ;
		}		

		char temp[2] ;
		float normal[3];
		Triangle* t = new Triangle() ;
		
		// Face normal
		fread( normal, sizeof( float ), 3, fin ) ;
		// Face vertices
		fread( t->vt[0], sizeof( float ), 3, fin ) ;
		fread( t->vt[1], sizeof( float ), 3, fin ) ;
		fread( t->vt[2], sizeof( float ), 3, fin ) ;
		// Tail
		fread( temp, sizeof( char ), 2, fin ) ;

		curtrian ++ ;

		return t ;
	};

	int getNextTriangle( int t[3] )
	{
		if ( curtrian == numTrians )
		{
			return 0 ;
		}		

		t[0] = curtrian * 3 ;
		t[1] = curtrian * 3 + 1 ;
		t[2] = curtrian * 3 + 2 ;

		curtrian ++ ;

		return 1 ;
	}

	void getNextVertex( float v[3] )
	{
		if ( curVert == 0 )
		{
			curT = getNextTriangle( ) ;
		}

		v[0] = curT->vt[curVert][0] ;
		v[1] = curT->vt[curVert][1] ;
		v[2] = curT->vt[curVert][2] ;

		curVert = ( curVert + 1 ) % 3 ;
	}

	/// Get bounding box
	float getBoundingBox ( float origin[3] )
	{
		for ( int i = 0 ; i < 3 ; i ++ )
		{
			origin[i] = min[i] ;
		}

		return maxsize ;
	};

	/// Get storage size
	int getMemory ( ) 
	{
		return sizeof ( class STLReader ) ;	
	};

	int getNumTriangles()
	{
		return this->numTrians ;
	}

	int getNumVertices ( )
	{
		return this->numTrians * 3 ;
	}

	void reset ( )
	{
		fseek( this->fin, offset, SEEK_SET ) ;
		this->curtrian = 0 ;
		curT = NULL ;
		curVert = 0 ;
	}

	void printInfo( )
	{
		printf( "Polygons: %d\n", this->numTrians ) ;
	}
};


#endif