

/*
 * $Id: fqdncache.c,v 1.112 1998/08/21 08:41:56 wessels Exp $
 *
 * DEBUG: section 35    FQDN Cache
 * AUTHOR: Harvest Derived
 *
 * SQUID Internet Object Cache  http://squid.nlanr.net/Squid/
 * ----------------------------------------------------------
 *
 *  Squid is the result of efforts by numerous individuals from the
 *  Internet community.  Development is led by Duane Wessels of the
 *  National Laboratory for Applied Network Research and funded by the
 *  National Science Foundation.  Squid is Copyrighted (C) 1998 by
 *  Duane Wessels and the University of California San Diego.  Please
 *  see the COPYRIGHT file for full details.  Squid incorporates
 *  software developed and/or copyrighted by other sources.  Please see
 *  the CREDITS file for full details.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111, USA.
 *
 */

#include "squid.h"

#define MAX_LINELEN (4096)

#define MAX_FQDN		 1024	/* Maximum cached FQDN */
#define FQDN_LOW_WATER       90
#define FQDN_HIGH_WATER      95

struct fqdncacheQueueData {
    struct fqdncacheQueueData *next;
    fqdncache_entry *f;
};

static struct {
    int requests;
    int replies;
    int hits;
    int misses;
    int pending_hits;
    int negative_hits;
    int errors;
    int ghba_calls;		/* # calls to blocking gethostbyaddr() */
} FqdncacheStats;

static dlink_list lru_list;

static void fqdncache_dnsHandleRead(int, void *);
static fqdncache_entry *fqdncacheParse(const char *buf, dnsserver_t *);
static void fqdncache_release(fqdncache_entry *);
static fqdncache_entry *fqdncache_create(const char *name);
static void fqdncache_call_pending(fqdncache_entry *);
static void fqdncacheAddHostent(fqdncache_entry *, const struct hostent *);
static int fqdncacheHasPending(const fqdncache_entry *);
static fqdncache_entry *fqdncache_get(const char *);
static FQDNH dummy_handler;
static int fqdncacheExpiredEntry(const fqdncache_entry *);
static void fqdncacheAddPending(fqdncache_entry *, FQDNH *, void *);
static void fqdncacheEnqueue(fqdncache_entry *);
static void *fqdncacheDequeue(void);
static void fqdncache_dnsDispatch(dnsserver_t *, fqdncache_entry *);
static void fqdncacheChangeKey(fqdncache_entry * i);
static void fqdncacheLockEntry(fqdncache_entry * f);
static void fqdncacheUnlockEntry(fqdncache_entry * f);
static FREE fqdncacheFreeEntry;

static hash_table *fqdn_table = NULL;
static struct fqdncacheQueueData *fqdncacheQueueHead = NULL;
static struct fqdncacheQueueData **fqdncacheQueueTailP = &fqdncacheQueueHead;
static int queue_length = 0;

static char fqdncache_status_char[] =
{
    'C',
    'N',
    'P',
    'D'
};

static long fqdncache_low = 180;
static long fqdncache_high = 200;

static void
fqdncacheEnqueue(fqdncache_entry * f)
{
    static time_t last_warning = 0;
    struct fqdncacheQueueData *new = xcalloc(1, sizeof(struct fqdncacheQueueData));
    new->f = f;
    *fqdncacheQueueTailP = new;
    fqdncacheQueueTailP = &new->next;
    queue_length++;
    if (queue_length < NDnsServersAlloc)
	return;
    if (squid_curtime - last_warning < 600)
	return;
    last_warning = squid_curtime;
    debug(35, 0) ("fqdncacheEnqueue: WARNING: All dnsservers are busy.\n");
    debug(35, 0) ("fqdncacheEnqueue: WARNING: %d DNS lookups queued\n", queue_length);
    if (queue_length > NDnsServersAlloc * 2)
	fatal("Too many queued DNS lookups");
    if (Config.dnsChildren >= DefaultDnsChildrenMax)
	return;
    debug(35, 1) ("fqdncacheEnqueue: Consider increasing 'dns_children' in your config file.\n");
}

static void *
fqdncacheDequeue(void)
{
    struct fqdncacheQueueData *old = NULL;
    fqdncache_entry *f = NULL;
    if (fqdncacheQueueHead) {
	f = fqdncacheQueueHead->f;
	old = fqdncacheQueueHead;
	fqdncacheQueueHead = fqdncacheQueueHead->next;
	if (fqdncacheQueueHead == NULL)
	    fqdncacheQueueTailP = &fqdncacheQueueHead;
	safe_free(old);
	queue_length--;
    }
    if (f && f->status != FQDN_PENDING)
	debug_trap("fqdncacheDequeue: status != FQDN_PENDING");
    return f;
}

/* removes the given fqdncache entry */
static void
fqdncache_release(fqdncache_entry * f)
{
    int k;
    assert(f->status != FQDN_PENDING);
    assert(f->status != FQDN_DISPATCHED);
    assert(f->pending_head == NULL);
    hash_remove_link(fqdn_table, (hash_link *) f);
    if (f->status == FQDN_CACHED) {
	for (k = 0; k < (int) f->name_count; k++)
	    safe_free(f->names[k]);
	debug(35, 5) ("fqdncache_release: Released FQDN record for '%s'.\n",
	    f->name);
    }
    dlinkDelete(&f->lru, &lru_list);
    safe_free(f->name);
    safe_free(f->error_message);
    memFree(MEM_FQDNCACHE_ENTRY, f);
}

/* return match for given name */
static fqdncache_entry *
fqdncache_get(const char *name)
{
    hash_link *e;
    static fqdncache_entry *f;

    f = NULL;
    if (fqdn_table) {
	if ((e = hash_lookup(fqdn_table, name)) != NULL)
	    f = (fqdncache_entry *) e;
    }
    return f;
}

static int
fqdncacheExpiredEntry(const fqdncache_entry * f)
{
    if (f->status == FQDN_PENDING)
	return 0;
    if (f->status == FQDN_DISPATCHED)
	return 0;
    if (f->locks != 0)
	return 0;
    if (f->expires > squid_curtime)
	return 0;
    return 1;
}

void
fqdncache_purgelru(void *notused)
{
    dlink_node *m;
    dlink_node *prev = NULL;
    fqdncache_entry *f;
    int removed = 0;
    eventAdd("fqdncache_purgelru", fqdncache_purgelru, NULL, 10.0, 1);
    for (m = lru_list.tail; m; m = prev) {
	if (memInUse(MEM_FQDNCACHE_ENTRY) < fqdncache_low)
	    break;
	prev = m->prev;
	f = m->data;
	if (f->status == FQDN_PENDING)
	    continue;
	if (f->status == FQDN_DISPATCHED)
	    continue;
	if (f->locks != 0)
	    continue;
	fqdncache_release(f);
	removed++;
    }
    debug(35, 3) ("fqdncache_purgelru: removed %d entries\n", removed);
}


/* create blank fqdncache_entry */
static fqdncache_entry *
fqdncache_create(const char *name)
{
    static fqdncache_entry *f;
    f = memAllocate(MEM_FQDNCACHE_ENTRY);
    f->name = xstrdup(name);
    f->expires = squid_curtime + Config.negativeDnsTtl;
    hash_join(fqdn_table, (hash_link *) f);
    dlinkAdd(f, &f->lru, &lru_list);
    return f;
}

static void
fqdncacheAddHostent(fqdncache_entry * f, const struct hostent *hp)
{
    int k;
    f->name_count = 0;
    f->names[f->name_count++] = xstrdup((char *) hp->h_name);
    for (k = 0; hp->h_aliases[k]; k++) {
	f->names[f->name_count++] = xstrdup(hp->h_aliases[k]);
	if (f->name_count == FQDN_MAX_NAMES)
	    break;
    }
}

static fqdncache_entry *
fqdncacheAddNew(const char *name, const struct hostent *hp, fqdncache_status_t status)
{
    fqdncache_entry *f;
    assert(fqdncache_get(name) == NULL);
    debug(35, 10) ("fqdncacheAddNew: Adding '%s', status=%c\n",
	name,
	fqdncache_status_char[status]);
    f = fqdncache_create(name);
    if (hp)
	fqdncacheAddHostent(f, hp);
    f->status = status;
    f->lastref = squid_curtime;
    return f;
}

/* walks down the pending list, calling handlers */
static void
fqdncache_call_pending(fqdncache_entry * f)
{
    fqdn_pending *p = NULL;
    int nhandler = 0;

    f->lastref = squid_curtime;

    fqdncacheLockEntry(f);
    while (f->pending_head != NULL) {
	p = f->pending_head;
	f->pending_head = p->next;
	if (p->handler) {
	    nhandler++;
	    dns_error_message = f->error_message;
	    p->handler((f->status == FQDN_CACHED) ? f->names[0] : NULL,
		p->handlerData);
	}
	memFree(MEM_FQDNCACHE_PENDING, p);
    }
    f->pending_head = NULL;	/* nuke list */
    debug(35, 10) ("fqdncache_call_pending: Called %d handlers.\n", nhandler);
    fqdncacheUnlockEntry(f);
}


static fqdncache_entry *
fqdncacheParse(const char *inbuf, dnsserver_t * dnsData)
{
    LOCAL_ARRAY(char, buf, DNS_INBUF_SZ);
    char *token;
    static fqdncache_entry f;
    int ttl;
    xstrncpy(buf, inbuf, DNS_INBUF_SZ);
    debug(35, 5) ("fqdncacheParse: parsing:\n%s", buf);
    memset(&f, '\0', sizeof(f));
    f.expires = squid_curtime;
    f.status = FQDN_NEGATIVE_CACHED;
    token = strtok(buf, w_space);
    if (NULL == token) {
	debug(35, 1) ("fqdncacheParse: Got <NULL>, expecting '$name'\n");
	return &f;
    }
    if (0 == strcmp(token, "$fail")) {
	f.expires = squid_curtime + Config.negativeDnsTtl;
	token = strtok(NULL, "\n");
	assert(NULL != token);
	f.error_message = xstrdup(token);
	return &f;
    }
    if (0 != strcmp(token, "$name")) {
	debug(35, 1) ("fqdncacheParse: Got '%s', expecting '$name'\n", token);
	return &f;
    }
    token = strtok(NULL, w_space);
    if (NULL == token) {
	debug(35, 1) ("fqdncacheParse: Got <NULL>, expecting TTL\n");
	return &f;
    }
    f.status = FQDN_CACHED;
    ttl = atoi(token);
    if (ttl > 0)
	f.expires = squid_curtime + ttl;
    else
	f.expires = squid_curtime + Config.positiveDnsTtl;
    token = strtok(NULL, w_space);
    if (NULL != token) {
	f.names[0] = xstrdup(token);
	f.name_count = 1;
    }
    return &f;
}

static void
fqdncacheNudgeQueue(void)
{
    dnsserver_t *dnsData;
    fqdncache_entry *f = NULL;
    while ((dnsData = dnsGetFirstAvailable()) && (f = fqdncacheDequeue()))
	fqdncache_dnsDispatch(dnsData, f);
}

static void
fqdncache_dnsHandleRead(int fd, void *data)
{
    dnsserver_t *dnsData = data;
    int len;
    int n;
    fqdncache_entry *f = NULL;
    fqdncache_entry *x = NULL;

    len = read(fd,
	dnsData->ip_inbuf + dnsData->offset,
	dnsData->size - dnsData->offset);
    fd_bytes(fd, len, FD_READ);
    debug(35, 5) ("fqdncache_dnsHandleRead: Result from DNS ID %d (%d bytes)\n",
	dnsData->id, len);
    if (len <= 0) {
	if (len < 0 && ignoreErrno(errno)) {
	    commSetSelect(fd,
		COMM_SELECT_READ,
		fqdncache_dnsHandleRead,
		dnsData, 0);
	    return;
	}
	debug(35, EBIT_TEST(dnsData->flags, HELPER_CLOSING) ? 5 : 1)
	    ("FD %d: Connection from DNSSERVER #%d is closed, disabling\n",
	    fd, dnsData->id);
	dnsData->flags = 0;
	commSetSelect(fd,
	    COMM_SELECT_WRITE,
	    NULL,
	    NULL,
	    0);
	comm_close(fd);
	return;
    }
    n = ++FqdncacheStats.replies;
    dnsData->offset += len;
    dnsData->ip_inbuf[dnsData->offset] = '\0';
    f = dnsData->data;
    if (f->status != FQDN_DISPATCHED)
	fatal_dump("fqdncache_dnsHandleRead: bad status");
    if (strchr(dnsData->ip_inbuf, '\n')) {
	/* end of record found */
	DnsStats.replies++;
	statHistCount(&Counter.dns.svc_time,
	    tvSubMsec(dnsData->dispatch_time, current_time));
	if ((x = fqdncacheParse(dnsData->ip_inbuf, dnsData)) == NULL) {
	    debug(35, 0) ("fqdncache_dnsHandleRead: fqdncacheParse failed?!\n");
	} else {
	    dnsData->offset = 0;
	    dnsData->ip_inbuf[0] = '\0';
	    f->name_count = x->name_count;
	    for (n = 0; n < (int) f->name_count; n++)
		f->names[n] = x->names[n];
	    f->error_message = x->error_message;
	    f->status = x->status;
	    f->expires = x->expires;
	    fqdncache_call_pending(f);
	}
	fqdncacheUnlockEntry(f);	/* unlock from FQDN_DISPATCHED */
    } else {
	debug(35, 5) ("fqdncache_dnsHandleRead: Incomplete reply\n");
	commSetSelect(fd,
	    COMM_SELECT_READ,
	    fqdncache_dnsHandleRead,
	    dnsData,
	    0);
    }
    if (dnsData->offset == 0) {
	dnsData->data = NULL;
	EBIT_CLR(dnsData->flags, HELPER_BUSY);
	if (EBIT_TEST(dnsData->flags, HELPER_SHUTDOWN))
	    dnsShutdownServer(dnsData);
	cbdataUnlock(dnsData);
    }
    fqdncacheNudgeQueue();
}

static void
fqdncacheAddPending(fqdncache_entry * f, FQDNH * handler, void *handlerData)
{
    fqdn_pending *pending = memAllocate(MEM_FQDNCACHE_PENDING);
    fqdn_pending **I = NULL;
    f->lastref = squid_curtime;
    pending->handler = handler;
    pending->handlerData = handlerData;
    for (I = &(f->pending_head); *I; I = &((*I)->next));
    *I = pending;
    if (f->status == FQDN_PENDING)
	fqdncacheNudgeQueue();
}

void
fqdncache_nbgethostbyaddr(struct in_addr addr, FQDNH * handler, void *handlerData)
{
    fqdncache_entry *f = NULL;
    dnsserver_t *dnsData = NULL;
    char *name = inet_ntoa(addr);

    if (!handler)
	fatal_dump("fqdncache_nbgethostbyaddr: NULL handler");

    debug(35, 4) ("fqdncache_nbgethostbyaddr: Name '%s'.\n", name);
    FqdncacheStats.requests++;

    if (name == NULL || name[0] == '\0') {
	debug(35, 4) ("fqdncache_nbgethostbyaddr: Invalid name!\n");
	handler(NULL, handlerData);
	return;
    }
    if ((f = fqdncache_get(name))) {
	if (fqdncacheExpiredEntry(f)) {
	    fqdncache_release(f);
	    f = NULL;
	}
    }
    if (f == NULL) {
	/* MISS: No entry, create the new one */
	debug(35, 5) ("fqdncache_nbgethostbyaddr: MISS for '%s'\n", name);
	FqdncacheStats.misses++;
	f = fqdncacheAddNew(name, NULL, FQDN_PENDING);
	fqdncacheAddPending(f, handler, handlerData);
    } else if (f->status == FQDN_CACHED || f->status == FQDN_NEGATIVE_CACHED) {
	/* HIT */
	debug(35, 4) ("fqdncache_nbgethostbyaddr: HIT for '%s'\n", name);
	if (f->status == FQDN_NEGATIVE_CACHED)
	    FqdncacheStats.negative_hits++;
	else
	    FqdncacheStats.hits++;
	fqdncacheAddPending(f, handler, handlerData);
	fqdncache_call_pending(f);
	return;
    } else if (f->status == FQDN_PENDING || f->status == FQDN_DISPATCHED) {
	debug(35, 4) ("fqdncache_nbgethostbyaddr: PENDING for '%s'\n", name);
	FqdncacheStats.pending_hits++;
	fqdncacheAddPending(f, handler, handlerData);
	if (squid_curtime - f->expires > 600) {
	    debug(35, 0) ("fqdncache_nbgethostbyname: '%s' PENDING for %d seconds, aborting\n", name,
		(int) (squid_curtime + Config.negativeDnsTtl - f->expires));
	    fqdncacheChangeKey(f);
	    fqdncache_call_pending(f);
	}
	return;
    } else {
	fatal_dump("fqdncache_nbgethostbyaddr: BAD fqdncache_entry status");
    }

    /* for HIT, PENDING, DISPATCHED we've returned.  For MISS we continue */

    if ((dnsData = dnsGetFirstAvailable())) {
	fqdncache_dnsDispatch(dnsData, f);
    } else if (NDnsServersAlloc > 0) {
	fqdncacheEnqueue(f);
    } else {
	/* abort if we get here */
	assert(NDnsServersAlloc);
    }
}

static void
fqdncache_dnsDispatch(dnsserver_t * dns, fqdncache_entry * f)
{
    char *buf = NULL;
    assert(EBIT_TEST(dns->flags, HELPER_ALIVE));
    if (!fqdncacheHasPending(f)) {
	debug(35, 0) ("fqdncache_dnsDispatch: skipping '%s' because no handler.\n",
	    f->name);
	f->status = FQDN_NEGATIVE_CACHED;
	fqdncache_release(f);
	return;
    }
    if (f->status != FQDN_PENDING)
	debug_trap("fqdncache_dnsDispatch: status != FQDN_PENDING");
    buf = xcalloc(1, 256);
    snprintf(buf, 256, "%s\n", f->name);
    EBIT_SET(dns->flags, HELPER_BUSY);
    dns->data = f;
    f->status = FQDN_DISPATCHED;
    comm_write(dns->outpipe,
	buf,
	strlen(buf),
	NULL,			/* Handler */
	NULL,			/* Handler-data */
	xfree);
    cbdataLock(dns);
    commSetSelect(dns->outpipe,
	COMM_SELECT_READ,
	fqdncache_dnsHandleRead,
	dns,
	0);
    debug(35, 5) ("fqdncache_dnsDispatch: Request sent to DNS server #%d.\n",
	dns->id);
    dns->dispatch_time = current_time;
    DnsStats.requests++;
    DnsStats.hist[dns->id - 1]++;
    fqdncacheLockEntry(f);	/* lock while FQDN_DISPATCHED */
}


/* initialize the fqdncache */
void
fqdncache_init(void)
{
    int n;
    if (fqdn_table)
	return;
    debug(35, 3) ("Initializing FQDN Cache...\n");
    memset(&FqdncacheStats, '\0', sizeof(FqdncacheStats));
    memset(&lru_list, '\0', sizeof(lru_list));
    fqdncache_high = (long) (((float) MAX_FQDN *
	    (float) FQDN_HIGH_WATER) / (float) 100);
    fqdncache_low = (long) (((float) MAX_FQDN *
	    (float) FQDN_LOW_WATER) / (float) 100);
    n = hashPrime(fqdncache_high / 4);
    fqdn_table = hash_create((HASHCMP *) strcmp, n, hash4);
    cachemgrRegister("fqdncache",
	"FQDN Cache Stats and Contents",
	fqdnStats, 0, 1);
}

/* clean up the pending entries in dnsserver */
/* return 1 if we found the host, 0 otherwise */
int
fqdncacheUnregister(struct in_addr addr, void *data)
{
    char *name = inet_ntoa(addr);
    fqdncache_entry *f = NULL;
    fqdn_pending *p = NULL;
    int n = 0;
    debug(35, 3) ("fqdncacheUnregister: name '%s'\n", name);
    if ((f = fqdncache_get(name)) == NULL)
	return 0;
    if (f->status == FQDN_PENDING || f->status == FQDN_DISPATCHED) {
	for (p = f->pending_head; p; p = p->next) {
	    if (p->handlerData != data)
		continue;
	    p->handler = NULL;
	    n++;
	}
    }
    if (n == 0)
	debug_trap("fqdncacheUnregister: callback data not found");
    debug(35, 3) ("fqdncacheUnregister: unregistered %d handlers\n", n);
    return n;
}

const char *
fqdncache_gethostbyaddr(struct in_addr addr, int flags)
{
    char *name = inet_ntoa(addr);
    fqdncache_entry *f = NULL;
    struct in_addr ip;

    if (!name)
	fatal_dump("fqdncache_gethostbyaddr: NULL name");
    FqdncacheStats.requests++;
    if ((f = fqdncache_get(name))) {
	if (fqdncacheExpiredEntry(f)) {
	    fqdncache_release(f);
	    f = NULL;
	}
    }
    if (f) {
	if (f->status == FQDN_NEGATIVE_CACHED) {
	    FqdncacheStats.negative_hits++;
	    dns_error_message = f->error_message;
	    return NULL;
	} else {
	    FqdncacheStats.hits++;
	    f->lastref = squid_curtime;
	    return f->names[0];
	}
    }
    /* check if it's already a FQDN address in text form. */
    if (!safe_inet_addr(name, &ip))
	return name;
    FqdncacheStats.misses++;
    if (flags & FQDN_LOOKUP_IF_MISS)
	fqdncache_nbgethostbyaddr(addr, dummy_handler, NULL);
    return NULL;
}


/* process objects list */
void
fqdnStats(StoreEntry * sentry)
{
    fqdncache_entry *f = NULL;
    int k;
    int ttl;
    if (fqdn_table == NULL)
	return;
    storeAppendPrintf(sentry, "FQDN Cache Statistics:\n");
    storeAppendPrintf(sentry, "FQDNcache Entries: %d\n",
	memInUse(MEM_FQDNCACHE_ENTRY));
    storeAppendPrintf(sentry, "FQDNcache Requests: %d\n",
	FqdncacheStats.requests);
    storeAppendPrintf(sentry, "FQDNcache Hits: %d\n",
	FqdncacheStats.hits);
    storeAppendPrintf(sentry, "FQDNcache Pending Hits: %d\n",
	FqdncacheStats.pending_hits);
    storeAppendPrintf(sentry, "FQDNcache Negative Hits: %d\n",
	FqdncacheStats.negative_hits);
    storeAppendPrintf(sentry, "FQDNcache Misses: %d\n",
	FqdncacheStats.misses);
    storeAppendPrintf(sentry, "Blocking calls to gethostbyaddr(): %d\n",
	FqdncacheStats.ghba_calls);
    storeAppendPrintf(sentry, "pending queue length: %d\n", queue_length);
    storeAppendPrintf(sentry, "FQDN Cache Contents:\n\n");

    hash_first(fqdn_table);
    while ((f = (fqdncache_entry *) hash_next(fqdn_table))) {
	if (f->status == FQDN_PENDING || f->status == FQDN_DISPATCHED)
	    ttl = 0;
	else
	    ttl = (f->expires - squid_curtime);
	storeAppendPrintf(sentry, " %-32.32s %c %6d %d",
	    f->name,
	    fqdncache_status_char[f->status],
	    ttl,
	    (int) f->name_count);
	for (k = 0; k < (int) f->name_count; k++)
	    storeAppendPrintf(sentry, " %s", f->names[k]);
	storeAppendPrintf(sentry, "\n");
    }
}

static void
dummy_handler(const char *bufnotused, void *datanotused)
{
    return;
}

static int
fqdncacheHasPending(const fqdncache_entry * f)
{
    const fqdn_pending *p = NULL;
    if (f->status != FQDN_PENDING)
	return 0;
    for (p = f->pending_head; p; p = p->next)
	if (p->handler)
	    return 1;
    return 0;
}

void
fqdncacheReleaseInvalid(const char *name)
{
    fqdncache_entry *f;
    if ((f = fqdncache_get(name)) == NULL)
	return;
    if (f->status != FQDN_NEGATIVE_CACHED)
	return;
    fqdncache_release(f);
}

const char *
fqdnFromAddr(struct in_addr addr)
{
    const char *n;
    static char buf[32];
    if (Config.onoff.log_fqdn && (n = fqdncache_gethostbyaddr(addr, 0)))
	return n;
    xstrncpy(buf, inet_ntoa(addr), 32);
    return buf;
}

static void
fqdncacheLockEntry(fqdncache_entry * f)
{
    if (f->locks++ == 0) {
	dlinkDelete(&f->lru, &lru_list);
	dlinkAdd(f, &f->lru, &lru_list);
    }
}

static void
fqdncacheUnlockEntry(fqdncache_entry * f)
{
    if (f->locks == 0) {
	debug_trap("fqdncacheUnlockEntry: Entry has no locks");
	return;
    }
    f->locks--;
    if (fqdncacheExpiredEntry(f))
	fqdncache_release(f);
}

static void
fqdncacheFreeEntry(void *data)
{
    fqdncache_entry *f = data;
    fqdn_pending *p = NULL;
    int k;
    while ((p = f->pending_head)) {
	f->pending_head = p->next;
	memFree(MEM_FQDNCACHE_PENDING, p);
    }
    for (k = 0; k < (int) f->name_count; k++)
	safe_free(f->names[k]);
    safe_free(f->name);
    safe_free(f->error_message);
    memFree(MEM_FQDNCACHE_ENTRY, f);
}

void
fqdncacheFreeMemory(void)
{
    hashFreeItems(fqdn_table, fqdncacheFreeEntry);
    hashFreeMemory(fqdn_table);
    fqdn_table = NULL;
}

static void
fqdncacheChangeKey(fqdncache_entry * f)
{
    static int index = 0;
    LOCAL_ARRAY(char, new_key, 256);
    hash_link *table_entry = hash_lookup(fqdn_table, f->name);
    if (table_entry == NULL) {
	debug(35, 0) ("fqdncacheChangeKey: Could not find key '%s'\n", f->name);
	return;
    }
    if (f != (fqdncache_entry *) table_entry) {
	debug_trap("fqdncacheChangeKey: f != table_entry!");
	return;
    }
    hash_remove_link(fqdn_table, table_entry);
    snprintf(new_key, 256, "%d/", ++index);
    strncat(new_key, f->name, 128);
    debug(35, 1) ("fqdncacheChangeKey: from '%s' to '%s'\n", f->name, new_key);
    safe_free(f->name);
    f->name = xstrdup(new_key);
    hash_join(fqdn_table, (hash_link *) f);
}

/* call during reconfigure phase to clear out all the
 * pending and dispatched reqeusts that got lost */
void
fqdncache_restart(void)
{
    fqdncache_entry *this;
    if (fqdn_table == 0)
	fatal_dump("fqdncache_restart: fqdn_table == 0\n");
    while (fqdncacheDequeue());
    hash_first(fqdn_table);
    while ((this = (fqdncache_entry *) hash_next(fqdn_table))) {
	if (this->status == FQDN_CACHED)
	    continue;
	if (this->status == FQDN_NEGATIVE_CACHED)
	    continue;
    }
    fqdncache_high = (long) (((float) MAX_FQDN *
	    (float) FQDN_HIGH_WATER) / (float) 100);
    fqdncache_low = (long) (((float) MAX_FQDN *
	    (float) FQDN_LOW_WATER) / (float) 100);
}

#ifdef SQUID_SNMP

int
fqdn_getMax()
{
    int i = 0;
    fqdncache_entry *fq = NULL;

    hash_first(fqdn_table);
    while ((fq = (fqdncache_entry *) hash_next(fqdn_table)))
	i++;
    return i;
}

variable_list *
snmp_fqdncacheFn(variable_list * Var, snint * ErrP)
{
    variable_list *Answer;
    static fqdncache_entry *fq = NULL;
    static struct in_addr fqaddr;
    int cnt = 0;

    debug(49, 5) ("snmp_fqdncacheFn: Processing request with %d.%d!\n", Var->name[11], Var->name[12]);

    cnt = Var->name[12];

    hash_first(fqdn_table);
    while (cnt--) {
	fq = (fqdncache_entry *) hash_next(fqdn_table);
	if (NULL == fq)
	    break;
    }
    if (fq == NULL || cnt != 0) {
	*ErrP = SNMP_ERR_NOSUCHNAME;
	return NULL;
    }
    Answer = snmp_var_new(Var->name, Var->name_length);
    *ErrP = SNMP_ERR_NOERROR;

    switch (Var->name[11]) {
    case NET_FQDN_ID:
	Answer->type = ASN_INTEGER;
	Answer->val_len = sizeof(snint);
	Answer->val.integer = xmalloc(Answer->val_len);
	*(Answer->val.integer) = Var->name[12];
	break;
    case NET_FQDN_NAME:
	Answer->type = SMI_STRING;
	Answer->val_len = strlen(fq->names[0]);
	Answer->val.string = (u_char *) xstrdup(fq->names[0]);
	break;
    case NET_FQDN_IP:
	Answer->type = SMI_IPADDRESS;
	Answer->val_len = sizeof(snint);
	Answer->val.integer = xmalloc(Answer->val_len);
	safe_inet_addr(fq->name, &fqaddr);
	*(Answer->val.integer) = (snint) fqaddr.s_addr;
	break;
    case NET_FQDN_LASTREF:
	Answer->type = SMI_TIMETICKS;
	Answer->val_len = sizeof(snint);
	Answer->val.integer = xmalloc(Answer->val_len);
	*(Answer->val.integer) = squid_curtime - fq->lastref;
	break;
    case NET_FQDN_EXPIRES:
	Answer->type = SMI_TIMETICKS;
	Answer->val_len = sizeof(snint);
	Answer->val.integer = xmalloc(Answer->val_len);
	*(Answer->val.integer) = fq->expires - squid_curtime;
	break;
    case NET_FQDN_STATE:
	Answer->type = ASN_INTEGER;
	Answer->val_len = sizeof(snint);
	Answer->val.integer = xmalloc(Answer->val_len);
	*(Answer->val.integer) = fq->status;
	break;
    default:
	*ErrP = SNMP_ERR_NOSUCHNAME;
	snmp_var_free(Answer);
	return (NULL);
    }
    return Answer;
}
#endif
