//::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
//
// NetscapePlugin.cpp
//
//
//::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
#include "CMacPluginView.h"

#include <string.h>
#include "Files.h"
#include "Resources.h"

#include <UDrawingState.h>
#include <UEnvironment.h>
#include <URegistrar.h>
#include <UReanimator.h>

#include <LScroller.h>
#include <LTextEdit.h>
#include <LActiveScroller.h>
#include <LCaption.h>

extern QDGlobals*	gQDPtr;
extern short		gResFile;

const ResIDT		rPPob_PluginView	= 1001;

//======================================================================
//		CMacPluginView::Initialize()
//======================================================================

NPError CMacPluginView::Initialize()
{
	UQDGlobals::SetQDGlobals(gQDPtr);
	
	Int32		qdVersion = gestaltOriginalQD;
	Gestalt(gestaltQuickdrawVersion, &qdVersion);
	UEnvironment::SetFeature(env_SupportsColor, (qdVersion > gestaltOriginalQD));

	URegistrar::RegisterClass(LGrafPortView::class_ID,	(ClassCreatorFunc) LGrafPortView::CreateGrafPortViewStream);
	URegistrar::RegisterClass(LPane::class_ID,			(ClassCreatorFunc) LPane::CreatePaneStream);
	URegistrar::RegisterClass(LView::class_ID,			(ClassCreatorFunc) LView::CreateViewStream);

	return NPERR_NO_ERROR;
}

//======================================================================
//		CMacPluginView::CMacPluginView()
//======================================================================

CMacPluginView::CMacPluginView( CNetscapePlugin* controller, Int16 inFrameThickness )
:	CPluginView( controller ),
	LGrafPortView(),
	mFramePenThickness( inFrameThickness ),
	mNetscapeWindow( nil ),
	mContentPane( nil )
{
//	SBooleanRect newBinding = { false, false, false, false };
//	SetFrameBinding( newBinding );
	mForeColor.red = mForeColor.green = mForeColor.blue = 32768;	// Grey

	mContentPane = CreateContentPane( );
	mContentPane->PutInside( this );
}

//======================================================================
//		CMacPluginView::~CMacPluginView()
//======================================================================

CMacPluginView::~CMacPluginView()
{
}

//::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
//
//	Object Interface Methods
//
//	These methods are called by the procedural methods above. They take care of basic housekeeping,
//	such as keeping track of streams and windows, then notify any inherited classes through the
//	virtual Handle_*() member functions.
//
//::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::


//------------------------------------------------------------------------------------
// CMacPluginView::RecalcFrameLayout
//------------------------------------------------------------------------------------
void	
CMacPluginView::RecalcFrameLayout( )
{
	if( mNetscapeWindow == 0 )
		return;
		
	NP_Port* npPort = (NP_Port*) mNetscapeWindow->window;
	mGrafPtr = (GrafPtr) GetNetscapeGrafPort();

	mPortOrigin.h = npPort->portx;
	mPortOrigin.v = npPort->porty;
	
	mFrameLocation.h = mNetscapeWindow->x;
	mFrameLocation.v = mNetscapeWindow->y;

	mFrameSize.width = mNetscapeWindow->width;
	mFrameSize.height = mNetscapeWindow->height;
	
	mContentPane->PlaceInSuperFrameAt( mFramePenThickness, mFramePenThickness, false );
	Int16 contentWidth, contentHeight;
	contentWidth = mNetscapeWindow->width - ( 2 * mFramePenThickness );
	contentHeight = mNetscapeWindow->height - ( 2 * mFramePenThickness );
	mContentPane->ResizeFrameTo( contentWidth, contentHeight, false );
	
	Rect clipRect;
	clipRect.top = mNetscapeWindow->clipRect.top;
	clipRect.left = mNetscapeWindow->clipRect.left;
	clipRect.bottom = mNetscapeWindow->clipRect.bottom;
	clipRect.right = mNetscapeWindow->clipRect.right;
	mRevealedRect = clipRect;

	mContentPane->AdaptToNewSurroundings();
	
}

//------------------------------------------------------------------------------------
// CMacPluginView::SetWindow
//------------------------------------------------------------------------------------
NPError	
CMacPluginView::SetWindow( NPWindow* window )
{
	mNetscapeWindow = window;
	

// Notification - Subclasses do whatever, and we redraw.	
	NPError result = Handle_SetWindow();	
	Paint();

	return result;
}




//------------------------------------------------------------------------------------
// CMacPluginView::Paint()
//------------------------------------------------------------------------------------
void
CMacPluginView::Paint()
{
	if( mNetscapeWindow == 0 )
		return;
	StColorPortState	savePortState(mGrafPtr);
	
	RecalcFrameLayout();
	
	ApplyForeAndBackColors();
	FocusDraw();
	Draw( nil );
	OutOfFocus(nil);
	
}

//------------------------------------------------------------------------------------
// CMacPluginView::UpdatePort()
//------------------------------------------------------------------------------------
void
CMacPluginView::UpdatePort()
{
//	FocusDraw();
	Draw( nil );
//	OutOfFocus(nil);
}

// ---------------------------------------------------------------------------
//		 DrawSelf
// ---------------------------------------------------------------------------

void
CMacPluginView::DrawSelf()
{
	LGrafPortView::DrawSelf();

	StColorPortState	savePortState(mGrafPtr);

	::PenSize( mFramePenThickness, mFramePenThickness );
		
	Rect	frame;
	CalcLocalFrameRect(frame);
	
	FrameRect(&frame);
}

// ---------------------------------------------------------------------------
//		 ActivateSelf
// ---------------------------------------------------------------------------
void
CMacPluginView::ActivateSelf()
{
	mForeColor.red = mForeColor.green = mForeColor.blue = 0;		// Black
	
	Paint();
}

// ---------------------------------------------------------------------------
//		 DeactivateSelf
// ---------------------------------------------------------------------------
void
CMacPluginView::DeactivateSelf()
{
	mForeColor.red = mForeColor.green = mForeColor.blue = 32768;	// Grey
	
	Paint();
}

//------------------------------------------------------------------------------------
// CMacPluginView::CreateContentPane
//------------------------------------------------------------------------------------
LPane*
CMacPluginView::CreateContentPane()
{
	// Navigator has it's own PPob Resources. We have to make sure that our
	// resource file is the current one, so that our PPob will be the first
	// one found.
	
	short saveResFile = CurResFile();
	UseResFile( gResFile );				// gResFile is set up in npmac.cpp, 
										// for exactly this purpose
	LView	*thePluginView = (LView*) UReanimator::ReadObjects('PPob', rPPob_PluginView);
	thePluginView->FinishCreate();	
	UseResFile(saveResFile);

	return thePluginView;
}

//------------------------------------------------------------------------------------
// CMacPluginView::Handle_Event
//------------------------------------------------------------------------------------

NPBool
CMacPluginView::Handle_Event( EventRecord& event )
{
	NPBool eventHandled = false;
	
	if( mNetscapeWindow == 0 )
		return eventHandled;
		
	StColorPortState	savePortState(mGrafPtr);
	
	RecalcFrameLayout();
	
	switch( event.what )
	{
		case nullEvent:
		{
			FocusDraw();
			LPeriodical::DevoteTimeToIdlers( event );
			OutOfFocus(nil);
			eventHandled = true;
			break;
		}
		case updateEvt:
		{
			ApplyForeAndBackColors();
			FocusDraw();
			Draw( nil );
			OutOfFocus(nil);	
			eventHandled = true;
			break;
		}
		case getFocusEvent:
		{
			Activate();
			eventHandled = true;
			break;
		}
		case loseFocusEvent:
		{
			Deactivate();
			eventHandled = true;
			break;
		}
		case mouseDown:
		{			
			ClickInContent( event );
			eventHandled = true;
			break;
		}
		case keyDown:
		{
		LCommander *theTarget = LCommander::GetTarget();
		if (theTarget != nil) 
			{
			theTarget->ProcessKeyPress( event );
			eventHandled = true;
			break;
			}
		}
		case activateEvt:
		{
			if( event.modifiers & activeFlag )
			{	// If we needed to do anything when the window became active, we would
				// do it here. We activate the view when it gets focus (getFocusEvent, above.)
			}
			else
			{	// If we needed to do anything when the window deactivated (such as stop
				// animation), we would do it here.
			}
			eventHandled = true;
			break;
		}
		default:
			eventHandled = false;
	}
	return eventHandled;
}
