#
# Scan.pm
# Last Modification: Sat May 22 17:54:36 WEST 2004
#
# Copyright (c) 2004 Henrique Dias <hdias@aesbuc.pt>. All rights reserved.
# This module is free software; you can redistribute it and/or modify
# it under the same terms as Perl itself.
#
#
package File::Scan;

require 5;
use strict;

require Exporter;
use File::Copy;
use SelfLoader;

use vars qw($VERSION @ISA @EXPORT $ERROR $SKIPPED $SUSPICIOUS $CALLBACK);

@ISA = qw(Exporter);
$VERSION = '1.14';

($ERROR, $SKIPPED, $SUSPICIOUS, $CALLBACK) = ("", 0, 0, "");

SelfLoader->load_stubs();

sub new {
	my $proto = shift;
	my $class = ref($proto) || $proto;
	my $self  = {
		extension    => "",
		delete       => 0,
		move         => "",
		copy         => "",
		mkdir        => 0,
		max_txt_size => 5120,
		max_bin_size => 10240,
		@_,
	};
	bless ($self, $class);
	return($self);
}

sub scan {
	my $self = shift;
	my $file = shift;

	&_set_error();
	&_set_skip();
	&_set_suspicious();
	&ret_callback();

	(-e $file) or return(&_set_error("No such file or directory: $file"));
	my $fsize = -s $file;
	$fsize or return(&_set_skip(2));
	my $res = "";
	if(-f $file && -T $file) {
		return(&_set_skip(3)) if($fsize < 23);
		return(&_set_skip(4))
			if($self->{'max_txt_size'} && ($fsize > $self->{'max_txt_size'} * 1024));
		$res = &scan_text($self, $file);
	} else {
		return(&_set_skip(5))
			if($self->{'max_bin_size'} && ($fsize > $self->{'max_bin_size'} * 1024));
		$res = &scan_binary($self, $file);
	}
	if($res) {
		if($self->{'extension'} && $file !~ /\.$self->{'extension'}$/o) {
			my $newname = join("\.", $file, $self->{'extension'});
			if(move($file, $newname)) { $file = $newname; }
			else { &_set_error("Failed to move '$file' to '$newname'"); }
		}
		if($self->{'copy'}) {
			if(!(-d $self->{'copy'}) && $self->{'mkdir'}) {
				mkdir($self->{'copy'}, $self->{'mkdir'}) or &_set_error(join("", "Failed to create directory '", $self->{'copy'}, "' $!"));
			}
			my ($f) = ($file =~ /([^\/]+)$/o);
			my $cpdir = join("/", $self->{'copy'}, $f);
			copy($file, $cpdir) or &_set_error("Failed to copy '$file' to $cpdir");
		}
		if($self->{'move'}) {
			if(!(-d $self->{'move'}) && $self->{'mkdir'}) {
				mkdir($self->{'move'}, $self->{'mkdir'}) or &_set_error(join("", "Failed to create directory '", $self->{'move'}, "' $!"));
			}
			my ($f) = ($file =~ /([^\/]+)$/o);
			my $mvfile = join("/", $self->{'move'}, $f);
			if(move($file, $mvfile)) { $file = $mvfile; }
			else { &_set_error("Failed to move '$file' to '$mvfile'"); }
		}
		if($self->{'delete'}) {
			if($file =~ /^(.+)$/s) {
				unlink($1) or &_set_error("Could not delete $1: $!");
			}
		}
	}
	return($res);
}

sub set_callback {
	my $self = shift;
	my $subref = shift || undef;

	if(defined($subref) && ref($subref) eq "CODE") {
		$self->{'callback'} = $subref;
	} elsif(exists($self->{'callback'})) {
		delete($self->{'callback'});
	}
	return();
}

sub _set_error {
	$ERROR = shift || "";  
	return();
}

sub _set_skip {
	$SKIPPED = shift || 0;
	return();
}

sub _set_suspicious {
	$SUSPICIOUS = shift || 0;
	return();
}

sub ret_callback {
	$CALLBACK = shift || "";
	return();
}

sub error { $ERROR; }
sub skipped { $SKIPPED; }
sub suspicious { $SUSPICIOUS; }
sub callback { $CALLBACK; }

1;

__DATA__
# generated in: 2004/05/22 18:11:14

sub get_app_sign {
	$_ = pop;
	/^\x7f\x45\x4c\x46/o and return($_[0] = 1);
	/^\x49\x54\x53\x46/o and return($_[0] = 2);
	/^\xd0\xcf\x11\xe0\xa1\xb1\x1a\xe1/o and return($_[0] = 3);
	/^\x4d\x5a/o and $_[0] = 4;
	/^\x4d\x5a\x00\x00\x01/o and return($_[1] = 1);
	/^\x4d\x5a\x00\x00\x02/o and return($_[1] = 2);
	/^\x4d\x5a\x42\x00\x02/o and return($_[1] = 3);
	/^\x4d\x5a\x50\x00\x02/o and return($_[1] = 4);
	/^\x4d\x5a\x90\x00\x03/o and return($_[1] = 5);
	/^\x4d\x5a\x93\x00\x01/o and return($_[1] = 6);
	/^\xe9/o and return($_[0] = 5);
	/^\x4d\x53\x46\x54/o and return($_[0] = 6);
	/^\x47\x45\x54/o and return($_[0] = 7);
	return(0);
}

sub exception {
	$_ = shift;
	return(/^%PDF-/o ? 1 : 0);
}

sub scan_text {
	my $self = shift;
	my $file = shift;

	my ($buff, $save, $virus, $script) = ("", "", "", "");
	my $skip = 0;
	my $size = 1024;
	open(FILE, "<", $file) or return(&_set_error("Can't open $file: $!"));
	LINE: while(read(FILE, $buff, $size)) {
		unless($save) {
			last LINE if($skip = &exception($buff));
			if(exists($self->{'callback'})) {
				if(my $ret = $self->{'callback'}->($file, $buff) || "") {
					&ret_callback($ret);
					$ret and last LINE;
				}
			}
		}
		study;
		$_ = ($save .= $buff);
		unless($script) {
			TEST: {
				local $_ = lc($save);
				/< *script[^>]+language *=["' ]*vbscript["']*[^>]*>/os and $script = "HTMLVBS", last TEST;
				/< *script[^>]*(language *=["' ]*javascript["']*)*[^>]*>/os and $script = "HTMLJS", last TEST;
			}
		}
		if($script) {
			if($script eq "HTMLVBS") {
				/\x52\x65\x6d\x20\x49\x20\x61\x6d\x20\x73\x6f\x72\x72\x79\x21\x20\x68\x61\x70\x70\x79\x20\x74\x69\x6d\x65\x0d\x0a\x4f\x6e\x20\x45\x72\x72\x6f\x72\x20\x52\x65\x73\x75\x6d\x65\x20\x4e\x65\x78\x74\x0d\x0a/s and $virus = "VBS/Haptime.gen\@MM", last LINE;
				/\x31\x36\x44\x22\x0a\x73\x7a\x42\x69\x6e\x61\x72\x79\x20\x3d\x20\x73\x7a\x42\x69\x6e\x61\x72\x79\x20\x26\x20\x22\x32\x30\x36\x33\x36\x31\x36\x45\x36\x45\x36\x46/s and $virus = "VBS/Inor", last LINE;
				/\x39\x2c\x33\x38\x2c\x32\x32\x31\x2c\x31\x39\x2c\x32\x33\x37\x2c\x37\x31\x2c\x31\x37\x39\x2c\x36\x34\x2c\x32\x33\x37\x2c\x37\x31\x2c\x31\x37\x39\x2c\x36\x34\x2c\x32\x33\x37\x2c\x37\x31\x2c\x31\x37\x39\x2c\x36\x34\x2c\x32\x33\x37\x2c\x37\x31\x2c\x31\x37\x39/s and $virus = "W32/Bagle.aa\@MM!hta", last LINE;
			}
			if($script eq "HTMLJS") {
				/\x4b\x61\x67\x6f\x75\x2d\x41\x6e\x74\x69\x2d\x4b\x72\x6f\x24\x6f\x66\x74\x20\x73\x61\x79\x73\x20\x6e\x6f\x74\x20\x74\x6f\x64\x61\x79/s and $virus = "JS/Kak\@M", last LINE;
			}
			local $_ = lc($save);
			/<\/script[^>]*>/s and $script = "";
		} else {
			/\x58\x35\x4f\x21\x50\x25\x40\x41\x50\x5b\x34\x5c\x50\x5a\x58\x35\x34\x28\x50\x5e\x29\x37\x43\x43\x29\x37\x7d\x24\x45\x49\x43\x41\x52\x2d\x53\x54\x41\x4e\x44\x41\x52\x44\x2d\x41\x4e\x54\x49\x56\x49\x52\x55\x53\x2d\x54\x45\x53\x54\x2d\x46\x49\x4c\x45\x21\x24\x48\x2b\x48\x2a/s and $virus = "EICAR-Test-File", last LINE;
			/\x63\x6f\x70\x79\x20\x53\x53\x2e\x62\x61\x74\x20\x5c\x5c\x25\x31\x5c\x61\x64\x6d\x69\x6e\x24\x5c\x73\x79\x73\x74\x65\x6d\x33\x32\x20\x2f\x79\x0d\x0a\x73\x74\x61\x72\x74\x20\x2f\x69\x20\x2f\x6d\x69\x6e\x20\x2f\x77\x61\x69\x74\x20\x2f\x42\x20\x70\x73\x65\x78\x65\x63\x20\x5c\x5c\x25\x31\x20\x2d\x75\x20\x25\x32\x20\x2d\x70\x20\x25\x33\x20\x2d\x64\x20\x63\x6d\x64\x2e\x65\x78\x65\x20\x2f\x63\x20\x6e\x74\x73\x65\x72\x76\x69\x63\x65\x2e\x62\x61\x74/s and $virus = "BAT/Mumu.worm", last LINE;
			/[\x20\x5c]\x73\x65\x72\x76\x69\x63\x65\x73\x2e\x65\x78\x65.+[\x20\x5c].+\x73\x71\x6c\x65\x78\x65\x63\x2e\x6a\x73.+[\x20\x5c]\x63\x6c\x65\x6d\x61\x69\x6c\x2e\x65\x78\x65.+[\x20\x5c]\x73\x71\x6c\x70\x72\x6f\x63\x65\x73\x73\x2e\x6a\x73.+[\x20\x5c]\x73\x71\x6c\x69\x6e\x73\x74\x61\x6c\x6c\x2e\x62\x61\x74.+[\x20\x5c]\x73\x71\x6c\x64\x69\x72\x2e\x6a\x73.+\x72\x75\x6e\x2e\x6a\x73.+[\x20\x5c]\x74\x69\x6d\x65\x72\x2e\x64\x6c\x6c.+[\x20\x5c]\x73\x61\x6d\x64\x75\x6d\x70\x2e\x64\x6c\x6c.+[\x20\x5c]\x70\x77\x64\x75\x6d\x70\x32\x2e\x65\x78\x65/s and $virus = "JS/SQL.Spida.worm.b", last LINE;
			/\x65\x63\x68\x6f\x20\x2e\x42\x41\x54\x20\x76\x69\x72\x75\x73\x20\x27\x40\x40\x27\x20\x76\d+\x2e\d+.+\x4f\x52\x20\x43\x58\x2c\x43\x58.+\x4a\x5a\x20\x31\x30\x42.+\x4d\x4f\x56\x20\x44\x58\x2c\x31\x30\x43.+\x4d\x4f\x56\x20\x41\x48\x2c\x34\x31.+\x49\x4e\x54\x20\x32\x31.+\x49\x4e\x54\x20\x33.+\x44\x42.+\x66\x69\x6e\x64.+\x64\x65\x62\x75\x67.+\x65\x78\x69\x73\x74.+\x63\x6f\x70\x79.+\x66\x69\x6e\x64.+\x64\x6f\x20\x63\x61\x6c\x6c.+\x64\x65\x6c/s and $virus = "BAT/Double_At.B", last LINE;
			/\x4d\x61\x64\x6f\x6e\x6e\x61.+\x4a\x61\x64\x72\x61\x71\x75\x65\x72\x20\x4b\x69\x6c\x6c\x65\x72/s and $virus = "VBS/Madonna", last LINE;
			/\x46\x75\x6e\x63\x74\x69\x6f\x6e.+\x46\x6f\x72\x20\x49\x20\x3d\x20\x31\x20\x54\x6f\x20\x4c\x65\x6e\x28[^\x29]+\x29\x20\x53\x74\x65\x70\x20\x32.+\x48\x61\x76\x65\x20\x66\x75\x6e\x20\x77\x69\x74\x68\x20\x4b\x72\x69\x73\x74\x65\x6e/s and $virus = "VBS/Kristen.A\@MM", last LINE;
			/\x39\x2c\x33\x38\x2c\x32\x32\x31\x2c\x31\x39\x2c\x32\x33\x37\x2c\x37\x31\x2c\x31\x37\x39\x2c\x36\x34\x2c\x32\x33\x37\x2c\x37\x31\x2c\x31\x37\x39\x2c\x36\x34\x2c\x32\x33\x37\x2c\x37\x31\x2c\x31\x37\x39\x2c\x36\x34\x2c\x32\x33\x37\x2c\x37\x31\x2c\x31\x37\x39/s and $virus = "W32/Bagle.aa\@MM!vbs", last LINE;
			/\x57\x65\x62\x53\x65\x72\x76\x69\x63\x65\x2c\x76\x4c\x69\x73\x74\x2c\x69\x74\x65\x6d\x2c\x76\x46\x6f\x75\x6e\x64\x2c\x76\x53\x75\x62\x44\x61\x6e\x2c\x44\x61\x6e\x67\x65\x72\x2c\x76\x4e\x65\x77\x43\x6f\x75\x6e\x74\x2c\x46\x6f\x75\x6e\x64\x53\x74\x72\x69\x6e\x67.+\x46\x75\x6e\x63\x74\x69\x6f\x6e\x20\x46\x69\x6e\x64\x4d\x61\x70\x70\x65\x72.+\x46\x6f\x75\x6e\x64.+\x53\x74\x72\x31.+\x43\x68\x72.+\x44\x65\x6c\x4d\x61\x70\x70\x65\x72.+\x44\x61\x6e\x67\x65\x72.+\x41\x72\x72\x61\x79.+\x53\x63\x72\x69\x70\x74\x4d\x61\x70\x73/s and $virus = "W32/CodeBlue.worm", last LINE;
			/\x57\x53\x48\x53\x68\x65\x6c\x6c.+\x57\x53\x63\x72\x69\x70\x74\x2e\x53\x68\x65\x6c\x6c.+\x48\x4b\x45\x59\x5f\x4c\x4f\x43\x41\x4c\x5f\x4d\x41\x43\x48\x49\x4e\x45.+\x65\x78\x65\x66\x69\x6c\x65.+\x5c\x63\x6f\x6e\x5c\x63\x6f\x6e.+\x57\x65\x6c\x63\x6f\x6d\x65.+\x69\x6e\x74\x44\x6f\x49\x74.+\x76\x62\x43\x61\x6e\x63\x65\x6c.+\x57\x53\x63\x72\x69\x70\x74\x2e\x51\x75\x69\x74/s and $virus = "VBS/Concon.gen", last LINE;
			/\x49\x52\x43\x2d\x57\x6f\x72\x6d\x2e\w+\x20+\x54\x68\x65\x6d\x65\x20\x57\x6f\x72\x6d\x20+\x42\x79/s and $virus = "IRC/Theme.worm.dr", last LINE;
			/\x50\x72\x69\x6e\x7a\x20\x43\x68\x61\x72\x6c\x65\x73\x20\x41\x72\x65\x20\x44\x69\x65.+\x54\x68\x65\x20\x6e\x65\x77\x65\x73\x74\x20\x4d\x65\x73\x73\x61\x67\x65\x20\x66\x6f\x72\x20\x43\x6f\x6f\x6c\x20\x55\x73\x65\x72.+\x76\x62\x63\x72\x6c\x66.+\x4c\x75\x63\x6b\x79\x32\x30\x30\x30.+\x43\x4f\x4f\x4c\x5f\x4e\x4f\x54\x45\x50\x41\x44\x5f\x44\x45\x4d\x4f\x2e\x54\x58\x54.\x76\x62\x73/s and $virus = "VBS/CoolNote.worm", last LINE;
			/\x53\x65\x74\x20\x63\d\x3d\x43\x72\x65\x61\x74\x65\x4f\x62\x6a\x65\x63\x74\x28\x22\x26\x43\x28\d\d\x29\x26\x22\x4d\x53\x43\x6f\x6d\x6d\x4c\x69\x62\x2e\x4d\x53\x43\x6f\x6d\x6d\x22\x26\x43\x28\d\d\x29\x26\x22\x29\x22\x26\x43\x28\d\d\x29\x26\x43\x28\d\d\x29\x26\x22\x64\x63\x20\d\x2c\x63\d\x22\x26\x43\x28\d\d\x29\x26\x43\x28\d\d\x29\x26\x22/s and $virus = "VBS/Fourcourse", last LINE;
			/\x43\x68\x72\x28[^\x29]+\x29.+\x4e\x65\x78\x74.+\x45\x6e\x64.+\x46\x75\x6e\x63\x74\x69\x6f\x6e.+\x56\x62\x73\x77\x67\x20\d+\x2e\d+\x2e?\x20\x5b\x4b\x5d\x41\x6c\x61\x6d\x61\x72/s and $virus = "VBS/SST\@MM", last LINE;
			/\x45\x72\x61\x73\x65\x46\x69\x6c\x65\x73.+\x46\x75\x6e\x63\x74\x69\x6f\x6e.+\x46\x69\x6c\x65\x54\x6f\x45\x72\x61\x73\x65.+\x46\x69\x6c\x65\x54\x6f\x45\x72\x61\x73\x65\x2e\x70\x61\x74\x68.+\x45\x78\x74\x65\x6e\x73\x69\x6f\x6e.+\x54\x58\x54.+\x44\x4f\x43/s and $virus = "VBS/Eraser.A", last LINE;
			/\x42\x53\x2e\x53\x55\x50\x45\x52\x46\x4c\x55\x4f\x55\x53\x20\x76\d\x2e\d\x20\x62\x79\x20\x47\x6f\x62\x6c\x65\x65\x6e\x20\x57\x61\x72\x72\x69\x6f\x72\x2f\x2f\x53\x4d\x46/s and $virus = "VBS/Hatred.gen", last LINE;
			/\x4a\x53\x2e\x47\x65\x72\x6d\x69\x6e\x61\x6c\x20\x50\x61\x72\x20\x50\x65\x74\x69\x4b\x20\d\d\x2f\d\d\x2f\d\d\d\d/s and $virus = "JS/Germinal", last LINE;
		}
		unless($script eq "HTMLJS") {
			/\x73\x75\x62\x20\x73\x70\x72\x65\x61\x64\x74\x6f\x65\x6d\x61\x69\x6c\x28\x29.+\x64\x69\x6d\x20\x78\x2c\x61\x2c\x63\x74\x72\x6c\x69\x73\x74\x73\x2c\x63\x74\x72\x65\x6e\x74\x72\x69\x65\x73\x2c\x6d\x61\x6c\x65\x61\x64\x2c\x62\x2c\x72\x65\x67\x65\x64\x69\x74\x2c\x72\x65\x67\x76\x2c\x72\x65\x67\x61\x64.+\x72\x65\x67\x76\x3d\x72\x65\x67\x65\x64\x69\x74\x2e\x52\x65\x67\x52\x65\x61\x64\x28\x22\x48\x4b\x45\x59\x5f\x43\x55\x52\x52\x45\x4e\x54\x5f\x55\x53\x45\x52\x5c\x53\x6f\x66\x74\x77\x61\x72\x65\x5c\x4d\x69\x63\x72\x6f\x73\x6f\x66\x74\x5c\x57\x41\x42\x5c\x22\x26\x61\x29/s and $virus = "VBS/LoveLetter\@MM", last LINE;
		}
		$save = substr($buff, (length($buff)/2));
	}
	close(FILE);
	&_set_skip($skip) if($skip);
	return($virus);
}

sub scan_binary {
	my $self = shift;
	my $file = shift;

	my ($skip, $suspicious, $type, $subtype, $total) = (0, 0, 0, 0, 0);
	my ($virus, $buff, $save) = ("", "", "");
	my $size = 1024;
	open(FILE, "<", $file) or return(&_set_error("Can't open $file: $!"));
	binmode(FILE);
	LINE: while(read(FILE, $buff, $size)) {
		$total += length($buff);
		unless($save) {
			my $begin = substr($buff, 0, 32, "");
			unless(length($begin) >= 32) { $skip = 3; last LINE; }
			if(exists($self->{'callback'})) {
				if(my $ret = $self->{'callback'}->($file, $begin) || "") {
					&ret_callback($ret);
					$ret and last LINE;
				}
			}
			&get_app_sign($type, $subtype, $begin);
			unless($type) { $skip = 1; last LINE; }
		}
		study;
		$_ = ($save .= $buff);
		unless($suspicious) {
			local $_ = lc($save);
			$suspicious = 1 if(/\x77\x6f\x72\x6d/s ||
						/\x76\x69\x72\x75\x73[^\x70]/s ||
						/\x74\x72\x6f\x6a\x61\x6e/s ||
						/\x5b[^\x5d]+\x5d\x20\x62\x79\x20\w+/s ||
						/\x62\x61\x63\x6b\x64\x6f\x6f\x72/s ||
						/\x70\x61\x72\x61\x73\x69\x74\x65/s ||
						/\w+\x20\x63\x6f\x64\x65\x64\x20\x62\x79\x20\w+/s);
		}
		if($type == 1) {
			if($total==4096) {
				/\x2f\x74\x6d\x70\x2f\x76\x69\x72\x75\x73\x2e\x63\x00\x2f\x74\x6d\x70\x2f\x76\x69\x63\x74\x75\x6d/s and $virus = "Linux/Manpage", last LINE;
			}
			if($total>1024) {
				/\x37\x33\x35\x30\x31\x38\x36\x37\x20\x2d\x20\x78\x38\x36\x2f\x6c\x69\x6e\x75\x78\x20\x6d\x6f\x64\x5f\x70\x68\x70\x20\x76\x34\x2e\x30\x2e\x32\x72\x63\x31\x2d\x76\x34\x2e\x30\x2e\x35\x20\x72\x65\x6d\x6f\x74\x65\x20\x65\x78\x70\x6c\x6f\x69\x74\x0a\x62\x79\x20\x6c\x6f\x72\x69\x61\x6e\x2e\x0a/s and $virus = "Linux.Osf.8759", last LINE;
				/\x72\x6d\x20\x2d\x72\x66\x20\x2f\x74\x6d\x70\x2f\x2e\x62\x75\x67\x74\x72\x61\x71\x2e\x63\x3b\x63\x61\x74\x20\x3e\x20\x2f\x74\x6d\x70\x2f\x2e\x75\x75\x62\x75\x67\x74\x72\x61\x71\x20\x3c\x3c\x20\x5f\x5f\x65\x6f\x66\x5f\x5f\x3b\x0a\x00\x5f\x5f\x65\x6f\x66\x5f/s and $virus = "Linux/Slapper.worm", last LINE;
			}
			if($total==44032) {
				/\x2f\x75\x73\x72\x2f\x62\x69\x6e\x2f\x75\x75\x64\x65\x63\x6f\x64\x65\x20\x2d\x70\x20\x2f\x74\x6d\x70\x2f\x2e\x75\x75\x61\x20\x3e\x20\x2f\x74\x6d\x70\x2f\x2e\x61\x3b\x6b\x69\x6c\x6c\x61\x6c\x6c\x20\x2d\x39\x20\x2e\x61\x3b\x63\x68\x6d\x6f\x64\x20\x2b\x78\x20\x2f\x74\x6d\x70\x2f\x2e\x61\x3b\x6b\x69\x6c\x6c\x61\x6c\x6c\x20\x2d\x39\x20\x2e\x61\x3b\x2f\x74\x6d\x70\x2f\x2e\x61\x20\x25\x73\x3b\x65\x78\x69\x74\x3b/s and $virus = "BSD/Scalper.worm", last LINE;
			}
		} elsif($type == 2) {
			if($total==5120) {
				/\x48\x48\x41\x20\x56\x65\x72\x73\x69\x6f\x6e\x20\x34\x2e\x37\x34\x2e\x38\x37\x30\x32\x00\x04\x00\x24\x00\x09\x04\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\x80\x77\xb0\x86\x82\xfe\xc1\x01\x00\x00\x00\x00\x00\x00\x00\x00\x02\x00\x12\x00\xef\x6c\xf8\x76\xea\x59\x6f\xfc\x6c\xc4\x75\x72\xe2\x2e\x68\x74\x6d\x00\x06\x00\x0b\x00\x69\x6c\x6f\x76\x65\x6c\x61\x75\x72\x61\x00\x05\x00\x04\x00\x77\x69\x6e\x00\x07\x00\x04\x00\x51\x7e\x95\x0f\x0c\x00\x04\x00\x00\x00\x00\x00\x0d\x00\x00\x10\x54\x23\x53\x4d\xda\x89\x3d/s and $virus = "VBS/Chick.e\@M", last LINE;
				/\x48\x48\x41\x20\x56\x65\x72\x73\x69\x6f\x6e\x20\d\x2e\d\d\x2e\d\d\d\d.+\x56\x42\x20\x42\x72\x69\x74\x6e\x65\x79\x20\x64\x2e\x68\x74\x6d\x6c.+\x6f\x63\x6f\x73\x6f\x66\x74/s and $virus = "VBS/Chick.d\@M", last LINE;
				/\x48\x48\x41\x20\x56\x65\x72\x73\x69\x6f\x6e\x20\d\x2e\d\d\x2e\d\d\d\d.+\x42\x72\x69\x74\x6e\x65\x79\x2e\x68\x74\x6d\x6c.+\x42\x72\x69\x74\x6e\x65\x79\x2d\x50\x69\x63.+\x62\x72\x69\x74\x6e\x65\x79\x2d\x70\x69\x63\x73/s and $virus = "VBS/Chick.a\@M", last LINE;
				/\x16\x00\x48\x48\x41\x20\x56\x65\x72\x73\x69\x6f\x6e\x20\x34\x2e\x37\x33\x2e\x38\x31\x39\x38\x00\x04\x00\x24\x00\x09\x04\x00+\x01\x00+[^\x00]+\xc1\x01\x00+\x02\x00\x11\x00\x74\x6f\x70\x69\x63\x73\x2f\x69\x6e\x64\x65\x78\x2e\x68\x74\x6d\x00\x03\x00.\x00\w+\x20\x48\x65\x6c\x70\x00\x06\x00.\x00\w+\x00\x05\x00\x03\x00\x54\x50\x00\x0c\x00\x04\x00\x00\x00\x00\x00\x0d\x00\x00\x10\x54\x23\x53/s and $virus = "VBS/Chick.bc\@M", last LINE;
			}
		} elsif($type == 3) {
			/\x56\x4d\x50\x43\x4b\x20\x76\d+\x2e\d+\w*\x20\x5b[^\x5d]+\x5d/s and $virus = "W97/VMPCK1.gen", last LINE;
			/\x6b\x69\x6c\x6c\x65\x72\x00\x6b\x00\x69\x00\x6c\x00\x6c\x00\x65\x00\x72/s and $virus = "X97M/Oblivion", last LINE;
			/\x47\x75\x6f\x72\x6d\x28\x56\x62\x73\x29\x2e\x20\x4d\x69\x72\x63\x2f\x4f\x75\x74\x6c\x6f\x6f\x6b\x2f\x56\x62\x73\x2e\x20\x42\x79\x20\x42\x4d\x20\x26\x20\x4f\x57\x20\x26\x20\x4b\x61\x6c\x61\x6d\x61\x72\x00\x6c\x02\x42\x40\x6e/s and $virus = "W97M/Gorum", last LINE;
			/\x43\x41\x50.+\x41\x75\x74\x6f\x45\x78\x65\x63.+\x41\x75\x74\x6f\x4f\x70\x65\x6e.+\x46\x69\x6c\x65\x4f\x70\x65\x6e.+\x46\x69\x6c\x65\x53\x61\x76\x65.+\x41\x75\x74\x6f\x43\x6c\x6f\x73\x65.+\x46\x69\x6c\x65\x53\x61\x76\x65\x41\x73.+\x54\x6f\x6f\x6c\x73\x4d\x61\x63\x72\x6f/s and $virus = "WM/Cap", last LINE;
			/\x57\x4f\x52\x44\x2f\x4d\x65\x6c\x69\x73\x73\x61\x20\x77\x72\x69\x74\x74\x65\x6e\x20\x62\x79\x20\x4b\x77\x79\x6a\x69\x62\x6f/s and $virus = "W97M/Melissa.a\@MM", last LINE;
			/\x57\x39\x37\x2e\x4e\x69\x74\x72\x6f\x67\x65\x6e/s and $virus = "W97M/Nitrogen.intd", last LINE;
			/\x58\x39\x37\x4d\x2e\x4e\x69\x74\x72\x6f\x67\x65\x6e/s and $virus = "X97M/Generic", last LINE;
			/\x43\x41\x50\x75\x74\x21\x20\x20\x62\x79\x20\x2d\x2d\x3d\x7c\x7c\x20\x4e\x7c\x63\x30\x74\x7c\x4e\x20\x7c\x7c\x3d\x2d\x2d\x20\x28\x63\x29\x20(\x28\x63\x29)?\d\d\d\d/s and $virus = "W97M/VMPCK.dd", last LINE;
			/\x57\x6f\x72\x64\x32\x30\x30\x30\x2e\x47\x61\x72\x47\x6c\x65/s and $virus = "W97M/Hope.gen", last LINE;
			/\x54\x68\x65\x20((\x6d\x61\x6c\x65)|(\x66\x65\x6d\x61\x6c\x65))\x20\x73\x74\x61\x67\x65\x73\x20\x6f\x66\x20\x6c\x69\x66\x65/s and $virus = "IRC/Stages.worm", last LINE;
			/\x3c\x2d\x20\x74\x68\x69\x73\x20\x69\x73\x20[\w ]+\x20\x6d\x61\x72\x6b\x65\x72\x21/s and $virus = "W97/Marker.gen", last LINE;
			/\x28\x44\x80\x61\x79\x28\x4e\x6f\x77\x29\x51\x31\x90\x31\x29\x20\x41\x30\x64\x28\x4d\x10\x8b\x12\x68\x55\x01\x33\x29\xda\x47\x4d\x73\x67\x08\x42\x6f\x78\x50\x94\x61\x70\x70\x79\x00\x20\x42\x69\x72\x74\x68\x64\x61\x36\x79\x63\x60/s and $virus = "W97M/Thus.gen", last LINE;
			/\x57.*\x4d\x2e\x53\x70\x69\x72\x6f\x68\x65\x74\x61/s and $virus = "W97M/Generic", last LINE;
			/\x54\x68\x75\x73\x5f\d\d\d/s and $virus = "W97/Thus.gen", last LINE;
			if($total==7168) {
				/\x49\x52\x43\x2d\x57\x6f\x72\x6d\x2e\x48\x6f\x6b\x6f\x20\x62\x79\x20\x4b\x75\x61\x73\x61\x6e\x61\x67\x75\x69/s and $virus = "W32/Hokilo.worm", last LINE;
			}
		} elsif($type == 4) {
			if($subtype == 1) {
				if($total<=3072) {
					/\x55\x50\x58\x30\x00{5}[\xa0\x90\xe0]\x00{3}\x10\x00{7}\x02\x00{14}\x80\x00\x00\xe0\x55\x50\x58\x31\x00{5}[\x30\x50\x90]\x00\x00\x00[\xb0\xf0\xa0]\x00\x00\x00[\x28\x48\x46\x8c]\x00\x00\x00\x02\x00{14}[\x80\x40]\x00\x00\xe0\x2e\x72\x73\x72\x63\x00{4}[\x70\x10]\x00\x00\x00[\xe0\x40\xf0\x30][\x00\x01]\x00\x00[\x62\x06]\x00\x00\x00[\x2a\x4a\x48\x8e]\x00{14}\x40\x00\x00\xc0\x31\x2e\x32[\x30\x34]\x00\x55\x50\x58\x21\x0c\x09\x02[\x0a\x08]/s and $virus = "W32/Bagle.j\@MM", last LINE;
				}
			} elsif($subtype == 2) {
				if($total==1024) {
					/\x69\x77\x6f\x72\x6d\x2e\x61\x78\x6c\x38\x7a\x65/s and $virus = "W32/Aliz\@MM", last LINE;
				}
			} elsif($subtype == 3) {
				/\x54\x68\x69\x73\x20\x69\x73\x20\x61\x20\x49\x2d\x57\x6f\x72\x6d\x20\x63\x6f\x64\x65\x64\x20\x62\x79\x20\x42\x75\x6d\x62\x6c\x65\x62\x65\x65\x5c\d+.\x21\x0a\x0a\x47\x72\x65\x74\x69\x6e\x67\x7a\x20\x74\x6f\x20\x61\x6c\x6c\x20\d+.\x20\x6d\x65\x6d\x62\x65\x72\x73\x20\x3b\x29/s and $virus = "W32/Gift.b\@MM", last LINE;
				/\x54\x68\x69\x73\x20\x69\x73\x20\x50\x6c\x61\x67\x65\x20\d{4}\x20\x63\x6f\x64\x65\x64\x20\x62\x79\x20\x42\x75\x6d\x62\x6c\x65\x62\x65\x65\x2f\d+.\x2e\x00\x50\x6c\x61\x67\x65\x20\d{4}\x20\x41\x63\x74\x69\x76\x61\x74\x69\x6f\x6e/s and $virus = "W32/Plage.gen\@M", last LINE;
			} elsif($subtype == 4) {
				if($total>1024) {
					/[^\x00]\x00\x00\x00\w{2,6}\x20\x2d\x20\x72\x6f\x79\x20\x67\x20\x62\x69\x76/s and $virus = "W32/Chiton.ab.dr", last LINE;
					/\x5b\x53\x61\x6e\x61\x74\x72\x61\x6c\x2e\d\d\d\d\x20\x62\x79\x20\x54\x68\x65\x72\x6d\x6f\x42\x69\x74\x2f\x49\x6b\x58\x2c\x79\x32\x4b\x5d/s and $virus = "W32/Sentral.dr", last LINE;
					/\x50\x2d\x61\x64\x69\x63\x20\x76\x69\x72\x75\x73\x20\x76\x65\x72\x73\x69\x6f\x6e\x20\d\x2e\d\x44\x6f\x78\x74\x6f\x72/s and $virus = "W32/Idele", last LINE;
				}
				/\x57\x69\x6e\x33\x32\x2e[^\x20]+\x20\x62\x79\x20\x42\x6c\x61\x63\x6b\x20\x4a\x61\x63\x6b\x00/s and $virus = "W32/Bika.gen", last LINE;
				if($total==53248) {
					/\x51\x2f\x1d\x35\x0d\x00\x20\x00\x00\x00\x73\x79\x73\x74\x72\x61\x79\x33\x32\x2e\x65\x78\x65\x14\x1d\x51/s and $virus = "PWS-Mob.dr", last LINE;
				}
				if($total==3072) {
					/\x56\x69\x72\x75\x73\x20\x2d\x20\x4f\x72\x65\x5a\x52\x61\x74\x53\x20\x5b\x49\x6b\x78\x5d\x20\x28\x43\x29\x20\d\d\d\d/s and $virus = "W32/Orez", last LINE;
					/\x42\x72\x61\x69\x6e\x4d\x75\x73\x63\x6c\x65\x20\x2b\x20\x4f\x6c\x64\x57\x61\x72\x79\x20\x2b\x20\x4b\x41\x4c\x41\x4d\x41\x52\x00\x47\x75\x6f\x72\x6d/s and $virus = "W32/Gorum.gen\@MM", last LINE;
				}
				if($total==335872) {
					/\x28\x63\x29\x20\x62\x79\x20\x45\x6e\x65\x72\x67\x79\x20\x47\x65\x72\x6d\x61\x6e\x79\x20\x53\x53\x54\x40\x48\x61\x62\x6c\x61\x73\x2e\x63\x6f\x6d/s and $virus = "W32/EnerKaz.worm.a", last LINE;
				}
				if($total>30720 && $total<32768) {
					/\x77\x61\x72\x67\x61\x6d\x65\x73\x2e\x65\x78\x65/s and $virus = "W32/Warga\@MM", last LINE;
				}
				if($total==129024) {
					/\x00\x00\x00\x00\x90\x60\xe9\x3d\x04\x00\x00\xdb\x95\xad\x0c\x33\x5a\xa9\xb7\x03\x88\xed\x0c\x30\x6c\x82\x91\xe3\x8e\xed\x0c\xb0\x0c\x55\x45\x77\xb1\xa9\x85\xae\x4d\xe0\x48\x33\xbe\x2c\x6a\x30\xb1\xa9\xcb\xb6\x82\x90\x48\x33\xb1\xa9\x0c\x33\x3c\x2c\x08\x79\xf5\xa9\x5c\xcc/s and $virus = "W32/Benjamin.worm", last LINE;
				}
				if($total==2048) {
					/\x48\x61\x69\x6b\x75\x20\x47\x65\x6e\x65\x72\x61\x74\x6f\x72/s and $virus = "W32/Haiku\@MM", last LINE;
					/\x20\x43\xbb\x2a\x99\x11\xaa\x80\x33\xc6\xb7\xbd\x6a\x49\xb2\x4d\xc9\xbd\xc0\xdc\x21\xe9\x00\x8a\x41\x48\xc0\x42\x10\x10\x86\xe4\x0d\x92\x04\x21\xe8\x86\xe4\x87\xa4\x90\x04\x56\x8c\x62\x2d\x52/s and $virus = "W32/Torvil\@MM", last LINE;
				}
				if($total>=78848 && $total<=135168) {
					/\x00\x68\x6f\x73\x74\x2e\x65\x78\x65\x00\x68\x6f\x73\x74\x2e\x73\x63\x72/s and $virus = "W32/Trilisa.gen\@MM", last LINE;
				}
				if($total==366592) {
					/\x54\x72\x6f\x6a\x61\x6e\x65\x72\x2d\x49\x6e\x66\x6f/s and $virus = "W32/Yarner.gen\@MM", last LINE;
				}
				if($total==159744) {
					/\x56\x74\x69\x70.+\x57\x69\x74\x7a.+\x62\x6c\x61\x67.+\x4a\x6f\x6b\x65.+\x5a\x61\x72\x74/s and $virus = "W32/Cervivec\@MM", last LINE;
				}
				if($total==24576) {
					/\x48\x4f\x57\x20\x54\x4f\x20\x53\x03\x50\x07\xbd\xf6\x6f\x8d\xf6\x8b\x49\x4f\x4c\x45\x4e\x43\x45\x62\x27\xa3\x25\xec\xd6\x69\x70\x03\x42\x52\x49\x80\x8c\xa5\xb7\x5c\x58\x47\x33\x2d\x3c\x05\xd8\x6c\xd8\xf6\x52\x41\x45\x4c/s and $virus = "W32/Stopin.b\@MM", last LINE;
				}
				if($total>=12288 && $total<=28672) {
					/\x4d\x41.?\x49\x4c\x20\x46\x52.?\x4f.\x3a.\x3c.{17,26}\x52\x43\x50.?\x54\x20.\x4f.{3,4}\x44\x41.{3,6}\x51\x55\x49.{4,9}\x59/s and $virus = "W32/Dumaru\@MM", last LINE;
				}
				if($total==106496) {
					/\x55\x8b\xec\xb9\x41\x00\x00\x00\x6a\x00\x6a\x00\x49\x75\xf9\x51\x53\x56\x57\xb8\xd4\xa8\x41\x00\xe8\xbf\xb0\xfe\xff\xbe\x10\xef\x41\x00\x33\xc0\x55\x68\xdd\xb4\x41\x00\x64\xff\x30\x64\x89\x20\x33\xc0\x55\x68\x8e\xb4\x41\x00\x64\xff\x30\x64\x89\x20\x6a\x20\xe8\xd3\xb1\xfe\xff\x50\xe8\xfd\xb2\xfe\xff\x6a\xff\xe8\xd6\xb1\xfe\xff\x50\xe8\xf8\xb2\xfe\xff\xe8\xeb\xfd\xff\xff\x8d\x4d\xf0\x33\xd2\xb8\xf4\xb4\x41\x00\xe8\x9c\xb5\xff\xff\x8b\x55\xf0\xb8\xfc\xee\x41\x00\xe8/s and $virus = "W32/SirCam\@MM", last LINE;
				}
				if($total==20480) {
					/\x43\x65\x63\x69\x6c\x65\x20\x63\x6f\x64\x65\x64\x20\x62\x79\x20\x53\x30\x2f\x42\x30\x5b\x69\x6b\x78\x5d\x2c\x20\x6d\x61\x64\x65\x20\x69\x6e\x20\x61\x73\x73\x65\x6d\x62\x6c\x79\x00\x68/s and $virus = "W32/Cecile.dr", last LINE;
				}
				if($total>184320) {
					/\x4b\x49\x4c\x54\x52\x4f\x20\x2a\x20\x4d\x53\x4e\x57\x48/s and $virus = "W32/Kitro\@MM", last LINE;
				}
				if($total==154624) {
					/\x52\x61\x5a\x13\x2f\x47\x45\x44\x5a\x41\x43\xaf\x0c\x2d\xa5\xfb\x37\x35\x37\x20\x31\x20\x34\x31\x34\x0f\x3c\xb3\xf1\xff\xa6\x2f\x54\x8d\x43\x4f\x4c\x4f\x4d\x42\x49\x41\x31\x5d/s and $virus = "W32/Blinkom", last LINE;
				}
				if($total>2048) {
					/\x04\x41\x40\x00\x0a\x41\x40\x00\x10\x41\x40\x00\x16\x41\x40\x00\x1c\x41\x40\x00\x24\x41\x40\x00\x2a\x2e\x6c\x6e\x6b\x00\x2a\x2e\x65\x78\x65\x00\x2a\x2e\x73\x63\x72\x00\x2a\x2e\x65\x6d\x6c\x00\x2a\x2e\x2a\x68\x74\x6d\x2a\x00\x2a\x2e\x64\x62\x78\x00\x3c\x73\x6b\x72\x61\x74\x74\x61\x68\x61\x68\x61\x40\x68\x6f\x74\x6d\x61\x69\x6c\x2e\x63\x6f\x6d\x3e/s and $virus = "W32/Ganda\@MM", last LINE;
				}
				if($total<=2048) {
					/\x49\x2d\x57\x6f\x72\x6d\x2e\x4a\x61\x70\x61\x6e\x69\x7a\x65/s and $virus = "W32/Fbound.c\@MM", last LINE;
				}
				if($total==70656) {
					/\x6c\x6f\x76\x65\x6c\x6f\x72\x6e\x40\x79\x61\x68\x6f\x6f\x2e\x63\x6f\x6d/s and $virus = "W32/Lovelorn\@MM", last LINE;
				}
				if($total==7168) {
					/\x62\x65\x67\x69\x6e\x20\x36\x34\x34\x20\x48\x61\x70\x70\x79\x39\x39\x2e\x65\x78\x65.+\x65\x6e\x64.+\x53\x6b\x61/s and $virus = "W32/Ska\@M", last LINE;
					/\xfd\xff\xff\xff\x4d\x41\x49\x4c\x20\x46\x52\x4f\x4d\x3a\x20\x3c\x61\x64\x6d\x69\x6e\x40\x64\x75\x6d\x61\x2e\x67\x6f\x76\x2e\x72\x75\x3e\x03\x78\xdd\xdf\x20\x52\x43\x50\x54\x20\x54\x4f/s and $virus = "W32/Pate.b", last LINE;
				}
				if($total==205824) {
					/\x7a\x65\x72\x6f\x2e\x65\x78\x65\x7f\x07\x32\xf2\x97\x06\x44\x6c\x44\x69\x72\x30\x18\x4b\xb5\x85\x58\xf8\x61\x7a\x61\x61\x5c\x27\x67\x31\x43\x56\x3a\xd7\x77\x84\x03\x17\xf1\x63\x70\x6c\x6f\x5c\x6b\x27\x0b\x7a\x10\x0f\x76\x31\x0a\xee\xa2\x19\x77\x09\x2e\x21\x60\x86\x4d\xc1\x5f\x2f\x17\x10\x08\x2c\x7b\x01\xa2\x2e\x68\x6f\x74\x0f\x86\x6f\x9c\x77\x32\x3e\x46\x6f\x6c\xc0\x96\x22\x07\x63\x36\x0c\x49\x6f\x73\xb4\xf5\x5f\x4a\x27\x73\x6b\x79\x4c\x61\x62/s and $virus = "W32/Duni.worm.b", last LINE;
				}
			} elsif($subtype == 5) {
				if($total==21504) {
					/\xb8\xc7\x64\x83\x71\xfe\xd2\x5c\x06\x38\x4a\xfc\x14\xa3\x98\x53\xd9\x1e\xf9\x69\x7f\x57\x52\x79\x65\x58\x40\x99\x7b\x36\xd6\x58\x06\x3e\xd3\xec\x24\xb4\xb9\x4b\x35\x19\x9a\x05\x3a\x1a\x6f/s and $virus = "W32/Sobig.e\@MM", last LINE;
					/\x53\x7f\xf3\xff\xff\x75\x6e\x4d\x6f\x6e\x54\x75\x65\x57\x65\x64\x54\x68\x75\x46\x72\x69\x53\x61\x74\x4a\x61\x6e\x46\x65\x62\x4d\xff\xb7\x76\xfb\x61\x72\x41\x70\x72\x05\x79\x4a\x26\x02\x6c\x41\x75\x67\x53\x65\x70\x4f\x63\x74\x5b\x81\xfa\xfd\x4e\x6f\x76\x44\x65\x63\x3f\x54\x5a\x1b\x1c\x74\x7b\xb7\xa9\xff\x69\x6d/s and $virus = "W32/Myparty.b\@MM", last LINE;
				}
				if($total>=30720 && $total<=43008) {
					/\x73\x72\x63\x3d\x33[^\x44]*\x44\x63\x69\x64\x3a\x57\x38\x64\x71\x77\x71\x38[^\x71]*\x71\x39\x31\x4f\x31\x33/s and $virus = "W32/Frethem.fl\@MM", last LINE;
				}
				if($total==46080) {
					/\x50\xe8\xce\x86\x00\x00\x83\xc4\x04\xc7\x46\x44\x00\x00\x00\x00\x8b\xce\xe8\xf9\xf8\xff\xff\x5e\xc3\x90\x90\x90\x90\x90\x90\x90\x8b\x44\x24\x08\x8b\x54\x24\x04\x50\x52\xe8\x31\xf9\xff\xff\xc2/s and $virus = "BackDoor-ARG", last LINE;
				}
				if($total==43008) {
					/\x7a\x49\x4e\x47\x0e\x3b\x4f\x4d\x6b\xbe\xdd\xfe\xb2\x4e\x11\x52\x36\x30\x32\x38\x08\x2d\x20\x47\x61\x62\x6c\x65\xeb\x70\x34\x3a\x50\x20\x1f/s and $virus = "W32/Palyh\@MM", last LINE;
				}
				if($total==29696 || $total==71680) {
					/\x3a\x2f\x2f\x77\x77\x32\x2e\x66.{1,2}\x2e\x76.{64,93}\x45\x4c\x45.{18,21}\x4f\x4b.{1,5}\x53\x54\x41.{1,5}\x50\x41\x53\x53/s and $virus = "W32/Swen\@MM", last LINE;
				}
				if($total==8192) {
					/\x43.{3,4}\x42\xdc?\x0d.{4}\x48\x45\x4c\x4c?\x4f.{3,4}\x52\x53[\x6c\xb0]\x54\x0f\x4d\x07?\x41\x49\x1f?\x4c\x20\x46[\xd7\x5e]\x4f[\x82\x09]\x3a\x3c.{3}\x30\x43\x50\x54\x20[\xc9\x25]\x4f[\x58\x61]\x0f/s and $virus = "W32/Bagle.u\@MM", last LINE;
					/\x96\xb4\x66\x55\xd6\xc5\xa3\xe2\x46\x4b\x4c\x54\x52\x61\x6d\x93\x14\x56\x81\xae\xc7\x17\xa2\x2f\x0c\x82\xff\xe8\x24\x2e\xb6\x6a/s and $virus = "W32/Lovgate.x\@MM", last LINE;
					/\x6a\x70\x67\x76\x69\x72/s and $virus = "W32/Perrun", last LINE;
				}
				if($total==3072) {
					/\x7a\x61\x72\x79\x32\x30+.+\x40\x65\x6d\x61\x69\x6c\x2e\x63\x6f\x6d/s and $virus = "W32/MyLife.e\@MM", last LINE;
				}
				if($total==6144 || $total==11264 || $total==13312) {
					/\x31\x35\x31\x2e.{0,4}\x32\x30[\x31\x03]?\x2e?\x30\x2e\x33/s and $virus = "W32/Bagle\@MM", last LINE;
				}
				if($total>=8192 && $total<=20480) {
					/\x51.{1,3}\x54.{14,36}\x41.{1,7}\x52.{0,4}\x43.{3,7}\x4f\x3a.{2,6}\x4d\x41.{0,4}\x49\x4c/s and $virus = "W32/Mimail\@MM", last LINE;
				}
				if(($total>=10240 && $total<=15360) || $total==45056) {
					/[\xee\x95\x00\x95\x2d]\x51\x55\x49\x54[\x8e\xfa\x0d\xa7\x83].{7,27}\x41.{6,35}\x20.{0,55}\x54[\x4f\x6f\x7e]/s and $virus = "W32/Mydoom\@MM", last LINE;
				}
				if($total==35840) {
					/\x73\x6d\x74\x70\x2e\x79\x65\x61\x68\x2e\x6e\x65.+\x2d\x20\x47\x45\x54\x20\x4f\x49\x43\x51/s and $virus = "W32/GOP\@MM", last LINE;
				}
				if($total==102400) {
					/\x90\x60\xe9\x3d\x04\x00\x00\x87\xc3\xa3\x9f\x9f\x8a\x9f\x5a\xcf\xd8\xe3\x9f\xa2\x7c\xca\x3c\x6f\xde\xe3\x9f\x22\x5c\x9b\xe8\xe3\x9f\x9f\x28\x3c\x9b\xe8\xe3\x9f\xae\x24\x05\xa2\x9f\x9f\x66\x24\xd2\xd8\xe3\x9f\x9f\x9f\x9f\x9f\x2c\x24\xa3\xe9\xe3\x9f\xef\x9e/s and $virus = "W32/Lovgate.g\@M", last LINE;
				}
				if($total==15360) {
					/\x57\x00\x69\x00\x6e\x00\x33\x00\x32\x00\x2e\x00\x6d\x00\x65\x00\x72\x00\x63\x00\x75\x00\x72\x00\x79\x00\x20\x00\x43\x00\x6f\x00\x64\x00\x65\x00\x64\x00\x20\x00\x62\x00\x79\x00\x20\x00\x49\x00\x6e\x00\x64\x00\x75\x00\x73\x00\x74\x00\x72\x00\x79\x00\x20\x00\x40\x00\x20\x00\x41\x00\x4e\x00\x56\x00\x58\x00\x67\x00\x72\x00\x6f\x00\x75\x00\x70/s and $virus = "W32/Merkur\@MM", last LINE;
				}
				if($total==267264) {
					/\x7a\x68\x61\x6e\x67\x70\x6f\x00\x58\x2d\x4d\x61\x69\x6c\x65\x72/s and $virus = "W32/Zhangpo\@MM", last LINE;
				}
				if($total==9216) {
					/\x6d\x6f\x76\x20\x5b\x77\xf0\xb7\x56\xb8\x00\x2e\x70\x6c\x57\xb7\x76\x69\x72\x75\x73\x2e\x9f\x6b\x77\xbf\x4d\x6d\x5d\x20\x2a\x2f\x37\x52\x65\x67\xf8\xc2\xa0/s and $virus = "BackDoor-CAG", last LINE;
				}
				if($total<=4096) {
					/\x70\x65\x6e\x74\x61\x67\x6f\x6e\x65/s and $virus = "W32/Goner\@MM", last LINE;
				}
				if($total==6144 || ($total>=14336 && $total<=22528)) {
					/[\x7c\x4c\x02]\x05?[\x20\xfc\x77]\x46\x3c?\x52.{0,16}[\x4f\xaf][\x57\x4d][\x3a\x2b\x12\x11\xad\xb6].{12,49}\x54/s and $virus = "W32/Netsky\@MM", last LINE;
				}
				if($total==5120) {
					/\x71\x9d\xcc\x1e\x2e\x77\xaa\x7a\x4d\x28\xd4\xa4\x00\xcd\xe3\xcd\xe4\x03\x8d\x15\x18\xb3\xd2\x7b\x64\x51\x32\xf1\x27\xc1\x76\x92\xeb\xe0\xd8\x54\x26\x0e\xda\x13\x57\xf3\x3f\xde\x09\x49\x5c\x0a/s and $virus = "W32/Sobig\@MM", last LINE;
					/\x71\x32\x31\x36\x33\x30\x39\x00\x71\x32\x31\x36\x33\x30\x39\x00\x00\x71\x32\x31\x36\x33\x30\x39\x00\x00\x00\x00\xf4\x01\x00\x00\x34\x1c\x40/s and $virus = "W32/Gibe\@MM", last LINE;
				}
				if($total==14336) {
					/\x47\x00\x6f\x00\x62\x00\x6f\x00.+\x74\x00\x65\x00\x61\x00\x6d\x00\x76\x00\x69\x00\x72\x00\x75\x00\x73.+\x4b\x00\x61\x00\x72\x00\x65\x00\x6e/s and $virus = "W32/Gokar\@MM", last LINE;
				}
				if($total>8192 && $total<=20480) {
					/\x44*\x65\x63.+\x4e*\x6f\x76.+\x4f*\x63\x74.+\x53*\x65\x70.+\x41*\x75\x67.+\x4a*\x75\x6c.+\x4d*\x61\x79.+\x46\x65\x62\x13\x61\x53\x61\x27\x46\x72\x69\x00\x54\x68\x75\x00.\x9d\x5b\xfe\x57\x65\x64\x00\x54\x75\x65\x6f\x17\x2f.+\x32\x75/s and $virus = "W32/BadTrans\@MM", last LINE;
				}
				if($total>=19456 && $total<=20480) {
					/\x3a\x2d\x29\x00\x21\x07\x21\x04[^\x5a]*\x5a\x4f\x4e\x45\x41\x4c\x41\x52\x4d.....\x41\x56\x50..\x4c*\x4f\x43\x4b\x44\x4f\x57\x4e\x32\x30/s and $virus = "W32/Yaha.gen\@MM", last LINE;
				}
				if($total==37888) {
					/\x91\xd9\x6f\x2d\x38\x38\x35\x39\x2d\xb9\x62\xb8\x44\x14\x4b\xe8\x07\xeb\xe9\xa7\x15\x25\x74\x2f\xad\x6d\x77\x77\x23\xe1\x35\x58\x2a\x8f\x27\x4d\x4b\x29\xa0\x21\xfa\x4d\x49\x4d\x45\x2d\x49\x3a\xca\xcd\x5a\x5a\xa1\x71\xaa\xfe\x2f\x5a\x58\x18\xee\x6d\x69\x78/s and $virus = "W32/Yaha.u\@MM", last LINE;
					/\x5c\x49\x6e\x74\x65\x72\x66\x61\x63\x65\x73\x00\x00\x00\x43\x6f\x6e\x63\x65\x70\x74\x20\x56\x69\x72\x75\x73\x28\x43\x56\x29\x20\x56\x2e\d\x2c\x20\x43\x6f\x70\x79\x72\x69\x67\x68\x74\x28\x43\x29\d\d\d\d.{10,}\x4d\x49\x4d\x45\x2d\x56\x65\x72\x73\x69\x6f\x6e\x3a\x20\x31\x2e\x30/s and $virus = "W32/Nimda.gen\@MM", last LINE;
				}
				if($total==41984) {
					/\x40\x03\x2e\x33\x3b\x7e\x6f\xad\x92\x6f\x41\x54\x41\x44\x32\x35\x87\x43\x50\x7d\x84\x76\x07\x47\x4f\x3a\x3c\x27\x7d\x4d\x41\x49\x4c\xc2\x4b\xe6\x68\xe8\xa3\x11\x27\x0d\x8b\xd2\x4d\xfa\x48\x45\x4c\x4f\x12\x0f\x32\xb5\x2b\x10\x6f\x61\x2b\x17\x75\xbb\xc3\x03/s and $virus = "W32/Bugbear\@MM", last LINE;
					/\x4d\x00\x61\x00\x63\x00\x68\x00\x69\x00\x6e\x00\x65\x00\x64\x00\x72\x00\x61\x00\x6d\x00\x6f\x00\x6e\x00\x64\x00\x61\x00\x72\x00\x6b\x00\x40\x00\x6c\x00\x61\x00\x74\x00\x69\x00\x6e\x00\x6d\x00\x61\x00\x69\x00\x6c\x00\x2e\x00\x63\x00\x6f\x00\x6d\x00/s and $virus = "W32/Sachiel.worm", last LINE;
				}
				if($total==6144 || $total==3072 || $total==6144) {
					/\x60\xe8\x01\x00\x00\x00\xe8\x83\xc4\x04\xe8\x01\x00\x00\x00\xe9\x5d\x81\xed\xd9\x21\x40\x00\xe8[\x05\x04]\x02\x00\x00\xe8\xeb\x08\xeb\x02\xcd\x20\xff\x24\x24\x9a\x66\xbe[\x35\x52][\x53\x47\xe1]\xe8\x01\x00\x00\x00/s and $virus = "W32/Bagle.e\@MM", last LINE;
				}
				if($total==17408) {
					/\x60\xe8\xed\x10\x00\x00\xc3\x83\x98\x87\x50\x2c\x58\x04\x81\x7c\xdb\xb6\xdf\x38\x34\x30\x2c\xa2\x61\xf9\xce\x6d\x0c\x1d\x13\x33\x04\x00\x86\xc8\xef\xf0\xc3\xef\x8f\xca\x08\xaf\xdb\x93\xcc\xfa/s and $virus = "W32/Netsky.j\@MM", last LINE;
				}
				/\x47\x69\x72\x6c\x73\x00\x5a\x69\x70\x57\x6f\x72\x6d\x00\x00\x7a\x69\x70\x57\x6f\x72\x6d/s and $virus = "IRC/Girls.worm", last LINE;
				if($total==1024) {
					/\x2e\x74\x65\x78\x74\x00\x00\x00\x00.\x00\x00\x00\x10\x00\x00..\x00\x00\x00\x02\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x20\x00\x00\xe0\x2e\x72\x64\x61\x74\x61\x00\x00\x00\x10\x00\x00\x00.\x00\x00\x5a\x00\x00\x00\x00.\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x40\x00\x00\xc0/s and $virus = "W32/Hybrys.gen\@MM", last LINE;
					/\x00\x20\x00\x2e\x6c\x70\x61\x63\x6b\x00\x00\x00\x60\x00\x00\x00\x10\x00\x00\x00\x34\x00\x00\x00\x04\x00{14}\x40\x00\x00\xc0\x2e\x6c\x70\x61\x63\x6b\x00\x00\x00\x10\x00\x00\x00\x70\x00\x00\x00\x0c\x00\x00\x00\x38\x00{14}\x40\x00\x00\xc0\x2e\x6c\x70\x61\x63\x6b\x00\x00\x00\x90\x01\x00\x00\x80\x00\x00\x00\x1c\x00\x00\x00\x44\x00{14}\x40\x00\x00\xc0\x2e\x72\x73\x72\x63\x00\x00\x00\x00\x10\x00\x00\x00\x10\x02\x00\x00\x02\x00\x00\x00\x60\x00{14}\x40\x00\x00\xc0\x2e\x6c\x70\x61\x63\x6b\x00\x00\x00\x30\x00\x00\x00\x20\x02\x00\x00\x22\x00\x00\x00\x62\x00{14}\x40\x00\x00\xc0\x00/s and $virus = "W32/Netsky.n\@MM", last LINE;
					/\x55\x50\x58\x30\x00\x00\x00\x00\x00\xe0\x06\x00\x00\x10\x00\x00\x00\x00\x00\x00\x00\x04\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x80\x00\x00\xe0\x55\x50\x58\x31\x00\x00\x00\x00\x00\x20\x01\x00\x00\xf0\x06\x00\x00\x14\x01\x00\x00\x04\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x40\x00\x00\xe0\x55\x50\x58\x32\x00\x00\x00\x00\x00\x10\x00\x00\x00\x10\x08\x00\x00\x02\x00\x00\x00\x18\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x40\x00\x00\xc0\x00\x00\x00\x00\x00\x00\x00\x00/s and $virus = "W32/Bugbear.b\@MM", last LINE;
					/\x50\x01\x00\x00\x10\x00\x00\xc9\x81\x00\x00\x00\x06\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x60\x00\x00\xe0\x2e\x46\x72\x61\x6c\x69\x27\x00\x73\x03\x00\x00\x00\x60\x01\x00\x00\x04\x00\x00\x00\x02\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x60\x00\x00\xe2\x00\x00/s and $virus = "W32/Higuy\@MM", last LINE;
				}
				if($total==31744) {
					/\x4e\x00\x61\x00\x76\x00\x69\x00\x64\x00\x61\x00\x64\x00\x20\x00\x56\x00\x65\x00\x72\x00\x73\x00\x69\x00\x6f\x00\x6e\x00\x20\x00\d+.+\x43\x00\x6f\x00\x70\x00\x79\x00\x72\x00\x69\x00\x67\x00\x68\x00\x74\x00\x20\x00\x28\x00\x43\x00\x29\x00\x20\x00\d\x00\d\x00\d\x00\d\x00/s and $virus = "W32/Navidad.gen\@M", last LINE;
				}
				if($total==2048) {
					/\x20\x03\x2b\x3a\x99\x12\x22\x80\x2b\x3e\xf7\x26\xe1\x36\x06\xc2\x40\x20\x91\x10\x58\xac\x14\x50\x54\x58\x0a\x0c\x05\x87\x48\x20\x84\x0d\xee\x40\x90\x10\x20\x2a\x23\xb6\x2a\x22\x1b\xda\x8b\x64\x10\x64\x8b\x37\x3f\x1b\x51\xb4\x1e\xc2\xa0\xa2\xa2\xa2\xa2\x22/s and $virus = "Exploit-DcomRpc.gen", last LINE;
					/\x4f\x54\x9a\xf6\xd9\xef\x93\xa0\x2a\xba\x53\x51\x93\xe5\x92\xb9\xc0\x2b\xa3\xf4\x1b\xd3\x6d\x80\x69\x91\xea\xec\x1b\x3d\xa3\x28\x02\x12\x0a\xc4\xa2\x2b\x13\x23\x7c\x82\xbf\xc7\x0c\xa4\x2e\xc1/s and $virus = "W32/Netsky.x\@MM", last LINE;
					/\xe8\x9b\x03\x00\x00\x85\xc0\x75\x01\xc3\xa1\x0c\x48\x00\x10\x53\x33\xdb\x3b\xc3\x75\x17\x68\x00\x14\x00\x10\x68\xe0\x93\x04\x00\x53\x53\xff\x15\x28\x20\x00\x10\xa3\x0c\x48\x00\x10\x6a\x1c\x68\x20\x30\x00\x10\xe8\x27\x04\x00\x00\x6a\x10\x68\x40\x30/s and $virus = "W32/Fizzer.dll", last LINE;
					/\xff\x01\x06\x00\x53\x48\x41\x52\x4f\x4e\x00\x19\x93\x42\x00\x22/s and $virus = "W32/MyLife.j\@MM", last LINE;
					/\xe8\x51\x50\x00\x00\xc3\x52\x53\xb6\x59\xb2\x73\xf9\x66\xe4\x1d\x98\x07\xde\x15\xa5\x94\xd0\x1d\x54\x7b\xf9\x34\x6b\x97\x44\x8f\xc6\xb0\xb9\xaa\x11\x70\xb4\x97\x8c\xc3\x0b\x33\x15\xa1\x60\x05\x51\x89\x80\xf5\x25\xba\x89\xa9\xfc\x3d\x67\x74\x47\xee\x91\x19/s and $virus = "W32/Bobax.worm.c", last LINE;
					/\x9d\x7e\x58\xd8\x63\x2f\xb7.\xce\x8c\x24\xe8\x25.\x6c\x68\x23\x36\x94\x0b\x6e\x51\x52.{6,7}\x0c\x10\xe4\xbf..\x41\x68\xc9\xff....\x39\xf2\xae\xf7\xd1\x2b\xf9\x8b\xf7\x8b\xd9\x8b\xfa\x10\x0e\x8b\xcb\x4f\xc1\xe9......\xa5\xa1.\xd2\x03\xcb\x83\xe1\x03\x95\xf3\xa4\x8b\x57....\xfc\xb4\xab\x81\x52\x66\xb4........\x15\x2c\xce\x52......\x88\x50\x53\x78\xc7\x0d\x04\x03/s and $virus = "W32/Lirva.gen\@MM", last LINE;
					/\x43\x6f\x64\x65\x64\x20\x2e\x2e\x2e\x62\x79\x20\x42\x65\x67\x62\x69\x65\x2c\x20\x53\x6c\x6f\x76\x61\x6b\x69\x61/s and $virus = "W32/Gibe.b\@MM", last LINE;
					/\x00\x63\x3a\x5c\x74\x6d\xff\xbf\xfc\xff\x70\x2e\x65\x78\x65\x00\x68\x74\x74\x70\x3a\x2f\x2f\x77\x2e\x61\x71\x75\x61\x72\x69\x75\x6d\x2d\x66\x69\x73\x20\xab\x6f\xff\x68\x2e\x72\x75\x2f\x70\x70\x61\x02\x2e\x62/s and $virus = "Downloader-GN", last LINE;
				}
				if($total==11264) {
					/\xe0\x5f\x4d\x69\x72\x63\x44\x43\x17\xa2/s and $virus = "W32_Darby\@MM", last LINE;
					/\x49\x00\x20\x00\x63\x00\x6f\x00\x6d\x00\x65\x00\x20\x00\x69\x00\x6e\x00\x20\x00\x70\x00\x69\x00\x65\x00\x63\x00\x65\x00\x2e\x00\x20\x00\x20\x00\x20\x00\x4d\x00\x79\x00\x20\x00\x6e\x00\x61\x00\x6d\x00\x65\x00\x20\x00\x69\x00\x73\x00\x20\x00\x4a\x00\x65\x00\x72\x00\x72\x00\x79/s and $virus = "W32/Choke.c.worm", last LINE;
				}
				if($total==25600) {
					/\x95\x2b\x66\x8a\xd1\xbd\xf0\xe1\x48\x8a\xd1\x97\x71\xc5\xff\x7b\xca\xe8\x0a\x48\x6f\x6f\x5d\xb6\x86\xe6\x3c\xcf\xcd\xae\xe4\x58\x31\x46\x54\x16\xef\xc3\xc1\xed\xc4\xcd\x52\xc6\x2e\xc6\xd0\x13\x58/s and $virus = "BackDoor-ARG.dr", last LINE;
				}
				if($total>=6144 && $total<=9216) {
					/\x2b\x0c\x01[\x94\x54\xcc\x2c].{2,7}\x6f\x6b\x75\x6d.{0,4}\x65\x6e\x74/s and $virus = "W32/Sober\@MM", last LINE;
				}
				if($total==12288) {
					/\x42\x72\x69\x64\x65\x00\x42\x72\x69\x64\x65\x00\x00\x42\x72\x69\x64\x65/s and $virus = "W32/Braid\@MM", last LINE;
				}
				if($total==44032) {
					/\xbd\xff\x57\x4f\x52\x4b\x2d\x53\x45\x58\x59\x33\x0f\x54\x55\x05\xa3\xde\x3b\x13\x4b\x61\x7a\x61\xd3\x5e\x07\x30\xe1\xef\x50\x9f\x76\x62\x73\x4b\x36\x94\x01\x68\x3a\x03/s and $virus = "W32/Oror.aa\@MM", last LINE;
				}
				if($total==22528) {
					/\x57\x00\x49\x00\x4e\x00\x4c\x00\x30\x00\x47\x00\x30\x00\x4e\x00\x2e\x00\x45\x00\x58\x00\x45/s and $virus = "W32/Shoho.gen\@MM", last LINE;
					/\x85\xff\x8a\xd9\x73\x5c\x04\xc7\x6f\x2e\x63\x6a\x64\x72\x61\xff\xcf\xfe\x6f\xc5\x47\x45\x54\x20\x68\x74\x74\x70\x3a\x2f\x2f\x66\x04\x20\x48\x54\x54\x50\x2f\x31\x2e\x30\x6c\xa4\x9d\x10\xc8\x2f/s and $virus = "Proxy-Cidra", last LINE;
					/\x68\x6f\x74\x6d\x61\x69\x6c\x5f\x68\x61\x63\x6b\x2e\x65\x78\x65\x46\x66\x72\x69\x65\x6e\x64\x73\x68\x69\x70\x2e\x5f\x68\xff\xef\x73\x63\x72\x1f\x77\x6f\x72\x6c\x64\x5f\x6f\x66\x5f\x31\x11\x61\x6b\xff\x61\xff\xb8\x65\x12\x0e\x77\x65\x65\x74\x17\x42\x65\x5f\x48\x61\x70\x70/s and $virus = "W32/Yaha.k", last LINE;
				}
				if($total==10240) {
					/\x43\x00\x68\x00\x6f\x00\x6b\x00\x65\x00\x20\x00\x2c\x00\x20\x00\x43\x00\x6f\x00\x70\x00\x79\x00\x72\x00\x69\x00\x67\x00\x68\x00\x74\x00\x20\x00\xae\x00\x20\x00\x31\x00\x38\x00\x38\x00\x36/s and $virus = "W32/Choke.gen.worm", last LINE;
					/\xdb\x0f\x84\xb5\x00\x00\x00\x8b\x3b\x3b\xfe\x73\xef\x52\x53\x56\x57\x51\x8b\x55\x18\x83\xe9\x02\x66\x8b\x04\x31\x66\x39\x04\x39\x0f\x85\x8c\x00\x00\x00\x8b\xd9\xc1\xf9\x02\x85\xc9\x74\x11\x8b\x06\x33\x07\x83\xc6\x04\x83\xc7\x04\x85\xc0\x75\x03\x49\x75\xef\x75\x16\x03\xcb\x83\xe1\x03\x85\xc9\x74\x0d\x8a\x06\x8a\x27\x46\x47\x38\xe0\x75\x03\x49\x75\xf3\x75\x58\x59\x5f\x5e\x56\x57\x03\xf9\x03\xf1\x83\xe9\x04\x83\xc1\x04\x8b\xc2\x83\xe8\x04\x78\x28\x3b\xc8/s and $virus = "W32/Fizzer.gen\@MM", last LINE;
					/\x3a\x00\x5c\x00\x76\x00\x69\x00\x72\x00\x75\x00\x73\x00\x5c\x00\x76\x00\x69\x00\x72\x00\x5c\x00\x63\x00\x75\x00\x72\x00\x72\x00\x5c\x00\x50\x00\x72\x00\x6f\x00\x6a\x00\x65\x00\x63\x00\x74\x00\x31\x00\x2e\x00\x76\x00\x62/s and $virus = "W32/ProLin\@MM", last LINE;
				}
				if($total==2048 || $total==4096) {
					/\x06\x00\x42\x00\x49\x00\x4e\x00\x41\x00\x52\x00\x59\x00\x01\x00\x30\x00\x00\x00\x00\x00\x00\x00\x6b\x7d\x66\x85\x94\x15\xad\x1d\xd6\x94\xdd\xc4\x89\xe6\x39\x31\x49\xad\xb5\x58\xf0\x93\x97\x32\x59\x2b\xd1\xc0\xfd\x16\x8e\x4e/s and $virus = "W32/Netsky.p\@MM", last LINE;
				}
				if($total==50176) {
					/\x34\x7b\x4d\x54\x50\xb4\x76\xc2\x11\xb7\x63\x70\xbb\x70\xb3\x26\x6f\x6f\x5d\x72\xc7\x4b\x2d\x38\x29\x39\x2d\x31\x0b\x51\x55\x49\x10\x86\xf0\x87\x54\x19\x45\x48\x4c\x4f\x24\x2b\x06\xe0\xc2\xf2\x50\x61\x77\x43\x64\x3a\xc6\xfd\xf7\x28\x15\xa7\xbb\x0b\x41\x55\x54\x48\x20\x7f\x03\x9a\xbd\x2e\x47\x49\x4e\x27\xbd\x4c\x20\x46\x52\x4f\x4d\xe3\xb1\x8a\x19\x58\xa3\x43\x50\x54\xc4\x5b\x31\x97/s and $virus = "W32/Sobig.c\@MM", last LINE;
				}
				if($total==29696) {
					/\x45\xf8\x79\x00\x2b\x6b\x23\x4c\x79\x6e\x61\x2e\x28\x63\x6d\xdb\x93\x6f\xd6\xed\x68\xa0/s and $virus = "W32/Sobig.f\@MM", last LINE;
				}
				if($total==66560) {
					/\x4e\x65\x74\x4d\x65\x65\x74\x69\x6e\x67\x20\x52\x65\x6d\x6f\x74\x65\x20\x44\x65\x73\x6b\x74\x6f\x70\x20\x28\x52\x50\x43\x29\x20\x53\x68\x61\x72\x69\x6e\x67\x00\x73\x6d\x74\x70\x2e\x31\x36\x33\x2e\x63\x6f\x6d\x00\x00\x00\x00/s and $virus = "BackDoor-AQJ", last LINE;
				}
				if($total==18432) {
					/\x4d\x61\x64\x65\x20\x62\x79\x20\x41\x78\x69\x61\x6c\x69\x73\x20\x41\x58\x2d\x49\x63\x6f\x6e\x20\d\x2e\d/s and $virus = "BackDoor-AJH", last LINE;
				}
			} elsif($subtype == 6) {
				if($total==162816) {
					/\xdb\x0f\x84\xb5\x00\x00\x00\x8b\x3b\x3b\xfe\x73\xef\x52\x53\x56\x57\x51\x8b\x55\x18\x83\xe9\x02\x66\x8b\x04\x31\x66\x39\x04\x39\x0f\x85\x8c\x00\x00\x00\x8b\xd9\xc1\xf9\x02\x85\xc9\x74\x11\x8b\x06\x33\x07\x83\xc6\x04\x83\xc7\x04\x85\xc0\x75\x03\x49\x75\xef\x75\x16\x03\xcb\x83\xe1\x03\x85\xc9\x74\x0d\x8a\x06\x8a\x27\x46\x47\x38\xe0\x75\x03\x49\x75\xf3\x75\x58\x59\x5f\x5e\x56\x57\x03\xf9\x03\xf1\x83\xe9\x04\x83\xc1\x04\x8b\xc2\x83\xe8\x04\x78\x28\x3b\xc8/s and $virus = "W32/Fizzer.gen\@MM", last LINE;
				}
			}
			if($total>1024) {
				/\x57\x69\x6e\x33\x32\x2e\x47\x69\x72\x69\x67\x61\x74\x20\x69\x73\x20\x6e\x6f\x77\x20\x61\x63\x74\x69\x76\x65\x21/s and $virus = "W32/Giri.dr", last LINE;
				/\x4e\x45\x54\x2e\x64\x6f\x74\x4e\x45\x54\x20\x62\x79\x20\x42\x65\x6e\x6e\x79\x2f\x32\x39\x41/s and $virus = "W32/Donut.dr", last LINE;
				/\x57\x71\x6b.+\x3d\x75\x73\x65\x72\x58\x75.+\x5c\x65\x78\x70\x6c\x6f\x72\x65\x72/s and $virus = "W95/Elkern.cav.c", last LINE;
			}
			if($total>=7168 && $total<8192) {
				/\x4a\x54\x4d\x20\x2d\x20\x66\x72\x6f\x6d\x20\x65\x5b\x61\x78\x5d\x20\x74\x6f\x20\x48\x6f\x6d\x65\x72\x20\x54\x68\x61\x20\x50\x69\x6c\x65/s and $virus = "W32/HLL.ow.24590", last LINE;
			}
			if($total>=1024 && $total<=5120) {
				/\xed.{0,13}\xe8.{0,17}\xe9.{0,12}\xff\xff\xff.{0,19}\x83/s and $virus = "W32/Netsky.c\@MM", last LINE;
			}
			if($total>2048) {
				/\x7e\x46\x75\x6e\x20\x4c\x6f\x76\x69\x6e\x67\x20\x43\x72\x69\x6d\x69\x6e\x61\x6c\x7e/s and $virus = "W32/FunLove.4099", last LINE;
			}
			if($total<=6144) {
				/\x77\x6f\x72\x6d\x49\x77\x69\x6e\x67.+\x57\x69\x6e\x33\x32\x2e\x20\x49\x6d\x65\x6c\x64\x61\x20\x74\x68\x65\x20\x56\x42\x20\x56\x69\x72\x75\x73/s and $virus = "W32/Alcop.gen\@MM", last LINE;
			}
			/\x53\x6f\x66\x74\x77\x61\x72\x65\x20\x70\x72\x6f\x76\x69\x64\x65\x20\x62\x79\x20\x5b\x4d\x41\x54\x52\x69\x58\x5d\x20\x56\x58\x20\x74\x65\x61\x6d/s and $virus = "W32/MTX.gen\@M", last LINE;
			/\x0d\x0a\x2e\x0d\x0a\x00\x00\x00\x44\x41\x54\x41\x20\x0d\x0a\x00\x48\x45\x4c\x4f\x20\x25\x73\x0d\x0a\x00\x00\x00\x3e\x0d\x0a\x00\x4d\x41\x49\x4c\x20\x46\x52\x4f\x4d\x3a\x20\x3c\x00\x00\x00\x00\x52\x43\x50\x54\x20\x54\x4f\x3a\x3c\x00\x00\x00\x25\x64\x00\x00/s and $virus = "W32/Klez.gen\@MM", last LINE;
			/\x5b\x57\x69\x6e\x32\x6b\x2e\x4a\x6f\x73\x73\x5d\x20\x62\x79\x20\x52\x61\x74\x74\x65\x72\x2f\d\d\x41/s and $virus = "W32/Joss.919", last LINE;
			/\x14\xff\x56\xb9\x36\xdc\x5a\xbd\x1b\x93\xeb\xea\x5f\x21\xb8\x35\x73\x1b\xfc\xa6\xdc\x6f\x01\x24\x8b\x14\x85\xb8\x6c\x28\x0d\x3b\xd1\x74\x09\x40\xb3\xbb\x95\x4a\x1a\x74\x15\x72\xe5\x1a\x89\x0c\x8b\x00\xcf\xb7\x90\x49\x24\xfe\x81\xc3\x22\x8d\xa5\x68\x7a\xb4/s and $virus = "BackDoor.arsd", last LINE;
			/\x43\x6f\x64\x65\x64\x20\x62\x79\x20\x47\x72\x69\x59\x6f.+\x32\x39\x41/s and $virus = "W32/GriYo.29A.by", last LINE;
			/\x2e\x41\x56\x58\x65\x6e\x63\x72/s and $virus = "W32/XTC\@MM", last LINE;
			/\x57\x49\x4e\x33\x32\x2e\x50\x49\x4c\x53\x45\x4e\x20\x56\x49\x52\x55\x53\x20\x62\x79\x20\x49\x6e\x74\x31\x33\x68\x2f\x49\x4b\x58\x00\x4d\x61\x44\x65\x20\x69\x4e\x20\x50\x61\x52\x61\x47\x75\x41\x79/s and $virus = "W32/Pilsen.cmp.4096", last LINE;
			/\x5b\x69\x4b\x78\x5d\x20\x28\x63\x29\x20\x31\x39\x39\x39\x20\x61\x6c\x6c\x20\x72\x69\x67\x68\x74\x20\x72\x65\x73\x65\x72\x76\x65\x64\x20\x2d\x20\x70\x72\x65\x73\x65\x6e\x74\x20\x41\x6c\x64\x65\x42\x61\x72\x61\x6e/s and $virus = "W32/Adebar.dr", last LINE;
			/\x76\x69\x72\x75\x73\x65\x73.+\x65\x78\x70\x6c\x6f\x69\x74.+\x70\x61\x74\x63\x68\x20\x68\x61\x73\x20\x62\x65\x65\x6e\x20\x73\x75\x70\x70\x6c\x69\x65\x64\x20\x77\x69\x74\x68\x20\x74\x68\x69\x73\x20\x65\x6d\x61\x69\x6c\x20\x61\x6e\x64\x20\x77\x69\x6c\x6c\x20\x66\x69\x78\x20\x74\x68\x65\x20\x73\x65\x63\x75\x72\x69\x74\x79\x20\x68\x6f\x6c\x65.+\x70\x61\x74\x63\x68\x2e\x65\x78\x65/s and $virus = "W32/Heidi\@MM", last LINE;
			/\x59\x61\x70\x21\x20\x57\x68\x61\x74\x20\x63\x6f\x75\x6c\x64\x20\x69\x74\x20\x42\x20\x3f\xa8\x20\x49\x74\x27\x73\x20\x59\x2e\x41\x2e\x50\x2e\x20\x28\x59\x65\x74\x20\x41\x6e\x6f\x74\x68\x65\x72\x20\x50\x61\x72\x61\x73\x69\x74\x65\x29/s and $virus = "HLLP.Yap.8421", last LINE;
			/\x68\x74\x74\x70\x3a\x2f\x2f\x77\x77\x70\x2e\x69\x63\x71\x2e\x63\x6f\x6d\x2f\x73\x63\x72\x69\x70\x74\x73\x2f\x57\x57\x50\x4d\x73\x67\x2e\x64\x6c\x6c\x3f\x66\x72\x6f\x6d\x3d\w+\x26\x66\x72\x6f\x6d\x65\x6d\x61\x69\x6c\x3d\w+\x40\w+\x2e\w+\x26\x73\x75\x62\x6a\x65\x63\x74\x3d\x50\x72\x6f\x67\x72\x61\x6d\x2b\x65\x78\x65\x63\x75\x74\x69\x6f\x6e\x26\x62\x6f\x64\x79\x3d\x54\x68\x65\x2b\x70\x65\x72\x73\x6f\x6e\x2b\x74\x68\x61\x74\x2b\x73\x65\x6e\x74\x2b\x74\x68\x69\x73\x2b\x70\x61\x67\x65\x72\x2c\x2b\x69\x73\x2b\x77\x69\x74\x68\x2b\x61\x2b\x66\x69\x6c\x65\x2b\x67\x65\x6e\x65\x72\x61\x74\x65\x64\x2b\x62\x79\x2b\x4a\x75\x6e\x74\x61\x64\x6f\x72\x26\x74\x6f\x3d\d{6,}\x26\x73\x65\x6e\x64\x3d.+/s and $virus = "MultiDropper-BN", last LINE;
			/\x5b\x57\x69\x6e\x33\x32\x2e\x4f\x72\x61\x6e\x67\x65\x20\x62\x79\x20\x45\x62\x6f\x6c\x61\x5d\x00\x44\x65\x64\x69\x63\x61\x74\x65\x64\x20\x74\x6f\x20\x74\x68\x65\x20\x4e\x59\x46\x44\x20\x61\x6e\x64\x20\x4e\x59\x50\x44\x2e/s and $virus = "W32/Awfull", last LINE;
			/\x57\x69\x6e\x33\x32\x2e\x48\x4c\x4c\x50\x2e\x5a\x61\x75\x73\x68\x6b\x61\x2e\x57\x6f\x72\x6d\x00\x5a\x61\x75\x73\x68\x6b\x61\x00/s and $virus = "W32/HLLP.32767.a", last LINE;
			/\x57\x69\x6e[\x32\x33\x35\x38\x39]{2}\x2e\w+\x2e*\w*\x20\x62\x79\x20\w+.+\x67\x65\x6e\x65\x72\x61*\x74\x69\x6f\x6e\x20\x76\x69\x72\x75\x73\x20/s and $virus = "W32/Blakan", last LINE;
			/\x5b\x57\x69\x6e\x33\x32\x2e[^\x5d]+\x5d\x20.+\x20\x47\x69\x67\x61\x62\x79\x74\x65\x2f\x4d\x65\x74\x61\x70\x68\x61\x73\x65/s and $virus = "W32/GMetaphase.by\@MM", last LINE;
			/\x5b\x57\x69\x6e[^\x5d]+\x5d\x20\x62\x79\x20\x52\x61\x74\x74\x65\x72\x2f\d+/s and $virus = "W32/Ratter.by", last LINE;
			/\x57\x69\x6e\x33\x32\x2f\x41\x73\x74\x72\x6f\x47\x69\x72\x6c\x20\x41\x73\x74\x72\x6f\x43\x6f\x64\x65\x64\x20\x62\x79\x20\x61\x20\x57\x61\x7a\x65\x78\x00\x59\x6f\x75\x72\x20\x73\x79\x73\x74\x65\x6d\x20\x69\x73\x20\x69\x6e\x66\x65\x63\x74\x65\x64\x20\x62\x79\x20\x57\x69\x6e\x33\x32\x2f\x41\x73\x74\x72\x6f\x47\x69\x72\x6c\x20\x76\d+\x2e\d+.+\x44\x65\x64\x69\x63\x61\x74\x65\x64\x20\x74\x6f\x20\x41\x6e\x69\x74\x61\x20\x61\x6e\x64\x20\x6f\x75\x72\x20\x70\x65\x6e\x67\x2d\x67\x75\x69\x6e\x20\x3b\x29\x0d/s and $virus = "Win32.Asorl", last LINE;
			/\x08\xb5\x6d\xea\x46\x82\x32\x67\x62\x42\x2b\x16\x59\x97\xcb\xdb\x40\x1c\x02\xd2\x43\x40\xa0\x99\x65\x20\x99\x2a\x9d\xa1\x21\xa1\xa1\x1d\x55\x05\x19\x01\x57\x55\x32\x8c\x41\xc5\x08\x01\x76\x0a\x43\x0f\x81\x87\xb0\xda\x18\x3d\x42\x28\x28\xa8\x80\xac\xd2\xe9/s and $virus = "W32/Navidad.e\@M", last LINE;
			if($total==1024) {
				/\x50\x45\x00\x00\x4c\x01.\x00....\x00\x00\x00\x00.\x00\x00\x00\xe0\x00..\x0b\x01..\x00..\x00\x00.\x00\x00\x00\x00\x00\x00...\x00\x00\x10\x00\x00\x00..\x00\x00\x00..\x00\x10\x00\x00\x00.\x00\x00.\x00\x00\x00.\x00\x00\x00.\x00.\x00\x70\x6c\x78\x72/s and $virus = "W32/Plex\@MM", last LINE;
				/\x00{2}..\x00{13}[^\x00]\x00\x00.\x2e.{5}\x00\x00\xec[^\x00](\x00|\x01)\x00\x00..\x00\x00.(\x00|\x01)\x00\x00..\x00{13}.\x00\x00[^\x00][^\x2e]/s and $virus = "W32/Magistr.a\@MM", last LINE;
				/\x00{2}..\x00{13}[^\x00]\x00\x00.\x2e.{5}\x00\x00\xed[^\x00](\x00|\x01)\x00\x00..\x00\x00.(\x00|\x01)\x00\x00..\x00{13}.\x00\x00[^\x00][^\x2e]/s and $virus = "W32/Magistr.b\@MM", last LINE;
			}
			if($total>102400 && $total<=160000) {
				/\x49\x2d\x57\x6f\x72\x6d\x2e\x53\x75\x70\x65\x72\x4e\x6f\x76\x61/s and $virus = "W32/Sintesys\@MM", last LINE;
			}
			if($total<=4096) {
				/\x49\x2d\x57\x6f\x72\x6d\x2e\x46\x72\x69\x65\x6e\x64\x73\x00\x43\x6f\x64\x65\x64\x20\x62\x79\x20\x50\x65\x74\x69\x4b\x20\x28\x63\x29\x32\x30\x30\x31.+\x54\x6f\x20\x6d\x79\x20\x66\x72\x69\x65\x6e\x64\x73\x20\x4d\x61\x79\x61\x20\x61\x6e\x64\x20\x4c\x61\x75\x72\x65\x6e\x74/s and $virus = "W32/PetTick\@MM", last LINE;
			}
		} elsif($type == 5) {
			/\x5b\x41\x4e\x4e\x41\x5d\x00\x53\x6c\x61\x72\x74\x69\x62\x61\x72\x74\x66\x61\x73\x74\x2c\x20\x41\x52\x43\x56\x20\x4e\x75\x4b\x45\x20\x74\x68\x65\x20\x46\x72\x65\x6e\x63\x68.+\x28\x63\x29\x20\x41\x52\x63\x56\x20\d\d\d\d\x20\x2d\x20\x45\x6e\x67\x6c\x61\x6e\x64\x20\x52\x61\x69\x6e\x69\x6e\x67\x20\x41\x67\x61\x69\x6e/s and $virus = "ARCV.Anna.742", last LINE;
			/\x59\x6f\x75\x20\x63\x61\x6e\x27\x74\x20\x63\x61\x74\x63\x68\x20\x74\x68\x65\x20\x47\x69\x6e\x67\x65\x72\x62\x72\x65\x61\x64\x20\x4d\x61\x6e\x21\x21\x95/s and $virus = "Ginger.mp", last LINE;
		} elsif($type == 6) {
			if($total==1281) {
				/\x25\x53\x79\x73\x74\x65\x6d\x52\x6f\x6f\x74\x25\x5c\x53\x79\x73\x74\x65\x6d\x33\x32\x5c\x66\x75\x6e\x74\x69\x6d\x65\d\d\x2e\x68\x74\x61/s and $virus = "VBS/Funtime", last LINE;
			}
		} elsif($type == 7) {
			/.+\x43\x6f\x64\x65\x52\x65\x64\x49\x49.+/s and $virus = "W32/CodeRed.c.worm", last LINE;
			/\x48\x4f\x53\x54\x3a\x77\x77\x77\x2e\x77\x6f\x72\x6d\x2e\x63\x6f\x6d\x0a\x20\x41\x63\x63\x65\x70\x74\x3a\x20\x2a\x2f\x2a\x0a\x43\x6f\x6e\x74\x65\x6e\x74\x2d\x6c\x65\x6e\x67\x74\x68\x3a/s and $virus = "W32/CodeRed.a.worm", last LINE;
		}
		$save = substr($buff, (length($buff)/2));
	}
	close(FILE);
	&_set_skip($skip) if($skip);
	$suspicious = 0 if($virus);
	&_set_suspicious($suspicious) if($suspicious);
	return($virus);
}

__END__

=head1 NAME

File::Scan - Perl extension for Scanning files for Viruses

=head1 SYNOPSIS

  use File::Scan;

  $fs = File::Scan->new([, OPTION ...]);
  $fs->set_callback(
    sub {
      my $filename = shift;
      my $bytes = shift;
      ...
      return("Callback Value");
    }
  );
  $fs->scan([FILE]);
  if(my $e = $fs->error) { print "$e\n"; }
  if(my $c = $fs->skipped) { print "file skipped ($c)\n"; }
  if($fs->suspicious) { print "suspicious file\n"; }
  if(my $res = $fs->callback) { print "$res\n"; }

=head1 DESCRIPTION

This module is designed to allows users to scan files for known viruses.
The purpose is to provide a perl module to make plataform independent
virus scanners.

=head1 METHODS

=head2 new([, OPTION ...])

This method create a new File::Scan object. The following keys are 
available:

=over 7

=item callback => 'subroutine reference'

if the item is set then use a callback subroutine reference to provide
extra information and functionalities. The callback subroutine have two
arguments: filename and first 1024 bytes read from the file. This only
work for binary files.

=item extension => 'string'

add the specified extension to the infected file

=item move => 'directory'

move the infected file to the specified directory

=item copy => 'directory'

copy the infected file to the specified directory

=item mkdir => octal_number

if the value is set to octal number then make the specified directories
(example: mkdir => 0755).

=item delete => 0 or 1

if the value is set to 1 delete the infected file

=item max_txt_size => 'size in kbytes'

scan only the text file if the file size is less then max_txt_size. The
default value is 5120 kbytes. Set to 0 for no limit.

=item max_bin_size => 'size in kbytes'

scan only the binary file if the file size is less then max_bin_size. The
default value is 10240 kbytes. Set to 0 for no limit.

=back

=head2 scan([FILE])

This method scan a file for viruses and return the name of virus if a
virus is found.

=head2 set_callback([SUBREF])

This method is another way to install a callback subroutine reference. 
Take a look in callback kay.

=head2 skipped()

This method return a code number if the file was skipped and 0 if not. The
following skipped codes are available:

=over 6

=item 0

file not skipped 

=item 1

file is not vulnerable

=item 2

file has zero size

=item 3

the size of file is small

=item 4

the text file size is greater that the 'max_txt_size' argument

=item 5

the binary file size is greater that the 'max_bin_size' argument

=back

=head2 suspicious()

This method return 1 if the file is suspicious and 0 if not.

=head2 callback()

This method return the result from the callback subroutine.

=head2 error()

This method return a error message if a error happens.

=head1 AUTHOR

Henrique Dias <hdias@aesbuc.pt>

=head1 CREDITS

Thanks to Rui de Castro, Sergio Castro, Ricardo Oliveira, Antonio
Campelo, Branca Silveira, Helena Gomes and Anita Afonso for the help.

Thanks to Fernando Martins for the personal collection of viruses.

=head1 SEE ALSO

perl(1).

=cut
