/*
 * Decompiled with CFR 0.152.
 */
package org.apache.accumulo.core.clientImpl.bulk;

import com.google.common.base.Preconditions;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.collect.Sets;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.SortedMap;
import java.util.TreeMap;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executor;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.accumulo.core.client.AccumuloException;
import org.apache.accumulo.core.client.AccumuloSecurityException;
import org.apache.accumulo.core.client.TableNotFoundException;
import org.apache.accumulo.core.client.admin.TableOperations;
import org.apache.accumulo.core.clientImpl.AccumuloBulkMergeException;
import org.apache.accumulo.core.clientImpl.ClientContext;
import org.apache.accumulo.core.clientImpl.TableOperationsImpl;
import org.apache.accumulo.core.clientImpl.bulk.Bulk;
import org.apache.accumulo.core.clientImpl.bulk.BulkSerialize;
import org.apache.accumulo.core.clientImpl.bulk.ConcurrentKeyExtentCache;
import org.apache.accumulo.core.conf.AccumuloConfiguration;
import org.apache.accumulo.core.conf.ClientProperty;
import org.apache.accumulo.core.conf.ConfigurationTypeHelper;
import org.apache.accumulo.core.conf.Property;
import org.apache.accumulo.core.crypto.CryptoFactoryLoader;
import org.apache.accumulo.core.data.Key;
import org.apache.accumulo.core.data.LoadPlan;
import org.apache.accumulo.core.data.Range;
import org.apache.accumulo.core.data.TableId;
import org.apache.accumulo.core.dataImpl.KeyExtent;
import org.apache.accumulo.core.file.FileOperations;
import org.apache.accumulo.core.file.FileSKVIterator;
import org.apache.accumulo.core.file.blockfile.impl.CachableBlockFile;
import org.apache.accumulo.core.spi.crypto.CryptoService;
import org.apache.accumulo.core.util.Retry;
import org.apache.accumulo.core.util.Validators;
import org.apache.accumulo.core.util.threads.ThreadPoolNames;
import org.apache.accumulo.core.volume.VolumeConfiguration;
import org.apache.commons.io.FilenameUtils;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.io.BinaryComparable;
import org.apache.hadoop.io.Text;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BulkImport
implements TableOperations.ImportDestinationArguments,
TableOperations.ImportMappingOptions {
    private static final Logger log = LoggerFactory.getLogger(BulkImport.class);
    private boolean setTime = false;
    private boolean ignoreEmptyDir = false;
    private Executor executor = null;
    private final String dir;
    private int numThreads = -1;
    private final ClientContext context;
    private String tableName;
    private LoadPlan plan = null;
    private static final byte[] byte0 = new byte[]{0};

    public BulkImport(String directory, ClientContext context) {
        this.context = context;
        this.dir = Objects.requireNonNull(directory);
    }

    @Override
    public TableOperations.ImportMappingOptions tableTime(boolean value) {
        this.setTime = value;
        return this;
    }

    @Override
    public TableOperations.ImportMappingOptions ignoreEmptyDir(boolean ignore) {
        this.ignoreEmptyDir = ignore;
        return this;
    }

    @Override
    public void load() throws TableNotFoundException, IOException, AccumuloException, AccumuloSecurityException {
        TableId tableId = this.context.getTableId(this.tableName);
        FileSystem fs = VolumeConfiguration.fileSystemForPath(this.dir, this.context.getHadoopConf());
        Path srcPath = this.checkPath(fs, this.dir);
        TableOperationsImpl tableOps = new TableOperationsImpl(this.context);
        Map<String, String> tableProps = tableOps.getConfiguration(this.tableName);
        int maxTablets = 0;
        String propValue = tableProps.get(Property.TABLE_BULK_MAX_TABLETS.getKey());
        if (propValue != null) {
            maxTablets = Integer.parseInt(propValue);
        }
        Retry retry = Retry.builder().infiniteRetries().retryAfter(100L, TimeUnit.MILLISECONDS).incrementBy(100L, TimeUnit.MILLISECONDS).maxWait(2L, TimeUnit.MINUTES).backOffFactor(1.5).logInterval(3L, TimeUnit.MINUTES).createRetry();
        boolean shouldRetry = true;
        while (shouldRetry) {
            SortedMap<KeyExtent, Bulk.Files> mappings = this.plan == null ? this.computeMappingFromFiles(fs, tableId, tableProps, srcPath, maxTablets) : this.computeMappingFromPlan(fs, tableId, srcPath, maxTablets);
            if (mappings.isEmpty()) {
                if (this.ignoreEmptyDir) {
                    log.info("Attempted to import files from empty directory - {}. Zero files imported", (Object)srcPath);
                    return;
                }
                throw new IllegalArgumentException("Attempted to import zero files from " + String.valueOf(srcPath));
            }
            BulkSerialize.writeLoadMapping(mappings, srcPath.toString(), arg_0 -> ((FileSystem)fs).create(arg_0));
            List<ByteBuffer> args = Arrays.asList(ByteBuffer.wrap(tableId.canonical().getBytes(StandardCharsets.UTF_8)), ByteBuffer.wrap(srcPath.toString().getBytes(StandardCharsets.UTF_8)), ByteBuffer.wrap(("" + this.setTime).getBytes(StandardCharsets.UTF_8)));
            try {
                tableOps.doBulkFateOperation(args, this.tableName);
                shouldRetry = false;
            }
            catch (AccumuloBulkMergeException ae) {
                if (this.plan != null) {
                    this.checkPlanForSplits(ae);
                }
                try {
                    retry.waitForNextAttempt(log, String.format("bulk import to %s(%s)", this.tableName, tableId));
                }
                catch (InterruptedException e) {
                    throw new RuntimeException(e);
                }
                log.info(ae.getMessage() + ". Retrying bulk import to " + this.tableName);
            }
        }
    }

    private void checkPlanForSplits(AccumuloBulkMergeException abme) throws AccumuloException {
        for (LoadPlan.Destination des : this.plan.getDestinations()) {
            if (!des.getRangeType().equals((Object)LoadPlan.RangeType.TABLE)) continue;
            throw new AccumuloException("The splits provided in Load Plan do not exist in " + this.tableName, abme);
        }
    }

    private Path checkPath(FileSystem fs, String dir) throws IOException, AccumuloException {
        Path ret = dir.contains(":") ? new Path(dir) : fs.makeQualified(new Path(dir));
        try {
            if (!fs.getFileStatus(ret).isDirectory()) {
                throw new AccumuloException("Bulk import directory " + dir + " is not a directory!");
            }
            Path tmpFile = new Path(ret, "isWritable");
            if (!fs.createNewFile(tmpFile)) {
                throw new AccumuloException("Bulk import directory " + dir + " is not writable.");
            }
            fs.delete(tmpFile, true);
        }
        catch (FileNotFoundException fnf) {
            throw new AccumuloException("Bulk import directory " + dir + " does not exist or has bad permissions", fnf);
        }
        return ret;
    }

    @Override
    public TableOperations.ImportMappingOptions executor(Executor service) {
        this.executor = Objects.requireNonNull(service);
        return this;
    }

    @Override
    public TableOperations.ImportMappingOptions threads(int numThreads) {
        Preconditions.checkArgument((numThreads > 0 ? 1 : 0) != 0, (String)"Non positive number of threads given : %s", (int)numThreads);
        this.numThreads = numThreads;
        return this;
    }

    @Override
    public TableOperations.ImportMappingOptions plan(LoadPlan plan) {
        this.plan = Objects.requireNonNull(plan);
        return this;
    }

    @Override
    public TableOperations.ImportMappingOptions to(String tableName) {
        this.tableName = Validators.EXISTING_TABLE_NAME.validate(tableName);
        return this;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static Map<KeyExtent, Long> estimateSizes(AccumuloConfiguration acuConf, Path mapFile, long fileSize, Collection<KeyExtent> extents, FileSystem ns, Cache<String, Long> fileLenCache, CryptoService cs) throws IOException {
        if (extents.size() == 1) {
            return Collections.singletonMap(extents.iterator().next(), fileSize);
        }
        long totalIndexEntries = 0L;
        TreeMap<KeyExtent, MLong> counts = new TreeMap<KeyExtent, MLong>();
        for (KeyExtent keyExtent : extents) {
            counts.put(keyExtent, new MLong(0L));
        }
        Text row = new Text();
        FileSKVIterator index = FileOperations.getInstance().newIndexReaderBuilder().forFile(mapFile.toString(), ns, ns.getConf(), cs).withTableConfiguration(acuConf).withFileLenCache(fileLenCache).build();
        try {
            while (index.hasTop()) {
                Key key = (Key)index.getTopKey();
                ++totalIndexEntries;
                key.getRow(row);
                for (Map.Entry entry : counts.entrySet()) {
                    if (!((KeyExtent)entry.getKey()).contains((BinaryComparable)row)) continue;
                    ++((MLong)entry.getValue()).l;
                }
                index.next();
            }
        }
        finally {
            try {
                if (index != null) {
                    index.close();
                }
            }
            catch (IOException e) {
                log.debug("Failed to close " + String.valueOf(mapFile), (Throwable)e);
            }
        }
        TreeMap<KeyExtent, Long> results = new TreeMap<KeyExtent, Long>();
        for (KeyExtent keyExtent : extents) {
            double numEntries = ((MLong)counts.get((Object)keyExtent)).l;
            if (numEntries == 0.0) {
                numEntries = 1.0;
            }
            long estSize = (long)(numEntries / (double)totalIndexEntries * (double)fileSize);
            results.put(keyExtent, estSize);
        }
        return results;
    }

    public static List<KeyExtent> findOverlappingTablets(Function<Text, KeyExtent> rowToExtentResolver, NextRowFunction nextRowFunction) throws IOException {
        ArrayList<KeyExtent> result = new ArrayList<KeyExtent>();
        Text row = new Text();
        while ((row = nextRowFunction.apply(row)) != null) {
            KeyExtent extent = rowToExtentResolver.apply(row);
            result.add(extent);
            row = extent.endRow();
            if (row == null) break;
            row = BulkImport.nextRow(row);
        }
        return result;
    }

    private static Text nextRow(Text row) {
        Text next = new Text(row);
        next.append(byte0, 0, byte0.length);
        return next;
    }

    public static List<KeyExtent> findOverlappingTablets(ClientContext context, KeyExtentCache keyExtentCache, Path file, FileSystem fs, Cache<String, Long> fileLenCache, CryptoService cs) throws IOException {
        try (FileSKVIterator reader = FileOperations.getInstance().newReaderBuilder().forFile(file.toString(), fs, fs.getConf(), cs).withTableConfiguration(context.getConfiguration()).withFileLenCache(fileLenCache).seekToBeginning().build();){
            List columnFamilies = Collections.emptyList();
            NextRowFunction nextRowFunction = row -> {
                reader.seek(new Range(row, null), columnFamilies, false);
                if (!reader.hasTop()) {
                    return null;
                }
                return ((Key)reader.getTopKey()).getRow();
            };
            List<KeyExtent> list = BulkImport.findOverlappingTablets(keyExtentCache::lookup, nextRowFunction);
            return list;
        }
    }

    private static Map<String, Long> getFileLenMap(List<FileStatus> statuses) {
        HashMap<String, Long> fileLens = new HashMap<String, Long>();
        for (FileStatus status : statuses) {
            fileLens.put(status.getPath().getName(), status.getLen());
        }
        return fileLens;
    }

    private static Cache<String, Long> getPopulatedFileLenCache(Path dir, List<FileStatus> statuses) {
        Map<String, Long> fileLens = BulkImport.getFileLenMap(statuses);
        HashMap absFileLens = new HashMap();
        fileLens.forEach((k, v) -> absFileLens.put(CachableBlockFile.pathToCacheId(new Path(dir, k)), v));
        Cache fileLenCache = CacheBuilder.newBuilder().build();
        fileLenCache.putAll(absFileLens);
        return fileLenCache;
    }

    private SortedMap<KeyExtent, Bulk.Files> computeMappingFromPlan(FileSystem fs, TableId tableId, Path srcPath, int maxTablets) throws IOException, AccumuloException, AccumuloSecurityException, TableNotFoundException {
        Map<String, List<LoadPlan.Destination>> fileDestinations = this.plan.getDestinations().stream().collect(Collectors.groupingBy(LoadPlan.Destination::getFileName));
        List<FileStatus> statuses = BulkImport.filterInvalid(fs.listStatus(srcPath, p -> !p.getName().equals("loadmap.json")));
        Map<String, Long> fileLens = BulkImport.getFileLenMap(statuses);
        if (!fileDestinations.keySet().equals(fileLens.keySet())) {
            throw new IllegalArgumentException("Load plan files differ from directory files, symmetric difference : " + String.valueOf(Sets.symmetricDifference(fileDestinations.keySet(), fileLens.keySet())));
        }
        ConcurrentKeyExtentCache extentCache = new ConcurrentKeyExtentCache(tableId, this.context);
        fileDestinations.values().stream().flatMap(Collection::stream).filter(dest -> dest.getRangeType() == LoadPlan.RangeType.FILE).flatMap(dest -> Stream.of(dest.getStartRow(), dest.getEndRow())).filter(Objects::nonNull).map(Text::new).sorted().distinct().forEach(row -> {
            try {
                extentCache.lookup((Text)row);
            }
            catch (Exception e) {
                throw new RuntimeException(e);
            }
        });
        TreeMap<KeyExtent, Bulk.Files> mapping = new TreeMap<KeyExtent, Bulk.Files>();
        for (Map.Entry<String, List<LoadPlan.Destination>> entry : fileDestinations.entrySet()) {
            String fileName = entry.getKey();
            List<LoadPlan.Destination> destinations = entry.getValue();
            Set<KeyExtent> extents = this.mapDestinationsToExtents(tableId, extentCache, destinations);
            log.debug("The file {} mapped to {} tablets.", (Object)fileName, (Object)extents.size());
            this.checkTabletCount(maxTablets, extents.size(), fileName);
            long estSize = (long)((double)fileLens.get(fileName).longValue() / (double)extents.size());
            for (KeyExtent keyExtent : extents) {
                mapping.computeIfAbsent(keyExtent, k -> new Bulk.Files()).add(new Bulk.FileInfo(fileName, estSize, 0L));
            }
        }
        return BulkImport.mergeOverlapping(mapping);
    }

    private Text toText(byte[] row) {
        return row == null ? null : new Text(row);
    }

    private Set<KeyExtent> mapDestinationsToExtents(TableId tableId, KeyExtentCache kec, List<LoadPlan.Destination> destinations) {
        HashSet<KeyExtent> extents = new HashSet<KeyExtent>();
        for (LoadPlan.Destination dest : destinations) {
            if (dest.getRangeType() == LoadPlan.RangeType.TABLE) {
                extents.add(new KeyExtent(tableId, this.toText(dest.getEndRow()), this.toText(dest.getStartRow())));
                continue;
            }
            if (dest.getRangeType() == LoadPlan.RangeType.FILE) {
                Text startRow = new Text(dest.getStartRow());
                Text endRow = new Text(dest.getEndRow());
                KeyExtent extent = kec.lookup(startRow);
                extents.add(extent);
                while (!extent.contains((BinaryComparable)endRow) && extent.endRow() != null) {
                    extent = kec.lookup(BulkImport.nextRow(extent.endRow()));
                    extents.add(extent);
                }
                continue;
            }
            throw new IllegalStateException();
        }
        return extents;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private SortedMap<KeyExtent, Bulk.Files> computeMappingFromFiles(FileSystem fs, TableId tableId, Map<String, String> tableProps, Path dirPath, int maxTablets) throws IOException, AccumuloException, AccumuloSecurityException {
        Executor executor;
        ExecutorService service = null;
        if (this.executor != null) {
            executor = this.executor;
        } else if (this.numThreads > 0) {
            service = this.context.threadPools().getPoolBuilder(ThreadPoolNames.BULK_IMPORT_CLIENT_LOAD_POOL).numCoreThreads(this.numThreads).enableThreadPoolMetrics().build();
            executor = service;
        } else {
            String threads = this.context.getConfiguration().get(ClientProperty.BULK_LOAD_THREADS.getKey());
            service = this.context.threadPools().getPoolBuilder(ThreadPoolNames.BULK_IMPORT_CLIENT_BULK_THREADS_POOL).numCoreThreads(ConfigurationTypeHelper.getNumThreads(threads)).enableThreadPoolMetrics().build();
            executor = service;
        }
        try {
            SortedMap<KeyExtent, Bulk.Files> sortedMap = this.computeFileToTabletMappings(fs, tableId, tableProps, dirPath, executor, this.context, maxTablets);
            return sortedMap;
        }
        finally {
            if (service != null) {
                service.shutdown();
            }
        }
    }

    public static List<FileStatus> filterInvalid(FileStatus[] files) {
        ArrayList<FileStatus> fileList = new ArrayList<FileStatus>(files.length);
        for (FileStatus fileStatus : files) {
            String fname = fileStatus.getPath().getName();
            if (fileStatus.isDirectory()) {
                log.debug("{} is a directory, ignoring.", (Object)fileStatus.getPath());
                continue;
            }
            if (FileOperations.isBulkWorkingFile(fname)) {
                log.trace("{} is an internal working file, ignoring.", (Object)fileStatus.getPath());
                continue;
            }
            if (!FileOperations.getValidExtensions().contains(FilenameUtils.getExtension((String)fname))) {
                log.warn("{} does not have a valid extension, ignoring", (Object)fileStatus.getPath());
                continue;
            }
            fileList.add(fileStatus);
        }
        return fileList;
    }

    public SortedMap<KeyExtent, Bulk.Files> computeFileToTabletMappings(FileSystem fs, TableId tableId, Map<String, String> tableProps, Path dirPath, Executor executor, ClientContext context, int maxTablets) throws IOException, AccumuloException, AccumuloSecurityException {
        ConcurrentKeyExtentCache extentCache = new ConcurrentKeyExtentCache(tableId, context);
        List<FileStatus> files = BulkImport.filterInvalid(fs.listStatus(dirPath, p -> !p.getName().equals("loadmap.json")));
        Cache<String, Long> fileLensCache = BulkImport.getPopulatedFileLenCache(dirPath, files);
        ArrayList<CompletableFuture<Map>> futures = new ArrayList<CompletableFuture<Map>>();
        CryptoService cs = CryptoFactoryLoader.getServiceForClientWithTable(context.instanceOperations().getSystemConfiguration(), tableProps, tableId);
        for (FileStatus fileStatus : files) {
            Path path = fileStatus.getPath();
            CompletableFuture<Map> future = CompletableFuture.supplyAsync(() -> {
                try {
                    long t1 = System.currentTimeMillis();
                    List<KeyExtent> extents = BulkImport.findOverlappingTablets(context, extentCache, filePath, fs, fileLensCache, cs);
                    this.checkTabletCount(maxTablets, extents.size(), filePath.toString());
                    Map<KeyExtent, Long> estSizes = BulkImport.estimateSizes(context.getConfiguration(), filePath, fileStatus.getLen(), extents, fs, fileLensCache, cs);
                    HashMap<KeyExtent, Bulk.FileInfo> pathLocations = new HashMap<KeyExtent, Bulk.FileInfo>();
                    for (KeyExtent ke : extents) {
                        pathLocations.put(ke, new Bulk.FileInfo(filePath, estSizes.getOrDefault(ke, 0L)));
                    }
                    long t2 = System.currentTimeMillis();
                    log.debug("Mapped {} to {} tablets in {}ms", new Object[]{filePath, pathLocations.size(), t2 - t1});
                    return pathLocations;
                }
                catch (Exception e) {
                    throw new CompletionException(e);
                }
            }, executor);
            futures.add(future);
        }
        TreeMap<KeyExtent, Bulk.Files> mappings = new TreeMap<KeyExtent, Bulk.Files>();
        for (CompletableFuture completableFuture : futures) {
            try {
                Map pathMapping = (Map)completableFuture.get();
                pathMapping.forEach((ext, fi) -> mappings.computeIfAbsent((KeyExtent)ext, k -> new Bulk.Files()).add((Bulk.FileInfo)fi));
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                throw new RuntimeException(e);
            }
            catch (ExecutionException e) {
                throw new RuntimeException(e);
            }
        }
        return BulkImport.mergeOverlapping(mappings);
    }

    static SortedMap<KeyExtent, Bulk.Files> mergeOverlapping(SortedMap<KeyExtent, Bulk.Files> mappings) {
        ArrayList<KeyExtent> extents = new ArrayList<KeyExtent>(mappings.keySet());
        for (KeyExtent ke : extents) {
            Set<KeyExtent> overlapping = KeyExtent.findOverlapping(ke, mappings);
            for (KeyExtent oke : overlapping) {
                if (ke.equals(oke)) continue;
                if (ke.contains(oke)) {
                    ((Bulk.Files)mappings.get(ke)).merge((Bulk.Files)mappings.remove(oke));
                    continue;
                }
                if (oke.contains(ke)) continue;
                throw new RuntimeException("Error during bulk import: Unable to merge overlapping tablets where neither tablet contains the other. This may be caused by a concurrent merge. Key extents " + String.valueOf(oke) + " and " + String.valueOf(ke) + " overlap, but neither contains the other.");
            }
        }
        return mappings;
    }

    private void checkTabletCount(int tabletMaxSize, int tabletCount, String file) {
        if (tabletMaxSize > 0 && tabletCount > tabletMaxSize) {
            throw new IllegalArgumentException("The file " + file + " attempted to import to " + tabletCount + " tablets. Max tablets allowed set to " + tabletMaxSize);
        }
    }

    private static class MLong {
        long l;

        public MLong(long i) {
            this.l = i;
        }
    }

    public static interface NextRowFunction {
        public Text apply(Text var1) throws IOException;
    }

    public static interface KeyExtentCache {
        public KeyExtent lookup(Text var1);
    }
}

