/**
 *
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.airavata.registry.core.app.catalog.resources;

import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.List;

import javax.persistence.EntityManager;
import javax.persistence.Query;

import org.apache.airavata.common.exception.ApplicationSettingsException;
import org.apache.airavata.common.utils.AiravataUtils;
import org.apache.airavata.registry.core.app.catalog.model.GridftpDataMovement;
import org.apache.airavata.registry.core.app.catalog.util.AppCatalogJPAUtils;
import org.apache.airavata.registry.core.app.catalog.util.AppCatalogQueryGenerator;
import org.apache.airavata.registry.core.app.catalog.util.AppCatalogResourceType;
import org.apache.airavata.registry.cpi.AppCatalogException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class GridftpDataMovementResource extends AppCatAbstractResource {
	private final static Logger logger = LoggerFactory.getLogger(GridftpDataMovementResource.class);
	private String dataMovementInterfaceId;
	private String securityProtocol;
    private Timestamp createdTime;
    private Timestamp updatedTime;

    public Timestamp getCreatedTime() {
        return createdTime;
    }

    public void setCreatedTime(Timestamp createdTime) {
        this.createdTime = createdTime;
    }

    public Timestamp getUpdatedTime() {
        return updatedTime;
    }

    public void setUpdatedTime(Timestamp updatedTime) {
        this.updatedTime = updatedTime;
    }
	
	@Override
	public void remove(Object identifier) throws AppCatalogException {
		EntityManager em = null;
		try {
			em = AppCatalogJPAUtils.getEntityManager();
			em.getTransaction().begin();
			AppCatalogQueryGenerator generator = new AppCatalogQueryGenerator(GRIDFTP_DATA_MOVEMENT);
			generator.setParameter(GridftpDataMovementConstants.DATA_MOVEMENT_INTERFACE_ID, identifier);
			Query q = generator.deleteQuery(em);
			q.executeUpdate();
			em.getTransaction().commit();
            if (em.isOpen()) {
                if (em.getTransaction().isActive()){
                    em.getTransaction().rollback();
                }
                em.close();
            }
		} catch (ApplicationSettingsException e) {
			logger.error(e.getMessage(), e);
			throw new AppCatalogException(e);
		} finally {
			if (em != null && em.isOpen()) {
				if (em.getTransaction().isActive()) {
					em.getTransaction().rollback();
				}
				em.close();
			}
		}
	}
	
	@Override
	public AppCatalogResource get(Object identifier) throws AppCatalogException {
		EntityManager em = null;
		try {
			em = AppCatalogJPAUtils.getEntityManager();
			em.getTransaction().begin();
			AppCatalogQueryGenerator generator = new AppCatalogQueryGenerator(GRIDFTP_DATA_MOVEMENT);
			generator.setParameter(GridftpDataMovementConstants.DATA_MOVEMENT_INTERFACE_ID, identifier);
			Query q = generator.selectQuery(em);
			GridftpDataMovement gridftpDataMovement = (GridftpDataMovement) q.getSingleResult();
			GridftpDataMovementResource gridftpDataMovementResource = (GridftpDataMovementResource) AppCatalogJPAUtils.getResource(AppCatalogResourceType.GRIDFTP_DATA_MOVEMENT, gridftpDataMovement);
			em.getTransaction().commit();
            if (em.isOpen()) {
                if (em.getTransaction().isActive()){
                    em.getTransaction().rollback();
                }
                em.close();
            }
			return gridftpDataMovementResource;
		} catch (ApplicationSettingsException e) {
			logger.error(e.getMessage(), e);
			throw new AppCatalogException(e);
		} finally {
			if (em != null && em.isOpen()) {
				if (em.getTransaction().isActive()) {
					em.getTransaction().rollback();
				}
				em.close();
			}
		}
	}
	
	@Override
	public List<AppCatalogResource> get(String fieldName, Object value) throws AppCatalogException {
		List<AppCatalogResource> gridftpDataMovementResources = new ArrayList<AppCatalogResource>();
		EntityManager em = null;
		try {
			em = AppCatalogJPAUtils.getEntityManager();
			em.getTransaction().begin();
			AppCatalogQueryGenerator generator = new AppCatalogQueryGenerator(GRIDFTP_DATA_MOVEMENT);
			Query q;
			if ((fieldName.equals(GridftpDataMovementConstants.DATA_MOVEMENT_INTERFACE_ID)) || (fieldName.equals(GridftpDataMovementConstants.SECURITY_PROTOCOL))) {
				generator.setParameter(fieldName, value);
				q = generator.selectQuery(em);
				List<?> results = q.getResultList();
				for (Object result : results) {
					GridftpDataMovement gridftpDataMovement = (GridftpDataMovement) result;
					GridftpDataMovementResource gridftpDataMovementResource = (GridftpDataMovementResource) AppCatalogJPAUtils.getResource(AppCatalogResourceType.GRIDFTP_DATA_MOVEMENT, gridftpDataMovement);
					gridftpDataMovementResources.add(gridftpDataMovementResource);
				}
			} else {
				em.getTransaction().commit();
                if (em.isOpen()) {
                    if (em.getTransaction().isActive()){
                        em.getTransaction().rollback();
                    }
                    em.close();
                }
				logger.error("Unsupported field name for Gridftp Data Movement Resource.", new IllegalArgumentException());
				throw new IllegalArgumentException("Unsupported field name for Gridftp Data Movement Resource.");
			}
			em.getTransaction().commit();
            if (em.isOpen()) {
                if (em.getTransaction().isActive()){
                    em.getTransaction().rollback();
                }
                em.close();
            }
		} catch (ApplicationSettingsException e) {
			logger.error(e.getMessage(), e);
			throw new AppCatalogException(e);
		} finally {
			if (em != null && em.isOpen()) {
				if (em.getTransaction().isActive()) {
					em.getTransaction().rollback();
				}
				em.close();
			}
		}
		return gridftpDataMovementResources;
	}

    @Override
    public List<AppCatalogResource> getAll() throws AppCatalogException {
        return null;
    }

    @Override
    public List<String> getAllIds() throws AppCatalogException {
        return null;
    }

    @Override
	public List<String> getIds(String fieldName, Object value) throws AppCatalogException {
		List<String> gridftpDataMovementResourceIDs = new ArrayList<String>();
		EntityManager em = null;
		try {
			em = AppCatalogJPAUtils.getEntityManager();
			em.getTransaction().begin();
			AppCatalogQueryGenerator generator = new AppCatalogQueryGenerator(GRIDFTP_DATA_MOVEMENT);
			Query q;
			if ((fieldName.equals(GridftpDataMovementConstants.DATA_MOVEMENT_INTERFACE_ID)) || (fieldName.equals(GridftpDataMovementConstants.SECURITY_PROTOCOL))) {
				generator.setParameter(fieldName, value);
				q = generator.selectQuery(em);
				List<?> results = q.getResultList();
				for (Object result : results) {
					GridftpDataMovement gridftpDataMovement = (GridftpDataMovement) result;
					GridftpDataMovementResource gridftpDataMovementResource = (GridftpDataMovementResource) AppCatalogJPAUtils.getResource(AppCatalogResourceType.GRIDFTP_DATA_MOVEMENT, gridftpDataMovement);
					gridftpDataMovementResourceIDs.add(gridftpDataMovementResource.getDataMovementInterfaceId());
				}
			} else {
				em.getTransaction().commit();
                if (em.isOpen()) {
                    if (em.getTransaction().isActive()){
                        em.getTransaction().rollback();
                    }
                    em.close();
                }
				logger.error("Unsupported field name for Gridftp Data Movement Resource.", new IllegalArgumentException());
				throw new IllegalArgumentException("Unsupported field name for Gridftp Data Movement Resource.");
			}
			em.getTransaction().commit();
            if (em.isOpen()) {
                if (em.getTransaction().isActive()){
                    em.getTransaction().rollback();
                }
                em.close();
            }
		} catch (ApplicationSettingsException e) {
			logger.error(e.getMessage(), e);
			throw new AppCatalogException(e);
		} finally {
			if (em != null && em.isOpen()) {
				if (em.getTransaction().isActive()) {
					em.getTransaction().rollback();
				}
				em.close();
			}
		}
		return gridftpDataMovementResourceIDs;
	}
	
	@Override
	public void save() throws AppCatalogException {
		EntityManager em = null;
		try {
			em = AppCatalogJPAUtils.getEntityManager();
			GridftpDataMovement existingGridftpDataMovement = em.find(GridftpDataMovement.class, dataMovementInterfaceId);
            if (em.isOpen()) {
                if (em.getTransaction().isActive()){
                    em.getTransaction().rollback();
                }
                em.close();
            }

			GridftpDataMovement gridftpDataMovement;
			em = AppCatalogJPAUtils.getEntityManager();
			em.getTransaction().begin();
			if (existingGridftpDataMovement == null) {
				gridftpDataMovement = new GridftpDataMovement();
                gridftpDataMovement.setCreationTime(AiravataUtils.getCurrentTimestamp());
			} else {
				gridftpDataMovement = existingGridftpDataMovement;
                gridftpDataMovement.setUpdateTime(AiravataUtils.getCurrentTimestamp());
			}
			gridftpDataMovement.setDataMovementInterfaceId(getDataMovementInterfaceId());
			gridftpDataMovement.setSecurityProtocol(getSecurityProtocol());
			if (existingGridftpDataMovement == null) {
				em.persist(gridftpDataMovement);
			} else {
				em.merge(gridftpDataMovement);
			}
			em.getTransaction().commit();
            if (em.isOpen()) {
                if (em.getTransaction().isActive()){
                    em.getTransaction().rollback();
                }
                em.close();
            }
		} catch (Exception e) {
			logger.error(e.getMessage(), e);
			throw new AppCatalogException(e);
		} finally {
			if (em != null && em.isOpen()) {
				if (em.getTransaction().isActive()) {
					em.getTransaction().rollback();
				}
				em.close();
			}
		}
	}
	
	@Override
	public boolean isExists(Object identifier) throws AppCatalogException {
		EntityManager em = null;
		try {
			em = AppCatalogJPAUtils.getEntityManager();
			GridftpDataMovement gridftpDataMovement = em.find(GridftpDataMovement.class, identifier);
            if (em.isOpen()) {
                if (em.getTransaction().isActive()){
                    em.getTransaction().rollback();
                }
                em.close();
            }

			return gridftpDataMovement != null;
		} catch (ApplicationSettingsException e) {
			logger.error(e.getMessage(), e);
			throw new AppCatalogException(e);
		} finally {
			if (em != null && em.isOpen()) {
				if (em.getTransaction().isActive()) {
					em.getTransaction().rollback();
				}
				em.close();
			}
		}
	}
	
	public String getDataMovementInterfaceId() {
		return dataMovementInterfaceId;
	}
	
	public String getSecurityProtocol() {
		return securityProtocol;
	}
	
	public void setDataMovementInterfaceId(String dataMovementInterfaceId) {
		this.dataMovementInterfaceId=dataMovementInterfaceId;
	}
	
	public void setSecurityProtocol(String securityProtocol) {
		this.securityProtocol=securityProtocol;
	}
}