/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.camel.component.springrabbit.integration;

import java.util.concurrent.TimeUnit;

import org.apache.camel.Exchange;
import org.apache.camel.ExchangePattern;
import org.apache.camel.RoutesBuilder;
import org.apache.camel.builder.RouteBuilder;
import org.apache.camel.component.mock.MockEndpoint;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Test;

public class RabbitMQInOutIT extends RabbitMQITSupport {

    @Test
    public void testInOut() throws Exception {
        getMockEndpoint("mock:input").expectedBodiesReceived("World");
        getMockEndpoint("mock:result").expectedBodiesReceived("Hello World");

        Exchange out = template.request("direct:start", e -> e.getMessage().setBody("World"));
        Assertions.assertNotNull(out);
        Assertions.assertFalse(out.isFailed());
        Assertions.assertEquals("Hello World", out.getMessage().getBody());
        Assertions.assertNotNull(out.getMessage().getHeader(Exchange.BREADCRUMB_ID));

        Object crumb = out.getMessage().getHeader(Exchange.BREADCRUMB_ID);
        Object crumb2 = context.getVariable("global:mycrmb");
        Assertions.assertEquals(crumb, crumb2);

        MockEndpoint.assertIsSatisfied(context, 30, TimeUnit.SECONDS);
    }

    @Override
    protected RoutesBuilder createRouteBuilder() throws Exception {
        return new RouteBuilder() {
            @Override
            public void configure() throws Exception {
                context.setUseBreadcrumb(true);

                from("direct:start")
                        .setVariable("global:mycrmb", header(Exchange.BREADCRUMB_ID))
                        .to("log:request")
                        .to(ExchangePattern.InOut, "spring-rabbitmq:cheese?routingKey=foo.bar")
                        .to("log:response")
                        .to("mock:result");

                from("spring-rabbitmq:cheese?queues=myqueue&routingKey=foo.bar")
                        .to("log:input")
                        .to("mock:input")
                        .transform(body().prepend("Hello "));
            }
        };
    }
}
