/*
 * Decompiled with CFR 0.152.
 */
package com.datastax.driver.core;

import com.datastax.driver.core.Crc;
import com.datastax.driver.core.FrameCompressor;
import com.datastax.driver.core.ProtocolOptions;
import com.datastax.driver.core.Segment;
import com.datastax.driver.core.exceptions.CrcMismatchException;
import com.datastax.shaded.netty.buffer.ByteBuf;
import com.datastax.shaded.netty.buffer.ByteBufAllocator;
import com.google.common.annotations.VisibleForTesting;
import java.io.IOException;
import java.util.List;

class SegmentCodec {
    private static final int COMPRESSED_HEADER_LENGTH = 5;
    private static final int UNCOMPRESSED_HEADER_LENGTH = 3;
    static final int CRC24_LENGTH = 3;
    static final int CRC32_LENGTH = 4;
    private final ByteBufAllocator allocator;
    private final boolean compress;
    private final FrameCompressor compressor;

    SegmentCodec(ByteBufAllocator allocator, ProtocolOptions.Compression compression) {
        this.allocator = allocator;
        this.compress = compression != ProtocolOptions.Compression.NONE;
        this.compressor = compression.compressor();
    }

    int headerLength() {
        return this.compress ? 5 : 3;
    }

    void encode(Segment segment, List<Object> out) throws IOException {
        ByteBuf encodedPayload;
        ByteBuf uncompressedPayload = segment.getPayload();
        int uncompressedPayloadLength = uncompressedPayload.readableBytes();
        assert (uncompressedPayloadLength <= Segment.MAX_PAYLOAD_LENGTH);
        if (this.compress) {
            uncompressedPayload.markReaderIndex();
            ByteBuf compressedPayload = this.compressor.compress(uncompressedPayload);
            if (compressedPayload.readableBytes() >= uncompressedPayloadLength) {
                uncompressedPayload.resetReaderIndex();
                encodedPayload = uncompressedPayload;
                compressedPayload.release();
                uncompressedPayloadLength = 0;
            } else {
                encodedPayload = compressedPayload;
                uncompressedPayload.release();
            }
        } else {
            encodedPayload = uncompressedPayload;
        }
        int payloadLength = encodedPayload.readableBytes();
        ByteBuf header = this.encodeHeader(payloadLength, uncompressedPayloadLength, segment.isSelfContained());
        int payloadCrc = Crc.computeCrc32(encodedPayload);
        ByteBuf trailer = this.allocator.ioBuffer(4);
        for (int i = 0; i < 4; ++i) {
            trailer.writeByte(payloadCrc & 0xFF);
            payloadCrc >>= 8;
        }
        out.add(header);
        out.add(encodedPayload);
        out.add(trailer);
    }

    @VisibleForTesting
    ByteBuf encodeHeader(int payloadLength, int uncompressedLength, boolean isSelfContained) {
        int shift;
        int i;
        assert (payloadLength <= Segment.MAX_PAYLOAD_LENGTH);
        assert (!this.compress || uncompressedLength <= Segment.MAX_PAYLOAD_LENGTH);
        int headerLength = this.headerLength();
        long headerData = payloadLength;
        int flagOffset = 17;
        if (this.compress) {
            headerData |= (long)uncompressedLength << 17;
            flagOffset += 17;
        }
        if (isSelfContained) {
            headerData |= 1L << flagOffset;
        }
        int headerCrc = Crc.computeCrc24(headerData, headerLength);
        ByteBuf header = this.allocator.ioBuffer(headerLength + 3);
        for (i = 0; i < headerLength; ++i) {
            shift = i * 8;
            header.writeByte((int)(headerData >> shift & 0xFFL));
        }
        for (i = 0; i < 3; ++i) {
            shift = i * 8;
            header.writeByte(headerCrc >> shift & 0xFF);
        }
        return header;
    }

    Header decodeHeader(ByteBuf buffer) throws CrcMismatchException {
        int uncompressedPayloadLength;
        int headerLength = this.headerLength();
        assert (buffer.readableBytes() >= headerLength + 3);
        long headerData = 0L;
        for (int i = 0; i < headerLength; ++i) {
            headerData |= ((long)buffer.readByte() & 0xFFL) << 8 * i;
        }
        int expectedHeaderCrc = 0;
        for (int i = 0; i < 3; ++i) {
            expectedHeaderCrc |= (buffer.readByte() & 0xFF) << 8 * i;
        }
        int actualHeaderCrc = Crc.computeCrc24(headerData, headerLength);
        if (actualHeaderCrc != expectedHeaderCrc) {
            throw new CrcMismatchException(String.format("CRC mismatch on header %s. Received %s, computed %s.", Long.toHexString(headerData), Integer.toHexString(expectedHeaderCrc), Integer.toHexString(actualHeaderCrc)));
        }
        int payloadLength = (int)headerData & Segment.MAX_PAYLOAD_LENGTH;
        headerData >>= 17;
        if (this.compress) {
            uncompressedPayloadLength = (int)headerData & Segment.MAX_PAYLOAD_LENGTH;
            headerData >>= 17;
        } else {
            uncompressedPayloadLength = -1;
        }
        boolean isSelfContained = (headerData & 1L) == 1L;
        return new Header(payloadLength, uncompressedPayloadLength, isSelfContained);
    }

    Segment decode(Header header, ByteBuf buffer) throws CrcMismatchException, IOException {
        ByteBuf payload;
        assert (buffer.readableBytes() == header.payloadLength + 4);
        ByteBuf encodedPayload = buffer.readSlice(header.payloadLength);
        encodedPayload.retain();
        int expectedPayloadCrc = 0;
        for (int i = 0; i < 4; ++i) {
            expectedPayloadCrc |= (buffer.readByte() & 0xFF) << 8 * i;
        }
        buffer.release();
        int actualPayloadCrc = Crc.computeCrc32(encodedPayload);
        if (actualPayloadCrc != expectedPayloadCrc) {
            encodedPayload.release();
            throw new CrcMismatchException(String.format("CRC mismatch on payload. Received %s, computed %s.", Integer.toHexString(expectedPayloadCrc), Integer.toHexString(actualPayloadCrc)));
        }
        if (this.compress && header.uncompressedPayloadLength > 0) {
            payload = this.compressor.decompress(encodedPayload, header.uncompressedPayloadLength);
            encodedPayload.release();
        } else {
            payload = encodedPayload;
        }
        return new Segment(payload, header.isSelfContained);
    }

    static class Header {
        final int payloadLength;
        final int uncompressedPayloadLength;
        final boolean isSelfContained;

        public Header(int payloadLength, int uncompressedPayloadLength, boolean isSelfContained) {
            this.payloadLength = payloadLength;
            this.uncompressedPayloadLength = uncompressedPayloadLength;
            this.isSelfContained = isSelfContained;
        }
    }
}

