/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.sidecar.restore;

import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;
import io.vertx.core.Future;
import java.io.File;
import java.io.IOException;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Comparator;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.stream.Stream;
import org.apache.cassandra.sidecar.cluster.instance.InstanceMetadata;
import org.apache.cassandra.sidecar.common.server.cluster.locator.TokenRange;
import org.apache.cassandra.sidecar.common.server.utils.ThrowableUtils;
import org.apache.cassandra.sidecar.concurrent.ExecutorPools;
import org.apache.cassandra.sidecar.config.RestoreJobConfiguration;
import org.apache.cassandra.sidecar.db.RestoreJob;
import org.apache.cassandra.sidecar.db.RestoreRange;
import org.apache.cassandra.sidecar.exceptions.RestoreJobFatalException;
import org.apache.cassandra.sidecar.restore.RestoreJobProgressTracker;
import org.apache.cassandra.sidecar.restore.RestoreJobUtil;
import org.apache.cassandra.sidecar.restore.RestoreProcessor;
import org.jetbrains.annotations.VisibleForTesting;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RestoreJobManager {
    private static final Logger LOGGER = LoggerFactory.getLogger(RestoreJobManager.class);
    private static final Object PRESENT = new Object();
    private final Map<UUID, RestoreJobProgressTracker> jobs = new ConcurrentHashMap<UUID, RestoreJobProgressTracker>();
    private final Cache<UUID, Object> deletedJobs;
    private final RestoreProcessor processor;
    private final ExecutorPools executorPools;
    private final InstanceMetadata instanceMetadata;
    private final RestoreJobConfiguration restoreJobConfig;

    public RestoreJobManager(RestoreJobConfiguration restoreJobConfig, InstanceMetadata instanceMetadata, ExecutorPools executorPools, RestoreProcessor restoreProcessor) {
        this(restoreJobConfig, instanceMetadata, executorPools, restoreProcessor, true);
    }

    @VisibleForTesting
    public RestoreJobManager(RestoreJobConfiguration restoreJobConfig, InstanceMetadata instanceMetadata, ExecutorPools executorPools, RestoreProcessor restoreProcessor, boolean deleteOnStart) {
        this.restoreJobConfig = restoreJobConfig;
        this.instanceMetadata = instanceMetadata;
        this.executorPools = executorPools;
        this.processor = restoreProcessor;
        this.deletedJobs = Caffeine.newBuilder().expireAfterAccess(1L, TimeUnit.DAYS).build();
        if (deleteOnStart) {
            this.deleteObsoleteDataAsync();
        }
    }

    public RestoreJobProgressTracker.Status trySubmit(RestoreRange range, RestoreJob restoreJob) throws RestoreJobFatalException {
        RestoreJobProgressTracker tracker = this.progressTracker(restoreJob);
        return tracker.trySubmit(range);
    }

    void updateRestoreJob(RestoreJob restoreJob) {
        RestoreJobProgressTracker tracker = this.progressTracker(restoreJob);
        tracker.updateRestoreJob(restoreJob);
    }

    Set<RestoreRange> discardOverlappingRanges(RestoreJob restoreJob, Set<TokenRange> otherRanges) {
        RestoreJobProgressTracker tracker = this.progressTracker(restoreJob);
        return tracker.discardOverlappingRanges(otherRanges);
    }

    void removeJobInternal(UUID jobId) {
        if (this.deletedJobs.getIfPresent((Object)jobId) == PRESENT) {
            LOGGER.debug("The job is already removed. Skipping. jobId={}", (Object)jobId);
            return;
        }
        this.executorPools.internal().runBlocking(() -> {
            RestoreJobProgressTracker tracker = this.jobs.remove(jobId);
            if (tracker != null) {
                tracker.cleanupInternal();
            }
        }).recover(cause -> {
            LOGGER.warn("Failed to clean up restore job. Recover and proceed to delete the on-disk files. jobId={}", (Object)jobId, cause);
            return Future.succeededFuture();
        }).compose(v -> this.deleteDataOfJobAsync(jobId)).onSuccess(v -> this.deletedJobs.put((Object)jobId, PRESENT));
    }

    void deleteObsoleteDataAsync() {
        this.findObsoleteJobDataDirs().compose(pathStream -> this.executorPools.internal().runBlocking(() -> {
            try (Stream stream = pathStream;){
                stream.forEach(this::deleteRecursively);
            }
        })).onFailure(cause -> LOGGER.warn("Unexpected error while deleting files.", cause));
    }

    Future<Stream<Path>> findObsoleteJobDataDirs() {
        Path rootDir = Paths.get(this.instanceMetadata.stagingDir(), new String[0]);
        if (!Files.exists(rootDir, new LinkOption[0])) {
            return Future.succeededFuture(Stream.empty());
        }
        return this.executorPools.internal().executeBlocking(() -> Files.walk(rootDir, 1, new FileVisitOption[0]).filter(this::isObsoleteRestoreJobDir));
    }

    private RestoreJobProgressTracker progressTracker(RestoreJob restoreJob) {
        return this.jobs.computeIfAbsent(restoreJob.jobId, id -> new RestoreJobProgressTracker(restoreJob, this.processor, this.instanceMetadata));
    }

    private Future<Void> deleteDataOfJobAsync(UUID jobId) {
        Path stagingDir = Paths.get(this.instanceMetadata.stagingDir(), new String[0]);
        if (!Files.exists(stagingDir, new LinkOption[0])) {
            return Future.succeededFuture();
        }
        String prefixedJobId = RestoreJobUtil.prefixedJobId(jobId);
        return this.executorPools.internal().runBlocking(() -> {
            try (Stream<Path> rootDirs = Files.walk(stagingDir, 1, new FileVisitOption[0]);){
                rootDirs.filter(path -> Files.isDirectory(path, new LinkOption[0]) && path.startsWith(prefixedJobId)).forEach(this::deleteRecursively);
            }
            catch (IOException ioe) {
                LOGGER.warn("Error on listing staged restore job directories. Path={}", (Object)stagingDir, (Object)ioe);
            }
        });
    }

    private void deleteRecursively(Path root) {
        try (Stream<Path> pathStream = Files.walk(root, new FileVisitOption[0]);){
            pathStream.sorted(Comparator.reverseOrder()).forEach(path -> ThrowableUtils.propagate(() -> Files.delete(path)));
        }
        catch (Exception exception) {
            LOGGER.warn("Error on deleting data. Path={}", (Object)root, (Object)exception);
        }
    }

    boolean isObsoleteRestoreJobDir(Path path) {
        long gapInMillis;
        File file = path.toFile();
        if (!file.isDirectory()) {
            return false;
        }
        long originTs = RestoreJobUtil.timestampFromRestoreJobDir(file.getName());
        if (originTs == -1L) {
            return false;
        }
        long delta = System.currentTimeMillis() - originTs;
        return delta > (gapInMillis = TimeUnit.DAYS.toMillis(this.restoreJobConfig.jobDiscoveryMinimumRecencyDays()));
    }

    @VisibleForTesting
    RestoreJobProgressTracker progressTrackerUnsafe(RestoreJob restoreJob) {
        return this.progressTracker(restoreJob);
    }
}

