/*
 * Decompiled with CFR 0.152.
 */
package org.apache.gobblin.source.extractor.watermark;

import com.google.common.base.Preconditions;
import com.google.common.collect.Maps;
import com.google.common.math.DoubleMath;
import com.google.common.primitives.Ints;
import java.math.RoundingMode;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;
import org.apache.gobblin.source.extractor.extract.QueryBasedExtractor;
import org.apache.gobblin.source.extractor.utils.Utils;
import org.apache.gobblin.source.extractor.watermark.Watermark;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class HourWatermark
implements Watermark {
    private static final Logger LOG = LoggerFactory.getLogger(HourWatermark.class);
    private static final String INPUTFORMAT = "yyyyMMddHHmmss";
    private static final String OUTPUTFORMAT = "yyyyMMddHH";
    private static final int deltaForNextWatermark = 3600;
    private final SimpleDateFormat inputFormatParser;
    private String watermarkColumn;
    private String watermarkFormat;

    public HourWatermark(String watermarkColumn, String watermarkFormat) {
        this.watermarkColumn = watermarkColumn;
        this.watermarkFormat = watermarkFormat;
        this.inputFormatParser = new SimpleDateFormat(INPUTFORMAT);
    }

    @Override
    public String getWatermarkCondition(QueryBasedExtractor<?, ?> extractor, long watermarkValue, String operator) {
        return extractor.getHourPredicateCondition(this.watermarkColumn, watermarkValue, this.watermarkFormat, operator);
    }

    @Override
    public int getDeltaNumForNextWatermark() {
        return 3600;
    }

    @Override
    public synchronized HashMap<Long, Long> getIntervals(long lowWatermarkValue, long highWatermarkValue, long partitionIntervalInHours, int maxIntervals) {
        Preconditions.checkArgument((maxIntervals > 0 ? 1 : 0) != 0, (Object)"Invalid value for maxIntervals, positive value expected.");
        Preconditions.checkArgument((partitionIntervalInHours > 0L ? 1 : 0) != 0, (Object)"Invalid value for partitionInterval, should be at least 1.");
        HashMap intervalMap = Maps.newHashMap();
        if (lowWatermarkValue > highWatermarkValue) {
            LOG.warn("The low water mark is greater than the high water mark, empty intervals are returned");
            return intervalMap;
        }
        Calendar calendar = Calendar.getInstance();
        Date lowWatermarkDate = HourWatermark.extractFromTimestamp(Long.toString(lowWatermarkValue));
        Date highWatermarkDate = HourWatermark.extractFromTimestamp(Long.toString(highWatermarkValue));
        long lowWatermark = lowWatermarkDate.getTime();
        long highWatermark = highWatermarkDate.getTime();
        int interval = HourWatermark.getInterval(highWatermark - lowWatermark, partitionIntervalInHours, maxIntervals);
        LOG.info("Recalculated partition interval:" + interval + " hours");
        Date startTime = new Date(lowWatermark);
        Date endTime = new Date(highWatermark);
        LOG.debug("Start time:" + startTime + "; End time:" + endTime);
        while (startTime.getTime() < endTime.getTime()) {
            long lwm = Long.parseLong(this.inputFormatParser.format(startTime));
            calendar.setTime(startTime);
            calendar.add(10, interval);
            Date nextTime = calendar.getTime();
            long hwm = Long.parseLong(this.inputFormatParser.format(nextTime.getTime() <= endTime.getTime() ? nextTime : endTime));
            intervalMap.put(lwm, hwm);
            LOG.debug("Partition - low:" + lwm + "; high:" + hwm);
            startTime = nextTime;
        }
        return intervalMap;
    }

    private static int getInterval(long diffInMilliSecs, long hourInterval, int maxIntervals) {
        int totalHours = DoubleMath.roundToInt((double)((double)diffInMilliSecs / 3600000.0), (RoundingMode)RoundingMode.CEILING);
        int totalIntervals = DoubleMath.roundToInt((double)((double)totalHours / (double)hourInterval), (RoundingMode)RoundingMode.CEILING);
        if (totalIntervals > maxIntervals) {
            hourInterval = DoubleMath.roundToInt((double)((double)totalHours / (double)maxIntervals), (RoundingMode)RoundingMode.CEILING);
        }
        return Ints.checkedCast((long)hourInterval);
    }

    private static synchronized Date extractFromTimestamp(String watermark) {
        SimpleDateFormat inputFormat = new SimpleDateFormat(INPUTFORMAT);
        SimpleDateFormat outputFormat = new SimpleDateFormat(OUTPUTFORMAT);
        Date outDate = null;
        try {
            Date date = inputFormat.parse(watermark);
            String dateStr = outputFormat.format(date);
            outDate = outputFormat.parse(dateStr);
        }
        catch (ParseException e) {
            LOG.error(e.getMessage(), (Throwable)e);
        }
        return outDate;
    }

    public static long adjustWatermark(String baseWatermark, int diff) {
        SimpleDateFormat parser = new SimpleDateFormat(INPUTFORMAT);
        Date date = Utils.toDate(baseWatermark, INPUTFORMAT, OUTPUTFORMAT);
        return Long.parseLong(parser.format(Utils.addHoursToDate(date, diff)));
    }
}

