/*
 * Decompiled with CFR 0.152.
 */
package org.apache.ignite.internal.processors.cache.persistence.checkpoint;

import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.file.CopyOption;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.function.Function;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.ignite.IgniteCheckedException;
import org.apache.ignite.IgniteLogger;
import org.apache.ignite.internal.pagemem.wal.record.CacheState;
import org.apache.ignite.internal.pagemem.wal.record.CheckpointRecord;
import org.apache.ignite.internal.processors.cache.persistence.StorageException;
import org.apache.ignite.internal.processors.cache.persistence.checkpoint.Checkpoint;
import org.apache.ignite.internal.processors.cache.persistence.checkpoint.CheckpointEntry;
import org.apache.ignite.internal.processors.cache.persistence.checkpoint.CheckpointEntryType;
import org.apache.ignite.internal.processors.cache.persistence.checkpoint.CheckpointHistory;
import org.apache.ignite.internal.processors.cache.persistence.checkpoint.CheckpointStatus;
import org.apache.ignite.internal.processors.cache.persistence.file.FileIO;
import org.apache.ignite.internal.processors.cache.persistence.file.FileIOFactory;
import org.apache.ignite.internal.processors.cache.persistence.file.FilePageStoreManager;
import org.apache.ignite.internal.processors.cache.persistence.wal.WALPointer;
import org.apache.ignite.internal.util.typedef.internal.U;
import org.jetbrains.annotations.Nullable;

public class CheckpointMarkersStorage {
    public static final Pattern CP_FILE_NAME_PATTERN = Pattern.compile("(\\d+)-(.*)-(START|END)\\.bin");
    protected IgniteLogger log;
    private CheckpointHistory cpHistory;
    private final FileIOFactory ioFactory;
    public final File cpDir;
    private final ByteBuffer tmpWriteBuf;

    CheckpointMarkersStorage(Function<Class<?>, IgniteLogger> logger, CheckpointHistory history, FileIOFactory factory, String absoluteWorkDir) throws IgniteCheckedException {
        this.log = logger.apply(this.getClass());
        this.cpHistory = history;
        this.ioFactory = factory;
        this.cpDir = Paths.get(absoluteWorkDir, "cp").toFile();
        if (!U.mkdirs(this.cpDir)) {
            throw new IgniteCheckedException("Could not create directory for checkpoint metadata: " + this.cpDir);
        }
        this.tmpWriteBuf = ByteBuffer.allocateDirect(16);
        this.tmpWriteBuf.order(ByteOrder.nativeOrder());
    }

    public void cleanupTempCheckpointDirectory() throws IgniteCheckedException {
        try (DirectoryStream<Path> files = Files.newDirectoryStream(this.cpDir.toPath(), FilePageStoreManager.TMP_FILE_MATCHER::matches);){
            for (Path path : files) {
                Files.delete(path);
            }
        }
        catch (IOException e) {
            throw new IgniteCheckedException("Failed to cleanup checkpoint directory from temporary files: " + this.cpDir, e);
        }
    }

    public void cleanupCheckpointDirectory() throws IgniteCheckedException {
        if (this.cpHistory != null) {
            this.cpHistory.clear();
        }
        try (DirectoryStream<Path> files = Files.newDirectoryStream(this.cpDir.toPath());){
            for (Path path : files) {
                Files.delete(path);
            }
        }
        catch (IOException e) {
            throw new IgniteCheckedException("Failed to cleanup checkpoint directory: " + this.cpDir, e);
        }
    }

    public void initialize() throws IgniteCheckedException {
        this.cpHistory.initialize(this.retrieveHistory());
    }

    public void removeCheckpointsUntil(@Nullable WALPointer highBound) throws IgniteCheckedException {
        List<CheckpointEntry> rmvFromHist = this.history().onWalTruncated(highBound);
        for (CheckpointEntry cp : rmvFromHist) {
            this.removeCheckpointFiles(cp);
        }
    }

    public void onCheckpointFinished(Checkpoint chp) throws IgniteCheckedException {
        List<CheckpointEntry> rmvFromHist = this.history().onCheckpointFinished(chp);
        for (CheckpointEntry cp : rmvFromHist) {
            this.removeCheckpointFiles(cp);
        }
    }

    public CheckpointStatus readCheckpointStatus() throws IgniteCheckedException {
        File[] files;
        long lastStartTs = 0L;
        long lastEndTs = 0L;
        UUID startId = CheckpointStatus.NULL_UUID;
        UUID endId = CheckpointStatus.NULL_UUID;
        File startFile = null;
        File endFile = null;
        WALPointer startPtr = CheckpointStatus.NULL_PTR;
        WALPointer endPtr = CheckpointStatus.NULL_PTR;
        File dir = this.cpDir;
        if (!dir.exists()) {
            this.log.warning("Read checkpoint status: checkpoint directory is not found.");
            return new CheckpointStatus(0L, startId, startPtr, endId, endPtr);
        }
        for (File file : files = dir.listFiles()) {
            Matcher matcher = CP_FILE_NAME_PATTERN.matcher(file.getName());
            if (!matcher.matches()) continue;
            long ts = Long.parseLong(matcher.group(1));
            UUID id = UUID.fromString(matcher.group(2));
            CheckpointEntryType type = CheckpointEntryType.valueOf(matcher.group(3));
            if (type == CheckpointEntryType.START && ts > lastStartTs) {
                lastStartTs = ts;
                startId = id;
                startFile = file;
                continue;
            }
            if (type != CheckpointEntryType.END || ts <= lastEndTs) continue;
            lastEndTs = ts;
            endId = id;
            endFile = file;
        }
        ByteBuffer buf = ByteBuffer.allocate(16);
        buf.order(ByteOrder.nativeOrder());
        if (startFile != null) {
            startPtr = this.readPointer(startFile, buf);
        }
        if (endFile != null) {
            endPtr = this.readPointer(endFile, buf);
        }
        if (this.log.isInfoEnabled()) {
            this.log.info("Read checkpoint status [startMarker=" + startFile + ", endMarker=" + endFile + ']');
        }
        return new CheckpointStatus(lastStartTs, startId, startPtr, endId, endPtr);
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private List<CheckpointEntry> retrieveHistory() throws IgniteCheckedException {
        if (!this.cpDir.exists()) {
            return Collections.emptyList();
        }
        try (DirectoryStream<Path> cpFiles = Files.newDirectoryStream(this.cpDir.toPath(), path -> CP_FILE_NAME_PATTERN.matcher(path.toFile().getName()).matches());){
            ArrayList<CheckpointEntry> checkpoints = new ArrayList<CheckpointEntry>();
            ByteBuffer buf = ByteBuffer.allocate(16);
            buf.order(ByteOrder.nativeOrder());
            for (Path cpFile : cpFiles) {
                CheckpointEntry cp = this.parseFromFile(buf, cpFile.toFile());
                if (cp == null) continue;
                checkpoints.add(cp);
            }
            ArrayList<CheckpointEntry> arrayList = checkpoints;
            return arrayList;
        }
        catch (IOException e) {
            throw new IgniteCheckedException("Failed to load checkpoint history.", e);
        }
    }

    @Nullable
    private CheckpointEntry parseFromFile(ByteBuffer buf, File file) throws IgniteCheckedException {
        Matcher matcher = CP_FILE_NAME_PATTERN.matcher(file.getName());
        if (!matcher.matches()) {
            return null;
        }
        CheckpointEntryType type = CheckpointEntryType.valueOf(matcher.group(3));
        if (type != CheckpointEntryType.START) {
            return null;
        }
        long cpTs = Long.parseLong(matcher.group(1));
        UUID cpId = UUID.fromString(matcher.group(2));
        WALPointer ptr = this.readPointer(file, buf);
        return this.createCheckPointEntry(cpTs, ptr, cpId, null, CheckpointEntryType.START);
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private WALPointer readPointer(File cpMarkerFile, ByteBuffer buf) throws IgniteCheckedException {
        buf.position(0);
        try (FileIO io = this.ioFactory.create(cpMarkerFile, StandardOpenOption.READ);){
            io.readFully(buf);
            buf.flip();
            WALPointer wALPointer = new WALPointer(buf.getLong(), buf.getInt(), buf.getInt());
            return wALPointer;
        }
        catch (IOException e) {
            throw new IgniteCheckedException("Failed to read checkpoint pointer from marker file: " + cpMarkerFile.getAbsolutePath(), e);
        }
    }

    private CheckpointEntry createCheckPointEntry(long cpTs, WALPointer ptr, UUID cpId, @Nullable CheckpointRecord rec, CheckpointEntryType type) {
        assert (cpTs > 0L);
        assert (ptr != null);
        assert (cpId != null);
        assert (type != null);
        Map<Integer, CacheState> cacheGrpStates = null;
        if (rec != null) {
            cacheGrpStates = rec.cacheGroupStates();
        }
        return new CheckpointEntry(cpTs, ptr, cpId, cacheGrpStates);
    }

    private void removeCheckpointFiles(CheckpointEntry cpEntry) throws IgniteCheckedException {
        Path startFile = new File(this.cpDir.getAbsolutePath(), CheckpointMarkersStorage.checkpointFileName(cpEntry, CheckpointEntryType.START)).toPath();
        Path endFile = new File(this.cpDir.getAbsolutePath(), CheckpointMarkersStorage.checkpointFileName(cpEntry, CheckpointEntryType.END)).toPath();
        try {
            if (Files.exists(startFile, new LinkOption[0])) {
                Files.delete(startFile);
            }
            if (Files.exists(endFile, new LinkOption[0])) {
                Files.delete(endFile);
            }
        }
        catch (IOException e) {
            throw new StorageException("Failed to delete stale checkpoint files: " + cpEntry, e);
        }
    }

    private void writeCheckpointEntry(ByteBuffer entryBuf, CheckpointEntry cp, CheckpointEntryType type, boolean skipSync) throws StorageException {
        String fileName = CheckpointMarkersStorage.checkpointFileName(cp, type);
        String tmpFileName = fileName + ".tmp";
        try {
            try (FileIO io = this.ioFactory.create(Paths.get(this.cpDir.getAbsolutePath(), skipSync ? fileName : tmpFileName).toFile(), StandardOpenOption.CREATE_NEW, StandardOpenOption.WRITE);){
                io.writeFully(entryBuf);
                entryBuf.clear();
                if (!skipSync) {
                    io.force(true);
                }
            }
            if (!skipSync) {
                Files.move(Paths.get(this.cpDir.getAbsolutePath(), tmpFileName), Paths.get(this.cpDir.getAbsolutePath(), fileName), new CopyOption[0]);
            }
        }
        catch (IOException e) {
            throw new StorageException("Failed to write checkpoint entry [ptr=" + cp.checkpointMark() + ", cpTs=" + cp.timestamp() + ", cpId=" + cp.checkpointId() + ", type=" + (Object)((Object)type) + "]", e);
        }
    }

    public CheckpointEntry writeCheckpointEntry(long cpTs, UUID cpId, WALPointer ptr, @Nullable CheckpointRecord rec, CheckpointEntryType type, boolean skipSync) throws StorageException {
        CheckpointEntry entry = this.prepareCheckpointEntry(this.tmpWriteBuf, cpTs, cpId, ptr, rec, type);
        if (type == CheckpointEntryType.START) {
            this.cpHistory.addCheckpoint(entry, rec.cacheGroupStates());
        }
        this.writeCheckpointEntry(this.tmpWriteBuf, entry, type, skipSync);
        return entry;
    }

    private CheckpointEntry prepareCheckpointEntry(ByteBuffer entryBuf, long cpTs, UUID cpId, WALPointer ptr, @Nullable CheckpointRecord rec, CheckpointEntryType type) {
        assert (ptr != null);
        entryBuf.rewind();
        entryBuf.putLong(ptr.index());
        entryBuf.putInt(ptr.fileOffset());
        entryBuf.putInt(ptr.length());
        entryBuf.flip();
        return this.createCheckPointEntry(cpTs, ptr, cpId, rec, type);
    }

    private static String checkpointFileName(long cpTs, UUID cpId, CheckpointEntryType type) {
        return cpTs + "-" + cpId + "-" + (Object)((Object)type) + ".bin";
    }

    public static String checkpointFileName(CheckpointEntry cp, CheckpointEntryType type) {
        return CheckpointMarkersStorage.checkpointFileName(cp.timestamp(), cp.checkpointId(), type);
    }

    public CheckpointHistory history() {
        return this.cpHistory;
    }
}

