/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hugegraph.computer.core.sender;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;
import org.apache.hugegraph.computer.core.common.ComputerContext;
import org.apache.hugegraph.computer.core.common.exception.ComputerException;
import org.apache.hugegraph.computer.core.config.ComputerOptions;
import org.apache.hugegraph.computer.core.config.Config;
import org.apache.hugegraph.computer.core.graph.id.Id;
import org.apache.hugegraph.computer.core.graph.partition.Partitioner;
import org.apache.hugegraph.computer.core.graph.value.Value;
import org.apache.hugegraph.computer.core.graph.vertex.Vertex;
import org.apache.hugegraph.computer.core.manager.Manager;
import org.apache.hugegraph.computer.core.network.TransportConf;
import org.apache.hugegraph.computer.core.network.message.MessageType;
import org.apache.hugegraph.computer.core.receiver.MessageStat;
import org.apache.hugegraph.computer.core.sender.MessageSendBuffers;
import org.apache.hugegraph.computer.core.sender.MessageSendPartition;
import org.apache.hugegraph.computer.core.sender.MessageSender;
import org.apache.hugegraph.computer.core.sender.QueuedMessage;
import org.apache.hugegraph.computer.core.sender.WriteBuffers;
import org.apache.hugegraph.computer.core.sort.sorting.SortManager;
import org.apache.hugegraph.util.Log;
import org.slf4j.Logger;

public class MessageSendManager
implements Manager {
    public static final Logger LOG = Log.logger(MessageSendManager.class);
    public static final String NAME = "message_send";
    private final MessageSendBuffers buffers;
    private final Partitioner partitioner;
    private final SortManager sortManager;
    private final MessageSender sender;
    private final AtomicReference<Throwable> exception;
    private final TransportConf transportConf;

    public MessageSendManager(ComputerContext context, SortManager sortManager, MessageSender sender) {
        this.buffers = new MessageSendBuffers(context);
        this.partitioner = (Partitioner)context.config().createObject(ComputerOptions.WORKER_PARTITIONER);
        this.transportConf = TransportConf.wrapConfig(context.config());
        this.sortManager = sortManager;
        this.sender = sender;
        this.exception = new AtomicReference();
    }

    @Override
    public String name() {
        return NAME;
    }

    @Override
    public void init(Config config) {
        this.partitioner.init(config);
    }

    @Override
    public void close(Config config) {
    }

    public void sendVertex(Vertex vertex) {
        this.checkException();
        int partitionId = this.partitioner.partitionId(vertex.id());
        WriteBuffers buffer = this.buffers.get(partitionId);
        this.sortIfTargetBufferIsFull(buffer, partitionId, MessageType.VERTEX);
        try {
            buffer.writeVertex(vertex);
        }
        catch (IOException e) {
            throw new ComputerException("Failed to write vertex '%s'", (Throwable)e, new Object[]{vertex.id()});
        }
    }

    public void sendEdge(Vertex vertex) {
        this.checkException();
        int partitionId = this.partitioner.partitionId(vertex.id());
        WriteBuffers buffer = this.buffers.get(partitionId);
        this.sortIfTargetBufferIsFull(buffer, partitionId, MessageType.EDGE);
        try {
            buffer.writeEdges(vertex);
        }
        catch (IOException e) {
            throw new ComputerException("Failed to write edges of vertex '%s'", (Throwable)e, new Object[]{vertex.id()});
        }
    }

    public void sendMessage(Id targetId, Value value) {
        this.checkException();
        int partitionId = this.partitioner.partitionId(targetId);
        WriteBuffers buffer = this.buffers.get(partitionId);
        this.sortIfTargetBufferIsFull(buffer, partitionId, MessageType.MSG);
        try {
            buffer.writeMessage(targetId, value);
        }
        catch (IOException e) {
            throw new ComputerException("Failed to write message", (Throwable)e);
        }
    }

    public void startSend(MessageType type) {
        Map<Integer, MessageSendPartition> all = this.buffers.all();
        all.values().forEach(MessageSendPartition::resetMessageWritten);
        Set<Integer> workerIds = all.keySet().stream().map(this.partitioner::workerId).collect(Collectors.toSet());
        this.sendControlMessageToWorkers(workerIds, MessageType.START);
        LOG.info("Start sending message(type={})", (Object)type);
    }

    public void finishSend(MessageType type) {
        Map<Integer, MessageSendPartition> all = this.buffers.all();
        MessageStat stat = this.sortAndSendLastBuffer(all, type);
        Set<Integer> workerIds = all.keySet().stream().map(this.partitioner::workerId).collect(Collectors.toSet());
        this.sendControlMessageToWorkers(workerIds, MessageType.FINISH);
        LOG.info("Finish sending message(type={},count={},bytes={})", new Object[]{type, stat.messageCount(), stat.messageBytes()});
    }

    public MessageStat messageStat(int partitionId) {
        return this.buffers.get(partitionId).messageWritten();
    }

    public void clearBuffer() {
        this.buffers.clear();
    }

    private void sortIfTargetBufferIsFull(WriteBuffers buffer, int partitionId, MessageType type) {
        if (buffer.reachThreshold()) {
            buffer.switchForSorting();
            this.sortThenSend(partitionId, type, buffer);
        }
    }

    private Future<?> sortThenSend(int partitionId, MessageType type, WriteBuffers buffer) {
        int workerId = this.partitioner.workerId(partitionId);
        return ((CompletableFuture)this.sortManager.sort(type, buffer).thenAccept(sortedBuffer -> {
            buffer.finishSorting();
            QueuedMessage message = new QueuedMessage(partitionId, type, (ByteBuffer)sortedBuffer);
            try {
                this.sender.send(workerId, message);
            }
            catch (InterruptedException e) {
                throw new ComputerException("Interrupted when waiting to put buffer into queue");
            }
        })).whenComplete((r, e) -> {
            if (e != null) {
                LOG.error("Failed to sort buffer or put sorted buffer into queue", e);
                this.exception.compareAndSet((Throwable)null, (Throwable)e);
            }
        });
    }

    private MessageStat sortAndSendLastBuffer(Map<Integer, MessageSendPartition> all, MessageType type) {
        MessageStat messageWritten = new MessageStat();
        ArrayList futures = new ArrayList(all.size());
        for (Map.Entry<Integer, MessageSendPartition> entry : all.entrySet()) {
            int partitionId = entry.getKey();
            MessageSendPartition partition = entry.getValue();
            for (WriteBuffers buffer : partition.buffers()) {
                if (buffer.isEmpty()) continue;
                buffer.prepareSorting();
                futures.add(this.sortThenSend(partitionId, type, buffer));
            }
            messageWritten.increase(partition.messageWritten());
        }
        this.checkException();
        try {
            for (Future future : futures) {
                future.get(300L, TimeUnit.SECONDS);
            }
        }
        catch (TimeoutException e) {
            throw new ComputerException("Timed out to wait for sorting task to finished", (Throwable)e);
        }
        catch (InterruptedException | ExecutionException e) {
            throw new ComputerException("Failed to wait for sorting task to finished", (Throwable)e);
        }
        return messageWritten;
    }

    private void sendControlMessageToWorkers(Set<Integer> workerIds, MessageType type) {
        ArrayList<CompletableFuture<Void>> futures = new ArrayList<CompletableFuture<Void>>();
        try {
            for (Integer workerId : workerIds) {
                futures.add(this.sender.send((int)workerId, type));
            }
        }
        catch (InterruptedException e) {
            throw new ComputerException("Interrupted when waiting to send message async");
        }
        long timeout = type == MessageType.FINISH ? this.transportConf.timeoutFinishSession() : this.transportConf.timeoutSyncRequest();
        try {
            for (CompletableFuture completableFuture : futures) {
                completableFuture.get(timeout, TimeUnit.MILLISECONDS);
            }
        }
        catch (TimeoutException e) {
            throw new ComputerException("Timeout(%sms) to wait for controling message(%s) to finished", (Throwable)e, new Object[]{timeout, type});
        }
        catch (InterruptedException | ExecutionException e) {
            throw new ComputerException("Failed to wait for controling message(%s) to finished", (Throwable)e, new Object[]{type});
        }
    }

    private void checkException() {
        if (this.exception.get() != null) {
            throw new ComputerException("Failed to send message", this.exception.get());
        }
    }
}

